/*+++
1 SaxsImage.c

2 PURPOSE
Input and output routines for time-resolved small-angle x-ray scattering
data that are stored in the ESRF data format (esrf_data_format.h).
A special sub-structure of this data format is defined which is
suitable for two dimensional time resolved data. The data are
stored in a float array of the size Dim_1 * Dim_2 (columns, rows).

Image data headers are marked with the keyword "image" followed by a positive
number larger than 0.

2 KEYWORDS and UNITS
Only SI units are used (meter, kilogramm, second).
Application programs may use other units, but they have to store their
values in SI units. The axes 1, 2 and 3 of the images form a right-handed
orthogonal coordinate system.
Because the direction of the axes are hardware depending they are not fixed
in the real world. Generally axis 1 is the horizontal axis, axis 2 the
vertical axis and axis 3 points from the detector to the sample.

The following keywords are used by the routines (default in square brackets):

Image header

  ImNum         = <image number>
  Dim[0]        = dimensions (2)
  Dim[1]        = number of columns
  Dim[2]        = number of rows
  Dummy         = dummy value (0.0 = no dummy) [0.0]
  DDummy        = delta dummy (range around a dummy) [0.1]
  Offset[1]     = column offset of all pixels [0.0]
  Offset[2]     = row offset of all pixels [0.0]
  Center[1]     = column center of the image
  Center[2]     = row center of the image

 optional

  PixSiz[1]       = distance in meters or seconds between two columns
  PixSiz[2]       = distance in meters or seconds between two rows
  SampleDistance  = distance of the sample in meters from the center that is
                    described by (Center_1, Center_2)
  WaveLength      = photon wavelength in meters
  Title           = one line with length InputLineLength-1 characters or shorter
  Time            = one line with start time,
                    preferred iso time: YYYY-MM-DD hh:mm:ss[.ssssss]
  Orientation     = orientation of the array (numbers from 1 to 8),
                    default 1

The keywords and the used coordinate systems are defined in the file
SaxsDefinition.h.

If a keyword has been found in the image header the corresponding .I flag
in the image header block is set to TRUE (+1) and the parameter is read.
It is also possible that a .I flag is set to -TRUE (-1).  This is the case
if a header block value has been overwritten by an image block value. The
test whether a flag is set must be : if (xxx.I!=0) the isset.

HISTORY
16-Nov-1995	Definition of return value of ReadImage and ReadImageHeader
		changed.
24-Nov-1995 PB	
27-Nov-1995 PB  ImgHeadBlock.I
04-Dec-1995 PB  IO_DontReadData in NewImage
21-Dec-1995 PB  all data format specific routines in external file
                (GetData2d.inc)
23-Dec-1995 PB  splitting in test_image.c and test_image.h
24-Dec-1995 PB  ReportSaxsError + more debug information for +test
25-Dec-1995 PB  ImageNumber int --> long
26-Dec-1995 PB  long --> long int
26-Jan-1996 PB  IO_DontRead --> IO_DontReadData (replaced 1:1)
                IO_DontWrite --> IO_DontReadWrite
                IO_DontReadWrite is now also checked in the read routines
                ReadImage and ReadImageHeader
27-Jan-1996 PB  ReadImageDimension added
28-Feb-1996 PB  alloc_image_memory: special case for ImgByteLen==0
29-Mar-1996 PB  MinClip and MaxClip
19-May-1996 PB  Gnaw, the order of operation for read and write is:
                clip, gnaw, rebin and transform image and image header
10-Oct-1996 PB  renamed from test_image.c to SaxsImage.c,
                include "ImageLoop.inc"
12-Oct-1996 PB  Error block modified, ReportSaxsError changed, SetSaxsError
14-Oct-1996 PB  V1.0 SAXSIMAGEVERSION="1.0" removed, replaced by
                SIM_Version, SaxsImageVersion
13-Dec-1996 PB  rebin_float... has only checked ..Bin[1].I, now also Bin[2].I
28-Aug-1997 PB  debug information added (SIM_Version V1.2)
31-Dec-1997 PB	SaxsGetData2d.inc -> ndf.inc (SIM_Version V1.3)
01-Jan-1997 PB  ndf.inc renamed to SaxsNewData2d.inc
12-Jan-1998 PB  all free(xyz); replaced against {free(xyz);xyz=NULL;}
                to avoid a second attempt to free the data (SIM_Version V1.31)
12-Jan-1998 PB  call to open_data_file now with: new, old, any (instead of
                any only) V1.32
17-Mar-1998 PB  include edfio.h V1.33
17-Mar-1998 PB  GeneralBlock option V1.34 in OpenImageFile 
17-May-1998 PB  call to open_data_file now with: new, old, any and read V1.35
18-Apr-1999 PB  new names of edfio routines V1.36
30-Apr-1999 PB  status set to true before calling edfio routines V1.37
25-Jun-1999 PB  edf_history V1.40:
                edf_read_header_history -> read_saxs_image 
                edf_write_header_history --> write_image_header
                The history is read from blkno 1 and written to
                blkno 0 together with the argument history (SaxsOption)
26-Jun-1999 PB  Time added to image header block V1.50
17-Jul-1999 PB  Comment corrected V1.51
17-Jul-1999 PB  V1.52
                definition of SAXSDATAVERSION removed (now in SaxsDefinition.h)
09-Nov-1999 PB  V1.53 DUMMYDEFINED updated
26-Nov-1999 PB  V1.54 cc on dec alpha: statements of the type *pd++=*pd+xyz;
                      changed to { *pd=*pd+xyz; pd++; }
21-Dec-1999 PB  V1.55 DummyFileName in OpenImageFile, IO_Dummy,
                      and alloc_image_memory in read_saxs_image 
18-Mar-2000 PB  V1.60 memory replaces ImageNumber 
                      modified: write_image_header, WriteImage,
                      read_saxs_image, ReadImage, ReadImageHeader,
                      SearchMinMaxImage, ReadImageDimension,
                      ReadHeaderLine, ReadHeaderLong, ReadHeaderFloat,
                      saxs_image_loop (default of output memory number
                      is memory number of first input)
07-Apr-2000 PB  V1.61 set_max_line_width
30-Jul-2000 PB  V1.62 Orientation added to image header block
                      orientation_float_2d, orientation_header_2d
                      included in read_saxs_image and WriteImage
31-Aug-2000 PB  V1.63 debug info in ReadHeader...
10-Nov-2000 PB  V1.7  condition_2d, changes for edfio V1.30
12-Nov-2000 PB  V1.71 changes for edfio V1.32
                V1.72 long ImgByteLen -> size_t ImgByteLen
                V1.73 changes for edfio V1.34
                      long ihb.Dim[3] -> long ihb.Dim[4]
2000-12-30      V1.74 changes for edfio V1.36
                      memory allocated for input image
2000-12-31      V1.75 element Dim[3] not used any more
2001-01-11      V1.76 NewImageHeader
2001-01-12      V1.77 u_long -> unsigned long
2001-07-03      V1.78 SaxsGetName.inc replaced by SaxsFilename.h
2001-07-06      V1.79 OpenImageFile uses edf_search_stream
2001-07-06      V1.80 default for ib[].Stream is -1 (no stream opened)
2001-07-07      V1.81 DoNotScaleWith
2001-07-08      V1.82 OpenImageFile(...,FileNumber,...)
2001-07-09      V1.83 OpenImageFile sets ib[].FileNumber and ib[].FileName
2001-07-10      V1.84 New SaxsImageLoop
2001-07-12      V1.85 New SaxsImageLoop.inc, ActualLoopCount
2001-08-19      V1.86 WriteHeaderLine, WriteHeaderFloat, WriteHeaderLong
                      ImageNumber removed from argument list of WriteImage
2001-08-20      V1.87 NewImageHeader
2001-08-30      V1.88 pihb->Write, SaxsImageMessage,
                      ImageIsNew, ImageToWrite, ImageIsFirst, ImageIsLast
2001-09-02      V1.89 edf_set_datatype, edf_set_datavalueoffset
2001-09-13      V1.90 history_key
2001-09-16      V1.91 PassHeader, header key in image header block, Header in
                      read_saxs_image, New_Image_Header, free_image_memory
2001-09-17      V1.92 write_image_header
---*/

/****************************************************************************
* SaxsImage.c                                                               *
****************************************************************************/
# define SIM_Version  "SaxsImage : V1.92 Peter Boesecke 2001-09-17"

# include <stdio.h>
# include <string.h>
# include <unistd.h>
# include <stdlib.h>

# include "SaxsImage.h"
# include "SaxsFilename.h"
# include "edfio.h"                        /* data format specific routines */

/*----------------------------------------------------------------------------
 Version String
----------------------------------------------------------------------------*/
static char * VersionString = (char *) NULL;

/*----------------------------------------------------------------------------
 Constants
----------------------------------------------------------------------------*/
const char * ModuleName    = "SaxsImage";
const char * DummyFileName = DUMMYFILENAME;

/*----------------------------------------------------------------------------
 Static Messages 
----------------------------------------------------------------------------*/
static char SaxsImageMessage[IO_len];

/*----------------------------------------------------------------------------
 Private include files
----------------------------------------------------------------------------*/
/* include ImageLoop */
# include "SaxsImageLoop.inc"

/*----------------------------------------------------------------------------
1 SaxsImageVersion

2 DESCRIPTION
  Returns a pointer to a string which explains the version of the used
  modules
  char * SaxsImageVersion ( void )
  "'SIM_Version' (Files = 'edf_dataformat_version')"

2 EXTERNAL FILES
  SaxsGetData2d.inc (edf_dataformat_version)

2 HISTORY
  14-Oct-1996 PB
  12-Jan-1997 PB free(xyz) -> { free(xyz);xyz=NULL;}
----------------------------------------------------------------------------*/
char * SaxsImageVersion ( void )
{ size_t t1, t2, t3;
  if ( VersionString!=(char *) NULL ) { 
    free( VersionString ); VersionString=NULL; }
  t1 = strlen(SIM_Version);
  t2 = strlen((char *) edf_dataformat_version() );
  t3 = strlen(SAXSDATAVERSION);
  VersionString =  (char *) malloc ( 30 + t1 + t2 + t3 + 1 );
  if ( VersionString!=(char *) NULL )
    sprintf(VersionString,"% s (SDV = % s, GDV = % s)",
              SIM_Version, SAXSDATAVERSION, edf_dataformat_version() );
  return ( VersionString );
} /* SaxsImageVersion */

/*--------------------------------------------------------------------------
NAME

  orientation_float_2d --- change orientation of 2d float data array

SYNOPSIS

  void orientation_float_2d( float *dataIn, float * dataOut,
                             long int dim_1, long int dim_2,
                             long relative_orientation,
                             int * pstatus );

DESCRIPTION
Change orientation of 2d float data array.

ARGUMENTS
  float * dataIn  : input data array
  float * dataOut : output data array (must be different from dataIn)
  long int dim_1  : dimensions of input and output arrays
  long int dim_2
  long relative_orientation : orientation of input array relative to the
                              desired final position
  int * pstatus   : Success or Failed

RETURN VALUE
  void

HISTORY
  2000-07-30 Peter Boesecke
--------------------------------------------------------------------------*/
void orientation_float_2d( float *dataIn, float * dataOut,
                           long int dim_1, long int dim_2,
                           long relative_orientation,
                           int * pstatus )
{ long data_dim[4];

  *pstatus = Success;

  data_dim[0] = 2l;
  if (relative_orientation<=4l) {
    data_dim[1] = dim_1; data_dim[2] = dim_2; 
    } else {
    data_dim[1] = dim_2; data_dim[2] = dim_1;
    }

  if ( edf_raster_normalization ( (void *) dataOut, (const void *) dataIn,
                data_dim, relative_orientation, sizeof( float ) ) ) {
    *pstatus = Failed;
    }

} /* orientation_float_2d */

/*--------------------------------------------------------------------------
NAME

  orientation_header_2d --- update header after a change of orientation

SYNOPSIS

  void orientation_header_2d ( ImgHeadBlk * pihb, long relative_orientation );

DESCRIPTION
The header values are changed according to a data transformation from
relative_orientation to 1

ARGUMENTS
  ImgHeadBlk * pihb         : image header block of the current data
  long relative_orientation : orientation relative to the final position

RETURN VALUE
  long orientation  > 0 final orientation of the data array
                   == 0 error

HISTORY
  2000-07-30 Peter Boesecke
--------------------------------------------------------------------------*/
long orientation_header_2d ( ImgHeadBlk * pihb, long relative_orientation )
{ long tmpl;
  IO_float tmpf;
  long inverse_orientation = edf_raster_inversion( relative_orientation );
  long orientation =
     edf_raster_multiplication( inverse_orientation, pihb->Orientation.V );
  long omod2, omod4;

  omod2 = (inverse_orientation-1) % 2l;
  omod4 = (inverse_orientation-1) % 4l;

  if ( omod2 >= 1l ) { // invert first coordinate
    pihb->Center[1].V = CSWAP(pihb->Center[1].V,pihb->Offset[1].V,pihb->Dim[1]);
    }

  if ( omod4 >= 2l ) { // invert second coordinate
    pihb->Center[2].V = CSWAP(pihb->Center[2].V,pihb->Offset[2].V,pihb->Dim[2]);
    }

  if ( inverse_orientation > 4l ) { // swap coordinates
    tmpl=pihb->Dim[1];    pihb->Dim[1]   =pihb->Dim[2];    pihb->Dim[2]   =tmpl;
    tmpf=pihb->Offset[1]; pihb->Offset[1]=pihb->Offset[2]; pihb->Offset[2]=tmpf;
    tmpf=pihb->Center[1]; pihb->Center[1]=pihb->Center[2]; pihb->Center[2]=tmpf;
    tmpf=pihb->PixSiz[1]; pihb->PixSiz[1]=pihb->PixSiz[2]; pihb->PixSiz[2]=tmpf;
    }

  pihb->Orientation.V = orientation; pihb->Orientation.I = TRUE;

  return ( orientation );

} /* orientation_header_2d */

/*----------------------------------------------------------------------------
rebin_float_2d
  Two dimensional rebinning of an array. The result is written into the
  output array and the new dimensions are returned. Input and output 
  arrays can be the same.
Parameters
  float * dataIn  (i)     pointer to input array
  float * dataOut (o)     pointer to output array
  long * dim_1    (i/o)   pointer to input/output dimension 1
  long * dim_2    (i/o)   pointer to input/output dimension 2
  float dummy     (i)     dummy value
  float ddummy    (i)     ddummy value
  long bin_1      (i)     rebin factor 1 (>=1)
  long bin_2      (i)     rebin factor 2 (>=1)
-----------------------------------------------------------------------------*/
void rebin_float_2d ( float * dataIn, float * dataOut,
                      long int * dim_1, long int * dim_2,
                      float dummy, float ddummy,
                      long int bin_1, long int bin_2)
{ register long int j_1, j_2, i_1, i_2;
  float * pin, * pout;
  float value, sum, count;

  if ((bin_1>1) || (bin_2>1)) {
    pout = dataOut;
    for (j_2=0;j_2<=*dim_2-bin_2;j_2+=bin_2)
      for (j_1=0;j_1<=*dim_1-bin_1;j_1+=bin_1) {
        sum = 0.0; count = 0.0;
        for (i_2=j_2;i_2<j_2+bin_2;i_2++) {
          pin = ABSPTR(dataIn,*dim_1,*dim_2,j_1,i_2);
          for (i_1=j_1;i_1<j_1+bin_1;i_1++) {
            value = *(pin++);
            if NODUMMY(value,dummy,ddummy)
              { sum += value; count += 1.0; }
            }
          }
        if (count>0.0) *(pout++) = sum/count; else *(pout++) = dummy;
        }
     *dim_1 /= bin_1; *dim_2 /= bin_2;
     }

} /* rebin_float_2d */

/*---------------------------------------------------------------------------
  transform_float_2d
  Multiplies the data field with a factor and adds a constant.
  ---------------------------------------------------------------------------*/
void transform_float_2d( float *data, long int dim_1, long int dim_2,
                         float dummy, float ddummy,
                         float factor, float constant )
{ register long int i;
  long int datanum;
  const float eps1 = 1e-9, eps2 = 1e-30;
  float *pdatain;
  float *pdataout;

  datanum = dim_1 * dim_2;
  pdatain = data;
  pdataout = data;
  if ((ABS(factor-1.0)>eps1) || (ABS(constant)>eps2)) { 

    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdatain,dummy,ddummy) 
           *pdataout = *pdatain * factor + constant;
        pdataout +=1; pdatain +=1;
        }
      } else for (i=0;i<datanum;i++) {
               *pdataout = *pdatain * factor + constant; 
               pdataout++; pdatain++;
               }
    } /* if ((ABS ...  */ 

} /* transform_float_2d */

/*---------------------------------------------------------------------------
  clip_float_2d
  Sets all values of the data field outside minclip and maxclip to dummy.
  ---------------------------------------------------------------------------*/
void clip_float_2d( float *data, long int dim_1, long int dim_2,
                         float dummy, float ddummy,
                         float minclip, int minclipI,
                         float maxclip, int maxclipI )
{ register long int i;
  long int datanum;
  float *pdata;

  datanum = dim_1 * dim_2;
  pdata = data;
  if (minclipI) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
           if (minclip>*pdata) *pdata = dummy;
           }
        pdata +=1;
        }
      } else for (i=0;i<datanum;i++) {
               if (minclip>*pdata) *pdata = dummy;
               pdata +=1;
               }
    } /* if (minclipI) */

  pdata = data;
  if (maxclipI) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
           if (maxclip<*pdata) *pdata = dummy;
           }
        pdata +=1;
        }
      } else for (i=0;i<datanum;i++) {
               if (maxclip>*pdata) *pdata = dummy;
               pdata +=1;
               }
    } /* if (maxclipI) */

} /* clip_float_2d */

/*----------------------------------------------------------------------------
gnaw_float_2d
  Sets all pixels inside an ellipse around a dummy to dummy.
  The result is written into the input array. The routine
  allocates a short integer array with the same dimensions as the
  data array for intermediate results. Its memory is released after
  successful termination of gnaw_float_2d.
Parameters
  float * data    (i)     pointer to input/output array
  long    dim_1   (i)     input/output/tmp dimension 1
  long    dim_2   (i)     input/output/tmp dimension 2
  float   dummy   (i)     dummy value
  float   ddummy  (i)     ddummy value
  long    gnaw_1  (i)     axis 1 of the ellipse (>=0)
  long    gnaw_2  (i)     axis 2 of the ellipse (>=0)
  int   * pstatus (o)     exit status
-----------------------------------------------------------------------------*/
void gnaw_float_2d ( float * data,
                     long int dim_1, long int dim_2,
                     float dummy, float ddummy,
                     long int gnaw_1, long int gnaw_2,
                     int * pstatus )
{ register long int i, j_1, j_2, i_1, i_2;
  long int i_1min;
  float * pdata;
  short int * tmp, * ptmp;
  size_t t_tmp;

  float value;
  long int gnaw2_1, gnaw2_2, gnaw2;
  long int dist2_1, dist2_2;

  *pstatus = Success;

  gnaw_1 = MAX2(0,gnaw_1);
  gnaw_2 = MAX2(0,gnaw_2);

  if ((gnaw_1>0) || (gnaw_2>0)) {

    /* allocate memory for tmp array*/
    t_tmp = sizeof(short int)*dim_1*dim_2;
    tmp = (short int *) malloc (t_tmp);
    if (tmp == (short int *) NULL) {
      *pstatus=NotEnoughMemoryAvailable; return; }

    /* clear tmp array */
    ptmp = tmp;
    for (i=0;i<dim_1*dim_2;i++) *ptmp++=0;

    /* search dummies in data array */
    gnaw2_1 = gnaw_1 * gnaw_1 + 1; /* to avoid stars with gnaw_1 = 1 */
    gnaw2_2 = gnaw_2 * gnaw_2 + 1; /* to avoid stars with gnaw_2 = 1 */
    gnaw2   = gnaw2_1*gnaw2_2;
    pdata = data;
    for (j_2=0;j_2<dim_2;j_2++)
      for (j_1=0;j_1<dim_1;j_1++) {
        value = *(pdata++);
        if DUMMY(value,dummy,ddummy) {
          /* create dummy ellipse around j_1, j_2 in tmp */
          for (i_2=MAX2(0,j_2-gnaw_2);i_2<MIN2(dim_2,j_2+gnaw_2+1);i_2++) {
            dist2_2 = i_2 - j_2; dist2_2 *= dist2_2;
            i_1min = MAX2(0,j_1-gnaw_1);
            ptmp = ABSPTR(tmp,dim_1,dim_2,i_1min,i_2);
            for (i_1=i_1min;i_1<MIN2(dim_1,j_1+gnaw_1+1);i_1++) {
              dist2_1 = i_1 - j_1; dist2_1 *= dist2_1;
              if (dist2_1*gnaw2_2+dist2_2*gnaw2_1<=gnaw2) *ptmp = 1;
              ptmp++;
              }
            }
          }
        }

    /* copy dummies from tmp to data */
    ptmp = tmp;
    pdata = data;
    for (i=0;i<dim_1*dim_2;i++) {
      if (*ptmp++) *pdata=dummy;
      pdata++;
      }

    /* free tmp array */
    free(tmp); tmp=NULL;

    }

} /* gnaw_float_2d */

/*----------------------------------------------------------------------------
  NAME
    condition_2d

  DESCRIPTION

    clip, gnaw, rebin, orient and transform image and image header 

  ARGUMENTS
    ImgHeadBlk * pihb     pointer to image header block
    ...
    long Orientation      transform data orientation from 
                          Orientation to 1, e.g.
                          read data:  pihb->Orientation.V
                          write data: edf_raster_inversion(pib->Orientation.V) 
    int Simulate          TRUE:  condition header only,
                          FALSE: condition header and data
    int TestBit           debug flag
    int *pstatus          exit status
-----------------------------------------------------------------------------*/
void condition_2d( ImgHeadBlk * pihb,
                   IO_float MinClip, IO_float MaxClip,
                   IO_long  Gnaw_1,  IO_long  Gnaw_2,
                   IO_long  Bin_1,   IO_long  Bin_2,
                   IO_float Factor,  IO_float Const,
                   long Orientation,
                   int Simulate, int TestBit, int *pstatus )
{
  float * ptmp;
  float * pdatain = (float *) NULL;
  float * pdataout = (float *) NULL;
  float * TmpBuffer = (float *) NULL;
  size_t  TmpBufferSize = (size_t) NULL;

    // Clipping
    if (MinClip.I || MaxClip.I) {
      if (!Simulate) {
        if (TestBit>1) printf(" --- clip_float_2d\n");
        clip_float_2d( pihb->Data, pihb->Dim[1], pihb->Dim[2],
                       pihb->Dummy.V, pihb->DDummy.V,
                       MinClip.V, MinClip.I,
                       MaxClip.V, MaxClip.I );
        } // if (!Simulate)
      } // Clipping

    // Gnawing
    if (Gnaw_1.I || Gnaw_2.I) {
      if (!Simulate) {
        if (TestBit>1) printf(" --- gnaw_float_2d\n");
        gnaw_float_2d ( pihb->Data, pihb->Dim[1], pihb->Dim[2],
                        pihb->Dummy.V, pihb->DDummy.V,
                        Gnaw_1.V, Gnaw_2.V,
                        pstatus );
        if (*pstatus!=Success) return;
        } // if (!Simulate)
      } // Gnawing

    // Binning
    if (!Bin_1.I) Bin_1.V = 1l; else Bin_1.V = MAX2(1,Bin_1.V);
    if (!Bin_2.I) Bin_2.V = 1l; else Bin_2.V = MAX2(1,Bin_2.V);
    if ((Bin_1.V>1) || (Bin_2.V>1)) {
      if (!Simulate) {
        if (TestBit>1) printf(" --- rebin_float_2d\n");
        rebin_float_2d ( pihb->Data, pihb->Data,
                         &pihb->Dim[1], &pihb->Dim[2],
                         pihb->Dummy.V, pihb->DDummy.V,
                         Bin_1.V, Bin_2.V );
        } else {
        // update dimension
        pihb->Dim[1] /= Bin_1.V;
        pihb->Dim[2] /= Bin_2.V;
        } // if (!Simulate)
      // update header
      AREBIN(pihb->Offset[1].V,pihb->PixSiz[1].V,pihb->Center[1].V,Bin_1.V);
      AREBIN(pihb->Offset[2].V,pihb->PixSiz[2].V,pihb->Center[2].V,Bin_2.V);
      } // Binning

    /* During orientation transformatino the input and output buffers
       must be different. Data is moved between the two buffers and
       finally copied to pihb->Data, either with transform or memcpy */

    pdatain  = pihb->Data; // pointer to actual input data
    pdataout = (float *) NULL; // currently no separate output buffer

    // Orientation

    if (Orientation > 1l) {
      if (!Simulate) {

        if (!pdataout) { // allocate output buffer
          if (TestBit>1) printf(" --- allocate memory for orientation\n");
          TmpBufferSize = pihb->Dim[1]*pihb->Dim[2]*sizeof(float);
          TmpBuffer = (float *) malloc ( TmpBufferSize );
          if (!TmpBuffer) { *pstatus=NotEnoughMemoryAvailable;return; }
          pdataout = TmpBuffer; // pointer to new output buffer
          } // allocate output buffer

        // transform data from Orientation to orientation 1
        if (TestBit>1) printf(" --- orientation_float_2d\n");
        orientation_float_2d( pdatain, pdataout,
                              pihb->Dim[1], pihb->Dim[2],
                              Orientation,
                              pstatus );
        if (*pstatus!=Success) {
          if (TmpBuffer) { free(TmpBuffer); TmpBuffer = NULL; } return;
          }
        ptmp = pdatain; pdatain = pdataout; pdataout = ptmp;
        } // if (!Simulate)

      // update header
      if (TestBit>1) printf(" --- orientation_header_2d\n");
      if (!orientation_header_2d ( pihb, Orientation )) {
        if (TmpBuffer) { free(TmpBuffer); TmpBuffer = NULL; }
        *pstatus=Failed; return; }
      } // Orientation

    // Scalar operations
    if (Factor.I || Const.I ) {
      if (!Simulate) {
        if (TestBit>1) printf(" --- transform_float_2d\n");
        transform_float_2d( pdatain, pihb->Dim[1], pihb->Dim[2],
                            pihb->Dummy.V, pihb->DDummy.V,
                            Factor.V, Const.V );
        } // if (!Simulate)
      } // Scalar operations

    if (!Simulate) {
      if (pdatain!=pihb->Data) { // data is still in temporary buffer
        // copy temporary buffer to image data
        if (TestBit>1) printf(" --- copy temporary buffer to image data\n");
        memcpy((void *) pihb->Data, (void *) TmpBuffer, TmpBufferSize); 
        }

      // release temporary buffer
      if (TmpBuffer) { free(TmpBuffer); TmpBuffer = (float *) NULL; }
      } // if (!Simulate)

} // condition_2d

/*--------------------------------------------------------------------------*/
/* set_header_defaults
/*--------------------------------------------------------------------------*/
void set_header_defaults( ImgHeadBlk * ihb )
{
        /* standard SAXS header values */
        if (!ihb->Dummy.I)     {ihb->Dummy.V = 0.0;ihb->Dummy.I = TRUE;}
        if (!ihb->DDummy.I)    {ihb->DDummy.V = DDSET(ihb->Dummy.V);
                               ihb->DDummy.I=TRUE;}
        if (!ihb->Offset[1].I) {ihb->Offset[1].V=0.0;ihb->Offset[1].I=TRUE;}
        if (!ihb->Offset[2].I) {ihb->Offset[2].V=0.0;ihb->Offset[2].I=TRUE;}

        /* optional SAXS header values
        if (!ihb->Center[1].I)     ;
        if (!ihb->Center[2].I)     ;
        if (!ihb->PixSiz[1].I)     ;
        if (!ihb->PixSiz[2].I)     ;
        if (!ihb->SampleDistance.I);
        if (!ihb->WaveLength.I)    ;
        if (!ihb->Title.I)         ;
        if (!ihb->Time.I)          ; 
        if (!ihb->Orientation.I)   ;
        */

} /*  set_header_defaults */

/*+++------------------------------------------------------------------------
1 geo_header_copy

2 DESCRIPTION
  Copies the geometrical keywords from pin to pout
----------------------------------------------------------------------------*/
void geo_header_copy ( ImgHeadBlk *pin, ImgHeadBlk *pout ) 
{ register int i;

  for (i=0;i<3;i++) {
    pout->Offset[i].I = (pin->Offset[i].I) ?  TRUE : FALSE;
    pout->Center[i].I = (pin->Center[i].I) ?  TRUE : FALSE;
    pout->PixSiz[i].I = (pin->PixSiz[i].I) ?  TRUE : FALSE; }
  pout->SampleDistance.I = (pin->SampleDistance.I) ?  TRUE : FALSE;
  pout->WaveLength.I =  (pin->WaveLength.I) ?  TRUE : FALSE;
  pout->Orientation.I = (pin->Orientation.I) ? TRUE : FALSE;

  for (i=0;i<3;i++) {
    pout->Offset[i].V = pin->Offset[i].V;
    pout->Center[i].V = pin->Center[i].V;
    pout->PixSiz[i].V = pin->PixSiz[i].V; }
  pout->SampleDistance.V = pin->SampleDistance.V;
  pout->WaveLength.V =  pin->WaveLength.V;
  pout->Orientation.V = pin->Orientation.V;

} /* geo_header_copy */

/*+++------------------------------------------------------------------------
1 template_header_block

2 DESCRIPTION
  Use the image header block *pin as a template for *pout.
  In case of a new header reset_header_block must be executed before
  calling template_header_block.

  return( void )                :

  ImgHeadBlk *pin       (i)     : input image header block
  ImgHeadBlk *pout      (o)     : output image header block


2 HISTORY
  12-Feb-1995   Peter Boesecke creation
  22-Feb-1995   PB      Copy renamed in Template
  26-Feb-1995   PB      Factor and Const added
  05-Mar-1995   PB      DataCtrl added
  08-Sep-1995   PB      Input .I flags checked with if (xxx.I).
                        All output .I flags are set to TRUE if input flag
                        is set.
  04-Dec-1995   Stream and StreamOpenMode
  07-Dec-1995   geo_header_copy
  12-Jul-2001   ActualLoopCount
  28-Aug-2001   Write
----------------------------------------------------------------------------*/
void template_header_block ( long int ImageNumber, 
                             long int DataCtrl, long int ActualLoopCount,
                             size_t ImgByteLen, long int Dim[], float * Data,
                             ImgHeadBlk *pin, ImgHeadBlk *pout )
{ register int i,j;
  /* control header */
  /*  DataCtrl and ActualLoopCount not changed */
  pout->ImNum =         ImageNumber;
  pout->Stream =        -1;
  pout->StreamOpenMode =0l;
  pout->DataCtrl =      DataCtrl;
  pout->ActualLoopCount=ActualLoopCount;
  pout->Write =         FALSE;
  pout->ImgByteLen =    ImgByteLen;
  for (i=0;i<3;i++) { pout->Dim[i] = Dim[i]; }
  pout->Data =          Data;
  pout->DoNotScale =    pin->DoNotScale;
  pout->Header.I =      FALSE; /* Header is not passed */
  /* SAXS keywords */
  pout->Dummy.I =       (pin->Dummy.I) ?  TRUE : FALSE;
  pout->DDummy.I =      (pin->DDummy.I) ?  TRUE : FALSE;
  pout->Title.I =       (pin->Title.I) ?  TRUE : FALSE;
  pout->Time.I =        (pin->Time.I) ? TRUE : FALSE;

  pout->Dummy.V =       pin->Dummy.V;
  pout->DDummy.V =      pin->DDummy.V;
  for (j=0;j<IO_len,pout->Title.V[j]=pin->Title.V[j];j++);
  for (j=0;j<IO_len,pout->Time.V[j]=pin->Time.V[j];j++);

  geo_header_copy(pin,pout);

} /* template_header_block */

/*+++------------------------------------------------------------------------
1 balance_header_block

2 DESCRIPTION
  Fill the image header block *pihb with all set values of the image block
  *pib.
  return( void )                :

  long int      ImageNumber (i)	: image number;
  ImgBlk     *pib       (i)     : input image block
  ImgHeadBlk *pihb      (i/o)   : image header block

2 HISTORY
  12-Feb-1995   Peter Boesecke creation
  21-Sep-1995   Offset set to defaults (0.0) if not set in ihb.
  08-Oct-1995   .I flags -TRUE (-1), if set by image block, others unchanged.
  04-Dec-1995   Actual Stream and StreamOpenMode copied from image block
  ----------------------------------------------------------------------------*/
void balance_header_block ( long int ImageNumber,ImgBlk *pib,ImgHeadBlk *pihb )
{
  register int i,j;

  pihb->ImNum = ImageNumber;

  /* copy actual stream and actual stream open mode */
  pihb->Stream = pib->Stream;
  pihb->StreamOpenMode = pib->StreamOpenMode;

  for (i=0;i<3;i++) {
    if (pib->Dim[i].I) { pihb->Dim[i] = pib->Dim[i].V; }
    }
  /* Dummy and DDummy must always be set. If not set use default values for
     Dummy and DDummy and set .I flags to (+TRUE).
     Check then whether DDummy has a sound number.
     Result overwritten and .I flags are set to (-TRUE) if
     values for Dummy and DDummy are found in the image block pib */
  /*    Set defaults */
  if (!pihb->Dummy.I) { /* if Dummy is not set, recalculate DDummy */
      pihb->Dummy.V = 0; pihb->DDummy.V = DDSET( 0 );
      pihb->Dummy.I = pihb->DDummy.I = TRUE; }
  if (!pihb->DDummy.I) { /* if DDummy is still not set, set it */
    pihb->DDummy.V = DDSET( pihb->Dummy.V ); pihb->DDummy.I=TRUE;
    } else { /* if DDummy is set, check it */
    pihb->DDummy.V = MAX2( DDSET( pihb->Dummy.V ), pihb->DDummy.V );
    }
  /*    Overwrite with image block values */
  if (pib->Dummy.I) { /* if new Dummy is used, recalculate DDummy */
    pihb->Dummy.V = pib->Dummy.V; pihb->DDummy.V = DDSET( pihb->Dummy.V );
    pihb->Dummy.I = pihb->DDummy.I = (-TRUE); }
  if (pib->DDummy.I) { /* if one really wants another one, OK */
    pihb->DDummy.V = pib->DDummy.V;
    pihb->DDummy.I = (-TRUE); }

  /* Offset[1] and Offset[2] must always be set, set default values,
     and flags to (+TRUE) if not set */
  /*    Set defaults */
  for (i=0;i<3;i++) {
    if (!pihb->Offset[i].I) {
      pihb->Offset[i].V = 0.0;
      pihb->Offset[i].I = TRUE; } /* This value is not taken from the image
                                     block, therefore not (-TRUE) */
    }
  /*    Overwrite with image block values */
  for (i=0;i<3;i++) {
    if (pib->Offset[i].I) {
      pihb->Offset[i].V = pib->Offset[i].V;
      pihb->Offset[i].I = (-TRUE); }
    if (pib->Center[i].I) {
      pihb->Center[i].V = pib->Center[i].V;
      pihb->Center[i].I = (-TRUE); }
    if (pib->PixSiz[i].I) {
      pihb->PixSiz[i].V = pib->PixSiz[i].V;
      pihb->PixSiz[i].I = (-TRUE); }
    }
  if (pib->SampleDistance.I) {
    pihb->SampleDistance.V = pib->SampleDistance.V;
    pihb->SampleDistance.I = (-TRUE); }
  if (pib->WaveLength.I) {
    pihb->WaveLength.V = pib->WaveLength.V;
    pihb->WaveLength.I = (-TRUE); }
  if (pib->Title.I) {
    for (j=0;j<IO_len,pihb->Title.V[j]=pib->Title.V[j];j++);
    pihb->Title.I = (-TRUE); }
  if (pib->Time.I) {
    for (j=0;j<IO_len,pihb->Time.V[j]=pib->Time.V[j];j++);
    pihb->Time.I = (-TRUE); }
  if (pib->Orientation.I) {
    pihb->Orientation.V = pib->Orientation.V;
    pihb->Orientation.I = (-TRUE); }

} /* balance_header_block */

/*+++------------------------------------------------------------------------
1 reset_header_block

2 DESCRIPTION
  Sets header control to 0, all .I flags to FALSE and all .V to 0.
  return( void )                :

  ImgHeadBlk *pihb      (i/o)   : image header block

2 HISTORY

----------------------------------------------------------------------------*/
void   reset_header_block( ImgHeadBlk * pihb )

{ register int i;


  /* header control */
  pihb->I          = 0; /* header not valid */
  pihb->ImNum      = 0l;
  pihb->Stream     = -1;
  pihb->StreamOpenMode   = 0l;
  pihb->DataCtrl   = 0l;
  pihb->ActualLoopCount = 0l;
  pihb->Write      = FALSE;
  pihb->ImgByteLen = (size_t) NULL;
  pihb->Dim[0]     = pihb->Dim[1] = pihb->Dim[2] = 0;
  pihb->Data       = (float *) NULL;
  pihb->DoNotScale = FALSE;
  pihb->Header.I   = FALSE;
  *(pihb->Header.V)= (char) NULL;

  /* SAXS keywords */
  pihb->Dummy.I    = FALSE;
  pihb->DDummy.I   = FALSE;
  for (i=0;i<3;i++) {
    pihb->Offset[i].I = FALSE;
    pihb->Center[i].I = FALSE;
    pihb->PixSiz[i].I = FALSE;
    } /* for */
  pihb->SampleDistance.I = FALSE;
  pihb->WaveLength.I = FALSE;
  pihb->Title.I    = FALSE;
  pihb->Time.I     = FALSE;
  pihb->Orientation.I = FALSE;

  /* SAXS keywords */
  pihb->Dummy.V    = 0.0;
  pihb->DDummy.V   = 0.0;
  for (i=0;i<3;i++) {
    pihb->Offset[i].V = 0.0;
    pihb->Center[i].V = 0.0;
    pihb->PixSiz[i].V = 0.0;
    } /* for */
  pihb->SampleDistance.V = 0.0;
  pihb->WaveLength.V = 0.0;
  pihb->Title.V[0] = (char) NULL;
  pihb->Time.V[0] = (char) NULL;
  pihb->Orientation.V = 1l; // default 1

} /* reset_header_block */

/*+++------------------------------------------------------------------------
1 alloc_image_memory

2 DESCRIPTION
  Allocates Image memory

  ImgHeadBlk *pihb      (i/o)   : image header block
  pihb->DataCtrl	(o)	: resetted to 0
  pihb->ActualLoopCount (o)     : resetted to 0
  pihb->Write           (o)     : resetted to FALSE
  pihb->Data[3]	 	(o)	: pointer to allocated memory
  pihb->Dim		(o)	: size of allocated memory

  float initvalue       (i)     : initialization value
  int initflag		(i)	: TRUE initialization of allocated memory
                                  FALSE no initialization

2 HISTORY
  15-Feb-1995   Peter Boesecke creation
  02-Mar-1995   PB DataCtrl
  08-Mar-1995   PB initialization
  28-Feb-1996   PB ImgByteLen==0 returns Data<-NULL and Dim[0], [1], [2]
  12-Jul-2001   PB ActualLoopCount
  28-Aug-2001   PB pihb->Write

----------------------------------------------------------------------------*/
void alloc_image_memory ( CmdBlk *pcb, long int Dim[], ImgHeadBlk *pihb,
                        float initvalue, int initflag, int *pstatus )
{ register long int i;
  float *pData;

  *pstatus=Success;

  pcb->seb.ReportError = NULL;
  pihb->DataCtrl = 0l;
  pihb->ActualLoopCount = 0l;
  pihb->Write = FALSE;
  pihb->ImgByteLen = sizeof(float) * Dim[1] * Dim[2];
  if (pihb->ImgByteLen) {
    /* allocate memory */
    pihb->Data = (float *) malloc (pihb->ImgByteLen);

    if (pihb->Data==NULL) {
      *pstatus=NotEnoughMemoryAvailable; return; }

    pihb->Dim[0] = 2; pihb->Dim[1] = Dim[1]; pihb->Dim[2] = Dim[2];

    if (initflag) {
      pData = pihb->Data;
      for (i=0;i<(Dim[1]*Dim[2]);i++) *(pData++)=initvalue;
      }
    } else {
    /* no memory to allocate */
    pihb->Data = (float *) NULL;
    pihb->Dim[0] = 2; pihb->Dim[1] = Dim[1]; pihb->Dim[2] = Dim[2]; 
    }

  *pstatus = Success;

} /* alloc_image_memory */

/*+++------------------------------------------------------------------------
1 write_image_header

2 DESCRIPTION
  Writes the values of image header block *pihb

  return (void)
  ImgHeadBlk *pihb (i)          : image header block to be written
  long int pihb->ImNum (i)      : image number
  int *pErrorValue (o)          : error value
  int *pstatus (o)              : SAXS status

2 HISTORY
  15-Feb-1995   Peter Boesecke
  18-Mar-2000   PB ImageChain as parameter
  19-Aug-2001   ImageNumber removed from arglist because it 
                is given by pihb->ImNum 
  17-Sep-2001   Orientation: pihb->Header.I
  ---------------------------------------------------------------------------*/
void write_image_header ( int stream, int ImageChain,
                          ImgHeadBlk * pihb, int *pErrorValue,
			  int * pstatus )
{       long int ImageNumber = pihb->ImNum;   /* Image Number */
 
        *pstatus = Success; /* edf_ routines */

        set_header_defaults( pihb );

        /* SaxsDataVersion */
        (void) edf_write_header_line( stream, ImageNumber, ImageChain,
                 KSaxsDataVersion, SAXSDATAVERSION, pErrorValue , pstatus );
        if (*pstatus!=Success) return;

        /* Dummy */
        if (pihb->Dummy.I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                          KDummy, (pihb->Dummy.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* DDummy */
        if (pihb->DDummy.I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                        KDDummy, (pihb->DDummy.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Offset_1 */
        if (pihb->Offset[1].I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KOffset_1, (pihb->Offset[1].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Offset_2 */
        if (pihb->Offset[2].I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KOffset_2, (pihb->Offset[2].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Center_1 */
        if (pihb->Center[1].I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KCenter_1, (pihb->Center[1].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Center_2 */
        if (pihb->Center[2].I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KCenter_2, (pihb->Center[2].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* PixSiz_1 */
        if (pihb->PixSiz[1].I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KPixSiz_1, (pihb->PixSiz[1].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* PixSiz_2 */
        if (pihb->PixSiz[2].I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KPixSiz_2, (pihb->PixSiz[2].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* SampleDistance */
        if (pihb->SampleDistance.I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KSampleDistance, (pihb->SampleDistance.V),
                   pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* WaveLength */
        if (pihb->WaveLength.I) {
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KWaveLength, (pihb->WaveLength.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Title */
        if (pihb->Title.I) {
          (void) edf_write_header_line( stream, ImageNumber, ImageChain,
                   KTitle, (pihb->Title.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Time */
        if (pihb->Time.I) {
          (void) edf_write_header_line( stream, ImageNumber, ImageChain,
                   KTime, (pihb->Time.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Orientation */
        if (pihb->Orientation.I) 
          if ( (pihb->Orientation.V > 1l) || (pihb->Header.I) ) { 
          (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                   KOrientation, (pihb->Orientation.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }
 
        /* History */
        (void) edf_write_header_history ( stream, ImageNumber, ImageChain, 
                                       SaxsHistoryKey, pErrorValue, pstatus );

	*pstatus = Success;

} /* End of write_image_header */

/*----------------------------------------------------------------------------
read_saxs_image

DESCRIPTION
The routine returns only FALSE if the image header was not found. In 
all other cases the return value is TRUE.
DontRead	TRUE  : works as ReadImageHeader (ImgByteLen -> 0)
		FALSE : works as ReadImage
IO_DontReadWrite : header will not be read (only balancing with image block)
IO_DontReadData  : binary data will not be read (ImgByteLen -> 0)
IO_Dummy         : header will not be read (only balancing with image block),
                   binary data will be allocated if DontRead is not set

HISTORY
2000-03-18  PB ImageChain = ib[blkno].Memory.V; 
2000-07-30  PB orientation added
2001-07-10  PB returns also 0 if header was not found due to error 
               FileMustBeOpened 
2001-09-16  PB PassHeader 
----------------------------------------------------------------------------*/
int	read_saxs_image		( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  long int ImageNumber, ImgHeadBlk ihb[], 
                                  int DontRead, int *pstatus)
{
  ImgBlk * pib;
  ImgHeadBlk * pihb;
  int retval;
  int * pErrorValue;
  int stream;
  int ImageChain;
  int simulate = TRUE;
  long *Dim;

  pcb->seb.ReportError = ReportSaxsImageError;
  pErrorValue = &pcb->seb.ExternalError;

  pib  = &ib[blkno];
  pihb = &ihb[blkno];
  stream = ib[blkno].Stream;
  ImageChain = ib[blkno].Memory.V;

  /* read image header and balance with image block,
     if (image header not found) return(FALSE) */

  reset_header_block( pihb );
  if (pcb->TestBit>1) { printf(" -- reset_header_block\n");
    PrintImageHeaderBlock(blkno,pihb); }

  /* Image Number */
  pihb->ImNum = ImageNumber;

  /* ImgByteLen */
  pihb->ImgByteLen=(size_t) NULL; 
  pihb->Data=(float*) NULL; /* no data available */

  *pstatus = Success; /* edf_ routines */

  /* --- Read header */
  if ( !(( IO_DontReadWrite | IO_Dummy ) & pib->StreamOpenMode)  ) {

    /* test stream */
    if ( stream < 0 ) {
      *pstatus = FileMustBeOpened;
      return(FALSE); // image header cannot be read because file is not opened
      }

    if (pcb->PassHeader.V) {
      /* create header key */ 
      sprintf(pihb->Header.V,"i%u",blkno);
      /* read full header */
      edf_new_header( pihb->Header.V );
      pihb->Header.I = TRUE;
      edf_read_header( stream, ImageNumber, ImageChain, 
                       pihb->Header.V, pErrorValue, pstatus );
      if (*pstatus!=Success) return(FALSE);
      }

    /* Dim */
    Dim=pihb->Dim;Dim[0]=2l;
    retval = edf_read_data_dimension( stream, ImageNumber, ImageChain,
                                     &Dim, &pihb->ImgByteLen,
                                     pErrorValue, pstatus );
    if (*pstatus!=Success) { /* image not found or other error*/
      pihb->Dim[0]=pihb->Dim[1]=pihb->Dim[2]=0;
      return(retval);
      }

    /* Dummy */
    if (edf_read_header_float( stream, ImageNumber, ImageChain,
          KDummy, &(pihb->Dummy.V), pErrorValue, pstatus ) ) {
      if (*pstatus!=Success) return(TRUE);
      pihb->Dummy.I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* DDummy */
    if (edf_read_header_float( stream, ImageNumber, ImageChain,
          KDDummy, &(pihb->DDummy.V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->DDummy.I = TRUE;
      } else *pstatus = Success; /* edf_ routines */


    /* Offset_1 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KOffset_1,
          &(pihb->Offset[1].V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->Offset[1].I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* Offset_2 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KOffset_2,
          &(pihb->Offset[2].V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->Offset[2].I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* Center_1 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KCenter_1,
          &(pihb->Center[1].V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->Center[1].I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* Center_2 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KCenter_2,
          &(pihb->Center[2].V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->Center[2].I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* PixSiz_1 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KPixSiz_1,
          &(pihb->PixSiz[1].V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->PixSiz[1].I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* PixSiz_2 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KPixSiz_2,
          &(pihb->PixSiz[2].V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->PixSiz[2].I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* SampleDistance */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KSampleDistance,
          &(pihb->SampleDistance.V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->SampleDistance.I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* WaveLength */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KWaveLength,
          &(pihb->WaveLength.V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->WaveLength.I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* Title */
    if (edf_read_header_line( stream, ImageNumber, ImageChain, KTitle,
         (pihb->Title.V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->Title.I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* Time */
    if (edf_read_header_line( stream, ImageNumber, ImageChain, KTime,
         (pihb->Time.V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->Time.I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* Orientation */
    if (edf_read_header_long( stream, ImageNumber, ImageChain, KOrientation,
          &(pihb->Orientation.V), pErrorValue, pstatus )) {
      if (*pstatus!=Success) return(TRUE);
      pihb->Orientation.I = TRUE;
      } else *pstatus = Success; /* edf_ routines */

    /* History (only for blkno==1) */
    if (blkno==1) 
      if (edf_read_header_history( stream, ImageNumber, ImageChain, 
           SaxsHistoryKey, pErrorValue, pstatus )) {
        if (*pstatus!=Success) return(TRUE);
        } else *pstatus = Success; /* edf_ routines */

    *pstatus = Success;
    } /* if IO_DontReadWrite | IO_Dummy*/

  if (pcb->TestBit>1) { printf(" -- edf_read_header_...\n");
    PrintImageHeaderBlock(blkno,pihb); }

  balance_header_block(ImageNumber,pib,pihb);
  if (pcb->TestBit>1) { printf(" -- balance_header_block\n");
    PrintImageBlock(blkno,pib);
    PrintImageHeaderBlock(blkno,pihb); }

  /* --- Read binary data */
  if ( !( (DontRead) || (IO_DontReadData & pib->StreamOpenMode) || 
          (IO_DontReadWrite  & pib->StreamOpenMode) ) ) {
    /* read image data */
    if (pcb->TestBit>1) printf(" --- read image data ... \n");
    if (IO_Dummy  & pib->StreamOpenMode) { // allocate dummy image 
      if (pcb->TestBit>1) printf(" -- allocate dummy image\n");  
      alloc_image_memory(pcb, pihb->Dim, pihb, pihb->Dummy.V, TRUE, pstatus);
      if (*pstatus!=Success) return(TRUE);
      } else {
      Dim=pihb->Dim;Dim[0]=2l;
      if (pcb->TestBit>1) printf(" -- allocate image\n");
      alloc_image_memory(pcb, Dim, pihb, 0.0, FALSE, pstatus);
      if (*pstatus!=Success) return(TRUE);
      if (pcb->TestBit>1) printf(" -- edf_read_data\n");
      edf_read_data ( stream, ImageNumber, ImageChain,
                      &Dim, &pihb->ImgByteLen, 
                      (void**) &(pihb->Data), MFloat, 
                      pErrorValue, pstatus );
      if (*pstatus!=Success) return(TRUE);
      }
    simulate = FALSE;
    } /* if (!(IO_DontReadData ... ) */

  /* clip, gnaw, rebin, orient and transform image */
  if (pcb->TestBit>1) printf(" --  condition_2d\n");
  condition_2d( pihb,
                pib->MinClip, pib->MaxClip,
                pib->Gnaw[1],  pib->Gnaw[2],
                pib->Bin[1],   pib->Bin[2],
                pib->Factor,  pib->Const,
                pihb->Orientation.V,
                simulate, pcb->TestBit, pstatus );
  if (*pstatus!=Success) return(TRUE);

  pihb->I = TRUE; /* header is valid */

  *pstatus = Success;

  return(TRUE);

} /* read_saxs_image */

/*+++------------------------------------------------------------------------
1 SearchMinMaxImage

2 DESCRIPTION
  Searches the minimum and maximum image number

  return( int ) : (o)   TRUE    : successful
                        FALSE   : not successful
  long int * pMinImage (o)      : minimum image number
  long int * pMaxImage (o)      : maximum image number
  int *pExternalError (o)       : external error value
  int *pstatus (o)		: SAXS status

  If IO_DontReadWrite or IO_Dummy is set, no read is done and the return 
  status is Success and the return value TRUE, *pMinImage -> 1 and 
  *pMaxImage -> 0.
---------------------------------------------------------------------------*/
int SearchMinMaxImage ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                          long int * pMinImage, long int *pMaxImage,
                          int *pstatus)
 { const char * RoutineName = "SearchMinMaxImage";
   int ImageChain = ib[blkno].Memory.V;

   *pstatus = Success; /* edf_ routines */

   if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
   SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );
   if ( ((IO_Dummy | IO_DontReadWrite) & ib[blkno].StreamOpenMode)  ) {
     *pMinImage = 1l; *pMaxImage = 0l; *pstatus = Success; return(TRUE);
     } else {
     return(edf_search_minmax_number(ib[blkno].Stream,ImageChain,
       pMinImage,pMaxImage, &pcb->seb.ExternalError, pstatus) );
     }

 } /* SearchMinMaxImage */

/*+++------------------------------------------------------------------------
1 ReadImageDimension

2 DESCRIPTION
  Returns the image dimensions Dim_1 and Dim_2.
  The routine test whether these dimensions are already available in the
  image block ((ib[blkno].Dim[1].V and ib[blkno].Dim[2].V). If yes, no read
  is done and these values are returned. If at least one of the dimensions
  is missing the missing one is read from the file. 

  return( int ) : (o)   TRUE    : dimensions are returned
                        FALSE   : no dimensions are returned
  CmdBlk * pcb			: command block
  ImgBlk ib[blkno]		: image block
    ib[blkno].Stream (i)
    ib[blkno].StreamOpenMode (i)
  int blkno (i)			: block number
  long int ImageNumber (i)      : image number of requested image
  long int * pDim_1 (o)         : dimension 1
  long int * pDim_2 (o)         : dimension 2
  int *pExternalError (o)       : external error value
  int *pstatus (o)              : SAXS status

  Returns TRUE if Dim_1 and Dim_2 have been read (either from file or
  image block) or if an error has occurred. In all other cases the
  return value is FALSE. If in case of FALSE Dim_1 and Dim_2 are not used
  the *pstatus value can be ignored.

  If IO_DontReadWrite or IO_Dummy is set, no read is done, the return 
  status is Success and the return value FALSE : Dim_1 --> -1, Dim_2 --> -1 .
---------------------------------------------------------------------------*/
int ReadImageDimension ( CmdBlk *pcb, ImgBlk ib[], int blkno,
			 long int ImageNumber,
                         long int * pDim_1, long int *pDim_2,
                         int *pstatus)
 { int retval = FALSE;
   long Tmp[4], *Dim; 
   size_t Ibl;
   const char * RoutineName = "ReadImageDimension";
   int ImageChain = ib[blkno].Memory.V;

   if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
   SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );

   *pstatus = Success;

   *pDim_1 = -1l; *pDim_2 = -1l;
   if ( !((ib[blkno].Dim[1].I) && (ib[blkno].Dim[2].I)) ) {

     if ( !((IO_DontReadWrite | IO_Dummy) & ib[blkno].StreamOpenMode)  ) {
      /* read from file */
       Dim=Tmp; Dim[0]=2l;
       if (retval = edf_read_data_dimension( ib[blkno].Stream,
             ImageNumber, ImageChain, &Dim, &Ibl,
             &pcb->seb.ExternalError, pstatus )) {
	 *pDim_1 = Dim[1]; *pDim_2 = Dim[2];
	 }
       }
     } else retval = TRUE;

   if (ib[blkno].Dim[1].I) *pDim_1 = ib[blkno].Dim[1].V;
   if (ib[blkno].Dim[2].I) *pDim_2 = ib[blkno].Dim[2].V;

   return(retval);

 } /* ReadImageDimension */

/*+++------------------------------------------------------------------------
1 ReadHeaderLine

2 DESCRIPTON
  Read a keyword value
  Return value TRUE if found or error, if not found return FALSE and the
  corresponding error message. In case of FALSE the error message can
  be ignored. The routines returnes only Success, if the keyword has been
  found or IO_DontReadWrite or IO_Dummy is set.
  Can only be used after ReadImage, ReadImageHeader, NewImage.
 
  return( int )			:		
  CmdBlk *pcb (i/o)		: command block
    pcb->seb.ExternalError (o)  : error value
    pcb->ib[blkno].Memory.V (i) : memory number
  int blkno (i)			: image block no
  ImgHeadBlk ihb[]		: image header block
     ihb[blkno].Stream (i)	: i/o stream
     ihb[blkno].ImNum (i)	: image, from which the value is read
  char * keyword (i)		: keyword
  char * Value (o)		: output value (line)
  int *pstatus (o)		: SAXS status

  If IO_DontReadWrite or IO_Dummy is set, no read is done and the return 
  status is Success and the return value FALSE (not found).
  ---------------------------------------------------------------------------*/
int ReadHeaderLine          ( CmdBlk *pcb, int blkno,
                              ImgHeadBlk ihb[], char * keyword,
                              char * Value, int * pstatus )
{  const char * RoutineName = "ReadHeaderLine";
   int ImageChain = pcb->ib[blkno].Memory.V;
   int retval;

   *pstatus = Success; /* edf_ routines */

   if (pcb->TestBit) 
     printf(" %s (blkno = %d, keyword = %s)\n",RoutineName,blkno,keyword);

   SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );

  if ( ((IO_Dummy | IO_DontReadWrite) & ihb[blkno].StreamOpenMode)  ) {
    *pstatus = Success; return(FALSE);
    } else {
    retval = edf_read_header_line ( ihb[blkno].Stream, ihb[blkno].ImNum, 
              ImageChain, keyword, Value, &pcb->seb.ExternalError, pstatus ) ;
    if (pcb->TestBit) if (retval) printf(" %s = %s\n", keyword, Value);
    return( retval );
    }
} /* ReadHeaderLine */

/*+++------------------------------------------------------------------------
1 ReadHeaderFloat

2 DESCRIPTON
  Read a keyword value
  Return value TRUE if found or error, if not found return FALSE and the
  corresponding error message. In case of FALSE the error message can
  be ignored. The routines returnes only Success, if the keyword has been
  found or IO_DontReadWrite or IO_Dummy is set.
  Can only used after ReadImage, ReadImageHeader, NewImage.

  return( int )			:
  CmdBlk *pcb (i/o)             : command block
    pcb->seb.ExternalError (o)  : error value
    pcb->ib[blkno].Memory.V (i) : memory number
  int blkno (i)                 : image block no
  ImgHeadBlk ihb[]              : image header block
     ihb[blkno].Stream (i)      : i/o stream
     ibh[blkno].ImNum (i)       : image, from which the value is read
  char * keyword (i)            : keyword
  float * Value (o)             : output value (float)
  int *pstatus (o)              : SAXS status

  If IO_DontReadWrite or IO_Dummy is set, no read is done and the return 
  status is Success and the return value FALSE (not found).
  ---------------------------------------------------------------------------*/
int ReadHeaderFloat ( CmdBlk *pcb, int blkno,
                      ImgHeadBlk ihb[], char * keyword,
                      float * Value, int * pstatus )
{  const char * RoutineName = "ReadHeaderFloat";
   int ImageChain = pcb->ib[blkno].Memory.V;
   int retval;

   *pstatus = Success; /* edf_ routines */

   if (pcb->TestBit) 
     printf(" %s (blkno = %d, keyword = %s)\n",RoutineName,blkno,keyword);

   SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );

  if ( ((IO_Dummy | IO_DontReadWrite) & ihb[blkno].StreamOpenMode)  ) {
    *pstatus = Success; return(FALSE);
    } else {
    retval = edf_read_header_float ( ihb[blkno].Stream, ihb[blkno].ImNum, 
               ImageChain, keyword, Value, &pcb->seb.ExternalError, pstatus );
    if (pcb->TestBit) if (retval) printf(" %s = %g\n", keyword, *Value);
    return( retval );
    }

} /* ReadHeaderFloat */

/*+++------------------------------------------------------------------------
1 ReadHeaderLong

2 DESCRIPTON
  Read a keyword value
  Return value TRUE if found or error, if not found return FALSE and the
  corresponding error message. In case of FALSE the error message can
  be ignored. The routines returnes only Success, if the keyword has been
  found or IO_DontReadWrite or IO_Dummy is set.
  Can only used after ReadImage, ReadImageHeader, NewImage.

  return( int )
  CmdBlk *pcb (i/o)             : command block
    pcb->seb.ExternalError (o)  : error value
    pcb->ib[blkno].Memory.V (i) : memory number
  int blkno (i)                 : image block no
  ImgHeadBlk ihb[]              : image header block
     ihb[blkno].Stream (i)      : i/o stream
     ihb[blkno].ImNum (i)       : image, from which the value is read
  char * keyword (i)            : keyword
  long int * Value (o)          : output value (long int)
  int *pstatus (o)              : SAXS status

  If IO_DontReadWrite or IO_Dummy is set, no read is done and the return 
  status is Success and the return value FALSE (not found).
  ---------------------------------------------------------------------------*/
int ReadHeaderLong  ( CmdBlk *pcb, int blkno,
                      ImgHeadBlk ihb[], char * keyword,
                      long int * Value, int * pstatus )
{  const char * RoutineName = "ReadHeaderLong";
   int ImageChain = pcb->ib[blkno].Memory.V;
   int retval;

   *pstatus = Success; /* edf_ routines */

   if (pcb->TestBit) 
     printf(" %s (blkno = %d, keyword = %s)\n",RoutineName,blkno,keyword);

   SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );

  if ( ((IO_Dummy | IO_DontReadWrite) & ihb[blkno].StreamOpenMode)  ) {
    *pstatus = Success; return(FALSE);
    } else {
    retval = edf_read_header_long ( ihb[blkno].Stream, ihb[blkno].ImNum, 
               ImageChain, keyword, Value, &pcb->seb.ExternalError, pstatus );
    if (pcb->TestBit) if (retval) printf(" %s = %d\n", keyword, *Value);
    return( retval );
    }

} /* ReadHeaderLong */

/*---------------------------------------------------------------------------
Name
   ReOpenImageFile --- tries to reopen an existing stream

DESCRIPTON
  Tries first to reopen an old stream, if it does not exist it opens a
  new stream.
  int *pstatus           : (o) SAXS status

HISTORY
  06-Jul-2001           PB
  ---------------------------------------------------------------------------*/
int ReOpenImageFile(const char * fname_full, const char * mode,
                    int * pErrorValue, int * pstatus)
{ int stream=-1;

  stream = edf_search_stream (fname_full, mode, pErrorValue, pstatus);
  if ( *pstatus != Success ) return;

  if ( stream < 0 ) // open new stream
    stream = edf_open_data_file(fname_full, mode, pErrorValue, pstatus);

  return( stream );

} // ReOpenImageFile

/*+++------------------------------------------------------------------------
1 OpenImageFile

2 DESCRIPTON
  Opens the data file
  return( void ):
  char * fname           : (i)   file name pattern terminated by comma, e.g.
                                 file%%%%.edf, 1,3,10. Only the first part
                                 before the comma is used. The placeholders
                                 %%%% are replaced by FileNumber. 
  int * ib[blkno].Stream : (o)   stream
  long int * ib[blkno].StreamOpenMode (o) : OpenMode
  long int OpenMode      : (i)   file opening mode, no action for IO_DontOpen
                                 and IO_Dummy
                             IO_New: open file and truncate, stream at start,
                             IO_Old: open file, stream at end
  int * pcb->seb.ExternalError : (o)       error value
  int *pstatus           : (o) SAXS status
2 HISTORY
  12-Jan-1998           PB open_data_file opens now with three
                           different modi: IO_New, IO_Old and nothing of both 
                           (Any)
  21-Dec-1999           PB if the filename is "dummy", do not read or
                           write any data to it
  03-Jul-2001           PB char *fname -> const char *fname
  08-Jul-2001           PB FileNumber, full name is written to image block
  ---------------------------------------------------------------------------*/
void OpenImageFile ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                     const char *fname, long int FileNumber,
                     long int OpenMode, int *pstatus)
{ int * pErrorValue;
  int * pstream;
  static char fname_full[IO_len], fname_pattern[IO_len];
  const char * RoutineName = "OpenImageFile";

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );

  /* set debug mode */
  if (pcb->TestBit>2) edfio_debug( True );

  /* get full file name */
  filename_parameter (fname_pattern,IO_len,fname,0);
  filename_pattern (fname_pattern,IO_len,fname_pattern,FileNumber);
  filename_full(fname_full,IO_len,fname_pattern);

  /* return filename as message in case of an error */
  pcb->seb.Message = fname_full; /* fname_full is static! */

  /* set general block or not */
  edf_general_block( pcb->GeneralBlock.V ); 

  /* set output data value offset type */
  edf_set_datavalueoffset( pcb->OutputDataValueOffset.V );

  /* set output data type */
  edf_set_datatype( pcb->OutputDataType.V );

  /* set byte order of all bsl input files */
  if (pcb->BslInputByteOrder.I) 
    switch ( pcb->BslInputByteOrder.V ) {
      case 1: edf_set_bsl_input_byteorder ( LowByteFirst ); break;
      case 2: edf_set_bsl_input_byteorder ( HighByteFirst ); break;
      default: break;
      }

  /* set maximum line width */
  if (pcb->MaxLineWidth.I)
     edf_set_max_line_width( (unsigned long) MAX2(0,pcb->MaxLineWidth.V) );

  if ( strcmp(fname_full,DummyFileName)==0 ) // dont read/write anything
     OpenMode = OpenMode | IO_Dummy;

  pErrorValue = &pcb->seb.ExternalError;
  pstream = &ib[blkno].Stream;
  ib[blkno].StreamOpenMode = OpenMode;

  if (*pstream>=0) { ;
    *pstatus = FileAlreadyOpened; return;
    }
  *pstatus = Success;

  // Save actual file number and file name in image block
  ib[blkno].FileNumber = FileNumber;
  CopyLine(fname_pattern,ib[blkno].FileName,IO_len,0);

  if ((IO_Dummy | IO_DontOpen) & OpenMode) { *pstatus = Success; return; }

  // open file
  if (IO_Old & OpenMode) {
    if (!filename_exists(fname_full)) {
      *pstatus=FileNotFound; return; 
      } else {
      if (IO_FileProtect & OpenMode) 
        *pstream = ReOpenImageFile(fname_full, "read", pErrorValue, pstatus);
       else
        *pstream = ReOpenImageFile(fname_full, "old", pErrorValue, pstatus);
      if (*pstatus!=Success) return;
      }
    } /* IO_Old */

  else if (IO_New & OpenMode) {
    if (filename_exists(fname_full)) {
      ; /* Rename old file to older version */
      } 
    *pstream = ReOpenImageFile(fname_full, "new", pErrorValue, pstatus);
    if (*pstatus!=Success) return;
    } /* IO_New */
  
  else {
    if (IO_FileProtect & OpenMode)
      *pstream = ReOpenImageFile(fname_full, "read", pErrorValue, pstatus);
     else
      *pstream = ReOpenImageFile(fname_full, "any", pErrorValue, pstatus);
    if (*pstatus!=Success) return;
    } /* Any */

} /* OpenImageFile */

/*---------------------------------------------------------------------------
NAME
  StreamCanBeClosed

DESCRIPTION
  Returns 1 if no other image block uses the same Stream as ib[blkno].
  and if it can be closed, otherwise 0. 
  ---------------------------------------------------------------------------*/
int StreamCanBeClosed( CmdBlk *pcb, ImgBlk ib[], int blkno, int * pstatus)
{ int stream=ib[blkno].Stream;
  int value=1;
  int i;

  if (stream<0) return(0);

  for (i=0;(i<pcb->ImgBlkLen)&&(value);i++) {
    if (i!=blkno) {
      if (stream==ib[i].Stream) value=0;
    }
  }

  return(value);

} // StreamCanBeClosed

/*---------------------------------------------------------------------------
NAME
  AlreadyOpened 

DESCRIPTION
  Returns 1 if the file is already opened otherwise 0.
  ---------------------------------------------------------------------------*/
int AlreadyOpened( CmdBlk *pcb, ImgBlk ib[], int blkno,
                   const char *fname, long int FileNumber, int *pstatus )
{ char fname_pattern[IO_len];

  *pstatus = Success;
  if (ib[blkno].Stream>=0) { // file is opened
    filename_parameter (fname_pattern,IO_len,fname,0);
    filename_pattern (fname_pattern,IO_len,fname_pattern,FileNumber);
    if ( filename_compare( fname_pattern, ib[blkno].FileName ) )
      return(0); else return(1);
  } else { // file is closed 
  return(0); 
  }

} // AlreadyOpened

/*+++------------------------------------------------------------------------
1 CloseImageFile

2 DESCRIPTION
  Closes the data file
  return( void ) :
  long int ib[blkno].StreamOpenMode: (i) file opening mode, no action for
                                     IO_DontOpen or IO_Dummy
  int *pErrorValue  :   error value
  ---------------------------------------------------------------------------*/
void CloseImageFile ( CmdBlk *pcb, ImgBlk ib[], int blkno, int * pstatus)
{ int *pErrorValue;
  const char * RoutineName = "CloseImageFile";

  *pstatus = Success; /* edf_ routines */

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );

  pErrorValue = &pcb->seb.ExternalError;

  if (!((IO_Dummy | IO_DontOpen) & ib[blkno].StreamOpenMode)) {
    // close if no other image blocks uses Stream
    if ( StreamCanBeClosed( pcb, ib, blkno, pstatus) ) {
      if (*pstatus!=Success) return;
      (void) edf_close_data_file(ib[blkno].Stream, pErrorValue, pstatus);
      if (*pstatus!=Success) return;
      }
    } /* if ((!IO_DontOpen) ... */

  ib[blkno].Stream = -1;
  ib[blkno].FileName[0] = (char) NULL;

  *pstatus = Success;

} /* CloseImageFile */

/*+++------------------------------------------------------------------------
1 ReadImage

2 DESCRIPTION
  Reads header and data of image number ImageNumber.

  Read the image data
  return( int )			FALSE : image header not found,
  				  *pstatus != Success
  				TRUE  : image header found or error,
  				  *pstatus = <SaxsStatus>
  CmdBlk *pcb (i)		: command block
    pcb->seb.ExternalError (o)	: <data error value>
  ImgBlk ib[blkno] (i)		: image block
    ib[blkno].StreamOpenMode (i): IO_DontReadData: only image header is read,
                                               no memory allocated,
                                               no image data available
    ib[blkno].Factor (i)	: multiplication factor
    ib[blkno].Const (i)		: addition constant
    ib[blkno].Bin (i)		: rebin factors
  int blkno (i)			: block no of image block and header block
  long int ImageNumber (i)	: number of requested image
  ImgHeadBlk ihb[blkno] (o)	: image header block
    ihb[blkno].I (o)		: TRUE  = header read from file
				  FALSE = no header read
    ihb[blkno].ImgByteLen (o)   : number of bytes allocated
    ihb[blkno].Stream (o)	: copy from Image Block
    ihb[blkno].StreamOpenMode(o): copy from Image Block
    ihb[blkno].ImNum (o)        : current image number (set to ImageNumber)
  int *pstatus (o)		: exit status

2 HISTORY
  27-Nov-1995
  ---------------------------------------------------------------------------*/
int	ReadImage 		( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  long int ImageNumber, ImgHeadBlk ihb[],
				  int *pstatus)
{ const char * RoutineName = "ReadImage";
  
  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );
  return(read_saxs_image(pcb,ib,blkno,ImageNumber,ihb,FALSE,pstatus));
} /* ReadImage */

/*+++------------------------------------------------------------------------
1 ReadImageHeader

2 DESCRIPTION
  Reads header of image number ImageNumber.

  return( int )			FALSE : image header not found,
  				        *pstatus != Success
    				TRUE  : image header found,
  				        *pstatus = <SaxsStatus>
  CmdBlk *pcb (i)		: command block
    pcb->seb.ExternalError (o)	: <data error value>
  ImgBlk ib[blkno] (i)		: image block
    ib[blkno].Bin (i)		: rebin factors
  int blkno (i)			: block no of image block and header block
  long int ImageNumber (i) 	: number of requested image
  ImgHeadBlk ihb[blkno] (o)	: image header block
    ihb[blkno].I                : TRUE  = header read from file
                                  FALSE = no header read
    ihb[blkno].ImgByteLen (o)   : 0 no bytes available
    ihb[blkno].Stream (o)       : copy from Image Block
    ihb[blkno].StreamOpenMode(o): copy from Image Block
    ihb[blkno].ImNum (o)        : current image number (set to ImageNumber)
  int *pstatus (o)              : exit status

2 HISTORY
  27-Nov-1995
  ---------------------------------------------------------------------------*/
int	ReadImageHeader		( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  long int ImageNumber, ImgHeadBlk ihb[],
				  int *pstatus)
{ const char * RoutineName = "ReadImageHeader";

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );
  return(read_saxs_image(pcb,ib,blkno,ImageNumber,ihb,TRUE,pstatus));
} /* ReadImageHeader */

/*+++------------------------------------------------------------------------
1 WriteHeaderLine 
 
2 DESCRIPTION
  Writes a keyword and a line value into the header of the specified image.
 
  Read the image data
  return( void )                :
  CmdBlk *pcb                   : command block
    pcb->seb.ExternalError (o)  : <data error value>
  long int ImageNumber (i)      : output image number
  ImgHeadBlk ihb[blkno]         : image header of current image
            .Stream (i)         : output stream
            .StreamOpenMode (i) :  no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain
  char * keyword                : keyword to write
  char * Value                  : Value to write
  int * pstatus                  : exit status
 
2 HISTORY
  2001-08-19 PB  creation 
----------------------------------------------------------------------------*/ 
void    WriteHeaderLine         ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[], 
                                  char * keyword, char * Value, int * pstatus )
{ ImgBlk * pib;
  ImgHeadBlk * pihb;
 
  int * pErrorValue;
  int stream;
  int ImageChain;
  long int ImageNumber;
  const char * RoutineName = "WriteHeaderLine";
 
  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );
 
  pErrorValue = &pcb->seb.ExternalError;
 
  pib  = &((pcb->ib)[blkno]);
  pihb = &ihb[blkno];
  if (!pihb->I) { // is header initialized?
    *pstatus = HeaderNotInitialized; return; }
  stream = pihb->Stream;
  ImageChain = pib->Memory.V;
  ImageNumber = pihb->ImNum;
 
  *pstatus = Success;
 
  if ((IO_Dummy | IO_DontReadWrite) & pib->StreamOpenMode) {
    *pstatus = Success; return; }
 
  (void) edf_write_header_line( stream, ImageNumber, ImageChain,
                keyword, Value, pErrorValue , pstatus );
  if (*pstatus!=Success) return;

} // WriteHeaderLine 

/*+++------------------------------------------------------------------------
1 WriteHeaderFloat
 
2 DESCRIPTION
  Writes a keyword and a float value into the header of the specified image. 
 
  Read the image data
  return( void )                :
  CmdBlk *pcb                   : command block
    pcb->seb.ExternalError (o)  : <data error value>
  ImgHeadBlk ihb[blkno]         : image header of current image
            .Stream (i)         : output stream
            .StreamOpenMode (i) :  no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain
  char * keyword                : keyword to write
  float Value                   : float value to write
  int * pstatus                 : exit status
 
2 HISTORY
  2001-08-19 PB  creation
----------------------------------------------------------------------------*/
void    WriteHeaderFloat        ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  char * keyword, float Value, int * pstatus )
{ ImgBlk * pib;
  ImgHeadBlk * pihb;
 
  int * pErrorValue;
  int stream;
  int ImageChain;
  long int ImageNumber;
  const char * RoutineName = "WriteHeaderFloat"; 
 
  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );
 
  pErrorValue = &pcb->seb.ExternalError;
 
  pib  = &((pcb->ib)[blkno]);
  pihb = &ihb[blkno];
  if (!pihb->I) { // is header initialized?
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageChain = pib->Memory.V;
  ImageNumber = pihb->ImNum;
 
  *pstatus = Success;
 
  if ((IO_Dummy | IO_DontReadWrite) & pihb->StreamOpenMode) {
    *pstatus = Success; return; }
 
  (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                keyword, Value, pErrorValue , pstatus );
  if (*pstatus!=Success) return;
 
} // WriteHeaderFloat

/*+++------------------------------------------------------------------------
1 WriteHeaderFloat
 
2 DESCRIPTION
  Writes a keyword and a long int value into the header of the specified image.
 
  Read the image data
  return( void )                :
  CmdBlk *pcb                   : command block
    pcb->seb.ExternalError (o)  : <data error value>
  ImgHeadBlk ihb[blkno]         : image header of current image
            .Stream (i)         : output stream
            .StreamOpenMode (i) :  no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain
  char * keyword                : keyword to write
  long int Value                : long int value to write
  int * pstatus                 : exit status
 
2 HISTORY
  2001-08-19 PB  creation
----------------------------------------------------------------------------*/
void    WriteHeaderLong         ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  char * keyword, long int Value, int * pstatus )
{ ImgBlk * pib;
  ImgHeadBlk * pihb;
 
  int * pErrorValue;
  int stream;
  int ImageChain;
  long int ImageNumber;
  const char * RoutineName = "WriteHeaderLong";

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );
 
  pErrorValue = &pcb->seb.ExternalError;
 
  pib  = &((pcb->ib)[blkno]);
  pihb = &ihb[blkno];
  if (!pihb->I) { // is header initialized?
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageChain = pib->Memory.V;
  ImageNumber = pihb->ImNum;
 
  *pstatus = Success;
 
  if ((IO_Dummy | IO_DontReadWrite) & pihb->StreamOpenMode) {
    *pstatus = Success; return; }
 
  (void) edf_write_header_long( stream, ImageNumber, ImageChain,
                keyword, Value, pErrorValue , pstatus );
  if (*pstatus!=Success) return;
 
  *pstatus = Success;
 
} // WriteHeaderLong

/*+++------------------------------------------------------------------------
1 WriteImage

2 DESCRIPTION
  Writes header and float data pdata[pihb->Dim[1],pihb->Dim[2]]
  In case of success pihb->ImNum is set to the ImageNumber.

  Read the image data
  return( void ) 		:

  CmdBlk *pcb                   : command block
    pcb->seb.ExternalError (o)  : <data error value>
  ImgHeadBlk ihb[blkno]         : image header of current image
                .Dim (i)        : Size of array (only [0],[1] and [2])
                .<any other header value> (i)
                .ImNum (o)      : image number to write
                .Data (i)       : data to write
            .Stream (i)         : output stream
            .StreamOpenMode (i) : no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain

  int *pstatus  		: exit status

2 HISTORY
  2000-03-18 PB   ImageChain = pib->Memory.V;
  2000-07-30 PB   orientation added
  2001-08-19 PB   ImageNumber removed, taken from ibh[blkno].ImNum
----------------------------------------------------------------------------*/
void    WriteImage              ( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  ImgHeadBlk ihb[], int *pstatus)
{ ImgBlk * pib;
  ImgHeadBlk * pihb;
  int *  pErrorValue;
  int stream;
  int ImageChain;
  long int ImageNumber;

  long Tmp[4], *Dim;
  size_t   Ibl;
  const char * RoutineName = "WriteImage";

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName,
                NULL, Success, 0, ReportSaxsImageError );

  pErrorValue = &pcb->seb.ExternalError;

  pib  = &ib[blkno];
  pihb = &ihb[blkno];
  if (!pihb->I) {
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageNumber = pihb->ImNum;
  ImageChain = pib->Memory.V;

  *pstatus = Success;

  if ((IO_Dummy | IO_DontReadWrite) & pihb->StreamOpenMode) { 
    *pstatus = Success; return; }

  /* clip, gnaw, rebin, orient and transform image */
  if (pcb->TestBit>1) printf(" --  condition_2d\n");
  condition_2d( pihb,
                pib->MinClip, pib->MaxClip,
                pib->Gnaw[1],  pib->Gnaw[2],
                pib->Bin[1],   pib->Bin[2],
                pib->Factor,  pib->Const,
                edf_raster_inversion( pib->Orientation.V ),
                FALSE, pcb->TestBit, pstatus );
  if (*pstatus!=Success) return;

  if (DUMMYDEFINED(pihb->Dummy.V,pihb->DDummy.V))
    mark_float_2d(pihb->Data,pihb->Dim[1],pihb->Dim[2],
                  pihb->Dummy.V,pihb->DDummy.V);

  /* write image header */
  if (pcb->TestBit>1) printf(" --  write_image_header\n");
  write_image_header ( stream, ImageChain, pihb, pErrorValue, pstatus );
  if (*pstatus!=Success) return;

  /* write image data */
  if (pcb->TestBit>1) printf(" --  edf_write_data\n");
  edf_write_data ( stream, ImageNumber, ImageChain, 
                   pihb->Dim, pihb->Data, MFloat, 
                   pErrorValue, pstatus );
  if (*pstatus!=Success) return;

} /* WriteImage */

/*+++------------------------------------------------------------------------
1 ReportSaxsImageError

2 DESCRIPTION

2 HISTORY
----------------------------------------------------------------------------*/
char * ReportSaxsImageError ( int ExternalError )
{ char * ErrorMsg;
  ErrorMsg = edf_report_data_error(ExternalError);
  return (ErrorMsg);
} /* ReportSaxsImageError */

/*---------------------------------------------------------------------------
1 NewImageHeader

2 DESCRIPTION
  Creates image header with ImageNumber. The file does not need 
  to be opened.

  return( void )
  CmdBlk *pcb (i)		: command block
    pcb->seb.ExternalError (o)	: <data error value>
  ImgBlk ib[blkno] (i)		: image block
                                  only image header will be created, 
                                  no memory will be allocated 
  int blkno (i)			: block no
  long int ImageNumber (i) 	: create image header with this number
  ImgHeadBlk ihb[blkno] (o)	: image header block
				  The header is created from the template
                                  and the image block.
    ihb[blkno].I (o)            : TRUE  = image header is created
                                  FALSE = no image header is not created
    ihb[blkno].Stream (o)       : copy from Image Block
    ihb[blkno].StreamOpenMode(o): copy from Image Block
  ImgHeadBlk * template (i)	: template for header block,
                                  not used if it is the NULL pointer
  int *pstatus (o)		: exit status

  Orientation is not copied from image block

2 HISTORY
  2001-01-11                     PB from NewImage
---------------------------------------------------------------------------*/
void	NewImageHeader           ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  long int ImageNumber, ImgHeadBlk ihb[],
                                  ImgHeadBlk * template, int *pstatus)
{ int ImageChain;
  int header_found;
  const char * RoutineName = "NewImageHeader";
  int * pErrorValue;

  int tmp;

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  filename_full(SaxsImageMessage,IO_len,ib[blkno].FileName); 
  SetSaxsError( &pcb->seb, ModuleName, RoutineName, SaxsImageMessage, 
                Success, 0, NULL );

  pErrorValue = &pcb->seb.ExternalError; 

  *pstatus = Success;

  ImageChain = ib[blkno].Memory.V;

  reset_header_block ( &ihb[blkno] );
  if (pcb->TestBit>1) { printf(" -- reset_header_block\n");
    PrintImageHeaderBlock(blkno,&ihb[blkno]); }

  if (template) {
    template_header_block ( ImageNumber, 0, 0,
                            (size_t) NULL, template->Dim, (float *) NULL,
                            template, &ihb[blkno] );
    if (pcb->TestBit>1) { printf(" -- template_header_block\n");
      PrintImageHeaderBlock(blkno,&ihb[blkno]); }
    }

  // do not take orientation from image block
  tmp = ib[blkno].Orientation.I; ib[blkno].Orientation.I = FALSE;
  balance_header_block( ImageNumber,&ib[blkno],&ihb[blkno]);
  ib[blkno].Orientation.I = tmp;
  if (pcb->TestBit>1) { printf(" -- balance_header_block\n");
    PrintImageBlock(blkno,&ib[blkno]);
    PrintImageHeaderBlock(blkno,&ihb[blkno]); }

  // copy Stream and StreamOpenMode
  ihb[blkno].Stream = ib[blkno].Stream;
  ihb[blkno].StreamOpenMode = ib[blkno].StreamOpenMode;

  /* error, if file is protected */
  if (ihb[blkno].StreamOpenMode & IO_FileProtect) {
    *pstatus = FileProtected; return;}

  if ((IO_Dummy | IO_DontReadWrite) & ihb[blkno].StreamOpenMode) {
    ihb[blkno].I = TRUE; /* header is valid */
    *pstatus = Success; return; }                                               

  /* error, if image exists and is protected */
  if (ib[blkno].Stream>=0) { // do this test only if file is opened
    header_found = edf_test_header( ihb[blkno].Stream, ImageNumber, ImageChain,
           pErrorValue, pstatus );
    if (*pstatus!=Success) return;
 
    if ( header_found && (ihb[blkno].StreamOpenMode & IO_ImageProtect) ) {
         *pstatus = ImageProtected; return;
      } /* edf_test_header */

    ihb[blkno].I = TRUE; /* header is valid for I/O*/

    if (template->Header.I) {
      /* write full header */
      edf_write_header( ihb[blkno].Stream, ImageNumber, ImageChain,
                        template->Header.V, pErrorValue, pstatus );
      if (*pstatus!=Success) return;
      }

    } // if (ib[blkno].Stream>=0)

  *pstatus = Success;

} /* NewImageHeader */

/*---------------------------------------------------------------------------
1 NewImage

2 DESCRIPTION
  Creates image ImageNumber.
  *ppdata[pihb->Dim[1],pihb->Dim[2]].

  Read the image data
  return( void )
  CmdBlk *pcb (i)               : command block
    pcb->seb.ExternalError (o)  : <data error value>
  ImgBlk ib[blkno] (i)          : image block
    ib[blkno].StreamOpenMode (i): IO_DontReadData: only image header will
                                  be created, no memory will be allocated
  int blkno (i)                 : block no
  long int ImageNumber (i)      : create image with this number
  ImgHeadBlk ihb[blkno] (o)     : image header block
                                  The header is created from the template
                                  and the image block.
    ihb[blkno].I (o)            : TRUE  = image header created
                                  FALSE = no image header created
    ihb[blkno].ImgByteLen (o)   : number of bytes allocated
    ihb[blkno].Stream (o)       : copy from Image Block
    ihb[blkno].StreamOpenMode(o): copy from Image Block
  ImgHeadBlk * template (i)     : template for header block
  ib[blkno].StreamOpenMode (i)  :
  int *pstatus (o)              : exit status

  Orientation is not copied from image block

2 HISTORY
---------------------------------------------------------------------------*/
void    NewImage                ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                                  long int ImageNumber, ImgHeadBlk ihb[],
                                  ImgHeadBlk * template, int *pstatus)
{ const char * RoutineName = "NewImage";
  int tmp;

  *pstatus = Success;

   NewImageHeader ( pcb, ib, blkno, ImageNumber, ihb, template, pstatus);
   if (*pstatus!=Success) return;

   if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
   filename_full(SaxsImageMessage,IO_len,ib[blkno].FileName);
   SetSaxsError( &pcb->seb, ModuleName, RoutineName, SaxsImageMessage, 
                 Success, 0, NULL ); 

   if (!(IO_DontReadData & ib[blkno].StreamOpenMode)) {
     if (pcb->TestBit>1) printf(" -- alloc_image_memory\n");
     alloc_image_memory(pcb,ihb[blkno].Dim,&ihb[blkno],ihb[blkno].Dummy.V,\
                        TRUE, pstatus);
     if (*pstatus!=Success) return;
     }
   *pstatus = Success;

} /* NewImage */

/*---------------------------------------------------------------------------
1 free_image_memory

2 DESCRIPTION
  Releases Image memory

  ImgHeadBlk *pihb      (i/o)   : image header block

2 HISTORY
  15-Feb-1995   Peter Boesecke creation
  28-Feb-1996   PB pihb->ImgByteLen = 0l; added
  12-Jan-1998   PB free(xyz) -> {free(xyz); xyz=NULL;}
  10-Jul-2001   PB pihb.I = FALSE; added
  12-Jul-2001   PB set DataCtrl, ActualLoopCount to 0
  28-Aug-2001   PB pihb->Write
----------------------------------------------------------------------------*/
void free_image_memory ( ImgHeadBlk *pihb )
{
  if (pihb->Header.I) {
    /* free header */
    edf_free_header ( pihb->Header.V );
    pihb->Header.I = FALSE;
    }

  if (pihb->Data!=NULL) { free(pihb->Data); pihb->Data=NULL; }
  pihb->Dim[0] = 0; pihb->Dim[1] = 0; pihb->Dim[2] = 0;pihb->Data=(float*) NULL;
  pihb->ImgByteLen = (size_t) NULL;
  pihb->I = FALSE;
  pihb->DataCtrl = 0;
  pihb->ActualLoopCount = 0;
  pihb->Write = FALSE;

} /* free_image_memory */

/*---------------------------------------------------------------------------
1 FreeImage

2 DESCRIPTION
  Releases Image memory

  ImgHeadBlk ihb[blkno]  (i/o)   : image header block

2 HISTORY
  24-Nov-1995   Peter Boesecke creation

----------------------------------------------------------------------------*/
void    FreeImage               ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[] )

{ const char * RoutineName = "FreeImage";

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName, NULL, Success, 0, NULL );

  free_image_memory ( &ihb[blkno] );
} /* FreeImage */

/*---------------------------------------------------------------------------
1 GeoHeaderCopy

2 DESCRIPTION
  Copies geometry keywords from ihb[in] to ihb[out]

  ImgHeadBlk ihb[in]      (i)   : image header block
  ImgHeadBlk ihb[in]      (i)   : image header block

2 HISTORY
  07-Dec-1995   Peter Boesecke creation
----------------------------------------------------------------------------*/
void    GeoHeaderCopy ( ImgHeadBlk ihb[], int blkin, int blkout )
{ geo_header_copy(&ihb[blkin],&ihb[blkout]);
} /* GeoHeaderCopy */

/*---------------------------------------------------------------------------
1 DoNotScaleWith 

2 DESCRIPTION
  Marks image header block ihb[blk] so that it will not be used for scaling

2 HISTORY
  2001-07-07   Peter Boesecke
----------------------------------------------------------------------------*/
void DoNotScaleWith   ( ImgHeadBlk ihb[], int blk )
{ ihb[blk].DoNotScale = TRUE;
} // DoNotScaleWith

/*--------------------------------------------------------------------------*/
/* PrintImageHeaderBlock
/*--------------------------------------------------------------------------*/
void PrintImageHeaderBlock ( int i , ImgHeadBlk * pihb )
{ char VName[IO_len];

  (void) sprintf(VName,"ihb[% d]",i);
  printf("Image Header Block (blkno = %d)\n",i);
  printf("% s.I              = % d\n", VName,pihb->I);
  printf("% s.ImNum          = % d\n", VName,pihb->ImNum);
  printf("% s.StreamOpenMode = %#x\n", VName,pihb->StreamOpenMode);
  printf("% s.DataCtrl       = % d\n", VName,pihb->DataCtrl);
  printf("% s.ActualLoopCount= % d\n", VName,pihb->ActualLoopCount);
  printf("% s.Write          = % s\n", VName,pihb->Write?"TRUE":"FALSE");
  printf("% s.ImgByteLen     = % u\n", VName,pihb->ImgByteLen);
  printf("% s.Dim[1]         = % d\n", VName,pihb->Dim[1]);
  printf("% s.Dim[2]         = % d\n", VName,pihb->Dim[2]);
  printf("% s.Data           = % p\n", VName,pihb->Data);
  printf("% s.DoNotScale     = % d\n", VName,pihb->DoNotScale);
  if (pihb->Header.I)         printf("% s.Header.V       = % s\n",
    VName,pihb->Header.V);
  if (pihb->Dummy.I)          printf("% s.Dummy.V        = % e\n",
    VName,pihb->Dummy.V);
  if (pihb->DDummy.I)         printf("% s.DDummy.V       = % e\n",
    VName,pihb->DDummy.V);
  if (pihb->Offset[1].I)      printf("% s.Offset[1].V    = % e\n",
    VName,pihb->Offset[1].V);
  if (pihb->Offset[2].I)      printf("% s.Offset[2].V    = % e\n",
    VName,pihb->Offset[2].V);
  if (pihb->Center[1].I)      printf("% s.Center[1].V    = % e\n",
    VName,pihb->Center[1].V);
  if (pihb->Center[2].I)      printf("% s.Center[2].V    = % e\n",
    VName,pihb->Center[2].V);
  if (pihb->PixSiz[1].I)      printf("% s.PixSiz[1].V    = % e\n",
    VName,pihb->PixSiz[1].V);
  if (pihb->PixSiz[2].I)      printf("% s.PixSiz[2].V    = % e\n",
    VName,pihb->PixSiz[2].V);
  if (pihb->SampleDistance.I) printf("% s.SampleDistance.V    = % e\n",
    VName,pihb->SampleDistance.V);
  if (pihb->WaveLength.I) printf("% s.WaveLength.V        = % e\n",
    VName,pihb->WaveLength.V);
  if (pihb->Title.I)          printf("% s.Title.V        = % s\n",
    VName,pihb->Title.V);
  if (pihb->Time.I)           printf("% s.Time.V         = % s\n",
    VName,pihb->Time.V);
  if (pihb->Orientation.I)    printf("% s.Orientation.V  = % d\n",
    VName,pihb->Orientation.V);

} /* PrintImageHeaderBlock */

/*---------------------------------------------------------------------------*/

