/*+++
1 SaxsExpression.c

2 PURPOSE
  Reading of float and long integer expressions from strings.
  See PUBLIC functions for detail.

2 CALL
  long int longexpr( char * s, int * pstatus);
  float floatexpr( char * s, int * pstatus);
  double doubleexpr( char * s, int * pstatus);

2 AUTHOR
  1995 Peter Boesecke (PB)

2 HISTORY
  11-Oct-1996 PB extracted from input.c
  13-Oct-1996 PB dpconstant : physical constants and units,
                 units are preceeded by an underscore '_'
  02-Aug-2000 PB dpterm : case '%' added
                 function doubleexpr added
  28-Nov-2000 PB dpfunction : GAMMA_ added
                 dpconstant : km3, ..., m3 added
  04-Dec-2000 PB ->SaxsExpression.c, .h
---*/

/****************************************************************************
*  Include                                                                  *
****************************************************************************/

# include "SaxsExpression.h"

/****************************************************************************
*  Parse Functions                                                          *
****************************************************************************/
double dpconstant( char **ps, int *pstatus);
double dpfunction( char **ps, int level, int *pstatus);
double dpexpression( char **ps, int level, int * pstatus);
double dpfactor( char **ps, int level, int * pstatus);
double dpterm( char **ps, int level, int * pstatus);
long int lvexpression( char **ps, int level, int * pstatus);
long int lvfactor( char **ps, int level, int * pstatus);
long int lvterm( char **ps, int level, int * pstatus);

int isfunction( char * s )
/* A function name starts with a character and contains characters and
   numbers. It ends with a parenthesis '('. This function returns 1
   if the string s starts with a function name */
{
   if (('0'<=*s) && (*s<'9')) return ( 0 ); /* no function */
   while ( (('0'<=*s) && (*s<'9')) || (('a'<=*s) && (*s<'z')) ) s++;
   if (*s=='(') return ( 1 ); else return ( 0 );
} /* isfunction */

double dpconstant( char **ps, int *pstatus)
/* physical constants from: 
   Lawrence Berkeley Laboratory
   University of California
   Berkeley, California 94720
   X-Ray data booklet, second printing, with corrections, April 1996 */
{
   const double gamma = 0.577215664901532861;   
   const double kb = 1.380662e-23;    /* Boltzmann constant (J/K) */
   const double me = 9.109534e-31;    /* electron rest mass (kg) */
   const double mp = 1.6726485e-27;   /* proton rest mass (kg) */
   const double na = 6.022045e23;     /* Avogadro number (1/mol) */
   const double pi = 3.1415926535897932384626;
   const double re = 2.8179380e-15;   /* classical electron radius (m) */
   const double c  = 2.99792458e8;    /* velocity of light (m/s) */
   const double e  = 1.6021892e-19;   /* electron charge (C) */
   const double h  = 6.626176e-34;    /* Planck's number (J*s) */
   const double barn_ = 1e-28;        /* barn (1/m^2) */
   const double amu_ = 1.6605655e-27; /* atomic mass unit (kg) */
   const double km_ = 1e3; const double m_ = 1.0; const double cm_ = 0.01;
   const double mm_ = 1e-3; const double um_ = 1e-6; const double nm_ = 1e-9;

   char * GAMMA_="gamma";
   char * KB_ ="kb"; char * ME_ ="me"; char * MP_ ="mp"; char * NA_= "na";
   char * PI_ ="pi"; char * RE_ ="re";
   char * C_  ="c";  char * E_  ="e";  char * H_ ="h";
   char * BARN_="barn";char * AMU_="amu";char * KEVU_="kev"; char * EVU_="ev";
   char * KM_ ="km"; char * KM2_ ="km2"; char * KM3_ ="km3";
   char * M_  ="m";  char * M2_  ="m2";  char * M3_  ="m3";
   char * CM_ ="cm"; char * CM2_ ="cm2"; char * CM3_ ="cm3";
   char * MM_ ="mm"; char * MM2_ ="mm2"; char * MM3_ ="mm3";
   char * UM_ ="um"; char * UM2_ ="um2"; char * UM3_ ="um3";
   char * NM_ ="nm"; char * NM2_ ="nm2"; char * NM3_ ="nm3";

   double value;

   *pstatus = Success;
   value = 1.0;

       /* --- constants with 5 characters */
         /* Gamma */
              if (!strncmp(*ps,GAMMA_,strlen(GAMMA_))) {
                  value=gamma; *ps=*ps+strlen(GAMMA_);}
       /* --- constants with 2 characters */
         /* kb Boltzmann constant (J/K) */
         else if (!strncmp(*ps,KB_,strlen(KB_))) {
                  value=kb; *ps=*ps+strlen(KB_);}
         /* me electron rest mass (kg) */
         else if (!strncmp(*ps,ME_,strlen(ME_))) {
                  value=me; *ps=*ps+strlen(ME_);}
         /* mp proton rest mass (kg) */
         else if (!strncmp(*ps,MP_,strlen(MP_))) {
                  value=mp; *ps=*ps+strlen(MP_);}
         /* na Avogadro number (1/mol) */
         else if (!strncmp(*ps,NA_,strlen(NA_))) {
                  value=na; *ps=*ps+strlen(NA_);}
         /* pi */
         else if (!strncmp(*ps,PI_,strlen(PI_))) {
                  value=pi; *ps=*ps+strlen(PI_);}
         /* re classical electron radius (m) */
         else if (!strncmp(*ps,RE_,strlen(RE_))) {
                  value=re; *ps=*ps+strlen(RE_);}
      /* --- constants with 1 character */
         /* c velocity of light */
         else if (!strncmp(*ps,C_,strlen(C_))) {
                  value=c; *ps=*ps+strlen(C_);}
         /* e electron charge */
         else if (!strncmp(*ps,E_,strlen(E_))) {
                  value=e; *ps=*ps+strlen(E_);}
         /* h Planck's number (J*s) */
         else if (!strncmp(*ps,H_,strlen(H_))) {
                  value=h; *ps=*ps+strlen(H_);}
      /* --- unit */
         else if (**ps=='_') { /* unit */ (*ps)++;
         /* barn (1/m^2) */
              if (!strncmp(*ps,BARN_,strlen(BARN_))) {
                  value=barn_; *ps=*ps+strlen(BARN_);}
         /* amu atomic mass unit (kg) */
         else if (!strncmp(*ps,AMU_,strlen(AMU_))) {
                  value=amu_; *ps=*ps+strlen(AMU_);}
         /* _keV (J) */
         else if (!strncmp(*ps,KEVU_,strlen(KEVU_))) {
                  value=e*1e3; *ps=*ps+strlen(KEVU_);}
         /* _eV (J) */
         else if (!strncmp(*ps,EVU_,strlen(EVU_))) {
                  value=e; *ps=*ps+strlen(EVU_);}
         /* _KM3 (m^3) */
         else if (!strncmp(*ps,KM3_,strlen(KM3_))) {
                  value=km_*km_*km_; *ps=*ps+strlen(KM3_);}
         /* _CM3 (m^3) */
         else if (!strncmp(*ps,CM3_,strlen(CM3_))) {
                  value=cm_*cm_*cm_; *ps=*ps+strlen(CM3_);}
         /* _MM3 (m^3) */
         else if (!strncmp(*ps,MM3_,strlen(MM3_))) {
                  value=mm_*mm_*mm_; *ps=*ps+strlen(MM3_);}
         /* _UM3 (m^3) */
         else if (!strncmp(*ps,UM3_,strlen(UM3_))) {
                  value=um_*um_*um_; *ps=*ps+strlen(UM3_);}
         /* _NM3 (m^3) */
         else if (!strncmp(*ps,NM3_,strlen(NM3_))) {
                  value=nm_*nm_*nm_; *ps=*ps+strlen(NM3_);}
         /* _M3 (m) */
         else if (!strncmp(*ps,M3_,strlen(M3_))) {
                  value=m_*m_*m_; *ps=*ps+strlen(M3_);}
         /* _KM2 (m^2) */
         else if (!strncmp(*ps,KM2_,strlen(KM2_))) {
                  value=km_*km_; *ps=*ps+strlen(KM2_);}
         /* _CM2 (m^2) */
         else if (!strncmp(*ps,CM2_,strlen(CM2_))) {
                  value=cm_*cm_; *ps=*ps+strlen(CM2_);}
         /* _MM2 (m^2) */
         else if (!strncmp(*ps,MM2_,strlen(MM2_))) {
                  value=mm_*mm_; *ps=*ps+strlen(MM2_);}
         /* _UM2 (m^2) */
         else if (!strncmp(*ps,UM2_,strlen(UM2_))) {
                  value=um_*um_; *ps=*ps+strlen(UM2_);}
         /* _NM2 (m^2) */
         else if (!strncmp(*ps,NM2_,strlen(NM2_))) {
                  value=nm_*nm_; *ps=*ps+strlen(NM2_);}
         /* _M2 (m) */
         else if (!strncmp(*ps,M2_,strlen(M2_))) {
                  value=m_*m_; *ps=*ps+strlen(M2_);}
         /* _KM (m) */
         else if (!strncmp(*ps,KM_,strlen(KM_))) {
                  value=km_; *ps=*ps+strlen(KM_);}
         /* _CM (m) */
         else if (!strncmp(*ps,CM_,strlen(CM_))) {
                  value=cm_; *ps=*ps+strlen(CM_);}
         /* _MM (m) */
         else if (!strncmp(*ps,MM_,strlen(MM_))) {
                  value=mm_; *ps=*ps+strlen(MM_);}
         /* _UM (m) */
         else if (!strncmp(*ps,UM_,strlen(UM_))) {
                  value=um_; *ps=*ps+strlen(UM_);}
         /* _NM (m) */
         else if (!strncmp(*ps,NM_,strlen(NM_))) {
                  value=nm_; *ps=*ps+strlen(NM_);}
         /* _M (m) */
         else if (!strncmp(*ps,M_,strlen(M_))) {
                  value=m_; *ps=*ps+strlen(M_);}

         /* no unit */
         else *pstatus = UnknownUnit;
           }
      /* --- no float constant */
         else *pstatus = NoFloatNumber;

         return( value );

} /* dpconstant */

double dpfunction( char **ps, int level, int *pstatus)
{
   const double pi = 3.1415926535897932384626;
   const double degtorad = pi/180.0;
   const double radtodeg = 180.0/pi;

   char * RAD_="rad("; char * DEG_="deg("; char * PI_ ="pi(";
   char * SIN_="sin("; char * COS_="cos("; char * TAN_="tan(";
   char * ASIN_="asin("; char * ACOS_="acos("; char * ATAN_="atan(";
   char * ATAN2_="atan2("; char * SINH_="sinh("; char * COSH_="cosh(";
   char * TANH_="tanh("; char * FLOOR_="floor("; char * CEIL_="ceil(";
   char * FABS_="abs("; char * EXP_="exp("; char * LOG_="log(";
   char * LOG10_="log10("; char * POW_="pow("; char * SQRT_="sqrt(";
   char * ROUND_="round("; char * GAMMA_="gamma(";

   double argument1, argument2;

   double value;

   *pstatus = Success;
   value = 1.0;

         /* rad-function */
              if (!strncmp(*ps,RAD_,strlen(RAD_))) { *ps+=strlen(RAD_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = degtorad * argument1;
             } /* rad_ */
         /* deg-function */
         else if (!strncmp(*ps,DEG_,strlen(DEG_))) { *ps+=strlen(DEG_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = radtodeg * argument1;
             } /* DEG_ */
         /* pi-function */
         else if (!strncmp(*ps,PI_,strlen(PI_))) { *ps+=strlen(PI_);
             if (*pstatus != Success) return( value );
             value = pi;
             } /* PI_ */
         /* sin-function */
         else if (!strncmp(*ps,SIN_,strlen(SIN_))) { *ps+=strlen(SIN_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = sin( argument1 );
             } /* SIN_ */
         /* cos-function */
         else if (!strncmp(*ps,COS_,strlen(COS_))) { *ps+=strlen(COS_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = cos( argument1 );
             } /* COS_ */
         /* tan-function */
         else if (!strncmp(*ps,TAN_,strlen(TAN_))) { *ps+=strlen(TAN_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = tan( argument1 );
             } /* TAN_ */
         /* asin-function */
         else if (!strncmp(*ps,ASIN_,strlen(ASIN_))) { *ps+=strlen(ASIN_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             if (fabs(argument1)>1) {*pstatus=DomainError; return( value );}
             value = asin( argument1 );
             } /* ASIN_ */
         /* acos-function */
         else if (!strncmp(*ps,ACOS_,strlen(ACOS_))) { *ps+=strlen(ACOS_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             if (fabs(argument1)>1) {*pstatus=DomainError; return( value );}
             value = acos( argument1 );
             } /* ACOS_ */
         /* atan-function */
         else if (!strncmp(*ps,ATAN_,strlen(ATAN_))) { *ps+=strlen(ATAN_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = atan( argument1 );
             } /* ATAN_ */
         /* atan2-function */
         else if (!strncmp(*ps,ATAN2_,strlen(ATAN2_))) { *ps+=strlen(ATAN2_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             if ((**ps)!=',') *pstatus = CommaExpected; else (*ps)++;
             if (*pstatus != Success) return( value );
             argument2 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = atan2( argument1 , argument2 );
             } /* ATAN2_ */
         /* sinh-function */
         else if (!strncmp(*ps,SINH_,strlen(SINH_))) { *ps+=strlen(SINH_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = sinh( argument1 );
             } /* SINH_ */
         /* cosh-function */
         else if (!strncmp(*ps,COSH_,strlen(COSH_))) { *ps+=strlen(COSH_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = cosh( argument1 );
             } /* COSH_ */
         /* tanh-function */
         else if (!strncmp(*ps,TANH_,strlen(TANH_))) { *ps+=strlen(TANH_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = tanh( argument1 );
             } /* TANH_ */
         /* floor-function */
         else if (!strncmp(*ps,FLOOR_,strlen(FLOOR_))) { *ps+=strlen(FLOOR_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = floor( argument1 );
             } /* FLOOR_ */
         /* ceil-function */
         else if (!strncmp(*ps,CEIL_,strlen(CEIL_))) { *ps+=strlen(CEIL_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = ceil( argument1 );
             } /* FLOOR_ */
         /* fabs-function */
         else if (!strncmp(*ps,FABS_,strlen(FABS_))) { *ps+=strlen(FABS_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = fabs( argument1 );
             } /* FABS_ */
         /* exp-function */
         else if (!strncmp(*ps,EXP_,strlen(EXP_))) { *ps+=strlen(EXP_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = exp( argument1 );
             } /* EXP_ */
         /* log-function */
         else if (!strncmp(*ps,LOG_,strlen(LOG_))) { *ps+=strlen(LOG_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             if (argument1<=0.0) {*pstatus=DomainError; return( value );}
             value = log( argument1 );
             } /* LOG_ */
         /* log10-function */
         else if (!strncmp(*ps,LOG10_,strlen(LOG10_))) { *ps+=strlen(LOG10_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             if (argument1<=0.0) {*pstatus=DomainError; return( value );}
             value = log10( argument1 );
             } /* LOG10_ */
         /* pow-function */
         else if (!strncmp(*ps,POW_,strlen(POW_))) { *ps+=strlen(POW_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             if ((**ps)!=',') *pstatus = CommaExpected; else (*ps)++;
             if (*pstatus != Success) return( value );
             argument2 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             if (((floor(argument2+0.5)-argument2)!=0.0)&&(argument1<=0.0 ))
                {*pstatus=DomainError; return( value );}
             if ( (fabs(argument2)<1.0) && (argument1<0.0) )
                {*pstatus=DomainError; return( value );}
             value = pow( argument1 , argument2 );
             } /* POW_ */
         else if (!strncmp(*ps,SQRT_,strlen(SQRT_))) { *ps+=strlen(SQRT_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             if (argument1<0.0) {*pstatus=DomainError; return( value );}
             value = sqrt( argument1 );
             } /* SQRT_ */
         /* round-function */
         else if (!strncmp(*ps,ROUND_,strlen(ROUND_))) { *ps+=strlen(ROUND_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = floor( argument1 + 0.5 );
             } /* ROUND_ */
         else if (!strncmp(*ps,GAMMA_,strlen(GAMMA_))) { *ps+=strlen(GAMMA_);
             argument1 = dpexpression(ps,level+1,pstatus);
             if (*pstatus != Success) return( value );
             value = gamma( argument1 );
             } /* GAMMA_ */
         else { /* unknown function */
             *pstatus = UnknownFloatFunction; return( value );
             }
         if ((**ps)!=')') *pstatus = BadParenthesis; else (*ps)++;
         return( value );

} /* dpfunction */

double dpterm( char **ps, int level, int * pstatus)
{
   const double deps = 1e-38;
   double value;
   double divisor;

   *pstatus = Success;

   value = dpfactor(ps,level,pstatus);
   if (*pstatus!=Success) return ( value );

   while (**ps) {
     switch (**ps) {
       case '*' :
         (*ps)++; value *= dpfactor(ps,level,pstatus); break;
       case '/' :
         (*ps)++; divisor = dpfactor(ps,level,pstatus);
         if ( fabs(divisor) > deps ) value /= divisor;
           else *pstatus = DivByZero;
         break;
       case '%' :
         (*ps)++; divisor = dpfactor(ps,level,pstatus);
         if ( divisor != 0l) 
            value = (double) ((long)floor(value+0.5)%(long)floor(divisor+0.5));
           else *pstatus=DivByZero;
         break;
       default  :
         return ( value );
     } /* switch */
     if (*pstatus!=Success) return ( value );
   } /* while */

   return ( value );

} /* dpterm */

double dpfactor( char **ps, int level, int * pstatus)
{
   char * DOUBLE_ = "(double)";

   double value;

   *pstatus = Success;
   value = 1.0;

     switch (**ps) {
       case '(' :
       /* --- (double) */
         if (!strncmp(*ps,DOUBLE_,strlen(DOUBLE_))) {
             *ps=*ps+strlen(DOUBLE_);
             value=(double) lvfactor(ps,level,pstatus);
             break;}
       /* --- expression */
         (*ps)++; value *= dpexpression(ps,level+1,pstatus);
         if (*pstatus != Success) break;
         if ((**ps)!=')') *pstatus = BadParenthesis; else (*ps)++;
         break;
       default  :
       /* --- number */
         if ( (('0'<=**ps)&&(**ps<='9'))||('.'==**ps) ) value = strtod(*ps,ps);
       /* --- function */
         else if (isfunction(*ps)) {
         value = dpfunction(ps,level,pstatus);
         if (*pstatus != Success) break;
         } /* functions */
       /* --- constant */
         else { value = dpconstant(ps,pstatus);
         if (*pstatus != Success) break;
         } /* constants */
         break;
     } /* switch */
     if (*pstatus!=Success) return ( value );

   return( value );
} /* dpfactor */

double dpexpression( char **ps, int level, int * pstatus)
{  double value;

   *pstatus = Success;
   value = 0.0;
   switch (**ps) {
     case '+':
       (*ps)++; value += dpterm(ps,level,pstatus); break;
     case '-':
       (*ps)++; value -= dpterm(ps,level,pstatus); break;
     default :
       value += dpterm(ps,level,pstatus); break;
   } /* switch */
   if (*pstatus!=Success) return ( value );

   while (**ps) {
     switch (**ps) {
       case '+':
         (*ps)++; value += dpterm(ps,level,pstatus); break;
       case '-':
         (*ps)++; value -= dpterm(ps,level,pstatus); break;
       case ')':
         if (level<=0) *pstatus = BadParenthesis;
         return ( value );
       case ',':
         return( value );
       default :
         *pstatus = ScanError; break;
     } /* switch */
     if (*pstatus!=Success) return ( value );
   } /* while */
   return(value);

} /* dpexpression */

/*--------------------------------------------------------------------------*/
/* doubleexpr : reads an expression of the type
/*              expression  = ["+"|"."] term {"+"|"-" term}
/*               term       = factor {"*"|"/" factor}
/*                factor    = number | function | constant | "(" expression ")"
/*                              | "(double)" lvfactor
/*                 number   = double precision type floating point number
/*                 function = name "(" expression ["," expression] ")"
/*                  name    = "a"|..|"z" {"a"|..|"z"|"0"|..|"9"}
/*                 constant = unit | name
/*                  unit    = "_" name
/*             The input string is collapsed (all blanks and tabs are removed)
/*             and characters are converted to lower case.
/*
/* 	return float value (o) : value of expression
/*      char * s           (i) : input string
/*      int * status       (o) : output status
/*                               Success        : successful conversion
/*                               BadParenthesis : wrong number of parentheses
/*                               NoFloatNumber  : mysterious character found
/*                               DomainError    :
/*                               etc.
/*--------------------------------------------------------------------------*/
PUBLIC double doubleexpr( char * s, int * pstatus)
{
     EditLine(LowerCase | Collapse | UnComment,s);
     return( dpexpression( &s, 0 , pstatus) );

} /* floatexpr */

/*--------------------------------------------------------------------------*/
/* floatexpr : like doubleexpr, but values are returned as float 
/*--------------------------------------------------------------------------*/
PUBLIC float floatexpr( char * s, int * pstatus)
{
     return( (float) doubleexpr( s, pstatus) );

} /* floatexpr */

long int lvterm( char **ps, int level, int * pstatus)
{
   long int value;
   long int divisor;

   *pstatus = Success;

   value = lvfactor(ps,level,pstatus);
   if (*pstatus!=Success) return ( value );

   while (**ps) {
     switch (**ps) {
       case '*' :
         (*ps)++; value *= lvfactor(ps,level,pstatus); break;
       case '/' :
         (*ps)++; divisor = lvfactor(ps,level,pstatus);
         if ( divisor != 0l) value /= divisor;
           else *pstatus=DivByZero;
         break;
       case '%' :
         (*ps)++; divisor = lvfactor(ps,level,pstatus);
         if ( divisor != 0l) value %= divisor;
           else *pstatus=DivByZero;
         break;

       default  :
         return ( value );
     } /* switch */
     if (*pstatus!=Success) return ( value );
   } /* while */

   return ( value );

} /* lvterm */

long int lvfactor( char **ps, int level, int * pstatus)
{
   const double long_max = (double) LONG_MAX;
   const double long_min = (double) LONG_MIN;

   double dpargument;

   long int value;
   char * TRUE_ = "true"; char * FALSE_ = "false";
   char * YES_  = "yes";  char * NO_    = "no";
   char * LONG_ = "(long int)"; char * ROUND_ = "(round)";

   *pstatus = Success;
   value = 1l;

     switch (**ps) {
       case '(' :
       /* --- (long int) */
         if (!strncmp(*ps,LONG_,strlen(LONG_))) {
             *ps=*ps+strlen(LONG_);
             dpargument = dpfactor(ps,level,pstatus);
             if (*pstatus!=Success) break;
             if ((long_min <= dpargument) && (dpargument <= long_max))
                value=(long int) dpargument; else *pstatus = IntegerOverflow;
             break;}
         if (!strncmp(*ps,ROUND_,strlen(ROUND_))) {
             *ps=*ps+strlen(ROUND_);
             dpargument = floor(dpfactor(ps,level,pstatus)+0.5);
             if (*pstatus!=Success) break;
             if ((long_min <= dpargument) && (dpargument <= long_max))
                value=(long int) dpargument; else *pstatus = IntegerOverflow;
             break;}
       /* --- expression */
         (*ps)++; value *= lvexpression(ps,level+1,pstatus);
         if (*pstatus != Success) break;
         if ((**ps)!=')') *pstatus = BadParenthesis; else (*ps)++;
         break;
       default  :
       /* --- number */
         if (('0'<=**ps) && (**ps<='9')) value=strtol(*ps,ps,10);
       /* --- function */
         else if (isfunction(*ps)) {
         dpargument = floor(dpfunction(ps,level,pstatus)+0.5);
         if (*pstatus!=Success) break;
         if ((long_min <= dpargument) && (dpargument <= long_max))
            value=(long int) dpargument; else *pstatus = IntegerOverflow;
         if (*pstatus != Success) break;
         } /* function */
       /* --- constant */
         else if (!strncmp(*ps,TRUE_,strlen(TRUE_))) {
                  value=1l; *ps=*ps+strlen(TRUE_);}
         else if (!strncmp(*ps,FALSE_,strlen(FALSE_))) {
                  value=0l; *ps=*ps+strlen(FALSE_);}
         else if (!strncmp(*ps,YES_,strlen(YES_))) {
                  value=1l; *ps=*ps+strlen(YES_);}
         else if (!strncmp(*ps,NO_,strlen(NO_))) {
                  value=0l; *ps=*ps+strlen(NO_);}
         else *pstatus = NoIntegerNumber;
       /* --- exit */
         break;
     } /* switch */
     if (*pstatus!=Success) return ( value );

   return( value );
} /* lvfactor */

long int lvexpression( char **ps, int level, int * pstatus)
{  long int value;

   *pstatus = Success;
   value = 0l;
   switch (**ps) {
     case '+':
       (*ps)++; value += lvterm(ps,level,pstatus); break;
     case '-':
       (*ps)++; value -= lvterm(ps,level,pstatus); break;
     default :
       value += lvterm(ps,level,pstatus); break;
   } /* switch */
   if (*pstatus!=Success) return ( value );

   while (**ps) {
     switch (**ps) {
       case '+':
         (*ps)++; value += lvterm(ps,level,pstatus); break;
       case '-':
         (*ps)++; value -= lvterm(ps,level,pstatus); break;
       case ')':
         if (level<=0) *pstatus = BadParenthesis;
         return ( value );
       default :
         *pstatus = ScanError; break;
     } /* switch */
     if (*pstatus!=Success) return ( value );
   } /* while */
   return(value);

} /* lvexpression */

/*--------------------------------------------------------------------------*/
/* longexpr : reads an expression of the type
/*              lvexpression  = ["+"|"-"] lvterm {"+"|"-" lvterm}
/*               lvterm       = lvfactor {"*"|"/" lvfactor}
/*                lvfactor    = lvnumber | lvconstant |"(" lvexpression ")"
/*                                 | dpfunction | "(long)" dpfactor
/*                                 | "(round)" dpfactor
/*                 lvnumber   = long integer type number
/*                 lvconstant = "true" | "false" | "yes" | "no"
/*                 dpfunction = see above
/*                 dpfactor   = see above
/*             The input string is collapsed (all blanks and tabs are removed)
/*             and characters are converted to lower case.
/*             The result of dpfunction is rounded to the closest long integer
/*             value.
/*
/* 	return long value  (o) : value of expression
/*      char * s           (i) : input string
/*      int * status       (o) : output status
/*                               Success         : successful conversion
/*                               BadParenthesis  : wrong number of parentheses
/*                               NoIntegerNumber : mysterious character found
/*--------------------------------------------------------------------------*/
PUBLIC long int longexpr( char * s, int * pstatus)
{
     EditLine(LowerCase | Collapse | UnComment,s);
     return( lvexpression( &s, 0 , pstatus) );

} /* longexpr */

/****************************************************************************/
