/*+++
1 SaxsError.c

2 DESCRIPTION
SaxsErrorVersion returns the version string of the module
InitSaxsError has to be called in main before any other error function
              is used.
SetSaxsError  has to be used to set the error block, alternatively
              it can be used to preset the block with ModuleName,
              RoutineName, Message, ReportError, ExternalError and
              status == Success. The block must be valid for
              status != Success.
ReportSaxsError   should be used to display the error message.
ReportSaxsStatus  uses external status variable
ReportInputStatus gives only error messages for status

2 CALL
char *SaxsErrorVersion ( void );
void InitSaxsError( SaxsErrBlk *pseb, int * pstatus );
void SetSaxsError( SaxsErrBlk *pseb, const char * ModuleName,
                   const char * RoutineName, const char * Message,
                   int status, int ExternalError,
                   char *  (*ReportError) (int ) );
void ReportSaxsError( SaxsErrBlk *pseb , int ExitFlag);
void ReportSaxsStatus(int Status, SaxsErrBlk *pseb , int ExitFlag);
void ReportInputStatus(int Status, int ExitFlag);
void PrintSaxsError ( SaxsErrBlk *pseb );

2 HISTORY
  1996-10-13 PB V1.0 Extracted from SaxsInput
  1996-10-15 PB V1.1 SaxsErrorVersion
  2000-01-07 PB V1.3 SizeMismatch, FileWriteError
  2000-03-19 PB V1.4 OutOfRange
  2001-01-01 PB V1.41 
  2001-07-06 PB V1.42 FileAlreadyOpened
  2001-07-10 PB V1.43 FileMustBeOpened
  2001-08-19 PB V1.44 HeaderNotInitialized
---*/

/***************************************************************************/
/* SaxsError.c                                                             */
/***************************************************************************/
# define SE_Version "SaxsError V 1.44 Peter Boesecke 2001-08-19"

# include <errno.h>
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include "SaxsError.h"
# include "edfio.h"

/****************************************************************************
1 SaxsErrorVersion

2 DESCRIPTION
  Returns a pointer to the version string.
  char * SaxsErrorVersion ( void )

2 HISTORY
  14-Oct-1996 PB
****************************************************************************/
char * SaxsErrorVersion ( void )
{ static char * Version = SE_Version;
  return ( Version );
} /* SaxsErrorVersion */

/****************************************************************************
1 ReportInputStatus

2 DESCRIPTION
   int Status        (i) : SAXS status
   int ExitFlag      (i) : 1 : if (Status!=Success) exit;
		           0 : do not exit
2 HISTORY
  11-Oct-1996 PB
****************************************************************************/
void ReportInputStatus(int Status, int ExitFlag)
{
  const char * cout;

  switch (Status) {
        case Failed:
             cout = "Failed: Routine failed"; break;
	case Success:
             cout = "Success: Normal successful operation"; break;
	case Abort:
             cout = "Abort: Input aborted"; break;
	case SkipInput:
             cout = "SkipInput: Input skipped"; break;
	case ScanError:
             cout = "ScanError: Input conversion error"; break;
	case NoDefault:
             cout = "NoDefault: No default possible"; break;
	case NoOption:
             cout = "NoOption: Unknown option"; break;
	case NoFloatNumber:
             cout = "NoFloatNumber: This is not a float number"; break;
	case NoIntegerNumber:
             cout = "NoIntegerNumber: This is not an integer number"; break;
	case BadParenthesis:
             cout = "BadParenthesis: Wrong number of parentheses"; break;
	case DivByZero:
             cout = "DivByZero: Division by zero"; break;
	case NoFlag:
             cout = "NoFlag: No flag (true|false|yes|no)"; break;
	case CommaExpected:
             cout = "CommaExpected"; break;
	case FloatingPointError:
             cout = "FloatingPointError"; break;
	case DomainError:
             cout = "DomainError"; break;
	case IntegerOverflow:
             cout = "IntegerOverflow"; break;
	case UnknownFloatFunction:
             cout = "UnknownFloatFunction"; break;
	case UnknownIntegerFunction:
             cout = "UnknownIntegerFunction"; break;
        case WrongImageNumber:
             cout = "WrongImageNumber"; break;
        case NotEnoughBlocks:
             cout = "NotEnoughBlocks"; break;
        case NotEnoughMemoryAvailable:
             cout = "NotEnoughMemoryAvailable"; break;
        case FileNotFound:
             cout = "FileNotFound"; break;
        case ImageProtected:
             cout = "ImageProtected"; break;
        case FileProtected:
             cout = "FileProtected"; break;
        case FileOpeningError:
             cout = "FileOpeningError"; break;
        case FileAlreadyOpened:
             cout = "FileAlreadyOpened"; break;
        case FileMustBeOpened:
             cout = "FileMustBeOpened"; break;
        case HeaderNotInitialized:
             cout = "HeaderNotInitialized"; break;
        case FilePositioningError:
             cout = "FilePositioningError"; break;
        case FileReadError:
             cout = "FileReadError"; break;
        case FileClosingError:
             cout = "FileClosingError"; break;
        case KeywordMissing:
             cout = "KeywordMissing"; break;
        case UnknownDataType:
             cout = "UnknownDataType"; break;
        case UnknownUnit:
             cout = "UnknownUnit"; break;
        case SizeMismatch:
             cout = "SizeMismatch"; break;
        case FileWriteError:
             cout = "FileWriteError"; break;
        case OutOfRange:
             cout = "OutOfRange"; break;
	case ExternalErrorReport:
             cout = "ExternalErrorReport"; break;
 	default: cout = "..."; break;
 	}
  printf ("Status = %d -- %s\n", Status, cout);
  if (ExitFlag && (Status!=Success)) exit (-1);
} /* ReportInputStatus */

/****************************************************************************
1 ReportSaxsStatus

2 DESCRIPTION
   int Status        (i) : SAXS status
   SaxsErrBlk * pseb (i) : SAXS error block (NULL : not used)
   char * ModuleName (i) : name of module
   char * RoutineName(i) : name of routine
   char * Message    (i) : message string
   int ExternalError (i) : external error number, only valid and used when
                           Status==ExternalErrorReport.
   (char *) (*ReportError) ( int ExternalError ) : error function
                           which returns an error message according to
                           ExternalError, only used when
                           Status==ExternalErrorReport.
   int ExitFlag      (i) : 1 : if (Status!=Success) exit;
		           0 : do not exit

2 HISTORY
  11-Oct-1996 PB
****************************************************************************/
void ReportSaxsStatus(int Status, SaxsErrBlk *pseb , int ExitFlag)
{ if (Status==Success) return;

  printf("\n");

  if (pseb) {
    if (pseb->ModuleName)
      printf("MODULE   : %s\n",pseb->ModuleName);
    if (pseb->RoutineName)
      printf("ROUTINE  : %s\n",pseb->RoutineName);
    if (pseb->Message)
      printf("%s\n",pseb->Message);
    }

  if (Status==ExternalErrorReport) {
    if ((pseb) && (pseb->ReportError))
      printf("%s\n",pseb->ReportError ( pseb->ExternalError ));
     else printf("%s\n",edf_report_data_error ( pseb->ExternalError ));
    }

  ReportInputStatus( Status, ExitFlag );

} /* ReportSaxsStatus */

/****************************************************************************
1 ReportSaxsError

2 DESCRIPTION
   SaxsErrBlk * pseb (i) : SAXS error block (NULL : not used)
   char * ModuleName (i) : name of module
   char * RoutineName(i) : name of routine
   char * Message    (i) : message string
   int *  pstatus    (i) : status
   int ExternalError (i) : external error number, only valid and used when
                           *seb->pstatus==ExternalErrorReport.
   (char *) (*ReportError) ( int ExternalError ) : error function
                           which returns an error message according to
                           ExternalError, only used when
                           *seb->pstatus==ExternalErrorReport.
   int ExitFlag      (i) : 1 : if (*seb->pstatus!=Success) exit;
		           0 : do not exit

2 CALL
  void ReportSaxsError( SaxsErrBlk *pseb , int ExitFlag);


2 HISTORY
  12-Oct-1996 PB
****************************************************************************/
void ReportSaxsError( SaxsErrBlk *pseb , int ExitFlag)
{ ReportSaxsStatus(*pseb->pstatus, pseb , ExitFlag);
} /* ReportSaxsError */

/****************************************************************************
1 InitSaxsError
  Initialisation of SaxsErrBlk with 0s and pointer to status.

2 DESCRIPTION
  Initialisation of SaxsErrBlk with 0s and pointer to status.

2 CALL
  void InitSaxsError( SaxsErrBlk *pseb, int * pstatus );

   SaxsErrBlk *pseb         (o) : SAXS error block

   int * pstatus            (i) : pointer to status variable

   ModuleName               (o) : NULL
   RoutineName              (o) : NULL
   Message	            (o) : NULL
   pstatus                  (o) : pstatus
   ExternalError            (o) : 0
   ReportError              (o) : NULL

2 HISTORY
  12-Oct-1996 PB
****************************************************************************/
void InitSaxsError( SaxsErrBlk *pseb, int * pstatus )
  { pseb->ModuleName    = NULL;
    pseb->RoutineName   = NULL;
    pseb->Message       = NULL;
    pseb->pstatus       = pstatus;
    pseb->ExternalError = 0;
    pseb->ReportError   = NULL;
  } /* InitSaxsError */

/****************************************************************************
1 SetSaxsError

2 DESCRIPTION
  void SetSaxsError( SaxsErrBlk *pseb, const char * ModuleName,
                     const char * RoutineName, const char * Message,
                     int status, int ExternalError,
                     char *  (*ReportError) (int ) );

   SaxsErrBlk *pseb         (o) : SAXS error block
   const char * ModuleName  (i) : pointer to module name
   const char * RoutineName (i) : pointer to routine name
   const char * Message     (i) : pointer to message
   int status               (i) : SAXS status value
   int ExternalError        (i) : external error number
   (char *) (*ReportError) ( int ExternalError ) :
                            pointer to error report function
2 HISTORY
  11-Oct-1996 PB
****************************************************************************/
void SetSaxsError( SaxsErrBlk *pseb, const char * ModuleName,
                   const char * RoutineName, const char * Message,
                   int status, int ExternalError,
                   char *  (*ReportError) (int ) )
{ if (!(pseb)) return;
  if (!(pseb->pstatus)) {
    printf("ERROR: Error block is not initialized\n");exit(-1);
    }
  *pseb->pstatus      = status;
  pseb->ModuleName    = (char *) ModuleName;
  pseb->RoutineName   = (char *) RoutineName;
  pseb->Message       = (char *) Message;
  pseb->ExternalError = ExternalError;
  pseb->ReportError   = ReportError;
} /* SetSaxsError */

/****************************************************************************
1 PrintSaxsError

2 DESCRIPTION
  void PrintSaxsError ( SaxsErrBlk *pseb )

2 HISTORY
  12-Oct-1996 PB
****************************************************************************/
void PrintSaxsError ( SaxsErrBlk *pseb )
  { const char * NullPointer = "(NULL)";
    const char * cout;

    printf("\n");
    if (!(pseb)) { printf("ErrBlk     = %s\n",NullPointer);return; }
    if (!(pseb->ModuleName)) cout=NullPointer; else cout=pseb->ModuleName;
    printf("ModuleName = %s\n",cout);
    if (!(pseb->RoutineName)) cout=NullPointer; else cout=pseb->RoutineName;
    printf("RoutineName = %s\n",cout);
    if (!(pseb->Message)) cout=NullPointer; else cout=pseb->Message;
    printf("Message = %s\n",cout);
    if (!(pseb->pstatus))
      printf("Message = %s (%p)\n",NullPointer,pseb->pstatus);
     else printf("Message = %d (%p)\n",*pseb->pstatus,pseb->pstatus);
    printf("ExternalError = %d\n",pseb->ExternalError);
    if (!(pseb->ReportError)) cout=NullPointer; else cout="Function";
    printf("ReportError = %s (%p)\n",cout,pseb->ReportError);

  }  /* PrintSaxsError */

/****************************************************************************/
