/*+++
1 SaxsDefinition

2 PURPOSE
  Definition of transformation macros and procedures between reference systems.
  All macros may contain several instructions separated by a semicolon. If
  necessary, e.g. in an if-statement, they MUST be put into parentheses:
  if (condition) {ARRAYREF(Off,Ps);}

2 HISTORY
  30-Jan-1995 PB creation
  02-Feb-1995 PB saxs_definitions.h SAXS_DEFINITIONS
  17-Feb-1995 PB saxs file keywords added
  19-Feb-1995 PB raw data keywords added
  24-Feb-1995 PB definition of "Index" changed
  27-Apr-1995 PB dummy macros corrected for dummy == 0
  04-Jun-1995 PB WaveLength0 changed from 1e-10 to 1e-9
                 Definition of saxs reference system changed from
    SAXSREF(Off,Ps,O,P,C,S,W) Off=(O)-(C)+DAI; Ps=((P)/(S))*(WaveLength0/W)
    SAXSREF(Off,Ps,O,P,C,S,W) Off=(O)-(C)+DAI; Ps=((P)/(S))*(_2W0_/W)
                 with _2W0_ = 2 * WaveLength0.
                 In the SAXS reference system the World Value corresponds now
                 to s = 2 Sin(Theta)/Lambda in the approximation for small
                 angles (s -> s Theta/Lambda).
                 To distinguish whether a program uses the new version the
                 version number of imageoption has been set to 4.0 at the
                 same time.
  19-Jun-1995 PB error in definition of s, s-value too big: _2W0_ --> _W0_
                 _2W0_ removed.
                 To distinguish whether a program uses the new version the
                 version number of imageoption has been set to 4.1 at the
                 same time.
  19-Jun-1995 PB definition of IPOLEPS. IPOLEPS is the width of the interval
                 in which no interpolation takes place
  17-Aug-1995 PB definition of INDEX2R
                 from INDEX2R(I,O,P) ((((I)+(O))*(P))+DAI)
                 to INDEX2R(I,O,P) (((I)+(O)+DAI)*(P))
  17-Aug-1995 PB KHScalerDepth added
  18-Aug-1995 PB RELECTRON added
  10-Sep-1995 PB SD_Version added
  12-Nov-1995 PB 2.0 SAXS_PI added
  14-Nov-1995 PB 2.0 NUMBERSTART added, ARRAYSTART 1 -> 1.0
                     REF2USER, USER2REF, AREBIN
                     Documentation added
                     KSaxsDataVersion missing
  23-Nov-1995 PB 2.02 KSaxsDataVersion added
  28-Nov-1995 PB 2.03 DREF2DUSER, DUSER2DREF added
  23-Dec-1995 PB 2.03 FALSE and TRUE added
  26-Jan-1996 PB 2.032 KDetectorPosition added
  27-Feb-1996 PB 2.033 I2OFFSET, R2OFFSET, S2OFFSET, S2PSIZE, R2PSIZE added
                       PSIZE2R, PSIZE2S
  28-May-1996 PB 2.034 UPDATE
  06-Oct-1996 PB 2.035 DUMMY definition corrected from ABS( ( V) - ( D)) < DD
                       to ABS( ( V) - ( D)) <= DD, NODUMMY respectively. Now,
                       DDummy==0 recognizes dummies.
  10-Oct-1996 PB 2.04  saxs_definitions.h renamed to SaxsDefinition.h
  26-Jun-1999 PB 2.05  keyword KTime added to saxs file keywords
  17-Jul-1999 PB 2.06  SAXSDATAVERSION defined here
  23-Jul-1999 PB 2.07  SAXSDATAVERSION 2.0
  09-Nov-1999 PB 2.08  DUMMYDEFINED( D, DD) parameter DD added
  21-Dec-1999 PB 2.09  definition of DUMMYFILENAME
  31-Dec-1999 PB 2.10  definition of reference system BEAM 
  10-Jan-2000 PB 2.11  definition of reference system CENTER, NORMAL 
                       NORMAL replaces BEAM
  30-Jul-2000 PB 2.12  CSWAP, KOrientation
  31-Aug-2000 PB 2.13  definition of KValueI0, KValueI1, KValueAnode, KValueTime
  10-Nov-2000 PB 2.14  Orientation -> RasterOrientation
  27-Nov-2000 PB 2.15  SETCTRDEF changed from (dim+DAI)*0.5 to 
                        (dim-1)*0.5+ArrayStart
  19-Apr-2001 PB 2.16  definition of WAVENUMBER(W) and suppression of _W0_ 

---*/
/****************************************************************************
* SaxsDefinition.h
****************************************************************************/

#ifndef SAXS_DEFINITIONS

/* version */
# define SD_Version "SaxsDefinition V 2.16 Peter Boesecke 2001-04-19"
# define SAXSDATAVERSION    "2.0"

/* general */
# define DUMMYFILENAME          "/dev/null"

/* saxs file keywords */
# define KSaxsDataVersion       "SaxsDataVersion"
# define KImage                 "Image"
# define KError                 "Image_error"
# define KGeneral               "General"
# define KDummy                 "Dummy"
# define KDDummy                "DDummy"
# define KOffset_1              "Offset_1"
# define KOffset_2              "Offset_2"
# define KCenter_1              "Center_1"
# define KCenter_2              "Center_2"
# define KPixSiz_1              "Psize_1"
# define KPixSiz_2              "Psize_2"
# define KSampleDistance        "SampleDistance"
# define KWaveLength            "WaveLength"
# define KTitle                 "Title"
# define KTime                  "Time"
# define KOrientation           "RasterOrientation"

/* additional raw data keywords */
# define KHScalerLen    "HS32Len"
# define KHScaler       "HS32"
# define KHScalerDepth  "HS32Depth"
# define KHC            "C"
# define KHN            "N"
# define KHZ            "Z"
# define KHF            "F"
# define KHFrame        "HMFrame"
# define KHFile         "HMFile"
# define KDetectorPosition "DetectorPosition"  
# define KDetectorType  "DetectorName"
# define KExperimentInfo "ExperimentInfo"
# define KMachineInfo   "MachineInfo"
# define KOpticsInfo    "OpticsInfo"
# define KStationInfo   "StationInfo"
# define KProposalInfo  "Proposal"
# define KHFraming      "HMFraming"
# define KHStartTime    "HMStartTime"
# define KHDeltaTime    "HMDeltaTime"
# define KHScalerI0     "HSI0"
# define KHScalerI1     "HSI1"
# define KHScalerAnode  "HSAnode"
# define KHScalerTime   "HSTime"
# define KHScalerI0S    "HSI0S"
# define KHScalerI1S    "HSI1S"
# define KHScalerAnodeS "HSAnodeS"
# define KHScalerTimeS  "HSTimeS"
# define KSubTitle      "SubTitle"
# define KValueI0       "Intensity0"
# define KValueI1       "Intensity1"
# define KValueAnode    "Anode"
# define KValueTime     "ExposureTime"
/* ---------------------------------------------------------------------------
  Physical Constants
  - RELECTRON classical electron radius in meter
---------------------------------------------------------------------------*/
# define RELECTRON	2.81794e-15
/* ---------------------------------------------------------------------------
  Mathematical Constants
  - SAXS_PI pi
---------------------------------------------------------------------------*/
# define SAXS_PI	3.1415926535897932384626
/* ---------------------------------------------------------------------------
  General Constants
  - TRUE  1,  - FALSE 0
---------------------------------------------------------------------------*/
# ifndef FALSE
#   define FALSE 0
# endif
# ifndef TRUE
#   define TRUE  1
# endif
/*---------------------------------------------------------------------------
 The constant IPOLEPS defines the interval around an array element in which
 no interpolation takes place.
---------------------------------------------------------------------------*/
# define IPOLEPS	1e-4
/*---------------------------------------------------------------------------
 The following macros calcuate the pointers to float array elements.
 - ABSPTR    calculates the pointer to the array element (column=i_1, row=i_2)
             from the base Data of the array.
 The next three macros are used for interpolation between neighbouring data
 point.
 -NEXTCOL    returns the pointer to the element in the next column
             of the same row
 -NEXTROW    returns the pointer to the element in the next row
             of the same column
 -NEXTCOLROW returns the pointer to the element in the next column
             of the next row.
 Usage:
 float Data[Dim_1,Dim_2] : float array of size Dim_1,Dim_2;
 int i_1,i_2 : array indices
 float *pdata, *pvalnxtcoli, *pvalnxtrow, *pvalnxtcolrow;
 pdata         = ABSPTR(Data,Dim_1,Dim_2,i_1,i_2);
 pvalnxtcol    = NEXTCOL(pdata,Dim_1,Dim_2);
 pvalnxtrow    = NEXTROW(pdata,Dim_1,Dim_2);
 pvalnxtcolrow = NEXTCOLROW(pdata,Dim_1,Dim_2);
---------------------------------------------------------------------------*/
/* #define PTRFA(A,D1,D2,I1,I2) (A)+(((I1)*(D2))+(I2)) */
#define ABSPTR(A,D1,D2,I1,I2) (A)+((I1)+((I2)*(D1)))
#define NEXTCOL(pA,D1,D2) (pA)+1
#define NEXTROW(pA,D1,D2) (pA)+(D1)
#define NEXTCOLROW(pA,D1,D2) (pA)+(1+(D1))
/*---------------------------------------------------------------------------
 The following macro calculates the absolute value of the argument
 Usage:
 absolut = ABS(-3.2);
---------------------------------------------------------------------------*/
#define ABS( x) ( ( x)> 0 ? ( x) : -( x) )
/*---------------------------------------------------------------------------
 The following macros calculate the maximum and minimum of the arguments
 Usage:
 minimum = MIN2(-3.2, 5.7);
 maximum = MAX2( 3.2, 5.9);
 minimum = MIN3( 2.2, 4.7, 0.3 );
 maximum = MAX3( 1.0, 3.0, -3.54 );
 minimum = MIN4( 2.2, 4.7, 0.3 , 4.5 );
 maximum = MAX4( 1.0, 3.0, -3.54 , 8.9 );
---------------------------------------------------------------------------*/
#define MAX2( x1, x2) ( ( x1)>( x2) ? ( x1) : ( x2) )
#define MIN2( n1, n2) ( ( n1)<( n2) ? ( n1) : ( n2) )
#define MAX3( y1, y2, y3) ( MAX2( MAX2( ( y1), ( y2) ), ( y3) ) )
#define MIN3( y1, y2, y3) ( MIN2( MIN2( ( y1), ( y2) ), ( y3) ) )
#define MAX4( z1, z2, z3, z4 ) MAX2( MAX2( ( z1), ( z2)), MAX2( ( z3), ( z4)) )
#define MIN4( z1, z2, z3, z4 ) MIN2( MIN2( ( z1), ( z2)), MIN2( ( z3), ( z4)) )
/*---------------------------------------------------------------------------
 Conventions
 -----------
  pixel index      : Machine dependent numbering of pixels inside an array.
                     The machine dependent numbering starts with INDEXSTART.
  pixel number     : Machine independent numbering of a pixel.
                     The machine indedpendend numbering starts with NUMBERSTART.
  pixel coordinate : Machine independent coordinate of a pixel array.
                     ARRAYSTART is the pixel coordinate of the
                     center of the pixel with the index INDEXSTART.
  world coordinate : Machine independent coordinate of a pixel array. The
                     actually used world coordinates are chosen with the
                     reference system. World coordinates are defined as
                     affine transformations of pixel coordinates:
                     world coordinate = (pixel coordinate + OFFSET) * Ps

  A pixel is covering the interval [pixel number - 0.5 .. pixel number + 0.5].
  Binning does not change world coordinates.

 Coordinate Systems
 ------------------
 The positioning of several pixel arrays to each other is chosen by the
 reference system. All basic lengths (pixel size, wave length,
 sample distance etc.) are given in meters. Four different systems are used:

 ARRAY  coordinate = pixel coordinate
 IMAGE  coordinate = array coordinate + offset
 CENTER coordinate = image coordinate - center
 REAL   coordinate = image coordinate * pixel size
 NORMAL coordinate = (image coordinate - center) * pixel size
 SAXS   coordinate = (image coordinate - center) *
                      (pixel size/sample distance) * (WaveLength0/wave length)

 The principal coordinate system is IMAGE.

 REFERENCE system : coordinate system that is used for calculations
 USER system      : coordinate system that is used for keyboard I/O by the
                    user.

 Description of macros
 ---------------------
 Only the following macros and constants should be used for transformations
 between the different coordinate systems.

 INDEXSTART  = lowest pixel index (machine dependent)
 NUMBERSTART = lowest pixel number (machine independent)
 ARRAYSTART  = pixel coordinate of center of pixel INDEXSTART
 LOWERBORDER = distance between pixel center and its lower border
 DAI         = ARRAYSTART - INDEXSTART = Difference (ARRAYSTART, INDEXSTART)
 WaveLength0 = reference wavelength in meters used in SAXS reference system

 WAVENUMBER
  Calculates the wavenumber as it is used for reference system coordinates

 Example
  Lower left corner has coordinates (0.5,0.5):
   INDEXSTART 0, ARRAYSTART 1.0, LOWERBORDER (-0.5), DAI 1.0
  Lower left corner has coordinates (0.0,0.0):
   INDEXSTART 0, ARRAYSTART 0.5, LOWERBORDER (-0.5), DAI 0.5 

 The world coordinates of the array boundaries are:
  lower boundary = WORLD(INDEXSTART+LOWERBORDER,Off,Ps);
  upper boundary = WORLD(INDEXSTART+DIM+LOWERBORDER,Off,Ps);

 The world coordinates of the first and the last column (row) of an array are:
  lower column = WORLD(INDEXSTART,Off,Ps);
  upper column = WORLD(INDEXSTART+DIM-1,Off,Ps);

 IDX
  Calculation of the integer part and the rest of a float pixel index.
  IDX calculates to the float value f the closest integer number I less
  than f and the difference R = f - (float) I.
 Usage :
  float f; int I; float R;
  IDX( f, I, R );

 ARRAYREF, IMAGEREF, CENTERREF, REALREF, NORMALREF, SAXSREF, WORLD, INDEX
  The preprocessor macros ARRAYREF, IMAGEREF, CENTERREF, REALREF, NORMALREF 
  and SAXSREF calculate offsets and pixel sizes to allow a direct affin 
  transformation between pixel indices and world coordinates. The 
  transformations are done with WORLD and INDEX. The calculated internal 
  offsets (Off) and pixel sizes (Ps) must not be confused with the basic 
  offsets and pixel sizes which are given in pixel coordinates and meters.

 Usage (transformation of coordinates) :
  float Off, Ps;
  float Offset, Psize, Center, SampleDistance, WaveLength;
  float IIndex;
  float WArray, WImage, WCenter, WReal, WNormal, WSaxs;
  {ARRAYREF(Off,Ps);}
  WArray = WORLD( IIndex, Off, Ps );
  IIndex = INDEX( WArray, Off, Ps );
  {IMAGEREF(Off,Ps,Offset);}
  WImage = WORLD( IIndex, Off, Ps );
  IIndex = INDEX( WImage, Off, Ps );
  {CENTERREF(Off,Ps,Offset,Center);}
  WCenter = WORLD( IIndex, Off, Ps );
  IIndex = INDEX( WCenter, Off, Ps );
  {REALREF(Off,Ps,Offset,Psize);}
  WReal = WORLD( IIndex, Off, Ps );
  IIndex = INDEX( WReal, Off, Ps );
  {NORMALREF(Off,Ps,Offset,Psize,Center);}
  WNormal = WORLD( IIndex, Off, Ps );
  IIndex = INDEX( WNormal, Off, Ps );
  {SAXSREF(Off,Ps,Offset,Psize,Center,SampleDistance,WaveLength);}
  WSaxs = WORLD( IIndex, Off, Ps );
  IIndex = INDEX( WSaxs, Off, Ps );

 A2INDEX, I2INDEX, R2INDEX, C2INDEX, S2INDEX, INDEX2A, INDEX2I, INDEX2R, 
  INDEX2C, INDEX2S
  The preprocessor macros A2INDEX, I2INDEX, R2INDEX, C2INDEX and S2INDEX 
  transform a world coordinate W directly directly into a pixel index IIndex.
  INDEX2A, INDEX2I, INDEX2R, INDEX2C and INDEX2S transform a pixel index 
  directly into a world coordinate W.

 Usage (direct transformation of coordinates) :
  float Offset, Psize, Center, SampleDistance, WaveLength;
  float IIndex;
  float WArray, WImage, WCenter, WReal, WNormal, WSaxs;
  IIndex  = A2INDEX(WArray);
  IIndex  = I2INDEX(WImage,Offset);
  IIndex  = C2INDEX(WCenter,Offset,Center);
  IIndex  = R2INDEX(WReal,Offset,Psize);
  IIndex  = N2INDEX(WNormal,Offset,Psize,Center);
  IIndex  = S2INDEX(WSaxs,Offset,Psize,Center,SampleDistance,WaveLength);
  WArray  = INDEX2A(IIndex);
  WImage  = INDEX2I(IIndex,Offset);
  WCenter = INDEX2C(IIndex,Offset,Center);
  WReal   = INDEX2R(IIndex,Offset,Psize);
  WNormal = INDEX2N(IIndex,Offset,Psize,Center);
  WSaxs   = INDEX2S(IIndex,Offset,Psize,Center,SampleDistance,WaveLength);

 REF2USER, USER2REF
  Transformation between a user system coordinate and
  a reference system coordinate

 Usage (transformation between user system and reference system ) :
  float ROff, RPs, UOff, UPs;
  float Offset, Psize, Center, SampleDistance, WaveLength;
  float RW, UW;

  {IMAGEREF(UOff,UPs,Offset);}
  {SAXSREF(ROff,RPs,Offset,Psize,Center,SampleDistance,WaveLength);}
  UW = REF2USER(RW,ROff,RPs,UOff,UPs);
  RW = USER2REF(UW,ROff,RPs,UOff,UPs);

  etc.

 AREBIN
  Calculation of new offset, pixel size and center coordinate after a
  binning by factor B

 Usage (binning) :
  AREBIN(Offset,PSize,Center,B)

 I2OFFSET, C2OFFSET, R2OFFSET, N2OFFSET, S2OFFSET
  Calculation of the offset value. The input value must be the coordinate 
  of the lower edge of the region or pixel (Image, Center, Real, 
  Normal and Saxs). An offset cannot be calculated for Array.
  
 Usage : 
  Offset = I2OFFSET(WImage);
  Offset = C2OFFSET(WNormal,Center);
  Offset = R2OFFSET(WReal,PSize);
  Offset = N2OFFSET(WNormal,PSize,Center);
  Offset = S2OFFSET(WSaxs,PSize,Center,SampleDistance,WaveLength);

 R2PSIZE, N2PSize, S2PSIZE
  Calculation of the pixel size from a distance in Real coordinates,
  Normal coordinates and Saxs coordinates.

 Usage :
  PSize = R2PSize(WRealDistance);
  PSize = N2PSize(WNormalDistance);
  PSize = S2PSize(WSaxsDistance,SampleDistance,WaveLength);

 CSWAP
  Calculation of a new center coordinate after a swap of the data ARRAY. 
  The new center coordinate is calculated in such a way that it points to 
  the same pixel as before the swap. Offset, PSize and Dim do not change
  when the array is swapped.

 Usage:
  Center = CSWAP(Center,Offset,Dim);

----------------------------------------------------------------------------*/
/* array specifications */
# define INDEXSTART  0
# define NUMBERSTART 1
// define ARRAYSTART  1.0 +++++++++++++++++++
# define ARRAYSTART  0.5 
# define LOWERBORDER (-0.5)
// define DAI         1.0 +++++++++++++++++++
# define DAI         0.5 
# define WaveLength0 1e-9
# define WAVENUMBER(W) (WaveLength0/(W)) 
/* transformation of coordinates*/
# define IDX(f,I,R) ((f)>=0)?((I)=(int)(f)):((I)=((int)(f))-1);(R)=(f)-(float)(I) 
# define WORLD( I, Off, Ps) (((I)+(Off))*(Ps))
# define INDEX( W, Off, Ps) (((W)/(Ps))-(Off))
# define ARRAYREF(Off,Ps) Off=DAI; Ps=1.0
# define IMAGEREF(Off,Ps,O) Off=(O)+DAI; Ps=1.0
# define CENTERREF(Off,Ps,O,C) Off=(O)-(C)+DAI; Ps=1.0;
# define REALREF(Off,Ps,O,P) Off=(O)+DAI; Ps=(P)
# define NORMALREF(Off,Ps,O,P,C) Off=(O)-(C)+DAI; Ps=(P);
# define SAXSREF(Off,Ps,O,P,C,S,W) Off=(O)-(C)+DAI; Ps=((P)/(S))*WAVENUMBER(W)
/* direct transformations */
# define A2INDEX(I) ((I)-DAI)
# define I2INDEX(I,O) ((I)-(O)-DAI)
# define C2INDEX(I,O,C) I2INDEX(I,(O)+(C))
# define R2INDEX(I,O,P) ((((I)/(P))-(O))-DAI)
# define N2INDEX(I,O,P,C) R2INDEX((I),(O)+(C),(P))
# define S2INDEX(I,O,P,C,S,W) R2INDEX((I),(O)+(C),((P)*(S))*WAVENUMBER(W))
# define INDEX2A(I) ((I)+DAI)
# define INDEX2I(I,O) ((I)+(O)+DAI)
# define INDEX2C(I,O) ((I)+(O)+DAI)
# define INDEX2R(I,O,P) (((I)+(O)+DAI)*(P))
# define INDEX2N(I,O,P,C) INDEX2R((I),(O)-(C),(P))
# define INDEX2S(I,O,P,C,S,W) INDEX2R((I),(O)-(C),((P)/(S))*WAVENUMBER(W))
/* transformation of coordinates between user system and reference system */
# define REF2USER(RW,ROff,RPs,UOff,UPs) WORLD(INDEX(RW,ROff,RPs),UOff,UPs)
# define USER2REF(UW,ROff,RPs,UOff,UPs) WORLD(INDEX(UW,UOff,UPs),ROff,RPs)
/* transformation of distances between user system and reference system */
# define DREF2DUSER(DRW,RPs,UPs) ((DRW) * ((UPs)/(RPs)))
# define DUSER2DREF(DUW,RPs,UPs) ((DUW) * ((RPs)/(UPs)))
/* binning */
# define AREBIN(O,P,C,B) \
          O=( ((ARRAYSTART+LOWERBORDER)*(1.0-MAX2(1,B))+(O))/MAX2(1,B) ); \
          P=( (P)*MAX2(1,B) ); C=( (C)/MAX2(1,B) )
/* calculation of Offset */
# define I2OFFSET(R) ((R) - (ARRAYSTART + LOWERBORDER) ) 
# define C2OFFSET(R,C) ((R) + (C) - (ARRAYSTART + LOWERBORDER) )
# define R2OFFSET(R,P) ((R)/(P) - (ARRAYSTART + LOWERBORDER) )
# define N2OFFSET(R,P,C) (((R)/(P)) + (C) - (ARRAYSTART + LOWERBORDER) )
# define S2OFFSET(R,P,C,S,W) \
          ( (((R)/(P))*(S)*WAVENUMBER(W)) + (C) - (ARRAYSTART + LOWERBORDER) )
/* calculation of PSize */
# define R2PSIZE(D) (D)
# define N2PSIZE(D) R2PSIZE(D)
# define S2PSIZE(D,S,W) ( ((D) * (S)) * WAVENUMBER(W) )
# define PSIZE2R(P) (P)
# define PSIZE2N(P) PSIZE2R(P)
# define PSIZE2S(P,S,W) ( ((P) / (S)) * WAVENUMBER(W) )
/* calculation of new center after swapping the data array */
# define CSWAP(C,O,D) ( (O) + (O) + (D) - (C) )
/*----------------------------------------------------------------------------
Dummy Value Definitions
MinDDummy minimum possible value for DDummy
AccDummy relative accuracy
DDSET(Dummy) calculation of a good DDummy value to Dummy
DUMMY(Value, Dummy, DDummy) TRUE if value is a Dummy
NODUMMY(Value, Dummy, DDummy) TRUE if value is not a Dummy
UPDATE(Destination,Value,Dummy,DDummy) adds Value to Destination if not Dummy
Usage:
DDummy = DDSET(Dummy);
if DUMMYDEFINED (Dummy, DDummy) ...
if DUMMY(Value, Dummy, DDummy) { do something } else { do something else };
if NODUMMY(Value, Dummy, DDummy) { do something } else { do something else };
UPDATE( Destination, Value, Dummy, DDummy);
---------------------------------------------------------------------------*/
# define MinDDummy 0.1
# define AccDummy  1e-5
# define DDSET(D) MAX2( MinDDummy, ABS( D) * AccDummy )
# define DUMMYDEFINED( D, DD) ((ABS( D) > MAX2( DD, MinDDummy) ) ? 1 : 0)
# define NODUMMYDEFINED(D, DD) ((ABS( D) > MAX2( DD, MinDDummy ) ) ? 0 : 1)
# define DUMMY( V, D, DD) ((ABS( ( V) - ( D)) <= DD) && DUMMYDEFINED( D, DD))
# define NODUMMY( V, D, DD) ((ABS( ( V) - ( D)) > DD) || NODUMMYDEFINED( D, DD))
# define UPDATE( DE, V, D, DD) if (DUMMY(DE,(D),(DD))) DE=( V); else DE+=( V)
/*-------------------------------------------------------------------------
Center Value Definitions
SETCTRDEF ( Dim ) calculation of a default center point to dimension Dim
Usage:
Center_1 = SETCTRDEF( Dim[1] );
Center_2 = SETCTRDEF( Dim[2] );
---------------------------------------------------------------------------*/
//# define SETCTRDEF(D ) ( ( (float) (D ) + DAI ) * 0.5 )
# define SETCTRDEF(D ) ( ( (float) (D ) - 1 ) * 0.5 + ARRAYSTART )
/*-------------------------------------------------------------------------*/
# define SAXS_DEFINITIONS
#endif /* SAXS_DEFINITIONS */

