"""
Py4datAsciiPlugin.py
Plugin which allows to save in one ascii file (or in several files)
eather a graph or a serie of azymuthal integration,
once the user has define a ROI on the AzimInt window.
If the fit option has been used, the fitting results are also calculated for each image 
and saved in the ascii file.
The peak tracking option available through the main window menu of py4dat
is taken into account here to adapt the ROI and display the results if required.
"""

from qt import *
import string,time,types,Numeric

from PyDisWindow import Plugin, ViewWindow
from PyDVT.ImageView import ImageView, ColormapFilter
from PyDVT.EdfFileData import EdfFileData
from MccdFileData import MccdFileData
from PyDVT.FitFilter import FitFilter
from PyDVT.GraphView import GraphView
import PyDVT.DataSelection as DataSelection
from AzimIntFilter import AzimIntFilter
from IntegratePhiFilter import IntegratePhiFilter
from AddConstantFilter import AddConstantFilter
from MultiplyByConstantFilter import MultiplyByConstantFilter
from SubstractBackgroundFilter import SubstractBackgroundFilter
from LeftRightFlipFilter import LeftRightFlipFilter
from TopBottomFlipFilter import TopBottomFlipFilter
from AsciiParametersDialog import AsciiParametersDialog
import Py4dat2DTransformPlugin

__version__=  '1.0.0'
__author__ =  'Anne-Cecile Gendrin'
__date__= '03/10/2002'

DEBUG=0

class Py4datAsciiPlugin(Plugin):
    	"""
	Py4datAsciiPlugin class. Inherits Plugin class.
	This plugin allows to save in one ascii file (or in several files)
	eather a graph or a serie of azymuthal integration,
	once the user has define a ROI on the AzimInt window.
	If the fit option has been used, the fitting results are also calculated for each image 
	and saved in the ascii file.
	The peak tracking option available through the main window menu of py4dat
	is taken into account here to adapt the ROI and display the results if required.
	"""
	########################################################
    	### Virtuals
    	########################################################
    	def OnStartInit (self):
		if DEBUG: print 'In Py4datAsciiPlugin.OnStartInit...'
		return 1

    	def OnFinishInit (self,app_window):
        	if DEBUG: print 'In Py4datAsciiPlugin.OnFinishInit...'
		self.app_window=app_window
		self.app_window.Display=0
	
	def OnFileSave (self,window,filename,format_string):
        	'''
		Check whether the user wants to save in one file or in several files,
		do the calculations for each file and save the results of integration and fit if required.
		For several files, the name of the result files begin with the chosen name 
		followed by the name of the original file.
		Parameters:
		- window
		- filename
		- format_string
		'''
		if DEBUG: print 'In Py4datAsciiPlugin.OnFileSave...'
		
		self.sourceWindow=self.GetSourceWindow(window)
		self.azimWindow=self.app_window.GetWindow('AzimInt - '+ self.sourceWindow.name)
                if self.azimWindow==None : self.azimWindow=self.app_window.GetWindow('AzimInt - Transformed - '+ self.sourceWindow.name)
		
		
		if hasattr(self.sourceWindow,'shortName') and self.sourceWindow.shortName[0:5]=='Stack':
			if self.azimWindow==None:
				print 'Do Azim Int and select Rois !........'
				return
			if self.azimWindow.rois=={}:
				print 'Select Rois !.........'
				return
			dial=AsciiParametersDialog(None,{'OneFileFlag':1,'SeveralFilesFlag':0,'Display':self.app_window.Display})
			if dial.dlg.result()==0: return
			self.OneFileFlag=dial.OneFileFlag
			self.SeveralFilesFlag=dial.SeveralFilesFlag
			self.app_window.Display=dial.Display
			if self.OneFileFlag==1: print 'Save in one file...'
			else: print 'Save in several files...'
			image=0	
    			
			if self.OneFileFlag==1:
				file=open(filename,'w')
				self.SaveCommonParameters(window,file)
				while image < self.sourceWindow.NumImages: 
					self.SaveImage(window,image,file)
					image=image+1
				file.close()
			elif self.SeveralFilesFlag==1:
				while image < self.sourceWindow.NumImages: 
					shortFileName=string.split(self.sourceWindow.fileName[image],'/')[-1]
					shortFileName=string.split(shortFileName,'.')[0]
					if string.count(filename,'.')!=0: newFilename=string.split(filename,'.')[0] + '_' + shortFileName + '.' + string.split(filename,'.')[1]
					else: newFilename=filename + '_' + shortFileName
					file=open(newFilename,'w')
					self.SaveCommonParameters(window,file)
					self.SaveImage(window,image,file)
					image=image+1
				file.close()
		else:
			print 'initialFile=',self.sourceWindow.name,'\n','destinationFile=',filename
			file=open(filename,'w')
			self.SaveCommonParameters(window,file)
			if window==None or isinstance(window.View,ImageView) == 1 :
				print 'The active window must be a GraphView...........'
				return
			file.write('###########################################\n')
			file.write('# File:' + window.name + '\n')
			xdata=window.View.GetSource()[0].GetOutput()['xdata'].tolist()
			ydata=window.View.GetSource()[0].GetOutput()['data'].tolist()
			file.write('# Xdata' + '    /    ' + 'Ydata' + '\n')
			i=0
			while i < len(xdata) :
				file.write(str(xdata[i]) + '    ' + str(ydata[i]) + '\n')
				i=i+1
			file.close()	

		
	def GetSaveFormatList(self,window):
        	if DEBUG: print 'In Py4datAsciiPlugin.GetSaveFormatList...'
		return ["Ascii files (*.txt)"]

	def GetInfoString(self):
		return "v. "+__version__+",  Author:"+__author__

    	########################################################
    	### Private
    	########################################################

	def SaveCommonParameters(self,window,file):
		"""
		Save as header the parameters common for each original image: 
			Date, CenterXY, OffsetXY, PixelXY, Wavelength, Distance, ROIx, ROIy
		and optionnal parameters if used:
			Transformed, Masked, Reduced Image, R Tracking, Phi Tracking
		"""
		file.write('# Py4dat Ascii Save for Graph Views'+'\n')
		file.write('# Date:'+ time.asctime() +'\n')
		file.write('# Parameters:'+'\n')
		file.write('#   CenterXY:'+str(self.sourceWindow.center)+'\n')
		file.write('#   OffsetXY:'+str(self.sourceWindow.offset)+'\n')
		file.write('#   PixelXY:'+str(self.sourceWindow.pixelsize)+'\n')
		file.write('#   Wavelength: '+str(self.sourceWindow.wavelength)+'\n')
		file.write('#   Distance: '+str(self.sourceWindow.distance)+'\n')
		if self.azimWindow!=None:
			ret=[]
			if self.azimWindow.rois=={}:
				print 'Select Rois !.........'
				return
			for roi in self.azimWindow.rois.keys():
				ret.append(self.azimWindow.rois[roi])
			self.Limitsx = [ret[0].Limitsx[0]]
			self.Limitsy = [ret[0].Limitsy[0]]
			ROIx=self.Limitsx[0]
			ROIy=self.Limitsy[0]
			file.write('#   ROIx:'+str(ROIx)+'\n')
			file.write('#   ROIy:'+str(ROIy)+'\n')
			if string.count(window.name,'Transformed')!=0:
				i=0
				file.write('#   Transformation: ')
				while i<len(self.sourceWindow.Transformation):
					file.write(str(self.sourceWindow.Transformation[i][0:2]) + ' ')
					i=i+1 
				file.write('\n')
			if hasattr(self.sourceWindow,'Mask') and type(self.sourceWindow.Mask) != types.NoneType: 
				file.write('#   Masked image:\n')
				if hasattr(self.sourceWindow,'GraphicalMaskRect') and self.sourceWindow.GraphicalMaskRect!=[]:
					file.write('#     Rectangular Masks: '+str(self.sourceWindow.GraphicalMaskRect)+' \n')
				if hasattr(self.sourceWindow,'GraphicalMaskPolygon') and self.sourceWindow.GraphicalMaskPolygon!=[]:
					file.write('#     Polygonal Masks: '+str(self.sourceWindow.GraphicalMaskPolygon)+' \n')
				if self.sourceWindow.minThreshold!=None: file.write('#     Min Threshold: '+str(self.sourceWindow.minThreshold)+' \n')
				if self.sourceWindow.maxThreshold!=None: file.write('#     Max Threshold: '+str(self.sourceWindow.maxThreshold)+' \n')
			if self.sourceWindow.shortName=='Stack - ReducedImage':
				file.write('#   Reduced Image: position=' + str(self.sourceWindow.pos) + ', size=' + str(self.sourceWindow.size) +'\n')
			if hasattr(self.app_window, 'RTrackingFlag') and self.app_window.RTrackingFlag==1:
				file.write('#   R Tracking: width=' + str(self.app_window.RTrackingWidth) + ', step=' + str(self.app_window.RTrackingStep) +'\n')
			if hasattr(self.app_window, 'PhiTrackingFlag') and self.app_window.PhiTrackingFlag==1:
				file.write('#   Phi Tracking: width=' + str(self.app_window.PhiTrackingWidth) + ', step=' + str(self.app_window.PhiTrackingStep) +'\n')
	
	###############################################################################
	
	def SaveImage(self,window,image,file):
		"""
		Save the results for the image in the given file.
		Parameters:
		- image: integer corresponding to the range of the image in the SourceWindow, which is a stack. 
		- file: resulting from : open(filename,'w')
		"""
		file.write('###########################################\n')
		file.write('# File:' + self.sourceWindow.fileName[image] + '\n')
		shortFileName=string.split(self.sourceWindow.fileName[image],'/')[-1]
		if self.sourceWindow.fileName[image][-3:]=='edf':
			print '######### Reading the file: ', shortFileName
			data=EdfFileData()
		elif self.sourceWindow.fileName[image][-4:]=='mccd':
			data=MccdFileData()
		else: 
			print '######### Reading the file: ', shortFileName
			data=EdfFileData()
                data.SetSource(self.sourceWindow.fileName[image])
		if self.sourceWindow.shortName=='Stack - ReducedImage': data.LoadSource(0,pos=self.sourceWindow.pos,size=self.sourceWindow.size)
		else: data.LoadSource()
		if hasattr(self.sourceWindow,'Mask') and type(self.sourceWindow.Mask) != types.NoneType:
			print 'Muliply by the mask...'
			if hasattr(self.sourceWindow,'ThresholdMask') and self.sourceWindow.ThresholdMask!=None:
				Mask=self.sourceWindow.GraphicalMask * self.sourceWindow.ThresholdMask
			else: Mask=self.sourceWindow.GraphicalMask
			data.Pages[0].Array=data.GetPageArray()*Mask
		dataSelection=DataSelection.RectSelection(data)
		if string.count(window.name,'Transformed')!=0:
			i=0
			while i<len(self.sourceWindow.Transformation):
				if self.sourceWindow.Transformation[i][0]=='ADD': dataSelection=ColormapFilter(None,AddConstantFilter(dataSelection,buffer_input=1, Constant=self.sourceWindow.Transformation[i][1]))
    				elif self.sourceWindow.Transformation[i][0]=='MULTIPLY': dataSelection=ColormapFilter(None,MultiplyByConstantFilter(dataSelection,buffer_input=1, Constant=self.sourceWindow.Transformation[i][1]))
				elif self.sourceWindow.Transformation[i][0]=='BKG': dataSelection=ColormapFilter(None,SubstractBackgroundFilter(dataSelection,buffer_input=1, Constant=self.sourceWindow.Transformation[i][1],BackgroundOutput=self.sourceWindow.Transformation[i][2]))	
				elif self.sourceWindow.Transformation[i][0]=='LRFLIP': dataSelection=ColormapFilter(None,LeftRightFlipFilter(dataSelection,buffer_input=1))
				elif self.sourceWindow.Transformation[i][0]=='TBFLIP': dataSelection=ColormapFilter(None,TopBottomFlipFilter(dataSelection,buffer_input=1))
				i=i+1
		azimFilter=AzimIntFilter(dataSelection,pars={'pixelsize':self.sourceWindow.pixelsize,'offset':self.sourceWindow.offset,'center':self.sourceWindow.center})
		azimDict=azimFilter.GetOutput()
		azimData=azimDict['data']
		ROIx=self.Limitsx[0]
		ROIy=self.Limitsy[0]	
		#############################
		# If tracking enable, the rois have to be adapted to find the peak:
		#### R Tracking:
		if hasattr(self.app_window,'RTrackingFlag') and self.app_window.RTrackingFlag==1 or hasattr(self.app_window,'PhiTrackingFlag') and self.app_window.PhiTrackingFlag==1:
			oldIntegration = Numeric.add.reduce(Numeric.add.reduce(azimData[ROIy[0]:ROIy[1],ROIx[0]:ROIx[1]],1))
		if hasattr(self.app_window,'RTrackingFlag') and self.app_window.RTrackingFlag==1:
			print 'R TRACKING..............'
			step=self.app_window.RTrackingStep
			width=self.app_window.RTrackingWidth
			newROI0=max(ROIx[0]-width,0)
			ROI0List=[]
			integrationList=[]
			while newROI0 < min(ROIx[0]+width, azimData.shape[1]-(ROIx[1]-ROIx[0])):
				ROI0List.append(newROI0)
				newIntegration = Numeric.add.reduce(Numeric.add.reduce(azimData[int(ROIy[0]):int(ROIy[1]),int(newROI0):int(newROI0+ROIx[1]-ROIx[0])],1))
				integrationList.append(newIntegration)
				print 'ROI0 x:',int(newROI0), 'Integration:', newIntegration
				newROI0=newROI0+step
			integrationMax=max(integrationList)
			print 'integrationMax:',integrationMax, 'old: ', oldIntegration
			index=integrationList.index(integrationMax)
			ROIx=(ROI0List[index],ROI0List[index]+ROIx[1]-ROIx[0])
			file.write('# New ROIx after peak tracking:')
			file.write(str(ROIx)+'\n')
		#### Phi tracking:
		if hasattr(self.app_window,'PhiTrackingFlag') and self.app_window.PhiTrackingFlag==1:
			print 'PHI TRACKING..............'
			step=self.app_window.PhiTrackingStep
			width=self.app_window.PhiTrackingWidth
			newROI0=max(ROIy[0]-width,0)
			ROI0List=[]
			integrationList=[]
			while newROI0 < min(ROIy[0]+width, azimData.shape[1]-(ROIy[1]-ROIy[0])):
				ROI0List.append(newROI0)
				newIntegration = Numeric.add.reduce(Numeric.add.reduce(azimData[int(newROI0):int(newROI0+int(ROIy[1])-int(ROIy[0])), int(ROIx[0]):int(ROIx[1])],1))
				integrationList.append(newIntegration)
				print 'ROI0 y:',newROI0, 'Integration:', newIntegration
				newROI0=newROI0+step
			integrationMax=max(integrationList)
			print 'integrationMax:',integrationMax, 'old: ', oldIntegration
			index=integrationList.index(integrationMax)
			ROIy=(ROI0List[index],ROI0List[index]+ROIy[1]-ROIy[0])
			file.write('# New ROIy after peak tracking:')
			file.write(str(ROIy)+'\n')
		#############################
		# Azymuthal integration and fit if required:
		iphiFilter=IntegratePhiFilter('',azimFilter,ROIx=ROIx,ROIy=ROIy,dict=azimDict)
		phiDict=iphiFilter.GetOutput()		
		xdata=phiDict['xdata']
		ydata=phiDict['data']
		if hasattr(window,'fit_filter') and type(window.fit_filter) != types.NoneType:
			fit_filter=FitFilter(iphiFilter,dict=phiDict)
			fit_filter.SetParameters(window.fit_filter.GetParameters())
			yfit=fit_filter.GetOutput()['data']
			file.write('# Fit Parameters:' + str(window.fit_filter.GetParameters()) + '\n')
			file.write('# Xdata' + '    /    ' + 'Ydata' + '    /    ' + 'Yfit' + '\n')
			i=0
			while i < len(ydata) :
				file.write(str(xdata[i]) + '    ' + str(ydata[i]) + '    ' + str(yfit[i])+ '\n')
				i=i+1
		else:
			file.write('# Xdata' + '    /    ' + 'Ydata' + '\n')
			i=0
			while i < len(ydata) :
				file.write(str(xdata[i]) + '    ' + str(ydata[i]) + '\n')
				i=i+1
		############################
		# Display of integration result for each image if option display selected:
		if hasattr(self.app_window,'Display') and self.app_window.Display==1:
			resultWindow=self.app_window.GetWindow('I(Phi) - '+ self.azimWindow.name)
			#resultWindow.SetTitle('Result: '+shortFileName)
			resultWindow.Filter=iphiFilter
                	resultWindow.View.SetSource(resultWindow.Filter)
		############################		
	
    	def GetSourceWindow(self,window):
    		"""
		Returns the source window corresponding to this window.
		Parameter:
		- window: a given window
		"""
		if window==None: return
		while window.master_window!=None:
			window=window.master_window
		return window
	
				
plugin=Py4datAsciiPlugin()
