"""
    ImageFileData.py
    Data derived class to access image files (JPG,GIF,...)
"""

from PyDVT import __version__,__date__,__author__


################################################################################  
from Data import *
import Image
################################################################################

SOURCE_TYPE = "ImageFile"


class ImageFileData(Data):
    """
    Specializes Data class in order to access image files.
    Stores image in a Numeric Array of type long (4 bytes), containing the RGBX
    representation of each pixel.
    Interface: Data class interface.
    """
    def __init__(self,refresh_interval=None,info={}):
        """
        See Data.__init__
        """
        self.EdfObj=None
        info["Class"]="ImageFileData"        
        Data.__init__(self,refresh_interval,info)


    def SetSource (self,source_name=None):
        """
        Sets a new source for data retrieving, an edf file.
        If the file exists, self.Source will be the EdfFile
        object associated to this file.
        Parameters:
        source_name: name of the edf file 
        """
        if source_name==self.SourceName: return 1
        if (source_name != None): 
            try:
                self.Source = Image.open(source_name)
            except:
                self.SourceName=None
                self.Source=None
                return 0
        else:
            self.Source=None
        self.SourceName=source_name
        return 1


    def GetSourceInfo (self):
        """
        Returns information about the EdfFile object created by
        SetSource, to give application possibility to know about
        it before loading.
        Returns a dictionary with the keys "Size" (number of possible
        keys to this source) and "KeyList" (list of all available keys
        in this source). Each element in "KeyList" is an integer
        meaning the index of the array in the file.
        """        
        if self.SourceName == None: return None
        NumImages=1
        source_info={}
        source_info["Size"]=1
        source_info["KeyList"]=(0,)
        source_info["Format"]=self.Source.format
        source_info["Size"]=self.Source.size
        source_info["Mode"]=self.Source.mode
        source_info["Palette"]=self.Source.palette
        return source_info        

        
    
    def LoadSource(self,key_list="ALL",append=0,invalidate=1):
        """
        Creates a given number of pages, getting data from the actual
        source (set by SetSource)
        Parameters:
        key_list: list of all keys to be read from source. It is a list of
                 integers, meaning the indexes to be read from the file.
                 It can be also one integer, if only one array is to be read.
        append: If non-zero appends to the end of page list.
                Otherwise, initializes the page list                
        invalidate: if non-zero performas an invalidade call after
                    loading        
        """
        if append==0: Data.Delete(self)
        info={}
        info["SourceType"]=SOURCE_TYPE
        info["SourceName"]=self.SourceName
        info["Key"]=0
        info["Source"]=self.Source
        
        info["Format"]=self.Source.format
        info["Size"]=self.Source.size
        info["Mode"]=self.Source.mode
        info["Palette"]=self.Source.palette
        info.update(self.Source.info)
        
        str=self.Source.convert("RGBX").tostring()
        array=Numeric.fromstring(str,"l")
        array=Numeric.reshape(array,(self.Source.size[1],self.Source.size[0]))
       
        self.AppendPage(info,array)        
        if invalidate: self.Invalidate()


    def GetCoordValue(self,coord):
        """
        Returns the value of data in a gives coordinate. In this case, overrides
        the original method to return a tuple of (R,G,B) values.        
        Parameters:
        coord:  a DataPosition object
        """
        val=Data.GetCoordValue(self,coord)
        R=val & 0xFF
        G= (val>>8) & 0xFF
        B= (val>>16) & 0xFF
        return (R,G,B)



################################################################################
#EXEMPLE CODE:
    
if __name__ == "__main__":
    import sys,time
    try:
        Filename=sys.argv[1]
        obj=ImageFileData()
        obj.SetSource(Filename)
        print GetSourceInfo()
        obj.LoadSource()
    except:
        print "Usage: EdfFileData.py <filename>"
        sys.exit()
    
    for i in range (obj.GetNumberPages()):
        print obj.GetPageInfo(i)
        print obj.GetPageArrayRegion((0,0),(2,2),i)
        print obj.GetPageSize(0)
    

        