"""
    FitFilter.py
    Filter derived class for 1d fitting
    Based on Specfit
"""  

from PyDVT import __version__,__date__,__author__

import Numeric
import string
from Filter import Filter
from SpecfitFunctions import SpecfitFunctions
from Specfit import Specfit

DEFAULT_FIT_PARS={"WeightFlag":0,"Theory":"Pseudo-Voigt Line","Background":"Linear", \
                  "Mode":"Normal","Sensitivity":3.25, "FwhmPoints":10,"Yscaling":1.0,\
                  "AutoFwhm":0,"AutoScaling":0}


class FitFilter(Filter):
    """
    Filter for 1d-fitting (Based on Specfit)
    """
    def __init__(self,source,synchronized=1,xdata=None):
        """
        See Filter.__init__
        """
        Filter.__init__(self,source,synchronized,buffer_input=1)
        self.FitFinishedEvent = self.eh.create("FitFinished")
        self.eh.register("ParametersChange", self.SetParameters)        
        self.Editor=None
        self.fit=Specfit()
        self.fit.importfun("SpecfitFunctions.py")
        self.Parameters={}
        self.Parameters.update(DEFAULT_FIT_PARS)
        self.DefaultAction="ESTIMATE_FIT"
        self.fit.settheory(DEFAULT_FIT_PARS["Theory"])
        self.xdata=xdata
        
            
    def GetOutput(self):
        """
        See Filter.GetOutput
        """
        input=self.GetInput()
        if "data" not in input.keys(): return {}
        ydata=input ["data"]        
        if "xdata" in input.keys(): xdata=input["xdata"]
        else: xdata=Numeric.arrayrange(ydata.shape[0])
        if self.Editor is not None: self.Parameters.update(self.Editor.GetParameters())
        if self.Parameters=={}: return {}
        self.fit.setdata(x=xdata,y=ydata)
        self.fit.settheory(self.Parameters["Theory"])
        self.fit.configure(Yscaling=self.Parameters["Yscaling"],
                      AutoScaling=self.Parameters["AutoScaling"],                      
                      WeightFlag=self.Parameters["WeightFlag"],
                      PosFwhmFlag=1,
                      HeightAreaFlag=1,
                      FwhmPoints=self.Parameters["FwhmPoints"],
                      AutoFwhm=self.Parameters["AutoFwhm"],
                      PositionFlag=1,
                      HypermetTails=5,sensitivity=self.Parameters["Sensitivity"])
        self.fit.setbackground(self.Parameters["Background"])

        if "Action" in self.Parameters.keys():
            action=self.Parameters["Action"]
            del self.Parameters["Action"]
        else: action=self.DefaultAction
    
        if action=="ESTIMATE" and self.Parameters["Mode"]!="MCA":
                self.fit.estimate()
                if self.eh is not None :                
                        self.eh.event(self.FitFinishedEvent, self.fit.paramlist)            
                return {}
        
        if action!="ESTIMATE_FIT" and action!="FIT": return {}
        if self.Parameters["Mode"]=="MCA":        
            mcaresult=self.fit.mcafit()
            if mcaresult != []:
                finalx=Numeric.array([])
                finaly=Numeric.array([])
                delimeters=[]                
                finalp=[]
                for result in mcaresult:               
                  delimeters.append((result['xbegin'],int(result['xend'])))
                  x=self.fit.xdata0[int(result['xbegin']):int(result['xend'])]
                  y=self.fit.ydata0[int(result['xbegin']):int(result['xend'])]
                  if self.xdata is not None: x=self.xdata
                  yfit=self.fit.gendata(x=x,parameters=result['paramlist'])
                  finalx=Numeric.concatenate((finalx,x))
                  finaly=Numeric.concatenate((finaly,yfit))
            else:
                    return {}
            if self.eh is not None : self.eh.event(self.FitFinishedEvent, mcaresult)
                
            return {"data": finaly, "xdata":finalx,"delimeters":delimeters}
        else:
            if action=="ESTIMATE_FIT": self.fit.estimate()
            elif "FitPars" in self.Parameters.keys(): self.fit.paramlist=self.Parameters["FitPars"]
            self.fit.startfit()
            if self.xdata is not None: xdata=self.xdata
            yfit=self.fit.gendata(x=xdata,parameters=self.fit.paramlist)
            if self.eh is not None : self.eh.event(self.FitFinishedEvent, self.fit.paramlist)            
            return {"data": yfit, "xdata":xdata}



    def ConnectParameterEditor(self,editor):
        """
        Connects filter to fit parameters editing dialog.
        """
        self.DisconnectParameterEditor()
        self.Editor=editor
        self.Editor.onInitialize(self.eh,self.fit)
        if self.Editor is not None: self.Editor.SetParameters(self.Parameters)
        

    def DisconnectParameterEditor(self):
        """
        Disconnects filter from fit parameters editing dialog.
        """
        if self.Editor is not None:
            self.Editor.onInitialize(None,None)
            self.Editor=None
        


    def GetParameters(self):
        """
        Returns fit parameter dictionary
        See SetParameters
        """
        return self.Parameters


    def SetParameters(self,parameters):
        """
        Sets fit parameter dictionary
        Parameters (See Specfit):
            parameter: Dictionary with the following keys:
                        "WeightFlag": 0 or 1
                        "Theory": 'Gaussians','Lorentz','Area Gaussians','Area Lorentz',...
                        "Background"= 'Linear','No Background', 'Constant', ...
                        "Mode"= "Normal" or "MCA"
                        "Action"= See SetDefaultAction
                        "FitPars"= Specfit parameters
                        "Sensitivity" : float
                        "FwhmPoints" : integer
                        "Yscaling": float
                        "AutoFwhm": 1 or 0
                        "AutoScaling":1 or 0
                        
        """
        #self.Parameters=parameters
        self.Parameters.update(parameters)
        if self.Editor is not None: self.Editor.SetParameters(parameters)
        self.Refresh()


    def SetDefaultAction(self,action):
        """
        Sets default behaviour in a data change event/
        Parameters:
            action: "ESTIMATE_FIT": Estimates, fits functions and plots graphic
                    "FIT": Fits and plots graphic
                    "ESTIMATE": Estimates and clear graphic
                    "CLEAR": Clear graphic
        """
        self.DefaultAction=action
