"""
    Data.py
    Generic class for Data storage.    
"""

from PyDVT import __version__,__date__,__author__


################################################################################  
import Numeric
import EventHandler
import types
import copy

from Binding import GUI_Binding

################################################################################

SOURCE_TYPE = "NumPyArray"


class Page:
    """
    Page class defines a structure composed by a NumPy array and a dictionary.
    Data objects contain a set of pages.
    The array stores the actual data.
    The dictionary contains variable information about the data in the array.    
    """
    def __init__(self,info={},array=None):
        """ Constructor:
        Pages can be initialized in the constructor, or be filled afterwards.
        All pages must have a field in the info dictionary called "SourceType",
        which indicates source type for this data. If not present in info,
        it is initialized to "None". This field is going to be verified by
        Data derived classes, in the method UpdatePage to control
        updating of objects having pages with diferent source types.
        """
        if "SourceType" not in info.keys():
            info["SourceType"]="None"
        self.Array=array
        self.Info=info



    def __getstate__(self):
        import copy
        dict=copy.copy(self.__dict__)
        info=copy.copy(dict['Info'])
        if "_RefreshFunction"  in info.keys(): info["_RefreshFunction"]=None
        dict['Info']=info
        return dict
                

 



class DataPosition:
    """
    Structure to store data coordinates.
    Consists of:
    page_index: the index of the page
    page_coord: tuple with the coordinates in the page, (x,), (x,y) or (x,y,z)
               according to the dimention of the page
    """
    def __init__(self,page_index=None,page_coord=None):
        self.PageIndex=page_index
        self.PageCoord=page_coord




class Data:
    """
    Data class is the base class for generic data retrieving and 
    storing for displaying.
    All operations needed for interfacing with displaying
    classes should be  coded in the base class. Derived classes 
    implement the specialized code to retrieve information from
    different data sources.

    Data objects are composed by a set of page objects. Each page has
    a NumPy array that stores the actual data, and a dictionary, containing
    information about this data and its source.

    The items in the page's information dictionary are defined by
    the derived data classes, but these keys in the dictionary are 
    standarized:
    
    "SourceType": string, identify the type of data source of the page.
                  Each derived class have to define it's own.
    "SourceName": the name of the data source of this page
    "Key": the key for this actual data in the data source, which is
           source dependent.
    "Source": optional, represent the source object itself (if possible)
           in order to allow the application to make direct calls to
           low level funcionality.
    Data class can add elements to this dictionary for internal purposes
    (not to be considered by application of View objects).
    In this case, the key starts by "_".

    Data objects have a general dictionary containing global information
    and, if needed, the relation between pages.

    Base class objects can be created to manage direct NumPy arrays
    manipulation, or to mix pages of different data sources.

    Data objects have an EventHandler member (self.eh) through which
    it sends "DataChange" and "DeleteEvent" events


    Interface:
    ===========================
    - __init__
    - GetNumberPages
    - AppendPage
    - InsertPage
    - GetSource
    - GetSourceName
    - GetInfo
    - GetPageDimention
    - GetPageSize
    - GetPageArray
    - GetPageInfo
    - GetPageArrayRegion
    - GetItemPageInfo
    - GetPageArrayRegion
    - IsCoordValid
    - GetCoordValue
    - CopyPages
    - Delete
    - Invalidate
    - Refresh
    - Destroy

    - SetSource       (virtual)
    - GetSourceInfo   (virtual)
    - LoadSource      (virtual)
    
    In most of the cases the applications will deal just with the
    second group of methods, they're a proposed way to have a
    standarized and higher level interface to data sources.

    SetSource, GetSourceInfo and LoadSource are the methods that derived 
    classes  should override (a fourth is the RefreshPage it wants to
    have automatic updating by polling). These three methods are signed as
    virtuals, but there's a default implementation in the Data class
    for loading numeric arrays (NumPy arrays can be handled by direct
    calling AppendPage/InsertPage/Delete methods, of course, this
    implementation is there as a simple example of what derived classes
    should do).

    New methods can be added to derived classes, but it's good to try to keep
    the same philosophy, in order all the sources to have a similar interface:
    1) SetSource links the Data object with a source, without actually loading
       any data.
    2) GetSourceInfo gets information about source. It returns at least a list
       of keys to identify every readable item in the source.
    3) LoadSource performs the reading of data. It indexes source's items with
       the same keys received by GetSourceInfo.
    
    If refresh_interval is not defined in the constructor, applications can
     poll for changes in data calling Refresh (if RefreshPage has been coded).

    Derived classes implement automatic refresh bahaviour by overriding
    RefreshPage (see doc).
    
    In automatic polling mode, if a GUI Binding is loaded then RefreshPage
    is called from a timer, with interval defined as in the Data object
    constructor. Otherwise, a thread is created to call RefreshPage.
    In this case the base class creates a self.Semaphore property to synchronize
    main thread. self.Semaphore.acquire() and self.Semaphore.release() are
    called from the thread befire and after RefreshPage.

    The base class doesn't take any other action about thread synchronization
    it is up to derived classes to call pairs self.Semaphore.acquire() and
    self.Semaphore.release() to protect concurrent code.

    The lower level interface is not thread save, so if the derived class
    is supposed to use thread, it is responsible to have a thread safe
    interface.    

    The indexing of pages (see interface methods, index parameter) in a data
    object can be done in two was:
    - With a integer that represents the position of the page. This is
      a simpler approach that suits for simpler applications.
    - With a dictionary, that logically indexes the page based on a number of
      keys (typically "SourceType", "SourceName" and "Key"). This is the
      way applications should deal if they are going to delete pages from
      a Data objects, in order to other objects to keep the link logically.

    Internal Events:
    ===========================
        Emits:
            "DataChange"
            "DeleteEvent"
    """
    def __init__(self,refresh_interval=None,info={}): 
        """
        Parameters:
        refresh_interval: time in milisseconds the Data will be refreshed
                         if None, no polling for data changes        
        info: dictionary containing generic information about this object
        """
        self.eh = EventHandler.EventHandler()
        if "Class" not in info.keys(): info["Class"]="Data"
        self.RefreshInterval=refresh_interval
        self.Info=info
        self.Pages=[]
        self.DataChange = self.eh.create("DataChange")
        self.DeleteEvent = self.eh.create("DeleteEvent")        
        self.SourceName=None
        self.Source=None
        self.Thread=None

        if self.RefreshInterval != None:
          if GUI_Binding!=None:
              from Binding import Timer
              self.timer=Timer(None,self.Refresh)
              self.timer.Start(self.RefreshInterval)
          else: 
            self.Semaphore=threading.Semaphore()        
            self.Event=threading.Event()
            self.MainThread=threading.currentThread()            
            self.Thread= threading.Thread(group=None, target=self.__ThreadProc, name="DataPoll")
            self.Thread.start()
            
        self.Alive=1


    def __getstate__(self):
        import copy
        dict=copy.copy(self.__dict__)
        info=copy.copy(dict['Info'])
        if "_RefreshFunction"  in info.keys(): info["_RefreshFunction"]=None
        dict['Info']=info
        return dict
        


    def __ThreadProc (self):
        """
        The polling thread, active if no GUI is loaded
        (otherwise a GUI Timer is used).
        """
        while (self.Event.isSet()==0) and (self.MainThread.isAlive()):
            self.Semaphore.acquire()
            self.Refresh()
            self.Semaphore.release()
            self.Event.wait(float(self.RefreshInterval)/1000.)            
        self.Event.clear()



    def Destroy(self,source=None):
        """
        Cleanup.
        As event handlers and callbacks, by keeping a reference to other
        objects, avoid them to be destroyed (and freed from memory), the
        Destroy method has to be explicitly called for Data objects
        before they get out of scope.
        I had tried to use weak references in order not to need this method,
        but this generated many other troubles (like impossibility to define
        selections locally, difficult control of the destruction sequence,
        not interceptable error mesassages...).
        """
        if self.Alive==0: return
        self.eh.event(self.DeleteEvent, self)     
        if self.RefreshInterval != None:
          if GUI_Binding!=None:
            if self.timer is not None:
                self.timer.Stop()
                self.timer=None
            else:
                if self.Thread != None:
                    self.Semaphore.acquire()
                    if self.Thread.isAlive():
                        self.Event.set()        
                        while self.Event.isSet():
                            pass
                    else:
                        self.Event.set()
                    self.Thread=None
        
        self.Info=None
        self.Pages=[]
        self.eh = None
        self.Alive=0
    
    def GetNumberPages(self):
        """
        Returns number of pages of the data object
        """
        return len(self.Pages)
    

    def AppendPage(self,info={},array=None):
        """
        Appends one page to the page list.
        This method just stores a reference to info and array, does not perform
        a deepcopy
        Parameters:
        info: Dictionary of the page
        array: NumPy data of the page
        """
        if "_RefreshFunction" not in info.keys(): info["_RefreshFunction"]=self.RefreshPage
        page=Page(info,array)
        self.Pages.append(page)

    def InsertPage(self,info={},array=None,index=None):
        """
        Inserts one page in a given position of the page list.
        This method just stores a reference to info and array, does not perform
        a deepcopy
        Parameters:
        info: Dictionary of the page
        array: NumPy data of the page
        index: Either an integer meaning the sequencial position of the page
               or a dictionary that logically index the page based on keys of
               the page's Info dictionary.
        """
        if "_RefreshFunction" not in info.keys(): info["_RefreshFunction"]=self.RefreshPage
        page=Page(info,array)
        index=self.GetPageListIndex(index)
        if index is None: self.Pages.append(page)
        else            : self.Pages.insert(index,page)
        


    def GetInfo(self):
        """
        Returns the general information about the object
        """
        return self.Info


    def GetSource(self):
        """
        Returns source object
        """
        return self.Source
        

    def GetSourceName(self):
        """
        Returns source name
        """
        return self.SourceName


    def GetPageDimention(self,index=0):
        """
        Returns the dimention of a given page
        Parameters:
        index: Either an integer meaning the sequencial position of the page
               or a dictionary that logically index the page based on keys of
               the page's Info dictionary.
        """
        index=self.GetPageListIndex(index)
        if index is None or index >= len(self.Pages): return None
        if self.Pages[index].Array is None: return 0        
        return len(self.Pages[index].Array.shape)


    def GetPageSize(self,index=0):
        """
        Returns a tuple containing the size of a page
        Parameters:
        index: Either an integer meaning the sequencial position of the page
               or a dictionary that logically index the page based on keys of
               the page's Info dictionary.
        """
        array=self.GetPageArray(index)
        if self.GetPageDimention(index)==1:
            return (array.shape[0],)             
        elif self.GetPageDimention(index)==2:
            return (array.shape[1],array.shape[0],)
        elif self.GetPageDimention(index)==3:
            return (array.shape[2],array.shape[1],array.shape[0],)
        else:
            return None


    def GetPageArray(self,index=0):
        """
        Returns page's data (NumPy array)
        Parameters:
        index: Either an integer meaning the sequencial position of the page
               or a dictionary that logically index the page based on keys of
               the page's Info dictionary.
        """
        index=self.GetPageListIndex(index)
        if index is None or index >= len(self.Pages): return None
        return self.Pages[index].Array


    def GetPageInfo(self,index=0):
        """
        Returns page's information (dictionary)
        Parameters:
        index: Either an integer meaning the sequencial position of the page
               or a dictionary that logically index the page based on keys of
               the page's Info dictionary.
        """
        index=self.GetPageListIndex(index)
        if index is None or index >= len(self.Pages): return None
        return self.Pages[index].Info


    def GetItemPageInfo(self,key,index=0):
        """
        Returns the value of a given key in page's information
        Parameters:
        key:   Key of page dictionary
        index: Either an integer meaning the sequencial position of the page
               or a dictionary that logically index the page based on keys of
               the page's Info dictionary.
        """
        index=self.GetPageListIndex(index)
        if index is None or index >= len(self.Pages): return None
        if key in self.Pages[index].Info.keys():
            return self.Pages[index].Info[key]
        else:
            return None

    def GetPageArrayRegion(self,pos=None,size=None,index=0,return_dimention=None):
        """
        Returns one slice of the data in a given page.
        This method may be used by higher level selection functions (they 
        should manage selection in multiple pages)
        
        Parameters:
        pos: Tuple containing the coordinates of the beggining of the slice.
             Coordinates must match the dimention of data:
             (x,),(x,y,) or (x,y,z,),
             If None,set to the beggining of the data (0,),(0,0,) or (0,0,0,),
             according to the dimention of data.
        size: Tuple containing the coordinates of the size of the slice.
             Coordinates must match the dimention of data:
             (x,),(x,y,) or (x,y,z,),
             If None,set to the size from pos to the end of data in all dimentions
             Any coordinate (x,y ou z) can be set to "ALL", which means up to the
             end of data in this dimention.
        index: Either an integer meaning the sequencial position of the page
               or a dictionary that logically index the page based on keys of
               the page's Info dictionary.
        return_dimention: Defines the dimention of the returned array. For instance,
                         if accessing 3d data, but we want the result as a 2d
                         array, instead of a 3d array with z=1, we should set
                         this parameter to 2. If None, it keeps the original
                         dimention.
        """
        index=self.GetPageListIndex(index)
        if index is None or index >= len(self.Pages): return None
        Dim=self.GetPageDimention(index)
        Arr=self.GetPageArray(index)
        if (pos==None) and (size==None): return Arr
        ArraySize=self.GetPageSize(index)          
        if (Dim==1):
            if (pos==None): pos=(0)
            if (size==None): size=(ArraySize[0]-pos[0])                    
            if (len(pos) != Dim)or(len(size) != Dim): return None                
            SizeX=size[0]
            if SizeX=="ALL": SizeX=Arr.shape[0]-pos[0]
            ArrRet=Numeric.take(Arr, range(pos[0],pos[0]+SizeX))
            if (return_dimention!=None) and (return_dimention!=Dim):
                if return_dimention == 2: ArrRet=Numeric.reshape(ArrRet,(1,SizeX))
                if return_dimention == 3: ArrRet=Numeric.reshape(ArrRet,(1,1,SizeX))
        elif (Dim==2):
            if (pos==None): pos=(0,0)
            if (size==None): size=(ArraySize[0]-pos[0],ArraySize[1]-pos[1])
            if (len(pos) != Dim)or(len(size) != Dim): return None                
            SizeX=size[0]
            SizeY=size[1]
            if SizeX=="ALL": SizeX=Arr.shape[1]-pos[0]
            if SizeY=="ALL": SizeY=Arr.shape[0]-pos[1]
            ArrRet=Numeric.take(Arr, range(pos[1],pos[1]+SizeY))
            ArrRet=Numeric.take(ArrRet, range(pos[0],pos[0]+SizeX),1)
            if (return_dimention!=None) and (return_dimention!=Dim):
                FullSize=SizeX*SizeY
                if return_dimention == 1:
                    ArrRet=Numeric.reshape(ArrRet,(FullSize,))
                if return_dimention == 3: ArrRet=Numeric.reshape(ArrRet,(1,SizeY,SizeX))
        elif (Dim==3):
            if (pos==None): pos=(0,0,0)
            if (size==None): size=(ArraySize[0]-pos[0],ArraySize[1]-pos[1],ArraySize[2]-pos[2])
            if (len(pos) != Dim)or(len(size) != Dim): return None                
            SizeX=size[0]
            SizeY=size[1]
            SizeZ=size[2]
            if SizeX=="ALL": SizeX=Arr.shape[2]-pos[0]
            if SizeY=="ALL": SizeY=Arr.shape[1]-pos[1]
            if SizeZ=="ALL": SizeZ=Arr.shape[0]-pos[2]
            ArrRet=Numeric.take(Arr, range(pos[2],pos[2]+SizeZ))
            ArrRet=Numeric.take(ArrRet, range(pos[1],pos[1]+SizeY),1)
            ArrRet=Numeric.take(ArrRet, range(pos[0],pos[0]+SizeX),2)
            if (return_dimention!=None) and (return_dimention!=Dim):
                FullSize=SizeX*SizeY*SizeZ
                if return_dimention == 1: ArrRet=Numeric.reshape(ArrRet,(FullSize,))
                if return_dimention == 2: ArrRet=ArrRet[0]
        else:
            ArrRet=None            
        return ArrRet


    def IsCoordValid(self,coord):
        """
        Returns if the coordinate is inside the data object
        Parameters:
        coord:  a DataPosition object
        """
        Arr=self.GetPageArray(coord.PageIndex)
        Dim=self.GetPageDimention(coord.PageIndex)
        if Dim != len(coord.PageCoord): return 0
        if   Dim==1 and coord.PageCoord[0] >=0 and coord.PageCoord[0] < Arr.shape[0]: return 1        
        elif Dim==2 and coord.PageCoord[0] >=0 and coord.PageCoord[0] < Arr.shape[1] and coord.PageCoord[1] >=0 and coord.PageCoord[1] < Arr.shape[0]: return 1
        elif Dim==3 and coord.PageCoord[0] >=0 and coord.PageCoord[0] < Arr.shape[2] and coord.PageCoord[1] >=0 and coord.PageCoord[1] < Arr.shape[1] and coord.PageCoord[2] >=0 and coord.PageCoord[2] < Arr.shape[0]: return 1
        return 0


    def GetCoordValue(self,coord):
        """
        Returns the value of data in a gives coordinate.
        Parameters:
        coord:  a DataPosition object
        """
        Arr=self.GetPageArray(coord.PageIndex)
        Dim=self.GetPageDimention(coord.PageIndex)
        if   Dim==1 and coord.PageCoord[0] >=0 and coord.PageCoord[0] < Arr.shape[0]:
                return Arr[coord.PageCoord[0]]
        elif Dim==2 and coord.PageCoord[0] >=0 and coord.PageCoord[0] < Arr.shape[1] and coord.PageCoord[1] >=0 and coord.PageCoord[1] < Arr.shape[0]:
                return Arr[coord.PageCoord[1],coord.PageCoord[0]]
        elif Dim==3 and coord.PageCoord[0] >=0 and coord.PageCoord[0] < Arr.shape[2] and coord.PageCoord[1] >=0 and coord.PageCoord[1] < Arr.shape[1] and coord.PageCoord[2] >=0 and coord.PageCoord[2] < Arr.shape[0]: 
                return Arr[coord.PageCoord[2],coord.PageCoord[1],coord.PageCoord[0]]
        return None
    

    def Delete(self,index=None):
        """
        Deletes a given page from the data object
        Parameters:
        index: Index of the page in the page list. If None, deletes all pages
        """
        if index == None: self.Pages=[]
        else:
            index=self.GetPageListIndex(index)
            if index is not None and index < len(self.Pages):  del self.Pages[index]
        
    
    def CopyPages (self,source_obj,index_list=0,position=None,synchronized=1,invalidate=1):
        """
        Copies  pages from other data object. This method makes a deepcopy operation.
        Parameters:
        source_obj: Source object of the copied pages
        index_list: index or tuple of indexes to be copied
        position: Position to be copied to. If None, it is appended
        synchronized: if non-zero stores the location of the refresh method of the
                          original object for updating the page. Otherwise, is no more
                          updated.
        invalidate: if non-zero performs an invalidade call after the operation
        """
        if index_list == "ALL": index_list=range(source_obj.GetNumberPages())
        elif type(index_list) is types.IntType or type(index_list) is types.DictType: index_list=[index_list]
        for i in index_list:
            index=self.GetPageListIndex(index)
            if index is None or index >= len(self.Pages): continue            
            oldinfo=source_obj.GetPageInfo(i)
            oldfunc=None
            if "_RefreshFunction" in oldinfo.keys():
                oldfunc=oldinfo["_RefreshFunction"]
                oldinfo["_RefreshFunction"]=None                
            info=copy.deepcopy(oldinfo)
            if (synchronized) and (oldfunc is not None): info["_RefreshFunction"]=oldfunc
            array=copy.deepcopy(source_obj.GetPageArray(i))
            
            position=self.GetPageListIndex(position)
            if position==None:
                self.AppendPage(info,array)
            else:
                self.InsertPage(info,array,position)
                position=position+1
        if invalidate: self.Invalidate()


    def Invalidate(self,page_list="ALL"):
        """
        This method have to be called from the derived classes when there's a change
        in the contents of data, in order to trigger the update events in all views
        and select objects related to this data.
        """
        self.eh.event(self.DataChange, (self,page_list))     


    def Refresh(self):
        """
        The application calls this method in order to poll for data changes.
        """
        ChangedPageList=[]
        for page in range(self.GetNumberPages()):
            info=self.GetPageInfo(page)
            if "_RefreshFunction" in info.keys():
                if (info["_RefreshFunction"](self,page)):
                    ChangedPageList.append(page)
        if len (ChangedPageList): self.Invalidate(ChangedPageList)
        

    def GetPageListIndex(self,index):
        """
        Converts a physical or logical index, into a physical one
        """
        if type(index) is not  types.DictType: return index
        for i in range(self.GetNumberPages()):
            found = 1
            for key in index.keys():
                if key not in self.Pages[i].Info.keys() or self.Pages[i].Info[key] != index[key]:
                    found=0
                    break
            if found: return i
        return None



#################################
#   VIRTUALS
#################################

    def RefreshPage (source_obj,self,page):        
        """
        Virtual method, implements seeking for changes in source for
        "page".
        Returns non-zero if the page was changed.
        If not implemented in the derived class, this class doesn't
        support dinamic changes monitoring.
        
        As pages can be copied to different Data objects, and can
        store the original RefreshPage method for updating, source_obj
        refers to the object that was origin of the page data, while
        self indicates the object that actually owns the page
        with index page.
        It was done this way because if it is stored the reference to
        the unbound method, python doesn't allow you to call it with
        an object of different data type.

        Important:
        Derived classes shall update the page:   self.Pages[page]
        but not:   source_obj.Pages[page]
        """
        return 0


    def SetSource (self,source_name=None):
        """
        Virtual method, sets a new source for data retrieving.
        Parameters:
        source_name: name of the source 

        Important: To be overridden 
        The default implementation works with a NumPy array
        """
        self.SourceName=source_name
        self.Source=source_name


    def GetSourceInfo (self):
        """
        Virtual method, Returns information about source, to give
        application possibility to know about it before loading.
        Returns a dictionary. The minimum suggested keys in this
        dictionary are "Size" (number of possible  keys to this
        source) and "KeyList" (list of all available keys in this
        source).

        Important: To be overridden 
        The default implementation works with a NumPy array
        """
        if self.SourceName == None: return None
        source_info["Size"]=1
        source_info["KeyList"]=(0,)


    def LoadSource (self,key_list="ALL",append=0,invalidate=1):
        """
        Virtual method, creates a given number of pages, getting data
        from the actual source (set by SetSource)
        Parameters:
        key_list: list of all keys to be read from source
        append: If non-zero appends to the end of page list.
                Otherwise, initializes the page list                
        invalidate: if non-zero performs an invalidate call after
                    loading

        Important: To be overridden 
        The default implementation works with a NumPy array
        """        
        if append==0: Data.Delete(self)
        info={}
        info["SourceType"]=SOURCE_TYPE
        info["SourceName"]=self.SourceName
        info["Key"]=0
        info["Source"]=self.Source

        info["Shape"]=self.Source.shape
        self.AppendPage(info,self.Source)
        if invalidate: self.Invalidate()

#################################
#   CONVENIENCE METHODS
#################################

    def GetSourceLoadedNames(self):
	""" Return a list of all different source name loaded
	"""
        names= []
        for i in range(self.GetNumberPages()):
            source=self.GetItemPageInfo("SourceName",i)
            if (source is not None) and (source not in names):
                 names.append(self.Pages[i].Info["SourceName"]) 
        return names

    def GetSourceLoadedKeys(self, source_name):
        """
        Returns the keys loaded for a given source name.
        """
        keys= []
        for i in range(self.GetNumberPages()):
            if self.GetItemPageInfo("SourceName",i)==source_name:
                key=self.GetItemPageInfo("Key",i)
                if key is not None: keys.append(key)
        return keys
    



        



