/*+++
1 sphere2saxs
  Generates the specific scattering cross section of diluted spheres

2 PURPOSE
  Generates either the small angle X-ray scattering cross section 
  1/A dSigma/dOmega of a solution of diluted spheres (+norm) or
  creates the expectation values (DeltaIs) of photons for each 
  detector pixel (-norm).
  In the latter case the number of incoming photons and the 
  sample transmission is required. To model a detector pattern
  the not normalized image must be additionally multiplied by 
  the detector sensitivity (flat field). The result can be
  fed into a poissonian noise generator (saxs_poisson) to 
  generate a detector pattern. 

  METHOD
  This approach is only valid for diluted particles. 
  
  +norm (default):

  (1/A * dSigma/dOmega) = 1/V * (dSigma/dOmega) * d 
                        = 1/V * (N * re^2 * DeltaNe^2 * |f|^2) * d

   -norm:
   DeltaIs = I0 * (1/A * dSigma/dOmega) * DeltaOmega

   V             : scattering volume
   dSigma/dOmega : differential scattering cross section
   d             : thickness
   re            : classical electron radius
   N = V * RhoS  : total number of spheres in the scattering volume V
   DeltaNe = Vs*DeltaRhoElektron : number of excess electrons in the 
                                   scattering particles compared to 
                                   the density of the surrounding 
                                   matrix. 
   For spheres:

   Vs = (4/3)*pi*R^3 : volume of a spherical particle with radius R

   |f|^2 = [ 3 * (sin(q*R) - (q*R)*cos(q*R))/(q*R)^3 ]^2

   All together:
   
   1/A*(dSigma/dOmega) = n * re^2 * DeltaRhoElektron^2 * Vs^2 * |f|^2 * d
      with Vs and |f|^2 as defined above 

   Parameters:

   DeltaRhoElektron :      electron volume density difference 
                      -dre <DeltaRhoElektron [#electrons/m^3]>
   R                :      radius of spheres
      mono disperse         -rad <R [m]>   
      Schultz distribution  -rad Schultz,<R [m]>,<Z>  with Z>-1
   vfr = n * Vs 
       = (N/V) * Vs :      volume fraction of particles 
                           (N: number of particles, V: scattering volume,
                            Vs: average volume of particles)
                      -vfr <vfr>
   d                :      thickness of scattering volume
                      -d   <d [m]>

   Hint: To give the numbers in nanometers type <number>/_nm, e.g.
         -rad 100*_nm sets the radius to 10 nm
         -dre 10/_nm/_nm/_nm

  Arguments:
  sphere2saxs [options] <onam> <odum> <odim 1> <odim 2> 
                        <opix 1> <opix 2> <odis> <owvl>
                        <dre> <rad> <vfr> <d>
  Defaults:

  output file name            : <onam>              : output.edf
  output dummy                : <odum>              : 0.0
  dimension 1                 : <odim 1>            : 512
  dimension 2                 : <odim 2>            : 512
  pixel size 1                : <opix 1>            : 100*_um
  pixel size 2                : <opix 2>            : 100*_um
  center 1                    : <ocen 1>            : half of dim 1 
  center 2                    : <ocen 2>            : half of dim 2 
  sample distance             : <odis>              : 10*_m
  wavelength                  : <owvl>              : 0.1*_nm

  electron density difference : <dre>               : 10/_nm/_nm/_nm
  radius of spheres or radius
    distribution function     : <rad>               : 100*_nm
  volume fraction             : <vfr>               : 0.01
  scattering volume thickness : <d>                 : 100*_um

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2000-11-25  PB from saxs_new
  2000-12-04  PB V3.1 -rad <radius or radius distribution function>
  2000-12-22  PB V3.2 f2_schultz
  2000-12-27  PB V3.3 normalization to volume fraction -vfr
  2001-01-10  PB V3.1 GetReference with output image after geometric setup
  2001-07-09  PB V4.00 new loop, no repetition
  2001-08-19  PB V4.01 normalization keywords are written
  2001-09-16  PB V4.02 no default for dummy


---*/
# define Version  "sphere2saxs V4.02 2001-09-16, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "schultz.h"

# define Usage "[options] \n\
             <onam> <odum> <odim 1> <odim 2> <opix 1> <opix 2> <odis> <owvl>\n\
             <dre> <rad> <vfr> <d>\n\
             <norm> <monv> <trm> <timv>\n\
             Generates the scattering cross section of diluted spheres\n\
             Options: -dre, -rad, -vfr, -d"

# define BlockNum 1       /* 1 output sequence */

/* float options */
# define NDre 0 // DeltaRhoElektron [1/m^3]
# define NVfr 1 // Volume fraction of particles 
# define ND   2 // Thickness of scattering volume [m]
# define NI0  3 // Incoming number of photons [ph]
# define NTrm 4 // Transmission
# define NTim 5 // Measurement time [s], info only, not used for calculations

/* line options */
# define NRad 0 // Radius Distribution Function 

/* flag options */
# define NNorm 0 // +norm/-norm : output of normalized/unnormalized data

enum DistributionType { InValidDistributionType,
                        MonoDisperse=1, Schultz,
                        EndDistributionType };

PRIVATE const char * DistributionTypeStrings[4] =
                      { "Invalid", 
                        "MonoDisperse", "Schultz",
                        (const char *) NULL };

/*---------------------------------------------------------------------------
calc_len
---------------------------------------------------------------------------*/
float calc_len( float A[] )
{ return(sqrt(A[1]*A[1]+A[2]*A[2]));
} /* calc_len */

/*---------------------------------------------------------------------------
maximum_radial_distance

   Returns the maximum distance (in [m]) between the center and one of the 
   four edges of the input array.
---------------------------------------------------------------------------*/
float maximum_radial_distance( CmdBlk * pcb, ImgHeadBlk ihb[], int blkno,
                               int * pstatus )
{ ImgHeadBlk * pihb;
  float Off_1, Off_2;
  float Ps_1, Ps_2;
  float edge1[3], edge2[3], edge3[3], edge4[3];
  float distance;
  float i_10, i_11, i_20, i_21;

  *pstatus = Success;

  pihb = &ihb[blkno];

  if (!( (pihb->Center[1].I) && (pihb->Center[2].I) ) ) {
    *pstatus = Failed; printf("Missing center definitions\n"); return( 0l );
    } 

  if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) ) ) {
    *pstatus = Failed; printf("Missing pixel size definitions\n"); return( 0l );
    } 

    NORMALREF(Off_1,Ps_1,pihb->Offset[1].V,pihb->PixSiz[1].V,pihb->Center[1].V);
    NORMALREF(Off_2,Ps_2,pihb->Offset[2].V,pihb->PixSiz[2].V,pihb->Center[2].V);

    i_10 = i_20 = A2INDEX(ARRAYSTART+LOWERBORDER);
    i_11 = A2INDEX(ARRAYSTART+LOWERBORDER+pihb->Dim[1]);
    i_21 = A2INDEX(ARRAYSTART+LOWERBORDER+pihb->Dim[2]);

    /* edge 1 */
    edge1[1] = WORLD(i_10,Off_1,Ps_1);
    edge1[2] = WORLD(i_20,Off_2,Ps_2);

    edge2[1] = WORLD(i_11,Off_1,Ps_1);
    edge2[2] = edge1[2];

    edge3[1] = edge2[1];
    edge3[2] = WORLD(i_21,Off_2,Ps_2);

    edge4[1] = edge1[1];
    edge4[2] = edge3[2];

    distance = MAX4( calc_len(edge1), calc_len(edge2),
                     calc_len(edge3), calc_len(edge4) );

  return ( distance );

} /* maximum_radial_distance */

/*---------------------------------------------------------------------------

   diluted_spheres

   Creates 1/A dSigma/dOmega of a system of diluted spheres 

   d/V*(dSigma/dOmega) = n * re^2 * DeltaRhoElektron^2 * Vs^2 * |f|^2 * d
   
   with 

   Vs = (4/3)*pi*R^3 : volume of a spherical particle with radius R

   |f|^2 = [ 3 * (sin(q*R) - (q*R)*cos(q*R))/(q*R)^3 ]^2

   dSigma/dOmega     : differential scattering cross section
   V                 : scattering volume 
   re                : classical electron radius

   n = N/V           : density of particles


 PARAMETERS
   float radial[]        : radial output buffer
   long dim              : length of output buffer
   float off             : offset
   float dummy           : dummy value
   float ddummy          : ddummy value
   float pix             : radial pixel size [m]
   float cen             : center
   float dis             : sample distance [m]
   float wvl             : wavelength [m]
   float DeltaRhoElektron:electron volume density difference [#electrons/m^3]
   float R               : radius of spheres [m]
   float vfr             : volume fraction of particles
   float d               : thickness scattering volume [m]
   float weight          : weight factor, all values are multiplied by weight
   int initialize        : 1: initializes radial[] with 0 and adds values
                           0: adds values to radial[] without initializing

 RETURN VALUE
   0: success
   1: failed 
---------------------------------------------------------------------------*/
int diluted_spheres( float radial[], long dim, float off, 
                     float dummy, float ddummy, float pix,
                     float cen, float dis, float wvl, 
                     float DeltaRhoElektron, float R, float vfr, float d,
                     float weight, int initialize, int debug )
{ 
   const float eps = 1e-10;
   const double pi = 3.1415926535897932384626;
   const double re = 2.8179380e-15;   /* classical electron radius (m) */

   long i;
   float Off, Ps;
   float qR;
   float radius; // radius around origin 
   float f, f2;
   float n;
   float factor1, factor2;
   float value;

   if (debug) {
     printf("off              = %g\n", off);
     printf("dummy            = %g\n", dummy);
     printf("ddummy           = %g\n", ddummy);
     printf("pix              = %g\n", pix);
     printf("cen              = %g\n", cen);
     printf("dis              = %g\n", dis);
     printf("wvl              = %g\n", wvl);
     printf("DeltaRhoElektron = %g\n", DeltaRhoElektron);
     printf("R                = %g\n", R);
     printf("vfr              = %g\n", vfr);
     printf("d                = %g\n", d);
     printf("initialize       = %s\n", initialize?"yes":"no");
     }

   n = vfr/((4.0/3.0)*pi*R*R*R);

   factor1 = weight *
             n * 
             re*re * 
             DeltaRhoElektron*DeltaRhoElektron * 
             ((4.0/3.0)*pi*R*R*R)*((4.0/3.0)*pi*R*R*R) *
             d;
   factor2 = 4*pi*R/wvl;

   /* Set the radial reference system to normal */
     NORMALREF(Off,Ps,off,pix,cen);


   if (initialize) for (i=0;i<dim;i++) radial[i] = dummy;

   for (i=0;i<dim;i++) {
     radius = WORLD(i,Off,Ps);
     if (fabs(radius)<eps) f2 = 1.0;
      else { 
       qR = factor2*sin(atan(radius/dis*0.5));
       f  = 3.0 * ( sin(qR) - (qR)*cos(qR) ) / ( qR*qR*qR );
       f2 = f*f;
       }
     value = factor1 * f2;

     UPDATE( radial[i], value, dummy, ddummy );
     }

   if (debug) {
     printf("%15s %15s %15s\n","i","radius","radial");
     for (i=0;i<dim;i++) {
       radius = WORLD(i,Off,Ps);
       printf("%15d %15g %15g\n",i,radius,radial[i]);
       }
     }

   return(1);

} /* diluted_spheres */

/*---------------------------------------------------------------------------

   diluted_schultz

   Creates 1/A dSigma/dOmega of a system of diluted schultz-distributed spheres

   d/V*(dSigma/dOmega) = n * re^2 * DeltaRhoElektron^2 * Vs^2 * f2_schultz * d

   with

   Vs = (4/3)*pi*R^3 : volume of a spherical particle with radius R

   R                 : average radius
   Z                 : width parameter of Schultz distribution
                       sigma = <R>/sqrt(Z+1)

   f2_schultz        : schultz averaged square of form factor of spheres with 
                       average radius R

   dSigma/dOmega     : differential scattering cross section
   V                 : scattering volume
   re                : classical electron radius

   vfr               : volume fraction of particles

 PARAMETERS
   float radial[]        : radial output buffer
   long dim              : length of output buffer
   float off             : offset
   float dummy           : dummy value
   float ddummy          : ddummy value
   float pix             : radial pixel size [m]
   float cen             : center
   float dis             : sample distance [m]
   float wvl             : wavelength [m]

   float DeltaRhoElektron:electron volume density difference [#electrons/m^3]
   float R               : average radius of spheres [m]
   float Z               : width parameter of Schultz distribution
   float vfr             : volume fraction of particles
   float d               : thickness scattering volume [m]
   float weight          : weight factor, all values are multiplied by weight
   int initialize        : 1: initializes radial[] with 0 and adds values
                           0: adds values to radial[] without initializing

 RETURN VALUE
   0: success
   1: failed
---------------------------------------------------------------------------*/
int diluted_schultz( float radial[], long dim, float off, 
                     float dummy, float ddummy, float pix,
                     float cen, float dis, float wvl, 
                     float DeltaRhoElektron, float R, float Z, 
                     float vfr, float d,
                     float weight, int initialize, int debug )
{
   const float eps = 1e-10;
   const double pi = 3.1415926535897932384626;
   const double re = 2.8179380e-15;   /* classical electron radius (m) */

   long i;
   float Off, Ps;
   float qR;
   float radius; // radius around origin
   float f2;
   float n;
   float factor1, factor2;
   float value;

   if (debug) {
     printf("off              = %g\n", off);
     printf("dummy            = %g\n", dummy);
     printf("ddummy           = %g\n", ddummy);
     printf("pix              = %g\n", pix);
     printf("cen              = %g\n", cen);
     printf("dis              = %g\n", dis);
     printf("wvl              = %g\n", wvl);
     printf("DeltaRhoElektron = %g\n", DeltaRhoElektron);
     printf("R                = %g\n", R);
     printf("Z                = %g\n", Z);
     printf("vfr              = %g\n", vfr);
     printf("d                = %g\n", d);
     printf("initialize       = %s\n", initialize?"yes":"no");
     }

   n = vfr/( (4.0/3.0)*pi*R3_schultz(R,Z) );

   factor1 = weight *
             n *
             re*re *
             DeltaRhoElektron*DeltaRhoElektron *
             ((4.0/3.0)*pi*R*R*R)*((4.0/3.0)*pi*R*R*R) *
             d;

   factor2 = 4*pi*R/wvl;

   /* Set the radial reference system to normal */
     NORMALREF(Off,Ps,off,pix,cen);

   if (initialize) for (i=0;i<dim;i++) radial[i] = dummy;

   for (i=0;i<dim;i++) {
     radius = WORLD(i,Off,Ps);
     if (fabs(dis)>eps) {
       qR = factor2*sin(atan(radius/dis*0.5));
       f2 = f2_schultz( Z, qR );
       if (f2>=0.0) {
         value = factor1 * f2;

         UPDATE( radial[i], value, dummy, ddummy );
         }
       }
     }

   if (debug) {
     printf("%15s %15s %15s\n","i","radius","radial");
     for (i=0;i<dim;i++) {
       radius = WORLD(i,Off,Ps);
       printf("%15d %15g %15g\n",i,radius,radial[i]);
       }
     }

   return(1);

} /* diluted_schultz */

/*---------------------------------------------------------------------------
1 unnormalize

2 PURPOSE
  Unnormalizes the absolute radial scattering pattern 

      1/A * dSigma/dOmega

  in radial[] by multiplicating each point with the number of transmitted 
  photons (I1) and by multiplicating each point with the covered spherical 
  angle DeltaOmega:

      I1 * (pix_area)/(L^2*cos(2Theta)) = pix_area*dis/L^3.            


  The result is the average number DeltaIs of scattered photons per pixel:

      DeltaIs = I0 * (1/A * dSigma/dOmega) * DeltaOmega

 PARAMETERS
   float radial[]        : radial output buffer
   long dim              : length of output buffer
   float off             : offset
   float dummy
   float ddummy
   float pix             : radial pixel size [m]
   float cen             : center
   float dis             : sample distance [m]
   float pix_area        : pixel area of the 2d detector (=pix_1*pix_2)
   float I1              : transmitted number of photons [ph]

 RETURN VALUE
   0: success
   1: failed

---------------------------------------------------------------------------*/
int unnormalize( float radial[], long dim, float off, 
                 float dummy, float ddummy,
                 float pix, float cen, float dis,
                 float pix_area, float I1,
                 int debug )
{ 
  int i;
  float factor1, factor2;
  float Off, Ps;
  float radius; // radius around origin
  float L, L2; // distance of pixel from detector

   if (debug) {
     printf("off              = %g\n", off);
     printf("dummy            = %g\n", dummy);
     printf("ddummy           = %g\n", ddummy);
     printf("pix              = %g\n", pix);
     printf("cen              = %g\n", cen);
     printf("dis              = %g\n", dis);
     printf("pix_area         = %g\n", pix_area);
     printf("I1               = %g\n", I1);
     }

  /* Set the radial reference system to normal */
  NORMALREF(Off,Ps,off,pix,cen);

  factor1 = I1*pix_area*dis;
  factor2 = dis*dis;

  for (i=0;i<dim;i++) {
    radius = WORLD(i,Off,Ps);
    L2 = radius*radius + factor2;
    L  = sqrt(L2);
    if ( NODUMMY( radial[i], dummy, ddummy ) )
      radial[i] *= factor1/(L*L2);
    }

  return(1);

} /* unnormalize */

/*---------------------------------------------------------------------------
NAME

  string2distributiontype --- converts a string to DistributionType 

SYNOPSIS

  (DistributionType) int string2distributiontype( char * string );

DESCRIPTION
  The input string might be changed (blanks removed etc)

RETURN VALUE
  DType == 0 : error, e.g. cannot convert
  DType >  0 : valid distribution type value

AUTHOR
2000-12-04  PB
  --------------------------------------------------------------------------*/
int string2distributiontype( char * string )
{ int  NE=1;
  int  i = 0;
  int  status;

  while ( (NE && DistributionTypeStrings[i]) ) {
    NE = strcmp(string,DistributionTypeStrings[i++]);
     }

  i = MAX2(0,i-1);

  if (NE) {
     (void) floatexpr( string, &status ); 
     if (status==Success) NE=0;
     i = MonoDisperse;
     }

  if (NE) i = InValidDistributionType;

  return( i );

} /* string2distributiontype */

/*---------------------------------------------------------------------------
1 sphere2saxs

2 PURPOSE
  Generates a new image with constant pixel values
---------------------------------------------------------------------------*/
void sphere2saxs (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  float W_12, W_22;
  float dre, rad, vfr, d;
  float arg1, arg2;
  char  *arg;
  int   distribution_type;
  float I0, trm, time;
  long  norm;
  float pixsiz, center = ARRAYSTART, offset = 0.0;
  float f_r;
  long radial_dim;
  float * radial;
  float radius;

  float dummy = -1.0;
  float ddummy = DDSET(dummy);
  float SampleDistance, WaveLength;
  char  line1[InputLineLength],
        line2[InputLineLength];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=1) {
     printf("%d images found, 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     pcb->RSys.V = IO_Normal;

 /* Set the angular reference system to normal */
   NORMALREF(Off_1[0],Ps_1[0],ihb[0].Offset[1].V,ihb[0].PixSiz[1].V,\
             ihb[0].Center[1].V);
   NORMALREF(Off_2[0],Ps_2[0],ihb[0].Offset[2].V,ihb[0].PixSiz[2].V,\
             ihb[0].Center[2].V);

   GetImageOverlap       ( pcb, ihb, 1, imax-1,
                           Off_1, Off_2, Ps_1,  Ps_2,
                           fmin_1, fmax_1, fmin_2, fmax_2,
                           &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

 /* loop over 1 output image */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  pixsiz         = MIN2(ihb[0].PixSiz[1].V,ihb[0].PixSiz[2].V);
  SampleDistance = ihb[0].SampleDistance.V;
  WaveLength     = ihb[0].WaveLength.V;
  dre            = pcb->Arg[NDre].V;
  arg            = pcb->Lin[NRad].V;
  vfr            = pcb->Arg[NVfr].V;
  d              = pcb->Arg[ND].V;
  I0             = pcb->Arg[NI0].V;
  trm            = pcb->Arg[NTrm].V;
  time           = pcb->Arg[NTim].V;
  norm           = pcb->Flg[NNorm].V;

  radial_dim = ceil( maximum_radial_distance(pcb,ihb,0,pstatus)/pixsiz ) + 1;
  if (!(radial = malloc( sizeof(float) * radial_dim ))) {
     *pstatus=NotEnoughMemoryAvailable; return;
     }

  LeftFL(arg,",",line1,InputLineLength);
  distribution_type = string2distributiontype( line1 );

  switch (distribution_type) {
    case MonoDisperse : // R
      rad  = floatexpr( line1, pstatus ); if (*pstatus!=Success) return; 

      if (!ihb[0].Title.I) {
        sprintf(ihb[0].Title.V,
              "Spheres: R=%g nm, vfr=%g, dre=%g e/nm^3, d=%g mm",
              rad*1e9, vfr, dre*1e-27, d*1e3 ); 
        ihb[0].Title.I = TRUE;
        }

      diluted_spheres( radial, radial_dim, offset, dummy, ddummy,
                       pixsiz, center, SampleDistance, WaveLength,
                       dre, rad, vfr, d , 1.0, TRUE, pcb->TestBit); 
      break;

    case Schultz : // "Schultz", R, Z
      RightFL(arg,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      arg1  = floatexpr( line2, pstatus ); if (*pstatus!=Success) return;

      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      arg2  = floatexpr( line2, pstatus ); if (*pstatus!=Success) return;

      if (!ihb[0].Title.I) {
        sprintf(ihb[0].Title.V,
               "Schultz: R=%g nm, Z=%g, vfr=%g, dre=%g e/nm^3, d=%g mm",
               arg1*1e9, arg2, vfr, dre*1e-27, d*1e3 );
        ihb[0].Title.I = TRUE;
        } 

      diluted_schultz( radial, radial_dim, offset, dummy, ddummy,
                       pixsiz, center, SampleDistance, WaveLength,
                       dre, arg1, arg2, vfr, d , 1.0, TRUE, pcb->TestBit);
      break;

    default :
       printf("Unknown distribution function : %s\n", arg); 
       *pstatus=Failed; return;

    }

  if (!norm) {
    unnormalize( radial, radial_dim, offset, dummy, ddummy,
                 pixsiz, center, SampleDistance,
                 ihb[0].PixSiz[1].V*ihb[0].PixSiz[2].V, I0*trm,
                 pcb->TestBit );
    }

  /* loop over the output array  */
  for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
    W_2 = WORLD( i_2, Off_2[0], Ps_2[0] );
    W_22 = W_2 * W_2;
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1[0],i_2);
    for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
        W_1 = WORLD( i_1, Off_1[0], Ps_1[0] );
        W_12 = W_1 * W_1;

        /* do something with the data */
        radius = sqrt(W_12+W_22);
        f_r = N2INDEX(radius, offset, pixsiz, center);

        if (Ipol2ld    ( radial, radial_dim, 1,
                         dummy, ddummy, f_r, 0, &Value) )
          UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

      pI0Data++;
      } /* for i_1 ... */
    } /* for i_2 ... */

   if (radial) free(radial);

   if (!norm) { // write intensity and time into header
     WriteHeaderFloat(pcb,0,ihb,KValueI0,I0, pstatus); 
     if (*pstatus!=Success) return; 
     WriteHeaderFloat(pcb,0,ihb,KValueI1,I0*trm, pstatus);
     if (*pstatus!=Success) return;
     WriteHeaderFloat(pcb,0,ihb,KValueTime,time, pstatus);
     if (*pstatus!=Success) return;
     }

 } /* sphere2saxs*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : pixel size 1 */
  argv_float(pcb,"Pixel size 1 [m]",&ib[0].PixSiz[1],100e-6,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : pixel size 2 */
  argv_float(pcb,"Pixel size 2 [m]",&ib[0].PixSiz[2],100e-6,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : center 1 */
  argv_float(pcb,"Center 1 [pixel coordinate]",
                 &ib[0].Center[1],SETCTRDEF(ib[0].Dim[1].V),pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : center 2 */
  argv_float(pcb,"Center 2 [pixel coordinate]",
                 &ib[0].Center[2],SETCTRDEF(ib[0].Dim[2].V),pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : sample distance */
  argv_float(pcb,"Sample distance [m]",&ib[0].SampleDistance,10.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : wavelength */
  argv_float(pcb,"Wavelength [m]",&ib[0].WaveLength,0.1e-9,pstatus);
  if (*pstatus!=Success) return;

  /* --- End of geometric setup */
  GetReference(pcb->RSys.V,0,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,0,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : DeltaRhoElektron */
  argv_float(pcb,"Electron density diff. between particles and matrix [#/m^3]",
    &pcb->Arg[NDre],pcb->Arg[NDre].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : Radius of particles */
  printf(" Particle radius distribution:\n");
  printf("   mono disperse        : radius [m]\n");
  printf("   Schultz distribution : \"Schultz, radius [m], Z\" with Z>-1\n");
  argv_line(pcb,"Particle radius [m] or distribution",
    &pcb->Lin[NRad],pcb->Lin[NRad].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : Particle density */
  argv_float(pcb,"Volume fraction",
    &pcb->Arg[NVfr],pcb->Arg[NVfr].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : Sample thickness */
  argv_float(pcb,"Thickness of scattering volume [m]",
    &pcb->Arg[ND],pcb->Arg[ND].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : Normalization? */
  argv_flag(pcb,"Should the data be normalized?",
    &pcb->Flg[NNorm],pcb->Flg[NNorm].V,pstatus);
  if (*pstatus!=Success) return;

  if (!pcb->Flg[NNorm].V) {
    /*--- Argument  : I0 */
    argv_float(pcb,"Incoming number of photons [ph]",
      &pcb->Arg[NI0],pcb->Arg[NI0].V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument  : Trm */
    argv_float(pcb,"Sample transmission",
      &pcb->Arg[NTrm],pcb->Arg[NTrm].V,pstatus);
    if (*pstatus!=Success) return;
    }

  printf("\n");
  if (ib[0].Name.I)           printf("o/p file           : %s\n",
    ib[0].Name.V);
  if (ib[0].Dummy.I)          printf("output dummy       : %g\n",
    ib[0].Dummy.V);
  if (ib[0].Dim[1].I)         printf("output dimension 1 : %d\n",
    ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)         printf("output dimension 2 : %d\n",
    ib[0].Dim[2].V);
  if (ib[0].PixSiz[1].I)      printf("pixel size 1       : %g m\n",
    ib[0].PixSiz[1].V);
  if (ib[0].PixSiz[2].I)      printf("pixel size 2       : %g m\n",
    ib[0].PixSiz[2].V);
  if (ib[0].Center[1].I)      printf("center 1           : %g\n",
    ib[0].Center[1].V);
  if (ib[0].Center[2].I)      printf("center 1           : %g\n",
    ib[0].Center[2].V);
  if (ib[0].SampleDistance.I) printf("sample distance    : %g m\n",
    ib[0].SampleDistance.V);
  if (ib[0].WaveLength.I)     printf("wavelength         : %g m\n",
    ib[0].WaveLength.V);
  if (pcb->Arg[NDre].I)       printf("electr. dens. diff.: %g /m^3\n",
    pcb->Arg[NDre].V);
  if (pcb->Arg[NRad].I)       printf("particle radius    : %s\n",
    pcb->Lin[NRad].V);
  if (pcb->Arg[NVfr].I)       printf("volume fraction    : %g\n",
    pcb->Arg[NVfr].V);
  if (pcb->Arg[ND].I)         printf("sample thickness   : %g m\n",
    pcb->Arg[ND].V);
  if (pcb->Flg[NNorm].I)      printf("normalization      : %s\n",
    (pcb->Flg[NNorm].V)?"yes":"no");
  if (pcb->Arg[NI0].I)        printf("incoming photons   : %g ph\n",
    pcb->Arg[NI0].V);
  if (pcb->Arg[NTrm].I)       printf("transmission       : %g\n",
    pcb->Arg[NTrm].V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

 /* Define options */
  DefFloatOption( &cb, "dre",    NDre);
  cb.Arg[NDre].V = 10.0/1e-9/1e-9/1e-9;     /* default */
  DefFloatOption( &cb,  "vfr",   NVfr);
  cb.Arg[NVfr].V = 0.01;                    /* default */
  DefFloatOption( &cb,   "d",    ND);
  cb.Arg[ND].V = 100.0*1e-6;                /* default */
  DefFloatOption( &cb,  "monv",   NI0);
  cb.Arg[NI0].V = 1e13;                     /* default */
  DefFloatOption( &cb,  "trm",   NTrm);
  cb.Arg[NTrm].V = 1.0;                     /* default */
  DefFloatOption( &cb,  "timv",  NTim);
  cb.Arg[NTim].V = 1.0;                     /* default */

  DefLineOption( &cb, "rad",    NRad);
  sprintf(cb.Lin[NRad].V,"1e-7");           /* default */

  DefFlgOption( &cb, "norm",   NNorm);
  cb.Flg[NNorm].V = TRUE;                   /* default */

  ib[0].Dim[1].V = 512;
  ib[0].Dim[2].V = 512;

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, sphere2saxs, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

