/*+++
1 saxs_tiff
  Transformation of an edf image sequence into 1byte/2byte/4byte tiff format

2 PURPOSE
  Transformation of an edf image sequence into 1byte/2byte/4byte tiff format
  (default 2 byte tiff)

  Arguments:
  saxs_tiff [options] 
            <i1nam> <tiff file prefix> <i1fst> <i1lst> <i1inc> 
            <odum> <odim1> <odim2> <8|16|32-bits> <orientation>
            <auto> [<min> <max>]
  Defaults:
  <input file name>   : input.edf 
  <tiff file prefix>  : input.tif (default conversion from input filename)
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <bps>               : 16 (number of bits per pixel in output file) 
  <orientation>       : orientation of output image (default: 3)
  <auto>              : autoscaling on/off (default on)
  <min>               : 0 (minimum of mapping region)
  <max>               : 2^bps-1 (maximum of mapping region)

  Specific arguments:
  -pref 'prefix'      : specify the prefix of the ouptput file

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  1999-10-31  PB from saxs_bsl V3.1
  1999-11-12  PB V3.0
  1999-12-21  PB V3.01 output file name without number 
                       if only 1 image is written
  1999-12-25  PB V3.02 default output file name without path
  2000-08-04  PB %f->%g
  2001-07-07  PB V3.03 IO_DontReadWrite
  2001-07-08  PB V3.04 FFirst
  2001-07-09  PB V3.05 (pcb->ib)[0].FileNumber
  2001-07-09  PB V4.00 new loop, no repetition
  2001-08-30  PB V4.01 

---*/
# define Version  "saxs_tiff 4.01 2001-08-30, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "edftiff.h"

/* Flg */
# define SAuto   "auto"
# define NAuto   0
/* Lin */
# define SPrefix "tnam"
# define NPrefix 0
/* Num */
# define SBps    "bps"
# define NBps    0
# define SRori   "rori"
# define NRori   1

/* Val */
# define SMin    "min"
# define NMin    0
# define SMax    "max"
# define NMax    1

# define Usage "[options] \n\
                <i1nam> <tiff file name -tnam> <i1fst> <i1lst> <i1inc>\n\
                <odum> <odim1> <odim2> <8|16|32-bps> <1-8-rori>\n\
                <auto> [<min> <max>]\n\n\
  Transformation of an edf image sequence to 1byte/2byte/4byte tiff images\n\
  -bps <8|16|32> : 8, 16 and 32 bits in tiff image (default 16)\n\
  -rori          : orientation of tiff image\n\
                   1: (x,y)\n\
                   2: (-x,y)  x-axis inverted\n\
                   3: (x,-y)  y-axis inverted (default)\n\
                   4: (-x,-y) both axes inverted\n\
                   5: (y,x)   axes swapped\n\
                   6: (y,-x)  axes swapped and x-axis inverted\n\
                   7: (-y,x)  axes swapped and y-axis inverted\n\
                   8: (-y,-x) axes swapped and both axes inverted\n\
  +auto          : automatic scaling between minimum and maximum,\n\
                   output dummy is set to upper range value,\n\
                   minimum and maximum are calculated, if not set\n\
                   (default +auto)\n\
  -min <minimum of mapping range>, -max <maximum of mapping range>\n"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
NAME

   tiffnewstr --- allocate memory and copy a character string into it

SYNOPSIS

   char * tiffnewstr( const char * string );

DESCRIPTION
  Allocates strlen(string)+1 bytes of memory and copies string into it.
  In case of success the pointer to the allocated memory is returned. The
  null pointer is returned in case of an error.
  If string is the NULL pointer the NULL pointer is returned.

RETURN VALUE
  Returns the pointer to the allocated string or (char *) NULL in case
  of an error.
---------------------------------------------------------------------------*/
char * tiffnewstr( const char * string )
{ char * newstring;

  if (!string) return( (char *) NULL );
  if (!(newstring = (char *) malloc(strlen(string)+1))) return((char *) NULL);
  (void) strcpy(newstring, string);

  return( newstring );

} /* tiffnewstr */

/*---------------------------------------------------------------------------
NAME
  serialfilename --- create a file name with prefix, number and extension.

DESCRIPTION
  Combines a file name from prefix, number and extension according to
  <prefix><num><extension>

SYNOPSIS
  char * serialfilename(char * buffer, const char * prefix,
                long num, const char * extension );

ARGUMENTS
  char       * buffer    : buffer, must be sufficiently long
  const char * prefix    : prefix string
  long         num       : serial number
  const char * extension : extension string including separator, e.g. dot

RETURN VALUE
  char * filename
---------------------------------------------------------------------------*/
char * serialfilename(char * buffer, const char * prefix,
                long num, const char * extension )
{      if (num<=-1000)sprintf( buffer,   "%s-%4u%s", prefix,-num, extension );
  else if (num<=-100) sprintf( buffer,  "%s-0%3u%s", prefix,-num, extension );
  else if (num<=-10 ) sprintf( buffer, "%s-00%2u%s", prefix,-num, extension );
  else if (num<=-1  ) sprintf( buffer,"%s-000%1u%s", prefix,-num, extension );
  else if (num>=1000) sprintf( buffer,   "%s+%4u%s", prefix, num, extension );
  else if (num>=100 ) sprintf( buffer,  "%s+0%3u%s", prefix, num, extension );
  else if (num>=10  ) sprintf( buffer, "%s+00%2u%s", prefix, num, extension );
  else                sprintf( buffer,"%s+000%1u%s", prefix, num, extension );

  return( buffer );
} /* serialfilename */

/*---------------------------------------------------------------------------
NAME
  getprefix --- return file name prefix

DESCRIPTION
  returns the string before the last occurrence of .. If filename does
  not contain a . the full filename is returned as prefix. Memory
  is allocated for prefix. It must be released with free().

SYNOPSIS
  char * getprefix( const char * filename );

ARGUMENTS
  const char * filename : string to be analysed (prefix.extension)

RETURN VALUE
  char * prefix  (NULL, if filename too long)
---------------------------------------------------------------------------*/
char * getprefix( const char * filename )
{ char buffer[InputLineLength];
  size_t str_len;
  char *ps;

  str_len = strlen(filename);
  if (str_len>=InputLineLength) return( (char *) NULL );

  if (ps = strrchr( filename, (int) '.' )) {
    memcpy( buffer, filename, (ps-filename) );
    buffer[(ps-filename)]='\0';
    } else {
    memcpy( buffer, filename, (str_len+1) );
    }

  return( tiffnewstr(buffer) );

} /* getprefix */

/*---------------------------------------------------------------------------
NAME
  getextension --- return file name extension

DESCRIPTION
  Returns the string from the last occurrence of .. If filename does
  not contain a . an empty string is returned. Memory is allocated for
  extension. It must be released with free().

SYNOPSIS
  char * getextension( const char * filename );

ARGUMENTS
  const char * filename : string to be analysed (prefix.extension)

RETURN VALUE
  char * extension (NULL, if filename too long)
---------------------------------------------------------------------------*/
char * getextension( const char * filename )
{ char buffer[InputLineLength];
  size_t str_len;
  char *ps;

  str_len = strlen(filename);
  if (str_len>=InputLineLength) return( (char *) NULL );

  if (ps = strrchr( filename, (int) '.' )) {
    memcpy( buffer, ps, (str_len-(ps-filename)) );
    buffer[(str_len-(ps-filename))]='\0';
    } else {
    buffer[0]='\0';
    }

  return( tiffnewstr(buffer) );

} /* getextension */

/*---------------------------------------------------------------------------
Saxs2TiffFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void Saxs2TiffFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[IO_len];
  char Prefix[IO_len];

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib,blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {
    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- File name ib[0].Name of output (not opened) */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",DefaultOutput);
      }

    /*--- Argument  : output tiff file name */
    if (!pcb->Lin[NPrefix].I) {
      /* Determine prefix from input file name */
      if (ib[0].Name.I) {
       filename_body(Prefix,IO_len-5,ib[0].Name.V);
      } else {
       filename_body(Prefix,IO_len-5,filename_name(Prefix,IO_len,ib[1].Name.V));
      }
      (void) sprintf(pcb->Lin[NPrefix].V,"%s%s", Prefix, ".tif");
      pcb->Lin[NPrefix].I = TRUE;
    }

    argv_line( pcb,"Output tiff file",&pcb->Lin[NPrefix],pcb->Lin[NPrefix].V,
               pstatus);
    if (*pstatus!=Success) return;

    /* extract output filenumbers from TIFF-file name */
    extract_filenumbers( &ib[0], &(pcb->Lin[NPrefix]), TRUE, pstatus );
    if (*pstatus!=Success) return;

    /*--- Take prefix as default for output filename */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",pcb->Lin[NPrefix].V);
      }

  } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

    /* Defaults for output file */
    if (!ib[0].Dummy.I) ib[0].Dummy.V   = ihb[1].Dummy.V; 
    if (!ib[0].Dim[1].I) ib[0].Dim[1].V = ihb[1].Dim[1];
    if (!ib[0].Dim[2].I) ib[0].Dim[2].V = ihb[1].Dim[2];
  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;

  } /* all other input files */

  if (block_1st <= 0) {
    /*--- Argument : output dummy */
    argv_float( pcb, "Output dummy", &ib[0].Dummy, ib[0].Dummy.V, pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    argv_long(pcb,"Output dimension 1", &ib[0].Dim[1],ib[0].Dim[1].V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    argv_long(pcb,"Output dimension 2", &ib[0].Dim[2],ib[0].Dim[2].V, pstatus);
    if (*pstatus!=Success) return;

    }

} /* Saxs2TiffFilenames */

/*---------------------------------------------------------------------------
1 saxs_tiff

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
  output to tiff file 
---------------------------------------------------------------------------*/

void saxs_tiff (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy;
  float I0DDummy;
  float *I1Data;
  float *pI1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1=0.0, W_2=0.0;
  float DW_1=0.0, DW_2=0.0;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  char *prefix, *extension;
  char OutputBinaryName[InputLineLength];
  FILE *OutputBinaryId;
  size_t t_length;
  char fname_full[IO_len], Temp[IO_len];

  short bps;
  int   orientation;
  int   autorange, automin, automax;
  float vmin, vmax;
  long  xdim, ydim;
  short compression=1;
  char  *title, *description=(char *) NULL;
  char  comment[InputLineLength];
  float xdpcm, ydpcm;

  void *tiffdata;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\nCalculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

     GetImageOverlap       ( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

 /* special parameters */
  bps         = (short) pcb->Num[NBps].V;
  orientation = (int) pcb->Num[NRori].V;
  autorange   = (int) pcb->Flg[NAuto].V;
  automin     = (int) (!pcb->Arg[NMin].I); 
  automax     = (int) (!pcb->Arg[NMax].I); 
  vmin        = pcb->Arg[NMin].V;
  vmax        = pcb->Arg[NMax].V;
  xdim        = I0Dim_1;
  ydim        = I0Dim_2;
  if (ihb[0].Title.I)
    title = ihb[0].Title.V; // output file title
   else title = pcb->ib[1].Name.V; 
  if (ihb->PixSiz[1].I)
    xdpcm = 0.01/ihb->PixSiz[1].V;  // dots per cm
   else xdpcm = 1.0;
  if (ihb->PixSiz[2].I)
    ydpcm = 0.01/ihb->PixSiz[2].V;  // dots per cm
   else ydpcm = 1.0;

  if (pcb->TestBit) {
    printf("         bps = %d\n",bps);
    printf("   autorange = %d\n",autorange);
    if (automin) 
      printf("        vmin = auto\n"); 
     else printf("        vmin = %g\n",vmin);
    if (automax)
      printf("        vmax = auto\n");
     else printf("        vmax = %g\n",vmax);
    printf("        xdim = %d\n",xdim);
    printf("        ydim = %d\n",ydim);
    printf(" compression = %d\n",compression);
    printf("       title = %s\n",title);
    printf("       xdpcm = %g\n",xdpcm);
    printf("       ydpcm = %g\n",xdpcm);
    }

 /* loop over 1 input and 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (pcb->TestBit)
    printf("maxloop_1 = % d, maxloop_2 = % d\n",maxloop_1,maxloop_2);

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  if (pcb->TestBit)
    printf("Wmin_1 = % g, DW_1 = % g, Wmin_2 = % g, DW_2 = % g.\n",
            Wmin_1,DW_1,Wmin_2,DW_2);

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  /* loop over the output array */
  W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
  for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
    W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
    for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {

      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);

      if (Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
        f_1[1], f_2[1], &Value)) {
        /* then do something with the data */
        *pI0Data = Value;
        } /* if (Ipol2d ... */

      W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
      } /* for i_2 ... */
    W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */

  if (pcb->TestBit) printf("Output into multiple tiff files\n");

  // Generate tiff filename
  filename_parameter (fname_full,IO_len,pcb->Lin[NPrefix].V,0);
  filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[0].FileNumber);
  filename_full (fname_full,IO_len,fname_full);

  /* Output into multiple tiff files */
   if ( (pcb->ib[0].Inc.V==0) || (pcb->ib[0].First.V==pcb->ib[0].Last.V) ) {
    // fixed output file name
    strncpy(OutputBinaryName,fname_full,InputLineLength);
    } else {
    // numbered output file name
    prefix = getprefix(fname_full);
    extension = getextension(fname_full);
    serialfilename(OutputBinaryName, prefix, ihb[0].ImNum, extension );
    free(prefix); free(extension);
    }
  if (pcb->TestBit) printf("OutputBinaryName = %s\n",OutputBinaryName);

  /* convert to tiff file */
  tiffdata = float2unsigned (I0Data, I0Dummy, I0DDummy, I0Dim_1, I0Dim_2, 
                             autorange,
                             automin, vmin, automax, vmax, bps, comment, 
                             pstatus);
  if (*pstatus!=Success) return;

  /* write description */
  description = (char *) malloc(strlen(comment)+strlen(title)+3);
  sprintf(description,"%s\r\n%s",title,comment);

  /* write tiff file */
  write_tiff ( OutputBinaryName, xdim, ydim, bps,
               compression, tiffdata, orientation,
               description, xdpcm, ydpcm, pstatus );
  free(description);
  if (*pstatus!=Success) return;

  if(tiffdata); free(tiffdata);

  printf("\n");

  printf("Input file         : % s\n",pcb->ib[1].FileName);
  printf("Input image number : % d\n",ihb[1].ImNum);
  printf("Output tiff file   : % s\n",OutputBinaryName);

 } /* saxs_tiff*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  Saxs2TiffFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /* --- Argument  : Bits per sample */
  if (!(pcb->Num[NBps].I)) pcb->Num[NBps].V = 16;
  argv_long( pcb,"Bits per sample (8, 16, 32)", \
             &pcb->Num[NBps],pcb->Num[NBps].V, pstatus);
  if (*pstatus!=Success) return;

  /* --- Argument  : Rasterorientation */
  if (!(pcb->Num[NRori].I)) pcb->Num[NRori].V = 3;
  argv_long( pcb,"Orientation of output image (1-8)", \
             &pcb->Num[NRori],pcb->Num[NRori].V, pstatus);
  if (*pstatus!=Success) return;

  /* --- Argument  : Autorange */
  if (!(pcb->Flg[NAuto].I)) pcb->Flg[NAuto].V = 1;
    
  argv_flag( pcb,"Range output image with minimum and maximum?", 
             &pcb->Flg[NAuto],pcb->Flg[NAuto].V, pstatus);
  if (*pstatus!=Success) return;

  if (!(pcb->Flg[NAuto].V)) {
    /* --- Argument  : Minimum value */
    if (!(pcb->Arg[NMin].I)) pcb->Arg[NMin].V = 0.0;
    argv_float( pcb,"Minimum value", &pcb->Arg[NMin],pcb->Arg[NMin].V, pstatus);
    if (*pstatus!=Success) return;

    /* --- Argument  : Maximum value */
    if (!(pcb->Arg[NMax].I)) pcb->Arg[NMax].V = pow(2,pcb->Num[NBps].V)-1.0;
    argv_float( pcb,"Maximum value", &pcb->Arg[NMax],pcb->Arg[NMax].V, pstatus);
    if (*pstatus!=Success) return;
    }

  printf("\n");
  if (ib[1].Name.I)    printf("input file         : %s\n",ib[1].Name.V);
  if (pcb->Lin[NPrefix].I)  
                       printf("output prefix      : %s\n",pcb->Lin[NPrefix].V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  if (pcb->Num[NBps].I)
                       printf("bits per sample    : %d\n",pcb->Num[NBps].V);
  if (pcb->Flg[NAuto].I)
                       printf("autorange          : %d\n",pcb->Flg[NAuto].V);
  if (!(pcb->Flg[NAuto].V)) {
    if (pcb->Arg[NMin].I)
                       printf("minimum value      : %g\n",pcb->Arg[NMin].V);
    if (pcb->Arg[NMax].I)
                       printf("maximum value      : %g\n",pcb->Arg[NMax].V);
    }
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  DefFlgOption   ( &cb, SAuto,   NAuto );
  DefLineOption  ( &cb, SPrefix, NPrefix );
  DefIntOption   ( &cb, SBps,    NBps );
  DefIntOption   ( &cb, SRori,   NRori);

  DefFloatOption ( &cb, SMin,    NMin );
  DefFloatOption ( &cb, SMax,    NMax );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_tiff, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

