/*+++
1 saxs_normn
  Normalization of an image sequence

2 PURPOSE
  Normalization of an image sequence

  Defaults:
  <input file>                     : input.edf
  <scaler file>                    : <input file>
  <output file name>               : output.edf
  <first image in input file>      : <first image number in first input file>
  <last image in input file>       : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>                      : 1
  <first image in scaler file>     : 1
  <increment in scaler file>       : <increment of 1st input file>
  <output dummy>          : <dummy value in 1st chosen image of 1st input file>
  <output dimension 1>    : <horizontal dimension of first chosen image>
  <output dimension 2>    : <horizontal dimension of first chosen image>
  <transmission>                   : 1 (-trm)
  <scaler number for I0 monitor>   : <HSI0 value of 1st chosen image> (-mons)
  <scaler number for exposure time>: <HSTime value of 1st chosen image (-tims)
  <scaler number for anode counts> : <HSAnode value of 1st chosen image (-anos)
  <scaler for 2nd I0 monitor>      : <HSI0S value of 1st chosen image> (-mo2s)
  <scaler for 2nd exposure time>   : <HSTimeS value of 1st chosen image (-ti2s)
  <scaler for 2nd anode counts>    : <HSAnodeS value of 1st chosen image (-an2s)

  Wildcard:
  A "=" in the argument list chooses the default.

2 METHOD
  The normalization is done in the following way:

        Iin          (anoc*anof)           rd       rd^2        1      1
 Iout = --- * -------------------------- * --- * ----------- * --- * -----
        sum   (monc-monz*timc*timf)*monf   dis    pix1*pix2    trm   Iflat

       with rd^2 = dis^2 +
                     INDEX2R(i1,off1-cen1,pix1)^2 +
                     INDEX2R(i2,off2-cen2,pix2)^2

       with trm  = (trmc-trmz*timc*timf)*trmf / (monc-monz*timc*timf)*monf
        or  trm  = <option>

  sum    : sum of input image                        [event]
  timc   : time counts                               [ct]
  timf   : time factor                               [s]
  timv   : time value                                [s]
  anoc   : anode counts                              [ct]
  anof   : anode factor                              [event/ct]
  anoz   : anode zero                                [ct/s]
  monc   : monitor counts                            [ct]
  monf   : monitor factor                            [photon/ct]
  monz   : monitor zero                              [ct/s]
  monv   : monitor value                             [photon]
  trmc   : transmission monitor counts               [ct]
  trmf   : transmission monitor factor               [photon/ct]
  trmz   : transmission monitor zero                 [ct/s]
  trmv   : transmission monitor value                [photon]
  trm    : tranmission
  rd     : distance sample-pixel                     [m]
  dis    : distance sample-detector (SampleDistance) [m]
  cen1   : horizontal beam center                    [pixel]
  cen2   : vertical beam center                      [pixel]
  off1   : horizontal offset                         [pixel]
  off2   : vertical offset                           [pixel]
  pix1   : horizontal pixel size (PixSiz_1)          [m/pixel]
  pix2   : vertical pixel size (PixSiz_2)            [m/pixel]
  re     : classical electron radius (RELECTRON)   = 2.81794e-15 m

  Iin    : input image                               [event/pixel]
  Iflat  : flatfield image                           [event/photon]
  Iout   : output image = (1/Vol) * dSigma/dOmega    [1/m]
 
  In case that the -flat option is set the result is not divided by the
  flatfield image (default +flat).

  The values of timc, timf etc. are stored as keywords in the header
  of the raw data file.
  To convert the output image to absolute values it has to be devided
  by the flat field file [event/photon].

  The second scalers are only used if (count2*calib2>max_count*calib1)
  where max_count is defined as maxcount = 2^ScalerDepth. If the number
  of the second scaler is set to 0 it will not be used.
  ATTENTION: Only the counts and the factors of the second scalers are 
  read. Only the zero values of the first scalers are read.  

2 HISTORY
  18-Aug-1995 PB      creation from saxs_normm program (PB)
  22-Aug-1995 PB      output is (1/V) * dSigma/dOmega [1/m]
  21-Sep-1995 PB      too long strings changed for SIG
  28-Oct-1995 PB      ScalerDepth changed from integer to float
  07-Dec-1995 PB 3.0
  08-Dec-1995 PB      Calculation of Accepted corrected (AnoCount changed 
                      to AnoVal)
  09-Dec-1995 PB      The binning correction factor cancels out. It is only
		      needed to calculate the accepted number of counts
  15-Dec-1995 PB 3.1  a) read KHScalerLen
  26-Dec-1995 PB 3.11 a) Missing \n\ in Usage added,
                      b) in user_io: Usage replaced by pcb->MainHelp
                                     Version replaced by pcb->MainVersion
                      c) RoutineSucceeded --> Success
  22-Apr-1996 PB 3.12 Title written before calculation
  18-May-1996 PB 3.2  Option rdis (relative distance, RelDis) added. 
                      If RelDis is set, the value of SampleDistance is 
                      ignored and replaced by the sum of DetectorPosition 
                      plus RelDis in input sequence 1 and 2 (i1 and i2) and 
                      in the output sequence (o). The value of
                      DetectorPosition is read from input sequence 2.
  02-Oct-1996 PB 3.3  Calculation and display of maximum pixel value
  20-Jul-1997 PB 3.4  Option +ccd: no dead time correction for ccd
  27-Apr-1998 PB 3.5  Option +tron: read transmission value from argument 
				    on/off (1: read from arguments and
				    use transmission 1 as default,
				    default 1)
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  29-Mar-2000 PB 3.6 AnoZero AnoZero2
  31-Mar-2000 PB 3.7 option -anoz <anode zero>
  2000-08-04  PB %f->%g
  2000-08-31  PB 3.8 flg option +/-scal, float option -monv, -trmv, anov, timv 
                    +scal: MonVal,TrmVal,TimVal,AnoVal are calculated from 
                           scalers counts
                    -scal: MonVal,TrmVal,TimVal,AnoVal are read from keywords
                           KValueI0, KValueI1, KValueAnode, KValueTime

                    Default is +scal. If the keyword KHScalerI0 is missing -scal
                    is used.

                    Values given with options (-monv, -trmv, -anov, -timv)
                    overwrite scaler and keyword values
                    -tron is default (instead of +tron)
                    -flat is default (instead of +flat)
  2000-11-03  PB 3.9 initialization of MonName, TimName etc.
  2000-11-06  PB 3.10 mons is no longer overwritten by ScalerOn test
                      trmv sets value of transmitted photons
  2001-05-22  PB 3.11 rd/dis = 1/cos(2Theta) (only wrong in displayed formula)
  2001-07-07  PB 3.12 do not use second image block to get image overlap
  2001-07-08  PB 3.13 FFirst
  2001-07-09  PB V4.00 new loop, no repetition

---*/

# define Version  "saxs_normn V4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options]\n\
    <i/p file> <scaler file> [<flatfield file>] <o/p file>\n\
    <first image> <last image> <inc>\n\
    <first scaler header> <inc scaler header>\n\
    [<flatfield image>]\n\
    <o/p dummy> <o/p dim1> <o/p dim2>\n\
    <transmission>\n\
    <scaler I0 mon.> <scaler exposure time> <scaler anode counts>\n\
    <2nd scaler I0 mon.> <2nd scaler exposure time> <2nd scaler anode counts>\n\
Purpose: Dead time correction and normalization of gasfilled detector data\n\
Options:  -trm  <transmission>, -bcf <binning correction factor>,\n\
          +-flat, +-ccd, +-tron\n\
         The following options are replacing header values:\n\
          -mons <scaler I0 mon.>,       -mo2s <second scaler I0 mon.>,\n\
          -tims <scaler exposure time>, -ti2s <second scaler exposure time>,\n\
          -anos <scaler anode counts>,  -an2s <second scaler anode counts>,\n\
          -trms <scaler I1 mon.>,       -tr2s <second scaler I1 mon.>,\n\
          -dpth <eff. scaler depth in bits>, -slen <scaler length>,\n\
         Additional options to change individual scaler values (see +h)" 

# define Default_flat      FALSE;
# define Default_ccd       FALSE;
# define Default_trmon     FALSE;
# define Default_extension DefaultExtension
# define Default_ipfn1     DefaultInput
# define Default_ipfn2     Default_ipfn1
# define Default_ipfn3     "flatfield"DefaultExtension
# define Default_opfn      DefaultOutput
# define Default_inc       1
# define Default_slen      32
# define Default_depth     24
# define Default_mons      4
# define Default_tims      17
# define Default_anos      19
# define Default_mo2s      0
# define Default_ti2s      0
# define Default_an2s      0
# define Default_tr2s      0
# define Default_trm       1.0
/* Special options */
# define SFlat     "flat"  /* use flatfield file */
# define NFlat     0
# define SCcd      "ccd"   /* correct ccd image, rejection=0 */
# define NCcd      1
# define STrmOn    "tron"  /* prompt for trm, default trm=1 */
# define NTrmOn    2
# define SScalOn   "scal"  /* use scaler file */
# define NScalOn   3

# define SMonSca   "mons" /* integer */
# define NMonSca   0
# define STimSca   "tims"
# define NTimSca   1
# define SAnoSca   "anos"
# define NAnoSca   2
# define STrmSca   "trms" 
# define NTrmSca   3
# define SMonScaS  "mo2s"
# define NMonScaS  4
# define STimScaS  "ti2s"
# define NTimScaS  5
# define SAnoScaS  "an2s"
# define NAnoScaS  6
# define STrmScaS  "tr2s"
# define NTrmScaS  7
# define SScaLen   "slen"
# define NScaLen   8

# define STrm       "trm" /* float */
# define NTrm       0
# define SScaDepth  "dpth"
# define NScaDepth  1
# define SMonCount  "monc"
# define NMonCount  2
# define SMonZero   "monz"
# define NMonZero   3
# define SMonFac    "monf"
# define NMonFac    4
# define SMonVal    "monv"
# define NMonVal    5
# define STimCount  "timc"
# define NTimCount  6
# define STimFac    "timf"
# define NTimFac    7
# define STimVal    "timv"
# define NTimVal    8 
# define SAnoCount  "anoc"
# define NAnoCount  9
# define SAnoZero   "anoz"
# define NAnoZero   10 
# define SAnoFac    "anof"
# define NAnoFac    11 
# define SAnoVal    "anov"
# define NAnoVal    12 
# define STrmCount  "trmc"
# define NTrmCount  13 
# define STrmZero   "trmz"
# define NTrmZero   14
# define STrmFac    "trmf"
# define NTrmFac    15 
# define STrmVal    "trmv"
# define NTrmVal    16
# define SMonCountS "mo2c"
# define NMonCountS 17 
# define SMonFacS   "mo2f"
# define NMonFacS   18
# define STimCountS "ti2c"
# define NTimCountS 19 
# define STimFacS   "ti2f"
# define NTimFacS   20 
# define SAnoCountS "an2c"
# define NAnoCountS 21 
# define SAnoFacS   "an2f"
# define NAnoFacS   22 
# define STrmCountS "tr2c"
# define NTrmCountS 23 
# define STrmFacS   "tr2f"
# define NTrmFacS   24
# define SI1sum     "sum"
# define NI1sum     25 
# define SBinCorFac "bcf"
# define NBinCorFac 26
# define SRelDis    "rdis"
# define NRelDis    27

# define BlockNum 4          /* 2 input + 1 output */

# define PRIVATE static        /* used to declare variables of private type */

/***************************************************************************
 * Global Initialised Variables                                            *
 ***************************************************************************/

PRIVATE char ScalerKeyNameBuffer[InputLineLength];

/*---------------------------------------------------------------------------
ScalerKey
return the full keyword of scaler number
---------------------------------------------------------------------------*/

char * ScalerKey ( const char * kh , const int scalernum )
{ if (scalernum<10)
     sprintf(ScalerKeyNameBuffer,"%s%s0%1u",KHScaler,kh,scalernum); else
     sprintf(ScalerKeyNameBuffer,"%s%s%2u",KHScaler,kh,scalernum);
  return ( ScalerKeyNameBuffer );
} /* ScalerKey */

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io( CmdBlk * pcb, ImgBlk * ib, int * pstatus )
{
  char  progfn[InputLineLength];

  long  image_1st;
  long  image_lst;

  int   next_block;
  int   INflat; /* image block number of flat field file */

  char  MonitorName[InputLineLength];
  char  TimeName[InputLineLength];
  char  AnodeName[InputLineLength];

  char  MonitorNameS[InputLineLength];
  char  TimeNameS[InputLineLength];
  char  AnodeNameS[InputLineLength];

  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn, InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,pcb->MainHelp);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,pcb->MainVersion);
  printf("\n");

  /*--- Reduce total number of block, if no flatfield correction requested */
  next_block = 3;
  if (!pcb->Flg[NFlat].I) pcb->Flg[NFlat].V = Default_flat;
  if (!pcb->Flg[NFlat].V) {
    INflat = -1; pcb->ImgBlkLen = BlockNum-1;
    } else { INflat = next_block++; }

  /*--- OpenMode of first input file */
  if (!ib[1].OpenMode.I) {
    ib[1].OpenMode.V = IO_Old | IO_FileProtect;
    ib[1].OpenMode.I = TRUE;
    }

  /*--- Argument  : input file name */
  argv_filename( pcb, "Input file name", ib, 1, Default_ipfn1, pstatus);

  /* Exit in all cases after an abort from the first argument */
  if (*pstatus!=Success) return;

  OpenImageFile( pcb,ib,1,ib[1].Name.V,ib[1].FFirst.V,
                 IO_Old|IO_FileProtect,pstatus);
  if (*pstatus!=Success) return;

  /*--- OpenMode of second input file */
  if (!ib[2].OpenMode.I) {
    ib[2].OpenMode.V = IO_Old | IO_FileProtect | IO_DontReadData;
    ib[2].OpenMode.I = TRUE;
    }

  /*--- Argument  : name of second input file (scaler file)*/
  argv_filename( pcb, "Scaler file name", ib, 2, ib[1].Name.V, pstatus);
  if (*pstatus!=Success) return;

  OpenImageFile( pcb,ib,2,ib[2].Name.V,ib[2].FFirst.V,
                 IO_Old|IO_FileProtect,pstatus);
  if (*pstatus!=Success) return;

  /*--- Rejection correction */
    if (!pcb->Flg[NCcd].I) pcb->Flg[NCcd].V = Default_ccd;

  /*--- Flatfield file */
  if (pcb->Flg[NFlat].V) {
    /*--- OpenMode of third input file (flatfield file) */
    if (!ib[INflat].OpenMode.I) {
      ib[INflat].OpenMode.V = IO_Old | IO_FileProtect;
      ib[INflat].OpenMode.I = TRUE;
      }

    /*--- Argument  : name of flat field file */
    argv_filename(pcb,"Flatfield file name",ib,INflat,Default_ipfn3, pstatus);
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,3,ib[INflat].Name.V,ib[3].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;
    } /* Flatfield file */


  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect;
    ib[0].OpenMode.I = TRUE;
    }

  /*--- Argument  : output file name */
  argv_filename(pcb,"Output file name",ib,0,Default_opfn, pstatus);
  if (*pstatus!=Success) return;

  /* Search for minimum and maximum image number */
  (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : number of first image */
  argv_long( pcb,"First image of input file", &ib[1].First, image_1st, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : number of last image */
  if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
  argv_long( pcb,"Last image of input file", &ib[1].Last, image_lst, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : increment */
  argv_long( pcb,"Increment of input file", &ib[1].Inc, Default_inc, pstatus);
  if (*pstatus!=Success) return;

  (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
  if (*pstatus!=Success) return;
  if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

  CloseImageFile( pcb, ib, 1, pstatus) ;
  if (*pstatus!=Success) return;

  /*--- Argument : number of first image */
  argv_long( pcb,"First scaler image", &ib[2].First, ib[1].First.V,
             pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : increment */
  argv_long( pcb,"Increment of scaler image", &ib[2].Inc, ib[1].Inc.V, pstatus);
  if (*pstatus!=Success) return;

  (void) ReadImageHeader( pcb, ib, 2, ib[2].First.V, ihb, pstatus);
  if (*pstatus!=Success) return;
  if (pcb->TestBit) PrintImageHeaderBlock(2,&ihb[2]);

  /*--- Flatfield file */
  if (pcb->Flg[NFlat].V) {
    /* Search for minimum and maximum image number */
    (void) SearchMinMaxImage ( pcb, ib, 3, &image_1st, &image_lst, pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of first image */
    argv_long( pcb,"Flatfield image", &ib[INflat].First, image_1st, pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, 3, pstatus) ;
    if (*pstatus!=Success) return;

    } /* Flatfield file */

  /*--- Argument  : output dummy */
  argv_float( pcb,"Output dummy", &ib[0].Dummy,ihb[1].Dummy.V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : output dimension 1 */
  argv_long( pcb,"Output dimension 1", &ib[0].Dim[1],ihb[1].Dim[1], pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : output dimension 2 */
  argv_long( pcb,"Output dimension 2", &ib[0].Dim[2],ihb[1].Dim[2], pstatus);
  if (*pstatus!=Success) return;

 /*--- Argument  : Transmission */
  if (!pcb->Flg[NTrmOn].I) pcb->Flg[NTrmOn].V = Default_trmon;
  if ( pcb->Flg[NTrmOn].V ) {
    argv_float( pcb,"Transmission", \
               &pcb->Arg[NTrm],Default_trm, pstatus);
    if (*pstatus!=Success) return;
    }

 /*--- Argument  : scaler number of I0 monitor */
  printf("\nThe following normalization will be done:\n");
  printf( "\n\
         Iin   rd       rd^2       1           (anoc*anof)              1\n\
  Iout = --- * ---- * --------- * --- * --------------------------  * -----\n\
         sum   dis    pix1*pix2   trm   (monc-monz*timc*timf)*monf    Iflat\n\
");

  printf( "\n\
         ^^^^^^^^^^^^^^^^^^^^^^^^^^^^   ^^^^^^^^^^^^^^^^^^^^^^^^^^    ^^^^^\n\
            first input file (i1)         scaler input file (i2)       (i3)\n\
");

  printf( "\n\
         with rd^2   = SampleDistance^2 + <radial distance on detector>^2\n\
              rd/dis = 1/cos(2*Theta)");
  if (!pcb->Arg[NTrm].I)
    printf( "\n\n\
                       (trmc-trmz*timc*timf)*trmf\n\
              trm    = --------------------------\n\
                       (monc-monz*timc*timf)*monf)");
    else printf( "\n\
              trm    = Transmission");
  printf( "\n\
              Iin  (detector raw data) : [event/pixel]\n\
              Iout (normalized data)   : [1/m]\n\n");
  if (pcb->Flg[NFlat].V) printf( "\n\
	      Iflat (flatfield image)  : [event/photon]\n");
  printf( "\n");
  if (pcb->Flg[NFlat].V) printf("  Division by flatfield\n\n"); 
  if (pcb->Flg[NCcd].V)  printf("  No rejection correction\n\n");

  if (!pcb->Num[NMonSca].I)
    if (ReadHeaderLong( pcb,2,ihb,KHScalerI0,&pcb->Num[NMonSca].V,pstatus)) {
      if (*pstatus!=Success) return;
      } else pcb->Num[NMonSca].V = Default_mons;
 
  argv_long( pcb,"Number of I0 monitor scaler", \
             &pcb->Num[NMonSca],pcb->Num[NMonSca].V, pstatus);
  if (*pstatus!=Success) return;

  if (ReadHeaderLine( pcb,2,ihb,ScalerKey(KHN,pcb->Num[NMonSca].V),\
       MonitorName,pstatus)) {
    if (*pstatus!=Success) return;
    } else MonitorName[0] = (char) 0;

 /*--- Argument  : time scaler number */
  if (!pcb->Num[NTimSca].I)
    if (ReadHeaderLong( pcb,2,ihb,KHScalerTime,&pcb->Num[NTimSca].V,pstatus)) {
      if (*pstatus!=Success) return;
      } else pcb->Num[NTimSca].V = Default_tims;

  argv_long( pcb,"Number of time scaler", \
             &pcb->Num[NTimSca],pcb->Num[NTimSca].V, pstatus);
  if (*pstatus!=Success) return;

  if (ReadHeaderLine( pcb,2,ihb,ScalerKey(KHN,pcb->Num[NTimSca].V),\
        TimeName, pstatus)) {
    if (*pstatus!=Success) return;
    } else TimeName[0] = (char) 0; 

 /*--- Argument  : anode scaler number */
  if (!pcb->Num[NAnoSca].I)
    if (ReadHeaderLong( pcb,2,ihb,KHScalerAnode,&pcb->Num[NAnoSca].V,pstatus)) {
      if (*pstatus!=Success) return;
      } else pcb->Num[NAnoSca].V = Default_anos;

  argv_long( pcb,"Number of anode scaler", \
             &pcb->Num[NAnoSca],pcb->Num[NAnoSca].V, pstatus);
  if (*pstatus!=Success) return;

  if (ReadHeaderLine( pcb,2,ihb,ScalerKey(KHN,pcb->Num[NAnoSca].V),\
       AnodeName,pstatus)) {
    if (*pstatus!=Success) return;
    } else AnodeName[0] = (char) 0;

  /*--- Argument  : number of 2nd I0 monitor scaler */
  if (!pcb->Num[NMonScaS].I)
    if (ReadHeaderLong( pcb,2,ihb,KHScalerI0S,&pcb->Num[NMonScaS].V,pstatus)) {
      if (*pstatus!=Success) return;
      } else pcb->Num[NMonScaS].V = Default_mo2s;

  printf("The second scalers are used in case of an overflow.\n");
  printf("Type 0 if you do not want to use a second scaler.\n");

  argv_long( pcb,"Number of 2nd I0 monitor scaler", \
             &pcb->Num[NMonScaS],pcb->Num[NMonScaS].V, pstatus);
  if (*pstatus!=Success) return;

  if (ReadHeaderLine( pcb,2,ihb,ScalerKey(KHN,pcb->Num[NMonScaS].V),\
        MonitorNameS, pstatus)) {
    if (*pstatus!=Success) return;
    } else MonitorNameS[0] = (char) 0;

 /*--- Argument  : number of 2nd time scaler */
  if (!pcb->Num[NTimScaS].I)
    if (ReadHeaderLong(pcb,2,ihb,KHScalerTimeS,&pcb->Num[NTimScaS].V,pstatus)) {
      if (*pstatus!=Success) return;
      } else pcb->Num[NTimScaS].V = Default_ti2s;

  argv_long( pcb,"Number of 2nd time scaler", \
             &pcb->Num[NTimScaS],pcb->Num[NTimScaS].V, pstatus);
  if (*pstatus!=Success) return;

  if (ReadHeaderLine( pcb,2,ihb,ScalerKey(KHN,pcb->Num[NTimScaS].V),\
       TimeNameS, pstatus)) {
    if (*pstatus!=Success) return;
    } else TimeNameS[0] = (char) 0;

 /*--- Argument  : number of 2nd anode scaler */
  if (!pcb->Num[NAnoScaS].I)
    if (ReadHeaderLong(pcb,2,ihb,KHScalerAnodeS,&pcb->Num[NAnoScaS].V,pstatus)){
      if (*pstatus!=Success) return;
      } else pcb->Num[NAnoScaS].V = Default_an2s;

  argv_long( pcb,"Number of 2nd anode scaler", \
             &pcb->Num[NAnoScaS],pcb->Num[NAnoScaS].V, pstatus);
  if (*pstatus!=Success) return;

  if (ReadHeaderLine( pcb,2,ihb,ScalerKey(KHN,pcb->Num[NAnoScaS].V),\
        AnodeNameS, pstatus)) {
    if (*pstatus!=Success) return;
    } else AnodeNameS[0] = (char) 0;

  CloseImageFile( pcb, ib, 2, pstatus) ;
  if (*pstatus!=Success) return;

 /* Binning correction factor (only for terminal info) */
  if (!pcb->Arg[NBinCorFac].I) { pcb->Arg[NBinCorFac].V = \
    (float) ( MAX2(1,ib[1].Bin[1].V) * MAX2(1,ib[1].Bin[2].V) );
    pcb->Arg[NBinCorFac].I = TRUE; }

  printf("\n");
  if (ib[1].Name.I)
  printf("Input file            : %s\n",ib[1].Name.V);
  if (ib[2].Name.I)
  printf("Scaler file           : %s\n",ib[2].Name.V);
/*  if ( (pcb->Arg[NFlat].V) && (ib[INflat].Name.I) ) */
  if (INflat>=0)
    printf("Flatfield file        : %s\n",ib[INflat].Name.V);
  if (ib[0].Name.I)
  printf("Output file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)
  printf("First image           : %d\n",ib[1].First.V);
  if (ib[1].Last.I)
  printf("Last image            : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)   
  printf("Increment             : %d\n",ib[1].Inc.V);
  if (ib[2].First.I)
  printf("First scaler image    : %d\n",ib[2].First.V);
  if (ib[2].Inc.I)
  printf("Inc. of scaler image  : %d\n",ib[2].First.V);
/*  if ( (pcb->Arg[NFlat].V) && (ib[INflat].First.I) ) */
  if (INflat>=0)
    printf("Flatfield image       : %d\n",ib[INflat].First.V);
  if (ib[0].Dummy.I)
  printf("Output dummy          : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)
  printf("Output dimension 1    : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)
  printf("Output dimension 2    : %d\n",ib[0].Dim[2].V);
  if (pcb->Num[NMonSca].I)
  printf("I0 monitor scaler     : %d (%s)\n",pcb->Num[NMonSca].V,MonitorName);
  if (pcb->Num[NTimSca].I)
  printf("Time scaler           : %d (%s)\n",pcb->Num[NTimSca].V,TimeName);
  if (pcb->Num[NAnoSca].I)
  printf("Anode scaler          : %d (%s)\n",pcb->Num[NAnoSca].V,AnodeName);
  if (pcb->Num[NMonScaS].I)
  printf("2nd I0 monitor scaler : %d (%s)\n",pcb->Num[NMonScaS].V,MonitorNameS);
  if (pcb->Num[NTimScaS].I)
  printf("2nd time scaler       : %d (%s)\n",pcb->Num[NTimScaS].V,TimeNameS);
  if (pcb->Num[NAnoScaS].I)
  printf("2nd anode scaler      : %d (%s)\n",pcb->Num[NAnoScaS].V,AnodeNameS);
  if (pcb->Arg[NTrm].I)
  printf("Transmission          : %g \n",pcb->Arg[NTrm].V);
  printf("\n");

  if (pcb->TestBit) { PrintBlocks ( pcb, ib ); printf("\n"); }

} /* user_io */

/*---------------------------------------------------------------------------
1 saxs_normn

2 PURPOSE
  Normalization of an image
  The normalization is done in the following way:

                 (anoc*anof)/sum                 rd^3         1      1
 Iout = Iin * -------------------------- * --------------- * --- * -----
              (monc-monz*timc*timf)*monf    dis*pix1*pix2    trm   Iflat

       with rd^2 = dis^2 +
                     INDEX2R(i1,off1-cen1,pix1)^2 +
                     INDEX2R(i2,off2-cen2,pix2)^2

            trm = ((trmc-trmz*timc*timf)*trmf)/((monc-monz*timc*timf)*monf)
         or trm = "Option"

  If the normalization is not possible, e.g. division by zero, the output
  image contains only dummies. 
  The division by Iflat is omitted if -flat option is set
  If +ccd is set no rejection correction is done: 
       AnoVal = I1Sum * BinCorFac
---------------------------------------------------------------------------*/
void saxs_normn (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int NumberOfImages;
  const float eps=1e-30;
  int i,imax;
  int j;
  int NormOK, FlatFieldCorrection;
  int RejectionCorrection;
  int ScalerOn=TRUE;
  float Divider;
  int ErrorValue;

  /* output image information */
  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy;
  /* input image information */
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float *pI1Data;
  float I1Dummy, I1DDummy;
  float I1Value;
  /* scaler image information */
  float *I2Data;
  int   I2Dim_1,I2Dim_2;
  float I2Dummy, I2DDummy;
  float I2Value;
  /* flatfield image information */
  int   INflat; /* number of flat field block */
  float *IFData;
  int   IFDim_1,IFDim_2;
  float *pIFData;
  float IFDummy, IFDDummy;
  float IFValue;
  int   ScalerLen;
  float ScalerDepth, MaxCount;
  float MonCount2, MonZero2, MonFac2;
  float TimCount2, TimFac2;
  float AnoCount2, AnoZero2, AnoFac2;
  float TrmCount2, TrmZero2, TrmFac2;
  int   MonScaler2, TimScaler2, AnoScaler2, TrmScaler2;
  float MonCount, MonZero, MonFac;
  float TimCount, TimFac;
  float AnoCount, AnoZero, AnoFac;
  float TrmCount, TrmZero, TrmFac;
  float MonVal, TrmVal, AnoVal, TimVal;
  int   MonScaler, TimScaler, AnoScaler, TrmScaler;
  float DetectorPosition;
  float off1, off2, rd2, rd20, rd3, dd2, re2, pix1, pix2, cen1, cen2, dis;
  IO_long tmp;

  char  MonName[InputLineLength];
  char  TimName[InputLineLength];
  char  AnoName[InputLineLength];
  char  TrmName[InputLineLength];

  int   MonI=FALSE, TimI=FALSE, AnoI=FALSE, TrmI=FALSE;

  float I1Sum, I1Max, BinCorFac, NormFac, Accepted, EffCountRate, Trm;
  float MaxPixRate, MaxPixVal, DetVal;

  /* Do not change the following definitions,
     they are automatically changed with BlockNum */
  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;
  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];
  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];
  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
/*  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
    for(i=1;i<imax;i++) {
      printf("ihb[% d,% d] ", i, ihb[i].ImNum); }
    printf(")\n\n");
*/
  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);}

 /* Check the number of images */
  NumberOfImages = 3;
  INflat = NumberOfImages; /* must be an existing block */
  if (pcb->Flg[NFlat].V) {
    FlatFieldCorrection=TRUE; NumberOfImages += 1; 
    } else { FlatFieldCorrection=FALSE; }

  if (pcb->ImgBlkLen!=NumberOfImages) {
     printf("%d images found, %d input and 1 output image required\n",
             pcb->ImgBlkLen, NumberOfImages-1); *pstatus=Failed; return; }

  /* Check, if rejecion correction requested */
  if (pcb->Flg[NCcd].V) RejectionCorrection=FALSE; 
    else RejectionCorrection=TRUE;

  /* Check, if scalers should be used */
  /* scalers are not used when keyword KHScalerI0 is not found */
 
  if (pcb->Flg[NScalOn].I) 
     ScalerOn=(int)pcb->Flg[NScalOn].V; 
   else /* test, if KHScalerI0 can be read */
     ScalerOn=ReadHeaderLong(pcb,2,ihb,KHScalerI0,&tmp.V,pstatus);

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  DoNotScaleWith(ihb,2); /* Do not use header ihb[2] for scaling */

  GetImageOverlap       ( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          fmin_1, fmax_1, fmin_2, fmax_2,
                          &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 input for data (input file -> I1),
    1 input for header (scaler file -> I2, only used to know the monitor value),
    1 output -> I0*/
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];
  I2Data   = ihb[2].Data;
  I2Dummy  = ihb[2].Dummy.V;
  I2DDummy = ihb[2].DDummy.V;
  I2Dim_1  = (int) ihb[2].Dim[1];
  I2Dim_2  = (int) ihb[2].Dim[2];
  IFData   = ihb[INflat].Data;
  IFDummy  = ihb[INflat].Dummy.V;
  IFDDummy = ihb[INflat].DDummy.V;
  IFDim_1  = (int) ihb[INflat].Dim[1];
  IFDim_2  = (int) ihb[INflat].Dim[2];

  /* if RelDis is set, calculate SampleDistance from DetectorPosition+RelDis */
  if (pcb->Arg[NRelDis].I) {
      ReadHeaderFloat(pcb,2,ihb,KDetectorPosition,&DetectorPosition,pstatus);
      if (*pstatus!=Success) return;
      ihb[0].SampleDistance.V = DetectorPosition + pcb->Arg[NRelDis].V;
      ihb[1].SampleDistance.V = ihb[0].SampleDistance.V;
      ihb[2].SampleDistance.V = ihb[1].SampleDistance.V;
      ihb[0].SampleDistance.I = TRUE;
      ihb[1].SampleDistance.I = TRUE;
      ihb[2].SampleDistance.I = TRUE;
      if (pcb->TestBit) {
        printf("RelativeDistance   = %9.3g m\n",pcb->Arg[NRelDis].V);
        printf("DetectorPosition   = %9.3g m\n",DetectorPosition);
        printf("New SampleDistance = %9.3g m\n",ihb[0].SampleDistance.V);
        }
      } 
    
  /* get geometrical parameters from the header */
  if (ihb[1].Offset[1].I) off1 = ihb[1].Offset[1].V;
    else {printf("Missing header information (Offset[1])\n");exit(-1);}
  if (ihb[1].Offset[2].I) off2 = ihb[1].Offset[2].V;
    else {printf("Missing header information (Offset[2])\n");exit(-1);}
  if (ihb[1].PixSiz[1].I) pix1 = ihb[1].PixSiz[1].V;
    else {printf("Missing header information (PixSiz[1])\n");exit(-1);}
  if (ihb[1].PixSiz[2].I) pix2 = ihb[1].PixSiz[2].V;
    else {printf("Missing header information (PixSiz[2])\n");exit(-1);}
  if (ihb[1].Center[1].I) cen1 = ihb[1].Center[1].V;
    else {printf("Missing header information (Center[1])\n");exit(-1);}
  if (ihb[1].Center[2].I) cen2 = ihb[1].Center[2].V;
    else {printf("Missing header information (Center[2])\n");exit(-1);}
  if (ihb[1].SampleDistance.I) dis = ihb[1].SampleDistance.V;
    else {printf("Missing header information (SampleDistance)\n");exit(-1);}
  re2  = RELECTRON*RELECTRON;

  if (pcb->TestBit) {
     printf("off1 = %9.3g pixel  , off2 = %9.3g pixel\n",off1,off2);
     printf("cen1 = %9.3g pixel  , cen2 = %9.3g pixel\n",cen1,cen2);
     printf("pix1 = %9.3g m/pixel, pix2 = %9.3g m/pixel\n",pix1,pix2);
     printf("dis  = %9.3g m      , re^2 = %9.3g m^2\n",dis,re2);
    }

  if (fabs(pix1*pix2)<eps) {
    printf("Pixel size too small (pix1*pix2 = %g)\n",pix1*pix2); exit(-1);
    }
  if (fabs(dis)<eps) {
    printf("Sample distance too small (dis = %g)\n",dis); exit(-1);
    }

  /* --- Calculate sum of input image 1 I1Sum */
  if (!pcb->Arg[NI1sum].I) {
    I1Sum=0.0;
    pI1Data=I1Data;
    for (j=0;j<I1Dim_1*I1Dim_2;j++) {
      if (NODUMMY(*pI1Data,I1Dummy,I1DDummy)) I1Sum+=*pI1Data;
      pI1Data++;
      }
    } else I1Sum=pcb->Arg[NI1sum].V;

  /* --- Binning correction factor (only for info) */
  if (pcb->Arg[NBinCorFac].I) BinCorFac = pcb->Arg[NBinCorFac].V;
    else BinCorFac = 1.0;

  /* --- Maximum value (only for info) */
  pI1Data=I1Data;
  /* skip all dummies */
  for (j=0;(j<I1Dim_1*I1Dim_2)&&(DUMMY(*pI1Data,I1Dummy,I1DDummy));j++) {
    pI1Data++;
    }
  if (j<I1Dim_1*I1Dim_2) I1Max=*pI1Data; else I1Max=I1Dummy;
  /* find maximum */
  for (j=j;j<I1Dim_1*I1Dim_2;j++) {
    if (NODUMMY(*pI1Data,I1Dummy,I1DDummy)) I1Max=MAX2(I1Max,*pI1Data);
    pI1Data++;
    }

  if (pcb->TestBit)
    printf("I1Max    = %g\n",I1Max);

  /* Initialize scaler names */
  MonName[0] = (char) 0;
  TimName[0] = (char) 0;
  AnoName[0] = (char) 0;
  TrmName[0] = (char) 0;

  if (ScalerOn) { /* read counter values from file */

    /* --- MaxCount */
    if (!pcb->Arg[NScaDepth].I) 
      if (ReadHeaderFloat(pcb,2,ihb,KHScalerDepth,&pcb->Arg[NScaDepth].V,
          pstatus)) {
        if (*pstatus!=Success) return;
        } else pcb->Arg[NScaDepth].V = (float) Default_depth; 

    ScalerDepth = pcb->Arg[NScaDepth].V;
    MaxCount = exp(ScalerDepth*log(2));

    if (pcb->TestBit) {
      printf("ScalerDepth = %g\n", ScalerDepth);
      printf("MaxCount = %g\n", MaxCount);
      }

    /* --- ScalerLen */
    if (!pcb->Num[NScaLen].I) {
      if (ReadHeaderLong(pcb,2,ihb,KHScalerLen, &pcb->Num[NScaLen].V,
          pstatus)) { if (*pstatus!=Success) return;
        } else  pcb->Num[NScaLen].V = (long) Default_slen;
      }
    ScalerLen = (int) pcb->Num[NScaLen].V;
    if (pcb->TestBit) printf("ScalerLen = %d\n", ScalerLen);

    /* --- Monitor */
    if (!pcb->Arg[NMonVal].I) {

    MonI = FALSE;
    /*     MonScaler1 */
    if (!pcb->Num[NMonSca].I) {
      if (!ReadHeaderLong(pcb,2,ihb,KHScalerI0,&pcb->Num[NMonSca].V,
          pstatus)) { *pstatus = KeywordMissing;
        printf("\nMissing keyword for MonScaler1 %s\n",KHScalerI0); } 
      if (*pstatus!=Success) return;
      }
    MonScaler = (int) pcb->Num[NMonSca].V;
    if (pcb->TestBit) printf("MonScaler = %d\n", MonScaler);

    if ((1<=MonScaler) && (MonScaler<=ScalerLen)) {
      /*     MonCount1 */
      if (!pcb->Arg[NMonCount].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHC,MonScaler),&MonCount,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for MonCount1 %s\n",
            ScalerKey(KHC,MonScaler));}
        if (*pstatus!=Success) return;
        } else MonCount = pcb->Arg[NMonCount].V;

      /*      MonFac1 */
      if (!pcb->Arg[NMonFac].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHF,MonScaler),&MonFac,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for MonFac1 %s\n",
            ScalerKey(KHF,MonScaler)); }
        if (*pstatus!=Success) return;
        } else MonFac = pcb->Arg[NMonFac].V;
  
      /*     MonZero1 */
      if (!pcb->Arg[NMonZero].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHZ,MonScaler),&MonZero,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for MonZero1 %s\n",
            ScalerKey(KHZ,MonScaler)); }
        if (*pstatus!=Success) return;
        } else MonZero = pcb->Arg[NMonZero].V;
      MonI = TRUE;
      } else { 
      MonCount = 0.0; MonFac = 1.0; MonZero = 0.0; 
      } /* if (MonScaler ... */

    /*     MonScaler2 */
    if (!pcb->Num[NMonScaS].I) {
      if (ReadHeaderLong(pcb,2,ihb,KHScalerI0S, &pcb->Num[NMonScaS].V,
          pstatus)) { if (*pstatus!=Success) return; 
        } else  pcb->Num[NMonScaS].V = (long) Default_mo2s;
      }
    MonScaler2 = (int) pcb->Num[NMonScaS].V;
  
    if ((1<=MonScaler2) && (MonScaler2<=ScalerLen)) {
      /*     MonCount2 */
      if (!pcb->Arg[NMonCountS].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHC,MonScaler2),&MonCount2,
            pstatus)) { *pstatus = KeywordMissing;
         printf("\nMissing keyword for MonCount2 %s\n",
           ScalerKey(KHC,MonScaler2));}
        if (*pstatus!=Success) return;
        } else MonCount2 = pcb->Arg[NMonCountS].V;
        
      /*      MonFac2 */
      if (!pcb->Arg[NMonFacS].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHF,MonScaler2),&MonFac2,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for MonFac2 %s\n",
            ScalerKey(KHF,MonScaler2)); }
        if (*pstatus!=Success) return;
        } else MonFac2 = pcb->Arg[NMonFacS].V;
 
      /*     MonZero2  (no separate option) */
      if (!pcb->Arg[NMonZero].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHZ,MonScaler2),&MonZero2,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for MonZero2 %s\n",
            ScalerKey(KHZ,MonScaler2));}
        if (*pstatus!=Success) return;
        } else MonZero2 = pcb->Arg[NMonZero].V;
      MonI = TRUE;

      /*      Find valid scaler */
      if (MonCount2*MonFac2>MaxCount*MonFac) {
        MonScaler = MonScaler2;
        MonCount  = MonCount2;
        MonFac    = MonFac2;
        MonZero   = MonZero2;
        if (pcb->TestBit) printf("Using alternativescaler for monitor.\n");
        }
      } /* if (MonScaler2 ... */

    /*     MonName */
    if (ReadHeaderLine(pcb,2,ihb,ScalerKey(KHN,MonScaler),MonName, 
        pstatus)) { if (*pstatus!=Success) return; 
      } else MonName[0] = (char) 0;

    if (pcb->TestBit) {
      printf("MonScaler = %d, MonName = %s\n", MonScaler, MonName);
      printf("MonCount = %g, MonZero = %g, MonFac = %g\n", \
              MonCount, MonZero, MonFac);
      }
      } /* if (!pcb->Arg[NMonVal].I) */ 

    /* --- Time */
    if (!pcb->Arg[NTimVal].I) {

    TimI = FALSE;
    /*     TimScaler1 */
    if (!pcb->Num[NTimSca].I) {
      if (!ReadHeaderLong(pcb,2,ihb,KHScalerTime, &pcb->Num[NTimSca].V,
          pstatus)) { *pstatus = KeywordMissing;
        printf("\nMissing keyword for TimScaler1 %s\n",KHScalerTime); }
      if (*pstatus!=Success) return;
      }
  
    TimScaler = (int) pcb->Num[NTimSca].V;
 
    if ((1<=TimScaler) && (TimScaler<=ScalerLen)) {
      /*     TimCount1 */
      if (!pcb->Arg[NTimCount].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHC,TimScaler),&TimCount, 
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for TimCount1 %s\n",
            ScalerKey(KHC,TimScaler));}
        if (*pstatus!=Success) return;
        } else TimCount = pcb->Arg[NTimCount].V;
  
      /*      TimFac1 */
      if (!pcb->Arg[NTimFac].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHF,TimScaler), &TimFac,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for TimFac1 %s\n",
            ScalerKey(KHF,TimScaler)); }
        if (*pstatus!=Success) return; 
        } else TimFac = pcb->Arg[NTimFac].V;
      TimI = TRUE;
      } else {
      TimCount = 0.0; TimFac = 1.0; 
      }

    /*     TimScaler2 */
    if (!pcb->Num[NTimScaS].I) {
      if (ReadHeaderLong(pcb,2,ihb,KHScalerTimeS,&pcb->Num[NTimScaS].V,
          pstatus)) { if (*pstatus!=Success) return;
        } else  pcb->Num[NTimScaS].V = (long) Default_ti2s;
      }
    TimScaler2 = (int) pcb->Num[NTimScaS].V;

    if ((1<=TimScaler2) && (TimScaler2<=ScalerLen)) {
      /*     TimCount2 */
      if (!pcb->Arg[NTimCountS].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHC,TimScaler2),&TimCount2, 
            pstatus)) { *pstatus = KeywordMissing;
         printf("\nMissing keyword for TimCount2 %s\n",
           ScalerKey(KHC,TimScaler2));}
        if (*pstatus!=Success) return;
        } else TimCount2 = pcb->Arg[NTimCountS].V;
  
      /*      TimFac2 */
      if (!pcb->Arg[NTimFacS].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHF,TimScaler2), &TimFac2, 
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for TimFac2 %s\n",
            ScalerKey(KHF,TimScaler2));}
        if (*pstatus!=Success) return;
        } else TimFac2 = pcb->Arg[NTimFacS].V;
      TimI = TRUE;

      /*      Find valid scaler */
      if (TimCount2*TimFac2>MaxCount*TimFac) {
        TimScaler = TimScaler2;
        TimCount  = TimCount2;
        TimFac    = TimFac2;
        if (pcb->TestBit) printf("Using alternative scaler for time.\n");
        }
      } /* if (1<=TimScaler2 ... */
  
    /*     TimName */
    if (ReadHeaderLine(pcb,2,ihb,ScalerKey(KHN,TimScaler),TimName, 
        pstatus)) { if (*pstatus!=Success) return;
      } else TimName[0] = (char) 0;
  
    if (pcb->TestBit) {
      printf("TimScaler = %d, TimName = %s\n", TimScaler, TimName);
      printf("TimCount = %g, TimFac = %g\n", TimCount, TimFac);
      }
      } /* if (pcb->Arg[NTimVal].I) */
  
    /* --- Anode */
    if (!pcb->Arg[NAnoVal].I) { 

    AnoI = FALSE;
    /*     AnoScaler1 */
     if (!pcb->Num[NAnoSca].I) {
      if (!ReadHeaderLong(pcb,2,ihb,KHScalerAnode, &pcb->Num[NAnoSca].V,
          pstatus)) { *pstatus = KeywordMissing;
        printf("\nMissing keyword for AnoScaler1 %s\n",KHScalerAnode); }
      if (*pstatus!=Success) return;
      }
    AnoScaler = (int) pcb->Num[NAnoSca].V;
 
    if ((1<=AnoScaler) && (AnoScaler<=ScalerLen)) {
      /*     AnoCount1 */
      if (!pcb->Arg[NAnoCount].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHC,AnoScaler),&AnoCount,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for AnoCount1 %s\n",
            ScalerKey(KHC,AnoScaler));}
        if (*pstatus!=Success) return;
        } else AnoCount = pcb->Arg[NAnoCount].V;

      /*      AnoFac1 */
      if (!pcb->Arg[NAnoFac].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHF,AnoScaler), &AnoFac,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for AnoFac1 %s\n",
            ScalerKey(KHF,AnoScaler)); }
        if (*pstatus!=Success) return;
        } else AnoFac = pcb->Arg[NAnoFac].V;
  
      /*     AnoZero1 */
      if (!pcb->Arg[NAnoZero].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHZ,AnoScaler),&AnoZero,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for AnoZero1 %s\n",
            ScalerKey(KHZ,AnoScaler)); }
        if (*pstatus!=Success) return;
        } else AnoZero = pcb->Arg[NAnoZero].V;
   
      AnoI = TRUE;
      } else {
      AnoCount = 0.0; AnoFac = 1.0;
      } /* if (AnoScaler ... */

    /*     AnoScaler2 */
   if (!pcb->Num[NAnoScaS].I) {
      if (ReadHeaderLong(pcb,2,ihb,KHScalerAnodeS, &pcb->Num[NAnoScaS].V,
          pstatus)) { if (*pstatus!=Success) return;
        } else  pcb->Num[NAnoScaS].V = (long) Default_an2s;
      }
    AnoScaler2 = (int) pcb->Num[NAnoScaS].V;
 
    if ((1<=AnoScaler2) && (AnoScaler2<=ScalerLen)) {
      /*     AnoCount2 */
      if (!pcb->Arg[NAnoCountS].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHC,AnoScaler2), &AnoCount2, 
            pstatus)) { *pstatus = KeywordMissing;
         printf("\nMissing keyword for AnoCount2 %s\n",
           ScalerKey(KHC,AnoScaler2));}
        if (*pstatus!=Success) return;
        } else AnoCount2 = pcb->Arg[NAnoCountS].V;

      /*      AnoFac2 */
      if (!pcb->Arg[NAnoFacS].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHF,AnoScaler2),&AnoFac2,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for AnoFac2 %s\n",
            ScalerKey(KHF,AnoScaler2)); }
        if (*pstatus!=Success) return;
        } else AnoFac2 = pcb->Arg[NAnoFacS].V;

      /*     AnoZero2  (no separate option) */
      if (!pcb->Arg[NAnoZero].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHZ,AnoScaler2),&AnoZero2,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for AnoZero2 %s\n",
            ScalerKey(KHZ,AnoScaler2));
          }
        if (*pstatus!=Success) return;
        } else AnoZero2 = pcb->Arg[NAnoZero].V; 

      AnoI = TRUE;

      /*      Find valid scaler */
      if (AnoCount2*AnoFac2>MaxCount*AnoFac) {
        AnoScaler = AnoScaler2;
        AnoCount  = AnoCount2;
        AnoFac    = AnoFac2;
        AnoZero   = AnoZero2;
        if (pcb->TestBit) printf("Using alternate scaler for anode.\n");
        }
      } /* if (AnoScaler2 ... */

    /*     AnoName */
    if (ReadHeaderLine(pcb,2,ihb,ScalerKey(KHN,AnoScaler), AnoName, 
        pstatus)) { if (*pstatus!=Success) return;
      } else AnoName[0] = (char) 0;

    if (pcb->TestBit) {
      printf("AnoScaler = %d, AnoName = %s\n", AnoScaler, AnoName);
      printf("AnoCount = %g, AnoZero = %g, AnoFac = %g\n", 
         AnoCount, AnoZero, AnoFac);
      }
      } /* if (pcb->Arg[NAnoVal].I) */

    /* --- Transmission monitor */
    if (!pcb->Arg[NTrmVal].I) { 

    TrmI = FALSE;
    /*     TrmScaler1 */
    if (!pcb->Num[NTrmSca].I) {
      if (!ReadHeaderLong(pcb,2,ihb,KHScalerI1,&pcb->Num[NTrmSca].V,
          pstatus)) { *pstatus = KeywordMissing;
        printf("\nMissing keyword for TrmScaler1 %s\n",KHScalerI1); }
      if (*pstatus!=Success) return;
      }
    TrmScaler = (int) pcb->Num[NTrmSca].V;

    if ((1<=TrmScaler) && (TrmScaler<=ScalerLen)) {
      /*     TrmCount1 */
      if (!pcb->Arg[NTrmCount].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHC,TrmScaler),&TrmCount,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for TrmCount1 %s\n",
            ScalerKey(KHC,TrmScaler));}
        if (*pstatus!=Success) return;
        } else TrmCount = pcb->Arg[NTrmCount].V;

      /*      TrmFac1 */
      if (!pcb->Arg[NTrmFac].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHF,TrmScaler),&TrmFac,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for TrmFac1 %s\n",
            ScalerKey(KHF,TrmScaler)); }
         if (*pstatus!=Success) return;
        } else TrmFac = pcb->Arg[NTrmFac].V;

      /*     TrmZero1 */
      if (!pcb->Arg[NTrmZero].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHZ,TrmScaler),&TrmZero,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for TrmZero1 %s\n",
            ScalerKey(KHZ,TrmScaler));}
        if (*pstatus!=Success) return;
        } else TrmZero = pcb->Arg[NTrmZero].V;
      TrmI = TRUE;
      } else {
      TrmCount = 0.0; TrmFac = 1.0; TrmZero = 0.0;
      }

    /*     TrmScaler2 */
    if (!pcb->Num[NTrmScaS].I) {
      if (ReadHeaderLong(pcb,2,ihb,KHScalerI1S, &pcb->Num[NTrmScaS].V,
          pstatus)) { if (*pstatus!=Success) return;
        } else  pcb->Num[NTrmScaS].V = (long) Default_tr2s;
      }
    TrmScaler2 = (int) pcb->Num[NTrmScaS].V;
  
    if ((1<=TrmScaler2) && (TrmScaler2<=ScalerLen)) {
      /*     TrmCount2 */
      if (!pcb->Arg[NTrmCountS].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHC,TrmScaler2),&TrmCount2,
            pstatus)) { *pstatus = KeywordMissing;
            printf("\nMissing keyword for TrmCount2 %s\n",
            ScalerKey(KHC,TrmScaler2));}
        if (*pstatus!=Success) return;
        } else TrmCount2 = pcb->Arg[NTrmCountS].V;

      /*      TrmFac2 */
      if (!pcb->Arg[NTrmFacS].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHF,TrmScaler2),&TrmFac2,
            pstatus)) { *pstatus = KeywordMissing;
          printf("\nMissing keyword for TrmFac2 %s\n",
            ScalerKey(KHF,TrmScaler2));}
        if (*pstatus!=Success) return;
        } else TrmFac2 = pcb->Arg[NTrmFacS].V;

      /*     TrmZero2  (no separate option) */
      if (!pcb->Arg[NTrmZero].I) {
        if (!ReadHeaderFloat(pcb,2,ihb,ScalerKey(KHZ,TrmScaler2),&TrmZero2,
            pstatus)) { *pstatus = KeywordMissing;
         printf("\nMissing keyword for TrmZero2 %s\n",
           ScalerKey(KHZ,TrmScaler2));}
        if (*pstatus!=Success) return;
        } else TrmZero2 = pcb->Arg[NTrmZero].V;
      TrmI = TRUE;

      /*      Find valid scaler */
      if (TrmCount2*TrmFac2>MaxCount*TrmFac) {
        TrmScaler = TrmScaler2;
        TrmCount  = TrmCount2;
        TrmFac    = TrmFac2;
        TrmZero   = TrmZero2;
        if (pcb->TestBit) 
          printf("Using alternative scaler for transmission.\n");
        }
      } /* if (TrmScaler2 ... */

      /*     TrmName */
      if (ReadHeaderLine(pcb,2,ihb,ScalerKey(KHN,TrmScaler),TrmName,
          pstatus)) { if (*pstatus!=Success) return;
         } else TrmName[0] = (char) 0;

    if (pcb->TestBit) {
      printf("TrmScaler = %d, TrmName = %s\n", TrmScaler, TrmName);
      printf("TrmCount = %g, TrmZero = %g, TrmFac = %g\n", \
              TrmCount, TrmZero, TrmFac);
      }
      } /* if (pcb->Arg[NTrmVal].I) */

    /* --- Calculation of the values */
    if (pcb->Arg[NTimVal].I) { TimVal = pcb->Arg[NTimVal].V; TimI = TRUE; }
      else TimVal = TimCount*TimFac; /* time in seconds */
    if (pcb->Arg[NMonVal].I) { MonVal = pcb->Arg[NMonVal].V; MonI = TRUE; }
      else MonVal = (MonCount-MonZero*TimVal) * MonFac; /*#incident photons*/
    if (pcb->Arg[NTrmVal].I) { TrmVal = pcb->Arg[NTrmVal].V; TrmI = TRUE; }
      else TrmVal = (TrmCount-TrmZero*TimVal) * TrmFac; /*#transmitted photons*/
    /* --- Set AnoVal to number of detected photons */
    if (RejectionCorrection)
      if (pcb->Arg[NAnoVal].I) { AnoVal = pcb->Arg[NAnoVal].V; AnoI = TRUE; }
        else AnoVal = (AnoCount-AnoZero*TimVal)*AnoFac; /*#detected photons*/
      else AnoVal = I1Sum*BinCorFac;

    } else {

    /* MonVal */
    if (pcb->Arg[NMonVal].I) MonVal = pcb->Arg[NMonVal].V;
      else {
      if (!ReadHeaderFloat(pcb,2,ihb,KValueI0,&MonVal, pstatus)) { 
        *pstatus = KeywordMissing;
        printf("\nMissing keyword for MonVal %s\n",KValueI0); }
      if (*pstatus!=Success) return;
      }
    sprintf(MonName,"%s",KValueI0);
    MonI = TRUE;

    /* TrmVal */
    if (pcb->Arg[NTrmVal].I) TrmVal = pcb->Arg[NTrmVal].V;
      else {
      if (!ReadHeaderFloat(pcb,2,ihb,KValueI1,&TrmVal, pstatus)) { 
        *pstatus = KeywordMissing;
        printf("\nMissing keyword for TrmVal %s\n",KValueI1); }
      if (*pstatus!=Success) return;
      }
    sprintf(TrmName,"%s",KValueI1);
    TrmI = TRUE;

    /* TimVal */
    if (pcb->Arg[NTimVal].I) TimVal = pcb->Arg[NTimVal].V;
      else {
      if (!ReadHeaderFloat(pcb,2,ihb,KValueTime,&TimVal, pstatus)) {
        *pstatus = KeywordMissing;
        printf("\nMissing keyword for TimVal %s\n",KValueTime); }
      if (*pstatus!=Success) return;
      }
    sprintf(TimName,"%s",KValueTime);
    TimI = TRUE;

    /* --- Set AnoVal to number of detected counts */
    if (RejectionCorrection) {
      if (pcb->Arg[NAnoVal].I) AnoVal = pcb->Arg[NAnoVal].V;
        else {       
        if (!ReadHeaderFloat(pcb,2,ihb,KValueAnode,&AnoVal, pstatus)) { 
          *pstatus = KeywordMissing; 
          printf("\nMissing keyword for AnoVal %s\n",KValueAnode); } 
        if (*pstatus!=Success) return;
        }
      AnoI = TRUE;
      } else AnoVal = I1Sum*BinCorFac;
    sprintf(AnoName,"%s",KValueAnode);

    } /* if (ScalerOn) */

  *pstatus = Success;

  /* --- Transmission */
  if (pcb->Arg[NTrm].I) {
    Trm = pcb->Arg[NTrm].V;
    TrmVal = MonVal * Trm;
    sprintf(TrmName,"trm*mon");
    } else 
    if (TrmI) {
      if (fabs(MonVal)>eps) Trm = TrmVal/MonVal; else Trm = 0.0;   
      } else {  
      Trm = 1.0;
      TrmVal = MonVal;
      sprintf(TrmName,"%s",MonName);
      }

    if (pcb->TestBit) {
      printf("TimVal = %g, AnoVal = %g, \n", TimVal, AnoVal);
      printf("MonVal = %g, TrmVal = %g, Trm = %g\n", MonVal, TrmVal, Trm);
      }

  /* --- Calculation of the normalization factor */
 
  NormOK = FALSE;
  Divider = I1Sum*MonVal*dis*pix1*pix2*Trm;
  if (ABS(Divider)>eps) {
    NormFac  = (AnoVal/Divider);
    NormOK = TRUE;
    }

  if (pcb->TestBit) 
    printf("I1Sum    = %g, NormFac = %g * rd3\n",I1Sum, NormFac);

  /* --- Write info */

  DetVal     = I1Sum*BinCorFac;

  if ( NODUMMY(I1Max,I1Dummy,I1DDummy) ) {
    MaxPixVal = I1Max*BinCorFac;
    } else MaxPixVal = -1;

  if ( (ABS(TimVal)>eps) && (NODUMMY(I1Max,I1Dummy,I1DDummy)) ) {
    MaxPixRate = I1Max*BinCorFac/TimVal;
    } else MaxPixRate = -1;

  if (ABS(TimVal)>eps) EffCountRate = (BinCorFac*I1Sum)/TimVal;
    else EffCountRate = -1;

  if (ABS(AnoVal)>eps) Accepted = (BinCorFac*I1Sum)/AnoVal;
    else Accepted = -1.0;

  if (pcb->TestBit)  {
    printf("Detector counts = %g\n", DetVal);
    printf("Accepted        = %g\n", Accepted);
    printf("EffCountRate    = %g\n", EffCountRate);
    printf("MaxPixVal       = %g\n", MaxPixVal);
    printf("MaxPixRate      = %g\n", MaxPixRate);
    }

if (ihb[1].Title.I) printf("\n%s\n",ihb[1].Title.V);

if (NormOK) {
  printf("\n Calculating ihb[0,% d] = %9.4g * (",ihb[0].ImNum,NormFac); 
    printf("ihb[% d,% d]", 1, ihb[1].ImNum);
    printf(")\n");
  } else printf("\n Calculating ihb[0,% d] = DUMMY\n",ihb[0].ImNum);
printf(" Total number of incident photons (%12s)               = %9.3g ph\n",\
         MonName, MonVal);
printf(" Total number of transmitted photons (%12s)            = %9.3g ph\n",\
         TrmName, TrmVal);
printf(" Counting time (%12s) = %9.3g s, transmission      = %9.3g\n",\
         TimName, TimVal, Trm);
if (RejectionCorrection)
  printf(" Anode counts (%12s)  = %9.3g  , Detector counts   = %9.3g\n",\
           AnoName, AnoVal, I1Sum*BinCorFac); else 
  printf("               %12s     %9s    Detector counts   = %9.3g\n",\
           " "," ", I1Sum*BinCorFac);
printf(" Maximum detector counts      = %9.3g ct/pix\n",\
         MaxPixVal);
printf(" Maximum detector rate        = %9.3g ct/s/pix\n",\
         MaxPixRate);
if (RejectionCorrection)
 printf(" Rejection                 = %9.3g %%, Effective count rate = %9.3g /s\
 \n", (1.0-Accepted)*100, EffCountRate); else
 printf("                             %9s    Effective count rate = %9.3g /s\
 \n", " ", EffCountRate); 
if (FlatFieldCorrection) printf(" Division by flatfield\n");
printf("\n");

 /* loop over 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

    if (NormOK) {
    /* loop over the output array */
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
    for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
      W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
      dd2  = INDEX2R(i_1,off1-cen1,pix1);
      rd20 = dis*dis + dd2*dd2;
      for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
        dd2 = INDEX2R(i_2,off2-cen2,pix2);
        rd2 = rd20 + dd2*dd2;
        rd3 = rd2*sqrt(rd2);

        pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);

        if (FlatFieldCorrection) {
          /* normalization and flatfield correction */
          if ( Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
             f_1[1], f_2[1], &I1Value ) &&
            Ipol2d (IFData,IFDim_1,IFDim_2,IFDummy,IFDDummy,
             f_1[INflat], f_2[INflat], &IFValue )) {

            /* then normalize and divide */

            if (IFValue>eps) *pI0Data = (I1Value / IFValue) * NormFac * rd3;
            } /* if (Ipol2d ... */ 
          } else {
          /* normalization only */
          if (Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
             f_1[1], f_2[1], &I1Value )) {

            /* then normalize */
            *pI0Data = I1Value * NormFac * rd3;

            } /* if (Ipol2d ... */
          } /* if (FlatFieldCorrection) */

        W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
        } /* for i_2 ... */
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
      } /* for i_1 ... */
   }
 } /* saxs_normn*/

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

   DefFlgOption( &cb, SFlat,    NFlat);
   DefFlgOption( &cb, SCcd,     NCcd);
   DefFlgOption( &cb, STrmOn,   NTrmOn);
   DefFlgOption( &cb, SScalOn,  NScalOn); cb.Flg[NScalOn].V = TRUE;
   DefIntOption( &cb, SMonSca,  NMonSca); 
   DefIntOption( &cb, STimSca,  NTimSca);
   DefIntOption( &cb, SAnoSca,  NAnoSca);
   DefIntOption( &cb, STrmSca,  NTrmSca);
   DefIntOption( &cb, SMonScaS, NMonScaS);
   DefIntOption( &cb, STimScaS, NTimScaS);
   DefIntOption( &cb, SAnoScaS, NAnoScaS);
   DefIntOption( &cb, STrmScaS, NTrmScaS);
   DefIntOption( &cb, SScaLen,  NScaLen);
   DefFloatOption( &cb, STrm,       NTrm);
   DefFloatOption( &cb, SScaDepth,  NScaDepth);
   DefFloatOption( &cb, SMonCount,  NMonCount);
   DefFloatOption( &cb, SMonZero,   NMonZero);
   DefFloatOption( &cb, SMonFac,    NMonFac);
   DefFloatOption( &cb, SMonVal,    NMonVal);
   DefFloatOption( &cb, STimCount,  NTimCount);
   DefFloatOption( &cb, STimFac,    NTimFac);
   DefFloatOption( &cb, STimVal,    NTimVal);
   DefFloatOption( &cb, SAnoCount,  NAnoCount);
   DefFloatOption( &cb, SAnoZero,   NAnoZero);
   DefFloatOption( &cb, SAnoFac,    NAnoFac);
   DefFloatOption( &cb, SAnoVal,    NAnoVal);
   DefFloatOption( &cb, STrmCount,  NTrmCount);
   DefFloatOption( &cb, STrmZero,   NTrmZero);
   DefFloatOption( &cb, STrmFac,    NTrmFac);
   DefFloatOption( &cb, STrmVal,    NTrmVal);
   DefFloatOption( &cb, SMonCountS, NMonCountS);
   DefFloatOption( &cb, SMonFacS,   NMonFacS);
   DefFloatOption( &cb, STimCountS, NTimCountS);
   DefFloatOption( &cb, STimFacS,   NTimFacS);
   DefFloatOption( &cb, SAnoCountS, NAnoCountS);
   DefFloatOption( &cb, SAnoFacS,   NAnoFacS);
   DefFloatOption( &cb, STrmCountS, NTrmCountS);
   DefFloatOption( &cb, STrmFacS,   NTrmFacS);
   DefFloatOption( &cb, SI1sum,     NI1sum);
   DefFloatOption( &cb, SBinCorFac, NBinCorFac);
   DefFloatOption( &cb, SRelDis,    NRelDis);

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, BlockNum );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_normn, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

