/*+++
1 saxs_csub
  Subtraction of a point mirrored sequence from itself 

2 PURPOSE
  Subtraction of a point mirrored sequence from itself 

  Arguments:
  saxs_csub [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc>  
             <odum> <odim1> <odim2> <center1> <center2>"

  Defaults:
  <input sequence 1>  : input.edf 
  <output sequence>   : output.edf
  <first image>       : <first image number in sequence 1>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <first image>       : <first image number in sequence 2>
  <output dummy>      : <dummy value in first image of input file>
  <output dimension 1>: <horizontal dimension of first image in input file>
  <output dimension 2>: <vertical dimension of first image in input file>

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  06-Oct-1996 PB
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  2000-08-04  PB %f -> %g
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "saxs_csub V4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
             <i1nam> <onam> <i1fst> <i1lst> <i1inc>\n\
             <odum> <odim1> <odim2> <center1> <center2>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

# define SCen1     "cen1" /* float */
# define NCen1     0
# define SCen2     "cen2"
# define NCen2     1

/*---------------------------------------------------------------------------
1 saxs_csub

2 PURPOSE
  Subtraction of a point mirrored sequence from a sequence
---------------------------------------------------------------------------*/

void saxs_csub (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ const int NumberOfImages=2;
  int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value;

  float WCenter_1, WCenter_2;
  float I1MirrorValue;
  float I1Off_1, I1Off_2;
  float I1Cen_1, I1Cen_2;
  float f_12, f_22;


  /* Do not change the following definitions,
     they are automatically changed with BlockNum */
  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;
  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];
  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];
  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=NumberOfImages) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

 /* 2 identical input files, 1 output */
  I0Data   = ihb[0].Data;
  I0Dummy  = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Data   = ihb[1].Data;
  I1Dummy  = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

 /* Don't calculate overlap, use full output image */
  fmin_1[0] = (float) 0.0;
  fmin_2[0] = (float) 0.0;
  fmax_1[0] = (float) (I0Dim_1-1);
  fmax_2[0] = (float) (I0Dim_2-1);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("fmin_1[% d] = % g, fmax_1[% d] = % g\n",i,fmin_1[i],i,fmax_1[i]);
    printf("fmin_2[% d] = % g, fmax_2[% d] = % g\n",i,fmin_2[i],i,fmax_2[i]);
    } /* for (i ... */

 /* loop over 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  Wmin_1 = WORLD(Imin_1[0],Off_1[0],Ps_1[0]);
  Wmin_2 = WORLD(Imin_2[0],Off_2[0],Ps_2[0]);
  Wmax_1 = WORLD(Imax_1[0],Off_1[0],Ps_1[0]);
  Wmax_2 = WORLD(Imax_2[0],Off_2[0],Ps_2[0]);

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  /* calculate the delta values of the indices */
  if (maxloop_1>1)
    Df_1[0]=(fmax_1[0]-fmin_1[0])/(maxloop_1-1); else Df_1[0]=0.0;
  if (maxloop_2>1)
    Df_2[0]=(fmax_2[0]-fmin_2[0])/(maxloop_2-1); else Df_2[0]=0.0;

  /* get actual center */
  if (pcb->Arg[NCen1].I) {
    WCenter_1 = pcb->Arg[NCen1].V;
    } else {
    if (ihb[1].Offset[1].I) I1Off_1 = ihb[1].Offset[1].V;
      else {printf("Missing header information (Offset[1])\n");exit(-1);}
    if (ihb[1].Center[1].I) I1Cen_1 = ihb[1].Center[1].V;
      else {printf("Missing header information (Center[1])\n");exit(-1);}
    WCenter_1 = WORLD(I2INDEX(I1Cen_1,I1Off_1),Off_1[1],Ps_1[1]);
    }
  if (pcb->Arg[NCen2].I) {
    WCenter_2 = pcb->Arg[NCen2].V;
    } else {
    if (ihb[1].Offset[2].I) I1Off_2 = ihb[1].Offset[2].V;
      else {printf("Missing header information (Offset[2])\n");exit(-1);}
    if (ihb[1].Center[2].I) I1Cen_2 = ihb[1].Center[2].V;
      else {printf("Missing header information (Center[2])\n");exit(-1);}
    WCenter_2 = WORLD(I2INDEX(I1Cen_2,I1Off_2),Off_2[1],Ps_2[1]);
    }

  if (pcb->TestBit) {
    printf("Wmin_1 = % g, Wmax_1 = % g, maxloop_1 = %d\n",
            Wmin_1,Wmax_1,maxloop_1);
    printf("Wmin_2 = % g, Wmax_2 = % g, maxloop_2 = %d\n",
            Wmin_2,Wmax_2,maxloop_2);
    printf("WCenter_1 = % g, WCenter_2 = % g\n",WCenter_1,WCenter_2);
    printf("fmin_1[% d] = % g, Df_1[% d] = % g\n",0,fmin_1[0],0,Df_1[0]);
    printf("fmin_2[% d] = % g, Df_2[% d] = % g\n",0,fmin_2[0],0,Df_2[0]);
    } /* for (i ... */

  /* loop over the output array */
  W_1 = Wmin_1; f_1[0]=fmin_1[0];
  for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
    f_1[1] = INDEX(W_1,Off_1[1],Ps_1[1]);
    f_12   = INDEX(WCenter_1+WCenter_1-W_1,Off_1[1],Ps_1[1]);
    W_2 = Wmin_2; f_2[0]=fmin_2[0];
    for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
      f_2[1] = INDEX(W_2,Off_2[1],Ps_2[1]);
      f_22   = INDEX(WCenter_2+WCenter_2-W_2,Off_2[1],Ps_2[1]);

      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);

      if (Ipol2ld (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
            f_1[1], f_2[1], &I1Value ) &&
          Ipol2ld (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
            f_12, f_22, &I1MirrorValue )) {

        /* then subtract both values */
        UPDATE( *pI0Data, I1Value - I1MirrorValue, I0Dummy, I0DDummy );

        } /* if (Ipol2ld ... */

      W_2+=DW_2; f_2[0]+=Df_2[0]; 
      } /* for i_2 ... */
    W_1+=DW_1; f_1[0]+=Df_1[0]; 
    } /* for i_1 ... */

 } /* saxs_csub*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument : world coordinate 1 of center */
  argv_coord(pcb,pcb->RSys.V,ROff_1,RPs_1,pcb->USys.V,UOff_1,UPs_1,
             "Center 1",&pcb->Arg[NCen1],
         WORLD(I2INDEX(ihb[1].Center[1].V,ihb[1].Offset[1].V),ROff_1,RPs_1),
             pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : world coordinate 2 of center */
  argv_coord(pcb,pcb->RSys.V,ROff_2,RPs_2,pcb->USys.V,UOff_2,UPs_2,
             "Center 2",&pcb->Arg[NCen2],
         WORLD(I2INDEX(ihb[1].Center[2].V,ihb[1].Offset[2].V),ROff_2,RPs_2),
             pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  if (pcb->Arg[NCen1].I) 
    printf("center 1 (%s)    : %g\n",reftostr(pcb->RSys.V),pcb->Arg[NCen1].V);
  if (pcb->Arg[NCen2].I) 
    printf("center 2 (%s)    : %g\n",reftostr(pcb->RSys.V),pcb->Arg[NCen2].V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

 /* Define options */
  DefFloatOption( &cb, SCen1,       NCen1);
  DefFloatOption( &cb, SCen2,       NCen2);

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_csub, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

