/*+++
1 saxs_circle
  Multiplies the input image with a circular intensity pattern

2 PURPOSE
  Multiplies the input image with a circular intensity pattern
  <output image> = <input image> * <slop> * pow( Radius(W_1, W_2), <pow> )
  where Radius is the distance between the zero point (in world 
  coordinates) and the actual world coordinates (W_1, W_2).

  Arguments:
  saxs_circle [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <ofac> <ocon> <slop> <pow>"

  Options:
  -slop <slop>
  -pow  <pow>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <output const>      : 0.0
  <output factor>     : 1.0
  <slop>              : 1.0 (intensity increase per rsys unit)

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  15-Nov-2000 PB V3.00 created from saxs_mac
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "saxs_circle V4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <ofac> <ocon> <slop[1.0]> <pow[1.0]>\n\
\n\
                Special options: -slop <slop>, -pow <pow>\n\
                <output> = <input>*<slop>*pow(radius,<pow>)"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Special options */
# define SSlop "slop" /* float */
# define NSlop 0
# define SPow "pow" 
# define NPow 1

/*---------------------------------------------------------------------------
1 saxs_circle

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void saxs_circle (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  float W_12, W_22;
  float slop, power, radius;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

     GetImageOverlap       ( pcb, ihb, 1, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

 /* loop over 1 input and 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  slop  = pcb->Arg[NSlop].V;
  power = pcb->Arg[NPow].V;
  /* loop over the output array  */
  for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
  for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1[0],i_2);
    for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
    W_2 = WORLD( i_2, Off_2[0], Ps_2[0] );
    W_22 = W_2 * W_2;
    for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
      if (Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
        f_1[1], f_2[1], &Value)) {
        /* then do something with the data */
        W_1 = WORLD( i_1, Off_1[0], Ps_1[0] );
        W_12 = W_1 * W_1;

        radius = sqrt(W_12+W_22);
        if (!((power<0)&&(radius<=0))) {
          Value *= slop*pow(radius,power);
          UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
          }
        } /* if (Ipol2d ... */
      pI0Data++;
      for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
      } /* for i_1 ... */
    for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
    } /* for i_2 ... */

 } /* saxs_circle*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_float(pcb,"Multiplication factor",&ib[0].Factor,1.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_float(pcb,"Addition constant",&ib[0].Const,0.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : slop */
  printf("<output image> = <input image> * <slop> * (Radius(W_1,W_2))^<pow>\n");
  argv_float(pcb,"Slope",&pcb->Arg[NSlop],pcb->Arg[NSlop].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : power */
  printf("<output image> = <input image> * <slop> * (Radius(W_1,W_2))^<pow>\n");
  argv_float(pcb,"Power",&pcb->Arg[NPow],pcb->Arg[NPow].V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  if (ib[0].Factor.I)  printf("factor             : %g\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("constant           : %g\n",ib[0].Const.V);
if (pcb->Arg[NSlop].I) printf("slope              : %g\n",pcb->Arg[NSlop].V);
if (pcb->Arg[NPow].I)  printf("power              : %g\n",pcb->Arg[NPow].V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

 /* Define special options */
  DefFloatOption( &cb, SSlop, NSlop);
  cb.Arg[NSlop].V = 1.0; /* default */
  DefFloatOption( &cb, SPow, NPow);
  cb.Arg[NPow].V = 1.0; /* default */

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_circle, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

