# define REFRACT_VERSION      "refract : V1.11 Peter Boesecke 2001-05-11"
/*+++------------------------------------------------------------------------
NAME
   refract --- routines for gisaxs refraction correction

SYNOPSIS

   # include refract.h

HISTORY
  2001-03-31 V1.0 
  2001-04-08 V1.1 approximation for Beamp.cosAlpha<<1
  2001-05-31 V1.11 warnings added in refract_Init

DESCRIPTION

  Refraction correction of 2d-surface scattering patterns. The calculation
  is valid for grazing incidence small angle scattering (GISAXS) and
  higher angle scattering. The incoming beam is refracted at the surface 
  scattered inside the sample and and again refracted at the surface before
  it hits the detector. This routine transforms the observed (external) 
  scattering pattern into the internal scattering pattern that would be 
  observed inside the sample. The calculation is done using Snell's law:

  (i)    cos(Tau) = (1-Delta)*cos(Taup).

  Tau (Sigma) is the angle between the incoming (outgoing) beam and the 
  surface, Taup (Sigmap) is the angle between the surface and the beam 
  inside the sample. n=(1-delta) is the refractive index of the sample. 
  Angles and vectors inside the sample a marked with a p at the end. 

  The following R-H-O-coordinate (right-handed orthogonal) system is used:

  axis 1: horizontal, axis 2: vertical, axis 3: axis1 X axis2

  The following relationship is used for calculation:

  (a x b) * (c x d) = (a * c) * (b * d) - (b * c) * (a * d)

  The direction of the incoming beam is denoted by ki^ (^ marks a unit
  vector). The direction of the scattered beam is described by ko^. The
  surface normal is N^.

  ki^, ko^ and N^ are parameterized in the following way:

  (ii)      ki^ = (  0,
                     0,
                    -1 )

  (iii)     ko^ = ( sin(2Theta)*cos(Alpha),
                    sin(2Theta)*sin(Alpha),
                    -cos(2Theta)            )

  The scattering angle between ki^ and ko^ is 2Theta. Alpha corresponds to 
  a ccw rotation of the scattered beam around axis3. The zero angle is 
  along axis1. 

  (iv)      N^  = ( cos(Chi),
                    cos(Psi)*sin(Chi),
                    sin(Psi)*sin(Chi) )

  N^ is pointing initially along axis1 and is first rotated ccw (with Chi)
  around axis3. Then N^ is rotated around axis1 (with Psi). 

  Rem: The angle Chi corresponds to the angle -CHI of the ID01/ESRF goniometer,
       and the angle Psi corresponds to the angle Theta of the goniometer.

  The angle Xi between the projections of ki^ and ko^ on the surface is the 
  same as the angle between the projections of the rays kip^ and kop^ inside
  the surface. This is expressed by the following products (*: scalar product,
  x: vector product):
 
  (v)     (ki^  x N^) * (ko^  x N^) = cos(Tau)  * cos(Sigma)  * cos(Xi)
        = (kip^ x N^) * (kop^ x N^) = cos(Taup) * cos(Sigmap) * cos(Xi)

  These products can be separated in the following way:
  
  (vi)    (ki^  x N^) * (ko^  x N^) =
                      (ki^ * ko^) * (N^ * N^) - (N^ * ko^) * (ki^ * N^)

  The scalar products can be calculated:

  (vii)   ki^ * N^  = cos(pi/2+Tau) = -sin(Tau) 
  (viii)  ko^ * N^  = cos(pi/2-Tau) =  sin(Sigma)
  (xi)    ki^ * ko^                 =  cos(2Theta)

  Inserted these values into eq. (vi) leads to an equation that combines
  the scattering angles:

  (x)     (ki^  x N^) * (ko^  x N^) =
                      cos(2Theta) * 1 + sin(Sigma) * sin(Tau)  

  or, by inserting the result into eq. (v) 

  (xi)    cos(Xi) = (cos(2Theta) +sin(Sigma) *sin(Tau) )/(cos(Tau) *cos(Sigma) )
                  = (cos(2Thetap)+sin(Sigmap)*sin(Taup))/(cos(Taup)*cos(Sigmap))

  The angles Tau and Taup are fixed during an experiment:

  (xii)   - (ki^ * N^) = sin(Psi) * Sin(Chi) = sin(Tau) 

  Taup can be calculated with eq. (i).

  By substituting eq. (iii) and (iv) in eq. (viii) leads to 

  (xiii) sin(Sigma)  =   cos(Chi) *            sin(2Theta)  * cos(Alpha) 
                       + cos(Psi) * sin(Chi) * sin(2Theta)  * sin(Alpha)
                       - sin(Psi) * sin(Chi) * cos(2Theta),

  where sin(Psi)*sin(Chi) can be replaced by sin(Tau) (eq. xii). The virtual 
  detection plane insde the sample is defined by the refracted beam kip and 
  the surface normal N. The virtual detection plane is normal to kip. The 
  vertical direction is defined as the projection of the surface normal N on 
  this plane. The inside R-H-O-coordinate system is: 

  (xiv)         e1p^ = ( kip^ x N^ ) / cos(Taup)
                e2p^ = ( kip^ x N^ ) x kip^ / cos(Taup)  
                e3p^ = -kip^

  Alphap is the angle between the axis e2p^ and the projection of kop^ on
  the virtual detection plane:

  (xv)   ( kip^ x kop^ ) * ( kip^ x N^ ) / ( sin(2Thetap) * cos(Taup) ) 
                     = cos(pi/2-Alphap) = sin(Alphap)
                         
  The vector products can be expressed by scalar products
  (xvi)  ( kip^ x kop^ ) * ( kip^ x N^ ) 
                     = ( kip^*kip^ ) * ( kop^*N^ ) - ( kop^*kip^ ) * ( kip^*N^ )
                     = 1 * sin(Sigmap) + cos(2Thetap)  * sin(Taup)

  By solving eq. (xvi) to sin(Sigmap) and replacing the vector products by 
  eq. (xv) one obtains:

  (xvii) sin(Sigmap) = sin(Alphap) * sin(2Thetap) * cos(Taup)
                       - cos(2Thetap) * sin(Taup) 

  eq. (xvii) allows the calculation of Sigmap from 2Thetap and Alphap. 
  Tau and Taup are known from eq. (xii) and (i). Inserting Sigmap and Sigma 
  (obtained with eq. (i)) into eq. (xi) returns cos(2Theta). 

  (xviii) cos(2Theta) = (cos(2Thetap)+sin(Sigmap)*sin(Taup))*(1-delta)^2
                         - sin(Sigma)*sin(Tau)

  After 2Theta is known, cos(Alpha) and sin(Alpha) are calculated with 
  help of eq. (xiii).

  (xix)   x1 = -p/2 + sqrt((p/2)^2-q) and x2 = -p/2 + sqrt((p/2)^2-q)

  For the calculation of x=sin(Alpha) p and q are as follows:

  (xx)    x1 = sin(Alpha1) and x2 = cos(Alpha2)

         cos(Psi)*sin(Chi)*(sin(Sigma)+sin(Tau)*cos(2Theta))
  -p/2 = ---------------------------------------------------
                        cos(Tau)^2 * sin(2Theta)

         (sin(Sigma)+sin(Tau)*cos(2Theta))^2-(cos(Chi)*sin(2Theta))^2
     q = ------------------------------------------------------------
                        cos(Tau)^2 * sin(2Theta)^2

  Solution x2 is valid for cos(Alphap)*cos(Chi)>0 otherwise x1.

  For the calculation of x=cos(Alpha) p and q are as follows:

  (xxi)   x1 = cos(Alpha1) and x2 = cos(Alpha2)


         cos(Chi) * (sin(Sigma)+sin(Tau)*cos(2Theta))
  -p/2 = --------------------------------------------
                   cos(Tau)^2 * sin(2Theta)^2

         (sin(Sigma)+sin(Tau)*cos(2Theta))^2-(cos(Psi)*sin(Chi)*sin(2Theta))^2
     q = ---------------------------------------------------------------------
                              cos(Tau)^2 * sin(2Theta)^2

  Solution x1 is valid for cos(Alphap)*sin(Chi)>0, otherwise x2.

  Special solutions are required for cos(Alaphp)<<1 and for n=1, 2Theta=0. 
  For n=1 and 2Theta=0 eq. xx sin(Alpha) and cos(Alpha) have only single 
  solutions.  Eq. xx and eq. xxi change to:

                       sin(Alphap)*cos(Psi)*sin(Chi) - cos(Chi)*cos(Alphap) 
  (xxii)  sin(Alpha) = ----------------------------------------------------
                                           cos(Tau) 

                       sin(Alphap)*cos(Chi) + cos(Psi)*sin(Chi)*cos(Alphap)
  (xxiii) cos(Alpha) = ----------------------------------------------------
                                           cos(Tau)

  The equations (xx) and (xxi) are not well conditioned for cos(Alphap)<<1. 
  There, sin(Sigma)+sin(Tau)*cos(2Theta) can be approximated as: 

  (xxiv)  sin(Sigma)+sin(Tau)*cos(2Theta) ~= sin(2Theta-Tau) - 
             1/2*n*cos(Tau)*sin(2Thetap)*sin(2Thetap-Taup)*cos(Alphap)^2+o(Alphap^4)
  
  The special solution for cos(Alphap)<<1 is:

  (xxv)  sin(Alpha) ~=  1 for cos(Alphap)*cos(Chi) < sin(Alphap)*sin(Chi),
                       -1 for cos(Alphap)*cos(Chi) > sin(Alphap)*sin(Chi)
 
                       cos(Chi)   cos(Psi)*sin(Chi)*cos(Alphap)
  (xxvi) cos(Alpha) ~= -------- + ----------------------------- *
                       cos(Tau)            cos(Tau)

                                 sin(2Thetap)*sin(2Thetap-Taup)
                          * sqrt(------------------------------ * n)
                                   sin(2Theta)*sin(2Theta-Tau) 

----------------------------------------------------------------------------*/
/******************************************************************************
* Include Files                                                               *
******************************************************************************/

# include "refract.h"

/******************************************************************************
* Private Constants                                                           *
******************************************************************************/

# define R_PI 3.1415926535897932384626

static const double deg2rad = R_PI/180.0;
static const double rad2deg = 180.0/R_PI;
static const double eps     = 1e-30;
static const double sqrteps = 1e-8;
static const double coseps  = 7.7e-2; // cosAlphap*cosTwoThetap/sinTwoThetap
static const double pi      = R_PI;
static const double halfpi  = R_PI*0.5;
static const double twopi   = R_PI*2.0;
static const double one     = 1.0;

/******************************************************************************
* Private Type Defs                                                           *
******************************************************************************/

typedef struct refract_params {
  int    Init;
  double Delta;
  double IndexN;
  double sinPsi;
  double cosPsi;
  double sinChi;
  double cosChi;
  double cosPsisinChi;
  double cosPsisinChiDcosTau;
  double cosChiDcosTau;
  double sinTau;
  double cosTau;
  double cos2Tau;
  double sinTaup;
  double cosTaup;

} RParams;

/******************************************************************************
* Private Variables                                                           *
******************************************************************************/

static RParams RefractParams;

/******************************************************************************
* Routines                                                                    *
******************************************************************************/

void refract_PrintParams ( FILE * out )
{ RParams * pParams = &RefractParams;
  if (!pParams->Init) return;
  fprintf(out," Delta               = %g\n", pParams->Delta);
  fprintf(out," IndexN              = %g\n", pParams->IndexN);
  fprintf(out," sinPsi              = %g\n", pParams->sinPsi);
  fprintf(out," cosPsi              = %g\n", pParams->cosPsi); 
  fprintf(out," sinChi              = %g\n", pParams->sinChi);
  fprintf(out," cosChi              = %g\n", pParams->cosChi);
  fprintf(out," cosPsisinChi        = %g\n", pParams->cosPsisinChi);
  fprintf(out," cosPsisinChiDcosTau = %g\n", pParams->cosPsisinChiDcosTau);
  fprintf(out," cosChiDcosTau       = %g\n", pParams->cosChiDcosTau);
  fprintf(out," sinTau              = %g\n", pParams->sinTau);
  fprintf(out," cosTau              = %g\n", pParams->cosTau);
  fprintf(out," cos2Tau             = %g\n", pParams->cos2Tau);
  fprintf(out," sinTaup             = %g\n", pParams->sinTaup);
  fprintf(out," cosTaup             = %g\n", pParams->cosTaup);
} //  refract_PrintParams 

void refract_PrintBeam ( FILE * out, RefractDir Beam )
{ 
  fprintf(out," sinAlpha           = %g\n", Beam.sinAlpha );
  fprintf(out," cosAlpha           = %g\n", Beam.cosAlpha );
  fprintf(out," sinTwoTheta        = %g\n", Beam.sinTwoTheta );
  fprintf(out," cosTwoTheta        = %g\n", Beam.cosTwoTheta );

} // refract_PrintBeam

/*+++------------------------------------------------------------------------
NAME
  refract_set --- set RefractDir 

SYNOPSIS

  RefractDir refract_set( RefractDir *pbeam, int status )

DESCRIPTION
 
  Changes error status in RefractDir to status.
  Other parameter are not changed.

RETURN VALUE

  RefractDir Beam with error status

----------------------------------------------------------------------------*/
RefractDir refract_set( RefractDir *pbeam, int status )
{ 
  pbeam->status = status;
  return( *pbeam );

} // refract_set 

/*+++------------------------------------------------------------------------
NAME
  refract_Init --- Initialisation of parameters

SYNOPSIS

  int refract_Init ( FILE * out, double Delta, double Psi, double Chi )

DESCRIPTION
It initializes all static parameters that do not depend on TwoThetap and Alphap.
Warnings are written to out.

RETURN VALUE
  returns status 0 if OK

----------------------------------------------------------------------------*/
int refract_Init ( FILE * out, double Delta, double Psi, double Chi )
{ RParams * pParams = &RefractParams;
  double tmp;

  pParams->Init   = 0; 
  pParams->sinPsi = sin(Psi);
  pParams->cosPsi = cos(Psi);
  pParams->sinChi = sin(Chi);
  pParams->cosChi = cos(Chi);
  pParams->cosPsisinChi = pParams->cosPsi*pParams->sinChi;

  // 0<= Tau <= pi
  pParams->sinTau = pParams->sinPsi * pParams->sinChi;
  tmp = pParams->sinTau;
  pParams->cosTau = (fabs(Psi)<=halfpi)?sqrt(one-tmp*tmp):-sqrt(one-tmp*tmp);
  pParams->cos2Tau = pParams->cosTau*pParams->cosTau;

  if ((pParams->sinTau<0)&&(fabs(Delta)>eps)) {
    fprintf(out, "\n WARNING: Incidence angle is less than zero: %g deg\n",
      asin(pParams->sinTau)*rad2deg);
    return( 1 );
    }

  // -1 <= cos(Tau) <= 1, 0 <= sin(Tau) <= 1
  pParams->Delta  = Delta;
  pParams->IndexN = one-Delta;
  if (pParams->IndexN<eps) {
    fprintf(out,"\n WARNING: Refractive index 1-Delta is too small: %g < %g\n",
      pParams->IndexN, eps);
    return( 2 );
    }

  pParams->cosTaup = pParams->cosTau/pParams->IndexN;
  tmp = pParams->cosTaup*pParams->cosTaup;
  if (tmp>one) {
    fprintf(out, 
   "\n WARNING: Incident beam is totally reflected (critical angle = %g deg)\n",
      (pParams->IndexN<1)?acos(pParams->IndexN)*rad2deg:0);
    return( 3 );
    }
  pParams->sinTaup = (pParams->sinTau>0)?sqrt(one-tmp):-sqrt(one-tmp);

  pParams->cosPsisinChiDcosTau = pParams->cosPsisinChi/pParams->cosTau;
  pParams->cosChiDcosTau = pParams->cosChi/pParams->cosTau;

  pParams->Init = 1;

  return( 0 );

} // refract_Init

/*+++------------------------------------------------------------------------
NAME
  refract_Angles --- GISAXS Refraction correction of 2Theta angle 

SYNOPSIS

  RefractDir refract_Angles ( RefractDir Beamp )

DESCRIPTION

  The external 2Theta angle is calculated by Snell's law from the 
  internal angles TwoThetap and Alphap.

  1-Delta is the real part of the refractive index
  psi is the inclination of the sample surface with respect to the 
  incoming beam
  chi is the tilt of the inclined sample 

  All angles are in radian.

RETURN VALUE
  
  .status==0 : sinTwoTheta, cosTwoTheta, sinAlpha, cosAlpha angles in rad
               (external angles)
  .status!=0 : error 

----------------------------------------------------------------------------*/
RefractDir refract_Angles ( RefractDir Beamp )
{ double Sigma, Sigmap;
  double sinSigma, sinSigmap;
  double cosSigma, cosSigmap;
  double sin2TwoTheta;
  double tmp, CC, DD, factor;
  double p1half, p1half2, q1; // for cosAlpha
  double p2half, p2half2, q2; // for sinAlpha
  RefractDir Beam;
  RParams * pParams = &RefractParams;

  // pParams initialized
  if (!pParams->Init) return(refract_set(&Beam,-1));

  // TwoThetap >= 0
  if (Beamp.sinTwoTheta<0) return(refract_set(&Beam,-2));

  // eq. xvii
  sinSigmap = Beamp.sinAlpha * Beamp.sinTwoTheta * pParams->cosTaup 
                - Beamp.cosTwoTheta * pParams->sinTaup;
  tmp = sinSigmap*sinSigmap;
  if (tmp>one) return(refract_set(&Beam,-3));
  cosSigmap = sqrt(one-tmp);

  cosSigma  = cosSigmap*pParams->IndexN;
  tmp = cosSigma*cosSigma;
  if (tmp>one) return(refract_set(&Beam,-4));
  sinSigma = (sinSigmap>0)?sqrt(one-tmp):-sqrt(one-tmp);

  // 0 <= Sigma <= pi
  if (sinSigma<0) return(refract_set(&Beam,-5));

  // eq. xi
  Beam.cosTwoTheta = (Beamp.cosTwoTheta + sinSigmap*pParams->sinTaup)
                   *pParams->IndexN*pParams->IndexN 
                  - sinSigma*pParams->sinTau ;

  // TwoTheta >= 0.0
  if (Beam.cosTwoTheta>one) return(refract_set(&Beam,-6));
  sin2TwoTheta = one-Beam.cosTwoTheta*Beam.cosTwoTheta;
  Beam.sinTwoTheta  = sqrt(sin2TwoTheta);

  // Alpha

  if (fabs(pParams->Delta)>eps) {
    if (fabs(Beamp.cosAlpha)>coseps) {
      factor = 1.0/(pParams->cos2Tau*sin2TwoTheta);
      CC     = sinSigma+pParams->sinTau*Beam.cosTwoTheta;

      // calculation of cosAlpha

      p1half  = - pParams->cosChi*Beam.sinTwoTheta*CC*factor; 
      p1half2 = p1half*p1half;
      DD      = pParams->cosPsisinChi*Beam.sinTwoTheta;
      q1      = (CC-DD)*(CC+DD)*factor;

      tmp = p1half2 - q1;

      if (tmp < sqrteps )
        if (tmp < -sqrteps) return(refract_set(&Beam,-7));
          else tmp = 0.0;

      if (Beamp.cosAlpha*pParams->sinChi>0) Beam.cosAlpha = -p1half + sqrt(tmp);
        else Beam.cosAlpha = -p1half - sqrt(tmp);

      // calculation of sinAlpha

      p2half  = - pParams->cosPsisinChi*Beam.sinTwoTheta*CC*factor;
      p2half2 = p2half*p2half;
      DD = pParams->cosChi*Beam.sinTwoTheta;
      q2 = (CC-DD)*(CC+DD)*factor;

      tmp = p2half2 - q2;

      if (tmp < sqrteps )
        if (tmp < -sqrteps) return(refract_set(&Beam,-8));
          else tmp = 0.0;

      if (Beamp.cosAlpha*pParams->cosChi>0) Beam.sinAlpha = -p2half - sqrt(tmp);
        else Beam.sinAlpha = -p2half + sqrt(tmp);
      } else {
      // Beamp.cosAlpha<coseps

      // calculation of cosAlpha
      tmp = (Beamp.sinTwoTheta/Beam.sinTwoTheta) * pParams->IndexN
          * ( (Beamp.sinTwoTheta*pParams->cosTaup-Beamp.cosTwoTheta*pParams->sinTaup)
             /(Beam.sinTwoTheta*pParams->cosTau-Beam.cosTwoTheta*pParams->sinTau) );

      if (tmp>=0) tmp=sqrt(tmp); else return(refract_set(&Beam,-9));

      Beam.cosAlpha = 
         (pParams->cosChiDcosTau + Beamp.cosAlpha * pParams->cosPsisinChiDcosTau*tmp);

      // calculation of sinAlpha
      tmp = sqrt(1.0-Beam.cosAlpha*Beam.cosAlpha);
      if (Beamp.cosAlpha*pParams->cosChi<Beamp.sinAlpha*pParams->sinChi) Beam.sinAlpha=tmp;
        else Beam.sinAlpha=-tmp;
      }
    } else {
    // Delta == 0.0

    Beam.cosAlpha =  
      (Beamp.sinAlpha*pParams->cosChi+Beamp.cosAlpha*pParams->cosPsisinChi)
       / pParams->cosTau;

    Beam.sinAlpha =  
      (Beamp.sinAlpha*pParams->cosPsisinChi-Beamp.cosAlpha*pParams->cosChi)
       / pParams->cosTau;

    }

  return( refract_set( &Beam, 0 ) );

} // refract_Angles
