/*+++
1 SaxsImageLoop.inc

2 DESCRIPTION
  Private include file of "SaxsImage.c".
  See PUBLIC function for details.

3 CALL
  void ImageLoop( CmdBlk * pcb, ImgBlk ib[],
                  void (*Function) (CmdBlk *, ImgHeadBlk [], int * ),
                  int allneeded, int * pstatus );

2 HISTORY
  12-Oct-1996 Peter Boesecke extracted from SaxsImage.c
  14-Nov-1996 PB mark_...
  21-Dec-1999 PB DummyFileName, IO_Dummy
  18-Mar-2000 PB default for output memory (saxs_image_loop) 
  30-Dec-2000 PB FreeImage removed for all input images 
  03-Jul-2001 PB FullFileName(in,out) -> !filename_full( out, IO_LEN, in )
  06-Jul-2001 PB filename_full now in OpenImageFile
  08-Jul-2001 PB OpenImageFile(...,FileNumber,...)
  09-Jul-2001 PB loop over file numbers, repetitions removed
  09-Jul-2001 PB single2multi, multi2single
  10-Jul-2001 PB saxs_new_all_images, saxs_open_all_images, 
                 saxs_read_all_images: only required images are read
  11-Jul-2001 PB single2multi, multi2single
  11-Jul-2001 PB NewImage only when output image header empty
                 control parameter Add
  12-Jul-2001 PB saxs_open_all_images opens only new output file when
                 image block ihb[0] is empty (!ihb[0].I).
  19-Aug-2001 PB WriteImage argument list has changed
  30-Aug-2001 PB single2multi -> SaxsImageLoop_single2multi
                 multi2single -> SaxsImageLoop_multi2single
                 ImageIs..., ImageToWrite
---*/

/****************************************************************************
*  External Functions                                                          *
****************************************************************************/

void free_image_memory ( ImgHeadBlk *pihb );

/****************************************************************************
*  Defines                                                                  *
****************************************************************************/

#ifndef PRIVATE
#  define PRIVATE static
#endif

#ifndef PUBLIC
#  define PUBLIC
#endif

/*---------------------------------------------------------------------------
 Static variables 
----------------------------------------------------------------------------*/
PRIVATE int SaxsImageLoop_single2multi = 0, SaxsImageLoop_multi2single = 0;

/*---------------------------------------------------------------------------
1 ImageIsNew
 
2 DESCRIPTION
  Return TRUE if ActualLoopCount <= 0, otherwise FALSE 
 
  int blkno             (i)   : blkno
  ImgHeadBlk *pihb      (i)   : image header block
----------------------------------------------------------------------------*/
int ImageIsNew( int blkno, ImgHeadBlk ihb[] )
{ return((ihb[blkno].ActualLoopCount>0)?FALSE:TRUE);
} // ImageIsNew

/*---------------------------------------------------------------------------
1 ImageToWrite
 
2 DESCRIPTION
  Return TRUE, when image will be written at end of loop (pihb->Write)
 
  int blkno             (i)   : blkno
  ImgHeadBlk *pihb      (i)   : image header block
----------------------------------------------------------------------------*/                                                                
int ImageToWrite( int blkno, ImgHeadBlk ihb[] )
{ return( ihb[blkno].Write );
} // ImageToWrite

/*---------------------------------------------------------------------------
1 ImageIsFirst 
 
2 DESCRIPTION
  Return TRUE, when image is first image in input or output file 
 
  CmdBlk * pcb          (i)   : control block
  int blkno             (i)   : blkno
  ImgHeadBlk *pihb      (i)   : image header block
----------------------------------------------------------------------------*/
int ImageIsFirst( CmdBlk * pcb, int blkno, ImgHeadBlk ihb[] )
{ int value;
  value = (ihb[blkno].ImNum == pcb->ib[blkno].First.V)?1:0;
  if ((SaxsImageLoop_multi2single)&&(blkno==0)) { // if blkno output block no
    value = (pcb->ib[0].FileNumber == pcb->ib[0].FFirst.V)?value:0; 
    }
  return( value );
} // ImageIsFirst 

/*---------------------------------------------------------------------------
1 ImageIsLast
 
2 DESCRIPTION
  Return TRUE, when image is last image in input or output file
 
  CmdBlk * pcb          (i)   : control block
  int blkno             (i)   : blkno
  ImgHeadBlk *pihb      (i)   : image header block
----------------------------------------------------------------------------*/
int ImageIsLast( CmdBlk * pcb, int blkno, ImgHeadBlk ihb[] )
{ int value;
  value = ((ihb[blkno].ImNum==pcb->ib[blkno].Last.V)||
           (pcb->ib[blkno].Inc.V==0))?1:0;
  if ((SaxsImageLoop_multi2single)&&(blkno==0)) { // if blkno output block no
    value = ((pcb->ib[0].FileNumber==pcb->ib[0].FLast.V)||
             (pcb->ib[0].FInc.V==0))?value:0;
    }
  return( value );
} // ImageIsLast

/*---------------------------------------------------------------------------
1 saxs_free_all_images

2 DESCRIPTION
  Releases memories of all initialized images

  ImgHeadBlk *pihb      (i/o)   : image header block

2 HISTORY
  24-Nov-1995   Peter Boesecke creation
  10-Jul-2001   PB release only images of initialized blocks

----------------------------------------------------------------------------*/
void saxs_free_all_images ( CmdBlk * pcb, ImgHeadBlk ihb[] )
/* Release all image memory */
{ register int i;
  for (i=0;i<pcb->ImgBlkLen;i++) {
    free_image_memory( &ihb[i] );
  }
} /* saxs_free_all_images */

/*---------------------------------------------------------------------------
1 saxs_new_all_images

2 DESCRIPTION
  Allocates an array with imax image header blocks and sets the
  indicators ihb[i].I to FALSE.

  int pcb->seb.ExternalError (o): error value
  int *pstatus (o)              : SAXS status

2 RETURN VALUE
  Pointer to created ImgHeadBlk array or NULL in case of an error;

2 HISTORY
  10-Jul-2001 PB 
----------------------------------------------------------------------------*/
ImgHeadBlk * saxs_new_all_images( int imax )
{ int i;
  ImgHeadBlk *ihb;

  ihb = (ImgHeadBlk*) malloc(sizeof(ImgHeadBlk)*imax);
  if (ihb==NULL) return(ihb);

  // Init .I-flag and pointer to Data
  for (i=0;i<imax;i++) {
    ihb[i].I    = FALSE;
    ihb[i].Data = (float *) NULL;
  }

  return( ihb );

} // saxs_new_all_images

/*---------------------------------------------------------------------------
1 saxs_open_all_images

2 DESCRIPTION
  Opens all data files according to the command block cb and the image
  blocks ib[i] (all defined data files from 0 to ImgBlkLen-1). 

  Checks, whether a data file is already opened and closes and reopens it
  only if necessary. 
  If a file cannot be opened the function tries either to open the remaining 
  files (allneeded==FALSE) or stops and returns 0. In case that all 
  required files could be opened the function returns 1. The stream 
  ib[].Stream of an unopened file is -1.  

  return( int )               : 1 for all needed files are opened
                                When 0 is returned saxs_image_loop
                                does not need not be called.

  CmdBlk *pcb           (i)   : ImgBlkLen
  ImgBlk ib[]           (i)   : OpenMode, Name
                        (o)   : Stream
  long int FNum[]       (i)   : FileNumber table (one value for each ib)
  ImgHeadBlk ihb[]      (o)   : image memory is released if file was closed
  int allneeded         (i)   : FALSE : The function tries to open all files 
                                        and skippes files that cannot be 
                                        opened due to a FileNotFound status. 
                                        If no other error has occured the 
                                        function returns 1, otherwise 0.
                                 TRUE : If a file is missing, the function
                                        stops without error and returns 0.
                                        If all files could be opened it 
                                        returns 1.

  int pcb->seb.ExternalError (o): error value
  int *pstatus (o)              : SAXS status

2 HISTORY
  12-Feb-1995   Peter Boesecke creation
  24-Nov-1995 PB
  09-Jul-2001 PB checks, whether a file is already opened
  10-Jul-2001 PB allneeded,
                 A data array of an image header block of a closed and
                 reopened file is released and the header block flag .I 
                 is set to FALSE.
  11-Jul-2001 PB output image 0 opened at the end
  12-Jul-2001 PB output image file is only opened when image block is empty
----------------------------------------------------------------------------*/
int saxs_open_all_images( CmdBlk *pcb, ImgBlk ib[], long FNum[], 
                          ImgHeadBlk ihb[], int allneeded, int * pstatus )
{
  int imax = pcb->ImgBlkLen;
  register int i;

  for (i=imax-1;0<=i;i--) { // open output image at the end to avoid empty files

    if (!(ib[i].OpenMode.I && (IO_DontOpen & ib[i].OpenMode.V)))
      if (!(ib[i].Name.I && ib[i].OpenMode.I))
        printf("WARNING (ImgBlk[% d]) : file name or open mode missing\n",i);

      if (!AlreadyOpened( pcb,ib,i,ib[i].Name.V,FNum[i],pstatus )) {
        // open only a new output file  when image block is empty (!ihb[0])
        if ( (i!=0)||(!ihb[i].I) ) { 
          // close and open
          CloseImageFile ( pcb, ib, i, pstatus);
          if (*pstatus!=Success) return(0);
          // release allocated image data and mark header block as empty
          FreeImage( pcb, i, ihb ); 
          OpenImageFile(pcb,ib,i,ib[i].Name.V,FNum[i],
                        ib[i].OpenMode.V,pstatus);
          if (*pstatus==FileNotFound) {
            *pstatus=Success;
            if (allneeded) return(0); // some files not found, stop
          }
        if (*pstatus!=Success) return(0);
        }
      } 

      if (pcb->TestBit) {
        printf("ib[% d].FileName = % s (stream = %d) opened\n",
               i,ib[i].FileName,ib[i].Stream);
        }

    } // for i

  *pstatus = Success;

  return(1);

} /* saxs_open_all_images */

/*---------------------------------------------------------------------------
NAME

  saxs_read_all_images --- read all required images 

SYNOPSIS

  int saxs_read_all_images( CmdBlk *pcb, ImgBlk ib[], long Num[],
                          ImgHeadBlk * ihb, int allneeded, int * pstatus );

2 DESCRIPTION
  Reads all images of ib[i] with image number Num[i] if they are not already
  in ihb[i]. The images must have been opened with saxs_open_all_images.

  Checks, whether an image needs to be read. If ihb[i] does not contain 
  the requested image the old image is released with FreeImage and the
  requested image is read from file.
  If an image cannot be read because the file is closed (stream<0) or
  the image number is not found in the file the image header block .I flag
  is set to false. Depending on the parameter allneeded the routine 
  stops (allneeded==TRUE) or tries to load the resting images 
  (allneeded==FALSE).

  return( int )               : 1 for all needed images are read 
                                When 0 is returned function
                                must not be called.
                                If returned value is 1 status must be
                                checked.
  CmdBlk *pcb           (i)   : ImgBlkLen
  ImgBlk ib[]           (i)   : OpenMode, Name
                        (o)   : Stream
  long int FNum[]       (i)   : FileNumber table (one value for each ib)
  ImgHeadBlk ihb[]      (o)   : image memory is released if file was closed
  int allneeded         (i)   : FALSE : The function tries to read all images 
                                        and skippes images that cannot be
                                        found. If no other error has occured 
                                        the function returns 1, otherwise 0.
                                 TRUE : If an image is missing, the function
                                        stops without error and returns 0.
                                        If all files could be opened it
                                        returns 1.

  int pcb->seb.ExternalError (o): error value
  int *pstatus (o)              : SAXS status

2 HISTORY
  10-Jul-2001 Peter Boesecke 
----------------------------------------------------------------------------*/
int saxs_read_all_images( CmdBlk *pcb, ImgBlk ib[], long Num[],
                          ImgHeadBlk * ihb, int allneeded, int * pstatus )
{
  int imax = pcb->ImgBlkLen;
  register int i;
  int retval=1;

  /* read input image(s) */
  for (i=1;i<imax;i++) {
    // test, whether new image must be loaded
    if ( (ihb[i].I)&&(ihb[i].ImNum!=Num[i]) ) { 
      // free image header
      FreeImage( pcb, i, ihb );
    }

    if  (!ihb[i].I) {
     // comment
      if (!((IO_Dummy | IO_DontReadWrite) & ib[i].StreamOpenMode))
        if (IO_DontReadData & ib[i].StreamOpenMode) {
          printf("Reading header: % s, Image : % d",
                 ib[i].FileName, Num[i]);
        } else {
          printf("Reading image: % s, Image : % d",
                 ib[i].FileName, Num[i]);
        } // comment

      /* +++ read image ihb[i] with image number Num[i] */
      if (!(retval=ReadImage(pcb,ib,i,Num[i],ihb,pstatus))) {
        printf(" missing\n");
      } else {
        if (*pstatus!=Success) { printf("\n"); return(1); } // error, check st.
        if (!((IO_Dummy | IO_DontReadWrite ) & ib[i].StreamOpenMode))
          printf(" done\n");
      }
      if (allneeded && (!retval)) break; else retval=1;
    }
  } /* for (i=1;... */

  *pstatus = Success;

  return(retval);

} // saxs_read_all_images

/*---------------------------------------------------------------------------
1 saxs_close_all_images

2 DESCRIPTION
  Closes all data files according to the command block cb and the image
  blocks ib[i].

  close all data files
  return( void )                :

  CmdBlk *pcb           (i)     : ImgBlkLen
  ImgBlk ib[]           (i)     : Stream
                        (o)     : Stream
  int pcb->seb.ExternalError (o): error value
  int *pstatus		(o)	: SAXS status

2 HISTORY
  12-Feb-1995   Peter Boesecke creation
  24-Nov-1995 PB
----------------------------------------------------------------------------*/
void saxs_close_all_images( CmdBlk *pcb, ImgBlk ib[], int * pstatus )
{ register int i;

  for (i=0;i<pcb->ImgBlkLen;i++) {
    CloseImageFile(pcb,ib,i,pstatus);
    if (*pstatus!=Success) return;
    }
  *pstatus = Success;

} /* saxs_close_all_images */

/*---------------------------------------------------------------------------
1 saxs_image_loop

2 PURPOSE
  N input sequences, 1 output sequence.
  Perform sequence analysis: one output image per calculation.
  Output[i0] = Function(image1[i1], image2[i2], image3[i3], ..., imageN[iN])
  In the case that the increment is zero, the corresponding image is only
  one times read at the beginning or one times written at the end:
  The loop is defined as follows,  t also negative increments are possible,
  if First and Last are set respectively:

  ImageLoop(...) {
  for (O=OFirst,I1=I1First,I2=I2First,...;
       O<OLast,I1<I1Last,I2<I2Last,...;O+=OInc,I1+=I1Inc,I2+=I2Inc) {
    Read(I1,I2,...);
    Function((CmdBlk * ) pcb, (ImgHeadBlk *) ihb, (int *) pstatus );
    Write(O); }
  }

  Not existing images are skipped.

2 PARAMETERS
  return value                  : void
  CmdBlk        * pcb           :  ommand block
  ImgBlk        ib[]            : image block
  void (*Function) (CmdBlk *,ImgHeadBlk *, int *) : image loop
  int           allneeded (i)   : FALSE : at least one image must be found,
                                  TRUE : if one image is missing, Function
                                         is not called and data not written
  ImgHeadBlk    ihb[]           : image header block
  int           *pstatus        : exit status

The following parameters of *pcb and ib[] are used:

pcb->ImgBlkLen  Number of image blocks>=1

ib[].First, ib[].Last, ib[].Inc
- default of sequence 1 is: 1, 1, 1
- sequence 1 is default for all following sequences

2 HISTORY
  27-Feb-1995 PB in case of ib[0].Inc==0, the output image is only allocated
                 once (for loop=1) and written only once for (loop=loopmax)
  07-Jun-1995 PB Template only, if at least one input image exist
                 (if (imax>1) Template ...)
  14-Nov-1995 PB For modified saxs_image routines.
  24-Nov-1995 PB small modifications for new routines
  25-Dec-1995 PB loop, loopmax, maxloop, Num, First, Last, Inc : int --> long
  26-Dec-1995 PB long --> long int
  18-Mar-2000 PB ib[0].Memory (default of output memory is input memory) 
  10-Jul-2001 PB saxs_free_all_images removed
---------------------------------------------------------------------------*/
void saxs_image_loop( CmdBlk * pcb, ImgBlk ib[],
                      void (*Function) (CmdBlk *, ImgHeadBlk [], int * ),
                      int allneeded, ImgHeadBlk ihb[], int * pstatus )
{ int imax;
  int i,j;
  int allfound;
  int firstblock;
  long int loop,loopmax;

  /* dynamic allocation of memory */
  long int *maxloop, *Num, *First, *Last, *Inc;

  *pstatus = NotEnoughMemoryAvailable;

  imax = pcb->ImgBlkLen;
  maxloop = (long int *) malloc(sizeof(int)*imax); if (maxloop==NULL) return;
  Num     = (long int *) malloc(sizeof(int)*imax); if (Num==NULL) return;
  First   = (long int *) malloc(sizeof(int)*imax); if (First==NULL) return;
  Last    = (long int *) malloc(sizeof(int)*imax); if (Last==NULL) return;
  Inc     = (long int *) malloc(sizeof(int)*imax); if (Inc==NULL) return;

  *pstatus = Success;

  /* Choose loop parameters */
  if (imax>1) {
    if (ib[1].First.I) First[1] = ib[1].First.V; else First[1] = 1l;
    if (ib[1].Last.I) Last[1] = ib[1].Last.V; else Last[1] = 1l;
    if (ib[1].Inc.I) Inc[1] = ib[1].Inc.V; else Inc[1] = 1l;
    if (ib[0].First.I) First[0] = ib[0].First.V; 
       else { ib[0].First.V = First[0] = First[1]; }
    if (ib[0].Last.I) Last[0] = ib[0].Last.V; 
       else { ib[0].Last.V = Last[0] = Last[1]; }
    if (ib[0].Inc.I) Inc[0] = ib[0].Inc.V; 
       else { ib[0].Inc.V = Inc[0] = Inc[1]; }
    if (!(ib[0].Memory.I)) ib[0].Memory.V = ib[1].Memory.V;
    } else {
    if (ib[0].First.I) First[0] = ib[0].First.V; 
       else ib[0].First.V = First[0] = 1l;
    if (ib[0].Last.I) Last[0] = ib[0].Last.V; 
       else ib[0].Last.V = Last[0] = 1l;
    if (ib[0].Inc.I) Inc[0] = ib[0].Inc.V; 
       else ib[0].Inc.V = Inc[0] = 1l;
    } /* if (imax>1) */

  /* All other sequences */
  for (i=2;i<imax;i++) {
    if (ib[i].First.I) First[i] = ib[i].First.V; else First[i] = First[1];
    if (ib[i].Last.I) Last[i] = ib[i].Last.V; else Last[i] = Last[1];
    if (ib[i].Inc.I) Inc[i] = ib[i].Inc.V; 
      else if (ib[i].Last.I) Inc[i] = Inc[1]; else Inc[i] = 0l;
    } /* for */

  /* Avoid infinite loops in case of Inc[i]==0 by setting maxloop[0]=1 in
     case that all Inc[i]==0 */

  for (i=0;i<imax;i++) {
    if (Inc[i]!=0l)
      maxloop[i] = MAX2(1l,1l + ((Last[i])-(First[i]))/(Inc[i]));
      else maxloop[i] = -1l; /* infinite */
    } /* i */

  loopmax=-1l; /* infinite */
  for (i=0;i<imax;i++)
    if (loopmax>0l) {
      if (maxloop[i]>0) {loopmax=MIN2(loopmax,maxloop[i]);}
      } else {
      if (maxloop[i]>0l) {loopmax=maxloop[i];}
      } /* if (loopmax>0l) */
  loopmax = MAX2(1l,loopmax);

  /* start values */
  for(i=0;i<imax;i++) { Num[i]=First[i]; }
  /* loop */
  for (loop=1l; loop<=loopmax; loop++) {

       // +++ read all required images
       saxs_read_all_images( pcb, ib, Num, ihb, allneeded, pstatus );
       if (*pstatus!=Success) { return; }

       /* +++ test, whether all required images are available */
       allfound = TRUE; firstblock = 0;
       for (i=imax-1;i>0;i--)
         if (ihb[i].I) firstblock = i; else allfound=FALSE;

       for (i=1;i<imax;i++) {
          if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);
          } /* for (i=1;... */
       /*----------------------------------------------------------------
        Create output image if all needed image headers have been read :
          allneeded=TRUE  : all headers are needed,
          allneeded=FALSE : at least one image must have been found
                            (firstblock != 0).
        allfound is always TRUE if no input images are required (imax=1).
        In the case that ib[0].Inc.V=0 the output header is only created
        once (when loop=1) and written once (when loop=loopmax).
        allfound || (   (!allneeded) && (firstblock!=0) )
        TRUE            FALSE           *               TRUE
        FALSE           FALSE           *               FALSE
        FALSE           TRUE            TRUE            TRUE
        FALSE           TRUE            FALSE           FALSE
       ------------------------------------------------------------------*/
       if ( (allfound) || ( (!allneeded) && (firstblock!=0) ) ) {
         /* Create output image only when ihb is empty (ihb[0].I=0) */
         if ( (!ihb[0].I)&&(!ihb[0].Data) ) {
           /* Use first valid image as template */
           NewImage(pcb,ib,0,Num[0],ihb,&ihb[firstblock],pstatus);
           if (*pstatus!=Success) { return; }
           } /* ((ib[0]. ... */

         /* Decide, whether output image should be written
            and set ihb[0].Write to TRUE if it should.
            Usually, write always the output image,
            if Add.I is set write only every Add.V's image,
            if Inc==0 write output image only once at end of loop */
         if ( ( (pcb->Add.I)&&((ihb[0].DataCtrl+1)>=(pcb->Add.V)) ) ||
              ( (!pcb->Add.I)&&((ib[0].Inc.V!=0l) || (loop==loopmax)) ) ) {
           ihb[0].Write = TRUE;
           }

         if (pcb->TestBit) PrintImageHeaderBlock(0,&ihb[0]);

        /********************* Calculation ********************************/

            SetSaxsError( &pcb->seb, pcb->MainName, "Function", NULL,
                          Success, 0, NULL );
            Function(pcb,ihb,pstatus);
            if (*pstatus!=Success) {return;}

        /******************************************************************/

         if (pcb->TestBit) PrintImageHeaderBlock(0,&ihb[0]);

         /* incrementation of ActualLoopCount */
         for(i=0;i<imax;i++) ihb[i].ActualLoopCount++;

         } /* if ( (allfound) ... */

       /* incrementation of DataCtrl */
       for(i=0;i<imax;i++) ihb[i].DataCtrl++;

       // +++ decide whether output image should be written
       if ( ImageToWrite( 0, ihb ) ) { 
         if (!((IO_Dummy | IO_DontReadWrite) & ib[0].StreamOpenMode)) { 
           printf("Writing : % s, Image : % d\n",
                   ib[0].FileName, ihb[0].ImNum); }

         WriteImage(pcb,ib,0,ihb,pstatus);
         if (pcb->TestBit) PrintImageHeaderBlock(0,&ihb[0]);
         if (*pstatus!=Success) { return; }
         printf("\n");
       } // if ImageToWrite 

       // +++ decide whether output image should be released 
       /* Usually, free always the output image,       
          if Add.I is set free only every Add.V's image, 
          if Inc==0 free output image only once at end of loop */
       if ( ( (pcb->Add.I)&&((ihb[0].DataCtrl)>=(pcb->Add.V)) ) ||
            ( (!pcb->Add.I)&&((ib[0].Inc.V!=0l) || (loop==loopmax)) ) ) 
         FreeImage( pcb, 0, ihb ); /* if ((ib[0]. ... */

       /* incrementation of loop variables */
       for(i=0;i<imax;i++) Num[i]+=Inc[i];

       } /* for (loop ... */

     free(maxloop); free(Num); free(First); free(Last); free(Inc); 

     *pstatus = Success;

} /* saxs_image_loop */

/*---------------------------------------------------------------------------
1 ImageLoop

2 PURPOSE
  Loop over file numbers (FFirst, FLast, FInc)
  open all files (saxs_open_all_images),
  do sequence analysis (see saxs_image_loop),
  close all files (saxs_close_all_images).

  single2multi : If the input filename does not contain percent characters
                 as placeholders for numbers and if the output filename
                 contains placeholders for numbers all images of the input
                 file are written to numbered output files. 

                 Input image number parameters (first, last, increment) are 
                 overwritten by input file number (if set). 

                 The default output image number is 1. 
                 
                 The default of the output file numbers are the input
                 image numbers, e.g.

                 "saxs_mac input.edf,2,10 output%%%.edf" writes images 
                 2 to 10 toe output002.edf to output010.edf with image 
                 number 1.

  multi2single:  If the input filename contains percent characters
                 as placeholders for file numbers and if the output 
                 filename does not contain placeholders all images 
                 i1first of the numbered input files are written 
                 into a single output file. 
                 
                 Output image number parameters (first, last, increment)
                 are overwritten by output file numbers. Default
                 output file numbers are input file numbers, e.g.

                 "saxs_mac -i1fst 1 input%%%.edf,2,10 output.edf,3,11" 
                 writes all images with image number 1 in input002.edf
                 to input010.edf to the output.edf with output image
                 numbers 3 to 11 (default would be 2 to 10).

2 CALL
  void ImageLoop( CmdBlk * pcb, ImgBlk ib[],
                  void (*Function) (CmdBlk *, ImgHeadBlk [], int * ),
                  int allneeded, int * pstatus );

2 HISTORY
  1995-11-27 PB
  2001-07-09 PB for file numbers, single2multi, multi2single
---------------------------------------------------------------------------*/
PUBLIC void ImageLoop( CmdBlk * pcb, ImgBlk ib[],
                       void (*Function) (CmdBlk *, ImgHeadBlk [], int * ),
                       int allneeded, int * pstatus )
{ const char * RoutineName = "ImageLoop";
  int f,fmax;
  long floop,floopmax;

  /* dynamic allocation of memory */
  long int *fmaxloop, *FNum, *FFirst, *FLast, *FInc;
  ImgHeadBlk *ihb;

  /* file number and image number conversion */
  int fhp0 = 0, fhp1 = 0;

  /* Reset static variables */
  SaxsImageLoop_single2multi = 0; 
  SaxsImageLoop_multi2single = 0;

  if (pcb->TestBit) printf(" %s\n",RoutineName);
  SetSaxsError( &pcb->seb, ModuleName, RoutineName, NULL, Success, 0, NULL );

  *pstatus = NotEnoughMemoryAvailable;

  fmax = pcb->ImgBlkLen;
  fmaxloop = (long int *) malloc(sizeof(int)*fmax); if (fmaxloop==NULL) return;
  FNum     = (long int *) malloc(sizeof(int)*fmax); if (FNum==NULL) return;
  FFirst   = (long int *) malloc(sizeof(int)*fmax); if (FFirst==NULL) return;
  FLast    = (long int *) malloc(sizeof(int)*fmax); if (FLast==NULL) return;
  FInc     = (long int *) malloc(sizeof(int)*fmax); if (FInc==NULL) return;
  ihb      = (ImgHeadBlk*) saxs_new_all_images( fmax );if (ihb==NULL) return;

  *pstatus = Success;

  /* file number and image number conversion */
  if (fmax>1) {
    fhp0 = filename_has_pattern(ib[0].Name.V);
    fhp1 = filename_has_pattern(ib[1].Name.V);
    if (fhp0!=fhp1)
      if (fhp0) SaxsImageLoop_single2multi=1; else SaxsImageLoop_multi2single=1;
  } // if fmax>1

  /* file number and image number conversion */
  if (SaxsImageLoop_single2multi) {
    /* Input image numbers (first, last, increment) are
       overwritten by input file numbers (if set). */
    if (ib[1].FFirst.I) {
      ib[1].First.V = ib[1].FFirst.V; ib[1].First.I = TRUE; 
    } 
    if (ib[1].FLast.I) {
      ib[1].Last.V = ib[1].FLast.V; ib[1].Last.I = TRUE;
    }
    if (ib[1].FInc.I) {
      ib[1].Inc.V = ib[1].FInc.V; ib[1].Inc.I = TRUE;
    }

    ib[1].FFirst.V = ib[1].First.V; ib[1].FFirst.I = TRUE;
    ib[1].FLast.V  = ib[1].Last.V;  ib[1].FLast.I  = TRUE;
    ib[1].FInc.V   = ib[1].Inc.V;   ib[1].FInc.I   = TRUE;
    // single image per file with image number 1 or ofst if set
    if (!ib[0].First.I) { ib[0].First.V = 1l; ib[0].First.I = TRUE; }
  } // SaxsImageLoop_single2multi

  /* Choose file loop parameters */
  if (fmax>1) {
    if (ib[1].FFirst.I) FFirst[1] = ib[1].FFirst.V; else FFirst[1] = 1l;
    if (ib[1].FLast.I) FLast[1] = ib[1].FLast.V; else FLast[1] = 1l;
    if (ib[1].FInc.I) FInc[1] = ib[1].FInc.V; else FInc[1] = 1l;
    if (ib[0].FFirst.I) FFirst[0] = ib[0].FFirst.V; 
       else ib[0].FFirst.V = FFirst[0] = FFirst[1];
    if (ib[0].FLast.I) FLast[0] = ib[0].FLast.V; 
       else ib[0].FLast.V = FLast[0] = FLast[1];
    if (ib[0].FInc.I) FInc[0] = ib[0].FInc.V; 
       else ib[0].FInc.V = FInc[0] = FInc[1];
    } else {
    if (ib[0].FFirst.I) FFirst[0] = ib[0].FFirst.V; 
      else ib[0].FFirst.V = FFirst[0] = 1l;
    if (ib[0].FLast.I) FLast[0] = ib[0].FLast.V; 
      else ib[0].FLast.V = FLast[0] = 1l;
    if (ib[0].FInc.I) FInc[0] = ib[0].FInc.V; 
      else ib[0].FInc.V = FInc[0] = 1l;
    } /* if (fmax>1) */

  /* All other sequences */
  for (f=2;f<fmax;f++) {
    if (ib[f].FFirst.I) FFirst[f] = ib[f].FFirst.V; else FFirst[f] = FFirst[1];
    if (ib[f].FLast.I) FLast[f] = ib[f].FLast.V; else FLast[f] = FLast[1];
    if (ib[f].FInc.I) FInc[f] = ib[f].FInc.V;
      else if (ib[f].FLast.I) FInc[f] = FInc[1]; else FInc[f] = 0l;
    } /* for */

  /* Avoid infinite loops in case of FInc[f]==0 by setting fmaxloop[0]=1 in
     case that all FInc[f]==0 */

  for (f=0;f<fmax;f++) {
    if (FInc[f]!=0l)
      fmaxloop[f] = MAX2(1l,1l + ((FLast[f])-(FFirst[f]))/(FInc[f]));
      else fmaxloop[f] = -1l; /* infinite */
    } /* f */

  floopmax=-1l; /* infinite */
  for (f=0;f<fmax;f++)
    if (floopmax>0l) {
      if (fmaxloop[f]>0) {floopmax=MIN2(floopmax,fmaxloop[f]);}
      } else {
      if (fmaxloop[f]>0l) {floopmax=fmaxloop[f];}
      } /* if (floopmax>0l) */
  floopmax = MAX2(1l,floopmax);

  /* start values */
  for(f=0;f<fmax;f++) { FNum[f]=FFirst[f]; }

  /* loop */
  for (floop=1l; floop<=floopmax; floop++) {

    /* file number and image number conversion */
    if (SaxsImageLoop_multi2single) { 
     // overwrite output image numbers with output file numbers
      ib[0].First.V = FNum[0]; ib[0].First.I = TRUE;
      ib[0].Inc.V   = 0l;      ib[0].Inc.I = TRUE; 
    }
    if (SaxsImageLoop_single2multi) {
      ib[1].First.V = FNum[1]; ib[1].First.I = TRUE;
      // only 1 loop in saxs_image_loop
      ib[1].Inc.V   = 0l;      ib[1].Inc.I = TRUE;
    }

    // call saxs_image_loop always (check for all images is done inside)
    saxs_open_all_images( pcb, ib, FNum, ihb, allneeded, pstatus );
    if (*pstatus!=Success) { saxs_free_all_images(pcb,ihb); return; }
    saxs_image_loop( pcb, ib, Function, allneeded, ihb, pstatus);
    if (*pstatus!=Success) { saxs_free_all_images(pcb,ihb); return; }

    /* incrementation of file loop variables */
    for(f=0;f<fmax;f++) FNum[f]+=FInc[f];

  } // for (floop ...

  saxs_close_all_images( pcb, ib, pstatus );
  if (*pstatus!=Success) return;

  saxs_free_all_images(pcb,ihb);

  free(fmaxloop); free(FNum); free(FFirst); free(FLast); free(FInc); free(ihb);

  *pstatus = Success;

} /* ImageLoop */

/*---------------------------------------------------------------------------
 mark_float_2d
---------------------------------------------------------------------------*/
# define MARK00( L, E )     L += ( E)
# define MARK06( L, E, V )  L[1] = L[2] = ( V); L += ( E)
# define MARK09( L, E, V )  L[0] = L[3] = ( V); L += ( E)
# define MARK0A( L, E, V )  L[1] = L[3] = ( V); L += ( E)
# define MARK0E( L, E, V )  L[1] = L[2] = L[3] = ( V); L += ( E)
# define MARK10( L, E, V )  L[4] = ( V); L += ( E)
# define MARK14( L, E, V )  L[2] = L[4] = ( V); L += ( E)
# define MARK1C( L, E, V )  L[2] = L[3] = L[4] = ( V); L += ( E)
# define MARK20( L, E, V )  L[5] = ( V); L += ( E)
# define MARK22( L, E, V )  L[1] = L[5] = ( V); L += ( E)
# define MARK24( L, E, V )  L[2] = L[5] = ( V); L += ( E)
# define MARK28( L, E, V )  L[3] = L[5] = ( V); L += ( E)
# define MARK2A( L, E, V )  L[1] = L[3] = L[5] = ( V); L += ( E)
# define MARK2E( L, E, V )  L[1] = L[2] = L[3] = L[5] = ( V); L += ( E)
# define MARK36( L, E, V )  L[1] = L[2] = L[4] = L[5] = ( V); L += ( E)
# define MARK38( L, E, V )  L[3] = L[4] = L[5] = ( V); L += ( E)
# define MARK3A( L, E, V )  L[1] = L[3] = L[4] = L[5] = ( V); L += ( E)
# define MARK3E( L, E, V )  L[1] = L[2] = L[3] = L[4] = L[5] = ( V); L += ( E)
# define MARK67( L, E, V )  L[0] = L[1] = L[2] = L[5] = L[6] = ( V); L += ( E)
# define MARK69( L, E, V )  L[0] = L[3] = L[5] = L[6] = ( V); L += ( E)
# define MARK99( L, E, V )  L[0] = L[3] = L[4] = L[7] = ( V); L += ( E)
# define MARKBA( L, E, V )  L[1]=L[2]=L[3]=L[4]=L[5]=L[7]=( V); L += ( E)

/*---------------------------------------------------------------------------
1 mark_float_2d

2 HISTORY
  14-NOV-1996 Peter Boesecke
---------------------------------------------------------------------------*/
void mark_float_2d ( float *array, long d_1, long d_2, float d, float dd )
/* Marks a region with value dummy + 0.5*ddummy. */
{ const long m_1 = 8, m_2 = 80;
  short *farray, *f;
  float *a, value;
  long i, i_1, i_2;

  if (!(f = farray = (short *) malloc(m_1*m_2*sizeof(short)))) return;
  for (i=0;i<m_1*m_2;i++) *f++ = 0;

  f = farray;
  MARK00(f,m_1);MARK00(f,m_1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK14(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK22(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK2E(f,m_1,1);MARK2A(f,m_1,1);MARK3A(f,m_1,1);MARK00(f,m_1);
  MARKBA(f,m_1,1);MARK22(f,m_1,1);MARKBA(f,m_1,1);MARK00(f,m_1);
  MARK36(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK00(f,m_1);MARK00(f,m_1);
  MARK3A(f,m_1,1);MARK24(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK20(f,m_1,1);MARK3E(f,m_1,1);MARK20(f,m_1,1);MARK00(f,m_1);
  MARK22(f,m_1,1);MARK2A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK38(f,m_1,1);MARK28(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK00(f,m_1),MARK00(f,m_1);
  MARK0E(f,m_1,1);MARK0A(f,m_1,1);MARK3E(f,m_1,1);MARK00(f,m_1);
  MARK3E(f,m_1,1);MARK2A(f,m_1,1);MARK3A(f,m_1,1);MARK00(f,m_1);
  MARK3E(f,m_1,1);MARK2A(f,m_1,1);MARK3A(f,m_1,1);MARK00(f,m_1);
  MARK3E(f,m_1,1);MARK10(f,m_1,1);MARK00(f,m_1);MARK00(f,m_1);
  MARK06(f,m_1,1);MARK69(f,m_1,1);MARK99(f,m_1,1);
  MARK67(f,m_1,1);MARK09(f,m_1,1);MARK09(f,m_1,1);
  MARK06(f,m_1,1);

  /* copy farray to array */
  value = d+0.5*dd;
  f = farray + m_1*(m_2-MIN2(m_2,d_2)); a = array;
  for (i_2 = 0;i_2<MIN2(m_2,d_2);i_2++) {
    for (i_1 = 0;i_1<MIN2(m_1,d_1);i_1++)
      if ((f[i_1]) && (DUMMY(a[i_1],d,dd)) ) a[i_1] = value;
    a+=d_1; f+= m_1;
    }

  free(farray);

} /* mark */

/*---------------------------------------------------------------------------
 mark_float_2d end
---------------------------------------------------------------------------*/

/***************************************************************************/

