"""
    SPSScanGraphFilter.py
    Convertion of circular to linear buffer
"""    

from PyDVT import __version__,__date__,__author__

from Filter import *
from GraphView import GraphFilter
import Numeric


class SPSScanGraphFilter(GraphFilter):
    """
    Filter disigned to interface SPS scan data (SPSData) to ScanGraphView.
    For a shared array arr in Spec, it should exist an shared array arr_ENV 
    with the scan information fields, at least:
      "nopts": number of valid points of the scan      
    The fields of arr_ENV(read by SPSData in "envdict" entry of page info)
    "xmin" and "xmax" are included in it's output.    
    """

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            See GraphFilter.GetOutput
            In addiction:
                "nopts": integer, number of valid points of the scan. If greater than
                         previous, draw just new points.
                "envdict": dictionary with all the scan environment parameters (SPS)
        """
        
        sel=GraphFilter.GetOutput(self)
        if "data" not in sel.keys(): return sel

        data=self.GetData()
        datasel=self.GetDataSelection()
        page=datasel.GetInfo()["index_list"][0]
        page_info=data.GetPageInfo(page)
        
        if "envdict" in page_info.keys():
            if "xmin" in page_info["envdict"].keys(): sel["xmin"]=int(page_info["envdict"]["xmin"])
            if "xmax" in page_info["envdict"].keys(): sel["xmax"]=int(page_info["envdict"]["xmax"])
            if "nopts" in page_info["envdict"].keys(): sel["nopts"]=int(page_info["envdict"]["nopts"])
        return sel


class SPSCircularBufferFilter(Filter):
    """
    Auxiliar filter to linearize circular buffer acquisitions from SPEC.
    This Filter has to be linked (through a DataSelection) to a page of
    Data with the entry "envdict" in it's Info dictionary. This is a
    dictionary itself, with these required entries:
            "cbufsize": size of the circular buffer read with SPS
            "lbufsize": size of the full scan array
            "nopts": number of valid points of the scan
            "scanno": number of the scan (have to change for each new scan)
    This fields are automatically read in a SPSData object if they are
    written into an arrayname_ENV array.
    """
    def __init__(self,*args,**keys):
        """
        Constructor
        See Filter.__init__
        """
        Filter.__init__(self,*args,**keys)
        self.data=None
        self.xdata=None
        self.position=0
        self.scanno=-1
        self.overflow=0

    def GetOutput(self):
        """

        See Filter.GetOutput
        """
        sel=self.GetSource().GetOutput()
        if "data" not in sel.keys(): return {}

        data=self.GetData()
        datasel=self.GetDataSelection()
        page=datasel.GetInfo()["index_list"][0]        

        try:
            arrayinfo=data.GetPageInfo(page)["envdict"]
            cbufsize=int(arrayinfo["cbufsize"])
            lbufsize=int(arrayinfo["lbufsize"])
            nopts=int(arrayinfo["nopts"])
            scanno=int(arrayinfo["scanno"])
        except: return {}

        if scanno!=self.scanno:
            self.position=0
            self.scanno=scanno
            self.overflow=0

        if nopts < cbufsize:
            rptr=self.position
            bytestoread=nopts-self.position            
        else:
            if self.position+cbufsize+10<nopts:
                if self.data is not None and self.overflow==0: print "CIRCULAR BUFFER OVERFLOW"
                self.overflow=1
                return {}
            rptr= self.position % cbufsize
            bytestoread=nopts-self.position

        if self.data is None or len(self.data)!=lbufsize:
            self.data=Numeric.zeros((lbufsize,),sel["data"].typecode())
            self.xdata=Numeric.zeros((lbufsize,),sel["xdata"].typecode())

        if rptr+bytestoread<cbufsize:
            self.data[self.position:self.position+bytestoread]=sel["data"][rptr:rptr+bytestoread]
            if "xdata" in sel.keys():
                self.xdata[self.position:self.position+bytestoread]=sel["xdata"][rptr:rptr+bytestoread]
        else:
            self.data[self.position:self.position+(cbufsize-rptr)]=sel["data"][rptr:cbufsize]
            self.data[self.position+(cbufsize-rptr):self.position+bytestoread]=sel["data"][0:bytestoread-(cbufsize-rptr)]
            if "xdata" in sel.keys():
                self.xdata[self.position:self.position+(cbufsize-rptr)]=sel["xdata"][rptr:cbufsize]
                self.xdata[self.position+(cbufsize-rptr):self.position+bytestoread]=sel["xdata"][0:bytestoread-(cbufsize-rptr)]
        self.position=self.position+bytestoread
        sel["data"]=self.data
        if "xdata" in sel.keys():
            sel["xdata"]=self.xdata

        return sel
    

