/****************************************************************************

                     uMatrix C++ Matrix Library

    Copyright (C) 1996  David Weber, Michael Sipe and Rajesh Shenoy

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    David Weber can be contacted at weber@ece.cmu.edu or 
    http://www.ece.cmu.edu/afs/ece/usr/weber/.home-page.html

****************************************************************************/

#ifndef uIndex_included
#define uIndex_included
#include <stdlib.h>
#include <iostream.h>

/// Index class for \Ref{uMatrix}
/** This class is used to index submatrices without having to use a
for loop. The following example demonstrates the extraction a
submatrix consisting of every other element of a given \Ref{uMatrix}
as well as the extraction of a quadrant from a given \Ref{uMatrix}:
\begin{verbatim}
#include <stdlib.h>
#include <iostream.h>
#include <nMatrix.h>

int main()
{
   // Declare 2 matrices 
   uMatrix<double> A(10,10);
   uMatrix<double> B;
   
   // Assign something to A

   uIndex I(0,2,8);
   uIndex J(0,2,8);
   // Extract submatrix consisting of all even indices in A
   B = A(I,J);
   cout << B;

   // Extract bottom left quadrant from A
   B = A( uIndex(5,1,9), uIndex(0,1,4) );
   cout << B;
   
   return 0;
}
\end{verbatim} */

class uIndex
{
      int Start, Stride, End, Dimension;

   public:
      /// Basic constructor
      /** The constructor takes the {\em start} index, {\em stride}
      and {\em end} index as parameters. The index uIndex I(4,3,12)
      will index every third element starting at index 4 and ending at
      12. If the end index is not reachable (because {\em start} -
      {\em end} is not divisable by {\em stride}) then indexing will
      stop at the last index less than {\em end}. Negative strides are
      allowed if {\em start} > {\em end}.
      */
      uIndex(int start, int stride, int end) 
      { 
	 if ( ( ( end - start ) % stride ) != 0  )
	 {
	    cerr << "Invalid uIndex Start/End/Stride choice in uIndex(i,j,k)\n";
	    abort();
	 } 
	 Start = start; Stride = stride; End = end;
	 Dimension = abs( ( End - Start ) / Stride ) + 1;
      }

      /// Index for a single row or column
      /** This constructor takes a single argument and constructs an
      index using uIndex(i,1,i). This will effectively index only one
      element. */ 
      uIndex(int i) 
      { 
	 if ( i < 0 )
	 {
	    cerr << "Invalid uIndex Start choice in uIndex(i)\n";
	    abort();
	 } 
	 Start = i; Stride = 1; End = i;
	 Dimension = abs( ( End - Start ) / Stride ) + 1;
	 //cout << "uIndex Dimension = " << Dimension << "\n";
      }

      /// Return the {\tt start} parameter
      inline   int start()  const   { return Start; }
      /// Return the {\tt stride} parameter
      inline  int stride()  const   { return Stride; }
      /// Return the {\tt end} parameter
      inline   int end() const    { return End; }
      /// Return the dimension of the index
      inline   int   dim() const { return Dimension; } ;
      /** This method returns the number of elements that will be
      indexed by this uIndex. */
};

#endif







