/****************************************************************************

                     uMatrix C++ Matrix Library

    Copyright (C) 1996  David Weber, Michael Sipe and Rajesh Shenoy

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    David Weber can be contacted at weber@ece.cmu.edu or 
    http://www.ece.cmu.edu/afs/ece/usr/weber/.home-page.html

****************************************************************************/

#ifndef uMatrix_included
#define uMatrix_included
/*
 $Id: nMatrix.h,v 1.1.1.1 2001/06/27 17:07:55 mirone Exp $
 $Author: mirone $
 $Date: 2001/06/27 17:07:55 $
 $Source: /scisoft/users/mirone/SOURCESCVS//nMatrix/nMatrix.h,v $ 
*/

// Comment this out if you do not want shallow copies on assign when
// assigning to the results of a function call

#define uMatrix_constructor_propagate_temporary_enable
#define uMatrix_enable_shallow_copy

#include <stdio.h>
#include "nIndex.h"
#include "nVector.h"
#include <string>
#include <fstream.h>
#include <strstream.h>
#include <ctype.h>
#include <complex.h>
// #include <Random.h>


// For compatibility with different processor like egcs
#include "compatibilita.h"

#ifdef Debug_on
#define DBG(Stuff) Stuff
#else
#define DBG(Stuff) /*don't do this */
#endif

#ifdef stupid_f2c_MAIN
extern "C"
{
     void MAIN__();
}
#endif

/// Matrix classes
/** This is the uMatrix class definition. It provides an easy and
    mostly efficent way to manipulate matrices from C++. It provides
    comprehensive indexing, manipulations, memory management, IO and
    casting operations. It is templated and can be intantiated for any
    type for which the basic math operators are defined. This includes
    char, int, double, Complex etc.

    The classes are mostly complete but the documentation is a work in 
    progress (performed while waiting for some heavy numerical codes to
    finish) so watch this space.
    */
template<class Type>
class uMatrix 
{
   uVector<Type> VectStorage; // Use a vector as a basis for a matrix.
   int  M, N;  // Size of Matrix (rows, cols);
   int shallowCopyFlag;  // Flag to enable shallow copy if construct parameter
   int temporaryFlag;    // Flag to signal this as a temporary variable 
 
   public:
 
      // ** Inizialise the matrix
      /// Inizialise the matrix. useful when allocating unformatted blocks of  memory
      void uMatrix::inizializza();

      // ** Constructor and destructor;
      /// Construct an empty uMatrix
      inline uMatrix( const int m, const int n = 1 );
      /// Construct a uMatrix from a pointer (CAUTION)

      /** Construct a uMatrix from a pointer variable. This
      constructor is provided to allow construction of matrices that
      point to a piece of memory allocated by someone else. If you
      wish to use a uMatrix to address a piece of video memory, this
      is your ticket.  {\em Be warned that absolutely no checks can be
      performed on the pointer.}  */
      inline uMatrix( const int m, const int n, Type data[] ); 
      /// Construct an empty matrix of zero rows and columns
      inline uMatrix(){ M = 0 ; N = 0; shallowCopyFlag = 0; temporaryFlag = 0;}
      /// Construct a uMatrix from another uMatrix
      /** The new matrix will be a {\em deep} copy of the matrix it was
      constructed from.*/
      inline uMatrix( const uMatrix<Type>& data );
      /// Destructor
      /** The \Ref{uVector} class does all the memory management so this is
      empty */
      inline ~uMatrix() { DBG(cout << "~uMatrix() destructor called: this=" << this << "\n";); }

   // ** uIndex and assignment operators
      /// Index operator
      /** Access an element in a uMatrix with index range checking. This is
      an lvalue so it can appear on the left hand side of an expression */
      inline Type& operator()(const int m, const int n = 0) const; 
      /// Index a submatrix
      /** Uses the \Ref{uIndex} class to index a submatrix. Warning: this 
      function does not return a modifyable lvalue so A(I,J)=B compiles 
      but won't work. I'll buy you a beer if you can figure an neat way to 
      do this without a mess.
 */
      uMatrix<Type> operator()(uIndex I, uIndex J) const;

      /// extract  a submatrix
      /** Uses the \Ref{uIndex} class to index a submatrix.  */
      void extract (uIndex I, uIndex J, uMatrix<Type> &res ) const;



      /// Assign to a uMatrix
      /** This assignment performs a deep copy of the uMatrix on the right
      hand side {\em unless} the RHS uMatrix was marked as a temporary. In
      this case, it is assumed that the RHS matrix is transient (such as 
      a function that returns a uMatrix constructed from inside the function).
      In this case, a shallow copy is performed by deallocating any existing
      storage and simpy copying the pointer rather than all the data. In all 
      cases, the LHS uMatrix is resized to be the same size as the RHS 
      uMatrix. */ 
      inline void operator=(const uMatrix<Type>& data); 
      /// Assign to a scalar
      /** Fill all of the elements of the LHS uMatrix with the scalar
      argument on the RHS. The LHS uMatrix dimensions are unchanged. */
      inline void operator=(Type); //scalar fill
      /// Assign to a pointer (CAUTION)
      /** Assign the LHS uMatrix to a pointer value. The dimensions of the 
      uMatrix are unchanged but the existing storage is deallocated and the
      new storage is set to the RHS pointer argument. {\em Be warned that 
      absolutely no checks can be performed on the pointer.} */
      inline void operator=(Type[]); // assign to pointer


   // ** Maths operators
      ///
      uMatrix<Type>& operator-=( const uMatrix<Type>& data );
      ///
      uMatrix<Type>& operator+=( const uMatrix<Type>& data );
      
      ///
      uMatrix<Type>& operator-= ( const Type );
      ///
      uMatrix<Type>& operator+= ( const Type );
      ///
      uMatrix<Type>& operator*= ( const Type );
      ///
      uMatrix<Type>& operator/= ( const Type );

   // ** Member functions
      /// Return number of rows
      int rows() const { return M; }
      /// Return number of columns
      int columns() const { return N; }
      /// Return true if marked as shallow copyable
      int shallow() const { return shallowCopyFlag; }
      /// Return true if marked as a temporary
      int temporary() const { return temporaryFlag; }

      /// Transpose method
      void transpose(); // You figure it out.
      /// Return the address of the allocated storage
      inline Type* address() const { return VectStorage.address(); }; 
      /// Return the allocated storage
      inline uVector<Type> & storage() { return VectStorage ; }

      /// Change number of rows and columns, preserve data
      /** Change the number of rows and columns but preserve the data.
      It is absolutely necessary that the new size have the same number
      of elements as the old size. */
      inline void reshape( int, int ); // Change dimensions but not total size
      /// Resize. If original size != m*n  destroy existing data 
      void resize( const int m, const int n = 1 );  // resize and trash data
      /// resize a matrix to the size of the data pointed to
      /** This resizes a matrix and sets its storage to the pointer.
      {\em Be warned that absolutely no checks can be performed on the 
      pointer.} The existing storage is dropped. */
      void resize( const int m, const int n, Type data[] ); 
      /// Element by element multiplication (Hadamard product)
      /** This multiplies two matrices elemet by element. It is used
      as follows:  
      \begin{verbatim}
   uMatrix<int> A(10,10);
   uMatrix<int> B(10,10);
    
   // Assign stuff to A and B

   uMatrix<int> C = A.pointMul(B);
      \end{verbatim}
      Note that A and B are {\em unchanged} and that the result is
      returned and assigned to C (in this example).
      */
      const uMatrix<Type> pointMul( const uMatrix<Type>& data ) const;
      /// Element by element division
      /** This is used the same way as \Ref{pointMul} */
      const uMatrix<Type> pointDiv( const uMatrix<Type>& data ) const;
      /// Read a binary file given a file name
      /** Read data from disk to a \Ref{uMatrix} given a file
      name. The data format must be raw binary data (no size
      information is read). The method reads in as many elements of
      <Type> as there are in the matrix. If the file has an extension
      of .Z or .gz, it is assumed to be compressed using uncompress or
      gunzip respectively and uncompressed on the fly. If the file does
      not exist or if there is insufficient data then the method
      aborts.  */
      void read( string filename ); // Open, read/write and then close file
      /// Write a \Ref{uMatrix} to disk
      /** Write a file to disk. The binary data representing the data
      in the \Ref{uMatrix} is simply dumped as raw binary data with no
      size information etc. If the file name ends in .Z or .gz, it is
      compressed on the fly using compress or gzip respectively. */
      void write( String filename ) const;
      /// Read a file to a \Ref{uMatrix} from a *FILE
      /** Read data from a *FILE opened for binary reading. */
      void read( FILE *Infile ); // Read and write previously opened files
      /// Write a file from a \Ref{uMatrix} to a *FILE
      /** Read data from a *FILE opened for binary writing. */
      void write( FILE *Outfile ) const;
      /// Read a flat ascii file. 
      /** The number of rows and columns is automatically determined
      but the data is not decompressed (yet) */
      void readAscii( const String filename ); 
      /// Mark the \Ref{uMatrix} as a temporary object
      /** Flag a matrix as both eligible for a shallow construction
      and shallow assignement a temporary. You should {\em ONLY} use
      this method immediately preceeding a return \Ref{uMatrix}
      statement. This method allows shallow copy construction and
      assignment from this \Ref{uMatrix}. It is used just before
      returning a \Ref{uMatrix} from a function. This considerably
      improves performance if the function is an argument for another
      function. In this case, the compiler constructs a temporary from
      the returned \Ref{uMatrix} which is then destroyed. The
      temporary is shallow copied from the returned value, saving the
      overhead of a full copy. The \ref{uMatrix} memory management
      ensures that the memory used is deleted when the last reference
      to the storage goes out of scope. Avoid using these unless you
      REALLY know what you are doing or if you REALLY like big memory
      leaks. The following code fragment illustrates how to use this:
      \begin{verbatim}
  uMatrix<Complex> doSomething( const uMatrix<Complex>& data )
  {
     uMatrix<Complex> A(data.rows(),data.columns());

     // Do something with the data
     A = Complex(4.0)*data;

     // Allow A to shallow construct and assign because it goes 
     // out of scope immediately
     A.markAsTemporary();
     return A;
  }
      \end{verbatim} */
      void markAsTemporary()
      { 
#ifdef uMatrix_enable_shallow_copy
	 shallowCopyFlag = 1; temporaryFlag = 1; 
#else  // Disable any hit of a shallow copy.
	 shallowCopyFlag = 0; temporaryFlag = 0; 
#endif
      }

      /// Opposite of \Ref{markAsTemporary}. 
      /** This method will explicitly disable any shallow copies 
      that may occur as a result of using the markAsTemporary method */
      void markAsPermanent(){ shallowCopyFlag = 0; temporaryFlag = 0; }
      
      /// Method to insert a \Ref{uMatrix} into *this as a submatrix
      /** This method allows insertion of a uMatrix as a subMatrix
          into this object. The following code fragment illustrates this:
\begin{verbatim}
   uMatrix<int> G(2,2);
   uMatrix<int> F(5,5);

   // This cast is needed to resolve assigment ambiguity with a null pointer
   F = (int) 0;  
   G(0,0) = 1; G(0,1) = 2;
   G(1,0) = 3; G(1,1) = 4;

   I = uIndex(1,2,3);
   J = uIndex(2,2,4);
   F.insert(I,J,G);
   // F now contains G, starting at (1,2) at every second index
   cout << F << endl;
\end{verbatim} 
*/
      void insert( const uIndex & I, const uIndex & J, const uMatrix<Type>& data );
      void info() const // Just print some (possibly) useful info.
      { 
	 cout << "uMatrix<Type> (" << M << "," << N 
	      << "), address = " << (int)VectStorage.address()
	      << " temporaryFlag = " << temporaryFlag
	      << " shallowCopyFlag = " << shallowCopyFlag << endl; 
      };

#ifdef stupid_f2c_MAIN
      void stupid_f2c(void)
      {
	 MAIN__();
      }
#endif

   // g++ 2.6.3 seems to think that this needs to be a friend. I think its bull
   // but I cannot argue here. Maybe uMatrix is a little lonely?
   // This does not seem to be a problem with gcc 2.6.3 on Linux

   friend ostream& operator<<   _EGCSCOMP_  (ostream& outStream, const uMatrix<Type>& data) ;
   friend istream& operator>> _EGCSCOMP_  (istream& inStream, uMatrix<Type>& data);

};

template<class Type>
inline void uMatrix<Type>::operator=( Type data[] )
{
   // The uVector class should take care of all this. We assume that data
   // points to a chunk of memory as big as this matrix is. Use with care.
   VectStorage = data;
   shallowCopyFlag = 0;
   temporaryFlag = 0;
}

template<class Type>
inline void uMatrix<Type>::operator=( const uMatrix<Type>& data )
{
   // If our source was a temporary or was flagged for a shallow copy then 
   // do a shallow copy again. Reset the flags of the new matrix
   // We assume that "data" is a temporary returned from an expression
   // containing a matrix. We also check if this was constructed from
   // a pointer. If so, we do not own the memory so it is necessary to
   // do a deep copy to preserve the intent of the matrix to access this
   // preallocated memory.
   if ( ( data.temporary() || data.shallow() ) && 
	VectStorage.ownsAllocatedMemory() )
   {
      VectStorage.shallow( data.VectStorage );
   }
   else  // Go nuts and do a deep copy
   {
      VectStorage = data.VectStorage;
   }

   // The vector shallow method or assignment operator resizes the storage
   // so just update LHS rows and columns to the RHS.
   M = data.rows();
   N = data.columns();

   // After an assign, this-> is no longer shallow flagged nor a temporary
   shallowCopyFlag = 0;
   temporaryFlag = 0;
}

template<class Type>  // Assign to scalar value
inline void uMatrix<Type>::operator=( Type data )
{
   VectStorage = data; // The uVector class does all the work
   shallowCopyFlag = 0;
   temporaryFlag = 0;
}

template<class Type>
uMatrix<Type>::uMatrix( int m, int n ) 
      : VectStorage( m*n )
{
   DBG(cout << "uMatrix(int,int) constructor called: this="
       << this << " size=(" << m << "," << n << ")\n";);
   M = m;
   N = n;
   shallowCopyFlag = 0;
   temporaryFlag = 0;
}

template<class Type>
void uMatrix<Type>::inizializza()
   
{
   DBG(cout << "uMatrix(int,int) constructor called: this="
       << this << " size=(" << m << "," << n << ")\n";);
   M = 0;
   N = 0;
   VectStorage.inizializza();
   shallowCopyFlag = 0;
   temporaryFlag = 0;
}

/*
   Construct from a pointer to some memory. We assume that you know what
   you were doing when you allocated *data!  */
template<class Type> 
inline uMatrix<Type>::uMatrix( const int m, const int n, Type data[] )
      : VectStorage( m*n, data )
{
   DBG(cout << "uMatrix(const int, const int, Type*) constructor called: this="
       << this << " param=" << &data << "\n";);
   M = m;
   N = n;
   // No shallow copy complications here.
   shallowCopyFlag = 0;
   temporaryFlag = 0;
}


template<class Type> 
inline uMatrix<Type>::uMatrix( const uMatrix<Type>& data ) 
      : VectStorage()
{
   DBG(cout << "uMatrix(uMatrix&) constructor called: this="
       << this << " param=" << &data << "\n";);
   M = data.rows();
   N = data.columns();

   
   // If our source was a shallow copy, then do a shallow copy again.
   // We assume that "data" is a temporary or reference returned from
   // a function.
   // We only reset the shallow flag as this-> is still a temporary. If we
   // call a constructor with this-> as an argument later then we will get
   // a deep copy on construction. We may only have a shallow copy if this
   // constructor is called as the result of a return statement.

   // I have used a compiler directive to disabled/enable  the propagation 
   // of the temporary flag as we cannot assign function local matrices to 
   // the functions parameter in  nested function calls without making a 
   // bad shallow copy or by calling the markAsPermanent() method.

   if ( data.shallow() && data.temporary() )
   {
      VectStorage.shallow( data.VectStorage );
      // this-> is still a temporary!
      shallowCopyFlag = 0;
      // Propagate temporary flag if we enable assigment shallow copies
#ifdef uMatrix_constructor_propagate_temporary_enable
      temporaryFlag = 1; // propagate flag and allow shallow copy on assign
#else
      temporaryFlag = 0; // force a deep copy on assign (SAFER OPTION)
#endif
   }
   else  // Go nuts and do a deep copy
   {
      VectStorage = data.VectStorage;
      // this-> is no longer a temporary so flag it as such.
      temporaryFlag = 0;
      shallowCopyFlag = 0;
   }
}

template<class Type>
inline Type& uMatrix<Type>::operator()(const int m, const int n) const 
{
#ifndef No_uIndex_Check
   if ( ( ( m < 0 ) || ( m >= M ) ) || ( ( n < 0 ) || ( n >= N ) ) )
   {
      cerr << "Bounds error in uMatrix::operator(): element (" 
           << m << "," << n << ") in a " << M << "x" << N << " matrix\n" ;
      abort();
   } 
#endif
   return VectStorage[M*n+m];
}
template<class Type>
void  uMatrix<Type>::extract (uIndex I, uIndex J, uMatrix<Type> &A ) const
{
  if(I.end() >= rows() || J.end() >= columns())
    {
      cerr << "Bounds error in uMatrix::extract(uIndex,uIndex,uMatrix&)\n"; 
      abort();
    }  
  A.resize(I.dim(),J.dim());
  int m = 0;
  for ( int r = I.start(); r <= I.end(); r += I.stride() )
    {
      int n = 0;
      for ( int c = J.start(); c <= J.end(); c += J.stride() )
	{
	  A(m,n) = VectStorage[M*c + r];
	  n++;
	}
      m++;
    }
};

// Warning: this function does not return a modifyable lvalue
// so A(I,J)=B compiles but won't work. I'll buy you a beer if you
// can figure an neat way to do this

template<class Type>
uMatrix<Type> uMatrix<Type>::operator()(uIndex I, uIndex J) const
{
  if(I.end() >= rows() || J.end() >= columns())
    {
      cerr << "Bounds error in uMatrix::extract(uIndex,uIndex,uMatrix&)\n"; 
      abort();
    } 
  uMatrix<Type> A(I.dim(),J.dim());
  int m = 0;
  for ( int r = I.start(); r <= I.end(); r += I.stride() )
    {
      int n = 0;
      for ( int c = J.start(); c <= J.end(); c += J.stride() )
	{
	  A(m,n) = VectStorage[r + c*M];
	  n++;
	}
      m++;
    }
  A.markAsTemporary();
  return A;
}


template<class Type>  
void  uMatrix<Type>::insert(const  uIndex &I, const uIndex &J, 
                             const uMatrix<Type>& data )
{
   int r,c;

   if ( ( I.dim() != data.rows() ) || ( J.dim() != data.columns() ) )
   {
      cerr << "Index dimensions do not match source data dimensions "
           << " in uMatrix::insert(uIndex,uIndex,uMatrix)" << endl;
      abort();
   }
   // Check that data and *this are not the same matrix so that
   // we do not tread on our own toes
   if ( this == &data )
   {
      cerr << "You cannot insert a matrix into itself in "
           << "uMatrix::insert(uIndex,uIndex,uMatrix)" << endl;
      abort();
   }


   for ( int m = 0; m < data.rows(); m++ )
   {
      for ( int n = 0; n < data.columns(); n++ )
      {
         r = m * I.stride() + I.start();
         c = n * J.stride() + J.start();

#ifndef No_uIndex_Check
	 if ( ( ( r < 0 ) || ( r >= M ) ) || ( ( c < 0 ) || ( c >= N ) ) )
	 {
	    cerr << "Bounds error in uMatrix::insert(uIndex,uIndex,uMatrix)" 
                 << endl; 
	    abort();
	 } 
#endif
	 // Actually copy the data over.
	 (*this)(r,c) = data(m,n);
      }
   }
}

template<class Type>
inline void uMatrix<Type>::reshape( int m, int n )
{
   if ( M*N != m*n )
   {
      cerr << "New allocated size is not equal to old size in uMatrix<Type>:reshape()\n";
      abort();
   }
   M = m;
   N = n;
}

template<class Type>
void uMatrix<Type>::resize( const int m, const int n ) // resize and trash data
{
   DBG( cout << "uMatrix<Type>::resize(const int m, const int n): New size = " 
	<< m << "x" << n << endl; );

   // Only resize if the allocated sizes are different. This speeds up cases
   // where the resize is not really required but called anyway for some reason
   if ( ( M * N ) != ( m * n ) )
   {
      VectStorage.resize(m*n);
   }
   // Just reshape anyway as the overhead is negligible
   M = m;
   N = n;
}

// Resize a matrix but use the supplied pointer as storage instead of
// allocating our own storage. Useful to assing and resize an existing
// uMatrix in one step

template<class Type>
void uMatrix<Type>::resize( const int m, const int n, Type data[] )
{
    DBG( cout << "uMatrix<Type>::resize(const int m,"
	      << "const int n, Type data[]): New size = " 
	      << m << "x" << n << endl; );

    // This will clobber existing storage  and setup new pointer
    VectStorage.resize(m*n, data);
    M = m;
    N = n;
}

template<class Type>
void uMatrix<Type>::transpose()
{
   int i,j; 
   Type temp;
   
   // If this is just a row or a column, just swap indecies.
   if ( ( M == 1 ) || ( N == 1 ) )
   { 
      int swapTemp = M;
      M = N;
      N = swapTemp;
      return;
   }
   if ( ( M == N ) )  // Square marices are easy
   {
      for ( i = 0; i < M; i++ )
      {
	 for ( j = i+1; j < N; j++ )
	 {
	    temp = VectStorage[i+j*M];
	    VectStorage[i+j*M] = VectStorage[j+i*M];
	    VectStorage[j+i*M] = temp;
	 }
      }
   }
   else  // This could probably be done quicker
   {
      uVector<Type> VectTemp(N*M);
      for ( i = 0; i < M*N; i++ )
      {
	 VectTemp[i] = VectStorage[(i+(i*(M-1))) % (M*N-1)];
      }
      VectTemp[M*N-1] = VectStorage[M*N-1];
      VectStorage = VectTemp; //Change this from a copy to a pointer fiddle
      j = M;
      M = N;
      N = j;
   }
}

template<class Type>
const uMatrix<Type> uMatrix<Type>::pointMul( const uMatrix<Type>& data ) const
{
   if ( ( M != data.rows() ) || ( N != data.columns() ) )
   {
      cerr << "Matrix size mismatch in uMatrix::pointMul\n";
      abort();
   }
   else
   {
      uMatrix<Type> A(data.rows(),data.columns());
      Type *d = A.address();
      Type *s1 = VectStorage.address();
      Type *s2 = data.address();
      for ( int i = 0; i < M*N; i++ )
      {
	 *d++ = *s1++ * *s2++;
      }
      A.markAsTemporary();
      return A;
   }
}

template<class Type>
const uMatrix<Type> uMatrix<Type>::pointDiv( const uMatrix<Type>& data ) const
{
   if ( ( M != data.rows() ) || ( N != data.columns() ) )
   {
      cerr << "Matrix size mismatch in uMatrix::pointDiv\n";
      abort();
   }
   else
   {
      uMatrix<Type> A(data.rows(),data.columns());
      Type *d = A.address();
      Type *s1 = VectStorage.address();
      Type *s2 = data.address();
      for ( int i = 0; i < M*N; i++ )
      {
	 *d++ = *s1++ / *s2++;
      }
      A.markAsTemporary();
      return A;
   }
}


/*
   Read a file pointed to by the filename. If the file extension contains
   a .gz or .Z, it is decompressed automatically.
*/
/*
template<class Type>
void  uMatrix<Type>::read( String filename )
{
   FILE *Infile;
   int Total, nRead;
   int pipe = 0;
   Regex gzip( ".gz$");
   Regex compress( ".Z$");


   // Test to see if we must decompress this on the fly
   if ( filename.contains(gzip) ||  filename.contains(compress) )
   {
      pipe=1; // Yes, we are reading from a pipe
      Infile = popen( "gzip -dc " + filename, "r" ); 
      if ( Infile == NULL ) // Check for a screwup
      {
	 cerr << "Error opening" << filename << " via decompress pipe\n";
	 abort();
      }
   }
   else
   {
      pipe = 0;
      Infile = fopen( filename, "rb" );
      if ( Infile == NULL )
      {
	 cerr << "File " << filename << " not found\n";
	 abort();
      }
   }
   
   // Read in data
   Total = M*N;  // total size to be read in
   nRead = fread( VectStorage.address(), sizeof( Type ), Total, Infile );
   if ( nRead != Total )
   {
      cerr << "Unable to read all required data in uMatrix<type>::read(String)"
	   << endl;
      abort();
   }

   // Use the right close statement.
   if ( pipe )
   {
      pclose( Infile );
   }
   else
   {
      fclose( Infile );
   }
}

template<class Type>
void  uMatrix<Type>::write( String filename ) const
{
   FILE *Outfile;
   int Total, nWrite;
   int pipe = 0;
   Regex gzip( ".gz$");
   Regex compress( ".Z$");

   if ( filename.contains(gzip) ||  filename.contains(compress) )
   {
      pipe=1; // Yes, we are reading from a pipe

      // We need to see if we must use gzip (.gz) or compress (.Z)
      if ( filename.contains(gzip) )
      {
	 Outfile = popen( "gzip -f > " + filename, "w" ); 
      }
      else
      {
	 Outfile = popen( "compress > " + filename, "w" ); 
      }
      if ( Outfile == NULL ) // Check for a screwup
      {
	 cerr << "Error opening" << filename << " via compress pipe\n";
	 abort();
      }
   }
   else
   {
      pipe = 0;
      Outfile = fopen( filename, "wb" );
      if ( Outfile == NULL )
      {
	 cerr << "Error in file " << filename << " during write\n";
	 abort();
      }
   }
   Total = M*N;  // total size to be written
   nWrite = fwrite( VectStorage.address(), sizeof( Type ), Total, Outfile );
   if ( nWrite != Total )
   {
      cerr << "Disk full or other failure during write to " << filename 
           << endl;
      abort();
   }

   // Use the right close statement.
   if ( pipe )
   {
      pclose( Outfile );
   }
   else
   {
      fclose( Outfile );
   }
}

*/

/*
   Read in matrix from an already opened file. Leave file open on return
*/


template<class Type>
void  uMatrix<Type>::read( FILE *Infile )
{
   int Total, nRead;
   Total = M*N;  // total size to be read in
   nRead = fread( VectStorage.address(), sizeof( Type ), Total, Infile );
   if ( nRead != Total )
   {
      cerr << "Unable to read all required data in uMatrix<type>::read(FILE *)"
           << endl;
      abort();
   }
}

/*
   Write a matrix to an existing file
*/
template<class Type>
void  uMatrix<Type>::write( FILE *Outfile ) const
{
   int Total, nWrite;

   Total = M*N;  // total size to be written
   nWrite = fwrite( VectStorage.address(), sizeof( Type ), Total, Outfile );
   if ( nWrite != Total )
   {
      cerr << "Disk full or other failure during uMatrix<type>::write( FILE *)"
           << endl;
      abort();
   }
}
/// Stream output operator for the \Ref{uMatrix} class
/** This operator allows the use of the "cout" C++ stream class as
well as file IO and any other form of IO supported by the iostreams
class. It simply dumps the \Ref{uMatrix} data row by row with a
newline at the end of each row to the output stream. The last row is
terminated with a newline.  */
template <class Type>
ostream& operator<<(ostream& outStream, const uMatrix<Type>& data)
{

   for ( int i = 0; i < data.rows(); i++ )
   {
      for ( int j = 0; j < data.columns(); j++ )
      {
 	 outStream << data(i,j) << " ";
      }
      outStream << "\n";
   }
   return outStream;
}


/// Stream input operator for the \Ref{uMatrix} class
/** This operator allows the use of the "cin" C++ stream class as
well as file IO and any other form of IO supported by the iostreams
class. It simply reads the number of <Type> elements needed to fill
the argument \Ref{uMatrix}. The argument is not resized. This will
fail if insufficient data is available or if there is invalid text for
<Type> type data. See the iostreams manual for more information on
iostreams. 
*/
template <class Type>
istream& operator>>(istream& inStream, uMatrix<Type>& data)
{
   for ( int i = 0; i < data.rows(); i++ )
   {
      for ( int j = 0; j < data.columns(); j++ )
      {
	 inStream >> data(i,j);
      }
   }
   return inStream;
}

template <class Type>
uMatrix<Type>& uMatrix<Type>::operator-=( const uMatrix<Type>& data )
{
   Type *s = data.address();
   Type *d = VectStorage.address();
   if ( ( M != data.rows() ) || ( N != data.columns() ) )
   {
      cerr << "Matrix size mismatch in operator-=(uMatrix&)\n";
      abort();
   }
   for ( int i = 0; i < M*N; i++ )
   {
      *d++ -= *s++;
   }
   return *this;
}

template <class Type>
uMatrix<Type>& uMatrix<Type>::operator+=( const uMatrix<Type>& data )
{
   Type *s = data.address();
   Type *d = VectStorage.address();
   if ( ( M != data.rows() ) || ( N != data.columns() ) )
   {
      cerr << "Matrix size mismatch in operator+=(uMatrix&)\n";
      abort();
   }
   for ( int i = 0; i < M*N; i++ )
   {
      *d++ += *s++;
   }
   return *this;
}


template <class Type>
uMatrix<Type>& uMatrix<Type>::operator+=( const Type data )
{
   Type *d = VectStorage.address();
   for ( int i = 0; i < M*N; i++ )
   {
      *d++ += data;
   }
   return *this;
}

template <class Type>

uMatrix<Type>& uMatrix<Type>::operator-=( const Type data )
{
   Type *d = VectStorage.address();
   for ( int i = 0; i < M*N; i++ )
   {
      *d++ -= data;
   }
   return *this;
}

template <class Type>
uMatrix<Type>& uMatrix<Type>::operator*=( const Type data )
{
   Type *d = VectStorage.address();
   for ( int i = 0; i < M*N; i++ )
   {
      *d++ *= data;
   }
   return *this;
}

template <class Type>
uMatrix<Type>& uMatrix<Type>::operator/=( const Type data )
{
   Type *d = VectStorage.address();
   for ( int i = 0; i < M*N; i++ )
   {
      *d++ /= data;
   }
   return *this;
}

/// Addition operator.
/** Allows the addition of 2 uMatrices. Overloaded versions exist
for uMatrix scalar operations. */
template <class Type>
uMatrix<Type> operator+( const uMatrix<Type>& data1,const uMatrix<Type>& data2 )
{
   if ( ( data1.rows() != data2.rows() ) || 
        ( data1.columns() != data2.columns() ) )
   {
      cerr << "uMatrix<Type> operator+(): Arguments differ in size\n";
      abort();
   }
   uMatrix<Type> temp(data1.rows(),data1.columns());
   Type *s1 = data1.address();
   Type *s2 = data2.address();
   Type *d = temp.address();
   for ( int i = 0; i < data1.rows()*data1.columns(); i++ )
   {
      *d++ = *s1++ + *s2++;
   }
   temp.markAsTemporary();
   return temp;
}
/// Subtraction operator.
/** Allows the subtraction of 2 uMatrices. Overloaded versions exist
for uMatrix scalar operations. */
template <class Type>
uMatrix<Type> operator-( const uMatrix<Type>& data1,const uMatrix<Type>& data2 )
{
   if ( ( data1.rows() != data2.rows() ) || 
        ( data1.columns() != data2.columns() ) )
   {
      cerr << "uMatrix<Type> operator+(): Arguments differ in size\n";
      abort();
   }
   uMatrix<Type> temp(data1.rows(),data1.columns());
   Type *s1 = data1.address();
   Type *s2 = data2.address();
   Type *d = temp.address();
   for ( int i = 0; i < data1.rows()*data1.columns(); i++ )
   {
      *d++ = *s1++ - *s2++;
   }
   temp.markAsTemporary();
   return temp;
}


// No frills matrix multiply. Some pointer fiddles may have helped...
/*
template <class Type>
uMatrix<Type> operator*( const uMatrix<Type>& data1, const uMatrix<Type>& data2 )
{
   if ( data1.columns() != data2.rows() )
   {
      cerr << "uMatrix<Type> operator*(): Arguments have incompatible sizes\n";
      abort();
   }
   uMatrix<Type> temp(data1.rows(),data2.columns());
   for ( int i = 0; i < temp.rows(); i++ )
   {
      for ( int j = 0; j < temp.columns(); j++ )
      {
	 temp(i,j) = 0;
	 for ( int k = 0; k < data1.columns(); k++ )
	 {
	    temp(i,j) += data1(i,k) * data2(k,j);
	 }
      }
   }
   temp.markAsTemporary();
   return temp;
}
*/

// Overloaded matrix multiplies that use the BLAS routines. This is
// screamingly faster on the ALPHA with the DXML

// These are not needed if we use the -fno-implicit-templates switch 
// I don't know what will happen if we don't use this switch.

/*
#ifdef uMatrix_use_BLAS_matrix_multiply
*/
uMatrix<float> operator*( const uMatrix<float>& data1, 
			  const uMatrix<float>& data2 );
uMatrix<double> operator*( const uMatrix<double>& data1, 
			   const uMatrix<double>& data2 );
uMatrix<fComplex> operator*( const uMatrix<fComplex>& data1, 
			     const uMatrix<fComplex>& data2 );
uMatrix<Complex> operator*( const uMatrix<Complex>& data1, 
			    const uMatrix<Complex>& data2 );
/*
#endif
*/

/// Multiplication operator.
/** Allows the multiplication of 2 uMatrices. Overloaded versions exist
for uMatrix scalar operations. If the
uMatrix\_use\_BLAS\_matrix\_multiply is defined then the BLAS library
will be used for the matrix-matrix multiplies. If not, a multiply
using pointer arithmetic is employed.
 */
// Matrix multiply based on pointer arithmetic for all other data types
template <class Type>
uMatrix<Type> operator*( const uMatrix<Type>& data1, const uMatrix<Type>& data2 )
{
  cout << " meglio usare le blas\n";
  exit(0);
   if ( data1.columns() != data2.rows() )
   {
      cerr << "uMatrix<Type> operator*(): Arguments have incompatible sizes\n";
      abort();
   }
   uMatrix<Type> temp(data1.rows(),data2.columns());
   Type *d1,*d2,*t;
   for ( int i = 0; i < temp.rows(); i++ )
   {
      for ( int j = 0; j < temp.columns(); j++ )
      {
	 t = &temp(i,j);
	 *t = (Type) 0;
	 d1 = &data1(i,0);
	 d2 = &data2(0,j);
	 for ( int k = 0; k < data1.columns(); k++ )
	 {
	    *t += *d1 * *d2++;
	    d1 += data1.rows();
	 }
      }
   }
   temp.markAsTemporary();
   return temp;
}

// Unary - operator
template <class Type>
uMatrix<Type> operator-( const uMatrix<Type>& data1)
{
   uMatrix<Type> temp(data1.rows(),data1.columns());
   Type *d = temp.address();
   Type *s = data1.address();
   for ( int i = 0; i < data1.rows()*data1.columns(); i++ )
   {
      *d++ = - *s++;
   }
   temp.markAsTemporary();
   return temp;
}

// Operators allowing scalar-Matrix operations

#define declareScalar_X_Matrix(Ts,Tm,Tr)\
  uMatrix<Tr>  operator*( const uMatrix<Tm>& data2, const Ts data1 );\
  uMatrix<Tr>  operator*( const Ts data1, const uMatrix<Tm>& data2 );

declareScalar_X_Matrix(double , Complex, Complex); 
declareScalar_X_Matrix(float  , Complex, Complex);
declareScalar_X_Matrix(float  , fComplex, fComplex);
declareScalar_X_Matrix(double  , fComplex, fComplex);
declareScalar_X_Matrix(Complex  , double , Complex);
declareScalar_X_Matrix(Complex  , float , Complex);
declareScalar_X_Matrix(fComplex  , double , Complex);
declareScalar_X_Matrix(fComplex  , float , fComplex);
#undef  declareScalar_X_Matrix



template <class Type>
uMatrix<Type> operator*( const uMatrix<Type>& data2, const Type data1 )
{
   uMatrix<Type> temp = data2;  // will shallow copy if required
   Type *d = temp.address();
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )
   {
      *d++ *= data1;
   }
   temp.markAsTemporary();
   return temp;
}

template <class Type>
uMatrix<Type> operator*( const Type data1, const uMatrix<Type>& data2 )
{
   uMatrix<Type> temp = data2;
   Type *d = temp.address();
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )
   {
      *d++ *= data1;
   }
   temp.markAsTemporary();
   return temp;
}

template <class Type>
uMatrix<Type> operator+( const uMatrix<Type>& data2, const Type data1 )
{
   uMatrix<Type> temp = data2; // will shallow copy if possible
   Type *d = temp.address();
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )
   {
      *d++ += data1;
   }
   temp.markAsTemporary();
   return temp;
}

template <class Type>
uMatrix<Type> operator+( const Type data1, const uMatrix<Type>& data2 )
{
   uMatrix<Type> temp = data2; // will shallow copy if possible
   Type *d = temp.address();
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )
   {
      *d++ += data1;
   }
   temp.markAsTemporary();
   return temp;
}

template <class Type>
uMatrix<Type> operator-( const uMatrix<Type>& data2, const Type data1 )
{
   uMatrix<Type> temp = data2;
   Type *d = temp.address();
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )
   {
      *d++ -= data1;
   }
   temp.markAsTemporary();
   return temp;
}

template <class Type>
uMatrix<Type> operator-( const Type data1, const uMatrix<Type>& data2 )
{
   uMatrix<Type> temp = data2;
   Type *d = temp.address();
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )
   {
      *d++ -= data1;
   }
   temp.markAsTemporary();
   return temp;
}

/// Matrix-scalar division
/** This operator allows a \Ref{uMatrix} to be divided by a
    scalar. Each element in the matrix is divided by the scalar argument.*/
template <class Type>
uMatrix<Type> operator/( const uMatrix<Type>& data2, const Type data1 )
{
   uMatrix<Type> temp = data2;
   Type *d = temp.address();
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )
   {
      *d++ /= data1;
   }
   temp.markAsTemporary();
   return temp;
}

/// Scalar-Matrix division
/** This operator allows a scalar to be divided by a \Ref{uMatrix}.
 The scalar argument is divided by each element in the matrix.*/
template <class Type>
uMatrix<Type> operator/( const Type data1, const uMatrix<Type>& data2)
{
  uMatrix<Type> temp = data2;
  Type *d = temp.address();
  for ( int i = 0; i < data2.rows()*data2.columns(); i++ )
  {
    *d = data1/(*d++);
  }
  temp.markAsTemporary();
  return temp;
}

/// Return the transpose of the argument
/** Simple transposition (no conjugation) 
 */
template <class Type>
uMatrix<Type> transpose( const uMatrix<Type>& data )
{
   if ( ( data.rows() == 1 ) || ( data.columns() == 1 ) )
   {
      uMatrix<Type> A(data);
      A.reshape(data.columns(),data.rows());
      A.markAsTemporary();
      return A;
      
   }
   else
   {
      uMatrix <Type> A(data.columns(),data.rows());
      
      for ( int i = 0; i < A.rows(); i++ )
      {
	 for ( int j = 0; j < A.columns(); j++ )
	 {
	    A(i,j) = data(j,i); // simple but slow.
	 }
      }
      A.markAsTemporary();
      return A;
   }
}

/*
   Return a copy of the argument matrix as a column vector
*/
template <class Type>
uMatrix<Type> vect( const uMatrix<Type>& data )
{
   uMatrix<Type> A = data; // This will copy the data over if needed
   A.reshape( data.rows()*data.columns(), 1 );
   A.markAsTemporary();
   return A;
}


/*
   Method to read in flat ascii files. This works the same way as Matlab
   ascii files are read. The dimension of the matrix is determined from
   the ascii file itself so there is no need to have prior knowledge about
   its size. The matrix will be resized appropriately

   NOTE: I assume that there is no weird crap in the file other than some
         reasonable numbers for the particular instantiation of the template.
*/
/*
template <class Type>
void uMatrix<Type>::readAscii( const String fileName )
{
   ifstream inFile( fileName );

   if ( ! inFile )
   {
      cout << "Unable to open file " << fileName << " in uMatrix<type>::readAscii" << endl;
      abort();
   }

   // First count how many times we get to swallow digits before we hit "\n"
   // to figure out what the dimension of this vector is

   char c;
   int nCols = 0;
   char done = 0;

   do
   {
      
      // Gobble any white space leading up to first digit
      while ( inFile.get(c) )
      {
	 if ( isspace(c) == 0 )
	 {
	    inFile.putback(c); // Ok, its not white so put it back and split
	    break;
	 }
	 if ( c == '\n' )
	 {
	    done = 1;
	 }
      }
      if ( inFile.eof() && ( nCols == 0 ) )
      {
	 cout << "File " << fileName << " seems to contain nothing :-)\n";
	 abort();
      }
      int toggle=0;
      while ( ( inFile.get(c) ) &&  !done  ) // Gobble digit
      {
         if ( c=='(' )
          {  toggle=1;
          }
	 if ( (toggle==0 && !( ( isdigit(c) ) 
		            || ( c == '.' ) 
		            || ( c == '-' ) 
            		    || ( c == 'e' )
		            || ( c == 'E' ) 
	                      ) 
	        )
               || (toggle == 1 && c == ')'  )
             )
	 {
            toggle = 0;
	    inFile.putback(c);
	    nCols++; // Ok, we just swallowed a number so chalk it up
	    break;
	 }
      }
   }
   while ( !done );

   // OK, we have the columns count so now do the rows count

   int nRows = 0; // Remember that we just ate one row already
   char buf[nCols*50]; // Generous buffer proportional to nCols

   while ( !inFile.eof() )
   {
      inFile.get( buf, nCols*50, '\n' );
      inFile.get(c); // slurp up '\n' left behind by get()
      nRows++;
   }
   
   // Rewind the file to its start. I am sure there is a beter way...
   inFile.close();
   inFile.open( fileName );

   // Resize and update parameters for this instance of a Matrix
   M = nRows;
   N = nCols;
   VectStorage.resize(M*N);

   // Read in the data
   for ( int i = 0; i < M*N; i++ )
   {
      inFile >> VectStorage(i);
   }
}
*/
// General zero padding. Data is centered in Zp
//@ManMemo: Zero pad a \Ref{uMatrix}
/** This function centers the \Ref{uMatrix} parameter "data" in the
    \Ref{uMatrix} "Zp". The size of "Zp" is the final size of the zero
    padded data. The top left corner of the data is at
    (Zp.rows()-data.rows())/2,Zp.columns()-data.columns())/2).
    */

template <class Type>
void zeroPad(  
//@ManMemo:
uMatrix<Type>& Zp, 
/** Contains the zero padded data on return. This variable must be set
    to the desired final size of the zero padded data. */
//@ManMemo:
/** Data to be zero padded. It is unchanged on return. */
const uMatrix<Type>& data)
{
   Zp = Type(0);
   for (int i = 0; i < data.columns(); i++ )
   {
      for ( int j = 0; j < data.rows(); j++ )
      {
	 Zp(i+(Zp.columns()-data.columns())/2,j+(Zp.rows()-data.rows())/2 ) = data(
	    i,j);
      }
   }
}

//@ManMemo: Fill a \Ref{uMatrix} with random numbers
/** Generate a matrix of random numbers using a random number
   generator from the libg++ library. It is the programmers
   responsibility to ensure that the generator creates appropriate
   random numbers. See the libg++ documentation for further information
   about the random number generator. Here is an example of how to use this:
\begin{verbatim}

#include <ACG.h>
#include <Uniform.h>
#include <nMatrix.h>
#include <iostream.h>

int main ( int argc, char *argv[] )
{
   int seed = 3453663363;
   ACG acg(seed);
   // This could be one of many different generators offered by libg++
   Uniform rng(-maxRand, maxRand, &acg);
   
   uMatrix<Complex> A(10,10);
   uRandom(A,rng);  // Fills A 
   cout << A;
}

\end{verbatim}
   */
/*
template <class Type> void uRandom(


     Random & rng )
{
   for ( int i = 0; i < A.rows(); i++ )
   {
      for ( int j = 0; j < A.columns(); j++ )
      {
	 A(i,j) = (Type) rng();
      }
   }
}
		     

void uRandom(uMatrix<Complex>& A, Random & rng );
void uRandom(uMatrix<fComplex>& A, Random & rng );
   */


/*
   These are prototypes for stuff that cannot be templated. The 
   implimentations and comments etc. are in uMatrix.cc
*/

uMatrix<Complex> conj( const uMatrix<Complex>& data );
uMatrix<fComplex> conj( const uMatrix<fComplex>& data );
uMatrix<double> conj( const uMatrix<double>& data );
uMatrix<float> conj( const uMatrix<float>& data );

uMatrix<double> norm( const uMatrix<Complex>& data );
uMatrix<float> norm( const uMatrix<fComplex>& data );
uMatrix<double> norm( const uMatrix<double>& data );
uMatrix<float> norm( const uMatrix<float>& data );

uMatrix<double> abs( const uMatrix<Complex>& data );
uMatrix<float> abs( const uMatrix<fComplex>& data );
uMatrix<double> abs( const uMatrix<double>& data );
uMatrix<float> abs( const uMatrix<float>& data );


uMatrix<double> real( const uMatrix<Complex>& data );
uMatrix<double> imag( const uMatrix<Complex>& data );
uMatrix<float> real( const uMatrix<fComplex>& data );
uMatrix<float> imag( const uMatrix<fComplex>& data );
uMatrix<double> real( const uMatrix<double>& data );
uMatrix<float> real( const uMatrix<float>& data );
uMatrix<float> imag( const uMatrix<float>& data );

/// Return the Hermitian (conjugate) transpose of the argument
/** This function is overloaded for the float, double, fComplex and
    Complex. For real arguments, the procedure simply returns the
    transpose. */
uMatrix<Complex> hTranspose( const uMatrix<Complex>& data );

uMatrix<fComplex> hTranspose( const uMatrix<fComplex>& data );
uMatrix<double> hTranspose( const uMatrix<double>& data );
uMatrix<float> hTranspose( const uMatrix<float>& data );

//@ManMemo: Normalize the argument uMatrix
/** This function normalizes the argument \Ref{uMatrix} so that all
the value of all its elements lie in the interval
[max,min). {\em Overloaded versions exist for the float and double
types only}*/ 

uMatrix<float> normalize( 

//@ManMemo: The data to be normalized. This variable is unchanged by this operation
const uMatrix<float>& data, 
//@ManMemo: Minimum value. Defaults to 0.0.
float min=0.0,
//@ManMemo:  Maximum value. Defaults to 1.0. 
float max = 1.0 );


uMatrix<double> normalize( const uMatrix<double>& data, double min=0.0, 
			   double max = 1.0 );

/// Joins matrix A and B. A and B must have the same number of columns
/** This function is overloaded for the double and
 is put over B  */
template<class TYPE>
uMatrix<TYPE> JoinV( uMatrix<TYPE> const &A,  uMatrix<TYPE> const &B)
{
   int nc;
   nc = A.columns();
   if(nc != B.columns() )
    {
       printf(" matrices incompatibles dans JoinV \n");
       exit(0);
    }
   int nr1,nr2;
   nr1= A.rows() ;
   nr2= B.rows() ;

   uMatrix<TYPE> res(nr1+nr2, nc );

   res.insert(uIndex(0,1,nr1-1)      ,uIndex(0,1,nc-1)      ,  A  );
   res.insert(uIndex(nr1,1,nr1+nr2-1),uIndex(0,1,nc-1)      ,  B  );

   res.markAsTemporary();
   return res;
}
/// Joins matrix A and B. A and B must have the same number of rows
/** This function is overloaded for the double and
 is put after  B, horizontally.  */
template<class TYPE>
uMatrix<TYPE> JoinH( const  uMatrix<TYPE> &  A, const uMatrix<TYPE>  &  B)
{
   int nr;
   nr = A.rows();
   if(nr != B.rows() )
    {
       printf(" matrices incompatibles dans JoinH \n");
       exit(0);
    }
   int nc1,nc2;
   nc1= A.columns() ;
   nc2= B.columns() ;

   uMatrix<TYPE> res(nr,nc1+nc2);

   res.insert(uIndex(0,1,nr-1),uIndex(0,1,nc1-1),  A  );
   res.insert(uIndex(0,1,nr-1),uIndex(nc1,1,nc2+nc1-1),  B  );
   res.markAsTemporary();
   return res;
}



// max and min functions for matrices. These are macros because I don't
// want a max and min for stuff like complex numbers.

#define MaxMinHeaderMacro(Type1)                             \
Type1 max( const uMatrix<Type1>& data1);                     \
Type1 min( const uMatrix<Type1>& data1);                     \

MaxMinHeaderMacro(char);
MaxMinHeaderMacro(unsigned char);
MaxMinHeaderMacro(int);
MaxMinHeaderMacro(unsigned int);
MaxMinHeaderMacro(short int);
MaxMinHeaderMacro(unsigned short int);
MaxMinHeaderMacro(long int);
MaxMinHeaderMacro(unsigned long int);
MaxMinHeaderMacro(float);
MaxMinHeaderMacro(double);

// Really ugly macros for the cast operators. Templates have a few problems
// like excuding certain typed from being instantiated so we have to do this
// by macro

#define castHeaderMacro(fName,Type1,Type2) \
uMatrix<Type1> fName ( const uMatrix<Type2>& data1);

castHeaderMacro(floatCast,float,char)
castHeaderMacro(floatCast,float,unsigned char)
castHeaderMacro(floatCast,float,int)
castHeaderMacro(floatCast,float,unsigned int)
castHeaderMacro(floatCast,float,short int)
castHeaderMacro(floatCast,float,unsigned short int)
castHeaderMacro(floatCast,float,long int)
castHeaderMacro(floatCast,float,unsigned long int)
castHeaderMacro(floatCast,float,float) 
castHeaderMacro(floatCast,float,double)


castHeaderMacro(doubleCast,double,char)
castHeaderMacro(doubleCast,double,unsigned char)
castHeaderMacro(doubleCast,double,int)
castHeaderMacro(doubleCast,double,unsigned int)
castHeaderMacro(doubleCast,double,short int)
castHeaderMacro(doubleCast,double,unsigned short int)
castHeaderMacro(doubleCast,double,long int)
castHeaderMacro(doubleCast,double,unsigned long int)
castHeaderMacro(doubleCast,double,float) 
castHeaderMacro(doubleCast,double,double)

castHeaderMacro(charCast,char,char)
castHeaderMacro(charCast,char,unsigned char)
castHeaderMacro(charCast,char,int)
castHeaderMacro(charCast,char,unsigned int)
castHeaderMacro(charCast,char,short int)
castHeaderMacro(charCast,char,unsigned short int)
castHeaderMacro(charCast,char,long int)
castHeaderMacro(charCast,char,unsigned long int)
castHeaderMacro(charCast,char,float) 
castHeaderMacro(charCast,char,double)

castHeaderMacro(unsignedCharCast,unsigned char,char)
castHeaderMacro(unsignedCharCast,unsigned char,unsigned char)
castHeaderMacro(unsignedCharCast,unsigned char,int)
castHeaderMacro(unsignedCharCast,unsigned char,unsigned int)
castHeaderMacro(unsignedCharCast,unsigned char,short int)
castHeaderMacro(unsignedCharCast,unsigned char,unsigned short int)
castHeaderMacro(unsignedCharCast,unsigned char,long int)
castHeaderMacro(unsignedCharCast,unsigned char,unsigned long int)
castHeaderMacro(unsignedCharCast,unsigned char,float) 
castHeaderMacro(unsignedCharCast,unsigned char,double)

castHeaderMacro(intCast,int,char)
castHeaderMacro(intCast,int,unsigned char)
castHeaderMacro(intCast,int,int)
castHeaderMacro(intCast,int,unsigned int)
castHeaderMacro(intCast,int,short int)
castHeaderMacro(intCast,int,unsigned short int)
castHeaderMacro(intCast,int,long int)
castHeaderMacro(intCast,int,unsigned long int)
castHeaderMacro(intCast,int,float) 
castHeaderMacro(intCast,int,double)

castHeaderMacro(unsignedIntCast,unsigned int,char)
castHeaderMacro(unsignedIntCast,unsigned int,unsigned char)
castHeaderMacro(unsignedIntCast,unsigned int,int)
castHeaderMacro(unsignedIntCast,unsigned int,unsigned int)
castHeaderMacro(unsignedIntCast,unsigned int,short int)
castHeaderMacro(unsignedIntCast,unsigned int,unsigned short int)
castHeaderMacro(unsignedIntCast,unsigned int,long int)
castHeaderMacro(unsignedIntCast,unsigned int,unsigned long int)
castHeaderMacro(unsignedIntCast,unsigned int,float) 
castHeaderMacro(unsignedIntCast,unsigned int,double)

castHeaderMacro(fComplexCast,fComplex,char)
castHeaderMacro(fComplexCast,fComplex,unsigned char)
castHeaderMacro(fComplexCast,fComplex,int)
castHeaderMacro(fComplexCast,fComplex,unsigned int)
castHeaderMacro(fComplexCast,fComplex,short int)
castHeaderMacro(fComplexCast,fComplex,unsigned short int)
castHeaderMacro(fComplexCast,fComplex,long int)
castHeaderMacro(fComplexCast,fComplex,unsigned long int)
castHeaderMacro(fComplexCast,fComplex,float) 
castHeaderMacro(fComplexCast,fComplex,double)
castHeaderMacro(fComplexCast,fComplex,fComplex)

castHeaderMacro(ComplexCast,Complex,char)
castHeaderMacro(ComplexCast,Complex,unsigned char)
castHeaderMacro(ComplexCast,Complex,int)
castHeaderMacro(ComplexCast,Complex,unsigned int)
castHeaderMacro(ComplexCast,Complex,short int)
castHeaderMacro(ComplexCast,Complex,unsigned short int)
castHeaderMacro(ComplexCast,Complex,long int)
castHeaderMacro(ComplexCast,Complex,unsigned long int)
castHeaderMacro(ComplexCast,Complex,float) 
castHeaderMacro(ComplexCast,Complex,double)
castHeaderMacro(ComplexCast,Complex,Complex)

castHeaderMacro(fComplexCast,fComplex,Complex)
castHeaderMacro(ComplexCast,Complex,fComplex)

castHeaderMacro(shortIntCast,short int,char)
castHeaderMacro(shortIntCast,short int,unsigned char)
castHeaderMacro(shortIntCast,short int,int)
castHeaderMacro(shortIntCast,short int,unsigned int)
castHeaderMacro(shortIntCast,short int,short int)
castHeaderMacro(shortIntCast,short int,unsigned short int)
castHeaderMacro(shortIntCast,short int,long int)
castHeaderMacro(shortIntCast,short int,unsigned long int)
castHeaderMacro(shortIntCast,short int,float) 
castHeaderMacro(shortIntCast,short int,double)

castHeaderMacro(unsignedShortIntCast,unsigned short int,char)
castHeaderMacro(unsignedShortIntCast,unsigned short int,unsigned char)
castHeaderMacro(unsignedShortIntCast,unsigned short int,int)
castHeaderMacro(unsignedShortIntCast,unsigned short int,unsigned int)
castHeaderMacro(unsignedShortIntCast,unsigned short int,short int)
castHeaderMacro(unsignedShortIntCast,unsigned short int,unsigned short int)
castHeaderMacro(unsignedShortIntCast,unsigned short int,long int)
castHeaderMacro(unsignedShortIntCast,unsigned short int,unsigned long int)
castHeaderMacro(unsignedShortIntCast,unsigned short int,float) 
castHeaderMacro(unsignedShortIntCast,unsigned short int,double)

castHeaderMacro(longIntCast,long int,char)
castHeaderMacro(longIntCast,long int,unsigned char)
castHeaderMacro(longIntCast,long int,int)
castHeaderMacro(longIntCast,long int,unsigned int)
castHeaderMacro(longIntCast,long int,long int)
castHeaderMacro(longIntCast,long int,unsigned short int)
castHeaderMacro(longIntCast,long int,long int)
castHeaderMacro(longIntCast,long int,unsigned long int)
castHeaderMacro(longIntCast,long int,float) 
castHeaderMacro(longIntCast,long int,double)

castHeaderMacro(unsignedLongIntCast,unsigned long int,char)
castHeaderMacro(unsignedLongIntCast,unsigned long int,unsigned char)
castHeaderMacro(unsignedLongIntCast,unsigned long int,int)
castHeaderMacro(unsignedLongIntCast,unsigned long int,unsigned int)
castHeaderMacro(unsignedLongIntCast,unsigned long int,short int)
castHeaderMacro(unsignedLongIntCast,unsigned long int,unsigned short int)
castHeaderMacro(unsignedLongIntCast,unsigned long int,long int)
castHeaderMacro(unsignedLongIntCast,unsigned long int,unsigned long int)
castHeaderMacro(unsignedLongIntCast,unsigned long int,float) 
castHeaderMacro(unsignedLongIntCast,unsigned long int,double)

#define castToComplexHeader(fName,Type1,Type2)               \
uMatrix<Type1> fName( const uMatrix<Type2>& data1,           \
		      const uMatrix<Type2>& data2);          \

castToComplexHeader(fComplexCast,fComplex,float)
castToComplexHeader(fComplexCast,fComplex,double)

castToComplexHeader(ComplexCast,Complex,float)
castToComplexHeader(ComplexCast,Complex,double)

#endif
