"""
    FitFilter.py
    Filter derived class: 1d fitting
    Based on NewSpecfit
 
    Contact: Alexandre Gobbo (gobbo@esrf.fr)
    Version: 1.0
"""  


import Numeric
import string
from Filter import Filter
from NewSpecfit import Specfit
from SpecfitFunctions import SpecfitFunctions


class FitFilter(Filter):
    def __init__(self,source,synchronized=1):
        Filter.__init__(self,source,synchronized,buffer_input=1)
        self.FitFinishedEvent = self.eh.create("FitFinished")
        self.eh.register("ParametersChange", self.SetParameters)
        
        self.Editor=None
        self.fit=Specfit()
        self.fit.importfun("SpecfitFunctions.py")
        self.fit.settheory('Area Gaussians')
        self.Parameters={}
        
            
    def GetOutput(self):
        if "data" not in self.Input.keys(): return {}
        ydata=self.Input ["data"]
        if "xdata" in self.Input.keys(): xdata=self.Input["xdata"]
        else: xdata=Numeric.arrayrange(ydata.shape[0])
        if self.Parameters=={}: return {}
        self.fit.setdata(x=xdata,y=ydata)
        self.fit.settheory(self.Parameters["Theory"])
        self.fit.configure(Yscaling=1.,
                      WeightFlag=self.Parameters["Weight"],
                      PosFwhmFlag=1,
                      HeightAreaFlag=1,
                      FwhmPoints=50,
                      PositionFlag=1,
                      HypermetTails=5)            
        self.fit.setbackground(self.Parameters["Background"])

        if self.Parameters["Action"]=="ESTIMATE":
                self.fit.estimate()
                if self.eh is not None :                
                        self.eh.event(self.FitFinishedEvent, self.fit.paramlist)            
                return {}

        if self.Parameters["Mode"]=="MCA":        
            mcaresult=self.fit.mcafit(sensitivity=3.5)
            if mcaresult != []:
                finalx=Numeric.array([])
                finaly=Numeric.array([])
                delimeters=[]                
                finalp=[]
                for result in mcaresult:               
                  delimeters.append((result['xbegin'],int(result['xend'])))
                  x=self.fit.xdata0[int(result['xbegin']):int(result['xend'])]
                  y=self.fit.ydata0[int(result['xbegin']):int(result['xend'])]
                  yfit=self.fit.gendata(x=x,parameters=result['paramlist'])
                  finalx=Numeric.concatenate((finalx,x))
                  finaly=Numeric.concatenate((finaly,yfit))
            else:
                    return {}
            #event fit finished
            if self.eh is not None :                
                self.eh.event(self.FitFinishedEvent, mcaresult)
                
            return {"data": finaly, "xdata":finalx,"delimeters":delimeters}
        else:
            #self.fit.estimate()
            if "FitPars" in self.Parameters.keys():
                    self.fit.paramlist=self.Parameters["FitPars"]
            self.fit.startfit()
            yfit=self.fit.gendata(x=xdata,parameters=self.fit.paramlist)
            
            #event fit finished
            if self.eh is not None :                
                self.eh.event(self.FitFinishedEvent, self.fit.paramlist)            
        
            return {"data": yfit, "xdata":xdata}



    def ConnectParameterEditor(self,editor):
        """
        """
        self.DisconnectParameterEditor()
        self.Editor=editor
        self.Editor.onInitialize(self.eh,self.fit)
        

    def DisconnectParameterEditor(self):
        """
        """        
        if self.Editor is not None:
            self.Editor.onInitialize(None,None)
            self.Editor=None
        


    def GetParameters(self):
        """
        """
        return self.Parameters


    def SetParameters(self,parameters):
        """
        """
        #TODO: If parameter have changed, processing must be done here,
        #otherwise it is assumes is going to call GetOutput
        self.Parameters=parameters
        self.Refresh()