"""
    Filter.py
    Filter base class
"""

from PyDVT import __version__,__date__,__author__


import EventHandler


class Filter:
    """
    Filter objects are inserted between Data and View objects. They represent
    any kind of transformation of data, or the way it is represented. Filters
    can be cascaded.
    These objects have just one source (a Data or another Filter) and
    can produce output to one or more objects (either View or another Filter
    object).
    A particular case of Filter is DataSelection classes, which represent
    a slice of the Data objects (they have Data objects as sources).
    Other Filters can be cascaded to perform operations on the data, and to
    conform the data to the format required by View object.
    Filter base class implements an interface that allows derived classes to
    comunicate to source and output objects. The base class is transparent,
    which means that a base class Filter object cascaded to others will just
    give it's input as output. But it can be useful sometimes to create a
    "buffer" of the data, when multiple views access the same information,
    to avoid duplicity in processing. This is done by cascading a Filter base
    class object before the View objects, initialized with the option
    buffer_input set to 1.

    Interface:
    ===========================
        __init__
        SetSynchronized
        GetData
        GetSource
        GetDataSelection
        GetInput
        Refresh
        Destroy

    Overridables
    ===========================
        DataCoord2SelectionCoord
            (default implementation works if filter doesn't change coordinate system)
        SelectionCoord2DataCoord
            (default implementation works if filter doesn't change coordinate system)
        DataChanged

    Virtuals:
    ===========================
        GetOutput


    Internal Events:
    ===========================
        Emits:
            "DataChange"
            "DeleteEvent"
        Receives:
            "DataChange"
            "DeleteEvent"

    """
    def __init__(self,source,synchronized=1,buffer_input=0):
        """
        Constructor
        Parameters:
          source: the source object (a Filter or Data derived object)
          synchronized: if zero the filter won't answer to DataChange
                        events from the source (no dynamic updating)
          buffer_input: if non-zero source's output is stored in DataChanged
                        event. Otherwise it is read just in time.
                        Setting to 1 is useful 
                        - When the data is accessed in more than one place than
                          GetOutput, by multiple receivers,or if reading/
                          processing data is slow.
                        - To set transformation parameters and promote a
                          redrawing without having to read the source.
                        The default implementation  (buffer_input=0)
                        don't buffer input in order not to keep reference to
                        useless data.
                        See GetInput documentation                  
        """        
        self.eh=EventHandler.EventHandler()
        self.DataChange = self.eh.create("DataChange")
        self.DeleteEvent = self.eh.create("DeleteEvent")        
        self.Synchronized=synchronized
        self.BufferInput=buffer_input
        if self.BufferInput: self.Input=None
        if source!=None:
            self.Source=source

            self.ConfigureEvents()
            

            if hasattr (source,"DataRef"):
                self.DataRef=source.GetData()
            else:
                self.DataRef=source
        else:
            self.DataRef=None
            self.Source=None
        self.Alive=1


    def ConfigureEvents(self):
       source = self.Source
       if source!=None:
        source.eh.register("DeleteEvent",self.Destroy)
        if self.Synchronized:
          source.eh.register("DataChange",self.DataChanged)
          
    def __setstate__(self, dict):
        self.__dict__=dict
        self.ConfigureEvents()
        

    def SetSynchronized(self,synchronized):
        """
        Set/reset on-line mode
        Parameters:
          synchronized: if zero the filter won't answer to DataChange
                        events from the source (no dynamic updating)
        """        
        source=self.GetSource()
        if source==None: return
        if synchronized!=self.Synchronized:
            self.Synchronized=synchronized
            if self.Synchronized:
                source.eh.register("DataChange",self.DataChanged)
            else:
                source.eh.unregister("DataChange",self.DataChanged)


    def Refresh(self):
        """
        Generates DataChange event to inform linked objects to read data
        from the filter.
        """        
        if self.BufferInput: self.Input=None
        source=self.GetSource()
        if source==None: return
        self.eh.event(self.DataChange, self)

    def GetSource(self):
        """
        Returns source object.
        """        
        return self.Source
        
    
    def GetData(self):
        """
        Returns linked Data object (as a filter has just une source, it can only
        be linked to a single Data object).
        """        
        return self.DataRef


    def GetDataSelection(self):
        """
        Returns linked DataSelection
        """        
        obj=self
        while (1):
            source=obj.GetSource()
            if source==None:return None
            if source==self.DataRef: return obj
            obj=source
            
            
    def GetInput(self):
        """
        Returns source's output(does't work for DataSelection). 
        If filter was initialized with buffer_input, input is buffered.
        """
        source=self.GetSource()
        if source is None: return {}
        if self.BufferInput:
            if self.Input is None: self.Input=source.GetOutput()
            return self.Input                                    
        else:
            return source.GetOutput()


    def Destroy(self,source=None):
        """
        Cleanup.
        As event handlers and callbacks, by keeping a reference to other
        objects, avoid them to be destroyed (and freed from memory), the
        Destroy method has to be explicitly called for Filter objects
        before they get out of scope.
        I had tried to use weak references in order not to need this method,
        but this generated many other troubles (like impossibility to define
        selections locally, difficult control of the destruction sequence,
        not interceptable error mesassages...).
        """
        if self.Alive==0: return
        self.eh.event(self.DeleteEvent, self)     

        if self.Source is not None:
            if self.Source.eh is not None:
                self.Source.eh.unregister("DataChange",self.DataChanged)
                if self.Synchronized:
                    self.Source.eh.unregister("DeleteEvent",self.Destroy)
        self.Source=None
        self.DataRef=None
        self.eh=None
        self.Alive=0

#################################
#       OVERRIDABLES
#################################
    def DataCoord2SelectionCoord(self,data_coord):
        """
        Overridable: Derived classes need to reimplement this method if they
        change the coordinate system (translation, rotation, reduction,...).
        The default implementation just calls the source's.
        """
        source=self.GetSource()
        if source is not None:
            return source.DataCoord2SelectionCoord(data_coord)
        return (-1,-1)


    def SelectionCoord2DataCoord(self,selection_coord):
        """
        Overridable: Derived classes need to reimplement this method if they
        change the coordinate system (translation, rotation, reduction,...).
        The default implementation just calls the source's.
        """
        source=self.GetSource()
        if source is not None:
            return source.SelectionCoord2DataCoord(selection_coord)
        return DataPosition(-1,(-1,-1))


    def DataChanged(self,par):
        """
        Overridable: By the default implementation, when a filter receive a
        DataChange event it generates it's own event (self.Refresh()).
        If buffer_input option in construnctor, invalidates input data buffer.        
        
        Derived classes can override this method if they want to inspect the
        input to find out if it is needed or not togenerate a DataChage event.
        """
        self.Refresh()
        

#################################
#       VIRTUAL
#################################

    def GetOutput(self):
        """
        Virtual: Derived classes implements the filter behaviour in here.
        This method should read source data, execute a transformation
        and return a dictionary with the output.
        The input data can be reached by two ways:
        - Stored in self.Input in DataChanged event, if option buffer_input
          set in constructor. In this case self.GetSource.GetOutput()
          should not be called from here, in order no to perform a
          redundant operation.
        - Queried from source inside GetOutput (default)
            If the source is a Filter it should call it's GetOutput
            to retrieve the input.
            If the source is a Data object, the filter should call Data's
            methods as GetPageArrayRegion to retrieve the input.
        The method returns a dictionary.
        {} means empty selection
        Some standarized keys:
            data:  NumPy array containing the actual data
            xdata: NumPy array containing x axis
            ydata: NumPy array containing y axis
            value: Numeric value
        Derived classes can use some of these, and define new ones.

        Defalut implementation returns Input: filter is transparent
        """
        return self.GetInput()
