/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                      lsqfit.c                                $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/*
Set of subroutines to do least-squares fitting.
Some are from "Numerical Recipes in C" (NR), by Press, Flannery, Teukolsky
and Vetterling (Cambridge University Press, 1988), some are modified. The
only subroutine not from NR is 'lsqfit', which is the subroutine to be
called by the user.

In NR the convention is adopted to start all arrays with index 1. Since the
C language encourages the starting index to be 0, the NR convention is
hidden from the user by offsetting the appropriate array adresses when
communicating between NR and the user. The only places where this occurs
are in 'lsqfit' and in 'mrqcof'.

Written by:     Elias Vlieg
		AT&T Bell Laboratories
		510E Brookhaven National Laboratory
		Upton, NY 11973

address since 01-Oct-90
		FOM-institute AMOLF
		Kruislaan 407
		1098 SJ Amsterdam
		The Netherlands

address since 01-Jan-98
		Dept. of Solid State Chemistry
		University of Nijmegen
		Toernooiveld 1
		6525 ED Nijmegen
		email: vlieg@sci.kun.nl

Extended by:    Martin Lohmeier
                FOM/AMOLF
                Kruislaan 407
                1098 SJ Amsterdam
                The Netherlands

Update 28/06/2007 R. Wilcke (wilcke@esrf.fr)
                  remove all code associated with the COOL preprocessor macro,
                  as this methode has now been replaced by ASA.
Update 20/03/2007 R. Wilcke (wilcke@esrf.fr)
                  add code from the experimental version to prevent I/O when
                  NFITAUTO is set. This is only for better compatibility with
                  the experimental version, no change in functionality as long
                  as NFITAUTO is not set.
Update 16/03/2007 R. Wilcke (wilcke@esrf.fr)
                  move definition of LSQ_FNAMETERMINAL here from "lsqfit.h";
                  move macro DELTA from here to "lsqfit.h" and rename it to
                  DELTASTP to avoid confusion with variable name in ANA.
Update 19/02/2007 R. Wilcke (wilcke@esrf.fr)
                  move all routines from "Numerical Recipes" into new file
                  "numrec.c";
                  move the routines for geneerating random numbers into new
                  file "random.c";
                  move the routines for the SET FITTING PARAMETERS menu into new
                  file "set.c";
                  move declaration, allocation and free of ASTEP into mrqmin()
                  in file "numreq.c";
                  removed special include of <mingw/mem.h> for CYGWIN, as this
                  has the same effect as including <memory.h>;
                  move definition of DEBUG and EXPORT from here to "lsqfit.h";
                  calc_error(): use macro MAXIT instead of "100" for maximum
                  number of iterations.
Update 22/03/2006 R. Wilcke (wilcke@esrf.fr)
                  removed check for __CYGWIN__ since __unix is now defined
                  in Cygwin 1.X.
Update 22/02/2006 R. Wilcke (wilcke@esrf.fr)
                  modified code for MacIntosh MACOSX using predefined macro
                  __APPLE__ for this architecture.
Update 15/11/02 E.Vlieg (vlieg@sci.kun.nl)
	Keep LSQ_LOGFNAME log file only for COOL option, since the
        FIT_LOGFILE gives a more convenient summary already.
Update 25/03/2002 R. Wilcke (wilcke@esrf.fr)
                  set_fitpar(): changed abbreviations for menu commands to
                  the agreed set.
Update 28/10/01 E.Vlieg (vlieg@sci.kun.nl)
	Allow setting of flag USER_INITIAL_PARAMETERS in control menu.
Update 26/10/01 E.Vlieg (vlieg@sci.kun.nl)
	Remove automatic range in asa() when not checked. Stop asa if no
        proper range defined.
Update 24/10/01 E.Vlieg (vlieg@sci.kun.nl)
	Add FIT_LOGFILE (and related variables) to allow opening of summary
	file with fit results (handy when doing multiple fits).
Update 23/10/01 E.Vlieg (vlieg@sci.kun.nl)
	Add #define COOL preprocessor flag to allow generation of code
	without conventional annealing. This saves memory, and the COOL
	may disappear since ASA appears to be superior.
Update 22/10/01 E.Vlieg (vlieg@sci.kun.nl)
	In ana() correct int *exit_code to int exit_code: otherwise no
	memory for the int itself is allocated (was a mistake...).
Update 20/10/01 E.Vlieg (vlieg@sci.kun.nl)
	Change "center all" command in set.fit.par menu to only center
	free parameters.
Update 19/10/01 E.Vlieg (vlieg@sci.kun.nl)
	Remove type_list() in list_fitresults() in order to avoid pausing
        of cooling algorithm.
Update 18/10/01 E.Vlieg (vlieg@sci.kun.nl)
	Add ASA_NPRINT variable to set printing interval, remove the
	previous prntstate variable in vfsr.
Update 13/10/01 E.Vlieg (vlieg@sci.kun.nl)
	Merge ASA version with latest lsqfit, remove some nasty things
Update 6/4/01 D. Kaminski & E. Vlieg (danielk@sci.kun.nl)
	Include ASA code from Lester Ingberg. Not the latest version,
	but the one we obtained from Herbert Zajonz.
Update 1/3/01 E.Vlieg (vlieg@sci.kun.nl)
	Prepare for adaptive simulated annealing (ASA)
Update 8/1/01 E.Vlieg (vlieg@sci.kun.nl)
	Add LSQ_ERRORCALC: global flag to denote whether error calculation
	uses covariance matrix or chi^2 increase. Changes in: levenberg(),
	set_controlpars() and write_controlpars(). This avoids the irritating
	question about the chi^2 increase after a minimization.
Update 26/06/2000 R. Wilcke (wilcke@esrf.fr)
                  replace UNIX by __unix and CYGWIN by __CYGWIN__ (those are
                  defined by CPP on the corresponding systems).

Earlier changes:
      19/10/91: penalty function with "soft borders" introduced
                for keeping parameters in range [min,max]. This
		replaces the older algorithm where parameters
		were simply wrapped around the range when falling
		outside.

      23/10/91: FITQUALITY calculates "quality" of recent fit
      20/02/92: exp- and sqrt-functions linked to FORTRAN library
      01/07/94: compute fit parameter errors by variation of parameters
      18/08/94: set_fitpar() subroutine taken over from ROD's fit module
      22/08/94: added thermal anneal/cooldown algorithms
      26/01/95: added pair crossing in cooldown algorithm
*/

/***************************************************************************/
/*      module flags                                                       */
/***************************************************************************/

/* #define FORTRAN_EXP */

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>

#if defined(__unix) || defined(__APPLE__)
#include <memory.h>
#define max(a,b) (a>b?a:b)
#define min(a,b) (a>b?b:a)
#elif defined(MSDOS)
#include <malloc.h>
#include <mem.h>
#endif /* defined(__unix) || defined(__APPLE__) */

#include <float.h>
#include <signal.h>
#include <time.h>
#include <platform.h>

#include <menu.h>
#include <lsqfit.h>

/***************************************************************************/
/*      type definitions                                                   */
/***************************************************************************/

/***************************************************************************/
/*      global parameters                                                  */
/***************************************************************************/

#define LSQ_BREAK_INACTIVE 0
#define LSQ_BREAK_FITTING  1
#define LSQ_BREAK_INIT     1
#define LSQ_BREAK_UPDATE   2

int LSQ_ERRORCALC = 0;         /* Flag to denote type of error calculation */
#define COVARIANCE 0           /* Use covariance matrix for error estimate */
#define CHISQR 1               /* Use chi^2 increase for error estimate */
float LSQ_DCHICONV = 0.01;     /* norm. chi^2 convergence criterion */

int MAXIT = 1000;              /* max. number of iterations */
int MAXCHI = 3;                /* max. number of iterations while converging in
                                  chi-square (lsqfit() only) */
#define CRITLAMDA 1E30         /* critical lamda value (lsqfit() only) */
int MAXLAMDA = 10;             /* max. number of iterations while converging or 
                                  diverging in lamda with no change in
                                  chi-square (lsqfit() only) */
FILE *FIT_LOGFILE;             /* File with summary of fit results */
char FIT_LOGFILE_NAME[128];    /* Name of the fit summary file */
int LOGFILE=FALSE;             /* Flag to denote whether logfile is kept
                                  yes/no */
int DERIV;                     /* Flag to denote whether fitted function has
                                  analytical derivatives or not */

#ifdef ASA
long int LIMIT_ACCEPTANCES = 500; /* ASA: max number of acceptances */
float TEMPERATURE_RATIO_SCALE = 1E-8;
float COST_PARAMETER_SCALE = 0.8;
int TESTING_FREQUENCY_MODULUS = 50;
int TEMPERATURE_ANNEAL_SCALE = 500;
int ASA_NPRINT = 20;
int USER_INITIAL_PARAMETERS = TRUE;

//extern long int steps;
#endif /* ASA */

int NFITAUTO;                  /* flag set if lsqfit is called by fit_auto() */

/***************************************************************************/
/*      internal function prototypes                                       */
/***************************************************************************/

void list_fitresults( int ntot, int nfree, int ndat,
    float *fpar, float *fmin, float *fmax, float *ferr,
    float *fstep,
    int *check, int *fix, float chisqr,
    char *names, int namelength);
float calc_error(float x[], float y[], float wgt[], int ndata,
    float a[],int ma, int par,
    float amin[], float amax[], float apen[],
    float chi2_inc,
    FitFunction funcs);
float fitquality(int, float);

void write_controlpars(void);
void write_logscreen(char *string);

#ifdef ASA /* All ASA releted functions */
int asa(float a[], int ma, int lista[], int mfit, float asig[],
    float amin[], float amax[], float *chisqr, FitFunctionASA funcsasa);
float vfsr(FitFunctionASA funcsasa,
    RandomASA randflt,
    int ma, int number,
    int *parameter_int_real,
    float a[],float amin[], float amax[], int *exit_code);
#endif /* ASA */

/***************************************************************************/
int EXPORT lsqfit(float x[], float y[], float wgt[], int ndata,
    float a[], int ma, int lista[], int mfit, float asig[],
    float amin[], float amax[], float apen[], float *chisqr,
    FitFunction funcs,
#ifdef ASA
    FitFunctionASA funcsasa,
#endif
    char *fittxt, int txtlength,
    int derivatives, int method, int calcquality)
/***************************************************************************/

/*
   User callable subroutine to do the fitting. Meaning of all variables:
     x[]          array with x-values of data
     y[]          array with y-values of data
     wgt[]        array with weight factors of data
     ndata        number of data points
     a[]          array with parameter values. The subroutine starts
                  with the initial values and returns the optimal ones.
     ma           total number of parameters
     lista[]      array with serial numbers of free parameters = 1,2,3,...
     mfit         total number of free parameters
     asig[]       standard statistical error in optimum parameters
     amin[]       lower limit on allowed variable range
     amax[]       upper limit on allowed variable range
     apen[]       penalty coeff's for the fit parameters:
                    penalties in Levenberg-Marquardt are introduced by simply
                    multiplying chi-square with a factor which is given for
                    example for parameter a being larger than amax:

                      penalty = ((a-amax) * apen/(amax-amin))^2

                  e.g. for apen = 10:

                  chi-square doubles if a is off by 0.1*range
                  chi-square quadruples if a is off by 0.2*range

                  apen = 0   means no range checking
                  apen = 10  means very soft range boundaries
                  apen = 100 means modest sharp boundaries
                  apen = 1e4 means sharp boundaries

                  For simulated annealing boundaries are always sharp
                  if apen != 0

     *chisqr      pointer to minimum chi-square value
     *(funcs)     pointer to function that calculates y and dy/da
     *(funcsasa)  pointer to function that calculates cost_value for ASA
                  subroutines
     fittxt       pointer to array of fit parameter names
     txtlength    length of parameter names
     derivatives  flag denoting whether *(funcs) is able to provide
                  analytical derivatives. If not, the subroutines
                  will do (the much slower) numerical differentiation.
     method       LSQ_LEVENBERG - use Levenberg/Marquardt algorithm
                  LSQ_COOLDOWN  - use simulated annealing algorithm
                  LSQ_ASA       - use adaptive simulated annealing
     calcquality  compute quality factor (TRUE/FALSE)

     The function *(funcs) has as arguments:
        float     x-value
        float *   array of parameter values
        float *   pointer to computed y-value
        float *   pointer to computed analytical derivatives dy/da
        int       total number of parameters
        int       flag denoting whether analytical derivatives should
                  be computed.

     The function *(funcsasa) returns the cost_value and has as arguments
        float *   array of parameter values
	int       some flag
 */

{
    int rc = 1, i;
    int *check, *fix;
    char string[200];
    time_t present_time;

    if (method != LSQ_LEVENBERG && method != LSQ_COOLDOWN &&
	method != LSQ_ASA) return(0);

    check = calloc(ma, sizeof(int));
    fix   = calloc(ma, sizeof(int));

    for (i=0; i<ma; i++) fix[i] = TRUE;
    for (i=0; i<mfit; i++) fix[lista[i]-1] = FALSE;
    for (i=0; i<ma; i++) check[i] = (apen[i] != 0.);

    /* write things to summary file */

    if (LOGFILE)
	{
	time(&present_time);
	strcpy(string,ctime(&present_time));
	for (i=0;i<8;i++) string[i]=string[i+11];/* no date */
	string[i] = '\0';
	fprintf(FIT_LOGFILE,"%s ",string);

	switch (method)
	    {
	    case LSQ_LEVENBERG:
		fprintf(FIT_LOGFILE,"lev %68s","");
		break;
#ifdef ASA
	    case LSQ_ASA:
		fprintf(FIT_LOGFILE,"asa ");
		break;
#endif /* ASA */
	    }
	}

    /* Do the actual fit */          /* only rc for levenberg?? */

    switch (method)
	{
        case LSQ_LEVENBERG:
            rc = levenberg(x, y, wgt, ndata, a, ma, lista, mfit, asig,
	    	amin, amax, apen, chisqr, funcs, derivatives);
	    break;
#ifdef ASA
	case LSQ_ASA:
	    rc = asa(a, ma, lista, mfit, asig, amin, amax, chisqr, funcsasa) ;
	    if (rc == 0) return(0);
	    break;
#endif /* ASA */
        }

    /* Show the fit results */
    /* Hardly a need for separate subroutine here */

    if (!NFITAUTO)
	list_fitresults(ma, mfit, ndata, a, amin, amax, asig, NULL, check, fix,
	    *chisqr, fittxt, txtlength);

    /* Write fit results to summary file */

    if (LOGFILE)
	{
	time(&present_time);
	strcpy(string,ctime(&present_time));
	for (i=0;i<8;i++) string[i]=string[i+11];/* no date */
	string[i] = '\0';
	fprintf(FIT_LOGFILE,"%s ",string);
	fprintf(FIT_LOGFILE,"chi^2= %7.3f ",*chisqr);
	for (i=0; i<ma; i++) fprintf(FIT_LOGFILE,"%9.2e ",a[i]);
	fprintf(FIT_LOGFILE,"\n");
	}

   /* Calculate goodness of fit */

    if (rc && calcquality)
        {
	sprintf(string,", Q = %.5f\n", fitquality(ndata-mfit, *chisqr));
	if (!NFITAUTO)
	    write_logscreen(string);
        }
    else
	if (!NFITAUTO)
	    write_logscreen("\n");

    free(check); free(fix);

    return(rc);
}

#ifdef ASA
/***************************************************************************/
int asa(float a[], int ma, int lista[], int mfit,
    float asig[], float amin[], float amax[],
    float *chisqr,
    FitFunctionASA funcsasa)
/***************************************************************************/

/*
   Find global minimum in Chi^2 using asa .

   Parameters are:

     ma       -  number of fit parameters
     lista[]  -  indices of free (unfixed) fit parameters = 1,2,3,...
     mfit     -  number of free fit parameters
     asig[]   -  standard errors of fit parameters (returned)
     amin[]   -  lower boundaries of fit parameters
     amax[]   -  upper boundaries of fit parameters
     chisqr   -  pointer to (non normalized) chi^2.
     funcsasa -  pointer to fit function of asa type

   The refined values of a[] as well as the errors asig[] and chi^2 are
   returned.
 */

{

    int *fix, /**check,*/ *parameter_int_real /*, number*/;
    int /*uselog_save,*/ i;
    int exit_code;
    float *bmin, *bmax;
    float cost_value /*, cost_tangents, cost_curvature*/;

    initialize_rng();

    /* enable crash protection */

    //lsq_breakcontrol(LSQ_BREAK_INIT, funcs, a, ma);   /* disable, EV */

    /* initialize algorithm */

    //uselog_save = LSQ_USELOG;

    /* Check if all free parameters have a proper range */

    for (i=0; i<mfit; i++)
	{
	if ((amax[lista[i]-1]-amin[lista[i]-1]) < 1e-5)
	    {
	    errtype("Error, no proper parameter range specified");
	    return(0);
	    }
	}

    fix = calloc(ma+1, sizeof(int));
    //check = calloc(ma+1, sizeof(int));
    parameter_int_real = calloc(ma+1, sizeof(int));
    bmin = calloc(ma+1, sizeof(float));
    bmax = calloc(ma+1, sizeof(float));

    for (i=0; i<ma; i++) fix[i] = 1;
    for (i=0; i<mfit; i++) fix[lista[i]-1]=0;
    //for (i=0; i<ma; i++) check[i] = ((acheck[i] != 0) ? TRUE : FALSE);
    for (i=0; i<ma; i++) asig[i] = 0.;

    /* All parameters are real*/

    for (i=0; i<ma; i++)
	{
	parameter_int_real[i]=0;
	}
	/* The range of all parameters is rewritten to bmin[i] and bmax[i]*/
/*
    for (i=0; i<ma; i++) {
	bmax[i]=a[i];
	bmin[i]=a[i];
	}
 */
    /* Setting range of parameters, fixed ones have 0 range */
    for (i=0; i<ma; i++)
	{
	if (fix[i])
	    {
	    bmin[i]=a[i];
	    bmax[i]=a[i];
	    }
	else
	    {
	    bmin[i]=amin[i];
	    bmax[i]=amax[i];
	    }
	}

//    number=mfit; /* number of free parameters*/

    /* begining of ASA*/

    type_line(
	"\n  *********************** ASA start ********************** \n\n");

    /* optimize the cost_function, returning the results in
       cost_value and cost_parameters */
    cost_value = vfsr(funcsasa, randflt, ma, mfit, parameter_int_real,
	a, bmin, bmax, &exit_code);
    *chisqr =cost_value  ;

    free(fix);
    //free(check);
    free(bmin);
    free(bmax);
    return(1);

    }
#endif /* ASA */

/***************************************************************************/
float EXPORT calc_chisq(float x[], float y[], float wgt[], int ndata,
    float a[], int ma,
    FitFunction funcs
    )
/***************************************************************************/

/*
   Evaluate chi^2 for given data set x[],y[],wgt[] by comparison with
   model function funcs, with ma parameters in array a[].

   Note:

   During fitting, chi2 is not computed by this function but by
   mrqcof(), which simultaneously computes derivatives and penalties.
   Calc_chisq() is called by lsqmin() for the computation of errors of
   fit parameters.
 */

{
    int i;
    float chi2=0.;
    float ymod;
    float *dyda;

    for (i=1; i<=ndata; i++)
	{

	(*funcs)(x[i], a+1, &ymod, dyda+1, ma, 0);
	chi2 += (ymod-y[i])*(ymod-y[i])*wgt[i];
	}

    return chi2;
}


/***************************************************************************/
float  EXPORT calc_error(float x[], float y[], float wgt[], int ndata,
    float a[],int ma, int par,
    float amin[], float amax[], float apen[],
    float chi2_inc,
    FitFunction funcs
    )
/***************************************************************************/

/*
   Compute the error of the fitparameter a[par].
   Do this by finding the value of the parameter
   for which chi^2 increases by chi2_inc.

   This function should only be called after optimizing chi2
   for the parameter of which the error is to be computed !

 */

{

    float chi0, chi1, p0, pinc, pinc0, root, sigma;
    float fl,fh,pl,ph,del,f;
    int i;

    /* get starting value of chi2 */

    chi1 = chi0 = calc_chisq(x,y,wgt,ndata,a,ma,funcs);

    p0 = a[par];

    if ( apen[par] > 0. && amax[par] != amin[par] )
    {
	pinc = pinc0 = fabs((amax[par]-amin[par]))*0.01;
    } else {
	pinc = pinc0 = 0.01;
    }

    /* find upper value of parameter */

    while (chi1 - chi0 - chi2_inc <= 0)
    {
	ph = p0 + pinc;
	pinc *= 1.6;
	a[par] = ph;
	chi1 = calc_chisq(x,y,wgt,ndata,a,ma,funcs);
    }

    fh = chi1 - chi0 - chi2_inc;
    pinc = pinc0;
    chi1 = chi0;
    a[par] = p0;

    while (chi1 - chi0 - chi2_inc <= 0)
    {
	pl = p0 - pinc;
	pinc *= 1.6;
	a[par] = pl;
	chi1 = calc_chisq(x,y,wgt,ndata,a,ma,funcs);
    }

    if (fabs(p0-pl) > fabs(p0-ph))
    {
	ph = pl;
	pl = p0;
	fh = chi1 - chi0 - chi2_inc;
    } else {
	pl = p0;
    }

    fl = -chi2_inc;

    pinc=ph-pl;

    for (i = 1; i <= MAXIT; i++) {
	root=pl+pinc*fl/(fl-fh);
	a[par] = root;
	f=calc_chisq(x,y,wgt,ndata,a,ma,funcs) - chi0 - chi2_inc;
	if (f < 0.0) {
	    del=pl-root;
	    pl=root;
	    fl=f;
	} else {
	    del=ph-root;
	    ph=root;
	    fh=f;
	}
	pinc=ph-pl;
	a[par] = p0;
	sigma=fabs(root-p0);
	if (fabs(del) < 0.001*sigma || f == 0.0) {
/*	    printf("CALC_ERROR: Converged after %d iterations.\n", i); */
	    return (sigma);
	}
    }

    errtype("ERROR, Maximum number of iterations exceeded in CALC_ERROR\n");
    return (-1.);

}

/****************************************************************************/
float EXPORT fitquality(int nfree, float chi2)
/****************************************************************************/
{
    if (nfree>=2)
	{
	return gammq((nfree-2)*0.5, chi2*0.5);
	}
    else
	{
	return 0.0;
	}
}

/***************************************************************************/
int EXPORT levenberg(float x[], float y[], float wgt[], int ndata,
    float a[], int ma, int lista[], int mfit, float asig[],
    float amin[], float amax[], float apen[], float *chisqr,
    FitFunction funcs,
    int derivatives)
/***************************************************************************/

/*
   Subroutine for Levenberg Marquardt fitting. Meaning of all variables:
     x[]          array with x-values of data
     y[]          array with y-values of data
     wgt[]        array with weight factors of data
     ndata        number of data points
     a[]          array with parameter values. The subroutine starts
                  with the initial values and returns the optimal ones.
     ma           total number of parameters
     lista[]      array with serial numbers of free parameters = 1,2,3,...
     mfit         total number of free parameters
     asig[]       standard statistical error in optimum parameters
     amin[]       lower limit on allowed variable range
     amax[]       upper limit on allowed variable range
     apen[]       penalty coeff's for the fit parameters:
                    penalties are introduced by simply multiplying
                    chi-square with a factor which is given for
                    example for parameter a being larger than amax:

                      penalty = ((a-amax) * apen/(amax-amin))^2

                    e.g. for apen = 10:

                      chi-square doubles if a is off by 0.1*range
                      chi-square quadruples if a is off by 0.2*range

                    apen = 0   means no range checking
                    apen = 10  means very soft range boundaries
                    apen = 100 means modest sharp boundaries
                    apen = 1e4 means sharp boundaries

     *chisqr      pointer to minimum chi-square value
     *(funcs)     pointer to function that calculates y and dy/da
     derivatives  flag denoting whether *(funcs) is able to provide
                  analytical derivatives. If not, the subroutines
                  will do (the much slower) numerical differentiation.

     The function *(funcs) has as arguments:
       float   x-value
       float*  array of parameter values
       float*  pointer to computed y-value
       float*  pointer to computed analytical derivatives dy/da
       int     total number of parameters
       int     flag denoting whether analytical derivatives should be computed.
 */

{

    float **covar,**alpha;
    float alamda,oldchi,oldalamda,chi2_inc;
    float *acoeff;
    int i,j,chistop,iter,lamdastop=0, res, nfree;
    char string[100];

    /* enable crash protection */

    lsq_breakcontrol(LSQ_BREAK_INIT, funcs, a, ma);

    /*
     * Set global flag denoting whether function has analytical derivatives
     * or not
     */

    DERIV = derivatives;

    covar = matrix(1,ma,1,ma);
    alpha = matrix(1,ma,1,ma);
    acoeff = vector(0,ma-1);

    nfree = ndata - mfit;
    if (nfree < 1) nfree = 1;

    /* Check the lower and upper bounds on variable ranges */

#define NDX lista[i]-1

    for (i = 0; i < mfit; i++)
    {
	if (amin[NDX] > amax[NDX])
	{
	    apen[NDX] = 0;
	    amin[NDX] = amax[NDX] = 0;
	}
	if ((amin[NDX] != amax[NDX]) &&
	    ((a[NDX] < amin[NDX]) || (a[NDX] > amax[NDX])))
	    a[NDX] = 0.5*(amin[NDX]+amax[NDX]);
    }

#ifdef DEBUG
    printf("-------------------------------------------------------------------\n");
    printf("INITIALIZATION\n\n");
#endif

    /* Initialize penalty coefficients */

    for (i = 0; i < mfit; i++)
	if (apen[NDX] && amax[NDX] > amin[NDX])
	    acoeff[NDX] = apen[NDX] / (amax[NDX]-amin[NDX]);
	else
	    acoeff[NDX] = 0;
#ifdef DEBUG
    printf("Pen. Coeff. = ");
    for (i=0; i<ma; i++)
    {
	printf("%10.2e  ",acoeff[i]);
    }
    printf("\n");
#endif

#undef NDX

    /* Initialize Marquardt algorithm */

    alamda = -1.;
    if (!mrqmin(x-1,y-1,wgt-1,ndata,a-1,ma,lista-1,mfit,amin-1,amax-1,
	acoeff-1, covar,alpha,chisqr,funcs,&alamda)) return(0);

#ifdef DEBUG
    printf("\n");
#endif
    /* Iterate to find minimum in chisqr */

    iter = 1;
    chistop = 0;
    if (mfit == 0)
	iter = -1; /* Don't iterate if no free parameters */

    while (!((chistop >= MAXCHI) && (abs(lamdastop) >= MAXLAMDA))
	&& (alamda < CRITLAMDA) && (iter <= MAXIT) && (iter != -1))
    {                   
#ifdef DEBUG
        printf("-------------------------------------------------------------------\n");
        printf("ITERATION %d\n\n", iter);
#else
	sprintf(string,"Iteration %2d\r",iter);
	type_line(string);
#endif
	iter++;
	oldchi = *chisqr;
	oldalamda = alamda;
	if (!mrqmin(x-1,y-1,wgt-1,ndata,a-1,ma,lista-1,mfit,amin-1,amax-1,
	    acoeff-1,covar,alpha,chisqr,funcs,&alamda)) return(0);
#ifdef DEBUG
	printf("alamda = %.1e\n",alamda);
#endif
	if (alamda < oldalamda)		/* are things getting better ? */
	{

	    // update good parameters
	    lsq_breakcontrol(LSQ_BREAK_UPDATE, NULL, a, 0);

	    if (lamdastop < 0)
	    {
		lamdastop = 2;
	    }
	    else
	    {
		lamdastop += 2;
	    }
	}
	else                            /* No, this is not convergence! */
	{
	    if (lamdastop > 0)
	    {
		lamdastop = -1;
	    }
	    else
	    {
		lamdastop -= 1;
	    }
	}
#ifdef DEBUG
	printf("lamdastop = %+2d\n",lamdastop);
#endif
	if (fabs((oldchi-*chisqr)/nfree) < LSQ_DCHICONV)
	{
	    chistop++;
	}
	else
	{
	    chistop = 0;
	}

    } /* while loop */

    if (iter != -1)	/* no free parameters */
    {
	if (chistop >= MAXCHI)
	{
	    if (lamdastop > 0)
		type_line("Iteration stopped after converging.\n\n");
	    else
		type_line("Iteration stopped with diverging lambda.\n\n");
	}
	if (oldalamda >= CRITLAMDA*0.1)
	{
	    type_line("Iteration stopped with critical lambda.\n\n");
	}
	if (iter >= MAXIT)
	{
	    errtype("Too many iterations.\n");
	}
/*	printf("Covariance matrix: \n\n");
	for (i=1; i<=ma; i++)
	{
	    for (j=1; j<=i; j++)
	    {
		printf("%10.2e  ",covar[i][j]);
	    }
	    printf("\n");
	}
	printf("\n");
*/

	} /* iter != -1 */

    // update good parameters

    lsq_breakcontrol(LSQ_BREAK_UPDATE, NULL, a, 0);

    /* Compute standard errors of fitted parameters */

    if (mfit) {
	if (LSQ_ERRORCALC == COVARIANCE)
	{
	    alamda = 0.;
	    res=mrqmin(x-1,y-1,wgt-1,ndata,a-1,ma,lista-1,mfit,amin-1,amax-1,
		acoeff-1,covar,alpha,chisqr,funcs,&alamda);

	    for (i = 0; i < ma; i++)
	    {
		if (covar[i+1][i+1] < 0. || !res)
		    asig[i] = -1.;    /* negative error indicates error */
		else
		    asig[i] = sqrt(covar[i+1][i+1]);
	    }
	}
	else if (LSQ_ERRORCALC == CHISQR)
	{
	    if (!NFITAUTO)
		chi2_inc = get_real(1.,
		    "Chi2 increase for error estimate: [1.0] ");
	    if (chi2_inc < 0.01) chi2_inc = 1.;
	    for (i = 0; i < ma; i++)
		asig[i] = -1.;

	    for (i = 0; i < mfit; i++)
		asig[lista[i]-1] = calc_error(x-1, y-1, wgt-1, ndata,
		    a-1, ma, lista[i],
		    amin-1, amax-1, apen-1,
		    chi2_inc,
		    funcs);
	}

    } /* if mfit */

    free_matrix(alpha,1,ma,1,ma);
    free_matrix(covar,1,ma,1,ma);
    free_vector(acoeff,0,ma-1);

    // restore saved parameters
    // this is necessary if anything went wrong during the error computation

    lsq_breakcontrol(LSQ_BREAK_RESTORE, NULL, a, 0);

    return(1);

}

/***************************************************************************/
void EXPORT lsq_breakcontrol(int command,FitFunction funcs,float *a,int na)
/***************************************************************************/

/*
   crash protection: restore old parameters if crashed or Ctrl-Break pressed
 */
{
    static float *a_saved;
    static int na_saved;
    static int status;
    static FitFunction funcs_saved;
    int i;
    float y;

    switch (command)
        {
        case LSQ_BREAK_INIT:
            a_saved = vector(0, na-1);
            if (a_saved != NULL)
                {
                for (i=0; i<na; i++) a_saved[i]=a[i];
                status = LSQ_BREAK_FITTING;
                na_saved = na;
                funcs_saved = funcs;
                }
            break;

        case LSQ_BREAK_UPDATE:
            if (status == LSQ_BREAK_FITTING && a_saved != NULL)
                for (i=0; i<na_saved; i++) a_saved[i]=a[i];
            break;

        case LSQ_BREAK_RESTORE:
            if (status == LSQ_BREAK_FITTING && a_saved != NULL)
                {
                if (a!=NULL) for (i=0; i<na_saved; i++) a[i]=a_saved[i];
                status = LSQ_BREAK_INACTIVE;
                funcs_saved(0., a_saved, &y, NULL, na_saved, 0);
                free_vector(a_saved, 0, na_saved-1);
                a_saved = NULL;
                }
            break;

        }
}

/***************************************************************************/
void EXPORT list_fitresults( int ntot, int nfree, int ndat,
    float *fpar, float *fmin, float *fmax, float *ferr,
    float *fstep,
    int *check, int *fix, float chisqr,
    char *names, int namelength)
/***************************************************************************/

/*
   List fit results
 */

{
    char STRING[200];
    float chisqrN;
    int  i;

    write_logscreen("Fit results:\n");
    for (i = 0; i<ntot; i++)
	{
	sprintf(STRING,"%*s = %10.4g",namelength,names+i*namelength,fpar[i]);
	write_logscreen(STRING);
	if (!fix[i])
	    {
            if (fstep != NULL)
                {
                sprintf(STRING, "    %-10.4g", fstep[i]);
                }
	    else
                {
                sprintf(STRING," +/- %-10.4g",ferr[i]);
                }
	    write_logscreen(STRING);
	    if (check[i] == TRUE)
		{
		sprintf(STRING,"    [%10.4g , %-10.4g]\n", fmin[i],fmax[i]);
		}
	    else
                {
                sprintf(STRING, "\n");
                }
	    }
	else
	    {
		sprintf(STRING,"       <FIXED>\n");
	    }
	write_logscreen(STRING);
	}
    chisqrN = chisqr/(ndat-nfree+1e-10);

    sprintf(STRING,"\nchisqr = %.4f, normalized = %.4f",chisqr,chisqrN);
    write_logscreen(STRING);
}

/***************************************************************************/
void EXPORT set_controlpars(char *prompt)
/***************************************************************************/

/*
   set control parameters for fitting
 */
{
    enum idx {
	i_zero,
	i_covariance,
	i_chisqr,
	i_itermax,
	i_conv,
	i_open,
	i_close,
#ifdef ASA
	i_asa_accept,
	i_asa_limit,
	i_asa_ratio,
	i_asa_cost,
	i_asa_freq,
	i_asa_print,
	i_asa_userinit,
#endif /* ASA */
	i_list,
	i_help,
	i_return
    };

    struct MENU menu[i_return] =
    {
	"covariance",2,i_covariance, "Use covariance matrix for error estimate",
	"chisqr",    2,i_chisqr,     "Use chi^2 for error estimate",
	"itermax",   2,i_itermax,    "Max. number of iterations (LM fit)",
	"conv",      3,i_conv,       "chi^2 convergence criterion",
	"open",      2,i_open,       "Open fit summary file",
	"close",     2,i_close,      "Close fit summary file",
#ifdef ASA
	"anneal",    3,i_asa_accept, "ASA: Temperature anneal scale",
	"limit",     3,i_asa_limit,  "ASA: Limit in number of acceptances",
	"ratio",     3,i_asa_ratio,  "ASA: Temperature ratio scale",
	"cost",      3,i_asa_cost,   "ASA: Cost parameter scale",
	"reanneal",  3,i_asa_freq,   "ASA: Reannealing interval",
	"nprint",    2,i_asa_print,  "ASA: printing inverval",
	"userinit",  2,i_asa_userinit,"ASA: user initial values (yes/no)",
#endif /* ASA */
	"list",      1,i_list,       "List parameters",
	"help",      1,i_help,       "This list",
	"return",    1,i_return,     "Return to main menu"
    };

    char token[100];
    char string[200];
    int stop = FALSE;
    int i;

    while (!stop)
    {
	if (!get_token(token, prompt)) return;
	switch (cmnd_match(token, menu, i_return))
	{

	    case i_zero:
		break;

	    case i_covariance:
		LSQ_ERRORCALC = COVARIANCE;
		break;

	    case i_chisqr:
		LSQ_ERRORCALC = CHISQR;
		break;

	    case i_itermax:
                sprintf(string,
		    "Maximum number of iterations (L/M fitting only [%d]: ",
		    MAXIT);
                MAXIT = get_int(MAXIT, string);
		break;

	    case i_conv:
		sprintf(string,
		    "Delta Chi^2 convergence criterion [%.2E]: ", LSQ_DCHICONV);
		LSQ_DCHICONV = get_real(LSQ_DCHICONV, string);
		break;

	    case i_open:
		if (LOGFILE)
		{
		    sprintf(string,
			"Warning, log file '%s' already open, new file? [no]: ",
			FIT_LOGFILE_NAME);
		    if (yesno(FALSE,string))
			fclose(FIT_LOGFILE);
		    else
			break;
		}
		while (!get_token(FIT_LOGFILE_NAME,
		    "Name of fit summary file [.log]: "));
		add_extension(FIT_LOGFILE_NAME,"log");
		if ((FIT_LOGFILE = fopen(FIT_LOGFILE_NAME,"w")) == NULL)
		{
		    sprintf(string,"Failed to open '%s'",FIT_LOGFILE_NAME);
		    errtype(string);
		    LOGFILE = FALSE;
		}
		else
		    LOGFILE = TRUE;

		break;

	    case i_close:
		if (LOGFILE)
		{
		    fclose(FIT_LOGFILE);
		    LOGFILE = FALSE;
		}
		break;

#ifdef ASA
	    case i_asa_accept:
		sprintf(string, "Temperature anneal scale [%d]: ",
		    TEMPERATURE_ANNEAL_SCALE);
		TEMPERATURE_ANNEAL_SCALE=
		    get_int(TEMPERATURE_ANNEAL_SCALE,string);
		break;

	    case i_asa_limit:
		sprintf(string, "Limit in total number of acceptances [%ld]: ",
		    LIMIT_ACCEPTANCES);
		LIMIT_ACCEPTANCES = get_int(LIMIT_ACCEPTANCES,string);
		break;

	    case  i_asa_ratio:
		sprintf(string, "Temperature ratio scale [%.2e]: ",
		    TEMPERATURE_RATIO_SCALE);
		TEMPERATURE_RATIO_SCALE = get_real(TEMPERATURE_RATIO_SCALE,
		    string);
		break;

	    case i_asa_cost:
		sprintf(string, "Cost parameter scale [%f]: ",
		    COST_PARAMETER_SCALE);
		COST_PARAMETER_SCALE = get_real(COST_PARAMETER_SCALE,string);
		break;
	    case i_asa_freq:
		sprintf(string, "Reannealing interval [%d]: ",
		    TESTING_FREQUENCY_MODULUS);
		TESTING_FREQUENCY_MODULUS = get_int(TESTING_FREQUENCY_MODULUS,
		    string);
		break;
	    case i_asa_print:
		sprintf(string, "Printing interval [%d]: ", ASA_NPRINT);
		ASA_NPRINT = get_int(ASA_NPRINT,string);
		break;
	    case i_asa_userinit:
		sprintf(string,"User initial values [%s]: ",
		    yesnostr(USER_INITIAL_PARAMETERS));
		USER_INITIAL_PARAMETERS = yesno(USER_INITIAL_PARAMETERS,string);
		break;
#endif /* ASA */

	    case i_list:
		type_line("Error calculation using ");
		if (LSQ_ERRORCALC == COVARIANCE)
		    type_line ("covariance matrix\n");
		if (LSQ_ERRORCALC == CHISQR) type_line("chi^2 increase\n");
		sprintf(string, "Delta Chi^2 convergence criterion : %.2E\n",
		    LSQ_DCHICONV);
		type_line(string);
		sprintf(string, "Maximum number of iterations      : %d\n",
		    MAXIT);
		type_line(string);
		if (LOGFILE)
		{
		    sprintf(string,"Fit summary file                  : %s\n",
			FIT_LOGFILE_NAME);
		    type_line(string);
		}
		else
                    type_line("No fit summary file\n");
#ifdef ASA
		sprintf(string, "Temperature anneal scale          : %d\n",
		    TEMPERATURE_ANNEAL_SCALE);
		type_line(string);
		sprintf(string, "Limit of accepted points          : %ld\n",
		    LIMIT_ACCEPTANCES);
		type_line(string);
		sprintf(string, "Temperature ratio scale           : %.2e\n",
		    TEMPERATURE_RATIO_SCALE);
		type_line(string);
		sprintf(string, "Cost parameter scale              : %f\n",
		    COST_PARAMETER_SCALE);
		type_line(string);
		sprintf(string, "Reannealing interval              : %d\n",
		    TESTING_FREQUENCY_MODULUS);
		type_line(string);
		sprintf(string, "Printing interval                 : %d\n",
		    ASA_NPRINT);
		type_line(string);
		sprintf(string, "User initial parameter values     : %s\n",
		    yesnostr(USER_INITIAL_PARAMETERS));
		type_line(string);
#endif /* ASA */
		break;

	    case i_help:
		list_menu("SET CONTROL PARAMETERS", menu, i_return);
		break;

	    case i_return:
		stop = TRUE;
		break;

	} /* end of switch */

    } /* end of while */

}

/***************************************************************************/
void write_logscreen(char *string)
/***************************************************************************/

/*
   write string to log file and screen
 */

{
    type_line(string);
}
