/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                         read.c                                $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: read.c,v 1.8 2006/03/28 15:43:39 wilcke Exp $

****************************************************************************/

/*
Update 22/03/2006 R. Wilcke (wilcke@esrf.fr)
                  modified code for MacIntosh MACOSX using predefined macro
                  __APPLE__ for this architecture.
Update 19/01/2004 R. Wilcke (wilcke@esrf.fr) 
                  removed declarations of fopen() (is declared in "stdio.h").
Update 21/12/2000 O. Svensson (svensson@esrf.fr)
                  Removed check for __CYGWIN__ since __unix is now defined
                  in Cygwin 1.X.
Update 26/06/2000 R. Wilcke (wilcke@esrf.fr)
                  replaced everywhere the buffer size for filenames by
                  the ISO-C defined macro FILENAME_MAX;
                  replace UNIX by __unix and CYGWIN by __CYGWIN__ (those are
                  defined by CPP on the corresponding systems).
*/

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#include "rod.h"

/***************************************************************************/
void    calc_rlat(float dlat[6], float rlat[6])
/***************************************************************************/

    /*
      Compute reciprocal lattice parameters. The convention used is that
      a[i]*b[j] = d[ij], i.e. no 2PI's in reciprocal lattice.
    */

    {

    double  volume,dummy;

    /* compute volume of real lattice cell */

    volume = dlat[0]*dlat[1]*dlat[2]*
	sqrt(1+2*cos(dlat[3])*cos(dlat[4])*cos(dlat[5])
	     -cos(dlat[3])*cos(dlat[3])
	     -cos(dlat[4])*cos(dlat[4])
	     -cos(dlat[5])*cos(dlat[5]));

    /* compute reciprocal lattice parameters */

    rlat[0] = dlat[1]*dlat[2]*sin(dlat[3])/volume;
    rlat[1] = dlat[0]*dlat[2]*sin(dlat[4])/volume;
    rlat[2] = dlat[0]*dlat[1]*sin(dlat[5])/volume;
    rlat[3] = acos((cos(dlat[4])*cos(dlat[5])-cos(dlat[3]))
		   /(sin(dlat[4])*sin(dlat[5])));
    rlat[4] = acos((cos(dlat[3])*cos(dlat[5])-cos(dlat[4]))
		   /(sin(dlat[3])*sin(dlat[5])));
    rlat[5] = acos((cos(dlat[3])*cos(dlat[4])-cos(dlat[5]))
		   /(sin(dlat[3])*sin(dlat[4])));

    }

/***************************************************************************/
void    read_bulk(void)
/***************************************************************************/

    /*
    Read file with coordinates of bulk atoms.
    First line: comments
    Second:     direct lattice parameters
    Data lines: El X Y Z (NDW), where
			El      element symbol
			X       x-position in reduced coordinates
			Y       y-position in reduced coordinates
			Z       z-position in reduced coordinates
			NDW     serial number of Debye-Waller par. (optional)
    */

    {

    char    filename[FILENAME_MAX];
    FILE    *infile;
    float   dlatb[6];
    int     end_of_file,dw_column;
    char    elbulk[MAXATOMS][3];
    char    dummy[20];
    float   dwdummy;
    int i,j;
    int newtype;

    /* Get filename and open file, return if not successful */

    get_token(filename,"Filename with bulk atoms (.bul): ");
    add_extension(filename,"bul");
    if ((infile = fopen(filename,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open file '%s'",filename);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Read comments and lattice parameters */

    fgets(INLINE,INLLEN,infile);
    sprintf(STRING,"Title: %s",INLINE);
    type_line(STRING);
    fgets(INLINE,INLLEN,infile);
    sscanf(INLINE,"%f%f%f%f%f%f",&dlatb[0],&dlatb[1],&dlatb[2],&dlatb[3],
	   &dlatb[4],&dlatb[5]);
    for (i = 3; i < 6; i++) dlatb[i] /= RAD;

    /* If a surface file has already been read in, check the lattice
       parameters for consistency */

    if (NSURF == 0)
	{
	for (i = 0; i < 6; i++) DLAT[i] = dlatb[i];
	calc_rlat(DLAT,RLAT);
	}
    else
	{
	for (i = 0; i < 6; i++)
	    {
	    if (fabs(DLAT[i]-dlatb[i]) > 1e-4)
		{
		errtype("WARNING, lattice parameters different from surface");
		}
	    }
	}

    /* Find out from first data line, whether a serial number for
       Debye-Waller parameters is given */

    fgets(INLINE,INLLEN,infile);
#if defined(__unix) || defined(__APPLE__)
    sscanf(INLINE,"%2s%f%f%f%d",
	   elbulk[0],&XB[0],&YB[0],&ZB[0],&NDWB[0]);
#else
    sscanf(INLINE,"%[ ]%2s%f%f%f%d",
	   dummy,elbulk[0],&XB[0],&YB[0],&ZB[0],&NDWB[0]);
#endif /* defined(__unix) || defined(__APPLE__) */
    if (NDWB[0] != 0)
	{
	dw_column = TRUE;
	}
    else
	{
	dw_column = FALSE;
	}

    /* Read the rest of the atoms */

    NBULK = 1;
    while ((fgets(INLINE,INLLEN,infile) != NULL) /*&& (NBULK < MAXATOMS)*/)
	{
	if (dw_column)
	    {
#if defined(__unix) || defined(__APPLE__)
	    sscanf(INLINE,"%2s%f%f%f%d",elbulk[NBULK],
		   &XB[NBULK],&YB[NBULK],&ZB[NBULK],&NDWB[NBULK]);
#else
	    sscanf(INLINE,"%[ ]%2s%f%f%f%d",dummy,elbulk[NBULK],
		   &XB[NBULK],&YB[NBULK],&ZB[NBULK],&NDWB[NBULK]);
#endif /* defined(__unix) || defined(__APPLE__) */
	    }
	else
	    {
#if defined(__unix) || defined(__APPLE__)
	    sscanf(INLINE,"%2s%f%f%f",elbulk[NBULK],
		   &XB[NBULK],&YB[NBULK],&ZB[NBULK]);
#else
	    sscanf(INLINE,"%[ ]%2s%f%f%f",dummy,elbulk[NBULK],
		   &XB[NBULK],&YB[NBULK],&ZB[NBULK]);
#endif /* defined(__unix) || defined(__APPLE__) */
	    }
	NBULK++;
	if (NBULK == MAXATOMS)
	    {
	    errtype("ERROR, maximum number of model atoms exceeded");
	    break;
	    }
	}
    fclose(infile);

    /* Find the different element types and store them in ELEMENT */

    if (NTYPES == 0)
	{
	NTYPES = 1;
	sprintf(ELEMENT[0],"%s",elbulk[0]);
	}
    for (i = 0; i < NBULK; i++)
	{
	newtype = TRUE;
	for (j = 0; j < NTYPES; j++)
	    {
	    if (el_equal(elbulk[i],ELEMENT[j])) newtype = FALSE;
	    }
	if (newtype)
	    {
	    if (NTYPES == MAXTYPES)
		{
		errtype("ERROR, too many atom types in unit cell");
		return;
		}
	    NTYPES++;
	    sprintf(ELEMENT[NTYPES-1],"%s",elbulk[i]);
	    }
	}

    /* Assign a type number to all atoms in the model */

    for (i = 0; i < NBULK; i++)
	{
	for (j = 0; j < NTYPES; j++)
	    {
	    if (el_equal(elbulk[i],ELEMENT[j])) TB[i] = j;
	    }
	}

    /* Store the coefficients for the computation of the atomic scattering
       factors for all the different elements in F_COEFF */

    for (i = 0; i < NTYPES; i++)
	{
	get_coeff(ELEMENT[i],F_COEFF[i]);
	}

    /* Find highest serial number of Debye-Waller, if read in */

    if (dw_column)
	{
	for (i = 0; i < NBULK; i++)
	    {
	    if (NDWB[i] > NDWTOT)
		{
		if (NDWB[i] > MAXPAR)
		    {
		    sprintf(STRING,
			    "ERROR, total # Debye-Wallers should be < %1d",MAXPAR);
		    errtype(STRING);
		    }
		else
		    {
		    NDWTOT = NDWB[i];
		    }
		}
	    }
	}
    }

/***************************************************************************/
void    read_data(void)
/***************************************************************************/

    /*
    Read file with truncation rod data.
    First line: comments
    Data:       h       k       l       f       error   lbragg.
    */

    {

    char    filename[FILENAME_MAX];
    FILE    *infile;
    int     i;
    int     end_of_file;

    /* Get filename and open file, return if not successful */

    get_token(filename,"Give filename (.dat): ");
    add_extension(filename,"dat");
    if ((infile = fopen(filename,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open file '%s'",filename);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Type comments */

    if (fgets(INLINE,INLLEN,infile) == NULL)
	{
	errtype("ERROR, file is empty");
	fclose(infile);
	clear_command();
	return;
	}
    else
	{
	type_line(INLINE);
	}

    /* Read data */

    NDAT = 0;
    end_of_file = FALSE;
    while ((fgets(INLINE,INLLEN,infile) != NULL) && (!end_of_file))
	{
	if (sscanf(INLINE,"%f%f%f%f%f%f",
		   &HDAT[NDAT],&KDAT[NDAT],&LDAT[NDAT],
		   &FDAT[NDAT],&ERRDAT[NDAT],&LBR[NDAT]) >= 4)
	    {
	    NDAT++;
	    if (NDAT == MAXDATA)
		{
		errtype(
		    "ERROR, maximum number of data points exceeded, file truncated");
		break;
		}
	    }
	else
	    {
	    end_of_file = TRUE;
	    }
	}
    fclose(infile);
    LBRAGG = LBR[0];
    if (NDAT == 0)
	{
	errtype("No data in file");
	clear_command();
	}

    }

/***************************************************************************/
void    read_fit(void)
/***************************************************************************/

    /*
    Read file with coordinates of surface atoms and parameters used for
    fitting.
    First line: comments
    Second:     direct lattice parameters
    Data lines: El X XCONST NXDIS X2CONST NX2DIS
		Y YCONST NYDIS Y2CONST NY2DIS Z NZDIS NDWS NDWS2 NOCCUP
		where
			El      element symbol
			X       x-position in reduced coordinates
			XCONST  multiplication factor of NXDIS
			NXDIS   serial number of x-displacement parameter
			X2CONST multiplication factor of NX2DIS
			NX2DIS  serial number of 2nd x-displacement parameter
			Y       y-position in reduced coordinates
			YCONST  multiplication factor of NYDIS
			NYDIS   serial number of y-displacement parameter
			Y2CONST multiplication factor of NY2DIS
			NY2DIS  serial number of 2nd y-displacement parameter
			Z       z-position in reduced coordinates
			NZDIS   serial number of z-displacement parameter
			NDWS    serial number of Debye-Waller parameter in
				the in-plane direction
			NDWS2   serial number of Debye-Waller parameter in
				the perpendicular direction.
			NOCCUP  serial number of occupancy parameter
    */

    {

    char    filename[FILENAME_MAX];
    FILE    *infile;
    float   dlats[6];
    int     end_of_file,dw_column;
    char    elsurf[MAXATOMS][3];
    char    dummy[20];
    float   dwdummy;
    int i,j;
    int newtype;

    /* Get filename and open file, return if not successful */

    get_token(filename,"Filename with surface fit model (.fit): ");
    add_extension(filename,"fit");
    if ((infile = fopen(filename,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open file '%s'",filename);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Read comments and lattice parameters */

    fgets(INLINE,INLLEN,infile);
    sprintf(STRING,"Title: %s",INLINE);
    type_line(STRING);
    fgets(INLINE,INLLEN,infile);
    sscanf(INLINE,"%f%f%f%f%f%f",&dlats[0],&dlats[1],&dlats[2],&dlats[3],
	   &dlats[4],&dlats[5]);
    for (i = 3; i < 6; i++) dlats[i] /= RAD;

    /* If a bulk file has already been read in, check the lattice
       parameters for consistency */

    if (NBULK == 0)
	{
	for (i = 0; i < 6; i++) DLAT[i] = dlats[i];
	calc_rlat(DLAT,RLAT);
	}
    else
	{
	for (i = 0; i < 6; i++)
	    {
	    if (fabs(DLAT[i]-dlats[i]) > 1e-4)
		{
		errtype("ERROR, lattice parameters different from bulk");
		return;
		}
	    }
	}

    /* Read the surface fit parameters */

    NSURF = 0;
    while ((fgets(INLINE,INLLEN,infile) != NULL) /*&& (NSURF < MAXATOMS)*/)
	{
#if defined(__unix) || defined(__APPLE__)
	sscanf(INLINE,"%2s%f%f%d%f%d%f%f%d%f%d%f%d%d%d%d%d",
	       elsurf[NSURF],
	       &XS[NSURF],&XCONST[NSURF],&NXDIS[NSURF],
	       &X2CONST[NSURF],&NX2DIS[NSURF],
	       &YS[NSURF],&YCONST[NSURF],&NYDIS[NSURF],
	       &Y2CONST[NSURF],&NY2DIS[NSURF],
	       &ZS[NSURF],&NZDIS[NSURF],
	       &NDWS[NSURF],&NDWS2[NSURF],&NOCCUP[NSURF]);
#else
	sscanf(INLINE,"%[ ]%2s%f%f%d%f%d%f%f%d%f%d%f%d%d%d%d%d",
	       dummy,elsurf[NSURF],
	       &XS[NSURF],&XCONST[NSURF],&NXDIS[NSURF],
	       &X2CONST[NSURF],&NX2DIS[NSURF],
	       &YS[NSURF],&YCONST[NSURF],&NYDIS[NSURF],
	       &Y2CONST[NSURF],&NY2DIS[NSURF],
	       &ZS[NSURF],&NZDIS[NSURF],
	       &NDWS[NSURF],&NDWS2[NSURF],&NOCCUP[NSURF]);
#endif /* defined(__unix) || defined(__APPLE__) */
	NSURF++;
	if (NSURF == MAXATOMS)
	    {
	    errtype("ERROR, maximum number of model atoms exceeded");
	    break;
	    }
	}
    fclose(infile);
    NSURFTOT = NSURF;

    /* Find the different element types and store them in ELEMENT */

    if (NTYPES == 0)
	{
	NTYPES = 1;
	sprintf(ELEMENT[0],"%s",elsurf[0]);
	}
    for (i = 0; i < NSURF; i++)
	{
	newtype = TRUE;
	for (j = 0; j < NTYPES; j++)
	    {
	    if (el_equal(elsurf[i],ELEMENT[j])) newtype = FALSE;
	    }
	if (newtype)
	    {
	    if (NTYPES == MAXTYPES)
		{
		errtype("ERROR, too many atom types in unit cell");
		return;
		}
	    NTYPES++;
	    sprintf(ELEMENT[NTYPES-1],"%s",elsurf[i]);
	    }
	}

    /* Assign a type number to all atoms in the model */

    for (i = 0; i < NSURF; i++)
	{
	for (j = 0; j < NTYPES; j++)
	    {
	    if (el_equal(elsurf[i],ELEMENT[j])) TS[i] = j;
	    }
	}

    /* Store the coefficients for the computation of the atomic scattering
       factors for all the different elements in F_COEFF */

    for (i = 0; i < NTYPES; i++)
	{
	get_coeff(ELEMENT[i],F_COEFF[i]);
	}

    /* Assign the values of the surface positions to the refined positions */

    for (i = 0; i < NSURF; i++)
	{
	XSFIT[i] = XS[i];
	YSFIT[i] = YS[i];
	ZSFIT[i] = ZS[i];
	}

    /* Find highest serial number of the displacement, Debye-Waller and
       occupancy parameters */

    for (i = 0; i < NSURF; i++)
	{
	if (NXDIS[i] > NDISTOT) NDISTOT = NXDIS[i];
	if (NX2DIS[i] > NDISTOT) NDISTOT = NX2DIS[i];
	if (NYDIS[i] > NDISTOT) NDISTOT = NYDIS[i];
	if (NY2DIS[i] > NDISTOT) NDISTOT = NY2DIS[i];
	if (NZDIS[i] > NDISTOT) NDISTOT = NZDIS[i];
	if (NDWS[i] > NDWTOT) NDWTOT = NDWS[i];
	if (NDWS2[i] > NDWTOT2) NDWTOT2 = NDWS2[i];
	if (NOCCUP[i] > NOCCTOT) NOCCTOT = NOCCUP[i];
	}
    if (NDISTOT > MAXPAR)
	{
	NDISTOT = MAXPAR;
	sprintf(STRING,
		"Error, total number of displacement parameters should be < %1d",
		MAXPAR+1);
	errtype(STRING);
	}
    if (NDWTOT > MAXPAR)
	{
	NDWTOT = MAXPAR;
	sprintf(STRING,
		"Error, total no. of paral. Debye-Waller param. should be < %1d",
		MAXPAR+1);
	errtype(STRING);
	}
    if (NDWTOT2 > MAXPAR)
	{
	NDWTOT2 = MAXPAR;
	sprintf(STRING,
		"Error, total no. of perp. Debye-Waller param. should be < %1d",
		MAXPAR+1);
	errtype(STRING);
	}
    if (NOCCTOT > MAXPAR)
	{
	NOCCTOT = MAXPAR;
	sprintf(STRING,
		"Error, total number of occupancy parameters should be < %1d",
		MAXPAR+1);
	errtype(STRING);
	}

    update_model();

    }

/***************************************************************************/
void    read_par(void)
/***************************************************************************/

    /*
    Read macro file with values of fitting parameters.
    */

    {

    char    filename[FILENAME_MAX];
    FILE    *infile;

    /* Get filename and open file, return if not successful */

    get_token(filename,"Filename with surface atoms (.par): ");
    add_extension(filename,"par");
/*
  if ((infile = fopen(filename,"r")) == NULL)
  {
  sprintf(STRING,"Failed to open file '%s'",filename);
  errtype(STRING);
  clear_command();
  return;
  }
*/

    /* Run macro file */

    put_command(filename);
    run_macro();

    }

/***************************************************************************/
void    read_surf(void)
/***************************************************************************/

    /*
    Read file with coordinates of surface atoms.
    First line: comments
    Second:     direct lattice parameters
    Data lines: El X Y Z (NDW NDW2), where
			El      element symbol
			X       x-position in reduced coordinates
			Y       y-position in reduced coordinates
			Z       z-position in reduced coordinates
			NDW     serial number of Debye-Waller parameter
				along in-plane direciton (optional)
			NDW2    serial number of Debye-Waller parameter
				along perpendicular direction (optional)
    */

    {

    char    filename[FILENAME_MAX];
    FILE    *infile;
    float   dlats[6];
    int     end_of_file,dw_column,dw2_column;
    char    elsurf[MAXATOMS][3];
    char    dummy[20];
    float   dwdummy;
    int i,j;
    int newtype;

    /* Get filename and open file, return if not successful */

    get_token(filename,"Filename with surface atoms (.sur): ");
    add_extension(filename,"sur");
    if ((infile = fopen(filename,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open file '%s'",filename);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Read comments and lattice parameters */

    fgets(INLINE,INLLEN,infile);
    sprintf(STRING,"Title: %s",INLINE);
    type_line(STRING);
    fgets(INLINE,INLLEN,infile);
    sscanf(INLINE,"%f%f%f%f%f%f",&dlats[0],&dlats[1],&dlats[2],&dlats[3],
	   &dlats[4],&dlats[5]);
    for (i = 3; i < 6; i++) dlats[i] /= RAD;

    /* If a bulk file has already been read in, check the lattice
       parameters for consistency */

    if (NBULK == 0)
	{
	for (i = 0; i < 6; i++) DLAT[i] = dlats[i];
	calc_rlat(DLAT,RLAT);
	}
    else
	{
	for (i = 0; i < 6; i++)
	    {
	    if (fabs(DLAT[i]-dlats[i]) > 1e-4)
		{
		errtype("ERROR, lattice parameters different from bulk");
		return;
		}
	    }
	}

    /* Find out from first data line, whether a serial number for
       Debye-Waller parameters is given */

    fgets(INLINE,INLLEN,infile);
#if defined(__unix__) || defined(__APPLE__)
    sscanf(INLINE,"%2s%f%f%f%d%d",
	   elsurf[0],&XS[0],&YS[0],&ZS[0],&NDWS[0],&NDWS2[0]);
#else
    sscanf(INLINE,"%[ ]%2s%f%f%f%d%d",
	   dummy,elsurf[0],&XS[0],&YS[0],&ZS[0],&NDWS[0],&NDWS2[0]);
#endif /* defined(__unix) || defined(__APPLE__) */
    if (NDWS2[0] != 0)
	{
	dw_column = dw2_column = TRUE;
	}
    else if (NDWS[0] != 0)
	{
	dw2_column = FALSE;
	dw_column = TRUE;
	}
    else
	{
	dw2_column = dw_column = FALSE;
	}

    /* Read the rest of the atoms */

    NSURF = 1;
    while ((fgets(INLINE,INLLEN,infile) != NULL) /*&& (NSURF < MAXATOMS)*/)
	{
	if (dw2_column)
	    {
#if defined(__APPLE__) || defined(__unix)
	    sscanf(INLINE,"%2s%f%f%f%d%d",elsurf[NSURF],
		   &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF],
		   &NDWS2[NSURF]);
#else
	    sscanf(INLINE,"%[ ]%2s%f%f%f%d%d",dummy,elsurf[NSURF],
		   &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF],
		   &NDWS2[NSURF]);
#endif /* defined(__APPLE__) || defined(__unix) */
	    }
	if (dw_column)
	    {
#if defined(__APPLE__) || defined(__unix)
	    sscanf(INLINE,"%2s%f%f%f%d",elsurf[NSURF],
		   &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF]);
#else
	    sscanf(INLINE,"%[ ]%2s%f%f%f%d",dummy,elsurf[NSURF],
		   &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF]);
#endif /* defined(__APPLE__) || defined(__unix) */
	    }
	else
	    {
#if defined(__APPLE__) || defined(__unix)
	    sscanf(INLINE,"%2s%f%f%f",elsurf[NSURF],
		   &XS[NSURF],&YS[NSURF],&ZS[NSURF]);
#else
	    sscanf(INLINE,"%[ ]%2s%f%f%f",dummy,elsurf[NSURF],
		   &XS[NSURF],&YS[NSURF],&ZS[NSURF]);
#endif /* defined(__APPLE__) || defined(__unix) */
	    }
	NSURF++;
	if (NSURF == MAXATOMS)
	    {
	    errtype("ERROR, maximum number of model atoms exceeded");
	    break;
	    }
	}
    fclose(infile);
    NSURFTOT = NSURF;

    /* Find the different element types and store them in ELEMENT */

    if (NTYPES == 0)
	{
	NTYPES = 1;
	sprintf(ELEMENT[0],"%s",elsurf[0]);
	}
    for (i = 0; i < NSURF; i++)
	{
	newtype = TRUE;
	for (j = 0; j < NTYPES; j++)
	    {
	    if (el_equal(elsurf[i],ELEMENT[j])) newtype = FALSE;
	    }
	if (newtype)
	    {
	    if (NTYPES == MAXTYPES)
		{
		errtype("ERROR, too many atom types in unit cell");
		return;
		}
	    NTYPES++;
	    sprintf(ELEMENT[NTYPES-1],"%s",elsurf[i]);
	    }
	}

    /* Assign a type number to all atoms in the model */

    for (i = 0; i < NSURF; i++)
	{
	for (j = 0; j < NTYPES; j++)
	    {
	    if (el_equal(elsurf[i],ELEMENT[j])) TS[i] = j;
	    }
	}

    /* Store the coefficients for the computation of the atomic scattering
       factors for all the different elements in F_COEFF */

    for (i = 0; i < NTYPES; i++)
	{
	get_coeff(ELEMENT[i],F_COEFF[i]);
	}

    /* Copy atom positions to fitted positions */

    for (i = 0; i < NSURF; i++)
	{
	XSFIT[i] = XS[i];
	YSFIT[i] = YS[i];
	ZSFIT[i] = ZS[i];
	}

    /* Find highest serial number of Debye-Waller, if read in */

    if (dw_column)
	{
	for (i = 0; i < NSURF; i++)
	    {
	    if (NDWS[i] > NDWTOT)
		{
		if (NDWS[i] > MAXPAR)
		    {
		    sprintf(STRING,
			    "ERROR, total # Debye-Wallers should be < %1d",MAXPAR);
		    errtype(STRING);
		    }
		else
		    {
		    NDWTOT = NDWS[i];
		    }
		}
	    }
	}
    if (dw2_column)
	{
	for (i = 0; i < NSURF; i++)
	    {
	    if (NDWS2[i] > NDWTOT2)
		{
		if (NDWS2[i] > MAXPAR)
		    {
		    sprintf(STRING,
			    "ERROR, total # Debye-Wallers should be < %1d",MAXPAR);
		    errtype(STRING);
		    }
		else
		    {
		    NDWTOT2 = NDWS2[i];
		    }
		}
	    }
	}

    }

/***************************************************************************/
void    read_type(void)
/***************************************************************************/

    /*
    Get type of file to be read in.
    */

    {

    /* define read_type_menu */

#define read_type_length 7       /* number of commands in read_type menu */

    static struct   MENU read_type_menu[read_type_length] =
	{
	"data",     1,  1,  "Structure factor data",
	"surface",  3,  2,  "Coordinates of surface atoms",
	"bulk",     1,  3,  "Coordinates of bulk atoms",
	"fit",      1,  4,  "File with fit model of surface atoms",
	"parameters",2, 5,  "File with fit parameters (macro)",
	"help",     1,  20, "Display menu",
	"return",   1,  21, "Return to main menu"
	};

    int     stop = FALSE;
    char    token[100];

    while (!stop)
	{
	if (!get_token(token,"File type: ")) break;
	switch (cmnd_match(token,read_type_menu,read_type_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		read_data();
		return;
	    case 2:
		read_surf();
		return;
	    case 3:
		read_bulk();
		return;
	    case 4:
		read_fit();
		return;
	    case 5:
		read_par();
		return;
	    case 20:
		list_menu("INPUT FILE TYPES",read_type_menu,read_type_length);
		break;
	    case 21:
		stop = TRUE;
	    }
	}
    }

