/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                         read.c                                $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: read.c,v 1.9 2004/01/19 16:55:03 wilcke Exp $

****************************************************************************/

/*
Update 19/01/2004 R. Wilcke (wilcke@esrf.fr) 
                  removed declarations of fopen() (is declared in "stdio.h").
Update 08/06/2001 E. Vlieg (vlieg@sci.kun.nl)
		  Add variable FILETYPE to store the type of file most
		  recently read in.
Update 15/12/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Add DUBBLE SPEC file type.
Update 06/10/2000 E. Vlieg (vlieg@sci.kun.nl)
                  Add (temporary) patch with label "femke" to read specular
                  data with correct indices. Activated if scan# negative.
Update 25/09/2000 R. Wilcke (wilcke@esrf.fr)
                  read_spec(): make the number of format specifiers in the
                  "sscanf()" calls agree with the number of variables to be read
                  (3 instances).
Update 13/09/2000 R. Wilcke (wilcke@esrf.fr)
                  limit the length of the strings printed (with sprintf()) into
                  field TITLE of structure DATA;
                  limit the length of the strings printed (with sprintf()) into
                  fields DATATITLE and DATAINFO.
Update 07/06/2000 R. Wilcke (wilcke@esrf.fr)
                  replaced everywhere the buffer size for filenames by the ISO-C
                  defined macro FILENAME_MAX.
Update 15/03/2000 R. Wilcke (wilcke@esrf.fr)
                  read_ftascom(): replace printf() calls by type_line().
*/

/******************************************************************************/

/*
	This version: 22-Aug-96

	reads :

	- SRS data files (old and new format)
	- SPEC data files (ESRF ID3 / ID10 / ID 32 format)
	- Brookhaven SUPER files
	- HASYLAB tascom files in single column format
*/

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define EXT extern
#include "ana.h"
#undef EXT
#include <ctype.h>

/***************************************************************************/
void    read_column(void)
/***************************************************************************/

	 /*
	 Read file of 'column' type:
	- HEADER_LINES header lines
	- data lines with arbirary amount of columns, with
		XCOL         the x-values
		YCOL         the y-values
		ECOL         the errors in y (no error if 0).
	 */

    {

    char    filename[FILENAME_MAX];
    FILE    *infile;
    int     i,in_spec,items,npoints;
    int     end_of_file,error_column;
    float   dummy[DATACOLUMNS];
    int     maxcolumn;

    /* Get filename and open file, return if not successful */

    get_token(filename,"Give filename (.dat): ");
    add_extension(filename,"dat");
    if ((infile = fopen(filename,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open file '%s'",filename);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Get spectrum number where data has to be stored */

    get_inspec(&in_spec);
    READSPEC = in_spec;

    /* Skip header lines */

    for (i = 0; i < HEADER_LINES; i++)
	{
	if (fgets(INLINE,INLLEN,infile) == NULL)
	    {
	    errtype("ERROR, less than HEADER_LINES lines in file");
	    clear_command();
	    return;
	    }
	}

    /* Find maximum column number to be read in */

    maxcolumn = XCOL;
    if (YCOL > maxcolumn) maxcolumn = YCOL;
    if (ECOL > maxcolumn) maxcolumn = ECOL;
    if (ECOL != 0)
	error_column = TRUE;
    else
	error_column = FALSE;

    /* Read in spectrum */

    npoints = 0;
    end_of_file = FALSE;
    while ((fgets(INLINE,INLLEN,infile) != NULL) && (!end_of_file))
	{
	/* Check whether enough data is read in */
	if (sscanf(INLINE,"%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f",
		   &dummy[0],&dummy[1],&dummy[2],&dummy[3],&dummy[4],&dummy[5],
		   &dummy[6],&dummy[7],&dummy[8],&dummy[9],&dummy[10],&dummy[11])
	    >= maxcolumn)
	    {
	    if (XCOL <= 0) /* Ennumerate x-values */
		DATA[in_spec].X[npoints] = (float) npoints+1;
	    else
		DATA[in_spec].X[npoints] = dummy[XCOL-1];
	    DATA[in_spec].Y[npoints] = dummy[YCOL-1];
	    if (error_column)
		DATA[in_spec].E[npoints] = dummy[ECOL-1];
	    else
		DATA[in_spec].E[npoints] = 1;
	    npoints++;
	    }
	else
	    {
	    end_of_file = TRUE;
	    }
	}
    fclose(infile);
    if (npoints == 0)
	{
	errtype("No data in spectrum");
	clear_command();
	return;
	}
    DATA[in_spec].NPNT = npoints;
    /* Find miniumum and maximum in spectrum */

    DATA[in_spec].MINY = DATA[in_spec].Y[0];
    DATA[in_spec].MAXY = DATA[in_spec].Y[0];
    for (i = 1; i < npoints; i++)
	{
	if (DATA[in_spec].Y[i] < DATA[in_spec].MINY)
	    DATA[in_spec].MINY = DATA[in_spec].Y[i];
	if (DATA[in_spec].Y[i] > DATA[in_spec].MAXY)
	    DATA[in_spec].MAXY = DATA[in_spec].Y[i];
	}

    /* Give spectrum a title */

    sprintf(DATA[in_spec].TITLE,"%.*s",TITLE_LENGTH,filename);
    sprintf(DATATITLE,"%.*s",TITINFLN-1,filename);

    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[in_spec].X[0],DATA[in_spec].X[DATA[in_spec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }

/***************************************************************************/
void    read_ftascom()
/***************************************************************************/

	 /*
	 Read HASYLAB tascom file in single column format.
	 We assume a z-axis mode.
	 */

    {

    char    filename[FILENAME_MAX],dummy_name[FILENAME_MAX],date[50],time[50];
    FILE    *infile;
    int     i,in_spec,items,npoints;
    int     end_of_file,error_column;
    float   dummy[DATACOLUMNS],dummyval;
    float   monitor[MAX_DATA];
    int     done,ncolumns,nheaders,xcol_save,timescan;
    char    column_name[20],name[20][20];
    float   *x,*y,*e;
    char    *spos;
    char    lastline[INLLEN];
    int     has_read_beta;
    int     digits;

    /* Get filename and open file, return if not successful */

    SCANNR = get_int(1,"Scan number: ");

    for (i=strlen(DATAFILE)-1, digits=5; i>=0; i--, digits--)
	{
	if (!isdigit(DATAFILE[i])) break;
	}

    sprintf(filename,"%s%0*d",DATAFILE,digits,SCANNR);
    add_extension(filename,"dat");
    if ((infile = fopen(filename,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open file '%s'",filename);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Extract a few things from the header */

    /* First line contains # header lines + #columns */

    fgets(INLINE,INLLEN,infile);
    sscanf(INLINE,"%d %d",&nheaders,&ncolumns);

    /* Now get date, time from second line */

    fgets(INLINE,INLLEN,infile);
    sscanf(INLINE," %s %s",&date,&time);
    sprintf(DATAINFO,"%.*s Date: %s time: %s",TITINFLN-40,filename,date,time);
    sprintf(STRING,"Date: %s time: %s\n",date,time);
    type_line(STRING);

    /* skip blank line */

    fgets(INLINE,INLLEN,infile);

    /* Rest may contain something interesting */

    ALPHA = BETAIN = DELTA = GAMMA = 0.;
    for (i = 0; i < nheaders-ncolumns-2; i++)
	{
	fgets(INLINE,INLLEN,infile);
	sscanf(INLINE,"%s = %f", &dummy_name, &dummyval);
	strupr(dummy_name);
	if (strcmp(dummy_name,"FTR") == 0) GAMMA = dummyval;
	if (strcmp(dummy_name,"ALF") == 0) ALPHA = dummyval;
	if (strcmp(dummy_name,"2TS") == 0) DELTA = dummyval;
	if (strcmp(dummy_name,"H") == 0) HMIL = dummyval;
	if (strcmp(dummy_name,"K") == 0) KMIL = dummyval;
	if (strcmp(dummy_name,"L") == 0) LMIL = dummyval;
	}
    BETAIN = ALPHA; /* This is, of course, only correct for z-axis mode */
    sprintf(STRING, "ALPHA=%.3f,  DELTA=%.3f, GAMMA=%.3f\n",
	    ALPHA, DELTA, GAMMA);
    type_line(STRING);

    /* Get the 'column' names */

    for (i = 0; i < ncolumns; i++)
	{
	fgets(INLINE,INLLEN,infile);
	sscanf(INLINE," %s ",name[i]);
	strlwr(name[i]);
	}

    sprintf(DATATITLE,"  ");  /* No comments line here in HASYLAB */

    xcol_save = XCOL;

    if (XCOL < 0) /* Get name of x */
	{
	done = FALSE;
	while (!done)
	    {
	    if (!get_token(column_name,
			   "Give column name (type <return> for options): "))
		{
		for (i = 0; i < ncolumns; i++)
		    {
		    type_line(name[i]);
		    }
		type_line("\n");
		}
	    else
		{
		strlwr(column_name);
		for (i = 0; i < ncolumns; i++)
		    {
		    if (strcmp(name[i],column_name) == 0)
			{
			XCOL = i+1;
			done = TRUE;
			sprintf(XCOLNAME,"%s",name[i]);
			}
		    }
		if (XCOL < 0)
		    {
		    errtype("ERROR, no matching column name found");
		    clear_command();
		    fclose(infile);
		    XCOL = xcol_save;
		    return;
		    }
		}
	    }
	}


    else if (XCOL > 0) /* Get column name of x */
	{
	sprintf(XCOLNAME,"%s",name[XCOL-1]);
	}
    else
	{
	sprintf(XCOLNAME,"x");
	}

    if (XCOL != 0) /* Put x name in graph */
	{
	sprintf(STRING,"x title %s break",XCOLNAME);
	put_command(STRING);
	plotxye(x,y,e,1);
	}

    /* Get column number of y values */

    for (i = 0; i < ncolumns; i++)
	{
	if (strcmp(name[i],YCOLNAME) == 0) YCOL = i+1;
	}

    /* Get column number of monitor values */

    if (MCOL != 0)
	{
	for (i = 0; i < ncolumns; i++)
	    {
	    if (strcmp(name[i],"mon") == 0) MCOL = i+1;
	    }
	}

    /* Get spectrum number where data has to be stored */

    get_inspec(&in_spec);
    READSPEC = in_spec;

    /* Read in spectrum */

    npoints = 0;
    end_of_file = FALSE;
    while (!end_of_file)
	{
	for (i = 0; i < ncolumns; i++)
	    {
	    if (fgets(INLINE,INLLEN,infile) == NULL)
		{
		end_of_file = TRUE;
		break;
		}
	    sscanf(INLINE,"%f",&dummy[i]);
	    }


	if (XCOL == 0) /* Ennumerate x-values */
	    {
	    DATA[in_spec].X[npoints] = (float) npoints+1;
	    }
	else
	    {
	    DATA[in_spec].X[npoints] = dummy[XCOL-1];
	    }
	DATA[in_spec].Y[npoints] = dummy[YCOL-1];
	if (MCOL != 0) monitor[npoints] = dummy[MCOL-1];
	npoints++;
	if (npoints == MAX_DATA)
	    {
	    errtype("ERROR, maximum number of data points exceeded");
	    break;
	    }
	} /* end of while */
    fclose(infile);

    if (npoints == 0)
	{
	errtype("No data in spectrum");
	clear_command();
	XCOL = xcol_save;
	return;
	}

    DATA[in_spec].NPNT = npoints;

    XCOL = xcol_save;

    /* Calculate statistical error */

    for (i = 0; i < npoints; i++) DATA[in_spec].E[i] = sqrt(DATA[in_spec].Y[i]);

    /* if required, normalize on monitor signal */

    if (MCOL != 0)
	{
	for (i = 0; i < npoints; i++)
	    {
	    if (monitor[i] < 1) monitor[i] = 1;
	    DATA[in_spec].Y[i] *= MONITOR_NORM/monitor[i];
	    DATA[in_spec].E[i] *= MONITOR_NORM/monitor[i];
	    }
	}

    /* Find miniumum and maximum in spectrum */

    DATA[in_spec].MINY = DATA[in_spec].Y[0];
    DATA[in_spec].MAXY = DATA[in_spec].Y[0];
    for (i = 1; i < npoints; i++)
	{
	if (DATA[in_spec].Y[i] < DATA[in_spec].MINY)
	    DATA[in_spec].MINY = DATA[in_spec].Y[i];
	if (DATA[in_spec].Y[i] > DATA[in_spec].MAXY)
	    DATA[in_spec].MAXY = DATA[in_spec].Y[i];
	}

    /* Give spectrum a title */

    sprintf(DATA[in_spec].TITLE,"%.*s",TITLE_LENGTH,filename);

    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[in_spec].X[0],DATA[in_spec].X[DATA[in_spec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }

/***************************************************************************/
void    read_spec(void)
/***************************************************************************/

	 /*
	Read scan from SPEC data file.
	 */

    {

    char    filename[FILENAME_MAX],par[3];
    FILE    *infile;
    int     in_spec;
    int     found,end_of_scan;
    char    hatch,date[40];
    int     i,scan;
    float   monitor[MAX_DATA];
    float   dummy[DATACOLUMNS];
    int     maxcolumn,npoints;
    float   norm_corr;
    int     done,ncolumns,xcol_save;
    static int  hcol,kcol,lcol;
    char    column_name[40],name[DATACOLUMNS][40];
    char    extension[4];
    float   *x,*y,*e;
    char    *strndx;
    float	 omega,alfa;

    /* femke */
    int	femke;

    /* Get scan number to be read in */

    sprintf(STRING,"Scan number to be read in [%1d]: ",SCANNR);
    SCANNR = get_int(SCANNR,STRING);

    /*femke*/
    if (SCANNR < 1)
	{
	SCANNR = -SCANNR;
	femke = TRUE;
	}
    else
	femke = FALSE;

    /* Open SPEC file. */

    if ((infile = fopen(DATAFILE,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open input file '%s'",DATAFILE);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Search for correct scan number */

    found = FALSE;
    while (!found)
	{
	if (fgets(INLINE,INLLEN,infile) == NULL)
	    {
	    sprintf(STRING,"Scan not found in file '%s'",DATAFILE);
	    errtype(STRING);
	    clear_command();
	    fclose(infile);
	    return;
	    }

	sscanf(INLINE,"%3c %d",par,&scan);
	if ((par[0]=='#') && (par[1] == 'S') && (scan == SCANNR))
	    {
	    found = TRUE;
	    }
	}

    /* We have found the correct scan number, now get the rest of the
       information */

    /* Get the date and time */

    found = FALSE;
    while (!found)
	{
	if (fgets(INLINE,INLLEN,infile) == NULL)
	    {
	    sprintf(STRING,"Error, emtpy scan");
	    errtype(STRING);
	    clear_command();
	    fclose(infile);
	    return;
	    }

	if ((INLINE[0]=='#') && (INLINE[1] == 'D'))
	    {
	    sscanf(INLINE,"%3c %24c",par,date);
	    found = TRUE;
	    }
	}

    date[24] = 0;

    sprintf(STRING,"%s\n",date);
    type_line(STRING);

    /* Get angles needed for computation of Lorentz factor,
       polarization factor and area correction */

    DELTA = 0;
    BETAIN = 0;
    ALPHA = 0;

    found = FALSE;
    while (!found)
	{
	if (fgets(INLINE,INLLEN,infile) == NULL)
	    {
	    sprintf(STRING,"Error, emtpy scan");
	    errtype(STRING);
	    clear_command();
	    fclose(infile);
	    return;
	    }

	if ((INLINE[0]=='#') && (INLINE[1] == 'P') && (INLINE[2] == '0'))
	    {

	    sscanf(INLINE,"%3c%f%f%f%f%f%f%f", par, &dummy[0],
		   &dummy[1],&dummy[2],&dummy[3],&dummy[4],&dummy[5],
		   &dummy[6]);

	    found = TRUE;
	    }
	}

    if (SPECTYPE == ID03)
	{
	DELTA = dummy[0];
	ALPHA = dummy[4];
	BETAIN = dummy[4]; /* Z-axis mode only !! */
	GAMMA = dummy[5];
	}
    if (SPECTYPE == ID32)
	{
	DELTA = dummy[1];
	ALPHA = dummy[3];
	GAMMA = dummy[0];
	BETAIN = dummy[3];
	}
    if (SPECTYPE == DUBBLE)
	{
	DELTA = dummy[1];
	ALPHA = dummy[2];
	GAMMA = dummy[0];
	BETAIN = dummy[2];
	}

    /* femke: trick to get special specular data read in with correct
       indices */

    if (femke)
	{
	ALPHA = (ALPHA+GAMMA)/2;
	GAMMA = ALPHA;
	}

    sprintf(STRING, "DELTA=%.3f,  ALPHA=%.3f, GAMMA=%.3f, BETA_IN=%.3f\n",
	    DELTA, ALPHA, GAMMA, BETAIN);
    type_line(STRING);

    /* Read line with column names. Skip this line if the column number
       of the x-values if given. If XCOL < 0, ask for name of column. */

    found = FALSE;
    while (!found)
	{
	if (fgets(INLINE,INLLEN,infile) == NULL)
	    {
	    sprintf(STRING,"Error, emtpy scan");
	    errtype(STRING);
	    clear_command();
	    fclose(infile);
	    return;
	    }

	if ((INLINE[0]=='#') && (INLINE[1] == 'L'))
	    {
	    found = TRUE;

	    /* SPEC column names are separated by double spaces,
	       thus replace all single spaces (that are part of column
	       name) by underscore */

	    for (i = 3; i < strlen(INLINE)-1; i++)
		{
		if ((INLINE[i] == ' ') && (INLINE[i-1] != ' ')
		    && (INLINE[i+1] != ' '))
		    INLINE[i] = '_';
		}

	    /* read column headers */

	    ncolumns = sscanf(INLINE,
			      "%3c %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s",
			      par, name[0],name[1],name[2],name[3],name[4],name[5],
			      name[6],name[7],name[8],name[9],name[10],name[11],
			      name[12],name[13],name[14],name[15],name[16],name[17],
			      name[18]);
	    ncolumns -= 1; /* the first item doesn't count */
	    if (ncolumns >= DATACOLUMNS)
		{
		sprintf(STRING,
			"Too many data columns, only first %d used",
			DATACOLUMNS);
		errtype(STRING);
		}
	    }
	}

    xcol_save = XCOL;
    hcol = 0;
    kcol = 0;
    lcol = 0;
    if (XCOL < 0)
	{
	done = FALSE;
	while (!done)
	    {
	    if (!get_token(column_name,
			   "Give column name (type <return> for options): "))
		{
		type_line(INLINE);
		}
	    else
		{
		string_tolower(column_name);
		for (i = 0; i < ncolumns; i++)
		    {
		    string_tolower(name[i]);
		    if (strcmp(name[i],column_name) == 0)
			{
			XCOL = i+1;
			done = TRUE;
			sprintf(XCOLNAME,"%s",name[i]);
			}
		    }
		if (XCOL < 0)
		    {
		    errtype("ERROR, no matching column name found");
		    clear_command();
		    fclose(infile);
		    return;
		    }
		}
	    }

	/* Assign x-column name to title of x-axis in plot */

	sprintf(STRING,"x title %s break",XCOLNAME);
	put_command(STRING);
	plotxye(x,y,e,1);
	}

    for (i = 0; i < ncolumns; i++)
	{
	string_tolower(name[i]);
	if (strcmp(name[i],YCOLNAME) == 0) YCOL = i+1;
	if (strcmp(name[i],"h") == 0) hcol = i+1;
	if (strcmp(name[i],"k") == 0) kcol = i+1;
	if (strcmp(name[i],"l") == 0) lcol = i+1;
	if (MCOL != 0)
	    {
	    if ( !strcmp(name[i],"monitor") ||
		 !strcmp(name[i],"monitor1") ||
		 !strcmp(name[i],"ion_ch")
		) MCOL = i+1;
	    }
	}

    /* Get spectrum number where data has to be stored */

    get_inspec(&in_spec);
    READSPEC = in_spec;


    /* Find maximum column number to be read in */

    if (XCOL > YCOL)
	maxcolumn = XCOL;
    else
	maxcolumn = YCOL;

    /* Read data and compute statistical error in y-values */

    npoints = 0;
    HMIL = 0;
    KMIL = 0;
    LMIL = 0;
    end_of_scan = FALSE;
    while ((fgets(INLINE,INLLEN,infile) != NULL) && (!end_of_scan))
	{
	/* Check whether enough data is read in */
	if (sscanf(INLINE,"%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f",
		   &dummy[0],&dummy[1],&dummy[2],&dummy[3],&dummy[4],&dummy[5],
		   &dummy[6],&dummy[7],&dummy[8],&dummy[9],&dummy[10],&dummy[11],
		   &dummy[12],&dummy[13],&dummy[14],&dummy[15],&dummy[16],
		   &dummy[17],&dummy[18])
	    >= maxcolumn)
	    {
	    if (XCOL == 0) /* Ennumerate x-values */
		DATA[in_spec].X[npoints] = (float) npoints+1;
	    else
		DATA[in_spec].X[npoints] = dummy[XCOL-1];
	    DATA[in_spec].Y[npoints] = dummy[YCOL-1];
	    if (MCOL > 0)
		monitor[npoints] = dummy[MCOL-1];
	    else
		monitor[npoints] = MONITOR_NORM;
	    HMIL += dummy[hcol-1];
	    KMIL += dummy[kcol-1];
	    LMIL += dummy[lcol-1];
	    if (dummy[YCOL-1] < 1e-10)
		{
		DATA[in_spec].E[npoints] = 1;
		}
	    else
		{
		DATA[in_spec].E[npoints] =  sqrt(dummy[YCOL-1]);
		}
	    npoints++;
	    if (npoints >= MAX_DATA)
		{
		errtype("Maximum number of data points exceeded");
		break;
		}
	    }
	else
	    {
	    end_of_scan = TRUE;
	    }
	}
    fclose(infile);

    XCOL = xcol_save;

    if (npoints == 0)
	{
	errtype("No data in spectrum");
	clear_command();
	return;
	}
    DATA[in_spec].NPNT = npoints;

    /* Make h, k, l and two-theta the average value */

    HMIL  = HMIL/npoints;
    KMIL  = KMIL/npoints;
    LMIL = LMIL/npoints;

    /* femke */

    if (femke)
	{
	HMIL = 0.;
	KMIL = 0.;
	LMIL = 20.369*sin(GAMMA/RAD);
	}

    sprintf(STRING,"(hkl) = (%4.2f,%4.2f,%4.2f)\n",HMIL,KMIL,LMIL);
    type_line(STRING);

    /* If requested, normalize data */

    if (MCOL != 0)
	{
	for (i = 0; i < npoints; i++)
	    {
	    if (monitor[i] < 1) monitor[i] = 1;
	    DATA[in_spec].Y[i] *= MONITOR_NORM/monitor[i];
	    DATA[in_spec].E[i] *= MONITOR_NORM/monitor[i];
	    }
	}

    /* Find miniumum and maximum in spectrum */

    DATA[in_spec].MINY = DATA[in_spec].Y[0];
    DATA[in_spec].MAXY = DATA[in_spec].Y[0];
    for (i = 1; i < npoints; i++)
	{
	if (DATA[in_spec].Y[i] < DATA[in_spec].MINY)
	    DATA[in_spec].MINY = DATA[in_spec].Y[i];
	if (DATA[in_spec].Y[i] > DATA[in_spec].MAXY)
	    DATA[in_spec].MAXY = DATA[in_spec].Y[i];
	}

    /* Give spectrum a title */

    sprintf(DATA[in_spec].TITLE,"%.*s #%1d",TITLE_LENGTH-7,DATAFILE,SCANNR);

    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[in_spec].X[0],DATA[in_spec].X[DATA[in_spec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }

/***************************************************************************/
void    read_srs(int version)
/***************************************************************************/

	 /*
	 Read Daresbury SRS file.
	 Since the file format is changing every now and then, the actions
	 differ somewhat according to the version flag
	 version 1   before 92
	 version 2   starting 92
	 */

    {

    char    filename[FILENAME_MAX],dummy_name[FILENAME_MAX],datetime[50];
    FILE    *infile;
    int     i,in_spec,items,npoints;
    int     end_of_file,error_column;
    float   dummy[DATACOLUMNS];
    float   monitor[MAX_DATA];
    static int     maxcolumn;
    int     done,ncolumns,xcol_save,timescan;
    char    column_name[20],name[20][20];
    float   *x,*y,*e;
    char    *spos;
    char    lastline[INLLEN];
    int     has_read_beta;
    int     digits;

    /* Get filename and open file, return if not successful */

    SCANNR = get_int(1,"Scan number: ");

    for (i=strlen(DATAFILE)-1, digits=5; i>=0; i--, digits--)
	{
	if (!isdigit(DATAFILE[i])) break;
	}

    sprintf(filename,"%s%0*d",DATAFILE,digits,SCANNR);
    add_extension(filename,"dat");
    if ((infile = fopen(filename,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open file '%s'",filename);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Extract a few things from the header */

    ALPHA = BETAIN = DELTA = 0.;
    GAMMA = GAMMA_FIX; /* Can't be found in data file */
    fgets(INLINE,INLLEN,infile);
    fgets(INLINE,INLLEN,infile);
    sprintf(DATAINFO,"%.*s Date: %.2s-%.2s-%.2s time: %.2s:%.2s:%.2s",
	    TITINFLN-40,filename,&INLINE[25],&INLINE[23],&INLINE[21],&INLINE[35],
	    &INLINE[37],&INLINE[39]);
    sprintf(datetime,"Date: %.2s-%.2s-%.2s time: %.2s:%.2s:%.2s",
	    &INLINE[25],&INLINE[23],&INLINE[21],&INLINE[35],&INLINE[37],
	    &INLINE[39]);
    fgets(INLINE,INLLEN,infile);
    fgets(INLINE,INLLEN,infile);
    sscanf(INLINE,"%8s%f %f %f %f",dummy_name,&HMIL,&KMIL,&LMIL,&BETAIN);
    if ((strstr(INLINE, "  ") != INLINE+9) /* first two characters must be spaces */
	||(BETAIN == 0.) )                   /* beta shouldn't be zero ... */
	{                                 /* for correct beta read from header */
	BETAIN = 0.;
	has_read_beta = FALSE;
	}
    else
	{
	has_read_beta = TRUE;
	}

    sprintf(DATATITLE,"%.40s",&INLINE[9]);

    if (version == 1)   /* that's before '92 */
	{
	for (i = 0; i < 11; i++) fgets(INLINE,INLLEN,infile);
	fgets(INLINE,INLLEN,infile);
	sscanf(INLINE,"%s%f",dummy_name,&DELTA);
	for (i = 0; i < HEADER_LINES-16; i++) fgets(INLINE,INLLEN,infile);

	/* Since alpha and beta are not necessarily correct, ignore them
	   for the time being */

	ALPHA = BETAIN = 0;

	}

    if (version == 2)   /* this is after Gordon's great rewrite */
	{

	/* We're looking in for the magic keywords DELTA and BETA_IN
	   somewhere in the header ... */

	ALPHA = DELTA = 0.;  /* BETAIN should still be zero if not in header */


	for(;;) /* loop until we find the start of data marker */
	    {

	    fgets(INLINE, INLLEN, infile); /* read a line from file */

	    strupr(INLINE);       /* all char's -> uppercase */

	    if (strstr(INLINE, "--------------------") != NULL)
		{
		break; /* go and read data now */
		}

	    if ((spos=strstr(INLINE, "ALPHA")) != NULL)
		{
		sscanf(spos+5, "%f", &ALPHA);  /* get alpha if you need it */
		}
	    if ((spos=strstr(INLINE, "DELTA")) != NULL)
		{
		sscanf(spos+5, "%f", &DELTA);   /* get twotheta = delta */
		}
	    if (((spos=strstr(INLINE, "BETA_IN")) != NULL) && (has_read_beta == FALSE) )
		{
		sscanf(spos+7, "%f", &BETAIN);   /* get incoming angle */
		}
	    if ((spos=strstr(INLINE, "DETECTOR")) != NULL)
		{
		strcpy(lastline, INLINE);  /* save this line for the headers */
		}
	    } /* end of for loop */

	sprintf(STRING, "DELTA=%.3f,  ALPHA=%.3f,  BETA_IN=%.3f\n",
		DELTA, ALPHA, BETAIN);
	type_line(STRING);

	/* look for column headers */

	if (version == 1)
	    {

	    rewind(infile);
	    for (i = 0; i < HEADER_LINES-1; i++) fgets(INLINE,INLLEN,infile);

	    }

	if (version == 2)
	    {

	    strcpy(INLINE, lastline);   /* this still has the headers */

	    }

	ncolumns = sscanf(INLINE,
			  " %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s",
			  name[0],name[1],name[2],name[3],name[4],name[5],name[6],
			  name[7],name[8],name[9],name[10],name[11],name[12]);
	for (i = 0; i < ncolumns; i++) string_tolower(name[i]);
	xcol_save = XCOL;
	if (XCOL < 0) /* Get name of x */
	    {
	    done = FALSE;
	    while (!done)
		{
		if (!get_token(column_name,
			       "Give column name (type <return> for options): "))
		    {
		    type_line(INLINE);
		    }
		else
		    {
		    string_tolower(column_name);
		    for (i = 0; i < ncolumns; i++)
			{
			if (strcmp(name[i],column_name) == 0)
			    {
			    XCOL = i+1;
			    done = TRUE;
			    sprintf(XCOLNAME,"%s",name[i]);
			    }
			}
		    if (XCOL < 0)
			{
			errtype("ERROR, no matching column name found");
			clear_command();
			fclose(infile);
			XCOL = xcol_save;
			return;
			}
		    }
		}
	    }
	else if (XCOL > 0) /* Get column name of x */
	    {
	    sprintf(XCOLNAME,"%s",name[XCOL-1]);
	    }
	else
	    {
	    sprintf(XCOLNAME,"x");
	    }

	if (XCOL != 0)
	    {
	    sprintf(STRING,"x title %s break",XCOLNAME);
	    put_command(STRING);
	    plotxye(x,y,e,1);
	    }

	/* Get column number of y values */

	for (i = 0; i < ncolumns; i++)
	    {
	    if (strcmp(name[i],YCOLNAME) == 0) YCOL = i+1;
	    }

	/* Get column number of monitor values */

	if (MCOL != 0)
	    {
	    for (i = 0; i < ncolumns; i++)
		{
		if (strcmp(name[i],"monitor") == 0) MCOL = i+1;
		}
	    }
/* Don't know why this has to be done (SdV, 16-7-96) (old data files?):
   fgets(INLINE,INLLEN,infile);
   BETAIN = 0;*/
	}

    if (strcmp(XCOLNAME,"time") == 0)
	timescan = TRUE;
    else
	timescan = FALSE;

    sprintf(STRING,"%s\n",datetime);
    type_line(STRING);
    sprintf(STRING,"Title: %s\n",DATATITLE);
    type_line(STRING);

    /* Get spectrum number where data has to be stored */

    get_inspec(&in_spec);
    READSPEC = in_spec;

    /* Find maximum column number to be read in */

    maxcolumn = XCOL;
    if (YCOL > maxcolumn) maxcolumn = YCOL;
    if (MCOL > maxcolumn) maxcolumn = MCOL;

    /* Read in spectrum */

    npoints = 0;
    end_of_file = FALSE;
    while ((fgets(INLINE,INLLEN,infile) != NULL) && (!end_of_file))
	{
	/* Check whether enough data is read in */
	if (sscanf(INLINE,"%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f",
		   &dummy[0],&dummy[1],&dummy[2],&dummy[3],&dummy[4],&dummy[5],
		   &dummy[6],&dummy[7],&dummy[8],&dummy[9],&dummy[10],&dummy[11])
	    >= maxcolumn)
	    {
	    if (XCOL == 0) /* Ennumerate x-values */
		{
		DATA[in_spec].X[npoints] = (float) npoints+1;
		}
	    else

		{
		if (npoints && timescan == TRUE) /* cumulate X column if */
		    {                            /* in time scan mode    */
		    DATA[in_spec].X[npoints] = DATA[in_spec].X[npoints-1] +
			dummy[XCOL-1];
		    }
		else
		    {
		    DATA[in_spec].X[npoints] = dummy[XCOL-1];
		    }
		}
	    DATA[in_spec].Y[npoints] = dummy[YCOL-1];
	    if (MCOL != 0) monitor[npoints] = dummy[MCOL-1];
	    npoints++;
	    if (npoints == MAX_DATA)
		{
		errtype("ERROR, maximum number of data points exceeded");
		break;
		}
	    }
	else
	    {
	    end_of_file = TRUE;
	    }
	} /* end of while */
    fclose(infile);
    if (npoints == 0)
	{
	errtype("No data in spectrum");
	clear_command();
	XCOL = xcol_save;
	return;
	}
    DATA[in_spec].NPNT = npoints;

    XCOL = xcol_save;

    /* If required, do the solid-state detector correction */

    if (SOLID_CORR)
	for (i = 0; i < npoints; i++) DATA[in_spec].Y[i] -= 1;

    /* Calculate statistical error */

    for (i = 0; i < npoints; i++) DATA[in_spec].E[i] = sqrt(DATA[in_spec].Y[i]);

    /* if required, normalize on monitor signal */

    if (MCOL != 0)
	{
	for (i = 0; i < npoints; i++)
	    {
	    if (monitor[i] < 1) monitor[i] = 1;
	    DATA[in_spec].Y[i] *= MONITOR_NORM/monitor[i];
	    DATA[in_spec].E[i] *= MONITOR_NORM/monitor[i];
	    }
	}

    /* Find miniumum and maximum in spectrum */

    DATA[in_spec].MINY = DATA[in_spec].Y[0];
    DATA[in_spec].MAXY = DATA[in_spec].Y[0];
    for (i = 1; i < npoints; i++)
	{
	if (DATA[in_spec].Y[i] < DATA[in_spec].MINY)
	    DATA[in_spec].MINY = DATA[in_spec].Y[i];
	if (DATA[in_spec].Y[i] > DATA[in_spec].MAXY)
	    DATA[in_spec].MAXY = DATA[in_spec].Y[i];
	}

    /* Give spectrum a title */

    sprintf(DATA[in_spec].TITLE,"%.*s",TITLE_LENGTH,filename);

    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[in_spec].X[0],DATA[in_spec].X[DATA[in_spec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }

/***************************************************************************/
void    read_super(int old)
/***************************************************************************/

	 /*
	 Read scan from SUPER file. There are two modes:
	- old = TRUE    old style: large file with many scans
	- old = FALSE   new style: file with one scan
	 */

    {

    char    filename[FILENAME_MAX];
    FILE    *infile;
    int     in_spec;
    int     found = FALSE,end_of_scan;
    char    hatch,date[21];
    int     i,scan;
    float   monitor;
    float   dummy[DATACOLUMNS];
    int     maxcolumn,npoints;
    float   norm_corr;
    int     done,ncolumns,xcol_save;
    static int  hcol,kcol,lcol,tthcol,alpcol,betcol;
    char    column_name[20],name[20][20];
    char    extension[4];
    float   *x,*y,*e;

    /* Get scan number to be read in */

    sprintf(STRING,"Scan number to be read in [%1d]: ",SCANNR);
    SCANNR = get_int(SCANNR,STRING);

    /* Open SUPER file. Different for the two modes */

    if (old)    /* Old style SUPER file */
	{
	if ((infile = fopen(DATAFILE,"r")) == NULL)
	    {
	    sprintf(STRING,"Failed to open input file '%s'",DATAFILE);
	    errtype(STRING);
	    clear_command();
	    return;
	    }
	}
    else        /* New style SUPER file */
	{

	/* Scannumbers > 999 get the extra digit as part of the filename */

	if (SCANNR/1000 > 0)
	    {
	    if (NAUTOSUB > 0)
		sprintf(filename,"%s%1d\\%s%1d.%03d",PATHNAME,
			SCANNR/NAUTOSUB,DATAFILE,SCANNR/1000,SCANNR%1000);
	    else
		sprintf(filename,"%s%1d.%03d",DATAFILE,SCANNR/1000,
			SCANNR%1000);
	    }
	else
	    {
	    if (NAUTOSUB > 0)
		sprintf(filename,"%s%1d\\%s.%03d",PATHNAME,
			SCANNR/NAUTOSUB,DATAFILE,SCANNR%1000);
	    else
		sprintf(filename,"%s.%03d",DATAFILE,SCANNR%1000);
	    }
	if ((infile = fopen(filename,"r")) == NULL)
	    {
	    sprintf(STRING,"Failed to open input file '%s'",filename);
	    errtype(STRING);
	    clear_command();
	    return;
	    }
	}

    /* Search for correct scan number */

    while (!found)
	{
	if (fgets(INLINE,INLLEN,infile) == NULL)
	    {
	    if (old)
		sprintf(STRING,"Scan not found in file '%s'",DATAFILE);
	    else
		sprintf(STRING,"Scan not found in file '%s'",filename);
	    errtype(STRING);
	    clear_command();
	    return;
	    }
	sscanf(INLINE,"%c%d",&hatch,&scan);
	if ((hatch == '#') && (scan == SCANNR))
	    {
	    found = TRUE;
#ifdef MSDOS
	    sprintf(DATAINFO,"@%s",INLINE);     /* The @ is for GraphiC */
#endif
#ifdef VMS
	    /* Remove newline character at the end */
	    i = strlen(INLINE) < TITINFLN ? strlen(INLINE) : TITINFLN;
	    sprintf(DATAINFO,"%.*s",i-1,INLINE);
#endif
	    sscanf(INLINE,"%80[^=]%*c%f%*6c%20c",STRING,&monitor,date);
	    }
	else if (hatch == '>')
	    {
	    /* Remove newline character at the end */
	    i = strlen(INLINE) < TITINFLN ? strlen(INLINE) : TITINFLN;
	    sprintf(DATATITLE,"%.*s",i-1,INLINE);
	    }
	}

    /* Read line with column names. Skip this line if the column number
       of the x-values if given. If XCOL < 0, ask for name of column. */

    if (fgets(INLINE,INLLEN,infile) == NULL)
	{
	errtype("Error, empty spectrum");
	clear_command();
	return;
	}
    xcol_save = XCOL;
    if (XCOL < 0)
	{
	hcol = 0;
	kcol = 0;
	lcol = 0;
	tthcol = 0;
	done = FALSE;
	while (!done)
	    {
	    if (!get_token(column_name,
			   "Give column name (type <return> for options): "))
		{
		type_line(INLINE);
		}
	    else
		{
		string_tolower(column_name);
		ncolumns = sscanf(INLINE,
				  "%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s",
				  name[0],name[1],name[2],name[3],name[4],name[5],name[6],
				  name[7],name[8],name[9],name[10],name[11],name[12]);
		for (i = 0; i < ncolumns; i++)
		    {
		    string_tolower(name[i]);
		    if (strcmp(name[i],column_name) == 0)
			{
			XCOL = i+1;
			done = TRUE;
			sprintf(XCOLNAME,"%s",name[i]);
			}
		    if (strcmp(name[i],YCOLNAME) == 0) YCOL = i+1;
		    if (strcmp(name[i],"h") == 0) hcol = i+1;
		    if (strcmp(name[i],"k") == 0) kcol = i+1;
		    if (strcmp(name[i],"l") == 0) lcol = i+1;
		    if (strcmp(name[i],"tth") == 0) tthcol = i+1;
		    if (strcmp(name[i],"bet") == 0) betcol = i+1;
		    if (strcmp(name[i],"mcr") == 0) alpcol = i+1;
		    }
		if (XCOL < 0)
		    {
		    errtype("ERROR, no matching column name found");
		    clear_command();
		    fclose(infile);
		    return;
		    }
		}
	    }
	sprintf(STRING,"x title %s break",XCOLNAME);
	put_command(STRING);
	plotxye(x,y,e,1);
	}

    /* Get spectrum number where data has to be stored */

    get_inspec(&in_spec);
    READSPEC = in_spec;

    /* Display some information about the file */

    DATATITLE[0] = ' ';
    sprintf(STRING,"Title: %s\n",DATATITLE);
    type_line(STRING);
    date[20] = '\0';
    sprintf(STRING,"Creation date: %s\n",date);
    type_line(STRING);

    /* Find maximum column number to be read in */

    if (XCOL > YCOL)
	maxcolumn = XCOL;
    else
	maxcolumn = YCOL;

    /* Read data and compute statistical error in y-values */

    npoints = 0;
    HMIL = 0;
    KMIL = 0;
    LMIL = 0;
    DELTA = 0;
    BETAIN = 0;
    ALPHA = 0;
    end_of_scan = FALSE;
    while ((fgets(INLINE,INLLEN,infile) != NULL) && (!end_of_scan))
	{
	/* Check whether enough data is read in */
	if (sscanf(INLINE,"%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f",
		   &dummy[0],&dummy[1],&dummy[2],&dummy[3],&dummy[4],&dummy[5],
		   &dummy[6],&dummy[7],&dummy[8],&dummy[9],&dummy[10],&dummy[11])
	    >= maxcolumn)
	    {
	    if (XCOL == 0) /* Ennumerate x-values */
		DATA[in_spec].X[npoints] = (float) npoints+1;
	    else
		DATA[in_spec].X[npoints] = dummy[XCOL-1];
	    DATA[in_spec].Y[npoints] = dummy[YCOL-1];
	    HMIL += dummy[hcol-1];
	    KMIL += dummy[kcol-1];
	    LMIL += dummy[lcol-1];
	    DELTA += dummy[tthcol-1];
	    if (betcol != 0) BETAIN += dummy[betcol-1];
	    if (alpcol != 0) ALPHA += dummy[alpcol-1];
	    if (dummy[YCOL-1] < 1e-10)
		{
		DATA[in_spec].E[npoints] = 1;
		}
	    else
		{
		DATA[in_spec].E[npoints] =  sqrt(dummy[YCOL-1]);
		}
	    npoints++;
	    }
	else
	    {
	    end_of_scan = TRUE;
	    }
	}
    fclose(infile);

    XCOL = xcol_save;

    if (npoints == 0)
	{
	errtype("No data in spectrum");
	clear_command();
	return;
	}
    DATA[in_spec].NPNT = npoints;

    /* Make h, k, l and two-theta the average value */

    HMIL  = HMIL/npoints;
    KMIL  = KMIL/npoints;
    LMIL = LMIL/npoints;
    DELTA = DELTA/npoints;
    BETAIN = BETAIN/npoints;
    ALPHA = ALPHA/npoints;
    sprintf(STRING,"(hkl) = (%4.2f,%4.2f,%4.2f)\n",HMIL,KMIL,LMIL);
    type_line(STRING);

    /* If requested, normalize data */

    if ((MCOL != 0) && (monitor > 0.1))
	{
	norm_corr = MONITOR_NORM/monitor;
	for (i = 0; i < npoints; i++)
	    {
	    DATA[in_spec].Y[i] *= norm_corr;
	    DATA[in_spec].E[i] *= norm_corr;
	    }
	}

    /* Find miniumum and maximum in spectrum */

    DATA[in_spec].MINY = DATA[in_spec].Y[0];
    DATA[in_spec].MAXY = DATA[in_spec].Y[0];
    for (i = 1; i < npoints; i++)
	{
	if (DATA[in_spec].Y[i] < DATA[in_spec].MINY)
	    DATA[in_spec].MINY = DATA[in_spec].Y[i];
	if (DATA[in_spec].Y[i] > DATA[in_spec].MAXY)
	    DATA[in_spec].MAXY = DATA[in_spec].Y[i];
	}

    /* Give spectrum a title */

    if (old)
        sprintf(DATA[in_spec].TITLE,"%.*s #%1d",TITLE_LENGTH-7,DATAFILE,SCANNR);
    else
	sprintf(DATA[in_spec].TITLE,"%.*s",TITLE_LENGTH,filename);

    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[in_spec].X[0],DATA[in_spec].X[DATA[in_spec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }

/***************************************************************************/
void    read_type(void)
/***************************************************************************/

	 /*
	 Get type of file to be read in.
	 */

    {

    /* define read_type_menu */

#define read_type_length 9       /* number of commands in read_type menu */

    static struct   MENU read_type_menu[read_type_length] =
	{
	"srs",      2,  1,  "SRS data file",
	"spec",     2,  6,  "SPEC data file",
	"super",    2,  2,  "Super data file (single scan)",
	"osuper",   1,  3,  "Old super scan (in bigdata file)",
	"xye",      1,  4,  "File with x y (and opt. error) columns",
	"column",   1,  5,  "File with arbitrary amount of columns",
	"ftascom",  1,  7,  "Single column tascom data file",
	"help",     1,  20, "Display menu",
	"return",   1,  21, "Return to main menu"
	};

    int     stop = FALSE,old;
    char    token[100];

    while (!stop)
	{
	type_header(0);
	if (!get_token(token,"File type: ")) return;
	switch (cmnd_match(token,read_type_menu,read_type_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		sprintf(FILETYPE,"srs");
		read_srs(2);
		return;
	    case 2:
		old = FALSE;
 		sprintf(FILETYPE,"super");
		read_super(old);
		return;
	    case 3:
		old = TRUE;
		sprintf(FILETYPE,"osuper");
		read_super(old);
		return;
	    case 4:
		sprintf(FILETYPE,"xye");
		read_xye();
		return;
	    case 5:
		sprintf(FILETYPE,"column");
		read_column();
		return;
	    case 6:
		sprintf(FILETYPE,"spec");
		read_spec();
		return;
	    case 7:
		sprintf(FILETYPE,"ftascom");
		read_ftascom();
		return;
	    case 20:
		list_menu("FILE TYPES",read_type_menu,read_type_length);
		break;
	    case 21:
		stop = TRUE;
	    }
	}
    }

/***************************************************************************/
void    read_xye(void)
/***************************************************************************/

	 /*
	 Read xye file:
	- one comment line
	- data line with x and y values, and optionally an error in third
	  column.
	 */

    {

    char    filename[FILENAME_MAX];
    FILE    *infile;
    int     i,in_spec,items,npoints;
    int     error_column,end_of_file;
    float   x,y,e;

    /* Get filename and open file, return if not successful */

    get_token(filename,"Give filename (.dat): ");
    add_extension(filename,"dat");
    if ((infile = fopen(filename,"r")) == NULL)
	{
	sprintf(STRING,"Failed to open file '%s'",filename);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Get spectrum number where data has to be stored */

    get_inspec(&in_spec);
    READSPEC = in_spec;

    /* Type comments */

    if (fgets(INLINE,INLLEN,infile) == NULL)
	{
	errtype("ERROR, file is empty");
	fclose(infile);
	clear_command();
	return;
	}
    else
	{
	type_line(INLINE);
	sprintf(DATATITLE,"%.*s",TITINFLN-1,INLINE);
	}

    /* Find out if there is an error column in file */

    if (fgets(INLINE,INLLEN,infile) == NULL)
	{
	errtype("ERROR, file is empty");
	fclose(infile);
	clear_command();
	return;
	}
    items = sscanf(INLINE,"%f%f%f",&x,&y,&e);
    if (items == 3)
	{
	error_column = TRUE;
	DATA[in_spec].X[0] = x;
	DATA[in_spec].Y[0] = y;
	DATA[in_spec].E[0] = e;
	}
    else if (items == 2)
	{
	error_column = FALSE;
	DATA[in_spec].X[0] = x;
	DATA[in_spec].Y[0] = y;
	DATA[in_spec].E[0] = 1;
	}
    else
	{
	errtype("ERROR, when reading file");
	fclose(infile);
	clear_command();
	return;
	}
	 
    npoints = 1;
    end_of_file = FALSE;
    while ((fgets(INLINE,INLLEN,infile) != NULL) && (!end_of_file))
	{
	if (error_column)       /* read in three values */
	    {
	    if (sscanf(INLINE,"%f%f%f",&x,&y,&e) == 3)
		{
		DATA[in_spec].X[npoints] = x;
		DATA[in_spec].Y[npoints] = y;
		DATA[in_spec].E[npoints] = e;
		npoints++;
		}
	    else
		{
		end_of_file = TRUE;
		}
	    }
	else                    /* read in two values */
	    {
	    if (sscanf(INLINE,"%f%f",&x,&y) == 2)
		{
		DATA[in_spec].X[npoints] = x;
		DATA[in_spec].Y[npoints] = y;
		DATA[in_spec].E[npoints] = 1;
		npoints++;
		}
	    else
		{
		end_of_file = TRUE;
		}
	    }
	}
    fclose(infile);
    if (npoints == 0)
	{
	errtype("No data in spectrum");
	clear_command();
	return;
	}
    DATA[in_spec].NPNT = npoints;
	 
    /* Find miniumum and maximum in spectrum */
	 
    DATA[in_spec].MINY = DATA[in_spec].Y[0];
    DATA[in_spec].MAXY = DATA[in_spec].Y[0];
    for (i = 1; i < npoints; i++)
	{
	if (DATA[in_spec].Y[i] < DATA[in_spec].MINY)
	    DATA[in_spec].MINY = DATA[in_spec].Y[i];
	if (DATA[in_spec].Y[i] > DATA[in_spec].MAXY)
	    DATA[in_spec].MAXY = DATA[in_spec].Y[i];
	}
	 
    /* Give spectrum a title */

    sprintf(DATA[in_spec].TITLE,"%.*s",TITLE_LENGTH,filename);
	 
    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[in_spec].X[0],DATA[in_spec].X[DATA[in_spec].NPNT-1]);
    type_line(STRING);
    type_header(1);
	 
    }

