/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                         calc.c                                $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: calc.c,v 1.9 2007/06/27 15:02:28 wilcke Exp $

****************************************************************************/

/***************************************************************************

changes:

Update 21/06/2007 R. Wilcke (wilcke@esrf.fr)
                  removed function fourier_test() (no longer needed).
Update 05/07/2001 E. Vlieg (vlieg@sci.kun.nl)
                  Changed menus to follow Rainer's scheme.
                  Replaced "Miller index" by the correct term
                  "diffraction index"
Update 05/10/00:  E. Vlieg
		  Correct ZEROFRAC flag (name should be changed as well).
    30/10/91:	Structure factors of bulk are averaged over domains
		if necessary.
    11/03/93:   Poisson bulk roughness model
    15/03/93:   Gauss bulk roughness model
    20/12/94:	Also coherent addition of rotational domains possible.
    09/02/95:	All roughness models applicable to both bulk and surface
		unit cells. Approxiamated beta and beta model still as they
		were.

***************************************************************************/

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define CALC
#include "rod.h"


/***************************************************************************/
float angle_calc(int atom1, int atom2, int atom3)
/***************************************************************************/

    /*
    Calculate bond angle between three atoms
    */

{

    float cart1[3],cart2[3],cart3[3];
    float vec21[3],vec23[3];
    float length21,length23;
    int i;

    /* Compute cartesian coordinates of atoms */

    get_cartesian(cart1,XSFIT[atom1],YSFIT[atom1],ZSFIT[atom1]);
    get_cartesian(cart2,XSFIT[atom2],YSFIT[atom2],ZSFIT[atom2]);
    get_cartesian(cart3,XSFIT[atom3],YSFIT[atom3],ZSFIT[atom3]);

    /* Compute vector from atom 2 to 1 and from atom 2 to 3, and their
       lengths */

    for (i = 0; i < 3; i++)
	{
	vec21[i] = cart1[i]-cart2[i];
	vec23[i] = cart3[i]-cart2[i];
	}
    length21 = sqrt(sqr(vec21[0])+sqr(vec21[1])+sqr(vec21[2]));
    length23 = sqrt(sqr(vec23[0])+sqr(vec23[1])+sqr(vec23[2]));
    if ((length21 < 1e-6) || (length23 < 1e-6))
	{
	errtype("Error, two atoms at same position");
	return(0);
	}

    /* Return the angle, which is the in-product divided by the vector
       lengths */

    return(acos((vec21[0]*vec23[0]+vec21[1]*vec23[1]+vec21[2]*vec23[2])/
	(length21*length23)));
}

/***************************************************************************/
void calculate(void)
/***************************************************************************/

    /*
    Calculate theoretical truncation rod profiles.
    */

{

    /* define calc_menu */

#define calc_length 10      /* number of commands in calc menu */

    static struct MENU calc_menu[calc_length] =
	{
	"rod",       2,  1,  "Calculate rod profile",
	"range",     2,  2,  "Calculate f's for range of h and k",
	"qrange",    1,  3,  "Calculate f's within q-max",
	"data",      1,  4,  "Calculate f's for all data points",
	"distance",  2,  5,  "Calculate the distance between two atoms",
	"angle",     1,  6,  "Calculate bond angle between three atoms",
	"leffective",2,  7,  "Calculate effective l from slit width",
        "roughness", 3,  8,  "Calculate roughness in atomic layers",
	"help",      1,  20, "Display menu",
	"return",    1,  21, "Return to main menu"
	};

    int stop = FALSE;
    char token[100];
    float lstep;
    static float hrod,krod;
    static float hstart,hend,hstep,kstart,kend,kstep,l;
    static float hmax,qmax;
    float sum, mean, rms;
    int i,j,atom1,atom2,atom3;

    /* Check whether model has been read in */

    if (NSURF+NBULK < 1)
	{
	errtype("ERROR, no model read in");
	clear_command();
	return;
	}

    while (!stop)
	{
	if (!get_token(token,"ROD.CALC>")) return;
	switch (cmnd_match(token,calc_menu,calc_length))
	    {
	    case -1:
		break;
	    case 0:
		break;

	    case 1:     /* Calculate rod profile */

		/* Input h and k */

		sprintf(STRING,"Diffraction index h [%3.1f]: ",hrod);
		hrod = get_real(hrod,STRING);
		sprintf(STRING,"Diffraction index k [%3.1f]: ",krod);
		krod = get_real(krod,STRING);

		/* Compute range of l-values */

		if (NL < 2) NL = 2;
		lstep = (LEND-LSTART)/(NL-1);
		NTH = NL;
		for (i = 0; i < NTH; i++)
		    {
		    HTH[i] = hrod;
		    KTH[i] = krod;
		    LTH[i] = LSTART+lstep*i;
		    }

		/* Compute the three structure factors */

		for (i = 0; i < NTH; i++)
		    {
		    f_calc(hrod,krod,LTH[i],ATTEN,LBRAGG,
			   &FTH[0][i],&FTH[1][i],&FTH[2][i],&PHASE[i]);
		    }
		return;

	    case 2:     /* Calculate structure factors for range of h,k */

		/* Input range of h and k */

		sprintf(STRING,"Start value of diffraction index h [%3.1f]: ",
		    hstart);
		hstart = get_real(hstart,STRING);
		sprintf(STRING,"End value of diffraction index h [%3.1f]: ",
		    hend);
		hend = get_real(hend,STRING);
		sprintf(STRING,"Step size of diffraction index h [%3.1f]: ",
		    hstep);
		hstep = get_real(hstep,STRING);
		if (fabs(hstep) < 0.1)
		    {
		    errtype("ERROR, too small step size");
		    clear_command();
		    break;
		    }
		sprintf(STRING,"Start value of diffraction index k [%3.1f]: ",
		    kstart);
		kstart = get_real(kstart,STRING);
		sprintf(STRING,"End value of diffraction index k [%3.1f]: ",
		    kend);
		kend = get_real(kend,STRING);
		sprintf(STRING,"Step size of diffraction index k [%3.1f]: ",
		    kstep);
		kstep = get_real(kstep,STRING);
		if (fabs(kstep) < 0.1)
		    {
		    errtype("ERROR, too small step size");
		    clear_command();
		    break;
		    }
		sprintf(STRING,"Diffraction index l [%3.1f]: ",l);
		l = get_real(l,STRING);

		/* loop through the range */

		NTH = 0;
		for (i = 0; (i <= (int) fabs((hend-hstart)/hstep+0.5))
		    && (NTH < MAXTHEO); i++)
		    {
		    for (j = 0; (j <= (int) fabs((kend-kstart)/kstep+0.5))
			&& (NTH < MAXTHEO); j++)
			{
			HTH[NTH] = hstart+hstep*i;
			KTH[NTH] = kstart+kstep*j;
			LTH[NTH] = l;
			f_calc(HTH[NTH],KTH[NTH],LTH[NTH],ATTEN,LBRAGG,
			    &FTH[0][NTH],&FTH[1][NTH],&FTH[2][NTH],&PHASE[NTH]);
			if ((fabs(HTH[NTH]) < 0.1) && (fabs(KTH[NTH]) < 0.1))
			    NTH--;
			NTH++;
			}
		    }
		return;

	    case 3:     /* Calculate f's within certain q-range */

		/* Input range of h and k */

		sprintf(STRING,"Maximum value of q (in terms of h) [%4.2f]: ",
		    hmax);
		hmax = get_real(hmax,STRING);
		sprintf(STRING,"Step size of diffraction index h [%3.1f]: ",
		    hstep);
		hstep = get_real(hstep,STRING);
		if (fabs(hstep) < 0.1)
		    {
		    errtype("ERROR, too small step size");
		    clear_command();
		    break;
		    }
		sprintf(STRING,"Step size of diffraction index k [%3.1f]: ",
		    kstep);
		kstep = get_real(kstep,STRING);
		if (fabs(kstep) < 0.1)
		    {
		    errtype("ERROR, too small step size");
		    clear_command();
		    break;
		    }
		sprintf(STRING,"Diffraction index l [%3.1f]: ",l);
		l = get_real(l,STRING);

		/* Estimate range in h and k */

		hend = (int) ((1+fabs(cos(RLAT[5])))*hmax);
		hstart = -hend;
		kend = (int) (hend*RLAT[0]/RLAT[1]);
		kstart = -kend;
		qmax = 1.01*q_length(hmax,0,0);

		/* loop through the range */

		NTH = 0;
		for (i = 0; (i <= (int) fabs((hend-hstart)/hstep+0.5))
		    && (NTH < MAXTHEO); i++)
		    {
		    for (j = 0; (j <= (int) fabs((kend-kstart)/kstep+0.5))
			&& (NTH < MAXTHEO); j++)
			{
			HTH[NTH] = hstart+hstep*i;
			KTH[NTH] = kstart+kstep*j;
			LTH[NTH] = l;
			if (q_length(HTH[NTH],KTH[NTH],0.) < qmax)
			    {
			    f_calc(HTH[NTH],KTH[NTH],LTH[NTH],ATTEN,LBRAGG,
				&FTH[0][NTH],&FTH[1][NTH],&FTH[2][NTH],
				&PHASE[NTH]);
			    if((fabs(HTH[NTH]) < 0.1) && (fabs(KTH[NTH]) < 0.1))
				NTH--;
			    NTH++;
			    }
			}
		    }
		return;

	    case 4:     /* Same range as experimental data points */

		if (NDAT < 1)
		    {
		    errtype("ERROR, no experimental data read in");
		    clear_command();
		    return;
		    }
		for (i = 0; i < NDAT; i++)
		    {
		    HTH[i] = HDAT[i];
		    KTH[i] = KDAT[i];
		    LTH[i] = LDAT[i];
		    LBRAGG = LBR[i];
		    f_calc(HTH[i],KTH[i],LTH[i],ATTEN,LBR[i],
			&FTH[0][i],&FTH[1][i],&FTH[2][i],&PHASE[i]);
		    }
		NTH = NDAT;
		return;
	    case 5:     /* Calculate distance between two atoms */
		{
		float dx, dy, dz, d;
		if (NSURF < 2)
		    {
		    errtype("Too few atoms in surface unit cell");
		    return;
		    }
		atom1 = get_int(1,"First atom [1]: ");
		atom2 = get_int(2,"Second atom [2]: ");
		if ((atom1 > NSURF) || (atom2 > NSURF))
		    {
		    sprintf(STRING,"Only %1d atoms in surface unit cell",NSURF);
		    errtype(STRING);
		    return;
		    }
		d = dist_calc(atom1-1, atom2-1, &dx, &dy, &dz);
		sprintf(STRING,
		    "Distance atoms %1d (%s) and %1d (%s): %7.4f   (%.4f  %.4f  %.4f)\n",
		    atom1,ELEMENT[TS[atom1-1]],atom2,ELEMENT[TS[atom2-1]],
		    d, dx, dy, dz);
		type_line(STRING);
		return;
		}
	    case 6:     /* Calculate bond angle between three atoms */
		if (NSURF < 3)
		    {
		    errtype("Too little atoms in surface unit cell");
		    return;
		    }
		atom1 = get_int(1,"First atom [1]: ");
		atom2 = get_int(2,"Second (centre) atom [2]: ");
		atom3 = get_int(3,"Third atom [3]: ");
		if ((atom1 > NSURF) || (atom2 > NSURF) || (atom3 > NSURF))
		    {
		    sprintf(STRING,"Only %1d atoms in surface unit cell",NSURF);
		    errtype(STRING);
		    return;
		    }
		sprintf(STRING,
		    "Bond angle from atom %1d (%s) via %1d (%s) to %1d (%s): %7.4f\n",
		    atom1,ELEMENT[TS[atom1-1]],atom2,ELEMENT[TS[atom2-1]],
		    atom3,ELEMENT[TS[atom3-1]],
		    angle_calc(atom1-1,atom2-1,atom3-1)*RAD);
		type_line(STRING);
		return;
	    case 7:
		l_effective();
		return;
            case 8:
                rms_calc(&sum, &mean, &rms);
		sprintf(STRING,
		    "Sum = %.3f layers, mean value = %.3f layers, rms = %.3f layers\n",
                    sum, mean, rms);
                type_line(STRING);

		/*
		   Fake rod scan for plotting roughness:
		   Compute range of "l"-values
		 */
		if (NL < 2) NL = 2;
		lstep = (mean+5.*rms)/(NL-1);
		NTH = NL;
		for (i = 0; i < NTH; i++)
		    {
		    HTH[i] = 0.;
		    KTH[i] = 0.;
		    LTH[i] = lstep*i;
		    }

		/* Compute layer occupancies */

		for (i = 0; i < NTH; i++)
		    {
                    FTH[0][i] = FTH[1][i] = FTH[2][i] = PHASE[i] =
			1000.*occ_calc((int)LTH[i]);
		    }
		return;

 	    case 20:
		list_menu("CALCULATION MENU",calc_menu,calc_length);
		break;
	    case 21:
		stop = TRUE;
	    }
	}
}

/***************************************************************************/
float dist_calc(int atom1, int atom2, float *dx, float *dy, float *dz)
/***************************************************************************/

    /*
    Calculate the distance between two atoms
    Also compute cartesian difference between x, y and z of atoms
    */

{

    float cart1[3],cart2[3],u1[3],u2[3],distance,d,ldx,ldy,ldz;
    int ix, iy;

    /* Compute cartesian coordinates of the two atoms */

    get_cartesian(cart1,XSFIT[atom1],YSFIT[atom1],ZSFIT[atom1]);
    get_cartesian(cart2,XSFIT[atom2],YSFIT[atom2],ZSFIT[atom2]);

    /* Calculate shortest distance possible by moving one atom
       around the unit cell */

    get_cartesian(u1,1,0,0);
    get_cartesian(u2,0,1,0);

    for (ix = -1; ix <= 1; ix++)
        {
        for (iy = -1; iy <= 1; iy++)
	    {
	    ldx = cart1[0]-cart2[0]+(float)ix*u1[0]+(float)iy*u2[0];
	    ldy = cart1[1]-cart2[1]+(float)ix*u1[1]+(float)iy*u2[1];
	    ldz = cart1[2]-cart2[2];
	    d = sqrt(sqr(ldx)+sqr(ldy)+sqr(ldz));
	    if ( (ix == -1 && iy == -1) || d < distance )
		{
		distance = d;
		*dx = fabs(ldx);
		*dy = fabs(ldy);
		*dz = fabs(ldz);
		}
	    }
	}

    return (distance);

}

/***************************************************************************/
float f_atomic(float q_half, int natom)
/***************************************************************************/

    /*
    Compute atomic scattering factor for atom 'natom'.
    */

{

    float f;
    int i;

    f = F_COEFF[natom][8];
    for (i = 0; i < 4; i++)
	{
	f += F_COEFF[natom][i*2]*exp(-F_COEFF[natom][i*2+1]*sqr(0.5*q_half/PI));
	}
    return(f);

}

/***************************************************************************/
void f_bulk(float h, float k, float l, float fat[], float dw[],
    float atten, float re_bulk[], float im_bulk[])
/***************************************************************************/

    /*
    Calculate real and imaginary parts of the structure factor of the
    bulk unit cell for all domains present.
    */

{

    int	i,j;
    float hnew,knew,re_bulk1,im_bulk1,fatom,factor,argument;

    for (j = 0; j < NDOMAIN; j++)
	{
	re_bulk[j] = im_bulk[j] = 0.;
	next_domain(h,k,&hnew,&knew,j);	/* Next equivalent refl. */
	if (!(!ZEROFRACT && ((fmod(hnew,1.) != 0) || (fmod(knew,1.) != 0))))
	    {
	    re_bulk1 = im_bulk1 = 0.;
	    for (i = 0; i < NBULK; i++)
		{
		/* the exp is added to fatom in order to avoid numerical
		   problems (suggestion by Yves Garreau */
		fatom = fat[TB[i]]*dw[NDWB[i]]*exp(atten*ZB[i]);
		argument = 2*PI*(hnew*XB[i]+knew*YB[i]+l*ZB[i]);
		re_bulk1 += fatom*cos(argument);
		im_bulk1 += fatom*sin(argument);
		}
	    factor = sqr(1-exp(-atten))+4*exp(-atten)*sqr(sin(PI*l));
	    if (factor < 1e-20) factor = 1e-20;
	    re_bulk[j] = ((1-exp(-atten)*cos(2*PI*l))*re_bulk1
		+exp(-atten)*sin(2*PI*l)*im_bulk1)/factor;
	    im_bulk[j] = ((1-exp(-atten)*cos(2*PI*l))*im_bulk1
		-exp(-atten)*sin(2*PI*l)*re_bulk1)/factor;
	    }
	}
}

/***************************************************************************/
float calc_roughness(float h, float k, float l, float lbragg)
/***************************************************************************/

    /*
    Calculate reduction in intensity due to roughness. Various models are
    available, whose effects can always be expressed as a multiplication
    factor of the calculated intensity. All models use a column
    approximation, i.e., each level terminates in an identical fashion.
    The programme does not deal with phase factors too carefully in these
    calculations, so be warned!

    The 'approximate beta' is very quick, but requires as input the
    diffraction index of the nearest Bragg peak.
    The other models are numerical and thus slower. They calculate the
    phase factor between layers from the bulk unit cell. The atoms in the
    bulk unit cell need to be in the correct order: ranging from top to
    bottom.

    */

{
    int integer_order, j, layer, cell, npl;
    double occ,occ_lower;
    float roughness;
    float im_total, re_total, phase;

    if (fabs(fmod(lbragg,1.)) < 0.01) integer_order = TRUE;
    else integer_order = FALSE;

    if (ROUGHMODEL == APPROXBETA)
	{
	if (integer_order)
	    roughness = (1-BETA)/
		sqrt(sqr(1-BETA)+4*BETA*sqr(sin(PI*(l-lbragg)/NLAYERS)));
	else /* for fractional-orders no l-dependence */
	    roughness = sqrt((1-BETA)/(1+BETA));
	}
    else if (ROUGHMODEL == EXACTBETA) /* Still treated in a funny way */
	roughness = 1.;
    else /* a numerical model has been chosen */
	{
	if (NBULK == 0)
	    roughness = 1.;
	else if (integer_order)
	    {

	    /* find correct phase shift between successive layers  */

	    npl = NBULK/NLAYERS;   /*number of atoms per layer*/
	    phase = 2*PI*((XB[0]-XB[npl])*h+(YB[0]-YB[npl])*k+
		(ZB[0]-ZB[npl])*l);

	    /* sum over all rough layers */

	    re_total=im_total=0.;
	    occ = 1.;
	    for (layer=0; (layer < 100) && (occ > 1e-5); layer++)
		{
		occ = occ_calc(layer);
		re_total += occ*cos(phase*layer);
		im_total += occ*sin(phase*layer);
		}
	    roughness = sqrt(sqr(re_total)+sqr(im_total));
	    }
	else /* fractional order */
	    {

	    /* inhorent sum over all visible surface layers */

	    re_total= 0.;
	    occ = 1.;
	    occ_lower = occ_calc(0);
	    for (layer=1; (layer < 100) && (occ > 1e-5); layer++)
		{
		occ = occ_calc(layer);
		re_total += sqr(occ_lower-occ);
		occ_lower = occ;
		}
	    roughness = sqrt(re_total);
	    }

	}
    return(roughness);
}

/***************************************************************************/
void f_bulk_beta(float h, float k, float l, float fat[], float dw[],
    float re_bulk[], float im_bulk[])
/***************************************************************************/

    /*
    Robinson's beta roughness model:

    Special function to test the influence of the roughness parameter
    beta. This function does an 'exact' calculation, rather than using
    the general roughness equation. The last atom in the bulk unit cell has
    to be the deepest one (lowest z-coordinate).
    */

{

    int	i,j;
    float hnew,knew,re_bulk1,im_bulk1,fatom,factor,argument;
    int	exponent;
    float z0,betapow;

    for (j = 0; j < NDOMAIN; j++)
	{
	re_bulk[j] = im_bulk[j] = 0.;
	next_domain(h,k,&hnew,&knew,j);	/* Next equivalent refl. */
	if (!(!ZEROFRACT && ((fmod(hnew,1.) != 0) || (fmod(knew,1.) != 0))))
	    {
	    re_bulk1 = im_bulk1 = 0.;
	    z0 = ZB[NBULK-1];
	    for (i = 0; i < NBULK; i++)
		{
		/* Calculate layer number of element i */
		exponent = (int) ((ZB[i]-z0)*NLAYERS+1.001);
		fatom = pow(BETA,exponent)*fat[TB[i]]*dw[NDWB[i]];
		argument = 2*PI*(hnew*XB[i]+knew*YB[i]+l*(ZB[i]+1));
		re_bulk1 += fatom*cos(argument);
		im_bulk1 += fatom*sin(argument);
		}
	    betapow = pow(BETA,NLAYERS);
	    factor = sqr(1-betapow)+4*betapow*sqr(sin(PI*l));
	    if (factor < 1e-20) factor = 1e-20;
	    re_bulk[j] = ((1-betapow*cos(2*PI*l))*re_bulk1
		-betapow*sin(2*PI*l)*im_bulk1)/factor;
	    im_bulk[j] = ((1-betapow*cos(2*PI*l))*im_bulk1
		+betapow*sin(2*PI*l)*re_bulk1)/factor;
	    }
	}
}

/***************************************************************************/
void f_calc(float h, float k, float l, float atten, float lbragg,
    float *fbulk, float *fsurf, float *fsum, float *phase)
/***************************************************************************/

    /*
    Compute bulk and surface contributions to rod profile, plus the
    interference sum, including the phase, of these two.
    */

{

    float q,q_par,q_perp;
    float dw[MAXPAR+1];   /* Debye-Waller factors. dw[0] contains DW for
                             atoms without serial DW number, i.e. having serial
                             number 0. */
    float dw_par[MAXPAR+1],dw_perp[MAXPAR+1];
    float fat[MAXTYPES];  /* Atomic scattering factors */
    float scale;
    volatile int i,j;
    float im_bulk[MAXDOM],re_bulk[MAXDOM];
    float im_surf[MAXDOM],re_surf[MAXDOM];
    float im_surf2[MAXDOM],re_surf2[MAXDOM],fsurf2,fsum1,fsum2;
    float im_bul,re_bul,im_sur1,re_sur1,im_sur2,re_sur2;
    float re_dom, im_dom;
    float fbulk_sqr_save;
    float roughness;

    /* Compute length of in-plane and out-of-plane momentum transfer */

    q_par = q_length(h,k,0);
    q_perp = q_length(0,0,l);
    // q = sqrt(q_par*q_par+q_perp*q_perp);
    q = q_length(h,k,l);

    /* Compute in-plane Debye-Waller factors */

    if (NDWTOT2 != 0)
	{
	dw_par[0] = 1.;         /* Unit factor if no parameter given */
	for (i = 1; i < NDWTOT+1; i++)
	    {
	    dw_par[i] = exp(-DEBWAL[i-1]*q_par*q_par/PIPI16);
	    }
	}

    /* Compute out-of-plane Debye-Waller factors */

    dw_perp[0] = 1.;         /* Unit factor if no parameter given */
    for (i = 1; i < NDWTOT2+1; i++)
	{
	dw_perp[i] = exp(-DEBWAL2[i-1]*q_perp*q_perp/PIPI16);
	}

    /*
       In case NDWS2[i] = 0, isotropic Debye-Waller parameters are assumed. Then
       the total momentum transfer is used together with the value of DEBWAL.
     */

    dw[0] = 1.;            /* Unit factor if no parameter given */
    for (i = 1; i < NDWTOT+1; i++)
	{
	dw[i] = exp(-DEBWAL[i-1]*q*q/PIPI16);
	}

    /* Compute atomic scattering factors */

    for (i = 0; i < NTYPES; i++)
	{
	fat[i] = f_atomic(q/2,i);
	}

    /* Compute intensity reduction due to roughness */

    roughness = calc_roughness(h,k,l,lbragg);

    /* Compute scale factor, a negative lbragg means a separate scale factor */

    if (lbragg < 0)
	scale = SCALE*SCALE2;
    else
	scale = SCALE;

    /* Compute bulk structure factor */

    if (NBULK > 0)
	{
	f_bulk(h,k,l,fat,dw,atten,re_bulk,im_bulk);
	*fbulk = 0;
	for (j = 0; j < NDOMAIN; j++)
	    *fbulk += DOMOCCUP[j]*(sqr(re_bulk[j])+sqr(im_bulk[j]));
	fbulk_sqr_save = *fbulk;
	*fbulk = scale*roughness*sqrt(*fbulk);
	}
    else
	{
	fbulk_sqr_save = *fbulk = 0;
	for (j = 0; j < NDOMAIN; j++)
	    re_bulk[j] = im_bulk[j] = 0;
	}

    /* Compute surface structure factor of 1st unit cell for all
       present domains */

    f_surf(h,k,l,fat,dw_par,dw_perp,dw,1,re_surf,im_surf);
    *fsurf = 0;
    for (j = 0; j < NDOMAIN; j++)
	*fsurf += DOMOCCUP[j]*(sqr(re_surf[j])+sqr(im_surf[j]));
    *fsurf = scale*roughness*sqrt(*fsurf*SURFFRAC*(1-SURF2FRAC));

    /* Calculate surface structure factor for 2nd surface unit cell */

    if (NSURF2 != 0)
	{
	f_surf(h,k,l,fat,dw_par,dw_perp,dw,2,re_surf2,im_surf2);
	fsurf2 = 0;
	for (j = 0; j < NDOMAIN; j++)
	    fsurf2 += DOMOCCUP[j]*(sqr(re_surf2[j])+sqr(im_surf2[j]));
	fsurf2 = scale*roughness*sqrt(fsurf2*SURFFRAC*SURF2FRAC);
	*fsurf = sqrt(sqr(*fsurf)+sqr(fsurf2));
	}

    /* This is only for test purposes for the time being */
    if (ROUGHMODEL == EXACTBETA)
	f_bulk_beta(h,k,l,fat,dw,re_surf,im_surf);

    /* Compute interference sum of bulk and surface. */

    fsum1 = re_dom = im_dom = 0.;
    for (j = 0; j < NDOMAIN; j++)
	{
	if (COHERENTDOMAINS == FALSE)
	    {
	    fsum1 += DOMOCCUP[j]*(sqr(re_bulk[j]+re_surf[j])
		+sqr(im_bulk[j]+im_surf[j]));
	    }
	else
	    {
	    re_dom += DOMOCCUP[j]*(re_bulk[j]+re_surf[j]);
	    im_dom += DOMOCCUP[j]*(im_bulk[j]+im_surf[j]);
	    fsum1 = sqr(re_dom)+sqr(im_dom);
	    }
	}

    *fsum = scale*roughness*sqrt(fsum1*SURFFRAC*(1-SURF2FRAC)+
	(1-SURFFRAC)*fbulk_sqr_save);

    /* Add corresponding fraction of 2nd surface unit cell to the result
       if necessary */

    if (NSURF2 != 0)
	{
	fsum2 = re_dom = im_dom = 0.;
	for (j = 0; j < NDOMAIN; j++)
	    {
	    if (COHERENTDOMAINS == FALSE)
		{
		fsum2 += DOMOCCUP[j]*(sqr(re_bulk[j]+re_surf2[j])
		    +sqr(im_bulk[j]+im_surf2[j]));
		}
	    else
		{
		re_dom += DOMOCCUP[j]*(re_bulk[j]+re_surf2[j]);
		im_dom += DOMOCCUP[j]*(im_bulk[j]+im_surf2[j]);
		fsum2 = sqr(re_dom)+sqr(im_dom);
		}
	    }

	fsum2 = scale*roughness*sqrt(fsum2*SURFFRAC*SURF2FRAC);
	*fsum = sqrt(sqr(*fsum)+sqr(fsum2));
	}

    /* Calculate the phase factor of the interference sum */

    im_bul = re_bul = 0;
    for (i = 0; i < NDOMAIN; i++)
	{
	im_bul += DOMOCCUP[i]*im_bulk[i];
	re_bul += DOMOCCUP[i]*re_bulk[i];
	}

    im_sur1 = re_sur1 = 0;
    for (i = 0; i < NDOMAIN; i++)
	{
	im_sur1 += DOMOCCUP[i]*im_surf[i];
	re_sur1 += DOMOCCUP[i]*re_surf[i];
	}

    im_sur2 = re_sur2 = 0;
    if (NSURF2 != 0)
	{
	for (i = 0; i < NDOMAIN; i++)
	    {
	    im_sur2 += DOMOCCUP[i]*im_surf2[i];
	    re_sur2 += DOMOCCUP[i]*re_surf2[i];
	    }
	}

    *phase = atan2(
	im_bul+SURFFRAC*(1-SURF2FRAC)*im_sur1+SURFFRAC*SURF2FRAC*im_sur2,
	re_bul+SURFFRAC*(1-SURF2FRAC)*re_sur1+SURFFRAC*SURF2FRAC*re_sur2+1e-10);

    if (!STRUCFAC)
	{
	*fbulk *= *fbulk;
	*fsurf *= *fsurf;
	*fsum *= *fsum;
	}

}

#ifdef SPEEDUP
/***************************************************************************/
float f_calc_fit(float h, float k, float l, float lbragg, float q_par,
    float q_perp, float fat[], float re_bulk[], float im_bulk[],
    float fbulk_sqr)
/***************************************************************************/

    /*
    Special version of f_calc that stores constant values (in particular
    the bulk structure factors) and is therefore faster when fitting.
    */

    {

    float roughness=1.;
    float dw[MAXPAR+1];   /* Debye-Waller factors. dw[0] contains DW for atoms
                             without serial DW number, i.e. having serial
                             number 0. */
    float dw_par[MAXPAR+1],dw_perp[MAXPAR+1];
    float scale;
    static int 	i,j;
    float im_surf[MAXDOM],re_surf[MAXDOM], re_dom, im_dom;
    float im_surf2[MAXDOM],re_surf2[MAXDOM],fsurf2,fsum,fsum1,fsum2;
    float re_rough,im_rough;
    static float prod, sum;

    /* Compute in-plane Debye-Waller factors */

    if (NDWTOT2 != 0)
	{
	dw_par[0] = 1.;         /* Unit factor if no parameter given */
	for (i = 1; i < NDWTOT+1; i++)
	    {
	    dw_par[i] = exp(-DEBWAL[i-1]*q_par*q_par/PIPI16);
	    }
	}

    /* Compute out-of-plane Debye-Waller factors */

    dw_perp[0] = 1.;         /* Unit factor if no parameter given */
    for (i = 1; i < NDWTOT2+1; i++)
	{
	dw_perp[i] = exp(-DEBWAL2[i-1]*q_perp*q_perp/PIPI16);
	}

    /* In case NDWS2[i] = 0, isotropic Debye-Waller parameters are assumed.
       Then the total momentum transfer is used together with the value of
       DEBWAL. */

    dw[0] = 1.;            /* Unit factor if no parameter given */
    for (i = 1; i < NDWTOT+1; i++)
	{
	prod= -DEBWAL[i-1];
	/* Not correct for things like KDP(101) */
	sum = q_par*q_par+q_perp*q_perp;
	prod *= sum;
	prod /= PIPI16;
	exp(1.);
	dw[i] = exp(prod);
	}

    /* Compute intensity reduction due to roughness */

    roughness = calc_roughness(h,k,l,lbragg);

    /* Compute scale factor, a negative lbragg means a separate scale factor */

    if (lbragg < 0)
	scale = SCALE*SCALE2;
    else
	scale = SCALE;

    /* Compute surface structure factor of 1st unit cell for all
       present domains */

    f_surf(h,k,l,fat,dw_par,dw_perp,dw,1,re_surf,im_surf);

    /* Calculate surface structure factor for 2nd surface unit cell */

    if (NSURF2 != 0)
	{
	f_surf(h,k,l,fat,dw_par,dw_perp,dw,2,re_surf2,im_surf2);
	}

    /* This is only for test purposes for the time being */
    if (ROUGHMODEL == EXACTBETA)
	f_bulk_beta(h,k,l,fat,dw,re_surf,im_surf);

    /* Compute interference sum of bulk and surface. */

    fsum1 = re_dom = im_dom = 0.;
    for (j = 0; j < NDOMAIN; j++)
	{
	if (COHERENTDOMAINS == FALSE)
	    {
	    fsum1 += DOMOCCUP[j]*(sqr(re_bulk[j]+re_surf[j])
		+sqr(im_bulk[j]+im_surf[j]));
	    }
	else
	    {
	    re_dom += DOMOCCUP[j]*(re_bulk[j]+re_surf[j]);
	    im_dom += DOMOCCUP[j]*(im_bulk[j]+im_surf[j]);
	    fsum1 = sqr(re_dom)+sqr(im_dom);
	    }
	}

    fsum = scale*roughness*sqrt(fsum1*SURFFRAC*(1-SURF2FRAC)+
	(1-SURFFRAC)*fbulk_sqr);

    /* Add corresponding fraction of 2nd surface unit cell to the result
       if necessary */

    if (NSURF2 != 0)
	{
	fsum2 = re_dom = im_dom = 0.;
	for (j = 0; j < NDOMAIN; j++)
	    {
	    if (COHERENTDOMAINS == FALSE)
		{
		fsum2 += DOMOCCUP[j]*(sqr(re_bulk[j]+re_surf2[j])
		    +sqr(im_bulk[j]+im_surf2[j]));
		}
	    else
		{
		re_dom += DOMOCCUP[j]*(re_bulk[j]+re_surf2[j]);
		im_dom += DOMOCCUP[j]*(im_bulk[j]+im_surf2[j]);
		fsum2 = sqr(re_dom)+sqr(im_dom);
		}
	    }

	fsum2 = scale*roughness*sqrt(fsum2*SURFFRAC*SURF2FRAC);
	fsum = sqrt(sqr(fsum)+sqr(fsum2));
	}

    if (STRUCFAC)
	return(fsum);
    else
	return(fsum*fsum);

}

/***************************************************************************/
void f_calc_fit_init(void)
/***************************************************************************/

    /*
    Store constant parameter values to speedup fitting.
    */

{
    int	i,j;
    float q,dw[MAXPAR+1],fbulk;

    for (i = 0; i < NDAT; i++)
	{
	Q_PAR[i] = q_length(HDAT[i],KDAT[i],0);
	Q_PERP[i] = q_length(0,0,LDAT[i]);
	/* q = sqrt(Q_PAR[i]*Q_PAR[i]+Q_PERP[i]*Q_PERP[i]); */
	q = q_length(HDAT[i],KDAT[i],LDAT[i]);

	dw[0] = 1.;            /* Unit factor if no parameter given */
	for (j = 1; j < NDWTOT+1; j++)
	    {
	    dw[j] = exp(-DEBWAL[j-1]*q*q/PIPI16);
	    }

	/* Compute atomic scattering factors */

	for (j = 0; j < NTYPES; j++)
	    {
	    FAT[i][j] = f_atomic(q/2,j);
	    }

	/* Compute bulk structure factor */

	if (NBULK > 0)
	    {
	    f_bulk(HDAT[i],KDAT[i],LDAT[i],FAT[i],dw,
		ATTEN,RE_BULK[i],IM_BULK[i]);
	    fbulk = 0;
	    for (j = 0; j < NDOMAIN; j++)
		fbulk += sqr(RE_BULK[i][j])+sqr(IM_BULK[i][j]);
	    FBULK_SQR[i] = fbulk/NDOMAIN;
	    }
	else
	    {
	    FBULK_SQR[i] = 0;
	    for (j = 0; j < NDOMAIN; j++)
		RE_BULK[i][j] = IM_BULK[i][j] = 0;
	    }
	}
}
#endif /* SPEEDUP */

/***************************************************************************/
void f_surf(float h, float k, float l, float fat[],
    float dw_par[], float dw_perp[], float dw[], int cell,
    float re_surf[], float im_surf[])
/***************************************************************************/

    /*
    Calculate real and imaginary parts of the structure factor of the
    surface unit cell for all domains present. Depending on the value
    of 'cell', this is done for the first or second surface unit cell.
    */

{

    int	i,j,istart,iend;
    float hnew,knew,fatom;

    if (cell == 1)
	{
	istart = 0;
	iend = NSURF;
	}
    else if (cell == 2)
	{
	istart = NSURF;
	iend = NSURF+NSURF2;
	}

    for (j = 0; j < NDOMAIN; j++)
	{
	re_surf[j] = im_surf[j] = 0.;
	/* Next equivalent diffraction indices */
	next_domain(h,k,&hnew,&knew,j);

	/*
	   Don't compute structure factor if the transformed diffraction indices
	   are fractional and the corresponding flag is set
	*/

	if (!(!ZEROFRACT && ((fmod(hnew,1.) != 0) || (fmod(knew,1.) != 0))))
	    {
	    for (i = istart; i < iend; i++)
		{
		if (NDWS2[i] != 0)
		    {
		    fatom = fat[TS[i]]*dw_par[NDWS[i]]*dw_perp[NDWS2[i]];
		    }
		else
		    {
		    fatom = fat[TS[i]]*dw[NDWS[i]];
		    }
		if (NOCCUP[i] != 0) fatom *= OCCUP[NOCCUP[i]-1];
		re_surf[j] += fatom*cos(2*PI*(hnew*XSFIT[i]+knew*YSFIT[i]+
		    l*ZSFIT[i]));
		im_surf[j] += fatom*sin(2*PI*(hnew*XSFIT[i]+knew*YSFIT[i]+
		    l*ZSFIT[i]));
		}
	    }
	}
}

/***************************************************************************/
float fourier(float x, float y, int mode)
/***************************************************************************/

    /*
    Compute a Fourier transform function for point (x,y) in reduced
    coordinates.
	mode 1  Patterson of experimental structure factors
	mode 2  Patterson function of theoretical structure factors
	mode 3  Electron density difference map
	mode 4  Electron density map of structure model

    For each structrure factor, a sum is made over all symmetry-equivalent
    reflections given in SYMSET
    */

{

    int i,j;
    float four,temp;

    four = 0;
    switch (mode)
	{
	case 1:
	    for (i = 0; i < NDAT; i++)
		{
		if (STRUCFAC) temp = FDAT[i]*FDAT[i];
		else	temp = FDAT[i];
		for (j = 0; j < SYMSET[i].N; j++)
		    {
		    four += temp*
			cos(2*PI*(SYMSET[i].H[j]*x+SYMSET[i].K[j]*y));
		    }
		}
	    break;
	case 2:
	    for (i = 0; i < NTH; i++)
		{
		if (STRUCFAC) temp = FTH[2][i]*FTH[2][i];
		else	temp = FTH[2][i];
		for (j = 0; j < SYMSET[i].N; j++)
		    {
		    four += temp*
			cos(2*PI*(SYMSET[i].H[j]*x+SYMSET[i].K[j]*y));
		    }
		}
	    break;
	case 3:
	    for (i = 0; i < NDAT; i++)
		{
		if (STRUCFAC) temp = FDAT[i]-FTH[2][i];
		else	temp = sqrt(FDAT[i])-sqrt(FTH[2][i]);
		for (j = 0; j < SYMSET[i].N; j++)
		    {
		    four += temp*
			cos(2*PI*(SYMSET[i].H[j]*x+SYMSET[i].K[j]*y)-PHASE[i]);
		    }
		}
	    break;
	case 4:
	    for (i = 0; i < NTH; i++)
		{
		if (STRUCFAC) temp = FTH[2][i];
		else	temp = sqrt(FTH[2][i]);
		for (j = 0; j < SYMSET[i].N; j++)
		    {
		    four += temp*
			cos(2*PI*(SYMSET[i].H[j]*x+SYMSET[i].K[j]*y)-PHASE[i]);
		    }
		}
	    break;
	}

    return(four);

}

/***************************************************************************/
void generate_equivalents(int h, int k, int n)
/***************************************************************************/

    /*
    Generate symmetry-equivalent reflections of (hk) and store these
    as the n-th item in SYMSET.
    */

{

    int i,j,hnew,knew;
    int in_set;

    SYMSET[n].N = 0;
    for (i = 0; i < NTRANS; i++)
	{
	hnew = TRANS[i][0][0]*h+TRANS[i][0][1]*k;
	knew = TRANS[i][1][0]*h+TRANS[i][1][1]*k;

	/* Check whether this reflection, or its Friedel sister, is not
	   allready in the set */

	in_set = FALSE;
	for (j = 0; j < SYMSET[n].N; j++)
	    {
	    if (((hnew == SYMSET[n].H[j]) && (knew == SYMSET[n].K[j])) ||
		((hnew == -SYMSET[n].H[j]) && (knew == -SYMSET[n].K[j])))
		in_set = TRUE;
	    }
	if (!in_set)
	    {
	    SYMSET[n].H[SYMSET[n].N] = hnew;
	    SYMSET[n].K[SYMSET[n].N] = knew;
	    SYMSET[n].N++;
	    }
	}
}

/***************************************************************************/
void get_cartesian(float cart[3], float x, float y, float z)
/***************************************************************************/

    /*
    Calculate cartesian coordinates in Angstroms for the atom at position
    (x,y,z) in reduced coordinates.
    */

{

    cart[0] = DLAT[0]*x+DLAT[1]*cos(DLAT[5])*y+DLAT[2]*cos(DLAT[4])*z;
    cart[1] = DLAT[1]*sin(DLAT[5])*y-DLAT[2]*sin(DLAT[4])*cos(RLAT[3])*z;
    cart[2] = z/RLAT[2];

}

/***************************************************************************/
void l_effective(void)
/***************************************************************************/

    /*
    Calculate effective l-value, as caused by finite value of l-defining
    slits. All l-values of the data are changed.
    This is only important for l-values close to Bragg peaks.
    */

{

#define LSTEP   0.001   /* Step size in l for calculation */

    float lwidth,l;
    float lstart,lftot,ftot;
    float fbulk,fsurf,fsum,phase;
    int i,j,nl;

    /* Get slit width */

    lwidth = get_real(0,"Slit width in units of l [0]: ");

    /* Make odd number of steps, to ensure that the center point is
       also calculated */

    nl = (((int) (lwidth/LSTEP))/2)*2+1;
    for (i = 0; i < NDAT; i++)
	{
	lstart = -lwidth/2+LDAT[i]+LSTEP/5;
	lftot = ftot = 0;
	for (j = 0; j < nl; j++)
	    {
	    l = lstart+j*LSTEP;
	    f_calc(HDAT[i],KDAT[i],l,ATTEN,LBR[i],
		&fbulk,&fsurf,&fsum,&phase);
	    lftot += l*fsum;
	    ftot += fsum;
	    }
	l = lftot/ftot;
	sprintf(STRING,"%6.3f -> %6.3f (%7.4f)\n",LDAT[i],l,l-LDAT[i]);
	type_line(STRING);
	LDAT[i] = l;
	}
    }

/***************************************************************************/
void make_symmetry_set(int mode)
/***************************************************************************/

    /*
    Generate all symmetry equivalent reflections belonging to the input
    structure factors.
	mode 1  experimental structure factors (for Patterson)
	mode 2  theoretical structure factors (for Patterson)
	mode 3  use experimental structure factors for difference map
	mode 4  theoretical structure factors (for electron density map)
    */

{

    int i;

    switch (mode)
	{
	case 1:
	case 3:
	    for (i = 0; i < NDAT; i++)
		{
		generate_equivalents(HDAT[i],KDAT[i],i);
		}
	    break;
	case 2:
	case 4:
	    for (i = 0; i < NTH; i++)
		{
		generate_equivalents(HTH[i],KTH[i],i);
		}
	    break;
	}
}

/***************************************************************************/
void next_domain(float hin, float kin, float *hout, float *kout, int ndomain)
/***************************************************************************/

    /*
    Calculate the equivalent diffraction indices (hout,kout) corresponding
    to the n-th domain.
    */

{
    *hout = hin*DOMMAT11[ndomain]+kin*DOMMAT12[ndomain];
    *kout = hin*DOMMAT21[ndomain]+kin*DOMMAT22[ndomain];
}

/***************************************************************************/
double occ_calc(int k)
/***************************************************************************/

   /*
   Calculate occupancy of layer k (= fraction of surface that is terminated
   by layer k) for the various roughness roughness models
   */

{
    float occ;

    if (k < 0) /* this should never happen! */
	{
	errtype("Error, called occ_calc with negative layer number");
	return(1.);
	}

    if (BETA < 1e-5) BETA = 1e-5;   /* avoid numerical problems */

    switch (ROUGHMODEL)
	{
	case APPROXBETA:
	case EXACTBETA:
	    occ = (1-BETA)*pow(BETA,k);
	    break;

	case POISSONROUGH:
	    occ = pow(BETA+1e-10,k)*exp(-BETA)/factrl(k);
	    break;

	case NUMBETA:
	    occ = (1-BETA)*pow(BETA,k);
	    break;

	case GAUSSROUGH:
	    occ = exp(-sqr(k/BETA))-exp(-sqr((k+1)/BETA));
	    break;

	case LINEARROUGH:
	    if (k < BETA )
		occ = 1/BETA;
	    else
		occ = 0.;
	    break;

	case COSINEROUGH:
	    if (k+1 < BETA)
		occ = 0.5*(cos(PI*k/BETA)-cos(PI*(k+1)/BETA));
	    else
		occ = 0.;
	    break;

	case TWOLEVEL:
	    if (k == 0)
		occ = 1-BETA;
	    else if
		(k == 1) occ = BETA;
	    else
		occ = 0.;
	    break;

	} /* endswitch */
    return (occ);
}

/***************************************************************************/
double factrl(int n)
/***************************************************************************/

    /*
    Calculate n!.
    */

{
    static int ntop=4;
    static double a[33]={1.0,1.0,2.0,6.0,24.0};
    int j;
    double gammln(float);

    if (n > 32)
	return exp(gammln(n+1.0));
    while (ntop<n)
	{
	j=ntop++;
	a[ntop]=a[j]*ntop;
	}
    return a[n];
}

/***************************************************************************/
float q_length(float h, float k, float l)
/***************************************************************************/

    /*
    Calculate length of momentum transfer vector q = (h,k,l).
    */

{
    return(2*PI*sqrt(
	sqr(h*RLAT[0]+k*RLAT[1]*cos(RLAT[5])+l*RLAT[2]*cos(RLAT[4]))
	+sqr(k*RLAT[1]*sin(RLAT[5])-l*RLAT[2]*sin(RLAT[4])*cos(DLAT[3]))
	+sqr(l/DLAT[2])
	));
}

/***************************************************************************/
float rms_calc(float *sum, float *mean, float *rms)
/***************************************************************************/

    /*
    Calculate sum, root mean square and mean value of roughness distribution
    */

{
    double p2, norm, fm, sm;
    int i;

    switch (ROUGHMODEL) {
    case APPROXBETA:
    case EXACTBETA:
	if (BETA > 0 && 1.-BETA > 1e-8) {
	    *mean = BETA/(1.-BETA);
	    *rms  = sqrt(BETA)/(1.-BETA);
	    *sum  = 1/(1.-BETA);
	} else {
	    *mean = *rms = -1.;
	} /* endif */
	break;

    default: /* numerical models: */

	/* Calculate normalization factor
	   Note this routine has changed compared to Martin's old routine.
	   Occupancy now denotes the difference between successive layers.*/

	*sum = 0.;
	for (i=0; TRUE; i++) {
	    p2 = occ_calc(i); /* occupancy layer 'i' */
	    if (p2 < 1e-5) {
		break;
	    }
	    *sum += p2;
	} /* endfor */
	if (*sum < 1e-10) {
	    *mean = *rms = -1.;
	    break;
	} else {
	    norm = 1/(*sum);
	} /* endif */

	/* Calculate first momentum = mean value : */

	fm = 0.;
	for (i=0; TRUE; i++) {
	    p2 = occ_calc(i);
	    if (p2 < 1e-5) {
		break;
	    } /* endif */
	    fm += p2*(double)i;
	} /* endfor */
	fm *= norm;       /* Normalization */

	/* Calculate second momentum = root mean square : */

	sm = 0.;
	for (i=0; TRUE; i++) {
	    p2 = occ_calc(i);
	    if (p2 < 1e-5) {
		break;
	    } /* endif */
	    sm += p2 * sqr( (double)i - fm );
	} /* endfor */
	sm = sqrt(norm*sm);    /* Normalization */
	*mean = fm;
	*rms = sm;
	break;

    } /* endswitch */
    return (*rms);
}

/***************************************************************************/
double sqr(float x)
/***************************************************************************/

    /*
    Return square of x.
    */

{
    return(x*x);
}
