
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$$$$$$                      menu.c                         $$$$$$$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

    /*
    Set of subroutines for menu-driven programs, including:
    - command interpreters
    - input/output functions
    - file extension manipulation
    - capability to run a macro file

    Written by: Elias Vlieg
		AT&T Bell Laboratories
		510E Brookhaven National Laboratory
		Upton, NY 11973

    Address since October 1, 1990
		FOM-institute AMOLF
		Kruislaan 407
		1098 SJ Amsterdam
		The Netherlands

    Address since January 1, 1998
		Dept. of Solid State Chemistry
		University of Nijmegen
		Toernooiveld 1
		6525 ED Nijmegen
		The Netherlands
		email: vlieg@sci.kun.nl

    Last patch: bug fix on opsys() subroutine  		  06-nov-92
		adapted for OS/2, cleaned up cursor stuff 26-jan-94
		nested macro files feature		  27-jan-94
		CLI messages, debug (trace) mode          30-jan-94
		FP exception handling			  16-jun-94
                set_output() function			  17-jul-94

    */


#ifdef DLL              /* make dynamic link library under OS/2 */
#define EXPORT _export _syscall
#define INTERNAL _syscall
#else
#define EXPORT
#define INTERNAL
#endif

/***************************************************************************/
/*  Declare include files                                                  */
/***************************************************************************/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <platform.h>

#ifdef	MSDOS
#include <dos.h>
#endif

#if ( defined(MSDOS) || defined(OS2) )
#include <conio.h>
#include <signal.h>
#include <float.h>
#include <memory.h>
#include <menu.h>
#endif

#ifdef VMS
#include "menu.h"
#endif

/***************************************************************************/
/*  Define ANSI video sequences        					   */
/***************************************************************************/

#if defined (MSDOS) || defined(OS2)

#define VIDEO_REVERSE       "\033[0;1;33;44m"    /* for errors: yellow on blue */
#define VIDEO_HIGHLIGHT     "\033[0;1;35;44m"    /* magenta highlight */
#define VIDEO_TRACE         "\033[0;1;36;44m"    /* cyan debug mode */
#define VIDEO_CLIQUERY      "\033[0;1;32;44m"    /* light green query attribute */
#define VIDEO_CLIMESSAGE    "\033[0;37;44m"      /* gray on blue messages */
#define VIDEO_NORMAL        "\033[0;1;37;44m"    /* white on blue - default */

#else

#define VIDEO_REVERSE   	"\033[0;7m"       /* reverse video attribute */
#define VIDEO_HIGHLIGHT 	"\033[0;1m"       /* intensive video attribute */
#define VIDEO_TRACE     	VIDEO_HIGHLIGHT /* debug mode attribute */
#define VIDEO_CLIQUERY     	VIDEO_HIGHLIGHT /* CLI query attribute */
#define VIDEO_CLIMESSAGE        VIDEO_HIGHLIGHT /* CLI message attribute */
#define VIDEO_NORMAL            "\033[0m"       /* turn attributes off */

#endif

/***************************************************************************/
/*  Define sound macro             					   */
/***************************************************************************/

#if defined(MSDOS) && defined(__BORLANDC__)

#define BEEP sound(2000);delay(30);nosound()    /* neat beep */

#endif

#ifdef OS2

#define INCL_DOSPROCESS
#include <os2.h>
#undef COMMENT
#define BEEP DosBeep(2000,30);

#endif

#ifdef VMS

#define BEEP type_line("\a")

#endif

/***************************************************************************/
/*  Exception handling definitions     					   */
/***************************************************************************/

#ifndef VMS

#ifndef _USERENTRY
typedef         void (cdecl *EXC_HND_F) (int);
#else
typedef		void (_USERENTRY *EXC_HND_F) (int);
#endif
EXC_HND_F 	SAVED_FPE_HANDLER;
EXC_HND_F	SAVED_INT_HANDLER;
jmp_buf 	FPE_JUMPADDR;
jmp_buf		INT_JUMPADDR;

void fpe_handler(int sig);
void exc_handler(int sig);

#endif

/***************************************************************************/
/*  Define global variables                                                */
/***************************************************************************/

/* General parameters */

#if defined(MSDOS) || defined (OS2)
#define TERMBUFFER		/* Use buffer for terminal input */
#define LINES		30	/* Number of lines to remember */
#endif

#define MAXCOLUMN	132	/* Maximum screen width */

#define FILENAME_LENGTH 256	/* default file name length */

/*  Globals defining input buffer   */

#define MLINE_LENGTH 1024       /* length of input buffer */
#define TOK_LENGTH   1024       /* length of input token */

char    MLINE[MLINE_LENGTH];    /* command line input buffer */
int     MLINEPTR = 0;           /* current position in input buffer */
int     MLINEEND = 0;           /* last character in input buffer */

/* Parameters for running macro files */

typedef struct _MACRO {
    FILE 	*file;
    char        filename[FILENAME_LENGTH];
    char 	saveline[MLINE_LENGTH];
    struct 	_MACRO *root;
    } MACRO, *PMACRO;

PMACRO PACTIVE_MACRO = NULL;	/* active macro file */

/* Standard output file */

FILE		*OUTPUTFILE = NULL;
unsigned        OUTPUT2TERMINAL = TRUE;

/* Buffer used in sprintf (saves a declaration everywhere) */

char    MSTRING[1024];

int     SCREENON = TRUE;        /* Flag to denote screen output on/off */
int	TRACEON  = FALSE;       /* Flag for tracing lines in macro files */

/* text attributes: */

/***************************************************************************/
/*     internal functions:						   */
/***************************************************************************/

void    INTERNAL  cli_message(char *message);
void    INTERNAL  video_message(void);
void    INTERNAL  video_query(void);
void    INTERNAL  video_trace(void);

/***************************************************************************/
void EXPORT add_extension(char filename[], char extension[])
/***************************************************************************/

    /*
    If 'filename' has no extension then add 'extension', else do nothing.
    */

    {

    int i = 0, j = 0, has_ext = 0;

    /* Find out whether filename has an extension. */

    for (i = strlen(filename); i > 0; i--)
        {
        if (filename[i-1] == '.')
            {
            has_ext = 1;
            break;
            }
#if defined(MSDOS) || defined(OS2)
        if (filename[i-1] == '\\')
            {
            break;
            }
#endif
        }

    /* return if filename has extension */

    if (has_ext) return;

    i = strlen(filename);

    filename[i] = '.';
    i++;
    do
	{
	filename[i] = extension[j];
	i++;
	j++;
	}
    while (extension[j-1] != '\0');
    }

/***************************************************************************/
void    EXPORT clear_command(void)
/***************************************************************************/

    /*
    Reset input buffer "MLINE"
    */

    {
    MLINE[0] = '\0';
    MLINEPTR = MLINEEND = 0;
    }

/***************************************************************************/
void    EXPORT clear_screen(void)
/***************************************************************************/

    /*
    Erase screen, set cursor at home position.
    Works for both MSDOS and VMS
    */

    {

    video_normal();
    if (SCREENON)
        {
#ifdef __BORLANDC__

        clrscr();
#else
        printf("\33[2J\33[u");
#endif
        }
    }

/***************************************************************************/
void  INTERNAL  cli_message(char *message)
/***************************************************************************/

    /*
    print Command Line Interpreter message
    */

    {
    video_message();
    type_line("CLI: ");
    type_line(message);
    video_normal();
    type_line("\n");
    }

/***************************************************************************/
int    EXPORT close_macro(void)
/***************************************************************************/

    /*
    If macro file is open, close it.

    Also restore contents of saveline in MLINE, return:
	0       if nothing in saveline,
	1       if something in saveline.
    */

    {

    PMACRO root;
    int	   ret_code=0;

    if (PACTIVE_MACRO != NULL)	/* macro file active ? */
	{
	if (SCREENON)
	    {
	    sprintf(MSTRING, "Closing macro file \"%s\"",
	       PACTIVE_MACRO->filename);
	    cli_message(MSTRING);
	    }
	fclose(PACTIVE_MACRO->file);
	ret_code = restore_line(PACTIVE_MACRO->saveline);
	root = PACTIVE_MACRO->root;
	free(PACTIVE_MACRO);
	PACTIVE_MACRO = root;
	if ((PACTIVE_MACRO != NULL) && SCREENON)
	    {
	    sprintf(MSTRING, "Returning to macro file \"%s\"",
	       PACTIVE_MACRO->filename);
	       cli_message(MSTRING);
	    }
	}
    return ret_code;
    }

/***************************************************************************/
int     EXPORT cmnd_match(char command[], struct MENU menu[], int menu_length)
/***************************************************************************/
    /*
    Compares the string in 'command' with the items in the menu. Returns:
	- return-number of command if match found
	- 0 if no match found
	- -1 if more than one match found
    */

    {
    return (cmnd_match_err(command, menu, menu_length, TRUE));
    }

/***************************************************************************/
int     EXPORT cmnd_match_err(char command[], struct MENU menu[],
			     int menu_length, int err_flag)
/***************************************************************************/
    /*
    Compares the string in 'command' with the items in the menu. Returns:
	- return-number of command if match found
	- 0 if no match found
	- -1 if more than one match found
    Depending on err_flag, error messages are displayed or not
    */

    {

    int i;
    int clength;    /* number of chars in command */
    int nmatch;     /* number of matching commands */
    int cnumber;    /* "serial" number of matching command */

    /* check length of command */

    clength = strlen(command);
    if ((clength < 1) || (clength > CMND_LENGTH-1))
	{
	return(0);
	}

    /* convert command to lowercase */

    for (i = 0; i < clength; i++)
	{
	command[i] = tolower(command[i]);
	}

    /* search for a match */

    nmatch = 0;
    for (i = 0; i < menu_length; i++)
	{
	/* only check if there are enough characters supplied */
	if (clength >= menu[i].SIGNIF_CHAR)
	    {
	    if (!strncmp(command,menu[i].COMMAND,clength))
		{
		nmatch += 1;
		cnumber = i;
		}
	    }
	}

    /* return appropriate number */

    if (nmatch == 0)        /* no match */
	{
	if (err_flag == TRUE)
	     {
	     sprintf(MSTRING,"ERROR, command \"%s\" not recognized",command);
	     errtype(MSTRING);
	     if (MLINE[MLINEPTR]) {
		 sprintf(MSTRING,"Ignoring remainder of line \"",
			 MLINE[MLINEPTR]);
		 strncat(MSTRING, MLINE, 20);
		 if (strlen(MLINE) > 20)
		    strcat(MSTRING, "...\"");
		 else
		    strcat(MSTRING, "\"");
		 cli_message(MSTRING);
		 clear_command();
		 }
	     }
	return(0);
	}
    else if (nmatch == 1)   /* one match found */
	{
	return(menu[cnumber].RET_NR);
	}
    else                    /* more than one match found */
	{
	if (err_flag == TRUE)
	    {
	     if (MLINE[MLINEPTR]) {
		 sprintf(MSTRING,"Ignoring remainder of line \"",
			 MLINE[MLINEPTR]);
		 strncat(MSTRING, MLINE, 20);
		 if (strlen(MLINE) > 20)
		    strcat(MSTRING, "...\"");
		 else
		    strcat(MSTRING, "\"");
		 cli_message(MSTRING);
		 clear_command();
		 }
	    sprintf(MSTRING,"WARNING, ambiguous command \"%s\"",command);
	    errtype(MSTRING);
	    }
	return(-1);
	}

    }

/***************************************************************************/
void    EXPORT del_extension(char filename[])
/***************************************************************************/

    /*
    Delete extension from 'filename.
    */

    {

    int i = 0;

    while ((filename[i] != '.') && (filename[i] != '\0')) i++;
    filename[i] = '\0';
    }

/***************************************************************************/
void    EXPORT errtype(char message[])
/***************************************************************************/

    /*
    Type error message to screen
    */

    {

    char   errstring[200], prompt[300];
    int    terminate;
    PMACRO pactive_macro;


    if (!SCREENON) return;

    sprintf(errstring,"%s",message);
    video_reverse();
    type_line(errstring);
    video_normal();
    type_line("\n");
    BEEP;
    if (PACTIVE_MACRO != NULL)
	{
	pactive_macro = PACTIVE_MACRO; /* save global pointer */
	PACTIVE_MACRO = NULL;          /* make sure user replies */

	clear_command();
	video_query();
	sprintf(prompt, "Terminate macro file \"%s\" (%s)? ",
		pactive_macro->filename, yesnostr(TRUE));
	terminate = yesno(TRUE, prompt);
	video_normal();

	PACTIVE_MACRO = pactive_macro; /* restore global pointer */

	if (terminate == TRUE)
	    {
	    SCREENON = FALSE;
	    while (PACTIVE_MACRO != NULL) close_macro();
	    SCREENON = TRUE;
	    }
	}
    }

/***************************************************************************/
void  EXPORT  get_extension(char filename[], char extension[])
/***************************************************************************/

    /*
    Finds extension in 'filename' and puts result in 'extension'.
    The '.' is not included in the extension.
    */

    {

    int i = 0, j = 0;

    /* Search for '.' in filename */

    while ((filename[i] != '.') && (filename[i] != '\0')) i++;

    if (filename[i] == '\0') /* No extension in filename */
	{
	extension[0] = '\0';
	}
    else
	{
	i++;    /* Skip the '.' */
	do
	    {
	    extension[j] = filename[i];
	    j++;
	    i++;
	    }
	while (filename[i-1] != '\0');
	}

    }


/***************************************************************************/
int EXPORT get_int(int dflt, char prompt[])
/***************************************************************************/
    /*
    Get the next argument from the input MLINE and convert it to int.
    If no argument is present, return the default value. If an illegal
    argument is passed, reset the input MLINE and read a new value.
    */

    {

    int     value;
    char    testchar;    /* Used to trap typing errors */
    char    token[TOK_LENGTH];

    /* sscanf reads an int until the first non-valid character. In
    order to trap typing errors try to read a test character after the
    int. If this is succesfull, there was more in the token and a
    typing error was presumably made. */

    while (TRUE)    /* stay in loop until correct value read in */
	{
	if (get_token(token,prompt))
	    {
	    if (sscanf(token,"%d%c",&value,&testchar) == 1)
		{
		return(value);
		}
	    else
		{
		errtype("ERROR, illegal int value");
		clear_command();
		}
	    }
	else        /* no token on input MLINE, return default value */
	    {
	    return(dflt);
	    }
	}
    }

/***************************************************************************/
double  EXPORT get_real(double dflt, char prompt[])
/***************************************************************************/
    /*
    Get a real (i.e float or double) value. The next argument in the
    input MLINE is converted to a double. If no argument is present,
    return the default value. If an illegal argument is passed, reset
    the input MLINE and read a new value.
    The return value is a double, which is converted to float in the
    calling function if it is assigned to a float.
    */

    {

    double  value;
    char    testchar;    /* Used to trap typing errors */
    char    token[TOK_LENGTH];

    /* sscanf reads a double until the first non-valid character. In
    order to trap typing errors try to read a test-character after the
    double. If this is succesfull, there was more in the token and a
    typing error was presumably made. */

    while (TRUE)    /* stay in loop until correct value read in */
	{
	if (get_token(token,prompt))
	    {
	    if (sscanf(token,"%lf%c",&value,&testchar) == 1)
		{
		return(value);
		}
	    else
		{
		errtype("ERROR, illegal value");
		clear_command();
        	return (dflt);
		}
	    }
	else        /* no token on input MLINE, return default value */
	    {
	    return(dflt);
	    }
	}
    }


/***************************************************************************/
void    EXPORT get_string(char string[], char prompt[])
/***************************************************************************/
    /*
    Read a string from the input line. If input buffer is empty, then all
    of the input after the prompt is assumed to be part of the string.
    Otherwise, the next token is assigned to the string (either one word
    of a sequence of words enclosed in quotes).
    */

    {

    while (TRUE)
	{
	if (MLINEPTR == MLINEEND)
	    {
	    sprintf(MSTRING,"%s",prompt);
	    type_line(MSTRING);
	    get_term_line(string);
	    return;
	    }
	else
	    {
	    if (get_token(string,prompt) == 1) return;
	    }
	}
    }

/***************************************************************************/
void    EXPORT get_dstring(char string[], char dflt[], char prompt[])
/***************************************************************************/
    /*
    Read a string from the input line. If no input on line, return
    default string.
    */

    {

    int i = 0;
    int flag = 0;

    while (TRUE)
	{
	flag = get_token(string,prompt);
	if (flag == 1) return;
	if (flag == 0)
	    {
	    do
		{
		string[i] = dflt[i];
		i++;
		}
	    while (dflt[i-1] != '\0');
	    return;
	    }
	}
    }

/***************************************************************************/
int     EXPORT get_token(char token[], char prompt[])
/***************************************************************************/
    /*
    Get next lexical entity from the input MLINE. If MLINE is empty, try
    reading a new one. Then return:
	- -1 if error in token
	-  0 if still nothing on line
	-  1 if lexical entity found.
    Also handle reserved words like DEBUG, NODEBUG, ?

    */

    {

    enum i_reserved { i_NULL,
		      i_debug,
		      i_nodebug,
		      i_tron,
		      i_troff,
		      i_tofileon,
		      i_tofileoff,
		      i_prompt,
		      i_help };

#define ITEMS i_help   /* last menu item */

    static struct MENU reserved_menu[ITEMS] =
	    {
	    "debug",   5, i_debug,   "",  /* type macro commands */
	    "nodebug", 7, i_nodebug, "",  /* stop typing macro commands */
	    "tron",    4, i_tron,    "",
	    "troff",   5, i_troff,   "",
	    "tofileon",8, i_tofileon,"",  /* output to file */
	    "tofileoff",9,i_tofileoff,"", /* stop output to file */
	    "prompt",  6, i_prompt,"",    /* print prompt to output */
	    "?",       1, i_help,    ""   /* convert '?' to 'help' */
	    };



    /* local variable declarations */

    int i,done,screenon;

    /* If input buffer empty, read a new MLINE from macro file or from
    the terminal. In the latter case the prompt is displayed */

    if (MLINEPTR == MLINEEND)
	{
	clear_command();
	if (PACTIVE_MACRO != NULL)
	    {
	    done = FALSE;
	    while (!done)
		{
		if (fgets(MLINE,MLINE_LENGTH,PACTIVE_MACRO->file) == NULL)
		    {
		    /* If no items on line after macro command, display
		    prompt and wait for input */

		    if (!close_macro())
			{
			if (PACTIVE_MACRO == NULL)
			   {
			   sprintf(MSTRING,"%s",prompt);
			   type_line(MSTRING);
			   get_term_line(MLINE);
			   MLINEEND = strlen(MLINE);
			   skip_white();
			   done = TRUE;
			   }
			}
		    else
			{
			done = TRUE;	/* still something left in MLINE */
			}
		    }
		else if (MLINE[0] == '!'); /* comment in macro file */
		else    /* replace trailing '\n' by '\0' */
		    {
		    MLINEEND = strlen(MLINE)-1;
		    MLINE[MLINEEND] = '\0';
		    skip_white();
		    done = TRUE;
		    }
		}
	    }
	else
	    {
	    sprintf(MSTRING,"%s",prompt);
	    type_line(MSTRING);
	    get_term_line(MLINE);
	    MLINEEND = strlen(MLINE);
	    skip_white();
	    }

    /* show line in debug mode, if this comes from a macro file */

	if ((PACTIVE_MACRO != NULL) && (TRACEON == TRUE))
	    {
	    screenon=SCREENON;
	    SCREENON=TRUE;
	    video_trace();
	    type_line(MLINE);
	    type_line("\n");
	    video_normal();
	    SCREENON=screenon;
	    }

	}

    /* Return with value 0 if still no input on MLINE */

    if (MLINEPTR == MLINEEND)
	{
	return(0);
	}

    /*
    If first character is a double quote ("):
	The next token is a string which should be read in until the
	next double quote. Two adjacent double quotes are interpreted
	as one double quote in the string.
    Otherwise:
	Simply read until next white character.
    */

    if (MLINE[MLINEPTR] == '\"')  /* Here the trouble starts */
	{
	MLINEPTR++;
	i = 0;
	done = FALSE;
	while (!done)   /* continue until terminating " found */
	    {
	    while ((MLINE[MLINEPTR] != '\"') && (MLINEPTR != MLINEEND))
		{
		token[i] = MLINE[MLINEPTR];
		MLINEPTR++;
		i++;
		}
	    /* If next char is also a ", then put one " into token and
	    continue. Otherwise, you are done */
	    if ((MLINEPTR != MLINEEND) && (MLINE[MLINEPTR+1] == '\"'))
		{
		token[i] = MLINE[MLINEPTR];
		MLINEPTR += 2;
		i++;
		}
	    else
		{
		done = TRUE;
		}
	    }
	if (MLINEPTR == MLINEEND) /* end of MLINE reached while looking for " */
	    {
	    errtype ("ERROR, double quote (\") expected");
	    clear_command();
	    token[0] = '\0';
	    return(-1);
	    }
	else
	    {
	    MLINEPTR++;      /* skip the terminating " */
	    }
	token[i] = '\0';
	skip_white();
	return(1);
	}
    else    /* Token is not a string, read until next white char */
	{
	i = 0;
	while ((MLINE[MLINEPTR] != ' ') && (MLINE[MLINEPTR] != '\t')
		&& (MLINEPTR != MLINEEND))
	    {
	    token[i] = MLINE[MLINEPTR];
	    MLINEPTR++;
	    i++;
	    };
	token[i] = '\0';
	skip_white();

	/* check for reserved words */
	switch (cmnd_match_err(token, reserved_menu, ITEMS, FALSE))
	    {
	    case i_debug:
	    case i_tron:
	        TRACEON = TRUE;
		cli_message("DEBUG set ON");
		return (0);
	    case i_nodebug:
	    case i_troff:
	       TRACEON = FALSE;
		cli_message("DEBUG set OFF");
	       return (0);
	    case i_tofileon:
		cli_message("Output to menulist.txt");
		put_command("menulist cli_output");
		set_output("txt");
		return(0);
	    case i_tofileoff:
		set_output(NULL);
		cli_message("Output to terminal");
		return(0);
	    case i_prompt:
		sprintf(MSTRING,"%s\n",prompt);
		type_line(MSTRING);
		return(0);
	    case i_help:
		strcpy(token, "help");
		return(1);
	    default:
	       return(1);     /* no reserved word, return token */
	    }

	}
    }

#undef ITEMS


#ifndef VMS

/***************************************************************************/
void fpe_handler(int sig)
/***************************************************************************/

    /*
    Floating point exception handler
    */

{
   if ( SAVED_FPE_HANDLER != SIG_ERR &&	/* call "old" handler first */
	SAVED_FPE_HANDLER != SIG_IGN &&
	SAVED_FPE_HANDLER != SIG_DFL &&
	SAVED_FPE_HANDLER != (EXC_HND_F) fpe_handler ) {

		signal(SIGFPE, SAVED_FPE_HANDLER);
		raise(sig);

   }

   _fpreset();
   signal(SIGFPE, (EXC_HND_F) fpe_handler);

   video_reverse();
   type_line("\nERROR: An exception has occurred somewhere in the execution of the current\n");
   type_line("       command.  I cannot recover that command and continue with execution,\n");
   type_line("       but I can restart at the main command.  Be aware that no cleanup has\n");
   type_line("       been performed, we simply jumped back to the beginning of the code.\n");
   type_line("       Good Luck.\n\n");

   video_normal();
   BEEP;

   longjmp (FPE_JUMPADDR,TRUE);

}
#endif

#ifndef VMS
/***************************************************************************/
void int_handler(int sig)
/***************************************************************************/

    /*
    Ctrl-C interrupt handler
    */

{

   int prompt, stop = 0;

   PMACRO pactive_macro;

   char    token[100];
   char    saveline[MLINE_LENGTH];
   int	   screenon;
   enum i_int_menu { i_zero, i_resume, i_main, i_abort };


   static struct MENU int_menu[i_abort] =
	{

	"resume", 1, i_resume, "Resume current command",
	"main",   1, i_main,   "Goto main menu",
	"abort",  5, i_abort,  "ABORT program"
	};

   if ( SAVED_INT_HANDLER != SIG_ERR &&	/* call "old" handler first */
	SAVED_INT_HANDLER != SIG_IGN &&
	SAVED_INT_HANDLER != SIG_DFL &&
	SAVED_INT_HANDLER != (EXC_HND_F) int_handler ) {

		signal(SIGINT, SAVED_INT_HANDLER);
		raise(sig);

   }

   signal(SIGINT, (EXC_HND_F) int_handler);

   pactive_macro = PACTIVE_MACRO; /* save global pointer */
   PACTIVE_MACRO = NULL;          /* make sure user replies */

   save_line(saveline);
   clear_command();

   screenon=SCREENON;
   SCREENON = TRUE;

   video_query();

   type_line("\n");
   list_menu("CTRL-C options", int_menu, i_abort);
   while (!stop)
	{
	while (!get_token(token,"CTRL-C>"));
	switch (prompt=cmnd_match(token,int_menu,i_abort))
		{
		case -1:
			break;
		case i_zero:
			video_query();
			list_menu("CTRL-C options", int_menu, i_abort);
			break;
		case i_resume:
		case i_main:
		case i_abort:
			stop = TRUE;
			break;
		}
	}


   clear_command();
   video_normal();

   PACTIVE_MACRO=pactive_macro;


   if (prompt==i_abort) exit(2);


   if (prompt==i_resume)
      {
      SCREENON = screenon;
      restore_line(saveline);
      return;
      }


   SCREENON = FALSE;
   while (PACTIVE_MACRO != NULL) close_macro();
   SCREENON = TRUE;

   video_reverse();
   type_line("\n	    *********************************************************\n");
   type_line("            *                                                       *\n");
   type_line("            *             HYPER-SPACE WARP to main menu ...         *\n");
   type_line("            *                                                       *\n");
   type_line("            *********************************************************\n\n");
   video_normal();

   longjmp (INT_JUMPADDR,TRUE);

}
#endif

#ifndef VMS
/***************************************************************************/
int EXPORT install_exc_handler(jmp_buf jumpaddr)
/***************************************************************************/

    /*
    This function installs our exception handlers.
    Previously installed handlers are saved.
    Returns :   1 = success
		0 = error
    */

{
    SAVED_FPE_HANDLER = signal(SIGFPE, (EXC_HND_F) fpe_handler);
    if (SAVED_FPE_HANDLER == SIG_ERR) {
	errtype("Error, cannot install float exception handler");
    }
    else
    {
	memcpy(FPE_JUMPADDR, jumpaddr, sizeof(jumpaddr[0]));
    }

    SAVED_INT_HANDLER = signal(SIGINT, (EXC_HND_F) int_handler);
    if (SAVED_INT_HANDLER == SIG_ERR) {
	errtype("Error, cannot install Ctrl-C handler");
    }
    else
    {
	memcpy(INT_JUMPADDR, jumpaddr, sizeof(jumpaddr[0]));
    }

    return (SAVED_FPE_HANDLER != SIG_ERR && SAVED_INT_HANDLER != SIG_ERR);

}
#endif

/***************************************************************************/
void    EXPORT list_menu(char title[], struct MENU menu[], int length)
/***************************************************************************/

    /*
    Short version of list_menus.
    */

    {
    list_menus(title,menu,length,25);
    }

/***************************************************************************/
void    EXPORT list_menus(char title[], struct MENU menu[], int length, int scroll)
/***************************************************************************/

    /*
    List the commands and comments of a menu. The characters of the
    command that are needed for a match, are displayed in upper case.
    Scroll is the number of lines available to display the menu.
    */

    {


    int i,j,k,item,remainder,tlength,start;
    char    command[CMND_LENGTH];
    char    title_bar[MAXCOLUMN+1];
    char    dummy[10];

    /* Generate title bar: *'s everywhere except the center with the
    title of the menu surrounded by a space */

    tlength = strlen(title);
    start = (COLUMNS-tlength)/2; /* start postion of menu title in bar */
    for (i = 0; i < start-1; i++)
	{
	title_bar[i] = '*';
	}
    title_bar[i] = ' ';
    i++;
    for (j = 0; j < tlength; j++)
	{
	title_bar[i+j] = title[j];
	}
    title_bar[i+j] = ' ';
    for (i = start+tlength+1; i < COLUMNS-1; i++)
	{
	title_bar[i] = '*';
	}
    title_bar[COLUMNS-1] = '\n';
    title_bar[COLUMNS] = '\0';

    /* List commands and comments. The significant characters of each
    command are displayed in upper case */

    for (k = 0;  length-k*(scroll-2) > scroll-2; k++)
	{
	/* clear_screen(); */
	type_line(title_bar);
	for (i = 0; i < scroll-2; i++)
	    {
	    item = k*(scroll-2)+i;
	    for (j = 0; j < menu[item].SIGNIF_CHAR; j++)
		{
		command[j] = toupper(menu[item].COMMAND[j]);
		}
	    for (;j < CMND_LENGTH; j++)
		{
		command[j] = menu[item].COMMAND[j];
		}
	    sprintf(MSTRING," %-*s: %-*s\n",CMND_LENGTH-1,command,
		CMNT_LENGTH-1,menu[item].COMMENT);
	    type_line(MSTRING);
	    }
	/*type_line(title_bar);*/
	if (get_token(dummy,"More, press <return> to continue"))
	    {
	    return;
	    }
	}

    /* Type last page of menu */

    remainder = length-k*(scroll-2);
    if (remainder > 0)
	{
	/*clear_screen();*/
	type_line(title_bar);
	}
    for (i = 0; i < remainder; i++)
	{
	item = length-remainder+i;
	for (j = 0; j < menu[item].SIGNIF_CHAR; j++)
	    {
	    command[j] = toupper(menu[item].COMMAND[j]);
	    }
	for (;j < CMND_LENGTH; j++)
	    {
	    command[j] = menu[item].COMMAND[j];
	    }
	sprintf(MSTRING," %-*s: %-*s\n",CMND_LENGTH-1,command,
	    CMNT_LENGTH-1,menu[item].COMMENT);
	type_line(MSTRING);
	}
	/*type_line(title_bar);*/

    }

/***************************************************************************/
int	EXPORT nomore_commands(void)
/***************************************************************************/

    /*
    See if there are still commands on the command line buffer.
    */

    {
    if (MLINEPTR == MLINEEND) return(TRUE);
    else return(FALSE);
    }

/***************************************************************************/
void    EXPORT opsys(void)
/***************************************************************************/

    /*
    Input one command line for the operation system and execute it.
    */

    {

    char command_line[129];

    get_string(command_line,"OS>");
    if (strlen(command_line) != 0)
	    system(command_line);
    }

/***************************************************************************/
void    EXPORT put_command(char string[])
/***************************************************************************/

    /*
    Write contents of string to input MLINE
    */

    {

    int i,slength;

    /* Clean up MLINE by moving all remaining characters to left */

    if (MLINEPTR != MLINEEND)
	{
	for (i = 0; i <= MLINEEND-MLINEPTR; i++)
	    {
	    MLINE[i] = MLINE[i+MLINEPTR];
	    }
	MLINEEND = MLINEEND-MLINEPTR;
	MLINEPTR = 0;
	}
    else
	{
	clear_command();
	}

    /* Proceed only if string is not empty */

    slength = strlen(string);
    if (slength > 0)
	{

	/* If there are remaining characters in MLINE, move these towards
	left to make space for the string that has to be placed there.
	Write a space in front of these characters.
	Otherwise, write string terminator at last position */

	if (MLINEEND > 0)
	    {
	    for (i = MLINEEND; i >= 0; i--)
		{
		MLINE[i+slength+1] = MLINE[i];
		}
	    MLINE[slength] = ' ';
	    }
	else
	    {
	    MLINE[slength] = '\0';
	    }

	/* Put string at leftmost postions in MLINE */

	for (i = 0; i < slength; i++)
	    {
	    MLINE[i] = string[i];
	    }
	if (MLINEEND == 0)
	    MLINEEND = slength;
	else
	    MLINEEND += slength+1;      /* Because of inserted space */
	}

    }

/***************************************************************************/
void    EXPORT replace_extension(char filename[], char extension[])
/***************************************************************************/

    /*
    Replace the extension in 'filename' by 'extension'.
    */

    {

    int i = 0, j = 0;

    /* Put i after the '.', or at last character if no extension. */

    while ((filename[i] != '.') && (filename[i] != '\0')) i++;
    if (filename[i] == '\0') filename[i] = '.';
    i++;

    do
	{
	filename[i] = extension[j];
	i++;
	j++;
	}
    while (extension[j-1] != '\0');
    }

/***************************************************************************/
int    EXPORT restore_line(char *saveline)
/***************************************************************************/

    /*
    Restore contents of saveline in MLINE, return:
	0       if nothing in saveline,
	1       if something in saveline.
    */

    {

    int length;

    length = strlen(saveline);
    if (length == 0)
	{
	return(0);
	}
    else
	{
	sprintf(MLINE,"%s",saveline);
	MLINEPTR = 0;
	MLINEEND = length;
	return(1);
	}

    }


/***************************************************************************/
int     EXPORT run_macro(void)
/***************************************************************************/

    /*
    Open macro file from where the next input is taken. Return:
	1 if successful
	0 if not successful
    */

    {

    FILE    *mac_file;
    char    filename[FILENAME_LENGTH], filename2[FILENAME_LENGTH];
    char    *macropath, *fname;
    PMACRO  root;
    int	    found;

    /* Get macro filename and open file */

    get_string(filename,"Give name of macro file (.mac): ");
    add_extension(filename,"mac");

#if defined(OS2) || defined(MSDOS) || defined(VMS)	/* only UNIX knows the difference */
    strupr(filename);
#endif

    found = 1;
    if ((mac_file = fopen(filename,"r")) == NULL)
	{

        found = 0;

        /* if not found in current directory:
           try to find in global macro directory */

#ifndef VMS
        if (NULL != (macropath = getenv("MACROS")))
            {
            strcpy(filename2, macropath);
            if (filename[strlen(filename2)-1] != '\\')
               strcat(filename2, "\\");
            fname = strrchr(filename, '\\');
            if (fname == NULL)
                fname = filename;
            else
                fname++;
            strcat(filename2, fname);
            found = (NULL != (mac_file = fopen(filename2, "r")));
            if (found)
                {
                strcpy(filename, filename2);
#if defined(OS2) || defined(MSDOS) || defined(VMS)	/* only UNIX knows the difference */
                strupr(filename);
#endif
		}
            }
#endif

        if (!found)
            {
            sprintf(MSTRING,"Failed to open \"%s\"",filename);
            errtype(MSTRING);
            clear_command();
            return(0);
            }

        }

    if (SCREENON)
        {
        if (PACTIVE_MACRO == NULL)
            {
            sprintf(MSTRING, "Opening macro file \"%s\"",
                filename);
            }
        else
            {
            sprintf(MSTRING, "Branching to macro file \"%s\"",
                filename);
            }
        cli_message(MSTRING);
        }

    root = PACTIVE_MACRO;	/* save old macro file if any */
    if (NULL==(PACTIVE_MACRO = malloc(sizeof(MACRO)))) 	/* allocate memory for new pointer */
        {
        errtype("Cannot allocate memory");
        clear_command();
        return(0);
        }

    PACTIVE_MACRO->root = root;
    PACTIVE_MACRO->file = mac_file;
    strcpy(PACTIVE_MACRO->filename, filename);
    save_line(PACTIVE_MACRO->saveline);    /* Save possible further tokens for later use */
    clear_command();
    return(1);

    }

/***************************************************************************/
void    EXPORT save_line(char *saveline)
/***************************************************************************/

    /*
    Save non-used portion of MLINE in saveline.
    */

    {

    int i;

    i = 0;
    while (MLINE[i+MLINEPTR] != '\0')
	{
	saveline[i] = MLINE[i+MLINEPTR];
	i++;
	}
    saveline[i] = '\0';

    }

/***************************************************************************/
void    EXPORT set_cursor(int column, int row)
/***************************************************************************/

    /*
    Position cursor at given column and row.
    The ANSI escape sequences work for MSDOS, OS/2 and VMS.

    BorlandC's implentation is faster

    column = 1..80, row = 1..25   in general

    */

    {
    if (!SCREENON) return;

    while (column>COLUMNS)
	{
	column-=COLUMNS;
	row++;
	}
    if (column < 1)
	{
	column=1;
	row--;
	}

    if (row < 1)
	{
	row = 1;
	}

    if (row > ROWS)
	{
	row = ROWS-1;
	}
#ifdef __BORLANDC__
    gotoxy(column, row);
#else
    printf("\033[%1d;%1dH",row,column);
#endif
    }


/***************************************************************************/
void EXPORT set_output(char *extension)
/***************************************************************************/

    /*
    Prompt user for output file, with terminal as an option.
    Redirect global output to this file.

    Parameter:  extension
                   -  some value like "DAT" will be default extension
                   -  NULL resets output to stdout
    */

{

    char filename[128], string[250];
    FILE *output;

    if (extension == NULL)
        {
        if (OUTPUTFILE != NULL) fclose (OUTPUTFILE);
        OUTPUTFILE = NULL;
        OUTPUT2TERMINAL = TRUE;
        return;
        }

    /* Ask for filename */

    sprintf(string,"Filename (.%s) (type 't' or <return> for terminal): ",
	extension);
    get_string(filename,string);
    if (!filename[0])
	{
	OUTPUT2TERMINAL = TRUE;
	}
    else if (((filename[0] == 't') || (filename[0] == 'T')) &&
	(filename[1] == '\0'))
	{
	OUTPUT2TERMINAL = TRUE;
	}
    else
	{
	add_extension(filename,extension);
	if ((output = fopen(filename,"w")) == NULL)
	    {
            OUTPUT2TERMINAL = TRUE;
            OUTPUTFILE = NULL;
	    sprintf(string,"Error, failed to open '%s'",filename);
	    errtype(string);
	    clear_command();
	    return;
	    }
	else
	    {
	    get_string(string,"Comments: ");
            if (string[0]) fprintf(output,"! %s\n",string);
	    OUTPUT2TERMINAL = FALSE;
            OUTPUTFILE = output;
	    }
	}
}

/***************************************************************************/
void    EXPORT set_scroll(int start_line, int end_line)
/***************************************************************************/

    /*
    Define a scrolling region on the screen.
    */

    {
#ifdef VMS
    int	start,end;
    start = start_line;
    end = end_line;
    lib$set_scroll(&start,&end);
#elif defined(__BORLANDC__)
    window(1, start_line, COLUMNS, end_line);
#else
    start_line; end_line;
#endif
    }

/***************************************************************************/
void    EXPORT skip_white(void)
/***************************************************************************/

    /*
    Set pointer of input MLINE to next non-white character
    */

    {
    while (((MLINE[MLINEPTR] == ' ') || (MLINE[MLINEPTR] == '\t')) &&
	    (MLINEPTR != MLINEEND))
	{
	MLINEPTR++;
	}
    }

/***************************************************************************/
void    EXPORT string_tolower(char string[])
/***************************************************************************/

    /*
    Convert string to lower case.
    */

    {
    int i = 0;

    while (string[i] != '\0')
	{
	string[i] = tolower(string[i]);
	i++;
	}
    }

/***************************************************************************/
void    EXPORT type_line(char tline[])
/***************************************************************************/

    /*
    Print line to standard output or output file if opened.
    */

    {

    if (OUTPUTFILE != NULL)
        {
	fprintf(OUTPUTFILE, "%s", tline);
        }

    else

    if (SCREENON) printf("%s",tline);

    }

/***************************************************************************/
void    EXPORT type_list(char tline[], int length)
/***************************************************************************/

    /*
    Print line to standard output or output file if opened, but avoid
    typing too many lines on terminal. After 'length' lines, a <return> has
    to be given to continue output.
    When length=0, the routine is initialized.
    */

    {
    static int counter = 0,all = FALSE;
    char	dummy[10];

    if (length == 0)
	{
	counter = 0;
	all = FALSE;
        }
    else if (all) type_line(tline);
    else
	{
	type_line(tline);
	counter++;
	if ((counter == length-1) && (OUTPUTFILE == NULL))
	    {
	    if ((get_token(dummy,"Type <return> to continue ")) != 0)
		all = TRUE;
	    //if (dummy[0] == 'a') all = TRUE;
	    counter = 0;
	    }
	}
    }

/***************************************************************************/
void    EXPORT typeoff(void)
/***************************************************************************/

    /*
    Switch off output to screen.
    */

    {
    SCREENON = FALSE;
    }

/***************************************************************************/
void    EXPORT typeon(void)
/***************************************************************************/

    /*
    Switch on output to screen.
    */

    {
    SCREENON = TRUE;
    }

/***************************************************************************/
void    EXPORT video_highlight(void)
/***************************************************************************/

    /*
    Switch to highlight text mode.
    */

    {
    type_line(VIDEO_HIGHLIGHT);
    }


/***************************************************************************/
void    EXPORT video_normal(void)
/***************************************************************************/

    /*
    Switch to normal text mode.
    */

    {
    type_line(VIDEO_NORMAL);
    }

/***************************************************************************/
void  INTERNAL  video_message(void)
/***************************************************************************/

    /*
    Switch to CLI message text mode.
    */

    {
    type_line(VIDEO_CLIMESSAGE);
    }

/***************************************************************************/
void  INTERNAL  video_query(void)
/***************************************************************************/

    /*
    Switch to CLI query text mode.
    */

    {
    type_line(VIDEO_CLIQUERY);
    }

/***************************************************************************/
void  INTERNAL  video_trace(void)
/***************************************************************************/

    /*
    Switch to trace (debug) text mode.
    */


    {
    type_line(VIDEO_TRACE);
    }

/***************************************************************************/
void    EXPORT video_reverse(void)
/***************************************************************************/

    /*
    Switch to reverse text mode.
    */

    {
    type_line(VIDEO_REVERSE);
    }


/***************************************************************************/
int     EXPORT yesno(int dflt, char prompt[])
/***************************************************************************/

    /*
    Returns:
	- TRUE if input is some form of YES
	- FALSE if input is some form of NO
	- default value if no input on MLINE.
    Retries if error in input
    */

    {

    static struct MENU yesno_menu[2] =
	{
	"yes",  1,1,"Affirmative answer",
	"no",   1,2,"Negative answer"
	};

    char    token[100];

    while (1)
	{

	/* Return default value if no input on line */

	if (!get_token(token,prompt)) return(dflt);
	switch (cmnd_match(token,yesno_menu,2))
	    {
	    case 0:
		errtype ("Warning: command not recognized");
		break;
	    case 1:
		return(TRUE);
	    case 2:
		return(FALSE);
            default:
                return(dflt);
	    }
	}
    }

/****************************************************************************/
char* EXPORT yesnostr(int flag)
/****************************************************************************/

/*
   returns "YES" if flag == TRUE
   else    "NO"
*/

{

   static char answerstr[2][4] = {"YES","NO"};

   if (flag == TRUE)
	return answerstr[0];
   else
	return answerstr[1];
}

/*
The following subroutines are used to facilitate input on a PC, it enables
one to scroll back to a previously used command line, and to modify it.
For other operating systems, only one line in 'get_term_line' is important.
*/

#ifdef	TERMBUFFER
/***************************************************************************/
void    EXPORT cursor_big(int flag)
/***************************************************************************/

    /*
    Set cursor to big for insert mode
    */
#ifdef __BORLANDC__	/* this is what we call a decent compiler */

    {
       _setcursortype(flag ? _SOLIDCURSOR : _NORMALCURSOR);
    }
#else			/* Microsoft deals it more delicately ... */

    {
    union REGS regs;

    regs.h.ah = 1;              /*  select cursor size function  */
    regs.h.cl = 7;              /*  bottom row of cursor  */
    if (flag)
	regs.h.ch = 4;          /*  larger cursor for insert mode  */
    else
	regs.h.ch = 6;          /*  normal cursor  */
    int86 (0x10, &regs, &regs);
    }
#endif

/***************************************************************************/
void    EXPORT get_cursor_position (int *row, int *column)
/***************************************************************************/

    /*
    Get the row and column of current cursor location
    */

#ifdef __BORLANDC__
    {
    *row = wherey()-1;
    *column = wherex()-1;
    }
#else
    {
    union REGS regs;

    regs.h.ah = 3;             /*  select function  */
    regs.h.bh = 0;             /*  video page number  */
    int86 (0x10, &regs, &regs);
    *row = (int) regs.h.dh;
    *column = (int) regs.h.dl;
    }
#endif

/***************************************************************************/
void    EXPORT set_cursor_position (int row, int col)
/***************************************************************************/

    /*
    Set cursor at specified row and column
    */

   {
#ifndef __BORLANDC__
    union REGS regs;
#endif

    if (!SCREENON) return;

    while (col>=COLUMNS)
	{
	col-=COLUMNS;
	row++;
	}
    if (col < 0)
	{
	col=0;
	row--;
	}

    if (row < 0)
	{
	row = 0;
	}

    if (row >= ROWS)
	{
	row = ROWS-1;
	}

#ifdef __BORLANDC__
    gotoxy(col+1, row+1);
#else
    regs.h.ah = 2;            /*  select function    */
    regs.h.bh = 0;            /*  video page number  */
    regs.h.dh = row;
    regs.h.dl = col;
    int86 (0x10, &regs, &regs);
#endif
    }

#endif /* #ifdef TERMBUFFER */

#ifdef __BORLANDC__
/***************************************************************************/
int     EXPORT get_rows(void)
/***************************************************************************/
    /*
    Get number of rows on screen
    */
    {

    struct text_info ti;

    gettextinfo(&ti);
    return (ti.screenheight);

    }

/***************************************************************************/
int     EXPORT get_columns(void)
/***************************************************************************/
    /*
    Get number of columns on screen
    */
    {

    struct text_info ti;

    gettextinfo(&ti);
    return (ti.screenwidth>MAXCOLUMN ? MAXCOLUMN : ti.screenwidth);

    }

#endif /* #ifdef __BORLANDC__ */

/***************************************************************************/
void    EXPORT get_term_line (char *term_line)
/***************************************************************************/

    /*
    Input a line from the terminal.
    This subroutine stores several command lines in a buffer, which can
    be recalled  and/or modified by using the arrow/edit keys.
    */

    {
#ifdef	TERMBUFFER
    static char old_term_line[LINES][MAXCOLUMN+1];
    static int ocp = 0;    /*  ocp points to next old_term_line to be used  */
    static int last_oc;
    int insert = FALSE;
    int printline = TRUE;
    char c;
    char last_arrow = '\0';
    char *cursor, *end, *p, *q;
    int len = 0;
    int start_row,start_column;

    end = cursor = term_line;
    get_cursor_position(&start_row,&start_column);
    *cursor = '\0';   /*  reset command line string  */
    while (len < MLINE_LENGTH - 1)
	{      /*  main loop  */
/*	while (!kbhit ()); */
	if ((c = (char) getch ()) != 0)  /*  some special keys return */
	    {                            /*  a zero first  */
	    switch (c)
		{
                case 3:			   /*  CTRL-C */
                    int_handler(SIGINT);
                    break;
		case 13:                   /*  ENTER        */
		    strcpy (old_term_line[ocp++ % LINES], term_line);
		    last_oc = ocp % LINES;
		    cursor_big(FALSE);
		    type_line("\n");
		    return;
		case 8:                    /*  BACKSPACE    */
		    if (cursor == term_line) break;
		    for (p = cursor--, q = cursor; p < end; p++, q++)
			*q = *p;
		    *--end = '\0';
		    len--;
		    break;
		case 27:                   /*  ESCAPE       */
		    end = cursor = term_line;
		    *cursor = '\0';    /*  erase line   */
		    len = 0;
		    set_cursor_position(start_row,start_column);
		    sprintf (MSTRING,"%*s",COLUMNS-start_column-1," ");
		    type_line(MSTRING);
		    insert = FALSE;
		    cursor_big(FALSE);
		    break;
		default:                   /*  CHARACTER    */
		    if (!insert)
			{
			if (cursor == end)
			    {
			    len++;
			    *++end = '\0';
			    }
			*cursor++ = c;
			sprintf(MSTRING,"%c",c);
			type_line(MSTRING);
			printline = FALSE;
			}
		    else
			{
			*++end = '\0';
			for (p = end-2*sizeof(char), q = end-sizeof(char);
			    p >= cursor; p--, q--)
			    *q = *p;
			*cursor++ = c;
			len++;
			}
		}
	    }
	else
	    {             /*  handles keys which return 0 first  */
	    switch (getch())
		{
		case 71:                   /*  HOME         */
		    cursor = term_line;
		    set_cursor_position(start_row,start_column);
		    printline = FALSE;
		    break;
		case 72:                   /*  UP ARROW     */
		    if (last_arrow == 'd') last_oc--;
		    if (last_oc == 0) last_oc = LINES;
		    while (*old_term_line[--last_oc] == '\0')
			if (last_oc == 0) break;
		    strcpy (term_line, old_term_line[last_oc]);
		    len = strlen (term_line) / sizeof (char);
		    cursor = end = term_line + len;
		    set_cursor_position(start_row,start_column);
		    /*  erase to end of line  */
		    sprintf (MSTRING,"\%*s",COLUMNS-start_column-1," ");
		    type_line(MSTRING);
		    last_arrow = 'u';
		    insert = FALSE;
		    cursor_big(FALSE);
		    break;
		case 80:                   /*  DOWN ARROW   */
		    if (last_arrow == 'u') last_oc++;
		    if (last_oc == LINES)
			last_oc = 0;
		    else
			while (*old_term_line[last_oc++] == '\0')
			    if (last_oc == LINES)
				{
				last_oc = 0;
				break;
				}
		    if (last_oc != 0) last_oc--;
		    strcpy (term_line, old_term_line[last_oc++]);
		    len = strlen (term_line) / sizeof (char);
		    cursor = end = term_line + len;
		    set_cursor_position(start_row,start_column);
		    sprintf (MSTRING,"\%*s",COLUMNS-start_column-1," ");
		    type_line(MSTRING);
		    last_arrow = 'd';
		    insert = FALSE;
		    cursor_big(FALSE);
		    break;
		case 75:                   /*  LEFT ARROW   */
		    cursor--;
		    if (cursor < term_line) cursor = term_line;
		    set_cursor_position(start_row,
			cursor-term_line+start_column);
		    printline = FALSE;
		    break;
		case 77:                   /*  RIGHT ARROW  */
		    cursor++;
		    if (cursor > end) cursor = end;
		    set_cursor_position(start_row,
			cursor-term_line+start_column);
		    printline = FALSE;
		    break;
		case 79:                   /*  END          */
		    cursor = end;
		    set_cursor_position(start_row,
			cursor-term_line+start_column);
		    printline = FALSE;
		    break;
		case 82:                   /*  INSERT       */
		    insert ^= 1;
		    cursor_big(insert);
		    break;
		case 83:                   /*  DELETE       */
		    if (cursor == end) break;
		    for (p = cursor+sizeof(char), q = cursor;
			p < end; p++, q++)
			*q = *p;
		    *--end = '\0';
		    len--;
		    break;
		}
	    }
	if (!printline)
	    {
	    printline = TRUE;
	    }
	else
	    {
	    if (len+start_column > COLUMNS)
		{
		start_column = 0;
		type_line("\n");
		}
	    set_cursor_position (start_row,start_column);

	    /* Print an extra space in case a character was deleted */

	    sprintf(MSTRING,"%s ",term_line);
	    type_line(MSTRING);
	    set_cursor_position (start_row,cursor-term_line+start_column);
	    }
	}
    type_line("\n");
    return;
#else
    gets(term_line);
    return;
#endif
    }
