/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                         fit.c                                 $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: fit.c,v 1.5 2001/07/09 12:40:26 svensson Exp $

****************************************************************************/
/*
Update 08/06/2001 E. Vlieg (vlieg@sci.kun.nl)
                  Change fit file headers from HWHM to FWHM.
                  Changed setfpar menu to follow Rainer's scheme.
Update 24/01/2001 R. Wilcke (wilcke@esrf.fr)
		  set_anafitpar(): determine length of setfpar_menu (parameter
		  setfpar_length) dynamically.
Update 09/10/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Change guess_fit for the use of FWHM instead of HWHM.
Update 06/10/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Allow exit from fit parameter menu without actual fit.
		  Include transmission correction.
*/
/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define	EXT extern
#include "ana.h"
#undef EXT
#include <lsqfit.h>

/***************************************************************************/
void    fit(void)
/***************************************************************************/

    /*
    Fit a spectrum.
    */

    {

    int 	i, outspec;
    int 	freepar[MAXPAR],nfree;
    float 	penalty[MAXPAR];
    float       chisqr;
    float       *a,*dyda;
    float	x,*y;
//    int 	derivatives;
    float	area, polarization, lorentz, correction, intercept, f, ferr;
    float	transmission,int_tot,int_tot_err;

    /* Enable fit parameter range checking */

    for (i = 0; i < MAXPAR; i++) penalty[i] = 10000;

    /* Choose a theory and get in- and output spectra */

    theory_type();
    get_inoutspec(&FITSPEC,&outspec);
    FITOUTSP = outspec;
    if  (DATA[FITSPEC].NPNT < 2)
	{
	errtype("ERROR, too few datapoints in spectrum");
	clear_command();
	return;
	}
    if (fabs(DATA[FITSPEC].X[DATA[FITSPEC].NPNT-1]-DATA[FITSPEC].X[0])
	< 1e-6)
	{
	errtype("ERROR, x-values in spectrum are constant");
	clear_command();
	return;
	}

    /* Get starting parameters of fit */

    if (!set_anafitpar()) return; /* stop here if no actual fit is
				     requested */

    /* Compute weight of data points */

    if (WEIGHTMODE == 0)
	{
	for (i = 0; i < DATA[FITSPEC].NPNT; i++)
	    {
	    if (DATA[FITSPEC].E[i] > 1e-5)
		WEIGHT[i] = 1/sqr(DATA[FITSPEC].E[i]);
	    else
		WEIGHT[i] = 0;
	    }
	}
    else if (WEIGHTMODE == 1)
	{
	for (i = 0; i < DATA[FITSPEC].NPNT; i++)
	    WEIGHT[i] = 1;
	}

    /* Get number of free parameters and make array with free parameter
       numbers */

    nfree = 0;
    for (i = 0; i < TH.NPAR; i++)
	{
	if (!TH.FIX[i])
	    {
	    freepar[nfree] = i+1;
	    nfree++;
	    }
	}

    /* Do the fit */

    if (!levenberg(DATA[FITSPEC].X,DATA[FITSPEC].Y,WEIGHT,DATA[FITSPEC].NPNT,
		   TH.PAR,TH.NPAR,freepar,nfree,TH.ERR,TH.MIN,TH.MAX,penalty,
		   &chisqr,ffit,1)) return;

    /* Show the results */

    sprintf(STRING,"Results of fit using %s\n",TH.TITLE);
    type_line(STRING);
    for (i = 0; i < TH.NPAR; i++)
	{
	sprintf(STRING,"%*s = %14.6e +/- %13.5e\n",
		TH_LENGTH,TH.NAME[i],TH.PAR[i],TH.ERR[i]);
	type_line(STRING);
	}
    CHISQR = chisqr/(DATA[FITSPEC].NPNT-nfree+1e-10);   /* Save chisqr */
    sprintf(STRING,"chisqr = %9.3f, normalized = %10.4f\n",
	    chisqr,CHISQR);
    type_line(STRING);

    /* Show the integrated intensities of the peaks and the structure
       factors if appropriate */

    int_tot = int_tot_err = 0;
    (*F_PTR)(x,a,y,dyda,PEAK);
    for (i = 0; i < TH.NPEAK; i++)
	{
	int_tot += TH.PEAKINT[i];
	int_tot_err += sqr(TH.PEAKERR[i]);
	sprintf(STRING,"Integrated intensity peak %1d: %10.2f +/- %10.2f\n",
		i+1,TH.PEAKINT[i],TH.PEAKERR[i]);
	type_line(STRING);
	if (DATARUN)
	    {
	    comp_cor(&lorentz, &polarization, &intercept,
		     &area, &transmission);
	    correction = lorentz*area*polarization*intercept*
		transmission/STEPSIZE;
	    if (TH.PEAKINT[i] < 0)
		{
		f = 0;
		ferr = 1;
		}
	    else
		{
		f = sqrt(TH.PEAKINT[i]*correction);
		ferr = sqrt(sqr(f)+TH.PEAKERR[i]*correction)-f;
		}
	    sprintf(STRING,
		    "Structure factor peak %1d    : %10.2f +/- %10.2f\n", i+1,f,ferr);
	    type_line(STRING);
	    }
	}

    /* If more than one peak, also show totals */

    if (TH.NPEAK > 1)
	{
	sprintf(STRING,"Total integrated intensity   : %10.2f +/- %10.2f\n",
		int_tot,sqrt(int_tot_err));
	type_line(STRING);
	if (DATARUN)
	    {
	    if (int_tot < 0)
		{
		f = 0;
		ferr = 1;
		}
	    else
		{
		f = sqrt(int_tot*correction);
		ferr = sqrt(sqr(f)+int_tot_err*correction)-f;
		}
	    sprintf(STRING,
		    "Total structure factor       : %10.2f +/- %10.2f\n",
		    f,sqrt(ferr));
	    type_line(STRING);
	    }
	}

    /* If required, write fit results to file */

    if (OUTFIT)
	{
	if ((!FITSWRITTEN) && (TH.NPAR == 5)) /* only for Gauss and Lorentzian */
	    {
	    fprintf(FITFILE,"%5s %6s %6s %6s %12s %12s %12s %12s %12s %8s %7s %7s %7s %18s\n",
		    "run#","H","K","L","const.bg.","slope bg.","center","height",
		    "FWHM","chisqr","Int.int","F","dF","type");
	    }
	if ((!FITSWRITTEN) && (TH.NPAR == 8)) /* for double Gauss/Lorentzian */
	    {
	    fprintf(FITFILE,
		    "%5s %6s %6s %6s %12s %12s %12s %12s %12s %12s %12s %12s %8s %7s %7s %7s %7s %7s %7s %7s %18s\n",
		    "run#","H","K","L","const.bg.","slope bg.","center1","height1","FWHM1",
		    "center2","heigth2","FWHM2","chisqr","Int.int","F","dF","F1","dF1","F2","dF2","type");
	    }
	FITSWRITTEN++;
	fprintf(FITFILE,"%5d %6.2f %6.2f %6.2f ",
		SCANNR,
		HMIL*T11+KMIL*T12+LMIL*T13,
		HMIL*T21+KMIL*T22+LMIL*T23,
		HMIL*T31+KMIL*T32+LMIL*T33);
	for (i = 0; i < TH.NPAR; i++)
	    {
	    fprintf(FITFILE,"%12.4e ",TH.PAR[i]);
	    }
	fprintf(FITFILE,"%8.2f %7.2f %7.2f ",CHISQR,int_tot,f);
	if (TH.NPAR == 8)
	    {
	    fprintf(FITFILE,"%7.2f ",sqrt(ferr));
	    f = sqrt(TH.PEAKINT[0]*correction);
	    ferr = sqrt(sqr(f)+TH.PEAKERR[0]*correction)-f;
	    fprintf(FITFILE,"%7.2f %7.2f ",f,ferr);
	    f = sqrt(TH.PEAKINT[1]*correction);
	    ferr = sqrt(sqr(f)+TH.PEAKERR[1]*correction)-f;
	    fprintf(FITFILE,"%7.2f %7.2f %18s\n",f,ferr,TH.TITLE);
	    }
	else
	    {
	    fprintf(FITFILE,"%7.2f %18s\n",ferr,TH.TITLE);
	    }
	}

    /* Make theoretical spectrum */

    DATA[outspec].NPNT = DATA[FITSPEC].NPNT;
    DATA[outspec].X[0] = DATA[FITSPEC].X[0];
    (*F_PTR)(DATA[outspec].X[0],TH.PAR,&DATA[outspec].Y[0],dyda,CALC);
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    DATA[outspec].E[0] = 1.;
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	DATA[outspec].X[i] = DATA[FITSPEC].X[i];
	(*F_PTR)(DATA[outspec].X[i],TH.PAR,&DATA[outspec].Y[i],dyda,CALC);
	DATA[outspec].E[i] = 1.;
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}

    /* Generate title */

    sprintf(DATA[outspec].TITLE,"Fit of %1d using %s",FITSPEC+1,TH.TITLE);
    type_header(1);

    }

/***************************************************************************/
void    ffit(float x, float a[], float *y, float dyda[], int na,
	     int derivatives)
/***************************************************************************/

    /*
    Function called by fitting routines to compute y-values and derivatives
    with respect to fitting parameters.
    This general routine makes a call to the specific theory used for
    fitting.
    */

    {
    if (derivatives)
	(F_PTR)(x,a,y,dyda,DERV);
    else
	(F_PTR)(x,a,y,dyda,CALC);

    na = na;
    }

/***************************************************************************/
void    guess_fit(int nspec, float *background, float *slope,
		  float *position, float *height, float *width)
/***************************************************************************/

    /*
    Estimates fit parameters of a spectrum with a single curve
    */

    {

    float	xleft,xright,yleft,yright,xmiddle,maxy;
    float       average,half_value,xhalf1,xhalf2;
    int         i,ipos,n;

    if (DATA[nspec].NPNT < 2*NLEVELING)
	{
	errtype("ERROR, too few datapoints for guess");
	return;
	}

    /* Determine average x and y values on the left and right sides of
       the spectrum */

    xleft=yleft=xright=yright=0.;
    for (i = 0; i < NLEVELING; i++)
	{
	yleft += DATA[nspec].Y[i];
	xleft += DATA[nspec].X[i];
	}
    yleft /= NLEVELING;
    xleft /= NLEVELING;
    for (i = DATA[nspec].NPNT-NLEVELING; i < DATA[nspec].NPNT; i++)
	{
	yright += DATA[nspec].Y[i];
	xright += DATA[nspec].X[i];
	}
    yright /= NLEVELING;
    xright /= NLEVELING;

    /* Deterimine slope and middle of the spectrum */

    *slope = (yright-yleft)/(xright-xleft+1e-10);
    xmiddle = (xleft+xright)/2;

    /* Store leveled curve in temporary spectrum */

    for (i = 0; i < DATA[nspec].NPNT; i++)
	{
	XD[i] = DATA[nspec].X[i];
	YD[i] = DATA[nspec].Y[i]-(DATA[nspec].X[i]-xmiddle)*(*slope);
	}

    /* Determine x and y value of maximum in leveled spectrum, plus the
       average */

    maxy = average = YD[0];
    ipos = 0;
    for (i = 1; i < DATA[nspec].NPNT; i++)
	{
	average += YD[i];
	if (YD[i] > maxy)
	    {
	    maxy = YD[i];
	    *position = XD[i];
	    ipos = i;
	    }
	}
    average /= DATA[nspec].NPNT;

    /* Find background */

    *background = 0.;
    n = 0;
    for (i = 0; i < DATA[nspec].NPNT; i++)
	{
	if (DATA[nspec].Y[i] < average)
	    {
	    n++;
	    *background += DATA[nspec].Y[i];
	    }
	}
    *background /= n;

    /* Find x-values corresponding to halfwidth and compute full width*/

    half_value = 0.5*(maxy+*background);
    for (i = 0; i <= ipos; i++)
	{
	if (YD[i] > half_value) break;
	}
    xhalf1 = XD[i];
    for (i = ipos; i < DATA[nspec].NPNT; i++)
	{
	if (YD[i] < half_value) break;
	}
    xhalf2 = XD[i];
    *width = fabs(xhalf2-xhalf1);

    /* Compute height */

    *height = maxy-*background;

    }

/***************************************************************************/
int    set_anafitpar(void)
/***************************************************************************/

    /*
    Give starting values to fit parameters, and set them fixed or loose.
    */

    {

    /* define setfitpar_menu */

    static struct   MENU setfpar_menu[] =
	{
	"value",     1,  1,  "Set parameter value",
	"lower",     3,  2,  "Set lower bound on parameter range",
	"upper",     1,  3,  "Set upper bound on parameter range",
	"fix",       1,  4,  "Fix a parameter value",
	"loose",     2,  5,  "Make parameter free",
	"guess",     1,  6,  "Estimate fitting parameters",
	"midpoint",  1,  7,  "Set peak position to center of scan",
	"list",      1, 10,  "List parameter values",
	"help",      1, 20,  "Display menu",
	"run",       2, 21,  "Start fit",
	"return",    1, 30,  "Return to main menu",
	};

    /* number of commands in setfitpar menu */
    int     setfpar_length = sizeof(setfpar_menu) / sizeof(setfpar_menu[0]);

    int     stop = FALSE;
    char    token[100];
    int     parameter,i;
    float   gbackground,gslope,gposition,gheight,gwidth;

    put_command("list");
    while (!stop)
	{
	while (!get_token(token,"ANA.FIT.PAR>"));
	switch (cmnd_match(token,setfpar_menu,setfpar_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		parameter = get_int(1,"Parameter number to change [1]: ");
		if ((parameter < 1) || (parameter > TH.NPAR))
		    {
		    errtype("ERROR, parameter out of range");
		    clear_command();
		    }
		else
		    {
		    parameter -= 1;
		    sprintf(STRING,"%s [%7.3f]: ",TH.NAME[parameter],
			    TH.PAR[parameter]);
		    TH.PAR[parameter] = get_real(TH.PAR[parameter],STRING);
		    }
		break;
	    case 2:
		parameter = get_int(1,"Parameter number to change [1]: ");
		if ((parameter < 1) || (parameter > TH.NPAR))
		    {
		    errtype("ERROR, parameter out of range");
		    clear_command();
		    }
		else
		    {
		    parameter -= 1;
		    sprintf(STRING,"Lower bound of %s [%7.3f]: ",
			    TH.NAME[parameter],TH.MIN[parameter]);
		    TH.MIN[parameter] = get_real(TH.MIN[parameter],STRING);
		    }
		break;
	    case 3:
		parameter = get_int(1,"Parameter number to change [1]: ");
		if ((parameter < 1) || (parameter > TH.NPAR))
		    {
		    errtype("ERROR, parameter out of range");
		    clear_command();
		    }
		else
		    {
		    parameter -= 1;
		    sprintf(STRING,"Upper bound of %s [%7.3f]: ",
			    TH.NAME[parameter],TH.MAX[parameter]);
		    TH.MAX[parameter] = get_real(TH.MAX[parameter],STRING);
		    }
		break;
	    case 4:
		parameter = get_int(1,"Parameter to be fixed [1]: ");
		if ((parameter < 1) || (parameter > TH.NPAR))
		    {
		    errtype("ERROR, parameter out of range");
		    clear_command();
		    }
		else
		    {
		    TH.FIX[parameter-1] = TRUE;
		    }
		break;
	    case 5:
		parameter = get_int(1,"Parameter to let free [1]: ");
		if ((parameter < 1) || (parameter > TH.NPAR))
		    {
		    errtype("ERROR, parameter out of range");
		    clear_command();
		    }
		else
		    {
		    TH.FIX[parameter-1] = FALSE;
		    }
		break;
	    case 6:
		guess_fit(FITSPEC,&gbackground,&gslope,&gposition,
			  &gheight,&gwidth);
		if (!TH.FIX[0]) TH.PAR[0] = gbackground;
		if (!TH.FIX[1]) TH.PAR[1] = gslope;
		if (!TH.FIX[2]) TH.PAR[2] = gposition;
		if (!TH.FIX[3]) TH.PAR[3] = gheight;
		if (!TH.FIX[4]) TH.PAR[4] = gwidth;

		/* Start with a Lorentzian for power-law line shape */

		if ((F_PTR == power) && (!TH.FIX[5])) TH.PAR[5] = 2;
		put_command("list");
		break;
	    case 7:
		TH.PAR[2] = DATA[FITSPEC].X[(DATA[FITSPEC].NPNT-1)/2];
		break;
	    case 10:
		clear_screen();
		type_header(1);
		set_cursor(1,MAX_SPECTRUM+4);
		sprintf(STRING,"%2s %-30s %12s %12s %12s\n",
			"#","name","start","lower","upper");
		type_line(STRING);
		for (i = 0; i < TH.NPAR; i++)
		    {
		    sprintf(STRING,"%2d %-30s %12.4f %12.4f %12.4f ",
			    i+1,TH.NAME[i],TH.PAR[i],TH.MIN[i],TH.MAX[i]);
		    type_line(STRING);
		    if (TH.FIX[i])
			type_line(" fixed\n");
		    else
			type_line(" loose\n");
		    }
		break;
	    case 20:
		list_menu("SET FITTING PARAMETERS",
			  setfpar_menu,setfpar_length);
		break;
	    case 21:
		stop = TRUE;
		return(1);
	    case 30:
		stop = TRUE;
		return(0);

	    }
	}
    }
