/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                         fit.c                                 $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/*
Update 14/08/2005 R. Wilcke (wilcke@esrf.fr)
                  fit(): prevent buffer overflow in DATA[outspec].TITLE;
                  fit() and set_anafitpar(): change code for new quadratic
                  background parameter, using the new TH.NBCK theory member;
                  fit(): add missing theory types to the print statements for
                  the output file;
                  set_anafitpar(): make "parameter" static and use its last
                  value in the command prompts.
Update 10/06/2005 R. Wilcke (wilcke@esrf.fr)
                  set_anafitpar(): use new enumerated variable "i_setfpar" for
                  the indices of the menu;
                  set_anafitpar(): rearrange structure of "switch" statement and
                  logic of the "while" loop over the menu items;
                  set_anafitpar(): use the full terminal window to list the
                  parameter values, and restore split window when leaving the
                  routine.
Update 27/04/2005 R. Wilcke (wilcke@esrf.fr)
                  set_anafitpar(): correct error in obtaining the left FWHM
                  point and the maximum value of a peak with the cursor.
Update 27/04/2004 R. Wilcke (wilcke@esrf.fr)
                  made additional text after "#endif" a comment.
Update 30/07/2003 R. Wilcke (wilcke@esrf.fr)
                  set_anafitpar(): correct error in obtaining the x-coordinate
                  of the left FWHM point of a peak with the cursor.
Update 20/02/2003 R. Wilcke (wilcke@esrf.fr)
                  fit(): changed output format for "Total integrated intensity"
                  slightly to align it with the other fit output.
Update 19/02/2003 R. Wilcke (wilcke@esrf.fr)
                  provide default values for spectrum numbers in all routines.
Update 18/02/2003 R. Wilcke (wilcke@esrf.fr)
                  fit(): first put and then get a (dummy) integer in the input
                  buffer to consume the fit parameters written by P. Howes -
                  they may otherwise mess up the command processing elsewhere.
Update 16/04/2002 O. Svensson (svensson@esrf.fr)
                  Renamed getPlotcursorPoints to getPlotcursorPoint,
                  changed the arguments.
Update 25/03/2002 R. Wilcke (wilcke@esrf.fr)
                  set_anafitpar(): changed abbreviations for menu commands to
                  the agreed set.
Update 25/09/2001 O. Svensson (svensson@esrf.fr)
                  Changed PLOT_CURSORPOINTS to getPlotcursorPoints().
Update 09/10/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Change guess_fit for the use of FWHM instead of HWHM.
Update 06/10/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Allow exit from fit parameter menu without actual fit.
		  Include transmission correction.
Update 03/10/2000 O. Svensson (svensson@esrf.fr)
                  Added cursor input method for starting values for
                  peak fitting.
Update 29/05/2000 R. Wilcke (wilcke@esrf.fr)
                  fit(): add all fit parameters for exponential, Pseudo-Voigt
                  and slit functions to the output file.
Update 24/05/2000 R. Wilcke (wilcke@esrf.fr)
                  fit(): change output format for "L" value and reorganize the
                  fprintf() calls;
                  make HMIL, KMIL and LMIL arrays with dimension MAX_SPECTRUM.
*/

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define	EXT extern
#include "ana.h"
#undef EXT
#include <lsqfit.h>

/***************************************************************************/
void    fit(void)
/***************************************************************************/

    /*
    Fit a spectrum.
    */

    {

    int 	i,outspec = 2;
    int 	freepar[MAXPAR],nfree;
    float 	penalty[MAXPAR];
    float       chisqr;
    float       *a,*dyda;
    float	x,*y;
//    int 	derivatives;
    float	area,polarization,lorentz,correction,intercept,f,ferr;
    float	transmission,int_tot,int_tot_err;

    /* Enable fit parameter range checking */

    for(i = 0; i < MAXPAR; i++) penalty[i] = 10000;

    /* Choose a theory and get in- and output spectra */

    theory_type();
    get_inoutspec(&FITSPEC,&outspec);
    FITOUTSP = outspec;
    if(DATA[FITSPEC].NPNT < 2)
	{
	errtype("ERROR, too few datapoints in spectrum");
	clear_command();
	return;
	}
    if(fabs(DATA[FITSPEC].X[DATA[FITSPEC].NPNT-1]-DATA[FITSPEC].X[0]) 
       < 1e-6)
	{
	errtype("ERROR, x-values in spectrum are constant");
	clear_command();
	return;
	}

    /* Get starting parameters of fit */

    if (!set_anafitpar()) return; /* stop here if no actual fit is requested */

    /* Compute weight of data points */

    if(WEIGHTMODE == 0)
	{
	for(i = 0; i < DATA[FITSPEC].NPNT; i++)
	    {
	    if(DATA[FITSPEC].E[i] > 1e-5)
		WEIGHT[i] = 1/sqr(DATA[FITSPEC].E[i]);
	    else
		WEIGHT[i] = 0;
	    }
	}
    else if(WEIGHTMODE == 1)
	{
	for(i = 0; i < DATA[FITSPEC].NPNT; i++)
	    WEIGHT[i] = 1;
	}
    /* 
     * Get number of free parameters and make array with free parameter 
     * numbers 
     */

    nfree = 0;
    for(i = 0; i < TH.NPAR; i++)
	{
	if(!TH.FIX[i])
	    {
	    freepar[nfree] = i+1;
	    nfree++;
	    }
	}

    /* Do the fit */

    if(!levenberg(DATA[FITSPEC].X,DATA[FITSPEC].Y,WEIGHT,DATA[FITSPEC].NPNT,
		  TH.PAR,TH.NPAR,freepar,nfree,TH.ERR,TH.MIN,TH.MAX,penalty,
		  &chisqr,ffit,1)) return;

    /* Show the results */

    sprintf(STRING,"Results of fit using %s\n",TH.TITLE);
    type_line(STRING);
    for(i = 0; i < TH.NPAR; i++)
      {
	sprintf(STRING,"%*s (%*s) = %14.6e +/- %13.5e\n",TH_LENGTH,TH.NAME[i],
	    S_LENGTH-1,TH.SHORT[i],TH.PAR[i],TH.ERR[i]);
	type_line(STRING);

#ifdef EXTENSIONS /* Howes */
        /*
         * The fit parameters are saved in internal "anarod" variables and thus
         * available to the user. Each parameter is stored twice:
         * - once in variables named "p1", "p2", ...
         * - then in variables named as the "TH.SHORT" name of the parameter.
         *
         * However, just putting them into the input command line may cause
         * problems with the command processing at some other part of the
         * program later. One has therefore to make sure that they have been
         * interpreted and thus removed from the command line before continuing
         * the program.
         *
         * This is achieved by first writing a dummy integer into the command
         * line, then the fit parameters, and then reading the dummy integer
         * back. Because of the way the command insertion and interpretation
         * works, reading the integer back will also interpret and remove the
         * expressions that are used to save the fit parameters.
         */ 
	put_command("3");
	/* This saves the parameters in variables p1,p2 ...*/
	sprintf(STRING,"$p%d=%f",i+1,TH.PAR[i]);
	put_command(STRING);
	/* and this saves them in "bkgnd" "slope" "hgt1" etc as well */
	sprintf(STRING,"$%s=%f",TH.SHORT[i],TH.PAR[i]);
	put_command(STRING);
	get_int(0," ");
#endif /* EXTENSIONS Howes */

      }
    CHISQR = chisqr/(DATA[FITSPEC].NPNT-nfree+1e-10);   /* Save chisqr */
    sprintf(STRING,"chisqr = %9.3f, normalized = %10.4f\n",
	    chisqr,CHISQR);
    type_line(STRING);
    /*
     * Show the integrated intensities of the peaks and the structure factors
     * if appropriate
     */
    int_tot = int_tot_err = 0;
    (*F_PTR)(x,a,y,dyda,PEAK);
    for(i = 0; i < TH.NPEAK; i++)
	{
	int_tot += TH.PEAKINT[i];
	int_tot_err += sqr(TH.PEAKERR[i]);
	sprintf(STRING,"Integrated intensity peak %1d: %10.2f +/- %10.2f\n",
	    i+1,TH.PEAKINT[i],TH.PEAKERR[i]);
	type_line(STRING);
	if(DATARUN)
	    {
	    comp_cor(&lorentz,&polarization,&intercept,
		     &area, &transmission);
	    correction = lorentz*area*polarization*intercept*
	      transmission/STEPSIZE;
	    if(TH.PEAKINT[i] < 0)
		{
		f = 0;
		ferr = 1;
		}
	    else
		{
		f = sqrt(TH.PEAKINT[i]*correction);
		ferr = sqrt(sqr(f)+TH.PEAKERR[i]*correction)-f;
		}
	    sprintf(STRING,"Structure factor peak %1d    : %10.2f +/- %10.2f\n",
		i+1,f,ferr);
	    type_line(STRING);
	    }
	}

    /* If more than one peak, also show totals */

    if(TH.NPEAK > 1)
	{
	sprintf(STRING,"Total integrated intensity : %10.2f +/- %10.2f\n",
	    int_tot,sqrt(int_tot_err));
	type_line(STRING);
	if(DATARUN)
	    {
	    if(int_tot < 0)
		{
		f = 0;
		ferr = 1;
		}
	    else
		{
		f = sqrt(int_tot*correction);
		ferr = sqrt(sqr(f)+int_tot_err*correction)-f;
		}
	    sprintf(STRING,
		    "Total structure factor       : %10.2f +/- %10.2f\n",
		    f,sqrt(ferr));
	    type_line(STRING);
	    }
	}

    /* If required, write fit results to file */

    if(OUTFIT)
	{
        /*
         * Reorganized the output statements to avoid duplication of code.
         */
	if(!FITSWRITTEN)
	    {
	    fprintf(FITFILE,"%5s %8s %8s %8s","run#","H","K","L");
#ifdef EXTENSIONS /* Wilcke */
	    /*
	     * Print theory parameters.
	     *
	     * First: background part of fit function
	     */
	    for(i = 0; i < TH.NBCK; i++)
		fprintf(FITFILE," %8s bg.",TH.SHORT[i]);
	    /*
	     * Then the peak fitting function, depending on its type.
	     *
	     * For single peak (Gauss, Lorentzian or combined Gauss-Lorenztian)
	     */
	    if(F_PTR == gauss_single || F_PTR == lorentz_single ||
		F_PTR == gauss_lorentz_combined)
		{
		fprintf(FITFILE," %12s %12s %12s","center","height","FWHM");
		if(F_PTR == gauss_lorentz_combined)
		    fprintf(FITFILE," %12s","L param.");
		}
	    else if(F_PTR == gauss_lorentz) /* for Gaussian plus Lorentzian */
		{
		fprintf(FITFILE," %12s %12s %12s %12s %12s %12s",
		    "cent.Gauss","hght.Gauss","FWHM Gauss",
		    "cent.Lor.","hght.Lor.","FWHM Lor.");
		}
	    else if(F_PTR == lorentz_gauss) /* for Lorentzian plus Gaussian */
		{
		fprintf(FITFILE," %12s %12s %12s %12s %12s %12s",
		    "cent.Lor.","hght.Lor.","FWHM Lor.",
		    "cent.Gauss","hght.Gauss","FWHM Gauss");
		}
	    else if(F_PTR == power) /* for exponential function */
		{
		fprintf(FITFILE," %12s %12s %12s %12s","peak",
		    "amplitude","width","expon.");
		}
	    else if(F_PTR == pseudo_voigt) /* for Pseudo Voigt */
		{
		fprintf(FITFILE," %12s %12s %12s %12s %12s",
		    "peak","height","FWHM Lor.","FWHM Gauss","Lr. fr. eta");
		}
	    /*
	     * For multiple Gauss or multiple Lorentzian
	     */
	    else if(F_PTR == gauss_double || F_PTR == gauss_triple ||
		F_PTR == gauss_4peak || F_PTR == gauss_5peak || 
		F_PTR == lorentz_double)
		{
		for(i = 1; i <= TH.NPEAK; i++)
		    fprintf(FITFILE," %11s%1d %11s%1d %11s%1d",
			"center",i,"height",i,"FWHM",i);
		}
	    else if(F_PTR == slit) /* for slit function */
		{
		fprintf(FITFILE,
		    " %12s %12s %12s %12s %12s %12s %12s",
		    "center","height","opening","blade offs.","screen dis.",
		    "wavelen","smearing");
		}
	    /*
	     * End of theory parameters. Print chi-square, intensity and
	     * structure factors. If more than one peak, print also individual
	     * structure factor for each peak.
	     */
	    fprintf(FITFILE," %8s %7s %7s %7s","chisqr","Int.int","F","dF");
	    if(TH.NPEAK > 1)
		{
		for(i = 1; i <= TH.NPEAK; i++)
		    fprintf(FITFILE," %7s%1d %7s%1d","F",i,"dF",i);
		}
#endif /* EXTENSIONS Wilcke */
	    fprintf(FITFILE," %18s\n","type");
	    }
	FITSWRITTEN++;
	/* Made HMIL, KMIL and LMIL arrays (see ana.h) */
	fprintf(FITFILE,"%5d %8.2f %8.2f %8.5f",SCANNR,
	    HMIL[FITSPEC]*T11 + KMIL[FITSPEC]*T12 + LMIL[FITSPEC]*T13,
	    HMIL[FITSPEC]*T21 + KMIL[FITSPEC]*T22 + LMIL[FITSPEC]*T23,
	    HMIL[FITSPEC]*T31 + KMIL[FITSPEC]*T32 + LMIL[FITSPEC]*T33);
	for(i = 0; i < TH.NPAR; i++)
	    {
	    fprintf(FITFILE," %12.4e",TH.PAR[i]);
	    }
	fprintf(FITFILE," %8.2f %7.2f %7.2f",CHISQR,int_tot,f);
	if(TH.NPEAK > 1)
	    {
	    fprintf(FITFILE," %7.2f",sqrt(ferr));
	    for(i = 0; i < TH.NPEAK; i++)
		{
		f = sqrt(TH.PEAKINT[i]*correction);
		ferr = sqrt(sqr(f)+TH.PEAKERR[i]*correction)-f;
		fprintf(FITFILE," %7.2f %7.2f",f,ferr);
		}
	    }
	else
	    {
	    fprintf(FITFILE," %7.2f",ferr);
	    }
	fprintf(FITFILE," %18s\n",TH.TITLE);
	}

    /* Make theoretical spectrum */

    DATA[outspec].NPNT = DATA[FITSPEC].NPNT;
    DATA[outspec].X[0] = DATA[FITSPEC].X[0];
    (*F_PTR)(DATA[outspec].X[0],TH.PAR,&DATA[outspec].Y[0],dyda,CALC);
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    DATA[outspec].E[0] = 1.;
    for(i = 1; i < DATA[outspec].NPNT; i++)
	{
	DATA[outspec].X[i] = DATA[FITSPEC].X[i];
	(*F_PTR)(DATA[outspec].X[i],TH.PAR,&DATA[outspec].Y[i],dyda,CALC);
	DATA[outspec].E[i] = 1.;
	if(DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if(DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}

    /* Generate title */

    i = sprintf(DATA[outspec].TITLE,"Fit of %1d using ",FITSPEC+1);
    if(i > 0)
	strncat(DATA[outspec].TITLE,TH.TITLE,TITLE_LENGTH-i);
    type_header(1);

    }

/***************************************************************************/
void    ffit(float x, float a[], float *y, float dyda[], int na,
	    int derivatives)
/***************************************************************************/

    /*
    Function called by fitting routines to compute y-values and derivatives
    with respect to fitting parameters.
    This general routine makes a call to the specific theory used for
    fitting.
    */

    {
    if(derivatives)
	(F_PTR)(x,a,y,dyda,DERV);
    else
	(F_PTR)(x,a,y,dyda,CALC);

    na = na;
    }

/***************************************************************************/
void    guess_fit(int nspec, float *background, float *slope,
	    float *position, float *height, float *width)
/***************************************************************************/

    /*
    Estimates fit parameters of a spectrum with a single curve
    */

    {

    float	xleft,xright,yleft,yright,xmiddle,maxy;
    float       average,half_value,xhalf1,xhalf2;
    int         i,ipos,n;

    if (DATA[nspec].NPNT < 2*NLEVELING)
	{
	errtype("ERROR, too few datapoints for guess");
	return;
	}

    /* Determine average x and y values on the left and right sides of
    the spectrum */

    xleft=yleft=xright=yright=0.;
    for (i = 0; i < NLEVELING; i++)
	{
	yleft += DATA[nspec].Y[i];
	xleft += DATA[nspec].X[i];
	}
    yleft /= NLEVELING;
    xleft /= NLEVELING;
    for (i = DATA[nspec].NPNT-NLEVELING; i < DATA[nspec].NPNT; i++)
	{
	yright += DATA[nspec].Y[i];
	xright += DATA[nspec].X[i];
	}
    yright /= NLEVELING;
    xright /= NLEVELING;

    /* Deterimine slope and middle of the spectrum */

    *slope = (yright-yleft)/(xright-xleft+1e-10);
    xmiddle = (xleft+xright)/2;

    /* Store leveled curve in temporary spectrum */

    for (i = 0; i < DATA[nspec].NPNT; i++)
	{
	XD[i] = DATA[nspec].X[i];
	YD[i] = DATA[nspec].Y[i]-(DATA[nspec].X[i]-xmiddle)*(*slope);
	}

    /* Determine x and y value of maximum in leveled spectrum, plus the
    average */

    maxy = average = YD[0];
    ipos = 0;
    for (i = 1; i < DATA[nspec].NPNT; i++)
	{
	average += YD[i];
	if (YD[i] > maxy)
	    {
	    maxy = YD[i];
	    *position = XD[i];
	    ipos = i;
	    }
	}
    average /= DATA[nspec].NPNT;

    /* Find background */

    *background = 0.;
    n = 0;
    for (i = 0; i < DATA[nspec].NPNT; i++)
	{
	if (DATA[nspec].Y[i] < average)
	    {
	    n++;
	    *background += DATA[nspec].Y[i];
	    }
	}
    *background /= n;

    /* Find x-values corresponding to halfwidth and compute full width */

    half_value = 0.5*(maxy+*background);
    for (i = 0; i <= ipos; i++)
	{
	if (YD[i] > half_value) break;
	}
    xhalf1 = XD[i];
    for (i = ipos; i < DATA[nspec].NPNT; i++)
	{
	if (YD[i] < half_value) break;
	}
    xhalf2 = XD[i];
    *width = fabs(xhalf2-xhalf1);

    /* Compute height */

    *height = maxy-*background;

    }

/***************************************************************************/
int    set_anafitpar(void)
/***************************************************************************/

    /*
    Give starting values to fit parameters, and set them fixed or loose.
    */

    {

    /* define setfitpar_menu */

    enum i_setfpar {
	i_zero,
	i_value,
	i_lower,
	i_upper,
	i_fix,
	i_loose,
	i_endpar,      /* Marks the end of the options that change parameters */
	i_guess,
	i_midpoint,
#ifdef EXTENSIONS /* Svensson */
	i_cursor,
#endif /* EXTENSIONS Svensson */
	i_list,
	i_help,
	i_run,
	i_return
	};

    static struct   MENU setfpar_menu[] =
	{
	"value",     1, i_value,    "Set parameter value",
	"lower",     3, i_lower,    "Set lower bound on parameter range",
	"upper",     1, i_upper,    "Set upper bound on parameter range",
	"fix",       1, i_fix,      "Fix a parameter value",
	"loose",     2, i_loose,    "Make parameter free",
	"guess",     1, i_guess,    "Estimate fitting parameters",
	"midpoint",  1, i_midpoint, "Set peak position to center of scan",
#ifdef EXTENSIONS /* Svensson */
        "cursor",    4, i_cursor,   "Input peak positions using the cursor",
#endif /* EXTENSIONS Svensson */
	"list",      1, i_list,     "List parameter values",
	"help",      1, i_help,     "Display menu",
	"run",       2, i_run,      "Start fit",
	"return",    1, i_return,   "Return to main menu",
	};

    /* number of commands in setfitpar menu */
    int     setfpar_length = sizeof(setfpar_menu) / sizeof(setfpar_menu[0]);

    static int parameter = 0;
    char    token[100];
    int     iret = -1;
    int     i,imatch;
    float   gbackground,gslope,gposition,gheight,gwidth;
    float   **plot_points, x1, x2, y1;

    if(TH.NPAR > 0)
	put_command("list");
    while (iret == -1)
	{
	while (!get_token(token,"ANA.FIT.PAR>"));
	imatch = cmnd_match(token,setfpar_menu,setfpar_length);
	/*
	 * Test if illegal number entered for commands that change parameters.
	 */
	if(imatch > i_zero && imatch < i_endpar)
	    {
	    sprintf(STRING,"Parameter number to change [%1d]: ",++parameter);
	    parameter = get_int(parameter,STRING);
 	    if ((--parameter < 0) || (parameter >= TH.NPAR))
		{
		errtype("ERROR, parameter out of range");
		clear_command();
		continue;
		}
	    }
	switch(imatch)
	    {
	    case i_value:
		sprintf(STRING,"%s [%7.3f]: ",TH.NAME[parameter],
		    TH.PAR[parameter]);
		TH.PAR[parameter] = get_real(TH.PAR[parameter],STRING);
		break;
	    case i_lower:
		sprintf(STRING,"Lower bound of %s [%7.3f]: ",
		    TH.NAME[parameter],TH.MIN[parameter]);
		TH.MIN[parameter] = get_real(TH.MIN[parameter],STRING);
		break;
	    case i_upper:
		sprintf(STRING,"Upper bound of %s [%7.3f]: ",
		    TH.NAME[parameter],TH.MAX[parameter]);
		TH.MAX[parameter] = get_real(TH.MAX[parameter],STRING);
		break;
	    case i_fix:
		TH.FIX[parameter] = TRUE;
		break;
	    case i_loose:
		TH.FIX[parameter] = FALSE;
		break;
	    case i_guess:
		guess_fit(FITSPEC,&gbackground,&gslope,&gposition,
		    &gheight,&gwidth);
		if (!TH.FIX[0]) TH.PAR[0] = gbackground;
		if (!TH.FIX[1]) TH.PAR[1] = gslope;
		if (!TH.FIX[TH.NBCK]) TH.PAR[TH.NBCK] = gposition;
		if (!TH.FIX[TH.NBCK+1]) TH.PAR[TH.NBCK+1] = gheight;
		if (!TH.FIX[TH.NBCK+2]) TH.PAR[TH.NBCK+2] = gwidth;

		/* Start with a Lorentzian for power-law line shape */

		if (F_PTR == power && !TH.FIX[TH.NBCK+3]) TH.PAR[TH.NBCK+3] = 2;
		put_command("list");
		break;
	    case i_midpoint:
		TH.PAR[TH.NBCK] = DATA[FITSPEC].X[(DATA[FITSPEC].NPNT-1)/2];
		break;
#ifdef EXTENSIONS /* Svensson */
	    case i_cursor:
		if (!TH.NPEAK)
		    {
		    errtype(
			"ERROR, cursor input impossible for this theory type");
		    }
		else
		    {
		    sprintf(STRING, "%1d plot ret", FITSPEC+1);
		    put_command(STRING);
		    plot();
		    for (i = 0; i < TH.NPEAK; i++)
			{

			/*
			 * Note: the x coordinate of the peak position will be
			 * calculated from the left and right FWHM positions,
			 * NOT taken from the "max" point! The "max" point only
                         * defines the peak height.
			 */
			sprintf(STRING,
                            "Click on max and full width half max of peak %d\n",
                            i + 1);
			type_line(STRING);

			/*
			 * Note that the "ncursor" command returns the
			 * coordinates sorted in ascending x-values, NOT in the
			 * chronological order of the clicks!
			 */
			sprintf(STRING, "%1d ncursor 3 ret", FITSPEC+1);
			put_command(STRING);
			plot();		   
			getPlotcursorPoint(0,0,&x1);  
			getPlotcursorPoint(1,1,&y1);  
			getPlotcursorPoint(2,0,&x2);  
			TH.PAR[TH.NBCK+i*3] = (x1 + x2)/2.0;
			TH.PAR[TH.NBCK+i*3+1] = y1;
			TH.PAR[TH.NBCK+i*3+2] = fabs(x1 - x2);
			}
		    put_command("list");
		    }
		break;
#endif /* EXTENSIONS Svensson */
	    case i_list:
		/*
		 * Clear the screen, list the parameter values in the upper part
		 * and redefine scroll area to keep this part unchanged while
		 * commands are accepted in the lower part.
                 *
                 * If there are too many parameters to fit on the screen, do
		 * not redefine scroll area.
		 *
		 * Technical note: we need one line more than the number of
		 * parameters for the title.
		 *
		 * The scroll area is reset to its normal value when leaving
		 * this routine.
		 */
		clear_screen();
		set_cursor(1,1);
		if(TH.NPAR + 2 < ROWS)
		    set_scroll(TH.NPAR + 2,ROWS);
		sprintf(STRING,"%2s %-30s %12s %12s %12s\n",
		    "#","name","start","lower","upper");
		type_line(STRING);
		for (i = 0; i < TH.NPAR; i++)
		    {
		    sprintf(STRING,"%2d %-30s %12.4f %12.4f %12.4f ",
			i+1,TH.NAME[i],TH.PAR[i],TH.MIN[i],TH.MAX[i]);
		    if(TH.FIX[i])
			strcat(STRING," fixed\n");
		    else
			strcat(STRING," loose\n");
		    type_line(STRING);
		    }
		break;
	    case i_help:
		list_menu("SET FITTING PARAMETERS",setfpar_menu,setfpar_length);
		break;
	    case i_run:
		iret = 1;
		break;
	    case i_return:
		iret = 0;
		break;
	    }
	}

	/*
	 * Reset scroll area to default value.
	 */
	clear_screen();
	set_scroll(MAX_SPECTRUM+4,ROWS);
	type_header(1);
	return(iret);
    }
