/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$$$$$$                  correct.c                          $$$$$$$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: correct.c,v 1.6 2001/11/26 08:16:11 svensson Exp $

****************************************************************************/
/*
Update 26/11/2001 O. Svensson
                  Replaced "0" and "1" with "0." and "1." at three locations.
Update 06/10/2000 E. Vlieg (vlieg@sci.kun.nl)
	 	  Add 2+3 mode.
Update 06/10/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Include correction for linear gamma table: LINEARGAMMA.
		  Include transmission correction: TRANSIN, TRANSOUT.
*/

/***************************************************************************/
/*      Declare include files                                              */
/***************************************************************************/

#define EXT	extern
#include "ana.h"
#undef EXT

/***************************************************************************/
void	comp_cor(float *lorentz, float *polarization, float *intercept,
		 float *area, float *transmission)
/***************************************************************************/

    /* Compute correction factors by calling the subroutine corresponding
    to the diffractometer geometry used.
    See Elias Vlieg, J. Appl. Cryst. 30 (1997) 532, and
    J. Appl. Cryst. 31 (1998) 198. */

    {

    if (CORRECTMODE == ZAXIS)
	comp_cor_zaxis(lorentz,polarization,intercept,area,transmission);
    if ((CORRECTMODE == TWOPTWO) || (CORRECTMODE == TWOPTHREE))
	comp_cor_2p2or3(lorentz,polarization,intercept,area,transmission);
    if (CORRECTMODE == REFROCK)
	comp_cor_refl(lorentz,polarization,intercept,area,transmission);
    if (CORRECTMODE == NOCORRECT)
	{
	*lorentz = *polarization = *intercept = *area = 1.;
	}

    }

/***************************************************************************/
void	comp_cor_zaxis(float *lorentz, float *polarization,
		       float *intercept, float *area, float *transmission)
/***************************************************************************/

    /* Compute the correction factors */

    {

    float	x,y,c1,c2,c3,phor,pver,area_sum,xlab,zlab,xdet,bs_eff;
    float	xlimit,xstep,ylimit,ystep;

    /* Calculate Lorentz factor */

    if (RSCAN)
	*lorentz = cos(BETAIN/RAD)*cos(GAMMA/RAD)*sin(DELTA/RAD);
    else
	*lorentz = 1;

    /* Calculate polarization correction */

    phor = ( 1. - sqr ( sin(ALPHA/RAD) * cos(DELTA/RAD) * cos(GAMMA/RAD)
		+ cos(ALPHA/RAD) * sin(GAMMA/RAD) ));
    pver = 1.-sqr(sin(DELTA/RAD)*cos(GAMMA/RAD));
    *polarization =  1. / (HPOLAR*phor+(1-HPOLAR)*pver);

    /* Calculate the rod interception correction */

    *intercept = (cos(ALPHA/RAD) + cos((ALPHA-2*BETAIN)/RAD) +
		  sin(2*(ALPHA-BETAIN)/RAD)*sin(GAMMA/RAD) +
		  2*sqr(sin((ALPHA-BETAIN)/RAD))*cos(DELTA/RAD)*cos(GAMMA/RAD)) /
	(2*cos(BETAIN/RAD)*cos(GAMMA/RAD));
    if (LINEARGAMMA) *intercept = (*intercept)/cos(GAMMA/RAD);

    /* Calculate area correction for used diffractometer setting */

    if (BEAMCOR) // take into account non-uniform intensity distribution
	{
	c1 = sin(BETAIN/RAD);
	c2 = cos(DELTA/RAD);
	c3 = sin(DELTA/RAD)*cos(ALPHA/RAD-BETAIN/RAD);
	area_sum = 0;

	/* determine integration limit along x (vertical) direction */

	if (VERSLIT > 0.01)
	    xlimit = VERSLIT/2+0.01;
	else
	    xlimit = 1.;
	xstep = xlimit/50.;

	/* determine integration limit along y (horizontal) direction */

	if (RADIUS > 0.01)
	    ylimit = RADIUS*1.1;
	else
	    ylimit = 10;
	if (fabs(2*HORBEAM/sin(BETAIN/RAD+0.001)) < ylimit)
	    ylimit = fabs(2*HORBEAM/sin(BETAIN/RAD+0.001));
	if (1.1*HORSLIT/(2*sin(BETAIN/RAD+0.001)) < ylimit)
	    ylimit = 1.1*HORSLIT/(2*sin(BETAIN/RAD+0.001));
	ystep = ylimit/50.;

	for (x = -xlimit; x < xlimit+0.01; x+= xstep)
	    {
	    for ( y = -ylimit; y < ylimit+0.01; y+= ystep)
		{
		area_sum += f_beam(x,y*c1)*f_detector(x*c2-y*c3)*f_onsample(x,y);
		}
	    }

	/* Calculate effective vertical beam slit size */

	bs_eff = 0.;
	for (x = -xlimit; x < xlimit; x += xstep/10.) bs_eff += f_beam(x,0);
	bs_eff *= xstep/10.;

	*area = (DSLIT*bs_eff)/(area_sum*xstep*ystep);

	}
    else         // no intensity distribution correction
	{
	*area = sin(DELTA/RAD)*cos(ALPHA/RAD-BETAIN/RAD);
	}

    /* Beam transmission correction */

    *transmission = pow(TRANSIN,1/cos(ALPHA/RAD)-1)*
	pow(TRANSOUT,1/cos(GAMMA/RAD)-1);

    }

/***************************************************************************/
void	comp_cor_2p2or3(float *lorentz, float *polarization,
			float *intercept, float *area, float *transmission)
/***************************************************************************/

    /* Compute the correction factors for 2+2 geometry */

    {

    float	x,y,c1,c2,c3,phor,pver,area_sum,xlab,zlab,xdet,bs_eff;
    float	cos_beta_out,xlimit,xstep,ylimit,ystep;

    /* Calculate Lorentz factor */

    if (RSCAN)
	*lorentz = cos(ALPHA/RAD)*sin(DELTA/RAD);
    else
	*lorentz = 1;

    /* Calculate polarization correction */

    phor = 1. - sqr ( sin(GAMMA/RAD) * cos(DELTA/RAD));
    pver = 1.-sqr(sin(DELTA/RAD));
    *polarization =  1. / (HPOLAR*phor+(1-HPOLAR)*pver);

    /* Calculate the rod interception correction */

    if (CORRECTMODE == TWOPTWO)
	*intercept = (sqr(sin((ALPHA-GAMMA)/RAD))*cos(DELTA/RAD)+
		      cos(ALPHA/RAD)*cos((ALPHA-GAMMA)/RAD))/cos(ALPHA/RAD);
    if (CORRECTMODE == TWOPTHREE)
	*intercept = 1/sqrt(1-sqr(cos(DELTA/RAD)*sin((GAMMA-ALPHA)/RAD)));

    /* Calculate area correction for used diffractometer setting */

    if (BEAMCOR) // take into account non-uniform intensity distribution
	{
	c1 = sin(ALPHA/RAD);
	c2 = cos(DELTA/RAD);
	c3 = sin(DELTA/RAD)*cos(GAMMA/RAD-ALPHA/RAD);
	area_sum = 0;

	/* determine integration limit along x (vertical) direction */

	if (VERSLIT > 0.01)
	    xlimit = VERSLIT/2+0.01;
	else
	    xlimit = 1.;
	xstep = xlimit/50.;

	/* determine integration limit along y (horizontal) direction */

	if (RADIUS > 0.01)
	    ylimit = RADIUS*1.1;
	else
	    ylimit = 10;
	if (fabs(2*HORBEAM/sin(ALPHA/RAD+0.001)) < ylimit)
	    ylimit = fabs(2*HORBEAM/sin(ALPHA/RAD+0.001));
	if (1.1*HORSLIT/(2*sin(ALPHA/RAD+0.001)) < ylimit)
	    ylimit = 1.1*HORSLIT/(2*sin(ALPHA/RAD+0.001));
	ystep = ylimit/50.;

	for (x = -xlimit; x < xlimit+0.01; x+= xstep)
	    {
	    for ( y = -ylimit; y < ylimit+0.01; y+= ystep)
		{
		area_sum += f_beam(x,y*c1)*f_detector(x*c2-y*c3)*f_onsample(x,y);
		}
	    }

	/* Calculate effective vertical beam slit size */

	bs_eff = 0.;
	for (x = -xlimit; x < xlimit; x += xstep/10.) bs_eff += f_beam(x,0);
	bs_eff *= xstep/10.;

	*area = (DSLIT*bs_eff)/(area_sum*xstep*ystep);

	}
    else         // no intensity distribution correction
	{
	cos_beta_out = sqrt(1-sqr(cos(DELTA/RAD)*sin((GAMMA-ALPHA)/RAD)));
	*area = sin(DELTA/RAD)/cos_beta_out;
	}

    /* Beam transmission correction */

    *transmission = pow(TRANSIN,1/cos(ALPHA/RAD)-1)*
	pow(TRANSOUT,1/cos_beta_out-1);

    }


/***************************************************************************/
void	comp_cor_refl(float *lorentz, float *polarization, float *intercept,
		      float *area, float *transmission)
/***************************************************************************/

    /* Compute the correction factors for reflectivity rocking scans. */

    {

    float	y,c,beam_sum,sample_sum;

    /* Calculate Lorentz factor */

    *lorentz = sin(2*ALPHA/RAD);

    /* Calculate polarization correction */

    *polarization =  1. / (1-HPOLAR*sqr(sin(2*ALPHA/RAD)));

    /* Calculate rod interception correction */

    *intercept = 1/cos(ALPHA/RAD);
    if (LINEARGAMMA) *intercept = (*intercept)/cos(GAMMA/RAD);

    /* Calculate area/beam profile correction.
       For a reflectivity scan, only the y direction is of interest */

    if (BEAMCOR) // take into account non-uniform intensity distribution
	{
	beam_sum = 0;
	sample_sum = 0;
	for ( y = -LIMIT; y < LIMIT+0.01; y+= STEP)
	    {
	    c = f_onsample(0,y);
	    beam_sum += f_beam(0,y*sin(ALPHA/RAD))*c;
	    sample_sum += c;
	    }
	*area = sample_sum/beam_sum;
	}
    else
	{
	*area = sin(ALPHA/RAD);
	}

    /* Beam transmission correction */

    *transmission = pow(TRANSIN,1/cos(ALPHA/RAD)-1)*
	pow(TRANSOUT,1/cos(GAMMA/RAD)-1);

    }

/***************************************************************************/
void	comp_cor_ridge(float beta_in, float *lorentz, float *polarization,
		       float *area, float *transmission)
/***************************************************************************/

    /* Compute the correction factors for reflectivity ridge scans. Beta_in
    is the angle of incidence */

    {

    float	y,c,beam_sum,sample_sum;

    /* Calculate Lorentz factor */

    *lorentz = sin(beta_in);

    /* Calculate polarization correction */

    *polarization =  1. / (HPOLAR*sqr(cos(2.*beta_in))+1.-HPOLAR);

    /* Calculate area/beam profile correction for used diffractometer setting.
       For a reflectivity scan, only the y direction is of interest */

    if (BEAMCOR) // take into account non-uniform intensity distribution
	{
	beam_sum = 0;
	sample_sum = 0;
	for ( y = -LIMIT; y < LIMIT+0.01; y+= STEP)
	    {
	    c = f_onsample(0,y);
	    beam_sum += f_beam(0,y*sin(beta_in))*c;
	    sample_sum += c;
	    }
	*area = sample_sum/beam_sum;
	}
    else    /* assume non-flooding case */
	{
	*area = sin(beta_in);
	}

    /* Beam transmission correction; beta_in = beta_out */

    *transmission = pow(TRANSIN,1/cos(beta_in)-1)*
	pow(TRANSOUT,1/cos(beta_in)-1);

    }


/***************************************************************************/
float	f_beam(float x, float z)
/***************************************************************************/

    /*
    Compute normalized incoming beam intensity.
    */

    {

    if ((fabs(x) > VERSLIT/2.) || (fabs(z) > HORSLIT/2.))
	{
	return(0.);
	}
    else
	{
	return(exp(-2.77*x*x/(VERBEAM*VERBEAM))*
	       exp(-2.77*z*z/(HORBEAM*HORBEAM)));
	}
    }


/***************************************************************************/
float	f_onsample (float x, float y)
/***************************************************************************/

    /*
    Compute whether point is on sample.
    */

    {

    if (((x*x)+(y*y)) > (RADIUS*RADIUS))
	{
	return(0.);
	}
    else
	{
	return(1.);
	}
    }

/***************************************************************************/
float	f_detector(float x)
/***************************************************************************/

    /*
    Compute whether point is observed by detector.
    */

    {

    if (fabs(x) > DSLIT/2.)
	{
	return(0.);
	}
    else
	{
	return(1.);
	}
    }


