/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                       keating.c                               $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: keating.c,v 1.6 2007/03/19 10:31:50 wilcke Exp $

****************************************************************************/

/***************************************************************************

Update 13/03/2007 R. Wilcke (wilcke@esrf.fr)
                  implement E. Vlieg's simplification in functional form of
                  Lennard-Jones potential;
                  change back to include of <lsqfit.h> (consequence of the
                  reorganization of the lsqfit routines);
                  remove include of <lsqfit.h> (is now in "rod.h").
Update 18/09/2003 O. Svensson (svensson@esrf.fr)
                  Include ../lsqfit/lsqfit.h instead of <lsqfit.h>
                  as done in the fit.c file.
                  Added #ifdef ASAROD for lsqfit calls, added ffit_asa
                  function.
Update 21/02/2002 O. Svensson (svensson@esrf.fr)
                  Ported changes from standard version: 
                  Add Lennard-Jones as possible potential. The subroutines 
                  no longer deal exclusively with Keating, so some names are 
                  changed as well. For simplicity, we will leave the name 
                  KEAT(ING) if this does not lead to confusion.
                  Change type_line to type_list in listing of bonds and angles.

Changes: 20/10/95 by Willem Jan program because it wasn't working

****************************************************************************/

#define LARGE_PENALTY 1e4	/* useful penalty factor */
#define NO_PENALTY    0

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define KEATING
#include "rod.h"

/***************************************************************************/
void    angle_search(void)
/***************************************************************************/

    /*
    Find all bond angles between atoms that are part of the energy
    minimization.
    */

    {

    int	i,j,k,l;
    int	xas[MAXBOND],yas[MAXBOND],bondlist[MAXBOND],nbonds;
    float	pos1[3],pos2[3],pos3[3],bond1[3],bond2[3],angle;

    for (i = 0; i < NSURF; i++)
	{
	NANGLE[i] = 0;
	if (IN_KEAT[i])
	    {

	    /* Find all bonds of atom i */

	    nbonds = 0;

	    /* See if i appears as a bond to an earlier atom */

	    for (j = 0; j < i; j++)
		{
		if (IN_KEAT[j])
		    {
		    for (k = 0; k < NBOND[j]; k++)
			{
			if (BOND_LIST[j][k] == i)
			    {
			    bondlist[nbonds] = j;
			    xas[nbonds] = -XBSHIFT[j][k];
			    yas[nbonds] = -YBSHIFT[j][k];
			    nbonds++;
			    }
			}
		    }
		}

	    /* Add to the bond list the bonds of the atom itself */

	    if (nbonds+NBOND[i] > MAXBOND)
		errtype("Maximum number of bonds exceeded");
	    else
		{
		for (j = 0; j < NBOND[i]; j++)
		    {
		    bondlist[nbonds] = BOND_LIST[i][j];
		    xas[nbonds] = XBSHIFT[i][j];
		    yas[nbonds] = YBSHIFT[i][j];
		    nbonds++;
		    }
		}

	    /* Generate all bond angles out of the bondlist */

	    get_cartesian(pos1,XSFIT[i],YSFIT[i],ZSFIT[i]);
	    for (j = 0; j < nbonds; j++)
		{
		get_cartesian(pos2,XSFIT[bondlist[j]]+xas[j],
			YSFIT[bondlist[j]]+yas[j],ZSFIT[bondlist[j]]);
		for (k = j+1; k < nbonds; k++)
		    {
		    if (NANGLE[i] == MAXANGLE)
			{
			errtype("Maximum number of bond angles exceeded");
			}
		    else
			{
			get_cartesian(pos3,XSFIT[bondlist[k]]+xas[k],
			    YSFIT[bondlist[k]]+yas[k],ZSFIT[bondlist[k]]);
			for (l = 0; l < 3; l++)
			    {
			    bond1[l] = pos2[l]-pos1[l];
			    bond2[l] = pos3[l]-pos1[l];
			    }
			angle = acos((bond1[0]*bond2[0]+
			    bond1[1]*bond2[1]+bond1[2]*bond2[2])
			    /sqrt(sqr(bond1[0])+sqr(bond1[1])+sqr(bond1[2]))
			    /sqrt(sqr(bond2[0])+sqr(bond2[1])+sqr(bond2[2])));
			if (angle < 1.1*EQU_ANGLE)
			    {
			    ANGLE_LIST[i][NANGLE[i]][0] = bondlist[j];
			    ANGLE_LIST[i][NANGLE[i]][1] = bondlist[k];
			    XASHIFT[i][NANGLE[i]][0] = xas[j];
			    YASHIFT[i][NANGLE[i]][0] = yas[j];
			    XASHIFT[i][NANGLE[i]][1] = xas[k];
			    YASHIFT[i][NANGLE[i]][1] = yas[k];
			    NANGLE[i]++;
			    }
			else
			    {
			    }
			}
		    }
		}
	    }
	}
    }

/***************************************************************************/
void    bond_search(void)
/***************************************************************************/

    /*
    Find the bonds between atoms that are part of the energy minimization.
    The subroutine stores the values of the displacements along the lattice
    vectors that are necessary to calculate the correct bond lengths.

    Bonds of an atom to itself (on an equivalent position) are ignored.
    These only occur for small unit cells that are normally not used in
    experiments.
    */

    {

    int	i,j,k,l;
    float	pos1[3],pos2[3],length;

    /* Allow the 2nd atom to be shifted over one lattice parameter in order
    to find the correct (= minimum) bond distance. */

    for (i = 0; i < NSURF; i++)
	{
	NBOND[i] = 0;
	if (IN_KEAT[i])
	    {
	    get_cartesian(pos1,XSFIT[i],YSFIT[i],ZSFIT[i]);
	    for (j = i+1; j < NSURF; j++)
		{
		if (IN_KEAT[j])
		    {
		    for (k = -1; k <= 1; k++)
			{
			for (l = -1; l <= 1; l++)
			    {
			    get_cartesian(pos2,XSFIT[j]+k,YSFIT[j]+l,
				ZSFIT[j]);
			    length = sqrt(sqr(pos1[0]-pos2[0])+
			       sqr(pos1[1]-pos2[1])+sqr(pos1[2]-pos2[2]));
			    if (length < 1.1*(ATOM_RAD[TS[i]]
				+ATOM_RAD[TS[j]]))
				{
				if (NBOND[i] == MAXBOND)
				    {
				    errtype(
				       "Error, maximum # bonds exceeded");
				    }
				else
				    {
				    BOND_LIST[i][NBOND[i]] = j;
				    XBSHIFT[i][NBOND[i]] = k;
				    YBSHIFT[i][NBOND[i]] = l;
				    NBOND[i]++;
				    }
				}
			    }
			}
		    }
		}
	    }
	}
    }

/***************************************************************************/
void    fit_energy(void)
/***************************************************************************/

    /*
    Minimize energy by optimizing the specified structural
    parameters of the surface unit cell.
    */

    {

    int	i;
    float	*x,*y,florisx,florisy;
    int		ndata, rc;
    int		freepar[MAXPAR],nfree;
    float	penalties[MAXPAR];
    float	lattice_energy;

    /* Check whether model has been read in */

    if (NSURF < 1)
	{
	errtype("ERROR, no model read in");
	clear_command();
	return;
	}

    /* Copy all fitting parameters to one array FITPAR and get starting
    values */

    for (i = 0; i < NDISTOT; i++)
	{
	FITPAR[i] = DISPL[i];
	FITMIN[i] = DISPLLIM[i][0];
	FITMAX[i] = DISPLLIM[i][1];
	FITPEN[i] = DISPLPEN[i];
	sprintf(FITTXT[i],"Displacement %2d",i+1);
	}

    x = &florisx;
    y = &florisy;
    florisx = 1.;
    florisy = 0.;

    FWEIGHT[0] = 1.;
    ndata = 1;

    rc = set_fitpar("ROD.ENERGY.FIT>",
	       NDISTOT,
	       FITPAR, FITMIN, FITMAX,
	       FITPEN, FIXPAR,
	       FITTXT[0], TITLE_LENGTH,
	       x, y, FWEIGHT, ndata, fenergy);

    if (rc == LSQ_NOFIT) goto DONTFIT;

    /* Find number of free parameters and make array with their numbers */

    nfree = 0;
    for (i = 0; i < NDISTOT; i++)
	{
	if (!FIXPAR[i])
	    {
	    freepar[nfree] = i+1;
	    nfree++;
	    }
	}

    /* Prepare penalty factors */

    for (i = 0; i < NDISTOT; i++)
	penalties[i] = (FITPEN[i] == TRUE ? LARGE_PENALTY : NO_PENALTY);

    /* Do the fit */

    rc = lsqfit(x,y,FWEIGHT,ndata,FITPAR,NDISTOT,freepar,nfree,FITERR,
	FITMIN,FITMAX,penalties,&lattice_energy,fenergy,
#ifdef ASAROD
		ffit_asa,
#endif /* ASAROD */
	FITTXT[0], TITLE_LENGTH, 0, rc, 0);

DONTFIT:

    /* Copy values in array 'FITPAR' to corresponding model parameters and
    update structure model */

    for(i = 0; i < NDISTOT; i++)
	FITPEN[i] = (penalties[i] != 0 ? TRUE : FALSE);

    for (i = 0; i < NDISTOT; i++)
	{
	DISPL[i] = FITPAR[i];
	DISPLLIM[i][0] = FITMIN[i];
	DISPLLIM[i][1] = FITMAX[i];
	DISPLPEN[i] = FITPEN[i];
	}

    update_model();

    /* Show the fit results */

    sprintf(STRING,"Lattice energy %7.4f eV\n",energy_calc());
    type_line(STRING);

    }

/***************************************************************************/
void    fenergy(float x, float a[], float *y, float dyda[], int na,
	        int derivatives)
/***************************************************************************/

    /*
    Function called by fitting routines to compute lattice energy for
    given displacement settings.
    */

    {

    int i;

    /* Copy values in array 'a' to corresponding model parameters */

    for (i = 0; i < NDISTOT; i++)
	{
	DISPL[i] = a[i];
	}

    update_model();

    *y = 100*energy_calc();



    x=x; dyda=dyda; na=na; derivatives=derivatives; /* prevent compiler warning */


    }

/***************************************************************************/
void    energy (void)
/***************************************************************************/

    /*
    Lattice-energy minimization of surfrace structure. Two potentials are
    implemented:
	Keating		bond lengths and bond angles
	Lennard-Jones	bond lenghts only
    */

    {

    /* define energy_menu */

#define energy_length 18       /* number of commands in energy menu */

    static struct   MENU energy_menu[energy_length] =
	{
	"alpha",	1,  1, "Set bond length deformation par.",
	"beta",		1,  2, "Set bond angle deformation par.",
	"radius",	2,  3, "Set atomic radius (angstrom)",
	"equangle",     2,  32,"Set equilibrium bond angle",
	"keating",      1,  33,"Use Keating potential",
	"lennard",      2,  34,"Use Lennard-Jones potential",
	"include", 	1,  4, "Include atom in Keating calculation",
	"exclude",      1,  5, "Exclude atom in Keating calculation",
	"remove",       3,  6, "Remove a bond from calculation",
	"findbonds",    1,  7, "Find all bonds between atoms",
	"chisqr",       2,  71,"Include energy in chi^2 minimization",
	"energy",       2,  8, "Calculate lattice energy",
	"minimize",     1,  9, "Minimize lattice energy",
	"list",		1, 20, "List parameter values",
	"bondlist",     2, 21, "List bonds between atoms",
	"anglelist",    2, 22, "List angles between atoms",
	"help",         1, 30, "Display menu",
	"return",       1, 31, "Return to main menu",
	};

    int     stop = FALSE;
    char    token[100];
    int     parameter,i,j,at1,at2,foundbond;
    char    kstring[2][5];
    
    while (!stop)
	{
	while (!get_token(token,"ROD.ENERGY>")) return;
	switch (cmnd_match(token,energy_menu,energy_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		sprintf(STRING,
		    "Bond length deformation parameter alpha [%7.4f]: ",
		    ALPHA_KEAT);
		ALPHA_KEAT = get_real(ALPHA_KEAT,STRING);
		break;
	    case 2:
		sprintf(STRING,
		    "Bond angle deformation parameter beta [%7.4f]: ",
		    BETA_KEAT);
		BETA_KEAT = get_real(BETA_KEAT,STRING);
		break;
	    case 3:
		i = get_int(1,"Serial number of element type: ");
		if ((i < 1) || (i > MAXTYPES)) i = 1;
		sprintf(STRING,"Atomic radius of %s [%7.4f] (Angstrom): ",
		    ELEMENT[i-1],ATOM_RAD[i-1]);
		ATOM_RAD[i-1] = get_real(ATOM_RAD[i-1],STRING);
		break;
	    case 32:
		sprintf(STRING,
		    "Equilibrium bond angle [%7.4f]: ",
		    EQU_ANGLE*RAD);
		EQU_ANGLE = get_real(EQU_ANGLE*RAD,STRING)/RAD;
		break;
	    case 33:
		POTENTIAL = VKEATING;
		break;
	    case 34:
		POTENTIAL = VLENNARDJONES;
		break;
	    case 4:
		at1 = get_natom("Atom to be included: ");
		IN_KEAT[at1-1] = TRUE;
		break;
	    case 5:
		at1 = get_natom("Atom to be excluded: ");
		IN_KEAT[at1-1] = FALSE;
		break;
	    case 6:
		at1 = get_natom("First atom of bond: ")-1;
		at2 = get_natom("Second atom of bond: ")-1;

		/* Search BOND_LIST for the specified bond */

		foundbond = FALSE;
		if (at2 < at1)
		    {
		    i = at1;
		    at1 = at2;
		    at2 = i;
		    }
		for (i = 0; (i < NBOND[at1]) && (!foundbond); i++)
		    {
		    if (BOND_LIST[at1][i] == at2)
			{
			foundbond = TRUE;
			NBOND[at1]--;
			for (j = i; j < NBOND[at1]; j++)
			    {
			    BOND_LIST[at1][j] = BOND_LIST[at1][j+1];
			    XBSHIFT[at1][j] = XBSHIFT[at1][j+1];
			    YBSHIFT[at1][j] = YBSHIFT[at1][j+1];
			    }
			}
		    }
		if (!foundbond) errtype("No such bond in structure");
		angle_search();
		break;
	    case 7:
		bond_search();
		angle_search();
		break;
	    case 71:
		sprintf(STRING,
		    "Include lattice energy in chi^2 minimization y/n [%s]: ",
		    yesnostr(KEAT_PLUS_CHI));
		KEAT_PLUS_CHI = yesno(KEAT_PLUS_CHI,STRING);
		break;
	    case 8:
		sprintf(STRING,"Lattice energy %6.3f eV\n",energy_calc());
		type_line(STRING);
		break;
	    case 9:
		angle_search();
		fit_energy();
		break;
	    case 20:
		if (POTENTIAL == VKEATING) type_line("Keating potential\n");
		if (POTENTIAL == VLENNARDJONES)
		    type_line("Lennard-Jones potential\n");
		sprintf(STRING,"alpha = %7.4f\n",ALPHA_KEAT);
		type_line(STRING);
		sprintf(STRING,"beta  = %7.4f\n",BETA_KEAT);
		type_line(STRING);
		for (i = 0; i < NTYPES; i++)
		    {
		    sprintf(STRING,"Atomic radius of %s = %7.4f\n",
			ELEMENT[i],ATOM_RAD[i]);
		    type_line(STRING);
		    }
		sprintf(STRING,"Equilibrium bond angle = %7.4f\n",
			EQU_ANGLE*RAD);
		type_line(STRING);
		sprintf(STRING,"Lattice energy included in chi^2: %s\n",
		    yesnostr(KEAT_PLUS_CHI));
		type_line(STRING);
		break;
	    case 21:
		type_list(STRING,0);
		sprintf(kstring[0],"excl");
		sprintf(kstring[1],"incl");
		sprintf(STRING,"%4s %2s %6s %6s %6s %7s %s\n",
			"atom","el","x-pos","y-pos","z-pos","energy",
			"bonded to atoms:");
		type_list(STRING,ROWS);
		for (i = 0; i < NSURF; i++)
		    {
		    sprintf(STRING,"%4d %2s %6.3f %6.3f %6.3f %7s",
			i+1,ELEMENT[TS[i]],XSFIT[i],YSFIT[i],ZSFIT[i],
			kstring[IN_KEAT[i]]);
		    type_list(STRING,ROWS);
		    for (j = 0; j < NBOND[i]; j++)
			{
			sprintf(STRING," %1d",BOND_LIST[i][j]+1);
			type_list(STRING,ROWS);
			}
		    type_list("\n",ROWS);
		    }
		break;
	    case 22:
	        type_list(STRING,0);
		angle_search();
		type_list("Atom  List of bond angles\n",ROWS);
		for (i = 0; i < NSURF; i++)
		    {
		    if (IN_KEAT[i])
			{
			sprintf(STRING,"%2d %2s ",i+1,ELEMENT[TS[i]]);
			type_list(STRING,ROWS);
			for (j = 0; j < NANGLE[i]; j++)
			    {
			    sprintf(STRING,"%1d-%1d ",ANGLE_LIST[i][j][0]+1,
				ANGLE_LIST[i][j][1]+1);
			    type_list(STRING,ROWS);
			    }
			type_list("\n",ROWS);
			}
		    }
		break;
	    case 30:
		list_menu("LATTICE ENERGY MENU",
		    energy_menu,energy_length);
		break;
	    case 31:
		stop = TRUE;
		break;
	    }
	}
    }

/***************************************************************************/
float    energy_calc(void)
/***************************************************************************/

    /*
    Calculate lattice energy.
    */

{

    float pos1[3],pos2[3],pos3[3];
    int	i,j;
    float bond_energy,angle_energy;
    float ratiosqr,ratio6;


    bond_energy = 0;
    angle_energy = 0;

    if (POTENTIAL == VKEATING)  /* Use Keating potential */
    {
	for (i = 0; i < NSURF; i++)
	{
	    if (IN_KEAT[i])
	    {
		get_cartesian(pos1,XSFIT[i],YSFIT[i],ZSFIT[i]);

		/* calculate bond-length deformation energy */

		for (j = 0; j < NBOND[i]; j++)
		{
		    get_cartesian(pos2,XSFIT[BOND_LIST[i][j]]+XBSHIFT[i][j],
			YSFIT[BOND_LIST[i][j]]+YBSHIFT[i][j],
			ZSFIT[BOND_LIST[i][j]]);
		    bond_energy += sqr(sqr(pos1[0]-pos2[0])+
			sqr(pos1[1]-pos2[1])+sqr(pos1[2]-pos2[2])-
			sqr(ATOM_RAD[TS[i]]+ATOM_RAD[TS[BOND_LIST[i][j]]]));
		}

		/* calculate bond-angle deformation energy */

		for (j = 0; j < NANGLE[i]; j++)
		{
		    get_cartesian(pos2,
			XSFIT[ANGLE_LIST[i][j][0]]+XASHIFT[i][j][0],
			YSFIT[ANGLE_LIST[i][j][0]]+YASHIFT[i][j][0],
			ZSFIT[ANGLE_LIST[i][j][0]]);
		    get_cartesian(pos3,
			XSFIT[ANGLE_LIST[i][j][1]]+XASHIFT[i][j][1],
			YSFIT[ANGLE_LIST[i][j][1]]+YASHIFT[i][j][1],
			ZSFIT[ANGLE_LIST[i][j][1]]);
		    angle_energy += sqr(
			(pos2[0]-pos1[0])*(pos3[0]-pos1[0])+
			(pos2[1]-pos1[1])*(pos3[1]-pos1[1])+
			(pos2[2]-pos1[2])*(pos3[2]-pos1[2])-cos(EQU_ANGLE)*
			(ATOM_RAD[TS[i]]+ATOM_RAD[TS[ANGLE_LIST[i][j][0]]])*
			(ATOM_RAD[TS[i]]+ATOM_RAD[TS[ANGLE_LIST[i][j][1]]]));
		}
	    }
	}

	return(ALPHA_KEAT*bond_energy+BETA_KEAT*angle_energy);
    }

    if (POTENTIAL == VLENNARDJONES) /* Use Lennard-Jones potential */
    {
	for (i = 0; i < NSURF; i++)
	{
	    if (IN_KEAT[i])
	    {
		get_cartesian(pos1,XSFIT[i],YSFIT[i],ZSFIT[i]);

		/* calculate bond-length deformation energy */

		for (j = 0; j < NBOND[i]; j++)
		{
		    get_cartesian(pos2,XSFIT[BOND_LIST[i][j]]+XBSHIFT[i][j],
			YSFIT[BOND_LIST[i][j]]+YBSHIFT[i][j],
			ZSFIT[BOND_LIST[i][j]]);
		    ratiosqr = sqr(ATOM_RAD[TS[i]]+
			ATOM_RAD[TS[BOND_LIST[i][j]]])/(sqr(pos1[0]-pos2[0])+
			sqr(pos1[1]-pos2[1])+ sqr(pos1[2]-pos2[2]));
		    ratio6 = ratiosqr*ratiosqr*ratiosqr;
		    bond_energy += 1+ratio6*(ratio6-2);
		}
	    }
	}

	return(ALPHA_KEAT*bond_energy);
    }
    return(0.); /* to avoid compiler warning */
}
