/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                       theory.c                                $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: theory.c,v 1.10 2006/03/29 12:13:20 wilcke Exp $

****************************************************************************/

/*
Update 06/12/2005 R. Wilcke (wilcke@esrf.fr)
                  added comments to most procedures to describe the fitting
                  functions used.
Update 05/12/2005 R. Wilcke (wilcke@esrf.fr)
                  gauss_lorentz_combined(): removed superfluous code for second
                  peak (function has only one peak), corrected formula for
                  Lorentz part of function value and derivatives;
                  pseudo_voigt(): corrected formula for Gaussian derivatives.
Update 23/08/2005 R. Wilcke (wilcke@esrf.fr)
                  theory_type(): added command to select the background function
                  as the fit function.
Update 15/08/2005 R. Wilcke (wilcke@esrf.fr)
                  backgrnd() and slit(): changed TH.SHORT[0] from "bkgrd" to
                  "const";
                  theory_type(): added command to select background type.
Update 12/08/2005 R. Wilcke (wilcke@esrf.fr)
                  all routines: prevent buffer overflow for TH.SHORT, TH.NAME
                  and TH.TITLE;
                  power(): changed short name of "Exponent" parameter from "exp"
                  to "expo" to avoid clash with reserved ANAROD keyword "exp".
Update 28/06/2005 R. Wilcke (wilcke@esrf.fr)
                  gauss_lorentz(), lorentz_gauss() and lorentz_double(): call
                  the corresponding function for the first peak instead of
                  doubling the code in the function;
                  add new function backgrnd() to calculate the background part
                  for the theories;
                  backgrnd(): add a new quadratic background term;
                  all theories except slit(): call the new backgrnd() function
                  instead of calculating the background part in the peak
                  function, and make parameter numbering dependent on the number
                  of background parameters as defined by the call to backgrnd();
                  theory_type(): use new enumerated variable "i_th_type" for
                  the indices of the menu;
                  theory_type(): if a theory is selected, store the
                  corresponding command into the COMMAND member of the theory.
Update 28/04/2005 R. Wilcke (wilcke@esrf.fr)
                  gauss_double() and gauss_triple(): call the corresponding
                  function with one Gaussian peak less to calculate the values
                  for the first "n-1" peaks, instead of repeating that code in
                  each routine;
                  theory_type(): add commands "4gauss" and "5gauss";
                  add functions gauss_4peak() and gauss_5peak() for fitting with
                  four and five Gaussian peaks;
Update 01/08/2003 R. Wilcke (wilcke@esrf.fr)
                  theory_type(void): added a "return" command.
Update 27/02/2003 R. Wilcke (wilcke@esrf.fr)
                  Removed all special code for the standing wave version.
Update 25/02/2003 R. Wilcke (wilcke@esrf.fr)
                  theory_type(): added menu item and call for "Combined Gaussian
                  Lorentzian" fit;
                  added function gauss_lorentz_combined() for the new fit type.
Update 25/03/2002 R. Wilcke (wilcke@esrf.fr)
                  theory() and theory_type(): changed abbreviations for menu
                  commands to the agreed set.
Update 25/01/2001 R. Wilcke (wilcke@esrf.fr)
                  modified text for "specx" command to agree with E. Vliegs
                  standard version.
Update 20/11/2000 O. Svensson (svensson@esrf.fr)
                  Added check for negative error estimates for the
		  standing wave version.
Update 09/10/2000 E. Vlieg (vlieg@sci.kun.nl)
 		  Change in all profiles hwhm to fwhm. (Done for complete
 		  equations, not just multiplier).
Update 02/10/2000 O. Svensson (svensson@esrf.fr)
                  Added initialization of TH.NPEAK.
Update 21/09/2000 O. Svensson (svensson@esrf.fr)
                  Added new function "gauss_triple" for fitting three
		  Gaussian peaks. Modified the menu of "theory_type"
		  to find out the number of elements in compile time.
		  Added CVS info.
*/

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define EXT	extern
#include "ana.h"
#undef EXT

/***************************************************************************/
void    gauss_double(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Double Gaussian.

    Fit a spectrum with a function containing two Gaussian peaks.

    See the general remark about the Gaussian function in gauss_single().

    Function: f(x) =
       const + (x - cent1) * slope                            (background)
       + hght1 * e**(-4*ln(2) * (x - cent1)**2 / fwhm1**2)    (1st Gauss peak)
       + hght2 * e**(-4*ln(2) * (x - cent2)**2 / fwhm2**2)    (2nd Gauss peak)

    with:
    1st Gauss peak parameters height = hght1, center = cent1, FWHM = fwhm1
    2nd Gauss peak parameters height = hght2, center = cent2, FWHM = fwhm2
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float xnew2,ex2,arg2,fac2;

    gauss_single(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = gauss_double;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Double Gaussian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 2;
	for(i = npsav - 3; i < npsav; i++)
	    {
	    if(strlen(TH.NAME[i]) < TH_LENGTH - 1)
		strcat(TH.NAME[i],"1");
	    if(strlen(TH.SHORT[i]) < S_LENGTH - 1)
		strcat(TH.SHORT[i],"1");
	    }
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Gaussian2");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos2");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Gaussian2");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght2");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Gaussian2");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"fwhm2");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
	 */
	*y += a[npsav1]*exp(-GAUSIG2*sqr(x-a[npsav])/(sqr(a[npsav2])+1e-10));
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the second Gaussian
	 */
	xnew2 = x-a[npsav];
	arg2 = xnew2/(a[npsav2]+1e-10);
	ex2 = exp(-GAUSIG2*arg2*arg2);
	fac2 = a[npsav1]*ex2*2*GAUSIG2*arg2;
	*y += a[npsav1]*ex2;
	dyda[npsav] = fac2/(a[npsav2]+1e-10);
	dyda[npsav1] = ex2;
	dyda[npsav2] = fac2*arg2/(a[npsav2]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 2;
	TH.PEAKINT[1] = TH.PAR[npsav1]*TH.PAR[npsav2]*GCONST2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[1] = 0;
	    }
	else
	    {
	    TH.PEAKERR[1] = TH.PEAKINT[1]*sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

#ifdef EXTENSIONS /* Svensson */

/***************************************************************************/
void    gauss_triple(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Triple Gaussian.

    Fit a spectrum with a function containing three Gaussian peaks.

    See the general remark about the Gaussian function in gauss_single().

    Function: f(x) =
       const + (x - cent1) * slope                            (background)
       + hght1 * e**(-4*ln(2) * (x - cent1)**2 / fwhm1**2)    (1st Gauss peak)
       + hght2 * e**(-4*ln(2) * (x - cent2)**2 / fwhm2**2)    (2nd Gauss peak)
       + hght3 * e**(-4*ln(2) * (x - cent3)**2 / fwhm3**2)    (3rd Gauss peak)

    with:
    1st Gauss peak parameters height = hght1, center = cent1, FWHM = fwhm1
    2nd Gauss peak parameters height = hght2, center = cent2, FWHM = fwhm2
    3rd Gauss peak parameters height = hght3, center = cent3, FWHM = fwhm3
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float xnew3,ex3,arg3,fac3;

    gauss_double(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = gauss_triple;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Triple Gaussian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 3;
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Gaussian3");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos3");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Gaussian3");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght3");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Gaussian3");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"fwhm3");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
	 */
	*y += a[npsav1]*exp(-GAUSIG2*sqr(x-a[npsav])/(sqr(a[npsav2])+1e-10));
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the third Gaussian
	 */
	xnew3 = x-a[npsav];
	arg3 = xnew3/(a[npsav2]+1e-10);
	ex3 = exp(-GAUSIG2*arg3*arg3);
	fac3 = a[npsav1]*ex3*2*GAUSIG2*arg3;
	*y += a[npsav1]*ex3;
	dyda[npsav] = fac3/(a[npsav2]+1e-10);
	dyda[npsav1] = ex3;
	dyda[npsav2] = fac3*arg3/(a[npsav2]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 3;
	TH.PEAKINT[2] = TH.PAR[npsav1]*TH.PAR[npsav2]*GCONST2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[2] = 0;
	    }
	else
	    {
	    TH.PEAKERR[2] = TH.PEAKINT[2]*sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

#endif /* EXTENSIONS Svensson */
#ifdef EXTENSIONS /* Wilcke */

/***************************************************************************/
void    gauss_4peak(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Four Gaussians.

    See the general remark about the Gaussian function in gauss_single().

    Fit a spectrum with a function containing four Gaussian peaks.

    Function: f(x) =
       const + (x - cent1) * slope                            (background)
       + hght1 * e**(-4*ln(2) * (x - cent1)**2 / fwhm1**2)    (1st Gauss peak)
       + hght2 * e**(-4*ln(2) * (x - cent2)**2 / fwhm2**2)    (2nd Gauss peak)
       + hght3 * e**(-4*ln(2) * (x - cent3)**2 / fwhm3**2)    (3rd Gauss peak)
       + hght4 * e**(-4*ln(2) * (x - cent4)**2 / fwhm4**2)    (4th Gauss peak)

    with:
    1st Gauss peak parameters height = hght1, center = cent1, FWHM = fwhm1
    2nd Gauss peak parameters height = hght2, center = cent2, FWHM = fwhm2
    3rd Gauss peak parameters height = hght3, center = cent3, FWHM = fwhm3
    4th Gauss peak parameters height = hght4, center = cent4, FWHM = fwhm4
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float xnew4,ex4,arg4,fac4;

    gauss_triple(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = gauss_4peak;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"4-peak Gaussian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 4;
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Gaussian4");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos4");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Gaussian4");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght4");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Gaussian4");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"fwhm4");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
	 */
	*y += a[npsav1]*exp(-GAUSIG2*sqr(x-a[npsav])/(sqr(a[npsav2])+1e-10));
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the fourth Gaussian
	 */
	xnew4 = x-a[npsav];
	arg4 = xnew4/(a[npsav2]+1e-10);
	ex4 = exp(-GAUSIG2*arg4*arg4);
	fac4 = a[npsav1]*ex4*2*GAUSIG2*arg4;
	*y += a[npsav1]*ex4;
	dyda[npsav] = fac4/(a[npsav2]+1e-10);
	dyda[npsav1] = ex4;
	dyda[npsav2] = fac4*arg4/(a[npsav2]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 4;
	TH.PEAKINT[3] = TH.PAR[npsav1]*TH.PAR[npsav2]*GCONST2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[3] = 0;
	    }
	else
	    {
	    TH.PEAKERR[3] = TH.PEAKINT[3]*sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

/***************************************************************************/
void    gauss_5peak(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Five Gaussians.

    See the general remark about the Gaussian function in gauss_single().

    Fit a spectrum with a function containing five Gaussian peaks.

    Function: f(x) =
       const + (x - cent1) * slope                            (background)
       + hght1 * e**(-4*ln(2) * (x - cent1)**2 / fwhm1**2)    (1st Gauss peak)
       + hght2 * e**(-4*ln(2) * (x - cent2)**2 / fwhm2**2)    (2nd Gauss peak)
       + hght3 * e**(-4*ln(2) * (x - cent3)**2 / fwhm3**2)    (3rd Gauss peak)
       + hght4 * e**(-4*ln(2) * (x - cent4)**2 / fwhm4**2)    (4th Gauss peak)
       + hght5 * e**(-4*ln(2) * (x - cent5)**2 / fwhm5**2)    (5th Gauss peak)

    with:
    1st Gauss peak parameters height = hght1, center = cent1, FWHM = fwhm1
    2nd Gauss peak parameters height = hght2, center = cent2, FWHM = fwhm2
    3rd Gauss peak parameters height = hght3, center = cent3, FWHM = fwhm3
    4th Gauss peak parameters height = hght4, center = cent4, FWHM = fwhm4
    5th Gauss peak parameters height = hght5, center = cent5, FWHM = fwhm5
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float xnew5,ex5,arg5,fac5;

    gauss_4peak(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = gauss_5peak;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"5-peak Gaussian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 5;
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Gaussian5");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos5");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Gaussian5");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght5");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Gaussian5");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"fwhm5");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
	 */
	*y += a[npsav1]*exp(-GAUSIG2*sqr(x-a[npsav])/(sqr(a[npsav2])+1e-10));
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the fifth Gaussian
	 */
	xnew5 = x-a[npsav];
	arg5 = xnew5/(a[npsav2]+1e-10);
	ex5 = exp(-GAUSIG2*arg5*arg5);
	fac5 = a[npsav1]*ex5*2*GAUSIG2*arg5;
	*y += a[npsav1]*ex5;
	dyda[npsav] = fac5/(a[npsav2]+1e-10);
	dyda[npsav1] = ex5;
	dyda[npsav2] = fac5*arg5/(a[npsav2]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 5;
	TH.PEAKINT[4] = TH.PAR[npsav1]*TH.PAR[npsav2]*GCONST2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[4] = 0;
	    }
	else
	    {
	    TH.PEAKERR[4] = TH.PEAKINT[4]*sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

#endif /* EXTENSIONS Wilcke */
#ifdef EXTENSIONS /* Svensson */

/***************************************************************************/
void    gauss_lorentz_combined(float x, float a[], float *y, float dyda[], 
			       int mode)
/***************************************************************************/

    /*
    Combined Gaussian and Lorentzian.

    Fit a spectrum with a function containing a mixture of a Gaussian and a
    Lorentzian peak:

       f(x) = (1 - eta) * Gauss(x) + eta * Lorentz(x)

    There is only one peak. The Gaussian and the Lorentzian functions have the
    same height, center position and FWHM.

    See the general remarks about the Lorentzian function in lorentz_single()
    and about the Gaussian function in gauss_single().

    Function: f(x) =
       const + (x - cent) * slope                                 (background)
       + hght * eta * fwhm**2 / (4 * (x - cent)**2 + fwhm**2)     (Lorentz part)
       + hght * (1-eta) * e**(-4*ln(2) * (x - cent)**2 / fwhm**2) (Gauss part)

    with:
    Lorentz and Gauss peak parameters height = hght, center = cent, FWHM = fwhm
    and eta = Lorentz fraction of the combined peak function.
    */

    {

    static int npsav,npsav1,npsav2,npsav3;
    int i;
    float xnew,arg,ex,num,den,lor;

    backgrnd(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = gauss_lorentz_combined;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Combined Gaussian Lorentzian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	npsav3 = npsav+3;
	TH.NPAR += 4;
	TH.NPEAK = 1;
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"fwhm");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Lorentz fraction eta");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"eta");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
         */
	*y += a[npsav1]*(
            a[npsav3]*sqr(a[npsav2])/(4*sqr(x-a[npsav])+sqr(a[npsav2])+1e-10)+
	    (1-a[npsav3])*exp(-GAUSIG2*sqr(x-a[npsav])/(sqr(a[npsav2])+1e-10)));
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the combined Gaussian
	 * and Lorentzian
         */
	xnew = x-a[npsav];
	arg = xnew/(a[npsav2]+1e-10);
	ex = exp(-GAUSIG2*arg*arg);
	num = a[npsav2]*a[npsav2];
	den = 4.0*xnew*xnew+num+1e-10;
	lor = num/den;
	*y += a[npsav1]*(a[npsav3]*lor+(1.0-a[npsav3])*ex);
	dyda[npsav] += a[npsav1]*(a[npsav3]*8.0*xnew*lor/den+
	    (1.0-a[npsav3])*ex*2.0*GAUSIG2*arg/a[npsav2]);
	dyda[npsav1] = a[npsav3]*lor+(1-a[npsav3])*ex;
	dyda[npsav2] = a[npsav1]*2.0*(a[npsav3]*a[npsav2]*(1.0-lor)/den+
	    (1-a[npsav3])*ex*GAUSIG2*arg*arg/a[npsav2]);
	dyda[npsav3] = a[npsav1]*(lor-ex);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 1;
	TH.PEAKINT[0] = TH.PAR[npsav1]*TH.PAR[npsav2]*GCONST2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]*sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

#endif /* EXTENSIONS Svensson */

/***************************************************************************/
void    gauss_lorentz(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Gaussian plus Lorentzian.

    Fit a spectrum with a function containing one Gaussian and one Lorentzian
    peak.

    The background is symmetric with respect to the Gaussian peak position.

    See the general remarks about the Lorentzian function in lorentz_single()
    and about the Gaussian function in gauss_single().

    Function: f(x) =
       const + (x - cent1) * slope                            (background)
       + hght1 * e**(-4*ln(2) * (x - cent1)**2 / fwhm1**2)    (Gauss peak)
       + hght2 * fwhm2**2 / (4 * (x - cent2)**2 + fwhm2**2)   (Lorentz peak)

    with:
    Gauss   peak parameters height = hght1, center = cent1, FWHM = fwhm1
    Lorentz peak parameters height = hght2, center = cent2, FWHM = fwhm2
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float xnew2,num2,den2;

    gauss_single(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = gauss_lorentz;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Gaussian plus Lorentzian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 2;

	i = npsav-3;
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"gpos");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"ghght");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"gfwhm");

	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Lorentzian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"lpos");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Lorentzian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"lhght");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Lorentzian");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"lfwhm");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
	 */
	*y += a[npsav1]*sqr(a[npsav2])/(4*sqr(x-a[npsav])+sqr(a[npsav2])+1e-10);
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the Lorentzian
	 */
	xnew2 = x-a[npsav];
	num2 = a[npsav1]*a[npsav2]*a[npsav2];
	den2 = 4*xnew2*xnew2+a[npsav2]*a[npsav2]+1e-10;
	*y += num2/den2;
	dyda[npsav] = 8*xnew2*num2/(den2*den2);
	dyda[npsav1] = a[npsav2]*a[npsav2]/den2;
	dyda[npsav2] = 2*a[npsav2]*(a[npsav1]-num2/den2)/den2;
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 2;
	TH.PEAKINT[1] = TH.PAR[npsav1]*TH.PAR[npsav2]*PI/2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[1] = 0;
	    }
	else
	    {
	    TH.PEAKERR[1] = TH.PEAKINT[1]*sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

/***************************************************************************/
void    gauss_single(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Single Gaussian.

    Fit a spectrum with a function containing one Gaussian peak.

    The programmed form of the Gauss function is

       f(x) = hght * e**(-4 * ln(2) * (x - cent)**2 / fwhm**2)

    Function: f(x) =
       const + (x - cent) * slope                             (background)
       + hght * e**(-4*ln(2) * (x - cent)**2 / fwhm**2)       (Gauss peak)

    with:
    Gauss peak parameters height = hght, center = cent, FWHM = fwhm
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float       xnew,ex,arg,fac;

    backgrnd(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = gauss_single;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Single Gaussian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 1;
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Gaussian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Gaussian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Gaussian");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"fwhm");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
	 */
	*y += a[npsav1]*exp(-GAUSIG2*sqr(x-a[npsav])/(sqr(a[npsav2])+1e-10));
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the Gaussian
	 */
	xnew = x-a[npsav];
	arg = xnew/(a[npsav2]+1e-10);
	ex = exp(-GAUSIG2*arg*arg);
	fac = a[npsav1]*ex*2*GAUSIG2*arg;
	*y += a[npsav1]*ex;
	dyda[npsav] += fac/(a[npsav2]+1e-10);
	dyda[npsav1] = ex;
	dyda[npsav2] = fac*arg/(a[npsav2]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 1;
	TH.PEAKINT[0] = TH.PAR[npsav1]*TH.PAR[npsav2]*GCONST2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]*sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

/***************************************************************************/
void    lorentz_gauss(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Lorentzian plus Gaussian.

    Fit a spectrum with a function containing one Lorentzian and one Gaussian
    peak.

    The background is symmetric with respect to the Lorentzian peak position.

    See the general remarks about the Lorentzian function in lorentz_single()
    and about the Gaussian function in gauss_single().

    Function: f(x) =
       const + (x - cent1) * slope                            (background)
       + hght1 * fwhm1**2 / (4 * (x - cent1)**2 + fwhm1**2)   (Lorentz peak)
       + hght2 * e**(-4*ln(2) * (x - cent2)**2 / fwhm2**2)    (Gauss peak)

    with:
    Lorentz peak parameters height = hght1, center = cent1, FWHM = fwhm1
    Gauss   peak parameters height = hght2, center = cent2, FWHM = fwhm2
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float xnew2,ex2,arg2,fac2;

    lorentz_single(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = lorentz_gauss;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Lorentzian plus Gaussian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 2;
	i = npsav-3;
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"lpos");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"lhght");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"lfwhm");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Gaussian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"gpos");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Gaussian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"ghght");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Gaussian");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"gfwhm");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
	 */
	*y += a[npsav1]*exp(-GAUSIG2*sqr(x-a[npsav])/(sqr(a[npsav2])+1e-10));
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the Gaussian
         */
	xnew2 = x-a[npsav];
	arg2 = xnew2/(a[npsav2]+1e-10);
	ex2 = exp(-GAUSIG2*arg2*arg2);
	fac2 = a[npsav1]*ex2*2*GAUSIG2*arg2;
	*y += a[npsav1]*ex2;
	dyda[npsav] = fac2/(a[npsav2]+1e-10);
	dyda[npsav1] = ex2;
	dyda[npsav2] = fac2*arg2/(a[npsav2]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 2;
	TH.PEAKINT[1] = TH.PAR[npsav1]*TH.PAR[npsav2]*GCONST2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[1] = 0;
	    }
	else
	    {
	    TH.PEAKERR[1] = TH.PEAKINT[1]*sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

/***************************************************************************/
void    lorentz_double(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Double Lorentzian.

    Fit a spectrum with a function containing two Lorentzian peaks.

    See the general remark about the Lorentzian function in lorentz_single().

    Function: f(x) =
       const + (x - cent1) * slope                            (background)
       + hght1 * fwhm1**2 / (4 * (x - cent1)**2 + fwhm1**2)   (1st Lorentz peak)
       + hght2 * fwhm2**2 / (4 * (x - cent2)**2 + fwhm2**2)   (2st Lorentz peak)

    with:
    1st Lorentz peak parameters height = hght1, center = cent1, FWHM = fwhm1
    2nd Lorentz peak parameters height = hght2, center = cent2, FWHM = fwhm2
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float xnew2,num2,den2;

    lorentz_single(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = lorentz_double;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Double Lorentzian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 2;
	for(i = npsav - 3; i < npsav; i++)
	    {
	    if(strlen(TH.NAME[i]) < TH_LENGTH - 1)
		strcat(TH.NAME[i],"1");
	    if(strlen(TH.SHORT[i]) < S_LENGTH - 1)
		strcat(TH.SHORT[i],"1");
	    }
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Lorentzian2");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos2");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Lorentzian2");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght2");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Lorentzian2");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"fwhm2");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
         */
	*y += a[npsav1]*sqr(a[npsav2])/(4*sqr(x-a[npsav])+sqr(a[npsav2])+1e-10);
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the second Lorentzian
         */
	xnew2 = x-a[npsav];
	num2 = a[npsav1]*a[npsav2]*a[npsav2];
	den2 = 4*xnew2*xnew2+a[npsav2]*a[npsav2]+1e-10;
	*y += num2/den2;
	dyda[npsav] = 8*xnew2*num2/(den2*den2);
	dyda[npsav1] = a[npsav2]*a[npsav2]/den2;
	dyda[npsav2] = 2*a[npsav2]*(a[npsav1]-num2/den2)/den2;
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 2;
	TH.PEAKINT[1] = TH.PAR[npsav1]*TH.PAR[npsav2]*PI/2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[1] = 0;
	    }
	else
	    {
	    TH.PEAKERR[1] = TH.PEAKINT[1]* sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

/***************************************************************************/
void    lorentz_single(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Single Lorentzian.

    Fit a spectrum with a function containing one Lorentzian peak.

    The usual form of the Lorentz function is

       f(x) = hght / (4 * ((x - cent) / fwhm)**2 + 1)

    The function is programmed here in a slightly different way, which however
    is mathematically equivalent.

    Function: f(x) =
       const + (x - cent) * slope                             (background)
       + hght * fwhm**2 / (4 * (x - cent)**2 + fwhm**2)       (Lorentz peak)

    with:
    Lorentz peak parameters height = hght, center = cent, FWHM = fwhm
    */

    {

    static int npsav,npsav1,npsav2;
    int i;
    float xnew,num,den;

    backgrnd(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = lorentz_single;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Single Lorentzian");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	TH.NPAR += 3;
	TH.NPEAK = 1;
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Center position Lorentzian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height Lorentzian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM Lorentzian");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"fwhm");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
         */
	*y += a[npsav1]*sqr(a[npsav2])/(4*sqr(x-a[npsav])+sqr(a[npsav2])+1e-10);
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the Lorentzian
         */
	xnew = x-a[npsav];
	num = a[npsav1]*a[npsav2]*a[npsav2];
	den = 4*xnew*xnew+a[npsav2]*a[npsav2];
	if (den < 1e-10) den = 1e-10;
	*y += num/den;
	dyda[npsav] += 8*xnew*num/(den*den);
	dyda[npsav1] = a[npsav2]*a[npsav2]/den;
	dyda[npsav2] = 2*a[npsav2]*(a[npsav1]-num/den)/den;
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 1;
	TH.PEAKINT[0] = TH.PAR[npsav1]*TH.PAR[npsav2]*PI/2;
	if ((fabs(TH.PAR[npsav1]) < 1e-6) || (fabs(TH.PAR[npsav2]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]* sqrt(sqr(TH.ERR[npsav1]/
		TH.PAR[npsav1])+sqr(TH.ERR[npsav2]/TH.PAR[npsav2]));
	    }
	}
    }

/***************************************************************************/
void    power(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Exponential function, but with small constant added to denumerator,
    to prevent crashing. Should work fine away from peak.

    Fit a spectrum with a function that has an exponential behaviour in the
    denominator:

    Function: f(x) =
       const + (x - cent) * slope                             (background)
       + hght / (4 * (x - cent)**2 / fwhm**2 + 1)**(expo / 2) (exponential part)

    with:
    peak parameters height = hght, center = cent, FWHM = fwhm,
    and expo = exponent of the function.
    */

    {

    static int npsav,npsav1,npsav2,npsav3;
    int i;
    float xnew,arg,denom;

    backgrnd(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = power;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Power law");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	npsav3 = npsav+3;
	TH.NPAR += 4;
	TH.NPEAK = 0;
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Peak position");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Amplitude");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Width");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"wdth");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Exponent");
	sprintf(TH.SHORT[i],"expo");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
         */
	*y += a[npsav1]/pow(4*(x-a[npsav])*(x-a[npsav])/a[npsav2]/a[npsav2]+1,
	    a[npsav3]/2);
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the exponential
	 * function
         */
	xnew = x-a[npsav];
	arg = 4*xnew*xnew/a[npsav2]/a[npsav2]+1;
	denom = pow(arg,a[npsav3]/2);
	*y += a[npsav1]/denom;
	dyda[npsav] += 4*a[npsav1]*a[npsav3]*xnew/
	    (denom*arg*a[npsav2]*a[npsav2]);
	dyda[npsav1] = 1/denom;
	dyda[npsav2] = 4*a[npsav1]*a[npsav3]*xnew*xnew/
	    (denom*arg*a[npsav2]*a[npsav2]*a[npsav2]);
	dyda[npsav3] = -0.5*a[npsav1]*log(arg)/denom;
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 0;
	}
    }

/***************************************************************************/
void    pseudo_voigt(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Pseudo Voigt function with independent Lorentzian and Gaussian widths.

    Fit a spectrum with a function containing a mixture of a Gaussian and a
    Lorentzian peak:

       f(x) = (1 - eta) * Gauss(x) + eta * Lorentz(x)

    There is only one peak. The Gaussian and the Lorentzian functions have the
    same height and center position, but different FWHMs.

    See the general remarks about the Lorentzian function in lorentz_single()
    and about the Gaussian function in gauss_single().

    Function: f(x) =
       const + (x - cent) * slope                                   (background)
       + hght * eta * fwhm1**2 / (4 * (x - cent)**2 + fwhm1**2)   (Lorentz part)
       + hght * (1-eta) * e**(-4*ln(2) * (x - cent)**2 / fwhm2**2)  (Gauss peak)

    with:
    Lorentz and Gauss common peak parameters height = hght, center = cent,
    Lorentz FWHM = fwhm1, Gauss FWHM = fwhm2,
    and eta = Lorentz fraction of the Pseudo Voigt function.
    */

    {

    static int npsav,npsav1,npsav2,npsav3,npsav4;
    int i;
    float xnew,num1,den1,lor,arg2,fac2,ex;

    backgrnd(x,a,y,dyda,mode);

    if (mode == INIT)
	{
	F_PTR = pseudo_voigt;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Pseudo Voigt");
	npsav = TH.NPAR;
	npsav1 = npsav+1;
	npsav2 = npsav+2;
	npsav3 = npsav+3;
	npsav4 = npsav+4;
	TH.NPAR += 5;
	TH.NPEAK = 0;
	i = npsav;
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Peak position");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"pos");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Height of peak");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"hght");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM of Lorentzian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"lfwhm");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"FWHM of Gaussian");
	sprintf(TH.SHORT[i++],"%.*s",S_LENGTH-1,"gfwhm");
	sprintf(TH.NAME[i],"%.*s",TH_LENGTH-1,"Lorentz fraction eta");
	sprintf(TH.SHORT[i],"%.*s",S_LENGTH-1,"eta");
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
         */
	*y += a[npsav1]*(
	    a[npsav4]*sqr(a[npsav2])/(4*sqr(x-a[npsav])+sqr(a[npsav2])+1e-10)+
	    (1-a[npsav4])*exp(-GAUSIG2*sqr(x-a[npsav])/(sqr(a[npsav3])+1e-10)));
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the parameters of the Lorentzian and
	 * Gaussian functions
         */
	xnew = x-a[npsav];
	num1 = a[npsav2]*a[npsav2];
	den1 = 4.0*xnew*xnew+num1+1e-10;
	lor = num1/den1;
	arg2 = xnew/(a[npsav3]+1e-10);
	ex = exp(-GAUSIG2*arg2*arg2);
	fac2 = ex*2*GAUSIG2*arg2/(a[npsav3]+1e-10);
	*y += a[npsav1]*(a[npsav4]*lor+(1-a[npsav4])*ex);
	dyda[npsav] += a[npsav1]*(a[npsav4]*8*xnew*lor/den1+(1-a[npsav4])*fac2);
	dyda[npsav1] = a[npsav4]*lor+(1-a[npsav4])*ex;
	dyda[npsav2] = 2*a[npsav1]*a[npsav4]*a[npsav2]*(1.0-lor)/den1;
	dyda[npsav3] = a[npsav1]*(1-a[npsav4])*fac2*arg2;
	dyda[npsav4] = a[npsav1]*(lor-ex);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 0;
	}
    }

/***************************************************************************/
void    slit(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Slit function.
    */

    {

    float       smear,xstep,xstart,xval,ysum,ssum;
    int		i;
#define	smearsteps	21

    if (mode == INIT)
	{
	F_PTR = slit;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Slit");
	TH.NPAR = 9;
	TH.NBCK = 2;
	TH.NPEAK = 0;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Center of slit function");
	sprintf(TH.NAME[3],"Height of slit function");
	sprintf(TH.NAME[4],"Slit opening");
	sprintf(TH.NAME[5],"Blade offset");
	sprintf(TH.NAME[6],"Screen distance");
	sprintf(TH.NAME[7],"Wavelength");
	sprintf(TH.NAME[8],"Smearing");
	sprintf(TH.SHORT[0],"const");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"pos");
	sprintf(TH.SHORT[3],"hght");
	sprintf(TH.SHORT[4],"s");
	sprintf(TH.SHORT[5],"h");
	sprintf(TH.SHORT[6],"l");
	sprintf(TH.SHORT[7],"lam");
	sprintf(TH.SHORT[8],"smear");
	}
    else if (mode == CALC)
	{
	xstep = 4*a[8]/(smearsteps-1.);
	xstart = -xstep*(smearsteps-1.)/2.;
	ysum = 0;
	ssum = 0;
	for (i = 0; i < smearsteps; i++)
	    {
	    xval = x-a[2]+xstart+i*xstep;
	    smear = exp(-sqr((xstart+i*xstep)/a[8]));
	    ssum += smear;
	    ysum += smear*(a[0]+xval*a[1]
		+a[3]*sqr(1000*
		sin(
		PI*xval*(a[4]/1000.+xval*a[5]/(a[6]*1000))/(a[6]*a[7]/10000)
		   )
		/(PI*xval/(a[6]*a[7]/10000)+1e-10)
		     ));
	    }
	    *y = ysum/ssum;
	}
    else if (mode == DERV)
	{
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 0;
	}
    }

/***************************************************************************/
void    backgrnd(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Background calculation.

    There are several background functions available, which are selected with
    the TH.BCKTYP parameter:

    - TH.BCKTYP = 0  linear background (default)
    - TH.BCKTYP = 1  quadratic background

    Function: const + slope * (x - cent)                        (linear)

              const + slope * (x - cent) + quadr * (x - cent)   (quadratic)

    with: const, slope and quadr the fit parameters for the constant, linear
                 and quadratic part of the background function; and
          cent = symmetry center point (usually first peak position)
    */

    {

    static int npsav;
    float xnew;

    if (mode == INIT)
	{
	F_PTR = backgrnd;
	sprintf(TH.TITLE,"%.*s",TH_LENGTH-1,"Background");
	TH.NPAR = 3;
	TH.NBCK = TH.NPAR;
	npsav = TH.NPAR;
	TH.NPEAK = 0;
	sprintf(TH.NAME[0],"%.*s",TH_LENGTH-1,"Constant background");
	sprintf(TH.SHORT[0],"%.*s",S_LENGTH-1,"const");
	sprintf(TH.NAME[1],"%.*s",TH_LENGTH-1,"Slope of background");
	sprintf(TH.SHORT[1],"%.*s",S_LENGTH-1,"slope");
	sprintf(TH.NAME[2],"%.*s",TH_LENGTH-1,"Quadratic background");
	sprintf(TH.SHORT[2],"%.*s",S_LENGTH-1,"quadr");
	sprintf(TH.NAME[TH.NBCK],"%.*s",TH_LENGTH-1,"Symmetry center");
	sprintf(TH.SHORT[TH.NBCK],"%.*s",S_LENGTH-1,"centr");
        /*
         * Set some of the background parameters to "fixed" and value 0.
         * depending on the background type selected.
         */
	TH.FIX[2] = 0;
        if(TH.BCKTYP == 0)
	    {
	    TH.FIX[2] = 1;
	    TH.PAR[2] = 0.;
	    }
	}
    else if (mode == CALC)
	{
	/*
	 * Function value
	 */
        xnew = x-a[npsav];
	*y = a[0]+xnew*(a[1]+xnew*a[2]);
	}
    else if (mode == DERV)
	{
	/*
	 * Derivatives with respect to the background parameters
	 */
        xnew = x-a[npsav];
	*y = a[0]+xnew*(a[1]+xnew*a[2]);
	dyda[0] = 1.;
	dyda[1] = xnew;
	dyda[2] = xnew*xnew;
	dyda[npsav] = -a[1]-2*a[2]*xnew;
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 0;
	}
    }

/***************************************************************************/
void    theory(void)
/***************************************************************************/

    /*
    Compute theoretical spectrum.
    */

    {

    /* define theory_menu */

    static struct   MENU theory_menu[] =
	{
	"theory",    1,  1,  "Choose type of theory",
	"lowerx",    2,  2,  "Lower x-value",
	"upperx",    1,  3,  "Upper x-value",
	"stepx",     1,  4,  "Step size in x",
	"specx",     2,  5,  "Use x-range of spectrum (if 0, use range)",
	"value",     1,  6,  "Set parameter value",
	"compute",   1, 10,  "Compute theoretical spectrum",
	"list",      1, 11,  "List parameter values",
	"help",      1, 20,  "Display menu",
	"return",    1, 21,  "Return to main menu",
	};

    /* number of commands in th_type menu */
    int theory_length = sizeof(theory_menu) / sizeof(theory_menu[0]);

    int     stop = FALSE;
    char    token[100];
    static float   lowerx,upperx,stepx;
    int     parameter,i;
    static int  spectrum = 0;
    int     outspec = 1;
    float   *a,*dyda;

    while (!stop)
	{
	type_header(0);
	while (!get_token(token,"ANA.THEORY>")) return;
	switch (cmnd_match(token,theory_menu,theory_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		theory_type();
		break;
	    case 2:
		sprintf(STRING,"Lower value of x-range [%6.3f]: ",lowerx);
		lowerx = get_real(lowerx,STRING);
		break;
	    case 3:
		sprintf(STRING,"Upper value of x-range [%6.3f]: ",upperx);
		upperx = get_real(upperx,STRING);
		break;
	    case 4:
		sprintf(STRING,"Step size of x-range [%6.3f]: ",stepx);
		stepx = get_real(stepx,STRING);
		break;
	    case 5:
		spectrum = get_int(0,
		"Spectrum to take x-values from (0 for manual range): ");
		if ((spectrum < 0) || (spectrum > MAX_SPECTRUM))
		    {
		    errtype("Error, invalid spectrum number");
		    clear_command();
		    spectrum = 0;
		    }
		if ((spectrum != 0) && (DATA[spectrum-1].NPNT < 1))
		    {
		    errtype("Error, no data in spectrum");
		    clear_command();
		    spectrum = 0;
		    }
		break;
	    case 6:
		parameter = get_int(1,"Parameter number to change [1]: ");
		if ((parameter < 1) || (parameter > TH.NPAR))
		    {
		    errtype("ERROR, parameter out of range");
		    clear_command();
		    }
		else
		    {
		    parameter -= 1;
		    sprintf(STRING,"%s [%7.3f]: ",TH.NAME[parameter],
			TH.PAR[parameter]);
		    TH.PAR[parameter] = get_real(TH.PAR[parameter],STRING);
		    }
		break;
	    case 10:

		/* get output spectrum number */

		get_inspec(&outspec);

		/* Compute x-values */

		if (spectrum == 0)
		    {
		    if (fabs(stepx) < 1e-6) stepx = upperx-lowerx;
		    if ((upperx-lowerx)*stepx < 0) stepx *= -1;
		    DATA[outspec].NPNT = (int) ((upperx-lowerx)/stepx+1.5);
		    if (DATA[outspec].NPNT > MAX_DATA)
			DATA[outspec].NPNT = MAX_DATA;
		    for (i = 0; i < DATA[outspec].NPNT; i++)
			{
			DATA[outspec].X[i] = lowerx+i*stepx;
			}
		    }
		else
		    {
		    DATA[outspec].NPNT = DATA[spectrum-1].NPNT;
		    for (i = 0; i < DATA[outspec].NPNT; i++)
			{
			DATA[outspec].X[i] = DATA[spectrum-1].X[i];
			}
		    }

		/* Calculate y-values, set error to unity */

		(*F_PTR)(DATA[outspec].X[0],TH.PAR,&DATA[outspec].Y[0],
		    dyda,CALC);
		DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
		DATA[outspec].E[0] = 1.;
		for (i = 1; i < DATA[outspec].NPNT; i++)
		    {
		    (*F_PTR)(DATA[outspec].X[i],TH.PAR,&DATA[outspec].Y[i],
			dyda,CALC);
		    DATA[outspec].E[i] = 1.;
		    if (DATA[outspec].Y[i] < DATA[outspec].MINY)
			DATA[outspec].MINY = DATA[outspec].Y[i];
		    if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
			DATA[outspec].MAXY = DATA[outspec].Y[i];
		    }

		/* Generate title */

		sprintf(DATA[outspec].TITLE,"%s",TH.TITLE);
		type_header(0);
		break;

	    case 11:
		sprintf(STRING,"Theory: %s\n",TH.TITLE);
		type_line(STRING);
		for (i = 0; i < TH.NPAR; i++)
		    {
		    sprintf(STRING,"%2d %-30s = %8.3f\n",
			i+1,TH.NAME[i],TH.PAR[i]);
		    type_line(STRING);
		    }
		if (spectrum == 0)
		    {
		    sprintf(STRING,
		    "Range of x-axis, lower %6.3f, upper %6.3f, step %6.3f\n",
			lowerx,upperx,stepx);
		    }
		else
		    {
		    sprintf(STRING,"x-values equal to spectrum %1d\n",
			spectrum);
		    }
		type_line(STRING);
		break;
	    case 20:
		list_menu("THEORY MENU",theory_menu,theory_length);
		break;
	    case 21:
		stop = TRUE;
		break;
	    }
	}
    }

/***************************************************************************/
void    theory_type(void)
/***************************************************************************/

    /*
    Get type of theory.
    */

    {

    /* define th_type_menu */

    enum i_th_type {
	i_zero,
	i_lorentz,
	i_llorentz,
	i_gauss,
	i_ggauss,
#ifdef EXTENSIONS /* Svensson */
	i_3gauss,
#endif /* EXTENSIONS Svensson */
#ifdef EXTENSIONS /* Wilcke */
	i_4gauss,
	i_5gauss,
#endif /* EXTENSIONS Wilcke */
	i_lgauss,
	i_glorentz,
#ifdef EXTENSIONS /* Svensson */
	i_combgaussl,
#endif /* EXTENSIONS Svensson */
	i_power,
	i_voigt,
	i_slit,
#ifdef EXTENSIONS /* Wilcke */
	i_bckgd,
	i_btype,
#endif /* EXTENSIONS Wilcke */
	i_help,
	i_return
	};

    static struct   MENU th_type_menu[] =
	{
	"lorentz",    2, i_lorentz,    "Single Lorentzian",
	"llorentz",   2, i_llorentz,   "Double Lorentzian",
	"gauss",      1, i_gauss,      "Single Gaussian",
	"ggauss",     2, i_ggauss,     "Double Gaussian",
#ifdef EXTENSIONS /* Svensson */
	"3gauss",     2, i_3gauss,     "Triple Gaussian",
#endif /* EXTENSIONS Svensson */
#ifdef EXTENSIONS /* Wilcke */
	"4gauss",     2, i_4gauss,     "4-peak Gaussian",
	"5gauss",     2, i_5gauss,     "5-peak Gaussian",
#endif /* EXTENSIONS Wilcke */
	"lgauss",     2, i_lgauss,     "Lorentzian plus Gaussian",
	"glorentz",   2, i_glorentz,   "Gaussian plus Lorentzian",
#ifdef EXTENSIONS /* Svensson */
	"combgaussl", 2, i_combgaussl, "Combined Gaussian/Lorentzian",
#endif /* EXTENSIONS Svensson */
	"power",      1, i_power,      "Power-law lineshape",
	"voigt",      1, i_voigt,      "Pseudo-Voigt lineshape",
	"slit",       1, i_slit,       "Slit function",
#ifdef EXTENSIONS /* Wilcke */
	"backgrnd",   2, i_bckgd,      "Background function fit",
	"btype",      2, i_btype,      "Select background type",
#endif /* EXTENSIONS Wilcke */
	"help",       1, i_help,       "Display menu",
	"return",     1, i_return,     "Return to main menu",
	};

    /* number of commands in th_type menu */
    int th_type_length = sizeof(th_type_menu) / sizeof(th_type_menu[0]);

    char    token[100];
    int     imatch,type;
    float   *a,*dyda,y;

    while (TRUE)
	{
	type_header(0);
	while (!get_token(token,"Give type of theory: "));
	imatch = cmnd_match(token,th_type_menu,th_type_length);

#ifdef EXTENSIONS /* Wilcke */
	/*
	 * If the token is a command that selects a theory, store it into the
         * COMMAND member of the TH structure.
         */
	if(imatch > i_zero && imatch < i_btype)
	    sprintf(TH.COMMAND,"%.*s",S_LENGTH - 1,token);
#endif /* EXTENSIONS Wilcke */
	switch (imatch)
	    {
	    case i_lorentz:
		lorentz_single(0.,a,&y,dyda,INIT);
		return;
	    case i_llorentz:
		lorentz_double(0.,a,&y,dyda,INIT);
		return;
	    case i_gauss:
		gauss_single(0.,a,&y,dyda,INIT);
		return;
	    case i_ggauss:
		gauss_double(0.,a,&y,dyda,INIT);
		return;
#ifdef EXTENSIONS /* Svensson */
	    case i_3gauss:
		gauss_triple(0.,a,&y,dyda,INIT);
		return;
#endif /* EXTENSIONS Svensson */
#ifdef EXTENSIONS /* Wilcke */
	    case i_4gauss:
		gauss_4peak(0.,a,&y,dyda,INIT);
		return;
	    case i_5gauss:
		gauss_5peak(0.,a,&y,dyda,INIT);
		return;
#endif /* EXTENSIONS Wilcke */
	    case i_lgauss:
		lorentz_gauss(0.,a,&y,dyda,INIT);
		return;
	    case i_glorentz:
		gauss_lorentz(0.,a,&y,dyda,INIT);
		return;
#ifdef EXTENSIONS /* Svensson */
	    case i_combgaussl:
		gauss_lorentz_combined(0.,a,&y,dyda,INIT);
		return;
#endif /* EXTENSIONS Svensson */
	    case i_power:
		power(0.,a,&y,dyda,INIT);
		return;
	    case i_voigt:
		pseudo_voigt(0.,a,&y,dyda,INIT);
		return;
	    case i_slit:
		slit(0.,a,&y,dyda,INIT);
		return;
#ifdef EXTENSIONS /* Wilcke */
	    case i_bckgd:
		backgrnd(0.,a,&y,dyda,INIT);
		TH.FIX[TH.NPAR] = TRUE;
		TH.NPAR++;
		return;
	    case i_btype:
		type = TH.BCKTYP;
		sprintf(STRING,
		    "Background type: 0 = linear, 1 = quadratic [%1d]: ",type);
		if((type = get_int(type,STRING)) != 0 && type != 1)
		    {
		    errtype("Error, invalid background type");
		    clear_command();
		    }
		else
		    /*
		     * If a valid background type was selected, re-initialize
		     * the theory with the new background type. If no theory
		     * is selected, just initialize the background function.
		     */
		    {
		    TH.BCKTYP = type;
		    if(F_PTR != 0)
			(*F_PTR)(0.,a,&y,dyda,INIT);
		    else
			backgrnd(0.,a,&y,dyda,INIT);
		    }
		break;
#endif /* EXTENSIONS Wilcke */
	    case i_help:
		list_menu("THEORY TYPES",th_type_menu,th_type_length);
		break;
	    case i_return:
		return;
	    }
	}
    }
