/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                         read.c                                $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: read.c,v 1.20 2007/03/19 13:07:23 wilcke Exp $

****************************************************************************/

/*
Update 08/11/2006 R. Wilcke (wilcke@esrf.fr)
                  read_fit(): automatically recognize if input file has the
                  additional parameters corresponding to Robach's extension,
                  and assigns the input values to the variables accordingly.
Update 07/11/2006 R. Wilcke (wilcke@esrf.fr)
                  read_fit(): corrected comments concerning parameter "NOCCUP2".
Update 26/10/2006 R. Wilcke (wilcke@esrf.fr)
                  read_bulk(): automatically recognize if input file has the
                  additional parameter NOCCUPB.
Update 22/03/2006 R. Wilcke (wilcke@esrf.fr)
                  modified code for MacIntosh MACOSX using predefined macro
                  __APPLE__ for this architecture.
Update 19/01/2004 R. Wilcke (wilcke@esrf.fr)
                  removed declarations of fopen() (is declared in "stdio.h").
Update 18/09/2003 O. Svensson (svensson@esrf.fr)
                  Added code written by Michael Dore for the "svensson"
                  extension (deformation of groups).
Update 25/02/2002 O. Svensson (svensson@esrf.fr)
                  Ported changes from standard version:
                    Removed check for __CYGWIN__ since __unix is now defined
                    in Cygwin 1.X
                    Changed menus to follow Rainer's scheme.
Update 18/05/2001 M.C. Saint Lager (stlager@polycnrs-gre.fr)
                  Commented out a printout in the Mcsl extension.
Update 24/04/2001 O.Svensson (svensson@esrf.fr)
                  Added occupancy for groups in the Svensson extension.
Update 15/02/2001 O.Svensson (svensson@esrf.fr)
                  Replaced UNIX with __unix in the Svensson extension.
Update 27/09/2000 O.Svensson (svensson@esrf.fr)
                  Fixed bug in Odile's (Robach) extension concerning NOCCUP2.
Update 22/09/2000 M.C. Saint Lager (stlager@polycnrs-gre.fr)
                  When using Mcsl's extension: 2 formats of lecture are possible
                  depending on the chosen calculation of ZSFIT.
                  They are reconized by the 3rd line of the .fit FILE
                  - which must begin by #
                  - the order of the names is not important
                  - a few can be omitted
                  - but the name of each column must be respected
                    if not they are not reconized
                    the value array must be connected to the name title

                  1: #el XS cxl nxl cx2 nx2 YS cyl nyl cy2 ny2
                         ZS czl nzl cz2 nz2 cz3 nz3 dwl dw2 ocl oc2
                     -> in that case NDELTAZS = FALSE

                  2: #el XS cxl nxl cx2 nx2 YS cyl nyl cy2 ny2
                         NP
                         DZS czl nzl cz2 nz2 cz3 nz3 dwl dw2 ocl oc2
                      -> it finds NP => switch NDELTAZS to TRUE

		  NDELTAZS determines the calculation mode of ZSFIT

	 	  1- ZS is defined as in the standard version from the interface
                     substrate/surf
		     => 3 additionnal columns compared to Odile Robach extension
		        Z3CONST = multiplication factor of DISPL[NZ3DIS]
		        NZ3DIS  = serial number of 3rd z-displacement parameter
		        NOCCUP2 = 2nd serial number for the atomic occupation

                     The list of columns becomes:
                       El X XCONST NXDIS X2CONST NX2DIS
                          Y YCONST NYDIS Y2CONST NY2DIS
                          Z ZCONST NZDIS Z2CONST NZ2DIS Z3CONST NZ3DIS
                          NDWS NDWS2 NOCCUP NOCCUP2

		  2- ZS is defined as the interplane distance (NDELTAZS=1)
		     => 1 additionnal column compared to the case (NDELTAZS=0)
		        NZSPLAN = serial number of the plane were the atom is
                                  (has to start at 1)

                     The list of columns becomes:
	 	       El X XCONST NXDIS X2CONST NX2DIS
                          Y YCONST NYDIS Y2CONST NY2DIS
                          NZSPLAN
                          Z ZCONST NZDIS Z2CONST NZ2DIS Z3CONST NZ3DIS
                          NDWS NDWS2 NOCCUP NOCCUP2
Update 26/06/2000 R. Wilcke (wilcke@esrf.fr)
                  replaced everywhere the buffer size for filenames by
                  the ISO-C defined macro FILENAME_MAX;
                  replace UNIX by __unix and CYGWIN by __CYGWIN__ (those are
                  defined by CPP on the corresponding systems).
*/

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#include "rod.h"

/***************************************************************************/
void calc_rlat(float dlat[6], float rlat[6])
/***************************************************************************/

	/*
	Compute reciprocal lattice parameters. The convention used is that
	a[i]*b[j] = d[ij], i.e. no 2PI's in reciprocal lattice.
	*/

{

  double volume,dummy;

  /* compute volume of real lattice cell */

  volume = dlat[0]*dlat[1]*dlat[2]*
	   sqrt(1+2*cos(dlat[3])*cos(dlat[4])*cos(dlat[5])
	   -cos(dlat[3])*cos(dlat[3])
	   -cos(dlat[4])*cos(dlat[4])
	   -cos(dlat[5])*cos(dlat[5]));

  /* compute reciprocal lattice parameters */

  rlat[0] = dlat[1]*dlat[2]*sin(dlat[3])/volume;
  rlat[1] = dlat[0]*dlat[2]*sin(dlat[4])/volume;
  rlat[2] = dlat[0]*dlat[1]*sin(dlat[5])/volume;
  rlat[3] = acos((cos(dlat[4])*cos(dlat[5])-cos(dlat[3]))
	    /(sin(dlat[4])*sin(dlat[5])));
  rlat[4] = acos((cos(dlat[3])*cos(dlat[5])-cos(dlat[4]))
	    /(sin(dlat[3])*sin(dlat[5])));
  rlat[5] = acos((cos(dlat[3])*cos(dlat[4])-cos(dlat[5]))
	    /(sin(dlat[3])*sin(dlat[4])));

}

/***************************************************************************/
void read_bulk(void)
/***************************************************************************/

    /*
    Read file with coordinates of bulk atoms.
    First line: comments
    Second:     direct lattice parameters
    Data lines: El X Y Z (NDW), where
			El      element symbol
			X       x-position in reduced coordinates
			Y       y-position in reduced coordinates
			Z       z-position in reduced coordinates
			NDW     serial number of Debye-Waller par. (optional)
    when using Robach's extension : one additionnal column :
			NOCCUP = serial number of bulk occupancy parameter
    */

{
  char filename[FILENAME_MAX];
  FILE *infile;
  float dlatb[6];
  int end_of_file,dw_column;
  char elbulk[MAXATOMS][3];
  char dummy[20];
  float dwdummy;
  int i,j;
  int newtype,nread,noccdum;

  /* Get filename and open file, return if not successful */

  get_token(filename,"Filename with bulk atoms (.bul): ");
  add_extension(filename,"bul");
  if ((infile = fopen(filename,"r")) == NULL)
  {
    sprintf(STRING,"Failed to open file '%s'",filename);
    errtype(STRING);
    clear_command();
    return;
  }

  /* Read comments and lattice parameters */

  fgets(INLINE,INLLEN,infile);
  sprintf(STRING,"Title: %s",INLINE);
  type_line(STRING);
  fgets(INLINE,INLLEN,infile);
  sscanf(INLINE,"%f%f%f%f%f%f",&dlatb[0],&dlatb[1],&dlatb[2],&dlatb[3],
    &dlatb[4],&dlatb[5]);
  for (i = 3; i < 6; i++)
    dlatb[i] /= RAD;

  /*
   * If a surface file has already been read in, check the lattice
   * parameters for consistency
   */
#ifdef EXTENSIONS /* Svensson */
  if (svensson_flag)
    if ((NSURF+NCYLINDERS+NELLIPSOIDS) == 0)
    {
       for (i = 0; i < 6; i++) DLAT[i] = dlatb[i];
       calc_rlat(DLAT,RLAT);
    }
    else
    {
      for (i = 0; i < 6; i++)
      {
        if (fabs(DLAT[i]-dlatb[i]) > 1e-4)
    	{
    	  errtype("WARNING, lattice parameters different from surface");
    	}
      }
    }
  else
#endif /* EXTENSIONS Svensson */
    if (NSURF == 0)
    {
      for (i = 0; i < 6; i++)
        DLAT[i] = dlatb[i];
      calc_rlat(DLAT,RLAT);
    }
    else
    {
      for (i = 0; i < 6; i++)
      {
	if (fabs(DLAT[i]-dlatb[i]) > 1e-4)
	{
	  errtype("WARNING, lattice parameters different from surface");
	}
      }
    }

  fgets(INLINE,INLLEN,infile);

  /*
   * Bulk atom files written by the experimental version of ANAROD can contain
   * an additional parameter defined by O. Robach.
   *
   * The program determines automatically from the number of items read
   * whether the file contains Robach's additional bulk occupancy parameter.
   * If so, it fills the corresponding variable NOCCUPB; if not, this variable
   * is not used here.
   */
#if (defined(__unix) || defined(__APPLE__))
  nread = sscanf(INLINE,"%2s%f%f%f%d%d",elbulk[0],
    &XB[0],&YB[0],&ZB[0],&NDWB[0],&noccdum);
#else
  nread-- = sscanf(INLINE,"%[ ]%2s%f%f%f%d%d",dummy,elbulk[0],
    &XB[0],&YB[0],&ZB[0],&NDWB[0],&noccdum);
#endif /* (defined(__unix) || defined(__APPLE__)) */
#ifdef EXTENSIONS /* Robach */
  if(nread >= 6)
    NOCCUPB[0] = noccdum;
#endif /* EXTENSIONS Robach */

  /*
   * Find out from first data line, whether a serial number for Debye-Waller
   * parameters is given.
   */
  if (NDWB[0] != 0)
  {
    dw_column = TRUE;
  }
  else
  {
    dw_column = FALSE;
  }

/* Read the rest of the atoms */

  NBULK = 1;
  while ((fgets(INLINE,INLLEN,infile) != NULL) /*&& (NBULK < MAXATOMS)*/)
#ifdef EXTENSIONS /* Svensson */
  /* Check that we're dealing with a valid atom line */
  if ((!svensson_flag) || (strlen(INLINE) > 3) && (strncmp(INLINE,"  ", 2)))
#endif /* EXTENSIONS Svensson */
  {

    if (dw_column)
    {
#if (defined(__unix) || defined(__APPLE__))
      nread = sscanf(INLINE,"%2s%f%f%f%d%d",elbulk[NBULK],
	&XB[NBULK],&YB[NBULK],&ZB[NBULK],&NDWB[NBULK],&noccdum);
#else
      nread-- = sscanf(INLINE,"%[ ]%2s%f%f%f%d%d",dummy,elbulk[NBULK],
	&XB[NBULK],&YB[NBULK],&ZB[NBULK],&NDWB[NBULK],&noccdum);
#endif /* (defined(__unix) || defined(__APPLE__)) */
#ifdef EXTENSIONS /* Robach */
      if(nread >= 6)
        NOCCUPB[NBULK] = noccdum;
#endif /* EXTENSIONS Robach */
    }
    else
    {
#if (defined(__unix) || defined(__APPLE__))
      sscanf(INLINE,"%2s%f%f%f",elbulk[NBULK],
	&XB[NBULK],&YB[NBULK],&ZB[NBULK]);
#else
      sscanf(INLINE,"%[ ]%2s%f%f%f",dummy,elbulk[NBULK],
	&XB[NBULK],&YB[NBULK],&ZB[NBULK]);
#endif /* (defined(__unix) || defined(__APPLE__)) */
    }

    NBULK++;
    if (NBULK == MAXATOMS)
    {
      errtype("ERROR, maximum number of model atoms exceeded");
      break;
     }
  }
  fclose(infile);

  /* Find the different element types and store them in ELEMENT */

  if (NTYPES == 0)
  {
    NTYPES = 1;
    sprintf(ELEMENT[0],"%s",elbulk[0]);
  }
  for (i = 0; i < NBULK; i++)
  {
    newtype = TRUE;
    for (j = 0; j < NTYPES; j++)
    {
      if (el_equal(elbulk[i],ELEMENT[j])) newtype = FALSE;
    }
    if (newtype)
    {
      if (NTYPES == MAXTYPES)
      {
	errtype("ERROR, too many atom types in unit cell");
	return;
      }
      NTYPES++;
      sprintf(ELEMENT[NTYPES-1],"%s",elbulk[i]);
    }
  }

  /* Assign a type number to all atoms in the model */

  for (i = 0; i < NBULK; i++)
  {
    for (j = 0; j < NTYPES; j++)
    {
      if (el_equal(elbulk[i],ELEMENT[j]))
	TB[i] = j;
    }
  }

  /* Store the coefficients for the computation of the atomic scattering
  factors for all the different elements in F_COEFF */

  for (i = 0; i < NTYPES; i++)
  {
    get_coeff(ELEMENT[i],F_COEFF[i]);
  }

  /* Find highest serial number of Debye-Waller, if read in */

  if (dw_column)
  {
    for (i = 0; i < NBULK; i++)
    {
      if (NDWB[i] > NDWTOT)
      {
	if (NDWB[i] > MAXPAR)
	{
	  sprintf(STRING,
	    "ERROR, total # Debye-Wallers should be < %1d",MAXPAR);
	    errtype(STRING);
	}
	else
	{
	  NDWTOT = NDWB[i];
	}
      }
#ifdef EXTENSIONS /* Robach + Mcsl */
      if ((robach_flag)||(mcsl_flag))
      {
	if (NOCCUPB[i] > NOCCTOT)
	{
          if (NOCCUPB[i] > MAXPAR)
	  {
	    sprintf(STRING,
	      "ERROR, total # Occup. param. should be < %1d", MAXPAR);
	    errtype(STRING);
	  }
	  else
	  {
	    NOCCTOT = NOCCUPB[i];
	  }
	}
		}
#endif /* EXTENSIONS Robach + Mcsl */

      }
   }
}

/***************************************************************************/
void read_data(void)
/***************************************************************************/

    /*
    Read file with truncation rod data.
    First line: comments
    Data:       h       k       l       f       error   lbragg.
    */

    {

  char filename[FILENAME_MAX];
  FILE *infile;
  int i;
  int end_of_file;

  /* Get filename and open file, return if not successful */

  get_token(filename,"Give filename (.dat): ");
  add_extension(filename,"dat");
  if ((infile = fopen(filename,"r")) == NULL)
  {
    sprintf(STRING,"Failed to open file '%s'",filename);
    errtype(STRING);
    clear_command();
    return;
  }

  /* Type comments */

  if (fgets(INLINE,INLLEN,infile) == NULL)
  {
    errtype("ERROR, file is empty");
    fclose(infile);
    clear_command();
    return;
  }
  else
  {
    type_line(INLINE);
  }

  /* Read data */

  NDAT = 0;
  end_of_file = FALSE;

#ifdef EXTENSIONS /* Robach */
  if ((robach_flag==1)&&(var_nlayers==1))
  {
    while ((fgets(INLINE,INLLEN,infile) != NULL) && (!end_of_file))
    {
      if (sscanf(INLINE,"%f%f%f%f%f%f%f",&HDAT[NDAT],&KDAT[NDAT],&LDAT[NDAT],
	&FDAT[NDAT],&ERRDAT[NDAT],&LBR[NDAT],&NL_OR[NDAT]) >= 4)
      {
	NDAT++;
	if (NDAT == MAXDATA)
	{
	  errtype(
	    "ERROR, maximum number of data points exceeded, file truncated");
	  break;
	}
      }
      else
      {
	end_of_file = TRUE;
      }
    }
  }
  else
#endif /* EXTENSIONS Robach */
    while ((fgets(INLINE,INLLEN,infile) != NULL) && (!end_of_file))
    {
      if (sscanf(INLINE,"%f%f%f%f%f%f",&HDAT[NDAT],&KDAT[NDAT],&LDAT[NDAT],
	&FDAT[NDAT],&ERRDAT[NDAT],&LBR[NDAT]) >= 4)
      {
	NDAT++;
	if (NDAT == MAXDATA)
	{
	  errtype(
	    "ERROR, maximum number of data points exceeded, file truncated");
	  break;
	}
    }
    else
    {
      end_of_file = TRUE;
    }
  }
  fclose(infile);
  LBRAGG = LBR[0];
  if (NDAT == 0)
  {
    errtype("No data in file");
    clear_command();
  }

}

/***************************************************************************/
void read_fit(void)
/***************************************************************************/

    /*
    Read file with coordinates of surface atoms and parameters used for
    fitting.
    First line: comments
    Second:     direct lattice parameters
    Data lines: El
		X XCONST NXDIS X2CONST NX2DIS
		Y YCONST NYDIS Y2CONST NY2DIS
		Z NZDIS
		NDWS NDWS2 NOCCUP

		where

		El      element symbol
		X       x-position in reduced coordinates
		XCONST  multiplication factor of NXDIS
		NXDIS   serial number of x-displacement parameter
		X2CONST multiplication factor of NX2DIS
		NX2DIS  serial number of 2nd x-displacement parameter
		Y       y-position in reduced coordinates
		YCONST  multiplication factor of NYDIS
		NYDIS   serial number of y-displacement parameter
		Y2CONST multiplication factor of NY2DIS
		NY2DIS  serial number of 2nd y-displacement parameter
		Z       z-position in reduced coordinates
		NZDIS   serial number of z-displacement parameter
		NDWS    serial number of Debye-Waller parameter in
			the in-plane direction
		NDWS2   serial number of Debye-Waller parameter in
			the perpendicular direction.
		NOCCUP  serial number of occupancy parameter
    */

#ifdef EXTENSIONS /* Robach */
    /* when using Robach's extension: 4 additionnal columns:

		ZCONST  multiplication factor of DISPL[NZDIS]
		Z2CONST multiplication factor of DISPL[NZ2DIS]
		NZ2DIS  serial number of 2nd z-displacement parameter
		NOCCUP2 2nd serial number for the atomic occupation (optional)

	The list of columns becomes:

		El
		X XCONST NXDIS X2CONST NX2DIS
		Y YCONST NYDIS Y2CONST NY2DIS
		Z ZCONST NZDIS Z2CONST NZ2DIS
		NDWS NDWS2 NOCCUP NOCCUP2
    */
#endif /* EXTENSIONS Robach */

#ifdef EXTENSIONS /* Mcsl */
    /* when using Mcsl's extension: 2 formats of lecture are possible depending
       on the calculation of ZSFIT

 	  1- ZS is defined as in the standard version from the interface
             substrate/surf (NDELTAZS=0)
 	     => 2 additionnal columns compared to Odile Robach extension:
		Z3CONST multiplication factor of DISPL[NZ3DIS]
		NZ3DIS  serial number of 3rd z-displacement parameter

		The list of columns becomes:
	  	El
		X XCONST NXDIS X2CONST NX2DIS
		Y YCONST NYDIS Y2CONST NY2DIS
	  	Z ZCONST NZDIS Z2CONST NZ2DIS Z3CONST NZ3DIS
	  	NDWS NDWS2 NOCCUP NOCCUP2

	   2- ZS is defined as the interplane distance (NDELTAZS=1)
	      => 1 additionnal column compared to the case (NDELTAZS=0)
		NZSPLAN serial number of the plane where the atom is

		The list of columns becomes:
	  	El
		X XCONST NXDIS X2CONST NX2DIS
		Y YCONST NYDIS Y2CONST NY2DIS
	  	NZSPLAN
                Z ZCONST NZDIS Z2CONST NZ2DIS Z3CONST NZ3DIS
	  	NDWS NDWS2 NOCCUP NOCCUP2

    */
#endif /* EXTENSIONS Mcsl */

{
  char filename[FILENAME_MAX];
  FILE *infile;
  float dlats[6];
  int end_of_file,dw_column,nread;
  char elsurf[MAXATOMS][3];
  char dummy[20];
  float dwdummy,fdummy[8];
  int i,j;
  int newtype;
#ifdef EXTENSIONS /* Svensson */
  int k,continue_flag,end_of_file_reached;
  float xorg, yorg, zorg, occupancy;
  char keyword[20], name[TITLE_LENGTH];
#endif /* EXTENSIONS Svensson */

#ifdef EXTENSIONS /* Mcsl */
  int automatic = 0;
  NDELTAZS = FALSE;
#endif /* EXTENSIONS Mcsl */


  /* Get filename and open file, return if not successful */

  get_token(filename,"Filename with surface fit model (.fit): ");
  add_extension(filename,"fit");
  if ((infile = fopen(filename,"r")) == NULL)
  {
    sprintf(STRING,"Failed to open file '%s'",filename);
    errtype(STRING);
    clear_command();
    return;
  }

  /* Read comments and lattice parameters */

  fgets(INLINE,INLLEN,infile);
  sprintf(STRING,"Title: %s",INLINE);
  type_line(STRING);
  fgets(INLINE,INLLEN,infile);
  sscanf(INLINE,"%f%f%f%f%f%f",&dlats[0],&dlats[1],&dlats[2],&dlats[3],
    &dlats[4],&dlats[5]);
  for (i = 3; i < 6; i++)
    dlats[i] /= RAD;

  /* If a bulk file has already been read in, check the lattice
  parameters for consistency */

  if (NBULK == 0)
  {
    for (i = 0; i < 6; i++)
      DLAT[i] = dlats[i];
    calc_rlat(DLAT,RLAT);
  }
  else
  {
    for (i = 0; i < 6; i++)
    {
      if (fabs(DLAT[i]-dlats[i]) > 1e-4)
      {
	errtype("ERROR, lattice parameters different from bulk");
	return;
      }
    }
  }

  /* Read the surface fit parameters */

  NSURF = 0;
#ifdef EXTENSIONS /* Svensson */
  if (svensson_flag)
  {
    NGROUPSS = 0;
    NCYLINDERS = 0;
    NELLIPSOIDS = 0;
    end_of_file_reached = FALSE;
  }
#endif /* EXTENSIONS Svensson */
  while ((fgets(INLINE,INLLEN,infile) != NULL) /*&& (NSURF < MAXATOMS)*/)
  {
#ifdef EXTENSIONS /* Svensson */
    if (svensson_flag)
    {
      /* Erase keyword */
      strcpy(keyword, "                    ");
      /* Check if a cylinder keyword is given */
      sscanf(INLINE,"%s %s", keyword, name);
      if (!strncmp(keyword, "cylinder", 8))
      {
	read_cylinder_fit(infile, name);
	if( fgets(INLINE,INLLEN,infile) == NULL)
	  end_of_file_reached = TRUE;
      } /* Cylinder */

      /* Check if an ellipsoid keyword is given */
      sscanf(INLINE,"%s %s", keyword, name);
      if (!strncmp(keyword,"ellipsoid", 9) && (!end_of_file_reached))
      {
	read_ellipsoid_fit(infile, name);
	if( fgets(INLINE,INLLEN,infile) == NULL)
	  end_of_file_reached = TRUE;
      } /* ellipsoid */

      if (!end_of_file_reached)
      {
	xorg = 0.0;
	yorg = 0.0;
	zorg = 0.0;
	occupancy = 1.0;
	sscanf(INLINE,"%s %s %f%f%f%f",keyword,name,&xorg,&yorg,&zorg,
	  &occupancy);
	if (!strncmp(keyword,"group",5))
	{
	  read_group_fit(infile, name, xorg, yorg, zorg, occupancy);
	  continue_flag = 0;
	}
	else
	  continue_flag = 1;
      }
    } /* if (svenssof_flag) */
    if ((!svensson_flag) || (continue_flag && (strlen(INLINE) > 3) &&
      (strncmp(INLINE,"  ", 2))))
    {
#endif /* EXTENSIONS Svensson */

#ifdef EXTENSIONS /* Mcsl */
       /* reading of the surface film parameters */
      if (mcsl_flag)
      {
	if (automatic)
	{
	  read_auto();
	  sscanf(INLINE,"%2s",elsurf[NSURF]);
	  /*
	  sprintf(STRING, " %2s\n",elsurf[NSURF]);
	  type_line(STRING);
	  */
	}
	else if (INLINE[0]=='#')
	{
	  read_name();
	  automatic=TRUE;
	  NSURF -= 1;
	}
      } else {
#endif /* EXTENSIONS Mcsl */

	/*
         * Surface fit model files written by the experimental version of ANAROD
         * can contain additional parameters defined by O. Robach.
	 *
	 * The program determines automatically from the number of items read
	 * whether the file contains Robach's additional parameters. If so, some
	 * of the "standard" parameters are in a different column of the input.
	 *
	 * It then always fills the standard parameters with their corresponding
	 * values. If the file contains the additional parameters and if
	 * Robach's extension is used, then Robach's additional variables are
	 * also filled, otherwise these parameters are simply ignored.
	 *
	 * This way the program can correctly read files containing the
	 * additional parameters, even if Robach's extension is not used.
	 */
#if (defined(__unix) || defined(__APPLE__))
	nread = sscanf(INLINE,"%2s%f%f%d%f%d%f%f%d%f%d%f%f%f%f%f%f%f%f%f",
	  elsurf[NSURF],
	  &XS[NSURF],&XCONST[NSURF],&NXDIS[NSURF],
	  &X2CONST[NSURF],&NX2DIS[NSURF],
	  &YS[NSURF],&YCONST[NSURF],&NYDIS[NSURF],
	  &Y2CONST[NSURF],&NY2DIS[NSURF],
	  &ZS[NSURF],&fdummy[0],&fdummy[1],&fdummy[2],&fdummy[3],&fdummy[4],
	  &fdummy[5],&fdummy[6],&fdummy[7]);
#else
	nread-- = sscanf(INLINE,"%[ ]%2s%f%f%d%f%d%f%f%d%f%d%f%f%d%f%d%d%d%d%d",
	  dummy,elsurf[NSURF],
	  &XS[NSURF],&XCONST[NSURF],&NXDIS[NSURF],
	  &X2CONST[NSURF],&NX2DIS[NSURF],
	  &YS[NSURF],&YCONST[NSURF],&NYDIS[NSURF],
	  &Y2CONST[NSURF],&NY2DIS[NSURF],
	  &ZS[NSURF],&fdummy[0],&fdummy[1],&fdummy[2],&fdummy[3],&fdummy[4],
	  &fdummy[5],&fdummy[6],&fdummy[7]);
#endif /* (defined(__unix) || defined(__APPLE__)) */

	if(nread > 16)
	{
	  NZDIS[NSURF] = (int)fdummy[1];
	  NDWS[NSURF] = (int)fdummy[4];
	  NDWS2[NSURF] = (int)fdummy[5];
          NOCCUP[NSURF] = (int)fdummy[6];
#ifdef EXTENSIONS /* Robach */
	  if (robach_flag)
	  {
	    ZCONST[NSURF] = fdummy[0];
            Z2CONST[NSURF] = fdummy[2];
	    NZ2DIS[NSURF] = (int)fdummy[3];
	    if(nread == 20)
	      NOCCUP2[NSURF] = (int)fdummy[7];
	  }
#endif /* EXTENSIONS Robach */
	}
	else
	{
	  NZDIS[NSURF] = (int)fdummy[0];
	  NDWS[NSURF] = (int)fdummy[1];
	  NDWS2[NSURF] = (int)fdummy[2];
          NOCCUP[NSURF] = (int)fdummy[3];
	}

#ifdef EXTENSIONS /* Mcsl */
      }
#endif /* EXTENSIONS Mcsl */

      NSURF++;
      if (NSURF == MAXATOMS)
      {
	errtype("ERROR, maximum number of model atoms exceeded");
	break;
      }
#ifdef EXTENSIONS /* Svensson */
    }
#endif /* EXTENSIONS Svensson */
  }
  fclose(infile);
  NSURFTOT = NSURF;

  /* Find the different element types and store them in ELEMENT */

  if (NTYPES == 0)
  {
    NTYPES = 1;
    sprintf(ELEMENT[0],"%s",elsurf[0]);
  }
  for (i = 0; i < NSURF; i++)
  {
    newtype = TRUE;
    for (j = 0; j < NTYPES; j++)
    {
      if (el_equal(elsurf[i],ELEMENT[j]))
	newtype = FALSE;
    }
    if (newtype)
    {
      if (NTYPES == MAXTYPES)
      {
	errtype("ERROR, too many atom types in unit cell");
	return;
      }
      NTYPES++;
      sprintf(ELEMENT[NTYPES-1],"%s",elsurf[i]);
    }
  }

#ifdef EXTENSIONS /* Svensson */
  for (i = 0; i < NGROUPSS; i++)
  {
    for (k = 0; k < NATOMSINGROUPS[i]; k++)
    {
      newtype = TRUE;
      for (j = 0; j < NTYPES; j++)
      {
	if (el_equal(ELSURFGROUPS[i][k],ELEMENT[j]))
	  newtype = FALSE;
      }
      if (newtype)
      {
	if (NTYPES == MAXTYPES)
	{
	  errtype("ERROR, too many atom types in unit cell");
	  return;
	}
	NTYPES++;
	sprintf(ELEMENT[NTYPES-1],"%s",ELSURFGROUPS[i][k]);
      }
    }
  }
#endif /* EXTENSIONS Svensson */

  /* Assign a type number to all atoms in the model */

  for (i = 0; i < NSURF; i++)
  {
    for (j = 0; j < NTYPES; j++)
    {
      if (el_equal(elsurf[i],ELEMENT[j]))
	TS[i] = j;
    }
  }

#ifdef EXTENSIONS /* Svensson */
  for (i = 0; i < NGROUPSS; i++)
  {
    for (k = 0; k < NATOMSINGROUPS[i]; k++)
    {
      for (j = 0; j < NTYPES; j++)
      {
	if (el_equal(ELSURFGROUPS[i][k],ELEMENT[j]))
	  TATOMINGROUPS[i][k] = j;
      }
    }
  }
#endif /* EXTENSIONS Svensson */

  /* Store the coefficients for the computation of the atomic scattering
  factors for all the different elements in F_COEFF */

#ifdef EXTENSIONS /* Svensson */
  if (NSURF+NGROUPSS)
#endif /* EXTENSIONS Svensson */
    for (i = 0; i < NTYPES; i++)
    {
      get_coeff(ELEMENT[i],F_COEFF[i]);
    }

  /* Assign the values of the surface positions to the refined positions */

  for (i = 0; i < NSURF; i++)
  {
    XSFIT[i] = XS[i];
    YSFIT[i] = YS[i];
    ZSFIT[i] = ZS[i];
  }

#ifdef EXTENSIONS /* Svensson */
  for (i = 0; i < NGROUPSS; i++)
  {
    XGROUPSFIT[i]   = XGROUPS[i];
    YGROUPSFIT[i]   = YGROUPS[i];
    ZGROUPSFIT[i]   = ZGROUPS[i];
    PHIGROUPSFIT[i] = PHIGROUPS[i];
    CHIGROUPSFIT[i] = CHIGROUPS[i];
    THGROUPSFIT[i]  = THGROUPS[i];

    /* Start addition by Michael Dore */
    for (j=0 ; j<NATOMSINGROUPS[i] ; j++)
    {
      XATOMINGROUPSFIT[i][j]=XATOMINGROUPS[i][j];
      YATOMINGROUPSFIT[i][j]=YATOMINGROUPS[i][j];
      ZATOMINGROUPSFIT[i][j]=ZATOMINGROUPS[i][j];
    }
    ORIGINOFATOMSINGROUPSFIT[i][0]=ORIGINOFATOMSINGROUPS[i][0];
    ORIGINOFATOMSINGROUPSFIT[i][1]=ORIGINOFATOMSINGROUPS[i][1];
    ORIGINOFATOMSINGROUPSFIT[i][2]=ORIGINOFATOMSINGROUPS[i][2];
    /* End addition by Michael Dore */
	
  }

  for (i = 0; i < NCYLINDERS; i++)
  {
    XCYLINDERSFIT[i] = XCYLINDERS[i];
    YCYLINDERSFIT[i] = YCYLINDERS[i];
    ZCYLINDERSFIT[i] = ZCYLINDERS[i];
    TCYLINDERSFIT[i] = TCYLINDERS[i];
    PSICYLINDERSFIT[i] = PSICYLINDERS[i];
    LENGTHOFCYLINDERSFIT[i] = LENGTHOFCYLINDERS[i];
    RADIUSOFCYLINDERSFIT[i] = RADIUSOFCYLINDERS[i];
    SOFTNESSOFCYLINDERSFIT[i] = SOFTNESSOFCYLINDERS[i];
    OCCUPOFCYLINDERSFIT[i] = OCCUPOFCYLINDERS[i];
  }

  for (i = 0; i < NELLIPSOIDS; i++)
  {
    XELLIPSOIDSFIT[i] = XELLIPSOIDS[i];
    YELLIPSOIDSFIT[i] = YELLIPSOIDS[i];
    ZELLIPSOIDSFIT[i] = ZELLIPSOIDS[i];
    PHIELLIPSOIDSFIT[i] = PHIELLIPSOIDS[i];
    CHIELLIPSOIDSFIT[i] = CHIELLIPSOIDS[i];
    THELLIPSOIDSFIT[i] = THELLIPSOIDS[i];
    ALENGTHOFELLIPSOIDSFIT[i] = ALENGTHOFELLIPSOIDS[i];
    BLENGTHOFELLIPSOIDSFIT[i] = BLENGTHOFELLIPSOIDS[i];
    CLENGTHOFELLIPSOIDSFIT[i] = CLENGTHOFELLIPSOIDS[i];
    SOFTNESSOFELLIPSOIDSFIT[i] = SOFTNESSOFELLIPSOIDS[i];
    OCCUPOFELLIPSOIDSFIT[i] = OCCUPOFELLIPSOIDS[i];
  }
#endif /* EXTENSIONS Svensson */

#ifdef EXTENSIONS /* Mcsl */
  /* Find the maximum of the number of plane */
  if (mcsl_flag)
  {
    NPLAN=0;
    for (i = 0; i < NSURF; i++)
      if (NZSPLAN[i]> NPLAN)
	NPLAN = NZSPLAN[i];

    sprintf (STRING,"Nplan = %d \n", NPLAN);
    type_line(STRING);
  }
#endif /* EXTENSIONS Mcsl */

  /* Find highest serial number of the displacement, Debye-Waller and
  occupancy parameters */

  for (i = 0; i < NSURF; i++)
  {
    if (NXDIS[i] > NDISTOT)
      NDISTOT = NXDIS[i];
    if (NX2DIS[i] > NDISTOT)
      NDISTOT = NX2DIS[i];
    if (NYDIS[i] > NDISTOT)
      NDISTOT = NYDIS[i];
    if (NY2DIS[i] > NDISTOT)
      NDISTOT = NY2DIS[i];
    if (NZDIS[i] > NDISTOT)
      NDISTOT = NZDIS[i];

#ifdef EXTENSIONS /* Robach + Mcsl */
    if ((robach_flag)||(mcsl_flag))
      if (NZ2DIS[i] > NDISTOT)
	NDISTOT = NZ2DIS[i];
#endif /* EXTENSIONS Robach + Mcsl */
#ifdef EXTENSIONS /* Mcsl */
    if (mcsl_flag)
      if (NZ3DIS[i] > NDISTOT)
	NDISTOT = NZ3DIS[i];
#endif /* EXTENSIONS Mcsl */

    if (NDWS[i] > NDWTOT)
      NDWTOT = NDWS[i];
    if (NDWS2[i] > NDWTOT2)
      NDWTOT2 = NDWS2[i];
    if (NOCCUP[i] > NOCCTOT)
      NOCCTOT = NOCCUP[i];

#ifdef EXTENSIONS /* Robach + Mcsl */
    if (robach_flag || mcsl_flag)
      if (NOCCUP2[i] > NOCCTOT)
	NOCCTOT = NOCCUP2[i];
#endif /* EXTENSIONS Robach + Mcsl */
  }

#ifdef EXTENSIONS /* Svensson */
  for (i = 0; i < NGROUPSS; i++)
  {
    if (NXDISGROUPS[i] > NDISGROUPSTOT)
      NDISGROUPSTOT = NXDISGROUPS[i];
    if (NYDISGROUPS[i] > NDISGROUPSTOT)
      NDISGROUPSTOT = NYDISGROUPS[i];
    if (NZDISGROUPS[i] > NDISGROUPSTOT)
      NDISGROUPSTOT = NZDISGROUPS[i];
    if (NPHIDISGROUPS[i] > NDISGROUPSTOT)
      NDISGROUPSTOT = NPHIDISGROUPS[i];
    if (NCHIDISGROUPS[i] > NDISGROUPSTOT)
      NDISGROUPSTOT = NCHIDISGROUPS[i];
    if (NTHDISGROUPS[i] > NDISGROUPSTOT)
      NDISGROUPSTOT = NTHDISGROUPS[i];
    for (j = 0; j < NATOMSINGROUPS[i]; j++)
    {
      if (NDWSATOMINGROUPS[i][j] > NDWGROUPSTOT)
	NDWGROUPSTOT = NDWSATOMINGROUPS[i][j];
    }
	/* Start addition by Michael Dore */
    if (SNEXPX[i] > NEXPANGROUPSTOT)
      NEXPANGROUPSTOT = SNEXPX[i];
    if (SNEXPY[i] > NEXPANGROUPSTOT)
      NEXPANGROUPSTOT = SNEXPY[i];
    if (SNEXPZ[i] > NEXPANGROUPSTOT)
      NEXPANGROUPSTOT = SNEXPZ[i];
    if (SNSHEARXY[i] > NSHEARGROUPSTOT)
      NSHEARGROUPSTOT = SNSHEARXY[i];
    if (SNSHEARZX[i] > NSHEARGROUPSTOT)
      NSHEARGROUPSTOT = SNSHEARZX[i];
    if (SNSHEARYZ[i] > NSHEARGROUPSTOT)
      NSHEARGROUPSTOT = SNSHEARYZ[i];
	/* End addition by Michael Dore */
  }

  for (i = 0; i < NCYLINDERS; i++)
  {
    if (NXDISCYLINDERS[i] > NDISCYLINDERSTOT)
      NDISCYLINDERSTOT = NXDISCYLINDERS[i];
    if (NYDISCYLINDERS[i] > NDISCYLINDERSTOT)
      NDISCYLINDERSTOT = NYDISCYLINDERS[i];
    if (NZDISCYLINDERS[i] > NDISCYLINDERSTOT)
      NDISCYLINDERSTOT = NZDISCYLINDERS[i];
    if (NTDISCYLINDERS[i] > NDISCYLINDERSTOT)
      NDISCYLINDERSTOT = NTDISCYLINDERS[i];
    if (NPSIDISCYLINDERS[i] > NDISCYLINDERSTOT)
      NDISCYLINDERSTOT = NPSIDISCYLINDERS[i];
    if (NLENGTHOFCYLINDERS[i] > NCHARGEDENSITYCYLINDERSTOT)
      NCHARGEDENSITYCYLINDERSTOT = NLENGTHOFCYLINDERS[i];
    if (NRADIUSOFCYLINDERS[i] > NCHARGEDENSITYCYLINDERSTOT)
      NCHARGEDENSITYCYLINDERSTOT = NRADIUSOFCYLINDERS[i];
    if (NSOFTNESSOFCYLINDERS[i] > NCHARGEDENSITYCYLINDERSTOT)
      NCHARGEDENSITYCYLINDERSTOT = NSOFTNESSOFCYLINDERS[i];
    if (NOCCUPOFCYLINDERS[i] > NCHARGEDENSITYCYLINDERSTOT)
      NCHARGEDENSITYCYLINDERSTOT = NOCCUPOFCYLINDERS[i];
  }
  for (i = 0; i < NELLIPSOIDS; i++)
  {
    if (NXDISELLIPSOIDS[i] > NDISELLIPSOIDSTOT)
      NDISELLIPSOIDSTOT = NXDISELLIPSOIDS[i];
    if (NYDISELLIPSOIDS[i] > NDISELLIPSOIDSTOT)
      NDISELLIPSOIDSTOT = NYDISELLIPSOIDS[i];
    if (NZDISELLIPSOIDS[i] > NDISELLIPSOIDSTOT)
      NDISELLIPSOIDSTOT = NZDISELLIPSOIDS[i];
    if (NPHIDISELLIPSOIDS[i] > NDISELLIPSOIDSTOT)
      NDISELLIPSOIDSTOT = NPHIDISELLIPSOIDS[i];
    if (NCHIDISELLIPSOIDS[i] > NDISELLIPSOIDSTOT)
      NDISELLIPSOIDSTOT = NCHIDISELLIPSOIDS[i];
    if (NTHDISELLIPSOIDS[i] > NDISELLIPSOIDSTOT)
      NDISELLIPSOIDSTOT = NTHDISELLIPSOIDS[i];
    if (NALENGTHOFELLIPSOIDS[i] > NCHARGEDENSITYELLIPSOIDSTOT)
      NCHARGEDENSITYELLIPSOIDSTOT = NALENGTHOFELLIPSOIDS[i];
    if (NBLENGTHOFELLIPSOIDS[i] > NCHARGEDENSITYELLIPSOIDSTOT)
      NCHARGEDENSITYELLIPSOIDSTOT = NBLENGTHOFELLIPSOIDS[i];
    if (NCLENGTHOFELLIPSOIDS[i] > NCHARGEDENSITYELLIPSOIDSTOT)
      NCHARGEDENSITYELLIPSOIDSTOT = NCLENGTHOFELLIPSOIDS[i];
    if (NSOFTNESSOFELLIPSOIDS[i] > NCHARGEDENSITYELLIPSOIDSTOT)
      NCHARGEDENSITYELLIPSOIDSTOT = NSOFTNESSOFELLIPSOIDS[i];
    if (NOCCUPOFELLIPSOIDS[i] > NCHARGEDENSITYELLIPSOIDSTOT)
      NCHARGEDENSITYELLIPSOIDSTOT = NOCCUPOFELLIPSOIDS[i];
  }
#endif /* EXTENSIONS Svensson */
  if (NDISTOT > MAXPAR)
  {
    NDISTOT = MAXPAR;
    sprintf(STRING,
      "Error, total number of displacement parameters should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
  if (NDWTOT > MAXPAR)
  {
    NDWTOT = MAXPAR;
    sprintf(STRING,
      "Error, total no. of paral. Debye-Waller param. should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
  if (NDWTOT2 > MAXPAR)
  {
    NDWTOT2 = MAXPAR;
    sprintf(STRING,
      "Error, total no. of perp. Debye-Waller param. should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
  if (NOCCTOT > MAXPAR)
  {
    NOCCTOT = MAXPAR;
    sprintf(STRING,
      "Error, total number of occupancy parameters should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
#ifdef EXTENSIONS /* Svensson */
  if (NDISGROUPSTOT > MAXPAR)
  {
    NDISGROUPSTOT = MAXPAR;
    sprintf(STRING,
      "Error, total number of group displacement parameters should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
  if (NDWGROUPSTOT > MAXPAR)
  {
    NDWGROUPSTOT = MAXPAR;
    sprintf(STRING,
      "Error, total number of group Debye-Waller parameters should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
  if (NDISCYLINDERSTOT > MAXPAR)
  {
    NDISCYLINDERSTOT = MAXPAR;
    sprintf(STRING,
      "Error, total number of cylinder displacement parameters should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
  if (NCHARGEDENSITYCYLINDERSTOT > MAXPAR)
  {
    NCHARGEDENSITYCYLINDERSTOT = MAXPAR;
    sprintf(STRING,
	"Error, total number of cylinder charge density parameters should be < %1d",
	MAXPAR+1);
    errtype(STRING);
  }
  if (NDISELLIPSOIDSTOT > MAXPAR)
  {
    NDISELLIPSOIDSTOT = MAXPAR;
    sprintf(STRING,
      "Error, total number of ellipsoid displacement parameters should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
  if (NCHARGEDENSITYELLIPSOIDSTOT > MAXPAR)
  {
    NCHARGEDENSITYELLIPSOIDSTOT = MAXPAR;
    sprintf(STRING,
      "Error, total number of ellipsoid charge density parameters should be < %1d",
      MAXPAR+1);
    errtype(STRING);
  }
#endif /* EXTENSIONS Svensson */

#ifdef EXTENSIONS /* Mcsl */
  if (mcsl_flag)
  {
    if (NSURF2 > NSURFTOT)
    {
      errtype(" error NSURF2 can't be larger than total number of atom");
      NSURF2 = 0;
    }
    NSURF=NSURFTOT-NSURF2;
  }
#endif /* EXTENSIONS Mcsl */

  update_model();

}

/***************************************************************************/
void read_par(void)
/***************************************************************************/

    /*
    Read macro file with values of fitting parameters.
    */

{

  char filename[FILENAME_MAX];
  FILE *infile;

  /* Get filename and open file, return if not successful */

  get_token(filename,"Filename with surface atoms (.par): ");
  add_extension(filename,"par");
/*
  if ((infile = fopen(filename,"r")) == NULL)
  {
    sprintf(STRING,"Failed to open file '%s'",filename);
    errtype(STRING);
    clear_command();
    return;
  }
*/

  /* Run macro file */

  put_command(filename);
  run_macro();

}

/***************************************************************************/
void read_surf(void)
/***************************************************************************/

    /*
    Read file with coordinates of surface atoms.
    First line: comments
    Second:     direct lattice parameters
    Data lines: El X Y Z (NDW NDW2), where
			El      element symbol
			X       x-position in reduced coordinates
			Y       y-position in reduced coordinates
			Z       z-position in reduced coordinates
			NDW     serial number of Debye-Waller parameter
				along in-plane direciton (optional)
			NDW2    serial number of Debye-Waller parameter
				along perpendicular direction (optional)
    */

{

  char filename[FILENAME_MAX];
  FILE *infile;
  float dlats[6];
  int end_of_file,dw_column,dw2_column;
  char elsurf[MAXATOMS][3];
  char dummy[20];
  float dwdummy;
  int i,j;
  int newtype;
#ifdef EXTENSIONS /* Svensson */
  char keyword[20], elsurf_tmp[3];
  char elsurf_group[MAXGROUPS][MAXATOMS][3];
  char name[TITLE_LENGTH];
  int k, group_flag, group_index, atom_index;
  int first_non_group_atom;
  float xs_tmp, ys_tmp, zs_tmp;
  int ndws_tmp, ndws2_tmp;
  float xorg, yorg, zorg, occupancy;
  int end_of_file_reached;

  /* Initialize surface reading */

  NSYMMCARDS = 0;
  NGROUPSS = 0;
  NCYLINDERS = 0;
  NELLIPSOIDS = 0;
  group_flag = FALSE;
  first_non_group_atom = TRUE;
  end_of_file_reached = FALSE;
#endif /* EXTENSIONS Svensson */

  /* Get filename and open file, return if not successful */

  get_token(filename,"Filename with surface atoms (.sur): ");
  add_extension(filename,"sur");
  if ((infile = fopen(filename,"r")) == NULL)
  {
    sprintf(STRING,"Failed to open file '%s'",filename);
    errtype(STRING);
    clear_command();
    return;
  }

  /* Read comments and lattice parameters */

  fgets(INLINE,INLLEN,infile);
  sprintf(STRING,"Title: %s",INLINE);
  type_line(STRING);
  fgets(INLINE,INLLEN,infile);
  sscanf(INLINE,"%f%f%f%f%f%f",&dlats[0],&dlats[1],&dlats[2],&dlats[3],
    &dlats[4],&dlats[5]);
  for (i = 3; i < 6; i++)
    dlats[i] /= RAD;

  /* If a bulk file has already been read in, check the lattice
  parameters for consistency */

  if (NBULK == 0)
  {
    for (i = 0; i < 6; i++) DLAT[i] = dlats[i];
    calc_rlat(DLAT,RLAT);
  }
  else
  {
    for (i = 0; i < 6; i++)
    {
      if (fabs(DLAT[i]-dlats[i]) > 1e-4)
      {
	errtype("ERROR, lattice parameters different from bulk");
	return;
      }
    }
  }

#ifdef EXTENSIONS /* Svensson */
  if (!svensson_flag)
  {
#endif /* EXTENSIONS Svensson */
    fgets(INLINE,INLLEN,infile);
#if (defined(__unix) || defined(__APPLE__))
    sscanf(INLINE,"%2s%f%f%f%d%d",elsurf[0],
      &XS[0],&YS[0],&ZS[0],&NDWS[0],&NDWS2[0]);
#else
    sscanf(INLINE,"%[ ]%2s%f%f%f%d%d",dummy,elsurf[0],
      &XS[0],&YS[0],&ZS[0],&NDWS[0],&NDWS2[0]);
#endif /* (defined(__unix) || defined(__APPLE__)) */

    /*
     * Find out from first data line, whether a serial number for Debye-Waller
     * parameters is given.
     */
    if (NDWS2[0] != 0)
    {
      dw_column = dw2_column = TRUE;
    }
    else if (NDWS[0] != 0)
    {
      dw2_column = FALSE;
      dw_column = TRUE;
    }
    else
    {
      dw2_column = dw_column = FALSE;
    }

  /* Read the rest of the atoms */

    NSURF = 1;
#ifdef EXTENSIONS /* Svensson */
  }
#endif /* EXTENSIONS Svensson */

  while ((fgets(INLINE,INLLEN,infile) != NULL) /*&& (NSURF < MAXATOMS)*/)
  {

#ifdef EXTENSIONS /* Svensson */
    if (svensson_flag)
    {
      /* Erase keyword */
      strcpy(keyword, "                    ");
      /* Check if a symmetry card keyword is given */
      sscanf(INLINE,"%s", keyword);
      if (!strncmp(keyword, "SYMM", 4))
      {
	if (parse_symmetry_card())
	{
	  sprintf(STRING, "Error when parsing symmetry card: %s\n", INLINE);
	  errtype(STRING);
	  return;
	}
      } else {
	if (!strcmp(keyword, "cylinder"))
	{
	  if (group_flag)
	  {
	    /* Already defining group! */
	    errtype("ERROR, cylinder encountered when group was being defined");
	    return;
	  }
	  read_cylinder_surf(infile, name);
	  if(fgets(INLINE,INLLEN,infile) == NULL)
	    end_of_file_reached = TRUE;
	} else {
	  if (!strcmp(keyword, "ellipsoid"))
	  {
	    if (group_flag)
	    {
	      /* Already defining group! */
	      errtype(
		"ERROR, ellipsoid encountered when group was being defined");
	      return;
	    }
	    read_ellipsoid_surf(infile, name);
	    if( fgets(INLINE,INLLEN,infile) == NULL)
	      end_of_file_reached = TRUE;
	  } else {
	    if (!strncmp(keyword,"endgr",5))
	    {
	      /* End of definition of group */
	      if (!group_flag)
	      {
		/* Not defining any group! */
		errtype(
		  "ERROR, endgroup encountered when no group was being defined");
		return;
	      }
	      group_flag = FALSE;
	    } else {

	      /* Check if a group keyword is given */

	      xorg = 0.0;
	      yorg = 0.0;
	      zorg = 0.0;
	      occupancy = 1.0;

	      sscanf(INLINE,"%s %s %f%f%f%f",keyword,name,&xorg,&yorg,&zorg,
		&occupancy);

	      if (!strncmp(keyword, "group", 5))
	      {
		/* Yes, we have found a group! */
		if (group_flag)
		{
		  /* Already definig a group! */
		  errtype("ERROR, trying to define a group whithin a group");
		  return;
		}
		group_flag = TRUE;
		read_group_surf(infile,name,xorg,yorg,zorg,occupancy);
	      } else {

		/* Check that we're dealing with a valid atom line */

		if ((strlen(INLINE) > 3) && (strncmp(INLINE,"  ", 2)))
		{

		  /* Now read the atom */

		  if (group_flag)
		  {

		    /* This atom goes into a group */

#if (defined(__unix) || defined(__APPLE__))
		    sscanf(INLINE,"%2s%f%f%f%d",
		      &elsurf_tmp,&xs_tmp,&ys_tmp,&zs_tmp,&ndws_tmp);
#else
		    sscanf(INLINE,"%[ ]%2s%f%f%f%d",
		      dummy,&elsurf_tmp,&xs_tmp,&ys_tmp,&zs_tmp,&ndws_tmp);
#endif /* (defined(__unix) || defined(__APPLE__)) */
		    group_index = NGROUPSS-1;
		    NATOMSINGROUPS[group_index]++;
		    atom_index = NATOMSINGROUPS[group_index]-1;
		    strcpy(elsurf_group[group_index][atom_index],elsurf_tmp);
		    XATOMINGROUPS[group_index][atom_index] = xs_tmp;
		    YATOMINGROUPS[group_index][atom_index] = ys_tmp;
		    ZATOMINGROUPS[group_index][atom_index] = zs_tmp;
		    NDWSATOMINGROUPS[group_index][atom_index] = ndws_tmp;

		  } else {

		    /* This atom is not part of a group */

		    if (first_non_group_atom)
		    {

		      /* We are reading the first atom. */

		      first_non_group_atom = FALSE;

		      /* Find out from first data line, whether a serial number
		         for Debye-Waller parameters is given */

#if (defined(__unix) || defined(__APPLE__))
		      sscanf(INLINE,"%2s%f%f%f%d%d",elsurf[0],
			&XS[0],&YS[0],&ZS[0],&NDWS[0],&NDWS2[0]);
#else
		      sscanf(INLINE,"%[ ]%2s%f%f%f%d%d",dummy,elsurf[0],
			&XS[0],&YS[0],&ZS[0],&NDWS[0],&NDWS2[0]);
#endif /* (defined(__unix) || defined(__APPLE__)) */
		      if (NDWS2[0] != 0)
		      {
			dw_column = dw2_column = TRUE;
		      } else if (NDWS[0] != 0)
		      {
			dw2_column = FALSE;
			dw_column = TRUE;
		      } else {
			dw2_column = dw_column = FALSE;
		      }
		      NSURF = 1;

		    } else {

		      /* We are now not reading the first atom */

		      if (dw2_column)
		      {
#if (defined(__unix) || defined(__APPLE__))
			sscanf(INLINE,"%2s%f%f%f%d%d",elsurf[NSURF],
			  &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF],
			  &NDWS2[NSURF]);
#else
			sscanf(INLINE,"%[ ]%2s%f%f%f%d%d",dummy,elsurf[NSURF],
			  &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF],
			  &NDWS2[NSURF]);
#endif /* (defined(__unix) || defined(__APPLE__)) */
		      } else if (dw_column)
		      {
#if (defined(__unix) || defined(__APPLE__))
		        sscanf(INLINE,"%2s%f%f%f%d",elsurf[NSURF],
		          &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF]);
#else
		        sscanf(INLINE,"%[ ]%2s%f%f%f%d",dummy,elsurf[NSURF],
		          &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF]);
#endif /* (defined(__unix) || defined(__APPLE__)) */
		      } else {
#if (defined(__unix) || defined(__APPLE__))
		        sscanf(INLINE,"%2s%f%f%f",elsurf[NSURF],
		          &XS[NSURF],&YS[NSURF],&ZS[NSURF]);
#else
		        sscanf(INLINE,"%[ ]%2s%f%f%f",dummy,elsurf[NSURF],
		          &XS[NSURF],&YS[NSURF],&ZS[NSURF]);
#endif /* (defined(__unix) || defined(__APPLE__)) */
		      }
		      NSURF++;
		      if (NSURF == MAXATOMS)
		      {
			errtype(
			  "ERROR, maximum number of model atoms exceeded");
			break;
		      }
		    } /* if (first_non_group_atom) */
		  } /* if (group_flag) */
		} /* if ((strlen(INLINE) > 3) && (strncmp(INLINE,"  ", 2))) */
	      } /* if (!strncmp(keyword, "group", 5)) */
	    } /* if (!strncmp(keyword,"endgr",5)) */
	  } /* if (!strcmp(keyword, "ellipsoid")) */
	} /* if (!strcmp(keyword, "cylinder")) */
      } /* if (!strncmp(keyword, "SYMM", 4)) */
    } else {
#endif /* EXTENSION Svensson */

      if (dw2_column)
      {
#if (defined(__unix) || defined(__APPLE__))
	sscanf(INLINE,"%2s%f%f%f%d%d",elsurf[NSURF],
	  &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF],&NDWS2[NSURF]);
#else
	sscanf(INLINE,"%[ ]%2s%f%f%f%d%d",dummy,elsurf[NSURF],
	  &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF],&NDWS2[NSURF]);
#endif /* (defined(__unix) || defined(__APPLE__)) */
      }
      if (dw_column)
      {
#if (defined(__unix) || defined(__APPLE__))
	sscanf(INLINE,"%2s%f%f%f%d",elsurf[NSURF],
	  &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF]);
#else
	sscanf(INLINE,"%[ ]%2s%f%f%f%d",dummy,elsurf[NSURF],
	  &XS[NSURF],&YS[NSURF],&ZS[NSURF],&NDWS[NSURF]);
#endif /* (defined(__unix) || defined(__APPLE__)) */
      }
      else
      {
#if (defined(__unix) || defined(__APPLE__))
	sscanf(INLINE,"%2s%f%f%f",elsurf[NSURF],
	  &XS[NSURF],&YS[NSURF],&ZS[NSURF]);
#else
	sscanf(INLINE,"%[ ]%2s%f%f%f",dummy,elsurf[NSURF],
	  &XS[NSURF],&YS[NSURF],&ZS[NSURF]);
#endif /* (defined(__unix) || defined(__APPLE__)) */
      }
      NSURF++;
      if (NSURF == MAXATOMS)
      {
	errtype("ERROR, maximum number of model atoms exceeded");
	break;
      }
#ifdef EXTENSIONS /* Svensson */
    } /* if (svensson_flag) */
#endif /* EXTENSIONS Svensson */
  }

  fclose(infile);
  NSURFTOT = NSURF;

  /* Find the different element types and store them in ELEMENT */

  if (NTYPES == 0)
  {
	NTYPES = 1;
	sprintf(ELEMENT[0],"%s",elsurf[0]);
  }
  for (i = 0; i < NSURF; i++)
  {
    newtype = TRUE;
    for (j = 0; j < NTYPES; j++)
    {
      if (el_equal(elsurf[i],ELEMENT[j]))
	newtype = FALSE;
    }
    if (newtype)
    {
      if (NTYPES == MAXTYPES)
      {
	errtype("ERROR, too many atom types in unit cell");
	return;
      }
      NTYPES++;
      sprintf(ELEMENT[NTYPES-1],"%s",elsurf[i]);
    }
  }
#ifdef EXTENSIONS /* Svensson */
  if (svensson_flag)
  {
    if (NGROUPSS)
    {
      for (i = 0; i < NGROUPSS; i++)
      {
	for (k = 0; k < NATOMSINGROUPS[i]; k++)
	{
	  newtype = TRUE;
	  for (j = 0; j < NTYPES; j++)
	  {
	    if (el_equal(elsurf_group[i][k],ELEMENT[j]))
	      newtype = FALSE;
	  }
	  if (newtype)
	  {
	    if (NTYPES == MAXTYPES)
	    {
	      errtype("ERROR, too many atom types in unit cell");
	      return;
	    }
	    NTYPES++;
	    sprintf(ELEMENT[NTYPES-1],"%s",elsurf_group[i][k]);
	  }
	}
      }
    }
  }
#endif /* EXTENSIONS Svensson */

  /* Assign a type number to all atoms in the model */

  for (i = 0; i < NSURF; i++)
  {
    for (j = 0; j < NTYPES; j++)
    {
      if (el_equal(elsurf[i],ELEMENT[j]))
	TS[i] = j;
    }
  }

#ifdef EXTENSIONS /* Svensson */
  if (svensson_flag)
  {
    for (i = 0; i < NGROUPSS; i++)
    {
      for (k = 0; k < NATOMSINGROUPS[i]; k++)
      {
	for (j = 0; j < NTYPES; j++)
	{
	  if (el_equal(elsurf_group[i][k],ELEMENT[j]))
	    TATOMINGROUPS[i][k] = j;
	}
      }
    }
  }
#endif /* EXTENSIONS Svensson */

  /* Store the coefficients for the computation of the atomic scattering
  factors for all the different elements in F_COEFF */

#ifdef EXTENSIONS /* Svensson */
  if (svensson_flag)
  {
    if (NSURF+NGROUPSS)
    {
      for (i = 0; i < NTYPES; i++)
      {
	get_coeff(ELEMENT[i],F_COEFF[i]);
      }
    }
  } else
#endif /* EXTENSIONS Svensson */
    for (i = 0; i < NTYPES; i++)
    {
      get_coeff(ELEMENT[i],F_COEFF[i]);
    }

  /* Copy atom positions to fitted positions */

  for (i = 0; i < NSURF; i++)
  {
    XSFIT[i] = XS[i];
    YSFIT[i] = YS[i];
    ZSFIT[i] = ZS[i];
  }

#ifdef EXTENSIONS /* Svensson */
  if (svensson_flag)
  {
    for (i = 0; i < NGROUPSS; i++)
    {
      XGROUPSFIT[i] = XGROUPS[i];
      YGROUPSFIT[i] = YGROUPS[i];
      ZGROUPSFIT[i] = ZGROUPS[i];
      PHIGROUPSFIT[i] = PHIGROUPS[i];
      CHIGROUPSFIT[i] = CHIGROUPS[i];
      THGROUPSFIT[i] =  THGROUPS[i];
    }

    for (i = 0; i < NCYLINDERS; i++)
    {
      XCYLINDERSFIT[i] = XCYLINDERS[i];
      YCYLINDERSFIT[i] = YCYLINDERS[i];
      ZCYLINDERSFIT[i] = ZCYLINDERS[i];
      TCYLINDERSFIT[i] = TCYLINDERS[i];
      PSICYLINDERSFIT[i] = PSICYLINDERS[i];
      LENGTHOFCYLINDERSFIT[i] = LENGTHOFCYLINDERS[i];
      RADIUSOFCYLINDERSFIT[i] = RADIUSOFCYLINDERS[i];
      SOFTNESSOFCYLINDERSFIT[i] = SOFTNESSOFCYLINDERS[i];
      OCCUPOFCYLINDERSFIT[i] = OCCUPOFCYLINDERS[i];
    }

    for (i = 0; i < NELLIPSOIDS; i++)
    {
      XELLIPSOIDSFIT[i] = XELLIPSOIDS[i];
      YELLIPSOIDSFIT[i] = YELLIPSOIDS[i];
      ZELLIPSOIDSFIT[i] = ZELLIPSOIDS[i];
      PHIELLIPSOIDSFIT[i] = PHIELLIPSOIDS[i];
      CHIELLIPSOIDSFIT[i] = CHIELLIPSOIDS[i];
      THELLIPSOIDSFIT[i] = THELLIPSOIDS[i];
      ALENGTHOFELLIPSOIDSFIT[i] = ALENGTHOFELLIPSOIDS[i];
      BLENGTHOFELLIPSOIDSFIT[i] = BLENGTHOFELLIPSOIDS[i];
      CLENGTHOFELLIPSOIDSFIT[i] = CLENGTHOFELLIPSOIDS[i];
      SOFTNESSOFELLIPSOIDSFIT[i] = SOFTNESSOFELLIPSOIDS[i];
      OCCUPOFELLIPSOIDSFIT[i] = OCCUPOFELLIPSOIDS[i];
    }
  }
#endif /* EXTENSIONS Svensson */

  /* Find highest serial number of Debye-Waller, if read in */

  if (dw_column)
  {
    for (i = 0; i < NSURF; i++)
    {
      if (NDWS[i] > NDWTOT)
      {
	if (NDWS[i] > MAXPAR)
	{
	  sprintf(STRING,
	    "ERROR, total # Debye-Wallers should be < %1d",MAXPAR);
	  errtype(STRING);
	}
	else
	{
	  NDWTOT = NDWS[i];
	}
      }
    }
#ifdef EXTENSIONS /* Svensson */
    if (svensson_flag)
    {
      for (i = 0; i < NGROUPSS; i++)
      {
	for (k = 0; k < NATOMSINGROUPS[i]; k++)
	{
	  if (NDWSATOMINGROUPS[i][k] > NDWGROUPSTOT)
	  {
	    if (NDWSATOMINGROUPS[i][k] > MAXPAR)
	    {
	      sprintf(STRING,
		"ERROR, total # Debye-Wallers should be < %1d",MAXPAR);
	      errtype(STRING);
	    }
	    else
	    {
	      NDWGROUPSTOT = NDWSATOMINGROUPS[i][k];
	    }
	  }
	}
      }
    }
#endif /* EXTENSIONS Svensson */
  }
  if (dw2_column)
  {
    for (i = 0; i < NSURF; i++)
    {
      if (NDWS2[i] > NDWTOT2)
      {
	if (NDWS2[i] > MAXPAR)
	{
	  sprintf(STRING,
	    "ERROR, total # Debye-Wallers should be < %1d",MAXPAR);
	  errtype(STRING);
	}
	else
	{
	  NDWTOT2 = NDWS2[i];
	}
      }
    }
  }

}

/***************************************************************************/
void read_type(void)
/***************************************************************************/

    /*
    Get type of file to be read in.
    */

{

    /* define read_type_menu */

#define read_type_length 7       /* number of commands in read_type menu */

  static struct MENU read_type_menu[read_type_length] =
  {
    "data",      1, 1,  "Structure factor data",
    "surface",   3, 2,  "Coordinates of surface atoms",
    "bulk",      1, 3,  "Coordinates of bulk atoms",
    "fit",       1, 4,  "File with fit model of surface atoms",
    "parameters",2, 5,  "File with fit parameters (macro)",
    "help",      1, 20, "Display menu",
    "return",    1, 21, "Return to main menu"
  };

  int stop = FALSE;
  char token[100];

  while (!stop)
  {
    if (!get_token(token,"File type: "))
      break;
    switch (cmnd_match(token,read_type_menu,read_type_length))
    {
      case -1:
	break;
      case 0:
	break;
      case 1:
	read_data();
	return;
      case 2:
	read_surf();
	return;
      case 3:
	read_bulk();
	return;
      case 4:
	read_fit();
	return;
      case 5:
	read_par();
	return;
      case 20:
	list_menu("INPUT FILE TYPES",read_type_menu,read_type_length);
	break;
      case 21:
	stop = TRUE;
    }
  }
}
