/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$$$$$$                  correct.c                          $$$$$$$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/*
Update 07/02/2001 R. Wilcke (wilcke@esrf.fr)
                  comp_cor_sixc(), comp_cor_w21v() and comp_cor_s2d2(): added
                  a new argument "transmission" to the functions and set the
                  value to 1 in the code.
Update 06/10/2000 E. Vlieg (vlieg@sci.kun.nl)
                  Add 2+3 mode.
Update 06/10/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Include correction for linear gamma table: LINEARGAMMA.
		  Include transmission correction: TRANSIN, TRANSOUT.
Update 12/05/2000 R. Wilcke (wilcke@esrf.fr)
                  moved declaration of variables "alp", "bet" and "psi" to file
                  "ana.h".
Update 26/01/2000 R. Wilcke (wilcke@esrf.fr)
                  replaced "RSCAN" by new variable "SCANTYPE" (three scan types
                  instead of two);
                  comp_cor(): converted the "if" statements for CORRECTMODE in
                  one "switch" statement and reordered the sequence;
                  comp_cor(): added HSIXC, VSIXC, W21V and S2D2 to the
                  CORRECTMODE types.
*/


/***************************************************************************/
/*      Declare include files                                              */
/***************************************************************************/

#define EXT extern
#include "ana.h"
#undef EXT

#ifdef EXTENSIONS /* Wilcke */
double f_cutrod(void);
double f_area(void);
double f_polar(void);
double f_lorent(void);
#endif /* EXTENSIONS Wilcke */

/***************************************************************************/
void	comp_cor(float *lorentz, float *polarization, float *intercept,
	float *area, float *transmission)
/***************************************************************************/

    /* Compute correction factors by calling the subroutine corresponding
    to the diffractometer geometry used.
    See Elias Vlieg, J. Appl. Cryst. 30 (1997) 532, and
    J. Appl. Cryst. 31 (1998) 198. */

    {

    if (CORRECTMODE == ZAXIS)
	comp_cor_zaxis(lorentz,polarization,intercept,area,transmission);
    if ((CORRECTMODE == TWOPTWO) || (CORRECTMODE == TWOPTHREE))
	comp_cor_2p2or3(lorentz,polarization,intercept,area,transmission);
    if (CORRECTMODE == REFROCK)
	comp_cor_refl(lorentz,polarization,intercept,area,transmission);
    if (CORRECTMODE == NOCORRECT)
	{
	*lorentz = *polarization = *intercept = *area = 1.;
	}
#ifdef EXTENSIONS /* Wilcke */
    switch (CORRECTMODE) 
	{
	case HSIXC:
	case VSIXC:
	    comp_cor_sixc(lorentz,polarization,intercept,area,transmission);
	    break;
	case W21V:
	    comp_cor_w21v(lorentz,polarization,intercept,area,transmission);
	    break;
	case S2D2:
	    comp_cor_s2d2(lorentz,polarization,intercept,area,transmission);
	    break;
	}
#endif /* EXTENSIONS Wilcke */

    }

/***************************************************************************/
void	comp_cor_zaxis(float *lorentz, float *polarization,
	float *intercept, float *area, float *transmission)
/***************************************************************************/

    /* Compute the correction factors */

    {

    float x,y,c1,c2,c3,phor,pver,area_sum,xlab,zlab,xdet,bs_eff;
    float xlimit,xstep,ylimit,ystep;

    /* Calculate Lorentz factor */

#ifdef EXTENSIONS /* Wilcke */
    if(SCANTYPE == rock)
#else
    if(RSCAN)
#endif /* EXTENSIONS Wilcke */
	*lorentz = cos(BETAIN/RAD)*cos(GAMMA/RAD)*sin(DELTA/RAD);
    else
	*lorentz = 1;

    /* Calculate polarization correction */

    phor = (1. - sqr(sin(ALPHA/RAD) * cos(DELTA/RAD) * cos(GAMMA/RAD)
	+ cos(ALPHA/RAD) * sin(GAMMA/RAD)));
    pver = 1. - sqr(sin(DELTA/RAD)*cos(GAMMA/RAD));
    *polarization =  1. / (HPOLAR*phor+(1-HPOLAR)*pver);

    /* Calculate the rod interception correction */

    *intercept = (cos(ALPHA/RAD) + cos((ALPHA-2*BETAIN)/RAD) +
	sin(2*(ALPHA-BETAIN)/RAD)*sin(GAMMA/RAD) +
	2*sqr(sin((ALPHA-BETAIN)/RAD))*cos(DELTA/RAD)*cos(GAMMA/RAD)) /
	(2*cos(BETAIN/RAD)*cos(GAMMA/RAD));
    if (LINEARGAMMA) *intercept = (*intercept)/cos(GAMMA/RAD);

    /* Calculate area correction for used diffractometer setting */

    if(BEAMCOR) // take into account non-uniform intensity distribution
	{
	c1 = sin(BETAIN/RAD);
	c2 = cos(DELTA/RAD);
	c3 = sin(DELTA/RAD)*cos(ALPHA/RAD-BETAIN/RAD);
	area_sum = 0;

	/* determine integration limit along x (vertical) direction */

	if(VERSLIT > 0.01)
	    xlimit = VERSLIT/2+0.01;
	else
	    xlimit = 1.;
	xstep = xlimit/50.;

	/* determine integration limit along y (horizontal) direction */

	if(RADIUS > 0.01)
	    ylimit = RADIUS*1.1;
	else
	    ylimit = 10;
	if(fabs(2*HORBEAM/sin(BETAIN/RAD+0.001)) < ylimit)
	    ylimit = fabs(2*HORBEAM/sin(BETAIN/RAD+0.001));
	if(1.1*HORSLIT/(2*sin(BETAIN/RAD+0.001)) < ylimit)
	    ylimit = 1.1*HORSLIT/(2*sin(BETAIN/RAD+0.001));
	ystep = ylimit/50.;

	for(x = -xlimit; x < xlimit+0.01; x+= xstep)
	    {
	    for(y = -ylimit; y < ylimit+0.01; y+= ystep)
		{
		area_sum += f_beam(x,y*c1)*f_detector(x*c2-y*c3)*f_onsample(x,y);
		}
	    }

	/* Calculate effective vertical beam slit size */

	bs_eff = 0.;
	for(x = -xlimit; x < xlimit; x += xstep/10.) bs_eff += f_beam(x,0);
	bs_eff *= xstep/10.;

	*area = (DSLIT*bs_eff)/(area_sum*xstep*ystep);

	}
    else         // no intensity distribution correction
	{
	*area = sin(DELTA/RAD)*cos(ALPHA/RAD-BETAIN/RAD);
	}

    /* Beam transmission correction */

    *transmission = pow(TRANSIN,1/cos(ALPHA/RAD)-1)*
	pow(TRANSOUT,1/cos(GAMMA/RAD)-1);

    }

/***************************************************************************/
void	comp_cor_2p2or3(float *lorentz, float *polarization, 
			     float *intercept, float *area, float *transmission)
/***************************************************************************/

    /* Compute the correction factors for 2+2 geometry */

    {

    float	x,y,c1,c2,c3,phor,pver,area_sum,xlab,zlab,xdet,bs_eff;
    float	cos_beta_out,xlimit,xstep,ylimit,ystep;

    /* Calculate Lorentz factor */

#ifdef EXTENSIONS /* Wilcke */
    if(SCANTYPE == rock)
#else
    if(RSCAN)
#endif /* EXTENSIONS Wilcke */
	*lorentz = cos(ALPHA/RAD)*sin(DELTA/RAD);
    else
	*lorentz = 1;

    /* Calculate polarization correction */

    phor = 1. - sqr(sin(GAMMA/RAD) * cos(DELTA/RAD));
    pver = 1. - sqr(sin(DELTA/RAD));
    *polarization =  1. / (HPOLAR*phor+(1-HPOLAR)*pver);

    /* Calculate the rod interception correction */

    if (CORRECTMODE == TWOPTWO)
	*intercept = (sqr(sin((ALPHA-GAMMA)/RAD))*cos(DELTA/RAD)+
		      cos(ALPHA/RAD)*cos((ALPHA-GAMMA)/RAD))/cos(ALPHA/RAD);
    if (CORRECTMODE == TWOPTHREE)
	*intercept = 1/sqrt(1-sqr(cos(DELTA/RAD)*sin((GAMMA-ALPHA)/RAD)));

    /* Calculate area correction for used diffractometer setting */

    if(BEAMCOR) // take into account non-uniform intensity distribution
	{
	c1 = sin(ALPHA/RAD);
	c2 = cos(DELTA/RAD);
	c3 = sin(DELTA/RAD)*cos(GAMMA/RAD-ALPHA/RAD);
	area_sum = 0;

	/* determine integration limit along x (vertical) direction */

	if(VERSLIT > 0.01)
	    xlimit = VERSLIT/2+0.01;
	else
	    xlimit = 1.;
	xstep = xlimit/50.;

	/* determine integration limit along y (horizontal) direction */

	if(RADIUS > 0.01)
	    ylimit = RADIUS*1.1;
	else
	    ylimit = 10;
	if(fabs(2*HORBEAM/sin(ALPHA/RAD+0.001)) < ylimit)
	    ylimit = fabs(2*HORBEAM/sin(ALPHA/RAD+0.001));
	if(1.1*HORSLIT/(2*sin(ALPHA/RAD+0.001)) < ylimit)
	    ylimit = 1.1*HORSLIT/(2*sin(ALPHA/RAD+0.001));
	ystep = ylimit/50.;

	for(x = -xlimit; x < xlimit+0.01; x+= xstep)
	    {
	    for( y = -ylimit; y < ylimit+0.01; y+= ystep)
		{
		area_sum += f_beam(x,y*c1)*f_detector(x*c2-y*c3)*f_onsample(x,y);
		}
	    }

	/* Calculate effective vertical beam slit size */

	bs_eff = 0.;
	for(x = -xlimit; x < xlimit; x += xstep/10.) bs_eff += f_beam(x,0);
	bs_eff *= xstep/10.;

	*area = (DSLIT*bs_eff)/(area_sum*xstep*ystep);

	}
    else         // no intensity distribution correction
	{
	cos_beta_out = sqrt(1-sqr(cos(DELTA/RAD)*sin((GAMMA-ALPHA)/RAD)));
	*area = sin(DELTA/RAD)/cos_beta_out;
	}

    /* Beam transmission correction */

    *transmission = pow(TRANSIN,1/cos(ALPHA/RAD)-1)*
	pow(TRANSOUT,1/cos_beta_out-1);

    }


/***************************************************************************/
void	comp_cor_refl(float *lorentz, float *polarization, float *intercept,
		      float *area, float *transmission)
/***************************************************************************/

    /* Compute the correction factors for reflectivity rocking scans. */

    {

    float	y,c,beam_sum,sample_sum;

    /* Calculate Lorentz factor */

    *lorentz = sin(2*ALPHA/RAD);

    /* Calculate polarization correction */

    *polarization =  1. / (1-HPOLAR*sqr(sin(2*ALPHA/RAD)));

    /* Calculate rod interception correction */

    *intercept = 1/cos(ALPHA/RAD);
    if (LINEARGAMMA) *intercept = (*intercept)/cos(GAMMA/RAD);

    /* Calculate area/beam profile correction.
    For a reflectivity scan, only the y direction is of interest */

    if (BEAMCOR) // take into account non-uniform intensity distribution
	{
	beam_sum = 0;
	sample_sum = 0;
	for (y = -LIMIT; y < LIMIT+0.01; y += STEP)
	    {
	    c = f_onsample(0,y);
	    beam_sum += f_beam(0,y*sin(ALPHA/RAD))*c;
	    sample_sum += c;
	    }
	*area = sample_sum/beam_sum;
	}
    else
	{
	*area = sin(ALPHA/RAD);
	}

    /* Beam transmission correction */

    *transmission = pow(TRANSIN,1/cos(ALPHA/RAD)-1)*
	pow(TRANSOUT,1/cos(GAMMA/RAD)-1);

    }

/***************************************************************************/
void	comp_cor_ridge(float beta_in, float *lorentz, float *polarization,
	float *area, float *transmission)
/***************************************************************************/

    /* Compute the correction factors for reflectivity ridge scans. Beta_in
    is the angle of incidence */

    {

    float y,c,beam_sum,sample_sum;

    /* Calculate Lorentz factor */

    *lorentz = sin(beta_in);

    /* Calculate polarization correction */

    *polarization =  1. / (HPOLAR*sqr(cos(2.*beta_in))+1.-HPOLAR);

    /* Calculate area/beam profile correction for used diffractometer setting.
    For a reflectivity scan, only the y direction is of interest */

    if (BEAMCOR) // take into account non-uniform intensity distribution
	{
	beam_sum = 0;
	sample_sum = 0;
	for (y = -LIMIT; y < LIMIT+0.01; y += STEP)
	    {
	    c = f_onsample(0,y);
	    beam_sum += f_beam(0,y*sin(beta_in))*c;
	    sample_sum += c;
	    }
	*area = sample_sum/beam_sum;
	}
    else    /* assume non-flooding case */
	{
	*area = sin(beta_in);
	}

    /* Beam transmission correction; beta_in = beta_out */

    *transmission = pow(TRANSIN,1/cos(beta_in)-1)*
	pow(TRANSOUT,1/cos(beta_in)-1);

    }


/***************************************************************************/
float	f_beam(float x, float z)
/***************************************************************************/

    /*
    Compute normalized incoming beam intensity.
    */

    {

    if ((fabs(x) > VERSLIT/2.) || (fabs(z) > HORSLIT/2.))
	{
	return(0.);
	}
   else
	{
	return(exp(-2.77*x*x/(VERBEAM*VERBEAM))*
	    exp(-2.77*z*z/(HORBEAM*HORBEAM)));
	}
    }


/***************************************************************************/
float	f_onsample (float x, float y)
/***************************************************************************/

    /*
    Compute whether point is on sample.
    */

    {

    if (((x*x)+(y*y)) > (RADIUS*RADIUS))
	{
	return(0.);
	}
    else
	{
	return(1.);
	}
    }

/***************************************************************************/
float	f_detector(float x)
/***************************************************************************/

    /*
    Compute whether point is observed by detector.
    */

    {

    if (fabs(x) > DSLIT/2.)
	{
	return(0.);
	}
    else
	{
	return(1.);
	}
    }
#ifdef EXTENSIONS /* Wilcke */

/***************************************************************************/
void	comp_cor_sixc(float *lorentz, float *polarization, float *intercept,
	float *area, float *transmission)
/***************************************************************************/

/*
 * Compute the correction factors for vertical and horizontal six circle
 * geometry.
 */

    {
    alp = ANGLE[4]/RAD;
    bet = ANGLE[5]/RAD;
    psi = ANGLE[0]/RAD;

    /* Calculate Lorentz factor */
    *lorentz = f_lorent();

    /* Calculate polarization correction */
    *polarization = f_polar();

    /* Calculate the rod interception correction */
    *intercept = f_cutrod();

    /* Calculate area correction for used diffractometer setting */
    *area = f_area();

    /* Calculate the beam transmission correction */
    /*
     * For the time being, this value is set to 1. The correct formula is still
     * being worked out.
     */

    *transmission = 1.;

    }

/***************************************************************************/
void	comp_cor_w21v(float *lorentz, float *polarization, float *intercept,
	float *area, float *transmission)
/***************************************************************************/

/*
 * Compute the correction factors for the 2+2 geometry (type w21v).
 */

    {
    alp = ANGLE[3]/RAD;
    bet = asin(cos(ANGLE[1]/RAD) * sin((ANGLE[0] - ANGLE[3])/RAD));
    psi = atan2(tan(ANGLE[1]/RAD),cos((ANGLE[0] - ANGLE[3])/RAD));

    /* Calculate Lorentz factor */
    *lorentz = f_lorent();

    /* Calculate polarization correction */
    *polarization = f_polar();

    /* Calculate the rod interception correction */
    *intercept = f_cutrod();

    /* Calculate area correction for used diffractometer setting */
    *area = f_area();

    /* Calculate the beam transmission correction */
    /*
     * For the time being, this value is set to 1. The correct formula is still
     * being worked out.
     */

    *transmission = 1.;

    }

/***************************************************************************/
void	comp_cor_s2d2(float *lorentz, float *polarization, float *intercept,
	float *area, float *transmission)
/***************************************************************************/

/*
 * Compute the correction factors for the 2+2 geometry (type s2d2).
 */

    {
    alp = ANGLE[2]/RAD;
    bet = asin(cos(ANGLE[1]/RAD) * sin((ANGLE[0] - ANGLE[2])/RAD));
    psi = atan2(tan(ANGLE[1]/RAD),cos((ANGLE[0] - ANGLE[2])/RAD));

    /* Calculate Lorentz factor */
    *lorentz = f_lorent();

    /* Calculate polarization correction */
    *polarization = f_polar();

    /* Calculate the rod interception correction */
    *intercept = f_cutrod();

    /* Calculate area correction for used diffractometer setting */
    *area = f_area();

    /* Calculate the beam transmission correction */
    /*
     * For the time being, this value is set to 1. The correct formula is still
     * being worked out.
     */

    *transmission = 1.;

    }

/***************************************************************************/
double f_lorent(void)
/***************************************************************************/
/*
 * Compute Lorentz factor for the various scan types.
 */
{
    double lorent;

    switch(SCANTYPE)
    {
    case rock:
	lorent = cos(alp) * sin(psi) * cos(bet);
	break;
    case sprd:
	lorent = sin(2. * alp);
	break;
    case hkl:
	lorent = 1.;
    }
    return(lorent);
}

/***************************************************************************/
double f_polar(void)
/***************************************************************************/
/*
 * Compute polarisation factor for the various detector geometries.
 */
{
    double polar;

    if(CORRECTMODE == HSIXC)
	polar = 1. / (1. - sqr(sin(psi) * cos(bet)));
    else
    {
	polar = 1. / (1. - sqr(sin(alp)*cos(psi)*cos(bet) + cos(alp)*sin(bet)));
    }
    return(polar);
}

/***************************************************************************/
double f_area(void)
/***************************************************************************/
/*
 * Compute area factor.
 * 
 * For details, see the internal note "Correction factors for integrated
 * intensities" by D. Smilgies. This note can be obtained from R. Wilcke
 * (e-mail: wilcke@esrf.fr,
 * normal mail: ESRF, BP 220, F-38043 Grenoble Cedex, FRANCE).
 */
{
    double area,a2i,b2i,dilli,sbwo,sinbet,sinpsi,cospsi,dcv;

    sinbet = sin(bet);
    sinpsi = sin(psi);
    cospsi = cos(psi);
    
    dcv = DSLIT * cospsi - VERSLIT;
    if(dcv < DBL_MIN)
	dcv = DBL_MIN;
    a2i = sinpsi / dcv;
    b2i = sinpsi / (DSLIT * cospsi + VERSLIT);
    /*
     * 1. / sbwo is the projection of the out-of-plane detector slit on the
     * surface of the sample. This is the limit in the out-of-plane dimension
     * of what the detector will see.
     */
    sbwo = sinbet / OSLIT;
    /*
     * In the scattering plane, the limit of what the detector will see is given
     * by the intersection of the incoming beam (as defined by the corresponding
     * beam slit) and the outgoing beam (as defined by the corresponding
     * detector slit.
     *
     * The total area that the detector will see is the intersection of the
     * limits given by the out-of-plane detector slit and the limits in the
     * scattering plane. For details of the cases to be distinguished, see the
     * note quoted above.
     */
    if(sbwo > a2i)
	area = sinbet * cospsi / VERSLIT / OSLIT;
    else if(sbwo < b2i)
	area = sinpsi / VERSLIT / DSLIT;
    else
	area = sinpsi / (VERSLIT * DSLIT - sqr(DSLIT * cospsi + VERSLIT -
	   sinpsi / sbwo) / 4. / cospsi);
    /*
     * The total area of the sample seen by the detector can, however, never be
     * larger than the area illuminated by the incoming beam as defined by its
     * out-of-plane slits.
     *
     * Moreover, it can also not be larger than the diameter of the sample.
     */
    dilli = sin(alp) / HORSLIT;
    if(dilli < 0.5 / RADIUS)
	dilli = 0.5 / RADIUS;
    if(area < dilli / VERSLIT)
	area = dilli / VERSLIT;

    return(area);
}

/***************************************************************************/
double f_cutrod(void)
/***************************************************************************/
/*
 * Compute "cut of the rod" factor.
 */
{
    double cutrod,oscb;
    oscb = OSLIT * cos(bet);
    if(oscb < DBL_MIN)
	oscb = DBL_MIN;
    cutrod = 1. / oscb;
    return(cutrod);
}
#endif /* EXTENSIONS Wilcke */
