/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                         ana.c                                 $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/*
Update 22/03/2006 R. Wilcke (wilcke@esrf.fr)
                  modified code for MacIntosh MACOSX using predefined macro
                  __APPLE__ for this architecture.
Update 19/01/2004 R. Wilcke (wilcke@esrf.fr) 
                  removed declarations of fopen() (is declared in "stdio.h").
Update 22/10/2003 R. Wilcke (wilcke@esrf.fr)
                  write_initmac(): print out new spec type ID32_k
Update 27/02/2003 R. Wilcke (wilcke@esrf.fr)
                  removed all special code for the standing wave version.
Update 19/02/2003 R. Wilcke (wilcke@esrf.fr)
                  provide default values for spectrum numbers in all routines.
Update 05/11/2002 R. Wilcke (wilcke@esrf.fr)
                  main(): removed variables "string" and "nspectrum" (not used).
Update 25/01/2001 R. Wilcke (wilcke@esrf.fr)
                  changed "miller" command to "hkl" and modified text for
                  "errors" command to agree with E. Vliegs standard version.
Update 10/01/2001 R. Wilcke (wilcke@esrf.fr)
                  main(): to obtain proper "long jump" behaviour in the SIGINT
                  interrupt handler:
                  - declared type of "mainloop" with the macro "JMPBUF_TYPE"
                    instead of type "jmp_buf" (this macro is defined in menu.h);
                  - replaced for __unix and __CYGWIN__ call to setjmp(mainloop)
                    by call to sigsetjmp(mainloop,1).
Update 21/12/2000 O. Svensson (svensson@esrf.fr)
                  Elias Vlieg has added initialisation of TRANSIN, 
		  TRANSOUT and LINEARGAMMA.
Update 20/12/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Small changes in menu help texts
Update 15/12/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Add Spec file of DUBBLE type.
		  Add (2+3) diffractometer corrections
Update 12/12/2000 R. Wilcke (wilcke@esrf.fr)
                  auto_plot(): changed the way the commands "title" and
                  "scannumber" work (GRAPHIC and PGPLOT case only).
Update 20/11/2000 O.Svensson (svensson@esrf.fr) 
                  Changed from "#ifdef EXTENSIONS" to "#ifdef STWAVE"
                  for the standing wave extensions.
Update 19/10/2000 O.Svensson (svensson@esrf.fr) 
                  Changed ANA menu item "standwave" to "stwave".
Update 03/10/2000 R. Wilcke (wilcke@esrf.fr)
                  auto_plot(): added new menu items "usertitle" and changed
                  the way the plotting of the title is handled (GRAPHIC and
                  PGPLOT case only).
Update 03/10/2000 O.Svensson (svensson@esrf.fr) 
                  Moved initialization of standing wave extension variables
		  to "standing_wave_init()" in "standing_wave.c".
Update 20/09/2000 O.Svensson (svensson@esrf.fr) 
                  Changed initialization of standing wave extension variables.
Update 15/09/2000 R. Wilcke (wilcke@esrf.fr)
                  auto_plot(): for the menu items "data" and "fit", do not
                  reset the plot title to blank (GRAPHIC and PGPLOT case only);
                  initialize(): set transformation matrix = unitary matrix;
                  write_initmac(): corrected wrong "if(SCANTYPE = rock)" to
                  "if(SCANTYPE == rock)"
Update 13/09/2000 R. Wilcke (wilcke@esrf.fr)
                  auto_plot() and plot_3d(): set dimension of the "menu" lists
                  to "[]", change the corresponding "lenght" from a macro to a
                  variable and set its value dynamically;
                  auto_plot(): when setting the standard data title, make sure
                  that the buffer does not overflow and that there is space for
                  SCANNR and DATATITLE.
Update 09/09/2000 O.Svensson (svensson@esrf.fr) 
                  Added "#include" for the "standing_wave.h" extension,
                  added a new sub menu "Standing waves" and initialization
                  of variables used by "standing_wave.c".
Update 19/07/2000 R. Wilcke (wilcke@esrf.fr)
                  define VERSIONSTR using the macro VERSION, if this is defined
                  (e.g. in the Makefile);
                  replaced everywhere the buffer size for filenames by the ISO-C
                  defined macro FILENAME_MAX.
Update 30/06/2000 R. Wilcke (wilcke@esrf.fr)
                  increased precision for output to macro of OSLIT, ..., HPOLAR.
Update 21/06/2000 R. Wilcke (wilcke@esrf.fr)
                  replace UNIX by __unix (is defined by CPP on UNIX systems).
Update 15/06/2000 R. Wilcke (wilcke@esrf.fr)
                  remove the #ifdef SCROLL around the set_scroll() calls, the
                  required action will be taken in set_scroll() itself.
Update 24/05/2000 R. Wilcke (wilcke@esrf.fr)
                  make HMIL, KMIL and LMIL arrays with dimension MAX_SPECTRUM.
Update 05/04/2000 R. Wilcke (wilcke@esrf.fr)
                  main(): remove the system("resize") call (does not work as
                  intended).
Update 31/03/2000 R. Wilcke (wilcke@esrf.fr)
                  add #define VERSIONSTR __DATE__ for "__STDC__", and update
                  the "else" string to "Mar 30 2000";
                  main(): close plot window when exiting program;
                  main(): for UNIX and __CYGWIN__, save stack environment with
                  setjmp() and install the exception handler with
                  install_exc_handler() (was only for MSDOS).
Update 30/03/2000 R. Wilcke (wilcke@esrf.fr)
                  main(): add call to end_menu() when exiting program.
Update 17/03/2000 R. Wilcke (wilcke@esrf.fr)
                  initialize(): add call to init_menu().
Update 26/01/2000 R. Wilcke (wilcke@esrf.fr)
                  replaced "RSCAN" by new variable "SCANTYPE" (three scan types
                  instead of two);
                  write_initmac(): added HSIXC, VSIXC, W21V and S2D2 to the
                  output.
Update 20/01/2000 R. Wilcke (wilcke@esrf.fr)
                  write_initmac(): added OSLIT to the output.
Update 10/12/1999 R. Wilcke (wilcke@esrf.fr)
                  replaced NBACK by NBACKL and NBACKR to allow independent
                  definition of the number of left and right background points
*/

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define	EXT
#include "ana.h"
#include <stdlib.h>
#if defined(MSDOS)
#include <alloc.h>
#include <dir.h>
#elif defined(__CYGWIN__)
#include <malloc.h>
#include <mingw/dir.h>
#endif

#undef EXT

#if defined(__BORLANDC__)
unsigned _stklen = 20000;
#endif /* defined(__BORLANDC__) */

#if defined(VERSION)
char VERSIONSTR[] = VERSION " " __DATE__;
#else
#define VERSIONSTR __DATE__
#endif /* defined(VERSION) */

/***************************************************************************/
int main(void)
/***************************************************************************/

    /*
    General program to manipulate and analyse spectra.

    Written by: Elias Vlieg
		AT&T Bell Laboratories
		510E Brookhaven National Laboratory
		Upton, NY 11973
		U.S.A.

    Address since October 1990:
		FOM Institute
		Kruislaan 407
		1098 SJ Amsterdam
		The Netherlands

    Address since January 1998:
		Solid State Chemistry
		University of Nijmegen
		Toernooiveld 1
		6525 ED Nijmegen
		The Netherlands
		email: vlieg@sci.kun.nl

    The present version works with two diffrent graphical packages:
	GraphiC V5.0 		on a PC running MSDOS / OS2
	IMSL Exponent Graphics	on a VAX running VMS
    A switch in ana.h allows a simple change between the two.
    */

    {

    /* define main_menu */

    enum i_main {
	i_zero,
	i_read,
	i_list,
	i_clear,
	i_set,
	i_operate,
	i_theory,
	i_fit,
	i_plot,
	i_autoplot,
#ifdef DO3D
	i_p3d,
#endif
	i_loop,
	i_macro,
	i_opsys,
	i_help,
	i_quit
	};

    static struct   MENU main_menu[i_quit] =
	{
	"read",     1,i_read,   "Read a spectrum from file",
	"list",     1,i_list,   "List a spectrum to terminal or file",
	"clear",    1,i_clear,  "Clear all spectra",
	"set",      1,i_set,    "Goto set parameter menu",
	"operate",  1,i_operate,"Goto operations menu",
	"theory",   1,i_theory, "Goto theory menu",
	"fit",      1,i_fit,    "Fit a spectrum",
	"plot",     1,i_plot,   "Plot a spectrum",
	"autoplot", 1,i_autoplot,  "Make automated plot",
#ifdef DO3D
	"p3d",      2,i_p3d,    "Plot spectrum as 3D curve",
#endif
	"loop",     2,i_loop,   "Execute loop command",
	"macro",    1,i_macro,  "Run macro file",
	":",        1,i_opsys,  "Execute an operating system command",
	"help",     1,i_help,   "Display menu",
	"quit",     4,i_quit,   "Quit program"
	};

    int stop = FALSE;
    int has_crashed = 0;
    char token[100];
    float *x,*y,*e;

    /*
     * Under POSIX, the setjmp() / longjmp() routines are not guaranteed to
     * restore the signal mask, which means that further occurences of signals
     * may be blocked after the first longjmp() call.
     *
     * To restore the signal mask, the sigsetjmp() / siglongjmp() routines
     * should be used under POSIX.
     *
     * For other systems (e.g. MSDOS), the availability of sigsetjmp() /
     * siglongjmp() is not known to us. Keep the old calls for the time being.
     *
     * The buffer used by sigsetjmp() / siglongjmp() has a different variable
     * type than the one for setjmp() / longjmp(). Use the macro JMPBUF_TYPE
     * (defined in menu.h) to have the correct variable type in functin of the
     * operating system.
     */
#if defined(MSDOS) || defined(__unix) || defined(__APPLE__)
    JMPBUF_TYPE mainloop;
#if defined(__unix) || defined(__APPLE__)
    has_crashed = sigsetjmp(mainloop,1);
#else
    has_crashed = setjmp(mainloop);
#endif /* defined(__unix) || defined(__APPLE__) */
    if(!has_crashed) install_exc_handler(mainloop);
#endif /* defined(MSDOS) || defined(__unix) || defined(__APPLE__) */

    /* Do some initialization */

    if(!has_crashed) initialize();

    while(!stop)
	{
	type_header(0);
	/* Trick to keep loop running after possible clear_command */
	/* Commented out by Olof Svensson in order to allow macro
           files to be called from within a loop
	if(LOOP)
	    {
	    if(nomore_commands()) put_command("loop");
	    }
	*/
	while(!get_token(token,"ANA>")) type_header(0);
	switch(cmnd_match(token,main_menu,i_quit))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case i_read:
		read_type();
		break;
	    case i_list:
		list_spectrum();
		break;
	    case i_clear:
		clear_data();
		break;
	    case i_set:
		set();
		break;
	    case i_operate:
		operate();
		break;
	    case i_theory:
		theory();
		break;
	    case i_fit:
		fit();
		break;
	    case i_plot:
		plot();
		break;
	    case i_autoplot:
		auto_plot();
		break;
#ifdef DO3D
	    case i_p3d:
		plot_3d();
		break;
#endif
	    case i_loop:
		run_loop();
		break;
	    case i_macro:
		run_macro();
		break;
	    case i_opsys:
		opsys();
		break;
	    case i_help:
		list_menu("MAIN MENU",main_menu,i_quit);
		break;
	    case i_quit:
		stop = TRUE;
	    }
	}

    /* Save parameters */

    write_initmac();
    put_command("save");
    plotxye(x,y,e,1);
    put_command("close");
    plotxye(x,y,e,1);
    if(OUTINT) fclose(INTFILE);
    end_menu();

    }

/***************************************************************************/
void    auto_plot(void)
/***************************************************************************/

    /*
    Make automated plot.
    */

    {

    /* define auto_plot_menu */

    static struct   MENU auto_plot_menu[] =
	{
	"data",      1,  1,  "Plot data only",
	"fit",       1,  2,  "Plot data and fit",
	"title",     1,  3,  "Set standard data title",
	"scannumber",1,  4,  "Make scan number the title",
#ifdef EXTENSIONS /* Wilcke */
	"usertitle", 1,  8,  "Enable user-defined title",
#endif /* EXTENSIONS Wilcke */
	"hkl",       2,  5,  "Put Diffraction indices in header",
	"errors",    1,  6,  "Error bar plotting on/off",
#ifdef EXTENSIONS /* Walker */
	"blankpage", 1,  7,  "Blank page",
#endif /* EXTENSIONS Walker */
	"help",      1,  20, "Display menu",
	"return",    1,  21, "Return to main menu"
	};
    /* number of commands in auto_plot menu */
    int auto_plot_length = sizeof(auto_plot_menu) / sizeof(auto_plot_menu[0]);

    float       *x,*y,*e;
    int i,j;
    int     stop = FALSE;
    char    token[100];
    static int errors = FALSE;
#ifdef EXTENSIONS /* Wilcke */
    static int titflg = 0;
#endif /* EXTENSIONS Wilcke */

    while(!stop)
	{
	type_header(0);
	if(!get_token(token,"ANA.AUTOPLOT>")) return;
	switch(cmnd_match(token,auto_plot_menu,auto_plot_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		/* Check whether spectrum contains data */

		if(DATA[READSPEC].NPNT < 1)
		    {
		    sprintf(STRING,"ERROR, no data in spectrum %1d",
			    READSPEC+1);
		    errtype(STRING);
		    return;
		    }
		/* Do the plotting */

#ifdef IMSLEG
		put_command("makeplot no ret");
		plotxye(x,y,e,1);
		sprintf(STRING,"title \"|lf||ht0.6||fn1|%s\" return",
			DATATITLE);
		put_command(STRING);
		plotxye(x,y,e,1);
		if(DATARUN)
		    {
		    sprintf(STRING,"header \"%s\" return",DATAINFO);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    /* Made HMIL, KMIL and LMIL arrays (see ana.h) */
		    sprintf(STRING,"header \"(%4.2f,%4.2f,%4.2f)\" return",
			HMIL[READSPEC],KMIL[READSPEC],LMIL[READSPEC]);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		if(errors)
		    {
		    sprintf(STRING,"%1d curve 2 plot errors ret ",
			    READSPEC+1);
		    }
		else
		    {
		    sprintf(STRING,"%1d curve 2 plot ret",
			    READSPEC+1);
		    }
		put_command(STRING);
		plot();
		put_command("makeplot yes replot ret");
		plotxye(x,y,e,1);
#endif /* IMSLEG */
#if defined(GRAPHIC)||defined(PGPLOT)
		typeoff();
#ifdef EXTENSIONS /* Wilcke */
		/*
		 * Generate title for the plot:
		 * if titflg == 1, then generate "standard data title";
		 * if titflg == 2, then make scannumber the title;
		 * if titflg == 0, do not generate new title, but retain user-
		 *                 defined title.
		 *
		 * Having the title generated here means that a title type can
		 * be set once (e.g. with "autoplot title"), and then a whole
		 * range of spectra can be plotted with consecutive calls to
		 * "autoplot data". The title will reflect the correct file and
		 * scan number without having to call "autoplot title" again.
		 */
		if(titflg == 1)
		    {
		    /*
		     * Make sure that DATAFILE does not overflow the STRING
		     * buffer and that there remains enough space for SCANNR and
		     * DATATITLE.
		     */
		    sprintf(STRING,"title \"%.*s #%1d %s\" break",
			INLLEN-strlen(DATATITLE)-10,DATAFILE,SCANNR,DATATITLE);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		else if(titflg == 2)
		    {
		    sprintf(STRING,"title \"%1d\" break",SCANNR);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		if(errors)
		    {
		    sprintf(STRING,
			    "%1d curve -1 plot err br",
			    READSPEC+1);
		    }
		else
		    {
		    sprintf(STRING,"%1d curve -1 plot br",
			    READSPEC+1);
		    }
#else
		if (errors)
		    {
		    sprintf(STRING,
		    "%1d tit \" \" curve -1 plot err br",READSPEC+1);
		    }
		else
		    {
		    sprintf(STRING,
			    "%1d tit \" \" curve -1 plot br", READSPEC+1);
		    }
#endif /* EXTENSIONS Wilcke */
		put_command(STRING);
		plot();
		sprintf(STRING,"text head \"%s\" ret bre",DATATITLE);
		put_command(STRING);
		plotxye(x,y,e,1);
		if(DATARUN)
		    {
		    sprintf(STRING,"text head \"%s\" ret bre",
			    DATAINFO);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    /* Made HMIL, KMIL and LMIL arrays (see ana.h) */
		    sprintf(STRING,"text head \"(%4.2f,%4.2f,%4.2f)\" ret bre",
			HMIL[READSPEC],KMIL[READSPEC],LMIL[READSPEC]);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		put_command("ret");
		plotxye(x,y,e,1);
		typeon();
#endif /* defined(GRAPHIC)||defined(PGPLOT) */

		return;
	    case 2:

		/* Check whether there is data in spectra used for fitting */

		if((DATA[FITSPEC].NPNT < 1) || (DATA[FITOUTSP].NPNT < 1))
		    {
		    errtype("ERROR, no data in fitting spectra");
		    return;
		    }

#ifdef IMSLEG

		/* Generate plot title */

		sprintf(STRING,"title \"|lf||ht0.6||fn1|%s\" return",
			DATATITLE);
		put_command(STRING);
		plotxye(x,y,e,1);
		if(DATARUN)
		    {
		    sprintf(STRING,"header \"%s\" return",DATAINFO);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    /* Made HMIL, KMIL and LMIL arrays (see ana.h) */
		    sprintf(STRING,"header \"(%4.2f,%4.2f,%4.2f)\" return",
			HMIL[FITSPEC],KMIL[FITSPEC],LMIL[FITSPEC]);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }

		sprintf(STRING,
	    	    "header \"%s  |fn4|h|fn1||se|2|ee| = %7.3f\" return",
		    TH.TITLE,CHISQR);
		put_command(STRING);
		plotxye(x,y,e,1);
		for(i = 0; i < TH.NPAR/2; i++)
		    {
		    sprintf(STRING,
"hea \"%5s = %10.2e |fn13|L|fn1| %9.1e  %5s = %10.2e |fn13|L|fn1| %9.1e\" ret",
			TH.SHORT[2*i],TH.PAR[2*i],TH.ERR[2*i],
			TH.SHORT[2*i+1],TH.PAR[2*i+1],TH.ERR[2*i+1]);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		if(TH.NPAR%2 > 0)
		    {
		    sprintf(STRING,
			"header \"%5s = %10.2e |fn13|L|fn1| %9.1e\" ret",
			TH.SHORT[TH.NPAR-1],TH.PAR[TH.NPAR-1],
			TH.ERR[TH.NPAR-1]);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }

		/* Do the plotting */

		put_command("makeplot no ret");
		plotxye(x,y,e,1);
		if(errors)
		    {
		    sprintf(STRING,"%1d curve 2 plot errors ret ",
			    FITSPEC+1);
		    }
		else
		    {
		    sprintf(STRING,"%1d curve 2 plot ret",
			    FITSPEC+1);
		    }
		put_command(STRING);
		plot();
		sprintf(STRING,"%1d curve 1 over ret",FITOUTSP+1);
		put_command(STRING);
		plot();
		put_command("makeplot yes replot yes");
		plotxye(x,y,e,1);
#endif /* IMSLEG */
#if defined(GRAPHIC)||defined(PGPLOT)
		typeoff();
#ifdef EXTENSIONS /* Wilcke */
		/*
		 * Generate title for the plot:
		 * if titflg == 1, then generate "standard data title";
		 * if titflg == 2, then make scannumber the title;
		 * if titflg == 0, do not generate new title, but retain user-
		 *                 defined title.
		 *
		 * Having the title generated here means that a title type can
		 * be set once (e.g. with "autoplot title"), and then a whole
		 * range of spectra can be plotted with consecutive calls to
		 * "autoplot data". The title will reflect the correct file and
		 * scan number without having to call "autoplot title" again.
		 */
		if(titflg == 1)
		    {
		    /*
		     * Make sure that DATAFILE does not overflow the STRING
		     * buffer and that there remains enough space for SCANNR and
		     * DATATITLE.
		     */
		    sprintf(STRING,"title \"%.*s #%1d %s\" break",
			INLLEN-strlen(DATATITLE)-10,DATAFILE,SCANNR,DATATITLE);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		else if(titflg == 2)
		    {
		    sprintf(STRING,"title \"%1d\" break",SCANNR);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		if(errors)
		    {
		    sprintf(STRING,
			    "%1d curve -1 plot err br",FITSPEC+1);
		    }
		else
		    {
		    sprintf(STRING,
			    "%1d curve -1 plot br",FITSPEC+1);
		    }
#else
		if (errors)
		    {
		    sprintf(STRING,
		    "%1d tit \" \" curve -1 plot err br",FITSPEC+1);
		    }
		else
		    {
		    sprintf(STRING,
		    "%1d tit \" \" curve -1 plot br",FITSPEC+1);
		    }
#endif /* EXTENSIONS Wilcke */
		put_command(STRING);
		plot();
		sprintf(STRING,"%1d curve 0 over br",FITOUTSP+1);
		put_command(STRING);
		plot();
		sprintf(STRING,"text head \"%s\" ret bre",DATATITLE);
		put_command(STRING);
		plotxye(x,y,e,1);
		if(DATARUN)
		    {
		    sprintf(STRING,"text head \"%s\" ret bre",
			    DATAINFO);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    /* Made HMIL, KMIL and LMIL arrays (see ana.h) */
		    sprintf(STRING,"text head \"(%4.2f,%4.2f,%4.2f)\" ret bre",
			HMIL[FITSPEC],KMIL[FITSPEC],LMIL[FITSPEC]);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		sprintf(STRING,"text head \"%s  \2h\1[2] = %7.3f\" ret br",
		    TH.TITLE,CHISQR);
		put_command(STRING);
		plotxye(x,y,e,1);
		for(i = 0; i < TH.NPAR/2; i++)
		    {
		    sprintf(STRING,
	"tex hea \"%5s = %10.2e \2+\1 %9.1e  %5s = %10.2e \2+\1 %9.1e\" ret br",
			TH.SHORT[2*i],TH.PAR[2*i],TH.ERR[2*i],
			TH.SHORT[2*i+1],TH.PAR[2*i+1],TH.ERR[2*i+1]);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		if(TH.NPAR%2 > 0)
		    {
		    sprintf(STRING,
			"tex hea \"%5s = %10.2e \2+\1 %9.1e\" ret br",
			TH.SHORT[TH.NPAR-1],TH.PAR[TH.NPAR-1],
			TH.ERR[TH.NPAR-1]);
		    put_command(STRING);
		    plotxye(x,y,e,1);
		    }
		put_command("ret");
		plotxye(x,y,e,1);
		typeon();
#endif /* defined(GRAPHIC)||defined(PGPLOT) */
		return;
	    case 3:
                /*
                 * Set the title type to 2 (for plotting with commands "data"
                 * and "fit"), but write the title also to the plot routines.
                 *
                 * This way a title can be set with "autoplot title" and the
                 * plotting then done e.g. with "plot" directly.
                 */
#ifdef EXTENSIONS /* Wilcke */
		titflg = 1;
#endif /* EXTENSIONS Wilcke */
		/*
		 * Make sure that DATAFILE does not overflow the STRING buffer
		 * and that there remains enough space for SCANNR and DATATITLE.
		 */
		sprintf(STRING,"title \"%.*s #%1d %s\" break",
		    INLLEN-strlen(DATATITLE)-10,DATAFILE,SCANNR,DATATITLE);
		put_command(STRING);
		plotxye(x,y,e,1);
		return;
	    case 4:
                /*
                 * Set the title type to 2 (for plotting with commands "data"
                 * and "fit"), but write the title also to the plot routines.
                 *
                 * This way a title can be set with "autoplot scan" and the
                 * plotting then done e.g. with "plot" directly.
                 */
#ifdef EXTENSIONS /* Wilcke */
		titflg = 2;
#endif /* EXTENSIONS Wilcke */
		sprintf(STRING,"title \"%1d\" break",SCANNR);
		put_command(STRING);
		plotxye(x,y,e,1);
		return;
	    case 5:
#ifdef IMSLEG
		/* Made HMIL, KMIL and LMIL arrays (see ana.h) */
		sprintf(STRING,"header \"(%4.2f,%4.2f,%4.2f)\" return",
		    HMIL[READSPEC],KMIL[READSPEC],LMIL[READSPEC]);
#endif /* IMSLEG */
#if defined(GRAPHIC)||defined(PGPLOT)
		/* Made HMIL, KMIL and LMIL arrays (see ana.h) */
		sprintf(STRING,"text hea \"(%4.2f,%4.2f,%4.2f)\" ret break",
		    HMIL[READSPEC],KMIL[READSPEC],LMIL[READSPEC]);
#endif /* defined(GRAPHIC)||defined(PGPLOT) */
		put_command(STRING);
		plotxye(x,y,e,1);
		return;
	    case 6:
		sprintf(STRING,"Plot error bars [%s]: ",yesnostr(errors));
		errors = yesno(errors,STRING);
		break;
#ifdef EXTENSIONS /* Walker */
	    case 7:
		sprintf(STRING, "printing a blank page\n");
		type_line(STRING);
		plot_blankpage();
		return;
#endif /* EXTENSIONS Walker */
#ifdef EXTENSIONS /* Wilcke */
	    case 8:
		titflg = 0;
		return;
#endif /* EXTENSIONS Wilcke */
	    case 20:
		list_menu("AUTO PLOT MENU",
			  auto_plot_menu,auto_plot_length);
		break;
	    case 21:
		stop = TRUE;
	    }
	}
    }

/***************************************************************************/
void clear_data(void)
/***************************************************************************/

    /*
    Erase all spectra.
    */

    {

    int i,j;

    for(i = 0; i < MAX_SPECTRUM; i++)
	{
	DATA[i].NPNT = 0.;
	DATA[i].MINY = 0.;
	DATA[i].MAXY = 0.;
	sprintf(DATA[i].TITLE,"No input");
	for(j = 0; j < MAX_DATA; j++)
	    {
	    DATA[i].X[j] = 0.;
	    DATA[i].Y[j] = 0.;
	    DATA[i].E[j] = 0.;
	    }
	}
    type_header(1);
    }

/***************************************************************************/
void    initialize(void)
/***************************************************************************/

    /*
    Initialize global parameters.
    */

    {

    float *x,*y,*e;

    /* Allocate memory for DATA blocks */

    DATA = (DATA_BLOCK far *)malloc(sizeof(DATA_BLOCK) * MAX_SPECTRUM);
    if(DATA == NULL)
	{
	errtype("FATAL ERROR, cannot allocate data blocks.\n");
	exit(1);
	}

    /* Initialize parameters local to analysis program */

    LOOP = FALSE;
    LOOP_INCREMENT = 1;
    init_menu();
    set_scroll(MAX_SPECTRUM+4,ROWS);
    clear_screen();
    set_cursor(1,ROWS);
    sprintf(STRING, "Data analysis program, version %s \n",VERSIONSTR);
    type_line(STRING);
    clear_data();
    type_header(1);
    DATARUN = TRUE;

    /* Initialize plotting parameters */

    put_command("init");
    plotxye(x,y,e,1);

    /* Give default names for the case there is no init file */

    sprintf(DATAFILE,"file");
    sprintf(YCOLNAME,"detector");
    sprintf(PATHNAME,"path");

    /* Initalize ANA parameters */

    GAMMA_FIX = 0.;
    HPOLAR = 1.;
    TRANSIN = TRANSOUT = 1.;
    LINEARGAMMA = FALSE;
    /*
     * Set transformation matrix = unitary matrix.
     * Note that this will only be effective if there is no "ana" initialization
     * macro file, otherwise the transformation matrix will be read from there.
     */
    T11 = T22 = T33 = 1.;
    /*
     * Read initialization macro file.
     */
    sprintf(STRING,"macro %s",INITFILE);
    put_command(STRING);

    }

/***************************************************************************/
void    list_spectrum(void)
/***************************************************************************/

    /*
    List a spectrum on screen or on a file. If first spectrum number is
    given a negative value, than the program asks for a second spectrum
    number.
    */

    {

    int nspec = 1,nspec2 = 2;
    int i,j,point,remainder,ok;
    char filename[FILENAME_MAX],dummy[10];
    int terminal;
    FILE *listfile;
    int onespec = TRUE;

    /* Get spectrum number, if negative number given, ask later for 2nd one */

    ok = FALSE;
    while(!ok)
	{
	nspec = get_int(1,
	    "Spectrum number (type -#1 #2 for two spectra) [1]: ");
	if((fabs(nspec) > 0) && (fabs(nspec) <= MAX_SPECTRUM))
	    ok = TRUE;
	else
	    clear_command();
	}
    if(nspec < 0)
	{
	nspec = -nspec;
	onespec = FALSE;
	get_inspec(&nspec2);
	}
    nspec -= 1;

    /* Stop if there are no data points in first spectrum or, in case two
    spectra are listed, if the two don't have the same number of points */

    if(DATA[nspec].NPNT == 0)
	{
	sprintf(STRING,"ERROR, no data in spectrum %1d",nspec+1);
	errtype(STRING);
	clear_command();
	return;
	}
    if(!onespec && (DATA[nspec].NPNT != DATA[nspec2].NPNT))
	{
	sprintf(STRING,
	    "ERROR, different number of points in spectra %1d and %1d",
	    nspec+1,nspec2+1);
	errtype(STRING);
	clear_command();
	return;
	}

    if(!get_token(filename,
	"Filename (.dat) (type 't' or <return> for terminal): "))
	{
	terminal = TRUE;
	}
    else if(((filename[0] == 't') || (filename[0] == 'T')) &&
	(filename[1] == '\0'))
	{
	terminal = TRUE;
	}
    else
	{
	terminal = FALSE;
	add_extension(filename,"dat");
	if((listfile = fopen(filename,"w")) == NULL)
	    {
	    errtype("ERROR, failed to open file");
	    clear_command();
	    return;
	    }
	}

    if(!terminal)      /* write spectrum to file */
	{
	get_string(STRING,"Comments: ");
	fprintf(listfile,"%s\n",STRING);
	for(i = 0; i < DATA[nspec].NPNT; i++)
	    {
	    if(onespec)
		{
		fprintf(listfile,"%12.5g %12.5g %12.5g\n",
		    DATA[nspec].X[i],DATA[nspec].Y[i],DATA[nspec].E[i]);
		}
	    else
		{
		fprintf(listfile,"%12.5g %12.5g %12.5g\n",
		    DATA[nspec].X[i],DATA[nspec].Y[i],DATA[nspec2].Y[i]);
		}
	    }
	fprintf(listfile,"End of list\n");
	fclose(listfile);
	}

    else                /* write spectrum to terminal */
	{

	/* Type spectrum one page at a time */
	set_scroll(1,ROWS);
	for(i = 0; i < DATA[nspec].NPNT/(ROWS-1); i++)
	    {
	    clear_screen();
	    set_cursor(1,1);
	    for(j = 0; j < (ROWS-1); j++)
		{
		point = i*(ROWS-1)+j;
		if(onespec)
		    {
		    sprintf(STRING,"%5d %12.5g %12.5g %12.5g\n",point+1,
			DATA[nspec].X[point],DATA[nspec].Y[point],
			DATA[nspec].E[point]);
		    }
		else
		    {
		    sprintf(STRING,"%5d %12.5g %12.5g %12.5g\n",point+1,
			DATA[nspec].X[point],DATA[nspec].Y[point],
			DATA[nspec2].Y[point]);
		    }
		type_line(STRING);
		}
	    sprintf(STRING,
		"Page %1d, More? (press <return> to continue): ",i+1);
	    if(get_token(dummy,STRING))
		{
		clear_screen();
		type_header(1);
		set_scroll(MAX_SPECTRUM+4,ROWS);
		return;
		}
	    }

	/* Type last page of data */

	remainder = DATA[nspec].NPNT-(DATA[nspec].NPNT/(ROWS-1))*(ROWS-1);
	if(remainder > 0)
	    {
	    clear_screen();
	    set_cursor(1,1);
	    }
	for(i = 0; i < remainder; i++)
	    {
	    point = (DATA[nspec].NPNT / (ROWS-1))*(ROWS-1)+i;
	    if(onespec)
		{
		sprintf(STRING,"%5d %12.5g %12.5g %12.5g\n",point+1,
		    DATA[nspec].X[point],DATA[nspec].Y[point],
		    DATA[nspec].E[point]);
		}
	    else
		{
		sprintf(STRING,"%5d %12.5g %12.5g %12.5g\n",point+1,
		    DATA[nspec].X[point],DATA[nspec].Y[point],
		    DATA[nspec2].Y[point]);
		}
	    type_line(STRING);
	    }
	sprintf(STRING,"Last page, press <return> to continue: ");
	if(!get_token(dummy,STRING));
	clear_screen();
	set_scroll(MAX_SPECTRUM+4,ROWS);
	type_header(1);
	return;

	}

    }

/***************************************************************************/
void    plot(void)
/***************************************************************************/

    /*
    Pass a spectrum to the plotting subroutines.
    */

    {

    int nspec = 1;

    /* Get spectrum number */

    get_inspec(&nspec);
    if(DATA[nspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Call plotting program */

    plotxye(DATA[nspec].X,DATA[nspec].Y,DATA[nspec].E,DATA[nspec].NPNT);

    }


#ifdef DO3D
/***************************************************************************/
void    plot_3d(void)
/***************************************************************************/

    /*
    Pass a spectrum to the plotting subroutines for 3D plotting.
    */

    {

    /* define plot options */

    static struct   MENU plot_opt_menu[] =
	{
	"x",	    1,  1,  "Spectrum along x-axis",
	"y",        1,  2,  "Spectrum along y-axis",
	"angle",    1,  3,  "Spectrum makes angle with x-axis",
	"help",     1, 10,  "Display menu",
	"return",   1, 11,  "Return to main menu"
	};
    /* number of commands in plot_opt menu */
    int plot_opt_length = sizeof(plot_opt_menu) / sizeof(plot_opt_menu[0]);


    int nspec = 1,i,stop = FALSE;
    char	token[100];
    float	value;

    /* Get spectrum number */

    get_inspec(&nspec);
    if(DATA[nspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Ask for type of spectrum */

    while(!stop)
	{
	type_header(0);
	while(!get_token(token,"Type of spectrum: ")) return;
	switch(cmnd_match(token,plot_opt_menu,plot_opt_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		value = get_real(0.,"Constant y value: ");
		for(i = 0; i < DATA[nspec].NPNT; i++)
			YD[i] = value;
		plot3dcurve(DATA[nspec].X,YD,DATA[nspec].Y,DATA[nspec].NPNT);
		return;
	    case 2:
		value = get_real(0.,"Constant x value: ");
		for(i = 0; i < DATA[nspec].NPNT; i++)
			XD[i] = value;
		plot3dcurve(XD,DATA[nspec].X,DATA[nspec].Y,DATA[nspec].NPNT);
		return;
	    case 3:
		value = get_real(0.,"Angle with x-axis (degrees) [0]: ")/RAD;
		for(i = 0; i < DATA[nspec].NPNT; i++)
			{
			XD[i] = DATA[nspec].X[i]*cos(value);
			YD[i] = DATA[nspec].X[i]*sin(value);
			}
		plot3dcurve(XD,YD,DATA[nspec].Y,DATA[nspec].NPNT);
		return;
	    case 10:
		list_menu("3D PLOT OPTIONS",plot_opt_menu,plot_opt_length);
		break;
	    case 11:
		stop = TRUE;
	    }
	}

    }
#endif

/***************************************************************************/
void    run_loop(void)
/***************************************************************************/

    /*
    Execute the loop strings in loop for range of input files.
    */

    {

    static int first,last,counter,increment;

    if(!LOOP)
	{
	first = get_int(1,"Give first scan number: ");
	last = get_int(1,"Give last scan number: ");
	if(last - first < 0) increment = -LOOP_INCREMENT;
	else	increment = LOOP_INCREMENT;
	counter = first-increment;
	LOOP = TRUE;
	}

    counter += increment;
    if(counter >= last)
	{
	LOOP = FALSE;
	sprintf(STRING,"%s %d %s",LOOP1,counter,LOOP2);
	}
    else
	{
	sprintf(STRING,"%s %d %s loop",LOOP1,counter,LOOP2);
	}
    put_command(STRING);

    /* The next line is not typed through 'type_line', in order
    to be able to turn the screen typing off while looping 
    (speed gain), but still to see the counter */

    if(SHOWCOUNTER) printf("Processing scan %1d\r",counter);

    }

/***************************************************************************/
void    write_initmac(void)
/***************************************************************************/

    /*
    Write current values of parameters to macro file which can be used
    for initialization.
    */

    {

    FILE    *init_mac;
    char    t;

    /* Open macro file */

    if((init_mac = fopen(INITFILE,"w")) == NULL)
	{
	sprintf(STRING,"Failed to open %s file",INITFILE);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Save read settings */

    fprintf(init_mac,"!Go to set read menu\n");
    fprintf(init_mac,"set read\n");
    fprintf(init_mac,"xcolumn %1d\n",XCOL);
    fprintf(init_mac,"ycolumn %1d\n",YCOL);
    fprintf(init_mac,"mcolumn %1d\n",MCOL);
    fprintf(init_mac,"ecolumn %1d\n",ECOL);
    fprintf(init_mac,"norm %10.4e\n",MONITOR_NORM);
    fprintf(init_mac,"header %1d\n",HEADER_LINES);
    fprintf(init_mac,"filename %s\n",DATAFILE);
    fprintf(init_mac,"solid %s\n",yesnostr(SOLID_CORR));
    fprintf(init_mac,"yname %s\n",YCOLNAME);
    fprintf(init_mac,"pathname %s\n",PATHNAME);
    fprintf(init_mac,"nauto %1d\n",NAUTOSUB);
    if(SPECTYPE == ID03) fprintf(init_mac,"id03\n");
    if(SPECTYPE == ID32) fprintf(init_mac,"id32\n");
    if(SPECTYPE == ID32_k) fprintf(init_mac,"k_id32\n");
    if(SPECTYPE == DUBBLE) fprintf(init_mac,"dubble\n");
    fprintf(init_mac,"return return\n");

    /* Save fit setting */

    fprintf(init_mac,"!Go to set fit menu\n");
    fprintf(init_mac,"set fit\n");
    if(WEIGHTMODE == 0)
	fprintf(init_mac,"standard\n");
    else if(WEIGHTMODE == 1)
	fprintf(init_mac,"unit\n");
    fprintf(init_mac,"return return\n");

    /* Save general settings */

    fprintf(init_mac,"!Go to set general menu\n");
    fprintf(init_mac,"set general\n");
    fprintf(init_mac,"datarun %s\n",yesnostr(DATARUN));
    fprintf(init_mac,"return return\n");

    /* Save loop parameters */

    fprintf(init_mac,"!Go to set loop menu\n");
    fprintf(init_mac,"set loop\n");
    fprintf(init_mac,"bef \"%s\"\n",LOOP1);
    fprintf(init_mac,"aft \"%s\"\n",LOOP2);
    fprintf(init_mac,"show %s\n",yesnostr(SHOWCOUNTER));
    fprintf(init_mac,"increment %1d\n",LOOP_INCREMENT);
    fprintf(init_mac,"return return\n");

    /* Save numerical integration parameters */

    fprintf(init_mac,"!Go to set integration menu\n");
    fprintf(init_mac,"set integr\n");
    fprintf(init_mac,
	"trans %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f\n",
	T11,T12,T13,T21,T22,T23,T31,T32,T33);
#ifdef EXTENSIONS /* Wilcke */
    fprintf(init_mac,"nback %1d %1d\n",NBACKL,NBACKR);
#else
    fprintf(init_mac,"nback %1d\n",NBACK);
#endif /* EXTENSIONS Wilcke */
    fprintf(init_mac,"autoback %s\n",yesnostr(AUTO_BACK));
    fprintf(init_mac,"range %.2f\n",INT_RANGE);
    fprintf(init_mac,"return return\n");

    /* Save correction parameters */

    fprintf(init_mac,"!Go to set correct menu\n");
    fprintf(init_mac,"set corr\n");
    if (CORRECTMODE == ZAXIS)
	fprintf(init_mac,"zaxis\n");
    if (CORRECTMODE == TWOPTWO)
	fprintf(init_mac,"twoptwo\n");
    if (CORRECTMODE == TWOPTHREE)
	fprintf(init_mac,"twopthree\n");
    if (CORRECTMODE == NOCORRECT)
	fprintf(init_mac,"nocorrect\n");
    if (CORRECTMODE == REFROCK)
	fprintf(init_mac,"refrock\n");
#ifdef EXTENSIONS /* Wilcke */
    switch(CORRECTMODE)
    {
    case HSIXC:
	fprintf(init_mac,"hzsixc\n");
	break;
    case VSIXC:
	fprintf(init_mac,"vzsixc\n");
	break;
    case W21V:
	fprintf(init_mac,"vwtpt\n");
	break;
    case S2D2:
	fprintf(init_mac,"vstpt\n");
	break;
    }
#endif /* EXTENSIONS Wilcke */
    fprintf(init_mac,"beam %s\n",yesnostr(BEAMCOR));
    fprintf(init_mac,"gammafix %.3f\n",GAMMA_FIX);
    fprintf(init_mac,"lingamma %s\n",yesnostr(LINEARGAMMA));
#ifdef EXTENSIONS /* Wilcke */
    if(SCANTYPE == rock)
	fprintf(init_mac,"rscan\n");
    else if(SCANTYPE == sprd)
	fprintf(init_mac,"sprdscan\n");
#else
    if(RSCAN)
	fprintf(init_mac,"rscan\n");
#endif /* EXTENSIONS Wilcke */
    else
	fprintf(init_mac,"hklscan\n");
#ifdef EXTENSIONS /* Wilcke */
    fprintf(init_mac,"oslit %4.2f\n",OSLIT);
#endif /* EXTENSIONS Wilcke */
    /* Increased precision for output */
    fprintf(init_mac,"dslit %4.2f\n",DSLIT);
    fprintf(init_mac,"hslit %4.2f\n",HORSLIT);
    fprintf(init_mac,"vslit %4.2f\n",VERSLIT);
    fprintf(init_mac,"hbeam %4.2f\n",HORBEAM);
    fprintf(init_mac,"vbeam %4.2f\n",VERBEAM);
    fprintf(init_mac,"radius %4.2f\n",RADIUS);
    fprintf(init_mac,"hpolar %7.4f\n",HPOLAR);
    fprintf(init_mac,"tin %7.4f\n",TRANSIN);
    fprintf(init_mac,"tout %7.4f\n",TRANSOUT);
    fprintf(init_mac,"return return\n");

    }
