
;+
; NAME:
;	XWINDOW
; PURPOSE:
;	Ceates a graphic window embedded in a widget.
;	useful to make (and print) graphics from widget applications.
; CATEGORY:
;	Widgets.
; CALLING SEQUENCE:
;	Xwindow [, keywords]
; INPUTS:
;	None.
; OPTIONAL INPUT PARAMETERS:
;	None.
; KEYWORD PARAMETERS:
;	GROUP = The widget ID of the widget that calls Xplot.
;	When this ID is specified, a death of the caller results
;	in a death of Xplot.
;	WINDOW_SIZE = a 2-dim vector containing the width and
;	height of the graphical window (in pixels).
;	WTITLE = title for the main window
;	BUFFER = String array with the idl command to be executed.
;	EXITBUFFER = String array with the idl command to be executed 
;	   when the application is closed (intended to clean pointers/objects
;	   if used in BUFFER)
;	DATA = idl variable containing the data needed by BUFFER
;	PARMS = idl structure containing the parameter data needed by BUFFER
;		This structure follows the conventions described in the
;		XScrMenu routine. 
;	EDIT = if set, creates EDIT buttons (Edit Buffer and Edit Parameters)
;	  to allow the user to interactively edit the buffer and parameters.
;	  Allowed values:
;		1 (or any other value different fron 2 or 3) create both 
;		  buttons
;		2= create only "Edit Buffer" button
;		3= create only "Edit Parameters" button
;		
;	EXTRA = extra keywords to be passed to the XScrMenu application
;		for editing the Parameters
;	ZOOMFLAG = when set, allows zooming a part of the graph area with
;		the cursor. In order to use this option it is essential
;		to to define the limits to the plot by the parms keyword, 
;		in the tags: xMin,xMax,yMin,yMax. See Example 7
;	ZBOXCOLOR = when ZOOMFLAG is set, use this keyword to define the
;		color for the zooming box (default = !p.color)
;	PARENT = set this keyword to a named variable to return the widget
;		ID of the xwindow widget_base. Useful to manipulate xwindow
;		from another program, e.g.:
;
;		xwindow,Parent=pa ; creates xwindow
;		xwindow_quit,pa ; destroys xwindow
;		xwindow_refresh,pa ; refreshes xwindow
;
; OUTPUTS:
;	Creates a Window with a graph.
; OPTIONAL OUTPUT PARAMETERS:
;    None.
; COMMON BLOCKS:
;	None.
; SIDE EFFECTS:
;	Starts xmanager, if not yest started.
; RESTRICTIONS:
;	Data to be used in the buffer commands is passes exclusively in
;	string format or by using the DATA keyword.
;	It is possible to use variables in the buffer, but this variables 
;	must have a name included in the list: i,j,a0,...a10,b0,...b10,
;	c0,...,c10 (this is due to the fact that EXECUTE cannot create
;	variables on the fly. The variable must be pre-initializad.
; PROCEDURE:
;	Uses EXECUTE to execute the commands in BUFFER
;	Uses XScrMenu to edit the parameters and XDisplayfile1 to edit
;	  the buffer.
; EXAMPLES:
;	1) Xwindow   (you will get a sample surface)
;	2) Xwindow,Buffer='tvscl,dist(500)' (you get an image)
;	3) Example of using the DATA keyword:
;	   a = dist(500)
;	   Xwindow,Buffer='tvscl,data',Data=a 
;	4) Example of using a anonymous structure to pass multiple data
;	   z= dist(20)
;	   x = findgen(20)*5
;	   y = findgen(20)*8
;	    xwindow,b='surface,data.z,data.x,data.y',data={z:z,x:x,y:y}
;	5) Example of using multiline buffer:
;	   xwindow,Buffer=['!p.multi=[0,2,2]', $
;		'for i=1,4 do surface,dist(20)','!p.multi=0']
;	6) Example of using "parameters" structure:
;		z= dist(20)
;		x = findgen(20)*5
;		y = findgen(20)*8
;		parms={xtitle:'x axis',ytitle:'y axis',ztitle:'z axis'}
;		xwindow,b='surface,data.z,data.x,data.y,xtitle=parms.xtitle'+$
;		  ',ytitle=parms.ytitle,ztitle=parms.ztitle',$
;		  data={z:z,x:x,y:y},/Edit, Parms=parms
;	7) Example using zoom (must use params keyword):
;		x = findgen(100)
;		y = x*x
;		buffer = $
;		'plot,data.x,data.y,xrange=[parms.xmin,parms.xmax],'+$
;		'yrange=[parms.ymin,parms.ymax]'
;		
;		XWindow,Buffer= buffer, $
;		data={x:x,y:y},/edit, /zoom, zboxcolor=3 , $
;		parms={xmin:10.0,xmax:50.0,ymin:0.0,ymax:0.0} 
;
; MODIFICATION HISTORY:
;	Created NOV-1995 by M. Sanchez del Rio (srio@esrf.fr) (c)ESRF
;	96-01-08 MSR When /edit is selected, and n_elements(buffer) EQ 1
;		presents the buffer in a text widget at the bottom.
;	96-09-24 MSR Uses xdisplayfile1 instead xscrmenu for editing
;		buffer. No "cancel" possibility.
;	96-10-02 MSR allows resizing window.
;	97-04-03 MSR uses /no_block kw in xmanager call for idl version NE 4
;	97-10-16 MSR adds the internal variables to be used within BUFFER.
;	98-12-21 MSR adds the PARMS keyword.
;	00-04-11 MSR adds zoomFlag option (following the ideas in zplot.pro
;		from D. Fanning www.dfanning.com)
;	02-07-24 MSR fixes a bug when editing buffer and printing.
;	02-09-18 MSR added EXITBUFFER keyword to clean pointers and objects
;		when window is closed. 
;	03-02-21 MSR added PARENT keyword. 
;	09-06-10 MSR added xwindow_refresh() function.
;-

;
;==============================================================================
;
PRO XWINDOW_QUIT,wiD

parent=Widget_Info(wId,/Parent)
IF parent NE 0 THEN wId = parent
Widget_control,wId,get_uvalue=info
data = info.data
parms = info.parms

   FOR i=0,N_Elements(info.exitBuffer)-1 DO BEGIN
     IF info.exitBuffer[i] NE '' THEN BEGIN
         tmp = Execute(info.exitbuffer[i])
         IF tmp NE 1 THEN BEGIN
           itmp = Dialog_Message(/Question,['Error executing command: ',$
             info.buffer[i],'Continue?'])
           IF itmp NE 'No' THEN RETURN
         ENDIF
     ENDIF
   ENDFOR
Widget_Control,wId,/DESTROY
END ; XWINDOW_QUIT
;
;==============================================================================
;
PRO XWINDOW_EXECUTEBUFFER,info


Catch,error_status
IF error_status ne 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XWINDOW_EXECUTEBUFFER: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF
;
; If we are zooming, set {xy}style to 1
;
IF info.zoomIn EQ 1 THEN BEGIN
  xstyle_old = !x.style
  ystyle_old = !y.style
  !x.style=1
  !y.style=1
ENDIF
;
; define user variables 
;
i=0 & j=0 & k=0 
a0=0 & a1=0 & a2=0 & a3=0 & a4=0 & a5=0 & a6=0 & a7=0 & a8=0 & a9=0 & a10=0
b0=0 & b1=0 & b2=0 & b3=0 & b4=0 & b5=0 & b6=0 & b7=0 & b8=0 & b9=0 & b10=0
c0=0 & c1=0 & c2=0 & c3=0 & c4=0 & c5=0 & c6=0 & c7=0 & c8=0 & c9=0 & c10=0
;
; define variables
;
data = info.data
buffer = info.buffer
parms = info.parms

Widget_Control,info.wdraw,Get_Value=windownum
windows=[windownum,info.pixIndex]
; 
; Executes active commands
;
Catch,/Cancel
FOR i=0,N_Elements(windows)-1 DO BEGIN
  IF windows[i] NE -1 THEN BEGIN
    WSet,windows[i]
    IF n_elements(info.buffer) EQ 1 THEN BEGIN
      tmp = execute(info.buffer[0]) 
      IF tmp NE 1 THEN BEGIN
        itmp = Dialog_Message(/Question,['Error executing command: ',$
          info.buffer[0],'Continue?'])
        IF itmp NE 'No' THEN RETURN
      ENDIF
    ENDIF ELSE BEGIN
       FOR ih=0L,N_Elements(info.buffer)-1 do BEGIN
         tmp = Execute(info.buffer[ih])
         IF tmp NE 1 THEN BEGIN
           itmp = Dialog_Message(/Question,['Error executing command: ',$
             info.buffer[ih],'Continue?'])
           IF itmp NE 'No' THEN RETURN
         ENDIF
       ENDFOR
    ENDELSE
  ENDIF
ENDFOR
Catch,error_status

;
; back {xy}style to its original value
;
IF info.zoomIn EQ 1 THEN  BEGIN
  !x.style = xstyle_old 
  !y.style = ystyle_old 
ENDIF

END ; XWINDOW_EXECUTEBUFFER
 
;
;==============================================================================
;

PRO xwindow_refresh,wid

Catch,error_status
IF error_status ne 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XWINDOW_REFRESH: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF widget_info(wid,/valid) NE 1 THEN RETURN
Widget_Control,wid,Get_UValue=info
xwindow_executebuffer,info
END ; xwindow_refresh

;
;==============================================================================
;

PRO XWINDOW_ZOOM_START_EVENT, event

; This event handler ONLY responds to button down events from the
; draw widget. If it gets a DOWN event, it does three things: (1) sets
; the static and dynamic corners of the zoom box, (2) changes the
; event handler for the draw widget to ZPLOT_DRAWBOX and turns on MOTION
; events, and (3) update the user's color table vectors.


Catch,error_status
IF error_status ne 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XWINDOW_ZOOM_START: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

possibleEventTypes = [ 'DOWN', 'UP', 'MOTION', 'SCROLL' ]
thisEvent = possibleEventTypes(event.type)
IF thisEvent NE 'DOWN' THEN RETURN
;
; Must be DOWN event to get here, so get info structure.
;
Widget_Control, event.top, Get_UValue=info ; , /No_Copy
;
; Set the static corners of the box to current
; cursor location.
;
info.xs = event.x
info.ys = event.y
;
; Change the event handler for the draw widget and turn MOTION
; events ON.
;
Widget_Control, event.id, Event_Pro='XWINDOW_ZOOM_DRAWBOX', $
   Draw_Motion_Events=1
;
; Put the info structure back into its storage location.
;
Widget_Control, event.top, Set_UValue=info ; , /No_Copy
END ; XWINDOW_ZOOM_START_EVENT 

;
;==============================================================================
;

PRO XWINDOW_ZOOM_DRAWBOX, event

; This event handler continuously draws and erases the zoom box until it
; receives an UP event from the draw widget. Then it turns draw widget motion
; events OFF and changes the event handler for the draw widget back to
; ZPLOT_PROCESS_EVENTS.


Catch,error_status
IF error_status ne 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XWINDOW_ZOOM_DRAWBOX: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

; Get the info structure out of the top-level base.

Widget_Control, event.top, Get_UValue=info ; , /No_Copy

;  Set up colors for line plot.

;backColor = GetColor('charcoal', !D.Table_Size-2)
;dataColor = GetColor('yellow', !D.Table_Size-3)
;axisColor = GetColor('green', !D.Table_Size-4)
;boxColor = GetColor('beige', !D.Table_Size-5)
boxColor = info.zoomBoxColor
IF boxColor EQ -1 THEN boxColor=!P.Color

   ; What type of an event is this?

possibleEventTypes = [ 'DOWN', 'UP', 'MOTION', 'SCROLL' ]
thisEvent = possibleEventTypes(event.type)

IF thisEvent EQ 'UP' THEN BEGIN

   ; If this is an UP event, you need to erase the zoombox, restore
   ; the user's color table, turn motion events OFF, set the
   ; draw widget's event handler back to ZPLOT_PROCESS_EVENTS, and
   ; draw the "zoomed" plot in both the draw widget and the pixmap.

   ; Erase the zoombox one final time by copying the plot from the pixmap.

   Widget_Control,info.wdraw,Get_Value=drawIndex
   WSet, drawIndex
   Device, Copy = [0, 0, info.zxsize, info.zysize, 0, 0, info.pixIndex]

   ; Turn motion events off and redirect the events to ZPLOT_PROCESS_EVENTS.

   Widget_Control, event.id, Draw_Motion_Events=0, $
       Event_Pro='XWINDOW_ZOOM_START_EVENT'

   ; Draw the "zoomed" plot. Start by getting the new limits to the plot
   ; (i.e., the LAST zoom box outline).

   x = [info.xs, event.x]
   y = [info.ys, event.y]

   ; Make sure the x values are ordered as [min, max].

   IF info.xs GT event.x THEN x = [event.x, info.xs]

   ; Restore plot system variables.

   !X = info.x
   !Y = info.y
   !P = info.p

   ; Convert the x device coordinates to data coordinates.

   coords = Convert_Coord(x, y, /Device, /To_Data)

   ; Make sure the x coordinates are within the data boundaries of the plot.
   ; srio comments this out to avoid problems with log plots

   ;x1 = !X.CRange(0) > coords(0,0) < !X.CRange(1)
   ;x2 = !X.CRange(0) > coords(0,1) < !X.CRange(1)

   ;y1 = !Y.CRange(0) > coords(1,0) < !Y.CRange(1)
   ;y2 = !Y.CRange(0) > coords(1,1) < !Y.CRange(1)

   x1 =  coords(0,0) 
   x2 =  coords(0,1) 

   y1 =  coords(1,0) 
   y2 =  coords(1,1) 


   IF (x1 EQ x2) AND (y1 EQ y2) THEN $  ; zoom out
     info.zoomIn=0 ELSE info.zoomIn=1


   IF x1 EQ x2 THEN BEGIN
     x1 = info.userlimits[0]
     x2 = info.userlimits[1]
   ENDIF
   IF y1 EQ y2 THEN BEGIN
     y1 = info.userlimits[2]
     y2 = info.userlimits[3]
   ENDIF

   IF info.zoomIn EQ 0 THEN $
     Widget_Control,info.wZoom,Sensitive=0  ELSE $
     Widget_Control,info.wZoom,Sensitive=1 

   info.parms.xmin=min([x1,x2])
   info.parms.xmax=max([x1,x2])
   info.parms.ymin=min([y1,y2])
   info.parms.ymax=max([y1,y2])

   ; Draw the "zoomed" plot in both the draw widget and the pixmap.

 
   ; 
   ; Executes active commands
   ;
   XWindow_ExecuteBuffer,info

   ; Update the plot system variables.

   info.x = !X
   info.y = !Y
   info.p = !P

   ; Put the info structure back into its storage location and then, 
   ; out of here!

   Widget_Control, event.top, Set_UValue=info ; , /No_Copy
   RETURN
ENDIF ; thisEvent = UP


; Most of the action in this event handler occurs here while we are waiting
; for an UP event to occur. As long as we don't get it, keep erasing the
; old zoom box and drawing a new one.

; Erase the old zoom box.

Widget_Control,info.wdraw,Get_Value=drawIndex
WSet, drawIndex
Device, Copy = [0, 0, info.zxsize, info.zysize, 0, 0, info.pixIndex]

; Update the dynamic corner of the zoom box to the current cursor location.

info.xd = event.x
info.yd = event.y

; Draw the zoom box.

PlotS, [info.xs, info.xs, info.xd, info.xd, info.xs], $
       [info.ys, info.yd, info.yd, info.ys, info.ys], $
       /Device, Color=boxColor

; Put the info structure back into its storage location.

Widget_Control, event.top, Set_UValue=info ; , /No_Copy

END ;  XWINDOW_ZOOM__DRAWBOX 


;
;==============================================================================
;
PRO Xwindow_Event,event

COMMON exodus_xwindow, xwindowParms

Catch,error_status
IF error_status ne 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XWINDOW_EVENT: error caught: '+!error_state.msg, $
	Dialog_Parent=event.top)
  Catch, /Cancel
  GoTo,out
ENDIF

Widget_control,event.id,get_uvalue=uvalue
Widget_control,event.top,get_uvalue=info

Widget_control,info.wdraw,get_value=windownum
Wset,windownum


;
; Resize window area
;
IF event.top eq event.id THEN BEGIN 
  widget_control,info.wdraw,scr_xsize=event.x-5,scr_ysize=event.y-37
  IF info.zoomFlag EQ 1 THEN BEGIN
    WDelete,info.pixIndex
    Window,XSize=event.x-5,YSize=event.y-37,/Free,/PixMap
    info.pixIndex = !D.Window
    info.zxsize = event.x-5
    info.zysize = event.y-37
  ENDIF
  XWindow_ExecuteBuffer,info
  GoTo,out
ENDIF

case uvalue of
  'QUIT':  XWindow_Quit,event.top
  'PRINT': BEGIN
	;
	; If we are zooming, set {xy}style to 1
	;
	IF info.zoomIn EQ 1 THEN BEGIN
	  xstyle_old = !x.style
	  ystyle_old = !y.style
	  !x.style=1
	  !y.style=1
	ENDIF
	XPrint,buffer=info.buffer, data = info.data, $
	  Parms=info.parms, Group=event.top
	;
	; back {xy}style to its original value
	;
	IF info.zoomIn EQ 1 THEN  BEGIN
	  !x.style = xstyle_old 
	  !y.style = ystyle_old 
	ENDIF
	END
  'EDIT': Begin
	tmp = info.buffer
        data = info.data
	parms = info.parms
	extra = info.extra
	wtitle = info.wtitle
	what = ''
	Widget_Control,event.id,Get_Value=what
	action=''
	IF what[0] EQ 'Edit Parameters' THEN BEGIN
	  IF info.zoomFlag EQ 1 THEN BEGIN
	    parms.xMin=info.userLimits[0]
	    parms.xMax=info.userLimits[1]
	    parms.yMin=info.userLimits[2]
	    parms.yMax=info.userLimits[3]
	  ENDIF
	  IF Type(extra) EQ 8 THEN $
            XscrMenu,parms,ACTION = action,  $
		Wtitle='Edit '+wtitle+' Parameters', $
		Group=event.top, Dialog_Parent=event.top, _extra=extra ELSE $
            XscrMenu,parms,ACTION = action, Wtitle=$
		'Edit '+wtitle+' Parameters', Group=event.top, /Interp, $
		Dialog_Parent=event.top
          IF action EQ 'DONT' THEN RETURN
	  xwindowParms = parms ; copy to common vlock (for Exodus)
	  info.parms = parms
	  IF info.zoomFlag EQ 1 THEN BEGIN
	    info.userLimits=[parms.xMin,parms.xMax,parms.yMin,parms.yMax]
	    Widget_Control,info.wZoom,Sensitive=0
	  ENDIF
	ENDIF ELSE BEGIN
	  if n_elements(tmp) GT 1 then begin
	    ; srio adds fieldlen 25 Apr 1996.
            ;XscrMenu,tmp,ACTION = action, FIELDLEN=50
	    xdisplayfile1,text=tmp,/modal,title='Buffer to edit'
	    ; CHANGED AGAIN!!!! 98-12-21
            ;if action EQ 'DONT' then return
            IF action EQ 'DONT' THEN RETURN
	  ;endif else widget_control,event.id,get_value=tmp
            info.buffer = tmp ; !!! added
	  endif else tmp=what
	ENDELSE

;	widget_control,event.top,set_uvalue = $
;	  {wdraw:info.wdraw,buffer:tmp, data:data, parms:parms, $
;	    extra:extra, wtitle:wtitle} 
	strnew = info
	strnew.buffer=tmp
	strnew.parms=parms
	widget_control,event.top,set_uvalue = strnew

        ; 
        ; refresh
        ; 
        xwindow_refresh,event.top

;	IF n_elements(tmp) EQ 1 THEN itmp = execute(tmp(0)) ELSE $
;	   FOR ih=0L,n_elements(tmp)-1 DO junk = execute(tmp(ih))
;	IF info.zoomFlag EQ 1 THEN BEGIN
;	  WSet,info.pixIndex
;	  IF n_elements(tmp) EQ 1 THEN itmp = execute(tmp(0)) ELSE $
;	     FOR ih=0L,n_elements(tmp)-1 DO junk = execute(tmp(ih))
;	ENDIF

	RETURN
	END

	'ZOOMOUT': BEGIN
	  !X = info.x
	  !Y = info.y
	  !P = info.p
	  info.parms.xmin=info.userlimits[0]
	  info.parms.xmax=info.userlimits[1]
	  info.parms.ymin=info.userlimits[2]
	  info.parms.ymax=info.userlimits[3]
	  info.zoomIn = 0
	  Widget_Control,info.wZoom,Sensitive=0
	  XWindow_ExecuteBuffer,info
	  info.x = !X
	  info.y = !Y
	  info.p = !P
	END
  else:
endcase

out:
IF (N_Elements(info) NE 0 AND Widget_Info(event.top,/Valid_ID) EQ 1) THEN $
  Widget_control,event.top,Set_UValue=info
;
;
end
;
;=========================================================================
;
pro xwindow,buffer=buffer,window_size = w_size, data = data, $
  wtitle=wtitle, edit=edit, group = group , Parms=parms, _Extra=extra, $
  zoomFlag=zoomFlag, ZBoxColor = zoomBoxColor, ExitBuffer=exitBuffer, $
  Parent=parent


Catch,error_status
IF error_status ne 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XWINDOW: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

if not(keyword_set(group)) then group=0
if not(keyword_set(buffer)) then buffer = 'surface,data'
if not(keyword_set(data)) then data = dist(30)
if not(keyword_set(wtitle)) then wtitle = 'Xwindow'
if not(keyword_set(parms)) then parms = 0
if not(keyword_set(extra)) then extra = 0
if not(keyword_set(exitBuffer)) then exitBuffer = ''


;
; widget def
;
wbase = widget_base(title=wtitle,/col,/TLB_SIZE_EVENTS, $
	Kill_Notify='xwindow_quit')
parent = wbase

wbuttons = widget_base(wbase,/ROW)
wquit = widget_button(wbuttons,VALUE='Quit',UVALUE='QUIT')
wprint = widget_button(wbuttons,VALUE='Print',UVALUE='PRINT')

if n_elements(w_size) lt 2 then begin     ; default window size
  device, GET_SCREEN = w_size
  w_size = (w_size(0) * [5,4]) /9
endif

IF N_Elements(zoomFlag) EQ 0 THEN zoomFlag=0
IF zoomFlag THEN BEGIN
  wdraw = widget_draw(wbase,XSIZE=w_size[0], YSIZE=w_size[1], $
  Button_Events=1, Event_Pro='XWINDOW_ZOOM_START_EVENT')
  Window,/Free,XSize=w_size[0], YSize=w_size[1], /PixMap
  pixIndex=!D.Window
  userlimits=[parms.xmin,parms.xmax,parms.ymin,parms.ymax]
  IF N_Elements(zoomBoxColor) EQ 0 THEN zoomBoxColor = -1L
ENDIF ELSE BEGIN
  wdraw = widget_draw(wbase,XSIZE=w_size(0), YSIZE=w_size(1))
  pixIndex=-1L
  userlimits=0
  zoomBoxColor = 0L
ENDELSE

button_edit = 0
button_parms = 0
IF keyword_set(edit) THEN BEGIN
   CASE edit OF
   2: button_edit = 1
   3: button_parms = 1
   else: BEGIN
	button_edit = 1
	button_parms = 1
	END
   ENDCASE
ENDIF

IF button_edit THEN BEGIN
  if n_elements(buffer) GT 1 then $
  wedit = widget_button(wbuttons,VALUE='Edit Buffer',UVALUE='EDIT') $
  else  wedit = widget_text(wbase,VALUE=buffer,/EDIT,/ALIGN_LEFT,$
    UVALUE='EDIT',XSIZE=strlen(buffer)+15)
ENDIF
IF button_parms THEN $
 wparms = widget_button(wbuttons,VALUE='Edit Parameters',UVALUE='EDIT')

IF zoomFlag THEN BEGIN
  wZoom = Widget_Button(wbuttons,VALUE='Zoom Out',UVALUE='ZOOMOUT')
  Widget_Control,wZoom,Sensitive=0
ENDIF ELSE wZoom=0L

widget_control,wbase,/REALIZE

;; 
;; Executes active commands
;;
;; define user variables 
;i=0 & j=0 & k=0 
;a0=0 & a1=0 & a2=0 & a3=0 & a4=0 & a5=0 & a6=0 & a7=0 & a8=0 & a9=0 & a10=0
;b0=0 & b1=0 & b2=0 & b3=0 & b4=0 & b5=0 & b6=0 & b7=0 & b8=0 & b9=0 & b10=0
;c0=0 & c1=0 & c2=0 & c3=0 & c4=0 & c5=0 & c6=0 & c7=0 & c8=0 & c9=0 & c10=0
;Widget_Control,wdraw,Get_Value=windownum
;windows=[windownum,pixIndex]
;FOR i=0,N_Elements(windows)-1 DO BEGIN
;  IF windows[i] NE -1 THEN BEGIN
;    WSet,windows[i]
;    IF n_elements(buffer) EQ 1 THEN tmp = execute(buffer(0)) ELSE $
;       FOR ih=0L,n_elements(buffer)-1 do tmp = execute(buffer(ih))
;  ENDIF
;ENDFOR
 
;
; info
;
info = { $
  wdraw:wdraw, $		; the draw widget id
  wZoom:wZoom, $		; the zoom widget id
  buffer:buffer, $	; the buffer to be executed
  exitBuffer:exitBuffer, $	; the exitbuffer to be executed
  data:data, $		; the data kwyword
  parms:parms, $	; the parms keyword
  extra:extra, $	; Extra keyword pointers
  wtitle:wtitle, $ 	; The window title
  pixIndex:pixIndex, $	; the index of the pixmap window (needed for zoom)
  xs:0, $	; X static corner of the zoom box (needed for zoom) 
  ys:0, $	; Y static corner of the zoom box (needed for zoom) 
  xd:0, $	; X dynamic corner of the zoom box (needed for zoom) 
  yd:0, $ 	; Y dynamic corner of the zoom box (needed for zoom) 
  zxSize:w_Size[0], $ ; X size of window (needed for zoom) 
  zySize:w_Size[1], $ ; Y size of window (needed for zoom) 
  x:!x, $ 		; (needed for zoom) 
  y:!y, $ 		; (needed for zoom) 
  p:!p, $ 		; (needed for zoom) 
  zoomflag:zoomflag, $ 	; (needed for zoom) 
  userlimits:userlimits, $	; (needed for zoom)
  zoomIn:0, $
  zoomBoxColor:zoomBoxColor } 


XWindow_ExecuteBuffer,info

widget_control,wbase,set_uval=info

xmanager, 'xwindow', wbase,group=group,/no_block
end

