;+
;
; =================================  EXODUS ===============================
;
;   EXODUS is a widget based application to manipulate SPEC files: 
;
;	1) It loads a SPEC file and user can  select, view, interpolate, 
;		average, and perform operations with data from multiple scans. 
;	   It also accepts MCS SPEC scans. Hete the XCol=1 means channel
;		numbet and XCol=2 means Energy (using calibration from
;		spec file).
;          It also loads data from ASCII files and from other XPlot windows.
;	2) It allows to export the selected data set from one or 
;		many SPEC files into files af different types
;		(Spec/Scan,Spac/Mesh, etc.)
;	3) It contains some tools to convert and operate with 
;		lists of Spec/MCA files. 
;
;		
;  COPYRIGHT:
; 	EXODUS belongs to XOP package and it is distributed within XOP.
; 	PLEASE REFER TO THE XOP COPYRIGHT NOTICE
; 
;  REFERENCE:
; 	Published calculations made with XOP should refer:
; 
; 	M. Sanchez del Rio and R. J. Dejus 
;         "Status of XOP: an x-ray optics software toolkit"
;         SPIE Proceedings Vol. 5536 (2004) pp.171-174
; 
;         http://dx.doi.org/10.1117/12.560903
; 
;  LAST MODIFICATION: srio@esrf.eu 2009-05-25
; 
; 
; ============= Non interactive use of EXODUS  ============================
;
;	NAME:
;		EXODUS
;	PURPOSE:
;		Manipulate and overplot multiple sets of data.
; 	CATEGORY:
;		Widgets. SPEC tools.
;	CALLING SEQUENCE:
;		EXODUS 
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;			When this ID is specified, a death of the caller 
;			results in a death of EXODUS.
;		NO_BLOCK = Set this keyword to start EXODUS and leaving
;			unblocked the IDL prompt (default: no_block=1)
;		XAFS = Set this keyword to start EXODUS with some
;			functionality for EXAFS data treatment. 
;		OnlyRight = Set this keyword if you want to eliminate
;			the spec preparation panel.
;		Quit_button = Set this keyword if you want to create
;			a QUIT button in the right panel.
;		Xlog = Set this keyword to start plots in log (X)
;		Ylog = Set this keyword to start plots in log (Y)
;               Wtitle = String with the window title
;
;	OUTPUTS:
;		Open the widget utility.
;	COMMON BLOCKS:
;		Exodus, to store some widget values
;	HISTORY:
;		EXODUS was  designed for being an application aiming
;		to average multiple scans for the ESRF beamline ID26
;		(at the time dedicated to EXafs Of DilUte Samples,
;		origin of the program title). However, as the
;		application revealed to be of general interest, it was 
;		promoted to be a general XOP tool. The "particularities" 
;		of EXAFS data analysis (i.e., interpolate and resample in 
;		k space, etc.), are hidden in the standard version, and are 
;		only available if EXODUS is started using the XAFS keyword on. 
;
;	MODIFICATION HISTORY:
;		Written by  Manuel Sanchez del Rio. ESRF. 13 April 2000
;		version 1.0 Beta2
;		00-05-10 srio@esrf.fr doc, added resample, 
;			and other changes. Release of version 1.0.
;		03-03-27 added MCA stuff. version 1.1
;			Bug fixed (did not use the first spectrum 
;			for average and sum...) M. Valvidares (id3).
;			use ompp inetead of mpp (obsoleted).
;			Export to mesh file. New routines:
;			exodus_interpolation and exodus_quit. 
;		03-03-31 Promoted from being an utility of xop/xaid
;			to be a general tool for XOP. 
;		03-11-17 srio@esrf.fr Added "Import from" options in
;			right panel.  Version 1.11
;		04-01-14 srio@esrf.fr Added Preferences (skip lines in ascii)
;		04-02-27 srio@esrf.fr Added import filter version 1.12
;		06-03-01 srio@esrf.fr Added onlyRight V 1.13
;			 Non interactive use via exodus_loadfile
;		06-04-12 srio@esrf.fr V 1.14
;			 Any operation allowed for ColX, ColY
;			 Export to Microsoft/Excel file
;		06-04-18 srio@esrf.fr export to multicolumn spec and excel
;		06-06-07 srio@esrf.fr Adds SPEC/MCA support. v. 1.15
;		06-07-03 srio@esrf.fr Bug fixed when def ColX=...,ColY=..
;		06-08-14 srio@esrf.fr keeps limits in xwindow 
;		06-09-12 srio@esrf.fr small bugs fixed
;		06-10-10 srio@esrf.fr fixes bug with fufifa_loadfile,/append
;		06-10-12 srio@esrf.fr adds operation on induvidual sets
;		06-11-07 srio@esrf.fr debug copying multiple columns.
;		07-09-07 srio@esrf.eu adds "append" data sets
;		07-11-07 srio@esrf.eu fixes some bugs in EXPORT options
;                        and adds Xplot view in them
;		07-11-07 srio@esrf.eu adds /xlog and /ylog
;		08-04-15 srio@esrf.eu adds wTitle keyword.
;		08-05-30 srio@esrf.eu adds Operations->Merge, Export->edf
;		08-11-11 srio@esrf.eu adds Export->CSV, v1.28
;		08-12-05 srio@esrf.eu skip recalculating data when col1 and
;                        col2 are selected. It keeps additional columns (for
;                        errors). v 1.29
;               09-09-02 srio@esrf.eu adds tolltips and cut data in operations
;                        v 1.30
;               10-06-11 srio@esrf.eu multi-file ASCII export allowed
;
;-
;
;========================================================================
;
Function Exodus_Version
Return,'1.31'
END
;
;========================================================================
;
PRO exodus_quit,event
;
Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'Exodus_quit: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

;Widget_Control,event.top, Get_UValue=state
stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state ; , /No_Copy


IF Ptr_Valid(state.ptrSpecFile) EQ 1 THEN Ptr_Free,state.ptrSpecFile
IF Obj_Valid(state.ompp) EQ 1 THEN Obj_Destroy,state.ompp

Widget_Control,event.top,/Destroy
END ; Exodus_quit

;
;========================================================================
;

PRO exodus_loadfile,stateid, isel=isel,Dialog_Parent=Dialog_Parent, $
	data=data,title=title

Widget_Control,stateid,Get_UValue=state

;+ 
;
; PRO EXODUS_LOADFILE, parent, isel=isel, data=data, Dialog_Parent=id, $
;	title=title
;
; 06-03-01 srio@esrf.fr
; isel: 0 Import data keyword (title=title for title)
;       1 Import from current Xplot window
;       2 Import from other Xplot window
;       3 Import from ASCII file (browser)
;       4 Import from ASCII file (wildcard)
;       5 Import filter (one file) 
;       6 Import filter (browser)
;       7 SPEC file (just a message)
;      10 delia (3cols: X,Y,Index)
;	0 data
;-
IF N_Elements(isel) EQ 0 THEN isel=0L ELSE isel=Long(isel)

	CASE isel OF
	 0: BEGIN ; Import from data kw
	  IF N_Elements(title) EQ 0 THEN title='Imported data'
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title=title
          state.ompp = ptr
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END
	 1: BEGIN ; Import from current Xplot window
          IF Widget_Info(state.xplot,/Valid_Id) EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
	     'No active xplot window created from Exodus')
	    Goto,out
	  ENDIF
          Widget_Control,/HourGlass
          Xplot_GetCurrentData,state.xplot,data
	  IF N_Elements(data) LE 1 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
             'No data imported')
            Goto,out
	  ENDIF
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title='Imported from xplot'
          state.ompp = ptr
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END
	 2: BEGIN ; Import Xplot window with ID...
	  id = state.xplot
	  XEdit,id,TEXT='XPlot window ID:',TITLE='Import from XPlot',$
	    Infotext=['Enter the Xplot ID number of the window from where'+$
		' you want to import data.', $
		'(See it in Xplot->View->XPlot ID number)'],Action=Action,$
		Dialog_Parent=dialog_Parent
	  IF action EQ 'CANCEL' THEN GoTo,out
          IF Widget_Info(id,/Valid_Id) EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
	     'XPLOT Window does not exist: '+StrCompress(id) )
	    Goto,out
	  ENDIF
          Widget_Control,/HourGlass
          Xplot_GetCurrentData,id,data
	  IF N_Elements(data) LE 1 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
             'No data imported')
            Goto,out
	  ENDIF
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title='Imported from xplot '+ $
		StrCompress(id)
          state.ompp = ptr
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END

	 3: BEGIN ; Import from ASCII file
	  dir = state.importDir 
	  file = Dialog_PickFile(Title='Select an ASCII file',$
	    Dialog_Parent=dialog_Parent,PATH=dir,GET_PATH=dir,$
	    /Multiple_Files)
	  IF file[0] EQ '' THEN GoTo,out
	  Widget_Control,/Hourglass
	  FOR i=0,N_Elements(file)-1 DO BEGIN
	    data = rascii(file[i],skip=state.pref.parameters.rascii_skip)
	    ptr = state.ompp
	    ptr->set,Value=data,/OverAdd, Title=file[i]
            state.ompp = ptr
	  ENDFOR
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END

	 4: BEGIN ; Import ASCII from list
	  wildfile=state.wildfile ; '*.txt'
	  XEdit,wildfile,TEXT='Match string: ',TITLE= 'Import from file list',$
	    Infotext=['Enter a string describing the files you want to use'],$
		Action=Action,Dialog_Parent=dialog_Parent,XSize=25
	  IF action EQ 'CANCEL' THEN GoTo,out
          state.wildfile=wildfile
	  files = FindFile(wildfile)
	  IF files[0] EQ '' THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
             'No files found')
            Goto,out
	  ENDIF
	  XDisplayFile1,TEXT=files,/Modal,Title= $
	    'Selected files (File->Accept to proceed)',Action=action, $
	    Dialog_Parent=dialog_Parent
	  IF action EQ 'DONT' THEN GoTo,out
	  Widget_Control,/Hourglass
	  nfiles = N_Elements(files)
	  FOR i=0,nfiles-1 DO BEGIN
	    data = rascii(files[i],skip=state.pref.parameters.rascii_skip)
	    ptr = state.ompp
	    ptr->set,Value=data,/OverAdd, Title=files[i]
            state.ompp = ptr
	  ENDFOR
          list = (state.ompp)->value(/Title)
	  nlist = N_Elements(list)
          sele=IndGen(nfiles)+(nlist-nfiles)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	  sele
	  END

	 5: BEGIN ; Import filter
	  Widget_Control,/Hourglass
	  x=0 & y = 0
	  xtmp=Temporary(x)
	  ytmp=Temporary(y)
	  command=state.pref.parameters.import_filter
	  itmp = Execute(command[0])
	  IF itmp NE 1 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
            'Error executing: '+command[0])
          Goto,out
	  ENDIF
	  IF (size(y))[0] GT 1 THEN BEGIN
	     data = y 
          ENDIF ELSE BEGIN
	    IF N_Elements(x) EQ 0 THEN x=FindGen(N_Elements(y))
	    data = make_set(x,y)
	  ENDELSE
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title=command[0]
          state.ompp = ptr

          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END


	 6: BEGIN ; Import filter from list
	  wildfile=state.wildfile ; '*.txt'
	  XEdit,wildfile,TEXT='Match string: ',TITLE= 'Import from file list',$
	    Infotext=['Enter a string describing the files you want to use'],$
		Action=Action,Dialog_Parent=dialog_Parent,XSize=25
	  IF action EQ 'CANCEL' THEN GoTo,out
	 state.wildfile=wildfile
	  files = FindFile(wildfile)
	  IF files[0] EQ '' THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
             'No files found')
            Goto,out
	  ENDIF
	  nfiles = N_Elements(files)
	  command  = Replicate(state.pref.parameters.import_filter,nfiles)
	  FOR i=0,nfiles-1 DO $
		command[i]=StrSubstitute(command[i],'file',"'"+files[i]+"'")
	  XDisplayFile1,TEXT=command,/Modal,Title= $
	    'Selected files (File->Accept to proceed)',Action=action, $
	    Dialog_Parent=dialog_Parent
	  IF action EQ 'DONT' THEN GoTo,out
	  Widget_Control,/Hourglass
	  FOR i=0,N_Elements(command)-1 DO BEGIN
	    x=0 & y = 0
	    xtmp=Temporary(x)
	    ytmp=Temporary(y)
	    itmp = Execute(command[i])
	    IF itmp NE 1 THEN BEGIN
	      itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
              'Error executing: '+command[i])
              Goto,out
	    ENDIF
	    
	    IF (size(y))[0] GT 1 THEN BEGIN
	       data = y 
            ENDIF ELSE BEGIN
	      IF N_Elements(x) EQ 0 THEN x=FindGen(N_Elements(y))
	      data = make_set(x,y)
	    ENDELSE
	    ptr = state.ompp
	    ptr->set,Value=data,/OverAdd, Title=command[i]
            state.ompp = ptr
	  ENDFOR
          list = (state.ompp)->value(/Title)
	  nlist = N_Elements(list)
          sele=IndGen(nfiles)+(nlist-nfiles)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	  sele
	  END
	 7: BEGIN  ; SPEC
            itmp = Dialog_message(/Info,Dialog_Parent=stateId,$
              'Please use the LEFT PANEL to import SPEC files')
            RETURN
	  END
	 10: BEGIN  ; XAID/DELIA
		IF N_Elements(data) EQ 0 THEN BEGIN
		  ffile = Dialog_Pickfile(Title='Read 3 col file: X, index, Y')
		  IF ffile EQ '' THEN GoTo,out
		  data=rascii(ffile)
		ENDIF
		IF Type(data) EQ 7 THEN data=rascii(data) ; File name
		arrx = Reform(data[0,*])
		arry = Reform(data[1,*])
		arri = Reform(data[2,*])
		idx = arri[UNIQ(arri, SORT(arri))]
		nspectra=N_Elements( idx ) 
	  FOR i=0,nspectra-1 DO BEGIN
	    igood = Where(arri EQ  idx[i]) 
	    data = Make_Set(arrx[igood],arry[igood])
	    ptr = state.ompp
	    ptr->set,Value=data,/OverAdd,  $
		Title='Index: '+StrCompress(idx[i],/Rem)
            state.ompp = ptr
	  ENDFOR
          list = (state.ompp)->value(/Title)
	  nlist = N_Elements(list)
          sele=IndGen(nspectra)+(nlist-nspectra)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	  sele
	  END
	 else: 
	ENDCASE

out:
if Widget_Info(stateid,/Valid_ID) then $
  Widget_Control, stateid, set_UValue=state ; , /No_Copy

END
;
;========================================================================
;
;
; calculate interpolated data (for sum, averaging, mesh writing, etc. )
;
PRO exodus_interpolate,event,state,isel,xInt=xInt,yInt=yInt, $
  leave_flag=leave_flag,ir_label=ir_label

COMMON exodus_interpolate, intstr
COMPILE_OPT IDL2
	leave_flag=0
        ;isel = Widget_Info(event.id,/DropList_Select)
	IF isel EQ 0 THEN BEGIN
	  leave_flag = 1
	  GoTo,out
	ENDIF
        ;Widget_Control,event.id,Set_DropList_Select=0

	iList = Widget_Info(state.wids.PRList,/List_Select)
	IF isel EQ 5 THEN BEGIN
	  IF N_Elements(iList) LE 2 THEN BEGIN
 	    itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	      ['Please select at least three lines including:',$
		'- the data set with the target (to be fitted)',$
		'- two or more components ','',$
		'The fit will minimize the difference between the',$
		'target and a linear combination of the components'])
	    leave_flag=1
	    GoTo,out
	  ENDIF
	  ptrMPP = state.ompp
	  pText=''
	  tmp = ompp_lcfit(ptrMPP,iList,Dialog_Parent=event.top, $
		PText=pText)
	  ;
	  ; store result
	  ;
	  IF N_Elements(tmp) LE 1 THEN BEGIN
	    leave_flag=1
	    GoTo,out
	  ENDIF
	  ptrMPP->set,Value=tmp,Title='Fit result '+pText,/OverAdd
          state.ompp = ptrMPP
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,$
                  Set_List_Select=iList
	  leave_flag=1
	  GoTo,out
	ENDIF
 
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    'Please select some data lines')
	  leave_flag=1
	  GoTo,out
	ENDIF
	IF N_Elements(iList) LE 1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    ['Please select MULTIPLE data lines',$
	     '(use <Ctr>+Click or <Shift>+Click for multiple selection.)'])
	  leave_flag=1
	  GoTo,out
	ENDIF

	if isel EQ 6 THEN RETURN

	mpp = state.ompp
	;
	; Check if all abscissas are the same and calculate extrema
	; 
	ideff = 0
	nn = N_Elements(iList)
	xMin = DblArr(nn)
	xMax = DblArr(nn)
	xStep = DblArr(nn)
	xPts = LonArr(nn)
	FOR i=0,nn-1 DO BEGIN
	  data = mpp->value(iList[i])
	  IF i EQ 0 THEN BEGIN
	    xx = Reform(data[0,*]) 
	    xMax[i] = Max(xx)
	    xMin[i] = Min(xx)
	    xPts[i] = N_Elements(xx)
	    xStep[i] = (xMax[i] - xMin[i])/(xPts[i] - 1)
          ENDIF ELSE BEGIN
	    xi = Reform(data[0,*])
	    xMax[i] = Max(xi)
	    xMin[i] = Min(xi)
	    xPts[i] = N_Elements(xi)
	    xStep[i] = (xMax[i] - xMin[i])/(xPts[i] - 1)
	    IF N_Elements(xx) NE N_Elements(xi) THEN BEGIN
	      ideff =1 
	      txt = 'Abscissas arrays have different number of elements.'
	    ENDIF ELSE IF Total(Abs(xx-xi)) NE 0 THEN BEGIN
	      ideff =1 
	      txt = 'Abscissas arrays have different values.'
	    ENDIF
	  ENDELSE
	ENDFOR

	IF ideff NE 0 THEN BEGIN  ; interpolation
 	  itmp = Widget_Message(/Info,Dialog_Parent=event.top,/Cancel, $
	    [txt,'Data will be interpolated','','Information: ',$
	    'Minima extrema = '+Vect2String([Min(xMin),Max(xMin)]),$
	    'Maxima extrema = '+Vect2String([Min(xMax),Max(xMax)]),$
	    'Step = '+Vect2String([Min(xStep),Max(xStep)]), $
	    'Number of points in = '+Vect2String([Min(xPts),Max(xPts)])] )
	  IF itmp EQ 'Cancel' THEN BEGIN
	     leave_flag=1
	     GoTo,out
	  ENDIF
	  x1 = Max(xMin)
	  x2 = Min(xMax)
	  nPts = Min(xPts)
	  deltaX = Max(xStep)

	  IF N_Elements(intStr) EQ 0 THEN BEGIN
	    ;intStr = { type:['0','Like 1st data','Customized'], x1:x1, x2:x2,$ 
	    ;      nPts:nPts, step:['0','energy (E)','wavenumber (k)'],$ 
            ;      e0:x1, method:['0','Interpolation','Resample'] }
	    intStr = { type:['0','Like 1st data','Customized'], x1:x1, x2:x2,$ 
	          deltaX:deltaX, step:['0','energy (E)','wavenumber (k)'],$ 
		  e0:x1, method:['0','Interpolation','Resample'] }
	  ENDIF ELSE BEGIN
	    intStr.x1=x1
	    intStr.x2=x2
	    ;intStr.nPts=nPts
	    intStr.deltaX=deltaX
	    intStr.e0=x1
	  ENDELSE
	  IF state.xafs EQ 1 THEN BEGIN
	    flags = ['1','w(0) EQ 1','w(0) EQ 1','w(0) EQ 1', $
		'w(0) EQ 1','w(0) EQ 1 AND w(4) EQ 1','1' ]
	    titles = ['Interpolated abscissas','X Min [Energy]', $
		'X Max [Energy]', $
		'Step value [Energy or k]','Step constant in', $
		'E0 (energy edge)','Method']
	  ENDIF ELSE BEGIN
	    flags = ['1','w(0) EQ 1','w(0) EQ 1','w(0) EQ 1', $
		'0','0','1' ]
	    titles = ['Interpolated abscissas','X Min', $
		'X Max', $
		'Step value','****', '****','Method']
	  ENDELSE
	  action = 0
	  XScrMenu,intstr,/NoType,/Inter,Flags=flags,Titles=titles, $
	    Dialog_Parent=event.top,WTitle='Interpolation parameters',$
	    Action=action
	  IF action EQ 'DONT' THEN BEGIN
		leave_flag=1
		GoTo,out
	  ENDIF
          Widget_Control,/HourGlass

	  ;
	  ; Interpolate
	  ;

	  IF StrCompress(intstr.type[0],/Rem) EQ '0' THEN xint = xx ELSE BEGIN
	    CASE StrCompress(intstr.step[0],/Rem) OF
	     '0': BEGIN
		  nPts = ((intstr.x2-intstr.x1)/intStr.deltaX)+1
		  ;xInt = Abscissas(intstr.x1,intstr.x2,intstr.nPts)
		  xInt = Abscissas(intstr.x1,intstr.x2,nPts)
		  END
	     '1': BEGIN
		  xlimE =[[intstr.x1,intstr.x1],[intstr.x2,intstr.x2]]
		  xlimK = e2k(xlimE,E0=intstr.e0)
		  nPts = ( (xlimK[0,1]-xlimK[0,0]) / intStr.deltaX)+1
		  ;xIntk = Abscissas(xlimK[0,0],xlimK[0,1],intstr.nPts) 
		  xIntk = Abscissas(xlimK[0,0],xlimK[0,1],nPts) 
		  xInt = Make_Set(xIntk,xIntk)
		  xInt = k2e(xInt)
		  xInt = Reform(xInt[0,*])+intstr.e0
	          END
	    ENDCASE
	  ENDELSE
	  Widget_Control,/HourGlass
	  nxInt = N_Elements(xInt)
	  yInt = FltArr(nn,nxInt)
	  ir_label=''
	  FOR i=0,nn-1 DO BEGIN
	    data = mpp->value(iList[i])
	    xx = Reform(data[0,*])
	    yy = Reform(data[1,*])
	    CASE StrCompress(intstr.method[0],/Rem) OF
	    '0': BEGIN
		;Message,/Info,'Using Interpolate'
	        ; y = Interpol(yy,xx,xInt)
		y = Interpolate(yy,FIndex(xx,xInt))
	    	yInt[i,*] = y
		ir_label='i'
		END
	    '1': BEGIN
		;Message,/Info,'Using Xaid_Resample'
		y = XAID_Resample(xx,yy,xInt)
	    	yInt[i,*] = y
		ir_label='r'
		END
	     else:
	     ENDCASE
	  ENDFOR
	ENDIF ELSE BEGIN
	  data = mpp->value(iList[0])
          xInt = Reform(data[0,*])
	  nxInt = N_Elements(xInt)
	  yInt = FltArr(nn,nxInt)
	  IF nn GT 0 THEN BEGIN
	    FOR i=0,nn-1 DO BEGIN
	      data = mpp->value(iList[i])
	      yInt[i,*] = Reform(data[1,*])
	    ENDFOR
	  ENDIF
	ENDELSE
	; 
 out:
END

;
;========================================================================
;
PRO exodus_event,event

; 
; WARNING!!
; If extending this common block, do the same thing on xwindow.pro
;
COMMON exodus_xwindow, xwindowParms
;
; 
COMMON exodus, stroper_cust, stroper_interp, stroper_merge, $
       stroper_parametric, stroper_xanes_normalize, filter, $
       strExportAscii

forward_function xanes_normalize

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'EXODUS_EVENT: error caught: '+!err_string, $
	DIALOG_PARENT=event.top)
   catch, /cancel
   on_error,2
   goto,out
endif

IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
  Exodus_Quit,event
  RETURN
ENDIF

;
;
Widget_Control, event.id, get_UValue=eventUValue
;Widget_Control, event.id, get_Value=Value
 
if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''
 
stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state ; , /No_Copy

 
back:
Case eventUValue OF
    'QUIT': Exodus_Quit,event
    'PREFERENCES': BEGIN
	   Widget_Control,event.id,Get_Value=val
	   CASE val OF
	     'Edit Preferences...': BEGIN
	        tmp = state.pref.parameters
	        titles = state.pref.titles
	        XScrMenu,tmp,/NoType,/Interp, Action=action, Titles=titles, $
		     WTitle='Exodus preferences',Dialog_Parent=event.top, $
		     FieldLen=35
	        IF action EQ 'DONT' THEN GoTo,out
	        state.pref.parameters=tmp
		END
	     'Save as default': BEGIN
                str_par = state.pref.parameters
                Xop_Input_Save,str_par,Group=event.top, $
                   Default='exodus.xop',Comment='; xop/exodus(v'+$
                   exodus_version()+') input file on '+SysTime()
		END
	   ENDCASE
	END
    'IMPORT': BEGIN
	isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0
	CASE isel OF
	0: 
	1: BEGIN ; load spec file
	  IF Widget_Info(state.Xplot,/Valid_Id) EQ 1 THEN $
	    Xplot_Quit,state.Xplot
	  dir = state.importDir 
          IF N_Elements(filter) EQ 0 THEN filter='*'
	  file = Dialog_PickFile(Title='Select a SPEC file',$
	    Dialog_Parent=event.top,PATH=dir,GET_PATH=dir,filter=filter)
	  IF file EQ '' THEN GoTo,out
          tmp = strpos(file,'.')
          IF tmp[0] NE -1 THEN filter = '*'+strmid(file,tmp[N_Elements(tmp)-1])
	  Widget_Control,/Hourglass
          Widget_Control,event.top,TLB_Set_Title= $
	  'EXODUS '+Exodus_Version()+' - Imported: '+file
	  n = Spec_Access(hSpec,file)
	  IF n EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error, DIALOG_PARENT=event.top, $
              ['EXODUS_EVENT: Error reading SPEC data file: ',file])
	    GoTo,out
	  ENDIF
	  list0 = StrCompress(hSpec.scan.scan_n)+' - '
	  list1 = hSpec.scan.name
	  Widget_Control,state.wids.PLList,Set_Value=list0+list1
	  state.ptrSpecFile=Ptr_New(hSpec)
	  state.specFile=file
	  state.importDir=dir
	  END
	2: BEGIN  ; Reload spec file
	  Widget_Control,/Hourglass
	  file = state.specFile
	  n = Spec_Access(hSpec,file)
	  IF n EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error, DIALOG_PARENT=event.top, $
              ['EXODUS_EVENT: Error reading SPEC data file: ',file])
	    GoTo,out
	  ENDIF
	  list0 = StrCompress(hSpec.scan.scan_n)+' - '
	  list1 = hSpec.scan.name
	  Widget_Control,state.wids.PLList,Set_Value=list0+list1
	  state.ptrSpecFile=Ptr_New(hSpec)
	END

	3: BEGIN  ;  mca files
	  itmp = Dialog_Message(Dialog_parent=event.top, /Info,/Cancel,$
		['MCA files cannot be load directly.',$
		 'They must be converted into a SPEC standard scan file.',$
		 'Do you want to do it?'])
	  IF itmp EQ 'Cancel' THEN GoTo,out
	  outFile = 0 ; variable to receive the file name
	  mca2scan,Group=event.top,info=info
	  IF N_Elements(info) EQ 0 THEN GoTo,out
	  file = info.outFile
	  dir = info.pathout

	  ; from here, the same as CASE=0
	  IF Widget_Info(state.Xplot,/Valid_Id) EQ 1 THEN $
	    Xplot_Quit,state.Xplot
	  IF file EQ '' THEN GoTo,out
	  Widget_Control,/Hourglass
          Widget_Control,event.top,TLB_Set_Title= $
	  'EXODUS '+Exodus_Version()+' - Imported: '+file
	  n = Spec_Access(hSpec,file)
	  IF n EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error, DIALOG_PARENT=event.top, $
              ['EXODUS_EVENT: Error reading SPEC data file: ',file])
	    GoTo,out
	  ENDIF
	  list0 = StrCompress(hSpec.scan.scan_n)+' - '
	  list1 = hSpec.scan.name
	  Widget_Control,state.wids.PLList,Set_Value=list0+list1
	  state.ptrSpecFile=Ptr_New(hSpec)
	  state.specFile=file
	  state.importDir=dir
	  END
	ENDCASE
	END
    'PLLIST': BEGIN
	IF event.clicks NE 2 THEN GoTo,out
	eventUValue = 'XPLOT_LEFT'
	GoTo,back
	; plot
	END
    'XSPECVIEW': BEGIN
	hSpec = *state.ptrSpecFile
	iList = Widget_Info(state.wids.PLList,/List_Select)
	tmp = Spec_Labels(hSpec,1+iList[0],/Index) ; To set current scan
	XSpecView,hSpec,Group=event.top,No_Block=state.no_block
	END
    'EXODUS': BEGIN
	Widget_Control,/HourGlass
	Exodus,No_Block=state.no_Block
	END
    'XAID': BEGIN
	Widget_Control,/HourGlass
	Xaid,No_Block=state.no_Block
	END
    'XPLOT': BEGIN
	Widget_Control,/HourGlass
	;Xplot,Group=event.top,No_Block=state.no_Block
	  XPlot,Parent=p,Group=event.top,No_Block=state.no_block
          state.Xplot=p
	END
    'DISPLAYFILE': BEGIN
	file = Dialog_Pickfile(Dialog_Parent=event.top, /Read)
	IF file EQ '' THEN GoTo,out
	XDisplayFile1,file,Dialog_Parent=event.top
	END
    'XPLOT_LEFT': BEGIN
        Widget_Control,/HourGlass
	hSpec = *state.ptrSpecFile
	iList = Widget_Info(state.wids.PLList,/List_Select)
	tmp = Spec_Labels(hSpec,1+iList[0],/Index) ; To set current scan
        IF Widget_Info(state.Xplot,/Valid_Id) EQ 0 THEN BEGIN

	  ; the default columns ate those selected to copy, in the case they are
	  ; defined as numbers
	  Widget_Control,state.wids.xCol,Get_Value=eqColX
	  Widget_Control,state.wids.yCol,Get_Value=eqColY
	  ; if column is only defined by a ore or two digit number
	  IF (strPos(eqColX[0],','))[0] EQ -1 AND StrLen(StrCompress(eqColX[0],/Rem)) LE 2 $
		  THEN xcol=Fix(eqColX[0]) ELSE xcol=0
	  IF (strPos(eqColY[0],','))[0] EQ -1 AND StrLen(StrCompress(eqColY[0],/Rem)) LE 2 $
		  THEN ycol=Fix(eqColY[0]) ELSE ycol=0

          p=0
	  XPlot,Parent=p,Group=event.top,No_Block=state.no_block,xcol=xcol,ycol=ycol
          state.Xplot=p
        ENDIF
        Xplot_LoadFile,state.Xplot,SPEC=hSpec,ScanId='CURRENT'

	END


    'HELPCOLUMNS': BEGIN
	    txt = [$
	    'Enter here the columns of the spec file to copy to the right panel.',$
	    ' ',$
	    'Simple copy: ',$
	    '  The simples way is to enter a the column number (starting from 1) ', $
	    '  Alternatively (mandatory for operations and complex expressions),',$
	    '  enter a string with the column number (starting from 1) after the word "col"', $
	    '  Examples: ',$
	    '    col1',$
	    '    col5',$
	    ' ',$
	    'Operations during copy: ',$
	    '  Enter the formula for the new column.',$
	    '  Examples: ',$
	    '    col1*1000',$
	    '    col1/col2',$
	    ' ',$
	    'Complex expressions could be coded in several command separated by &',$
	    '  Example: ',$
	    '    colX=col1*1000 & colX[10:100]=0 ',$
	    ' ',$
	    'Multiple copying can be entered using columns.',$
	    '  Example: ',$
	    '    col1,col2/col1', $
	    ' ',$
	    'Multi Channel Analysers scan will have: ',$
	    '  col1 is the channel column (not calibrated)',$
	    '  col2 is the calibrated data (using the corresponding spec keyword)',$
	    '  col3 is the first MCA data of the scan, and so on.' ]
	     XDisplayFile1,Text=txt,Group=event.top,Title='Help copying columns',height=36
	END
    ; 
    ; Central panel
    ;
    'COPY1': BEGIN
	Widget_Control,event.id,Get_Value=val
	hSpec = *state.ptrSpecFile
	IF Fix( state.pref.parameters.specNumbering[0]) EQ 1 THEN BEGIN
	  list = '['+StrCompress(hSpec.scan.scan_n,/Rem)+'] ' + hSpec.scan.name
	ENDIF ELSE BEGIN
	  list = hSpec.scan.name
	ENDELSE
	CASE val OF
	  '=>': iList = Widget_Info(state.wids.PLList,/List_Select)
	  '*=>': iList=LIndGen(N_Elements(list))
	  '?=>': BEGIN
		tmp= state.wildcard_Transfer
		Xedit,tmp,Text='Enter text with wildcard to copy:',$
		  Dialog_Parent=event.top,Action=action,XSize=40
		IF action EQ 'CANCEL' THEN GoTo,out
		state.wildcard_transfer=tmp
		iList = Where( StrMatch(List,tmp) EQ 1)
		IF iList[0] EQ -1 THEN BEGIN
		  itmp = Dialog_message(/Error,Dialog_Parent=event.top, $
			'No matches')
		  GoTo,out
		ENDIF
		END
	  else: 
	ENDCASE
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    'Please select a data line')
	  GoTo,out
	ENDIF

	ptr = state.ompp
	Widget_Control,state.wids.xCol,Get_Value=eqColXs
	Widget_Control,state.wids.yCol,Get_Value=eqColYs

	titles = list[ilist]

	Widget_Control,/HourGlass

	FOR i=0,N_Elements(iList)-1 DO BEGIN
	  index = iList[i]+1
          mc=0
	  npoints = Spec_Points(hSpec,index,/Index,MCData=mc) 
	  IF npoints LE 0 THEN BEGIN
	    mc=1
            npoints  = Spec_Points(hSpec,index,/Index,MCData=mc)
	  ENDIF
	  IF npoints LE 0 THEN BEGIN
	        tmp = Dialog_Message(/Question,['Scan contains no data',$
	          '','Abort? '],Dialog_Parent=event.top)
	        CASE tmp OF
	          'No': 
	          'Yes': GoTo,out
	        ENDCASE
	        Widget_Control,/HourGlass
	  ENDIF ELSE BEGIN
		IF mc THEN BEGIN
	          yy = Spec_Data(hSpec,index,/Index,/MCData,$
		      Calibrated=xcal, Channel=xcha	)
		  data=Make_Set(xcha,xcal,yy) 
		ENDIF ELSE BEGIN
	          ncol = Spec_Columns(hSpec,index,/Index)
	          data = Spec_Data(hSpec,index,/Index)
		ENDELSE

		;
		; info window
		;
		col1 = 0  &  col2 = 0  &  col3 = 0  &  col4 = 0  &  col5 = 0
		col6 = 0  &  col7 = 0  &  col8 = 0  &  col9 = 0  &  col10 = 0
		col11 = 0 &  col12 = 0  &  col13 = 0  &  col14 = 0  &  col15 = 0
		col16 = 0 &  col17 = 0  &  col18 = 0  &  col19 = 0  &  col20 = 0
		col21 = 0 &  col22 = 0  &  col23 = 0  &  col24 = 0  &  col25 = 0
		col26 = 0 &  col27 = 0  &  col28 = 0  &  col29 = 0  &  col30 = 0
		col31 = 0 &  col32 = 0  &  col33 = 0  &  col34 = 0  &  col35 = 0
		col36 = 0 &  col37 = 0  &  col38 = 0  &  col39 = 0  &  col40 = 0
		col41 = 0 &  col42 = 0  &  col43 = 0  &  col44 = 0  &  col45 = 0
		col46 = 0 &  col47 = 0  &  col48 = 0  &  col49 = 0  &  col50 = 0
			
		ncol = n_elements(data[*,0])
		if ncol gt 50 then begin
  		txt = ['Operation with columns are only allowed', $
    		'with set holding up to 50 columns.', $
    		'The current set has '+strcompress(ncol,/rem)+' columns.',$
    		'Problems could appear.']
  		tmp = Dialog_Message(dialog_parent=event.top,txt)
		end
		;
		; load data
		;
		for _i=1,ncol do begin
   		  command = 'col'+strcompress(_i,/rem)+' = Reform(data['+$
    		  strcompress(_i-1,/rem)+',*]) '
   		  tmp = execute(command[0])
		endfor

		
		if not(keyword_set(data)) then begin
    		  tmp = Dialog_Message(dialog_parent=event.top,$
      		  ['Empty data set','Returning'],/ERROR)
    		  return
		endif
		
                ; 
                ;  
                skip_execute=0
                IF StrCompress(eqColXs[0],/Rem) EQ 'col1' AND $
                   StrCompress(eqColYs[0],/Rem) EQ 'col2' THEN skip_execute=1
		;
		; evaluate the expression
		;
		nx = strParse(eqColXs[0],',',listX) 
		IF nx GT 0 THEN eqColX=listX ELSE eqColX=eqColXs[0]
		ny = strParse(eqColYs[0],',',listY) 
		IF ny GT 0 THEN eqColY=listY ELSE eqColY=eqColYs[0]

	FOR ii=0L,nx DO BEGIN
	FOR jj=0L,ny DO BEGIN
        ; skip recalculating data if col1 and col2 are selected
        IF skip_execute NE 1 THEN BEGIN
		; if column is only defined by a ore or two digit number
		IF (strPos(eqColX[ii],','))[0] EQ -1 AND StrLen(StrCompress(eqColX[ii],/Rem))LE 2 $
			THEN BEGIN
		  commandX = 'col'+StrCompress(eqColX[ii],/Rem)
		ENDIF ELSE BEGIN
		  commandX = eqColX[ii]
		ENDELSE

		; If command does not contain "=" then add 
		IF (strPos(commandX,'='))[0] EQ -1 THEN $
  		  commandX = 'colX = '+commandX 
		colX=0 & colY=0
		tmp = execute(commandX[0])
		if tmp ne 1 then begin
  		  tmp = Dialog_Message(dialog_parent=event.top,$
    		  'Operation expression not understood:'+commandX[0],/ERROR)
  		  return
		endif

		; if column is only defined by a one or two digit number
		IF (strPos(eqColY[jj],','))[0] EQ -1 AND StrLen(StrCompress(eqColY[jj],/Rem))LE 2 $
			THEN BEGIN
		  commandY = 'col'+StrCompress(eqColY[jj],/Rem)
		ENDIF ELSE BEGIN
		  commandY = eqColY[jj]
		ENDELSE

		; If command does not contain "=" then add 
		IF (strPos(eqColY,'='))[0] EQ -1 THEN $
  		  commandY = 'colY = '+commandY
		tmp = execute(commandY[0])
		if tmp ne 1 then begin
  		  tmp = Dialog_Message(dialog_parent=event.top,$
    		  'Operation expression not understood:'+commandY[0],/ERROR)
  		  return
		endif
		;
		; store data
		;
		IF N_Elements(colX) NE N_Elements(colY) THEN BEGIN
  		  tmp = Dialog_Message(dialog_parent=event.top,$
    		  'Operation error : colX and colY have different number of points ',/ERROR)
  		  return
		ENDIF
		IF N_Elements(colX) LE 1 THEN BEGIN
  		  tmp = Dialog_Message(dialog_parent=event.top,$
    		  'Operation error : colX has less than two points ',/ERROR)
  		  return
		ENDIF
		IF N_Elements(colY) LE 1 THEN BEGIN
  		  tmp = Dialog_Message(dialog_parent=event.top,$
    		  'Operation error : colY has less than two points ',/ERROR)
  		  return
		ENDIF
		data=Make_Set(colx,coly)
		IF Fix( state.pref.parameters.specNumbering[0]) EQ 1 THEN BEGIN
		  tmp = titles[i]+'; '+commandX+'; '+commandY
		ENDIF ELSE BEGIN
		  tmp = titles[i]
		ENDELSE
        ENDIF ELSE BEGIN
	  tmp = titles[i]
        ENDELSE
	        ptr->set,Value=data,/OverAdd, Title=tmp
	ENDFOR ;jj
	ENDFOR ;ii
	


	  ENDELSE
	ENDFOR
	state.ompp = ptr
	list = (state.ompp)->value(/Title)
	Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	  N_Elements(list)-1

	END
    ; 
    ; Right panel
    ;
    'IMPORTRIGHT': BEGIN
	isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0
	exodus_loadfile,stateid,isel=isel,Dialog_Parent=event.top
	return
	END

    'VIEWEDIT': BEGIN
	isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0
	CASE isel OF
	 0:
	 1: BEGIN ; View Info
;	    tmp = MPP_Info(state.ptrMPP)
	    tmp = (state.ompp)->info()
	    XDisplayFile1,Text=tmp, Group=event.top, Title='Information ', $
		Dialog_Parent=event.top
	    END
	 2: BEGIN  ; Edit Labels
	      iList = Widget_Info(state.wids.PRList,/List_Select)
	      IF iList[0] EQ -1 THEN BEGIN
 	        itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	          'Please select a data line')
	        GoTo,out
	      ENDIF
;	      txt_old = MPP_Value(state.ptrMPP,ilist,/Title)
	      txt_old = (state.ompp)->value(ilist,/Title)
	      txt_new = txt_old
	      action = ''
	      XDisplayFile1,Text=txt_new,Title='Edit data labels', $
		Group=event.top,/Modal,Action=action, $
		Dialog_Parent=event.top
	      IF action EQ 'DONT' THEN GoTo,out
	      IF N_Elements(txt_new) NE N_Elements(txt_old) THEN BEGIN
		itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
		 'Number of label lines has changed. Aborted')
		GoTo,out
	      ENDIF
;	      MPP_Set,state.ptrMPP,ilist,Title=txt_new
	      (state.ompp)->set,ilist,Title=txt_new
;	      list = MPP_Value(state.ptrMPP,/Title)
	      list = (state.ompp)->value(/Title)
	      Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	        iList[0]
	    END
	 else:
	ENDCASE
	END
    'EXPORT': BEGIN
	isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0
	Widget_Control,event.id,Get_Value=export_items
	iList = Widget_Info(state.wids.PRList,/List_Select)
        iSelTxt = export_items[iSel]
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Please select a data line')
	  GoTo,out
	ENDIF

	; isel: 
;    		1: 'ASCII file(s)'
;		2: 'SPEC file'
;		3: 'SPEC/Mesh file'
;		4: 'SPEC multicolumn file'
;		5: 'EXCEL-XML multisheet file',
;		6: 'EXCEL -XML multicolumn file',
;		7: 'Application...'
;		8: 'FuFiFa'
;		9 (8): 'XAID/mu2chi'
;		10 (9): 'XAID/ff'
;		11 (10): 'DELIA binary file'
;	CASE isel OF
;	  1: flag_single=1
;	  7: flag_single=1
;	  9: flag_single=1
;	  10: flag_single=1
;	  else: flag_single=0
;	ENDCASE

	CASE iselTxt OF
           ;'ASCII file(s)': flag_single=1
           'Application...': flag_single=1
	   ;'XAID/mu2chi': flag_single=1
           ;'XAID/ff': flag_single=1
	  else: flag_single=0
	ENDCASE




	IF flag_single AND N_Elements(iList) GT 1 THEN BEGIN
 	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Only the data from a single data line can be exported')
	  GoTo,out
	ENDIF
	
	CASE iselTxt OF
	'Export: ': 
	'ASCII file(s)': BEGIN ; Save to a new ASCII file 

	   nn = N_Elements(iList)
	   titles = (state.ompp)->value(iList,/Title)


           CASE nn OF
            0: Message,'Please select a data line'
            1: BEGIN
               file = Dialog_PickFile(File='exodus.ascii',Title= $
		    'Select a file for writing (ASCII format)',$
                    Dialog_Parent=event.top)
               IF file EQ '' THEN GoTo,out

	       IF CheckFile(file) EQ 1 THEN BEGIN
	         itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		    'Overwrite it?'],Dialog_Parent=event.top)
	         IF itmp EQ 'No' THEN BEGIN
	           itmp = Dialog_Message(/Error, 'Operation aborted',$
		    Dialog_Parent=event.top)
	           GoTo,out
	         ENDIF
	       ENDIF
	       END
               else: BEGIN
                 IF N_Elements(strExportAscii) EQ 0 THEN $
                 strExportAscii = {prefix:'',$
                        use:['0','Full set title','First word'],$
                        suffix:'.ascii'}

                 xscrmenu,strExportAscii,titles=['Prefix for the new files',$
                  'file name:', 'Suffix: '],Action=action,/NoType,/Interpret,$
                  Dialog_Parent=event.top,wtitle='Write multiple files'

                 IF action EQ 'CANCEL' THEN GoTo,out
                 IF Fix( (strExportAscii.use)[0] ) EQ 0 THEN BEGIN
                    file = strExportAscii.prefix+titles+strExportAscii.suffix
                 ENDIF ELSE BEGIN
                    file=titles
                    FOR j=0L,N_Elements(file)-1 DO BEGIN
                      n = strparse(titles[j],' ',list)
                      file[j]=list[0]
                    ENDFOR
                    file = strExportAscii.prefix+file+strExportAscii.suffix
                 ENDELSE
	       END
           ENDCASE 


	   FOR i=0L,nn-1 DO BEGIN
	     set = (state.ompp)->value(iList[i])
	     OpenW,unit,file[i],/Get_Lun
	     format = '(2(G16.10," "))'
	     printf,unit,format=format,set 
	     Free_Lun,unit
             print,'File written to disk: '+file[i]
	   ENDFOR


	   END
	'SPEC multiscan file': BEGIN ; Save to a file (SPEC format)
           file = Dialog_PickFile(File='exodus.spec',Title= $
		'Select a file for writing (SPEC format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out
	   IF checkfile(file) EQ 1 THEN BEGIN
             itmp=Dialog_Message(/Question,/Cancel,Dialog_Parent=$
                event.top,['File exists: ',file,'Append data?','',$ 
		'(Yes: data appended)','(No: file rewritten)', $
		'(Cancel: Abort operation)'])
	      CASE itmp OF
	        'Yes':
	        'No': Delete_Files,file
		'Cancel': GoTo,out
	      ENDCASE
	   ENDIF
	   nn = N_Elements(iList)
	   hspec = 0
;	   titles = MPP_Value(state.ptrMPP,iList,/Title)
	   titles = (state.ompp)->value(iList,/Title)
	   FOR i=0L,nn-1 DO BEGIN
;	     set = MPP_Value(state.ptrMPP,iList[i])
	     set = (state.ompp)->value(iList[i])
	     IF i EQ 0 THEN BEGIN
		Widget_Control,/Hourglass
		tmp=Spec_Save(hspec,set,file,Name=titles[i],Dialog_Parent=$
			event.top,/No_Confirm) 
		IF tmp EQ 0 THEN BEGIN
		  tmp=Dialog_Message(/Error,Dialog_Parent=$
		  event.top,'Writing operation aborted.')
		  GoTo,out
		ENDIF
	     ENDIF ELSE BEGIN
		tmp=Spec_Save(hspec,set,file,Name=titles[i],/No_Confirm)
	     ENDELSE
	   ENDFOR
	   END

	'SPEC/MCA multiscan file': BEGIN 
           file = Dialog_PickFile(File='exodusMCA.spec',Title= $
		'Select a file for writing (SPEC/MCA format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out
	   IF checkfile(file) EQ 1 THEN BEGIN
             itmp=Dialog_Message(/Question,/Cancel,Dialog_Parent=$
                event.top,['File exists: ',file,'Append data?','',$ 
		'(Yes: data appended)','(No: file rewritten)', $
		'(Cancel: Abort operation)'])
	      CASE itmp OF
	        'Yes':
	        'No': Delete_Files,file
		'Cancel': GoTo,out
	      ENDCASE
	   ENDIF
	   nn = N_Elements(iList)
	   hspec = 0
	   titles = (state.ompp)->value(iList,/Title)
	   FOR i=0L,nn-1 DO BEGIN
;	     set = MPP_Value(state.ptrMPP,iList[i])
	     set = (state.ompp)->value(iList[i])
             setX = Reform(set[0,*])
             xx = FindGen(N_Elements(setX))
             ccc = Poly_Fit(xx,setX,2)
             setY = Reform(set[N_Elements(set[*,0])-1,*])
	     IF i EQ 0 THEN BEGIN
		Widget_Control,/Hourglass
		tmp=Spec_SaveMCA(hspec,setY,file,Name=titles[i],Dialog_Parent=$
			event.top,/No_Confirm,Calib=ccc) 
		IF tmp EQ 0 THEN BEGIN
		  tmp=Dialog_Message(/Error,Dialog_Parent=$
		  event.top,'Writing operation aborted.')
		  GoTo,out
		ENDIF
	     ENDIF ELSE BEGIN
		tmp=Spec_SaveMCA(hspec,setY,file,Name=titles[i],/No_Confirm,$
                   calib=ccc)
	     ENDELSE
	   ENDFOR
	   END



	'XPLOT/SPEC mesh': BEGIN ; Save to a file (SPEC/MESH format)
	   isel=1
	   exodus_interpolate,event,state,isel,yInt=yInt,xInt=xInt, $
		leave_flag=leave_flag
	   IF leave_flag EQ 1 THEN GoTo,out
	   list = (state.ompp)->value(/Title)
           ;
           ; start in XPlot
           ;
           ss = size(yInt) ; exemple 2 2 100 4 200
           xx = Replicate(1,ss[1])#xInt
	   xx = Reform(transpose(xx),ss[4])
	   yy = Reform(transpose(yInt),ss[4])
           ii = reform(Replicate(1,ss[2])#indgen(ss[1]),ss[4])
	   aa = Make_Set(xx,ii,yy)
	   xplot,parent=p,/no_block,aa,colTitles=['x','index','y']
	   ; set mesh display in xplot
           ; this is not very elegant, but it works until the 
           ; procedure xplot_mesh will be available...
           widget_control, p , GET_UVALUE = xplotState
           tmp=xplotState.xpdt.mesh
           flag=tmp.flag
           flag[0]='1'
           tmp.flag=flag
           
           split=tmp.split
           split[0]='1'
           tmp.split=split
           
           tmp.col=2
           xplotState.xpdt.mesh=tmp
           widget_control, p , SET_UVALUE = xplotState
           xplot_controls_action,p,linestyle='0' ; refresh
           
           ;
           ; write file
           ;
           itmp = Dialog_Message(/Question,'Write the SPEC file?',$
		Dialog_Parent=event.top)
	   IF itmp EQ 'No' THEN GoTo,out

           file = Dialog_PickFile(File='exodus.spec',Title= $
		'Select a file for writing (SPEC/MESH format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF
	   OpenW,unit,file,/Get_Lun
           printf,unit,'#F '+file
           printf,unit,'#D '+sysTime()
           printf,unit,'#UINFO File written by xop/exodus containing an interpolated mesh from:'
           FOR i=0,N_Elements(list)-1 DO BEGIN
              printf,unit,'#UINFO      '+list[i]
           ENDFOR
           printf,unit,'#S  1 mesh from exodus data'
           printf,unit,'#N  3'
           printf,unit,'#L X  index  Y'
           FOR i=0,N_Elements(yInt[*,0])-1 DO BEGIN ; columns
             FOR j=1,N_Elements(yInt[0,*])-1 DO BEGIN ; points=rows
	       printf,unit, String([xInt[j],i,yInt[i,j]],Format='(3G14.6)')
	     ENDFOR
	   ENDFOR
           Free_Lun,unit
           Print,'EXODUS: File written to disk: '+file

	   END

	'XPLOT2D/EDF mesh': BEGIN ; 
	   isel=1
	   exodus_interpolate,event,state,isel,yInt=yInt,xInt=xInt, $
		leave_flag=leave_flag
	   IF leave_flag EQ 1 THEN GoTo,out
	   list = (state.ompp)->value(/Title)
           titles = list[iList]
           ;
           ; start in XPlot2d
           ;
           xx = FindGen(N_Elements(xInt))
           ccc = Poly_Fit(xx,xInt,2)
           axis1 = [ccc[0],1.0/ccc[1],0,1] 
           image = transpose(yInt)
           xplot2d,image,wtitle='(spectrum number vs abscissa|channel) %C',$
             axis=axis1
           ;
           ; write file
           ;
           itmp = Dialog_Message(/Question,'Write the EDF file?',$
		Dialog_Parent=event.top)
	   IF itmp EQ 'No' THEN GoTo,out
;
           file = Dialog_PickFile(File='exodus.edf',Title= $
		'Select a file for writing (EDF format)',$
                Filter='*.edf',Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out
;
	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF
           addkw = [ $
               'MCA a = '+StrCompress(ccc[0],/Rem), $
               'MCA b = '+StrCompress(ccc[1],/Rem), $
               'MCA c = '+StrCompress(ccc[2],/Rem), $
               'MCA start ch = 0 ']

	   write_edf,transpose(yInt),file=file,title= $
             'File with data exported by XOP/Exodus', $
             addKw=addKw

	   END

	'XPLOT/SPEC multicolumn': BEGIN ; Spec multicolumn

	   isel=1
	   exodus_interpolate,event,state,isel,yInt=yInt,xInt=xInt, $
		leave_flag=leave_flag
	   IF leave_flag EQ 1 THEN GoTo,out
	   npoints = N_Elements(xInt)
	   nrows = N_Elements(yInt[*,0])
	   list = (state.ompp)->value(/Title)
	   titles=StrCompress(list)

           ;
           ; start in XPlot
           ;
           aa = [reform(xInt,1,N_Elements(xInt)),yInt]
	   xplot,/no_block,aa,colTitles=['x',titles[iList]]
           ;
           ; write file
           ;

           itmp = Dialog_Message(/Question,'Write the SPEC file?',$
		Dialog_Parent=event.top)
	   IF itmp EQ 'No' THEN GoTo,out
           file = Dialog_PickFile(File='exodus.spec',Title= $
		'Select a file for writing (SPEC/Multicolumn format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF
	   OpenW,unit,file,/Get_Lun
           printf,unit,'#F '+file
           printf,unit,'#D '+sysTime()
           printf,unit,'#UINFO File written by xop/exodus containing an interpolated mesh from:'
	   titles2 = 'x'
	   FOR i=0,nrows-1 DO titles2=titles2+'  '+titles[iList[i]]
           printf,unit,'#S  1 interpolated data from exodus'
           printf,unit,'#N  '+StrCompress(nrows+1)
           printf,unit,'#L '+titles2
           FOR i=0,npoints-1 DO BEGIN
	     printf,unit, String([xInt[i],Reform(yInt[*,i])], $
		Format='('+StrCompress(nrows+1,/Rem)+'G14.6)')
	   ENDFOR
           Free_Lun,unit
           Print,'EXODUS: File written to disk: '+file

	   END
	'EXCEL-XML multisheet file': BEGIN ; EXCEL-XML Multisheet

           file = Dialog_PickFile(File='exodus.xls',Title= $
		'Select a file for writing (EXCEL-XLM format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF
	   OpenW,unit,file,/Get_Lun
	   xmlexcel_workbook,unit,/open

	   nn = N_Elements(iList)
	   titles = (state.ompp)->value(iList,/Title)
	   Widget_Control,/HourGlass
	   FOR i=0L,nn-1 DO BEGIN
	     xmlexcel_worksheet,unit,/open,name=StrCompress(i,/Rem)
	     xmlexcel_putline,unit,titles[i]
	     set = (state.ompp)->value(iList[i])
	     FOR j=0L,n_elements(set[0,*])-1 DO BEGIN
		xmlexcel_putline,unit,set[*,j]
	     ENDFOR
	     xmlexcel_worksheet,unit,/close
	   ENDFOR
	   xmlexcel_workbook,unit,/close
           Free_Lun,unit
           Print,'EXODUS: File written to disk: '+file
	   END
	'EXCEL-XML multicolumn file': BEGIN ; EXCEL-XML Multicolumn
	   isel=1
	   exodus_interpolate,event,state,isel,yInt=yInt,xInt=xInt, $
		leave_flag=leave_flag
	   IF leave_flag EQ 1 THEN GoTo,out

           file = Dialog_PickFile(File='exodus_interpolated.xls',Title= $
		'Select a file for writing (EXCEL-XLM format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF
	   OpenW,unit,file,/Get_Lun
	   xmlexcel_workbook,unit,/open
	   xmlexcel_worksheet,unit,/open
	   nn = N_Elements(iList)
	   titles = (state.ompp)->value(iList,/Title)
	   Widget_Control,/HourGlass
	   npoints=N_Elements(xInt)
	   nrows=1L+N_Elements(yInt[*,0])
	   xmlexcel_putline,unit,['x',titles]
	   FOR j=0L,npoints-1 DO BEGIN
	      xmlexcel_putline,unit,[xInt[j],Reform(yInt[*,j])]
	   ENDFOR

	   xmlexcel_worksheet,unit,/close
	   xmlexcel_workbook,unit,/close
           Free_Lun,unit
           Print,'EXODUS: File written to disk: '+file
	   END


	'EXCEL-CSV multicolumn file': BEGIN ; EXCEL-CSV Multicolumn
	   isel=1
	   exodus_interpolate,event,state,isel,yInt=yInt,xInt=xInt, $
		leave_flag=leave_flag
	   IF leave_flag EQ 1 THEN GoTo,out

           file = Dialog_PickFile(File='exodus_interpolated.csv',Title= $
		'Select a file for writing (EXCEL-CSV Comma Separated Values format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF
	   Widget_Control,/HourGlass
	   npoints=N_Elements(xInt)
	   nrows=1L+N_Elements(yInt[*,0])
	   OpenW,unit,file,/Get_Lun
	   titles = (state.ompp)->value(iList,/Title)
           stmp=''
           FOR i=0,N_Elements(titles)-1 DO stmp=stmp+','+titles[i]
	   printf,unit, stmp
           FOR i=0,npoints-1 DO BEGIN
             stmp= String(xInt[i],Format='(G14.6)')+','
             FOR j=0,nrows-2 DO stmp= stmp+String(yInt[j,i],Format='(G14.6)')+','
	     printf,unit, stmp
           ENDFOR

           Free_Lun,unit
           Print,'EXODUS: File written to disk: '+file
           IF sdep() EQ 'WINDOWS' THEN BEGIN
              command = 'start '+dospath(file)
           ENDIF ELSE BEGIN
              command = 'soffice '+file
           ENDELSE

           itmp = Dialog_Message(['EXODUS: File written to disk: '+file,$
                  '  ' , 'Open with native application, i.e., spawn:',$
                  command,'',''],Dialog_Parent=event.top,/Question)
           IF itmp EQ 'Yes' THEN spawn,command

	   END

	'Application...': BEGIN ; other applications
	   set = (state.ompp)->value(iList[0])
	   x=set[0,*]
	   y=set[1,*]
	   txt = 'Xplot,x,y,/No_Block'
	   XEdit,txt,InfoText= $
		['Type the command that will accept the data','  Use: ',$
		 'appl,x,y [,Keywords]','  or','appl,set [,Keywords] '],$
		Title='Export to application',text='Command: ',$
		Action=action,XSize=100, Dialog_Parent=event.top
	   IF action EQ 'CANCEL' THEN GoTo,out
	   itmp = Execute(txt)	
	   END

	'FuFiFa (fitting)': BEGIN ; FuFiFa
           IF Widget_Info(state.fufifa,/Valid_Id) THEN BEGIN
		p=state.fufifa
           ENDIF ELSE BEGIN
		fufifa,p=p
		state.fufifa=p
           ENDELSE

	   nn = N_Elements(iList)
	   titles = (state.ompp)->value(iList,/Title)
	   FOR i=0L,nn-1 DO BEGIN
	     set = (state.ompp)->value(iList[i])
             fufifa_loadfile,p,isel=2,data=set,title=titles[i],/append
	   ENDFOR
           fufifa_list_select,p,-1
           fufifa_plot,p
	   END
	'XAID/mu2chi': BEGIN ; mu2chi
	   nn = N_Elements(iList)

	   FOR i=0L,nn-1 DO BEGIN
	     set = (state.ompp)->value(iList[i])
	     title = (state.ompp)->value(iList[i],/Title)
             p = state.xaid_mu2chi
             IF Widget_Info(p,/Valid_Id) THEN BEGIN
	       XAID_mu2chi_loadfile,p,set,plot=0,Message= $
               'Imported from exodus: '+title
             ENDIF ELSE BEGIN
	       XAID_mu2chi,set,/No_Block,parent=p
               state.xaid_mu2chi=p
               ; state updated after the CASE block...
             ENDELSE
	     XAID_mu2chi_plot,p,xx=xx,yy=yy,xtitle=xtitle,ytitle=ytitle 
             ; add exodus set with calculated data
             exodus_loadfile,stateid,isel=0,data=Make_Set(xx,yy), $
               Title=ytitle+' vs '+xtitle+' | '+title
           ENDFOR
	   END
	'XAID/ff': BEGIN ; ff
	   ;set = (state.ompp)->value(iList[0])
	   ;XAID_ff,set,/No_Block

	   nn = N_Elements(iList)

	   FOR i=0L,nn-1 DO BEGIN
	     set =   (state.ompp)->value(iList[i])
	     title = (state.ompp)->value(iList[i],/Title)
             p = state.xaid_ff
             IF Widget_Info(p,/Valid_Id) THEN BEGIN
	       XAID_ff_loadfile,p,set ; ,plot=0,Message= $
               ; 'Imported from exodus: '+title
             ENDIF ELSE BEGIN
	       XAID_ff,set,/No_Block,parent=p
               state.xaid_ff=p
               ; state updated after the CASE block...
             ENDELSE
	     XAID_ff_plot,p,ft=setf
             exodus_loadfile,stateid,isel=0,data=setf,Title='FT of '+title
           ENDFOR

	   END
	'DELIA binary file': BEGIN ; Save to a file (DELIA binary format)

           file = Dialog_PickFile(File='exodus.delia',Title= $
		'Select a file for writing (DELIA binary format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF

	   Widget_Control,/Hourglass
	   OpenW,unit,file,/Get_Lun,/Swap_If_Little_Endian

	   nn = N_Elements(iList)
	   FOR i=0L,nn-1 DO BEGIN
;	     set = MPP_Value(state.ptrMPP,iList[i])
	     set = (state.ompp)->value(iList[i])
	     set1=FltArr(3,N_Elements(set[0,*]))
	     set1[1:2,*]=set
	     ii = Float(i)
	     set1[0,*]=ii
	     WriteU,unit,set1
	   ENDFOR
	   Free_Lun,unit
	   END
	else:
	ENDCASE
	END
    'DELETEALL': BEGIN
	itmp = Dialog_Message(/Question, $
	  ['This option deletes all data in the right panel', $
	   'Please confirm'],Dialog_Parent=event.top)
	IF itmp EQ 'No' THEN GoTo,out
;	MPP_Free,state.ptrMPP
	(state.ompp)->remove
	Widget_Control,state.wids.PRList,Set_Value=''
	END
    'DELETE': BEGIN
	iList = Widget_Info(state.wids.PRList,/List_Select)
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    'Please select a data line')
	  GoTo,out
	ENDIF
	(state.ompp)->remove,iList
	IF (state.ompp)->info(/N_Good_Elements) EQ 0 THEN $
	  list = '' ELSE list = (state.ompp)->value(/Title)
	Widget_Control,state.wids.PRList,Set_Value=list
	END
    'GETXPLOT': BEGIN
          IF Widget_Info(state.xplot,/Valid_Id) EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	     'No active xplot window created from Exodus')
	    Goto,out
	  ENDIF
          Widget_Control,/HourGlass
          Xplot_GetCurrentData,state.xplot,data
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title='Imported from xplot'
          state.ompp = ptr
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)

	END
    'OPERATIONS': BEGIN
        ;isel = Widget_Info(event.id,/DropList_Select)
	;Widget_Control,event.id,Set_DropList_Select=0
	Widget_Control,event.id, Get_Value=val
        ir_label=''
	CASE val OF
	  'Interpolate': isel=1
	  'Average': BEGIN
             isel=2
             group_nn=-1
             END
	  'Average every n sets': BEGIN
             isel=2
             group_nn=2 
	     XEdit,group_nn,InfoText=['Average every n sets.'],$
		Title='Average every n sets',text='n= ',$
		Action=action,Dialog_Parent=event.top
	     IF action EQ 'CANCEL' THEN GoTo,out
             END
	  'Sum': isel=3
	  'Customized': isel=4
	  'Fit LinCom': isel=5
	  'Customized operation': isel=6
	  'XANES Normalization': isel=7
	  'Append': isel=31
          'Merge two blocks...': isel=32
          'Parameters from blocks...': isel=33
	  else: isel=0
	ENDCASE


	iList = Widget_Info(state.wids.PRList,/List_Select)
	mpp = state.ompp
	; 
	; Perform requested operation
	;
	CASE isel OF
	0: 
	1: BEGIN  ; Interpolation
	   exodus_interpolate,event,state,isel,xInt=xInt,yInt=yInt, $
	     leave_flag=leave_flag,ir_label=ir_label
	   IF leave_flag EQ 1 THEN GoTo,out
	   nn = N_Elements(yInt[*,0])
	   nxInt = N_Elements(xInt)
	   IF N_Elements(stroper_interp) EQ 0 THEN BEGIN
	     stroper_interp = {output:['0','Add to List','Xplot 2D (multicolumn)',$
		'Xplot 3D view']}
	   ENDIF
	   XScrMenu,stroper_interp,Titles=['Output to:'],/NoType,/Interp,$
	     Dialog_Parent=event.top,Action=action,WTitle=$
		'Destination of interpolated data'
		
	   IF action EQ 'DONT' THEN GoTo,out
           Widget_Control,/HourGlass
	   CASE StrCompress(stroper_interp.output[0],/Rem) OF
	   '0': BEGIN ; Add to list
		FOR i=0,nn-1 DO BEGIN
		  title = mpp->value(iList[i],/Title)
	          data = Make_Set(xInt,Reform(yInt[i,*]))
		  mpp->set,Value=data,Title=ir_label+title,/OverAdd
		ENDFOR
	        state.ompp = mpp
                list = (state.ompp)->value(/Title)
                Widget_Control,state.wids.PRList,Set_Value=list,$
	          Set_List_Select=iList
		END
	   '1': BEGIN ; Xplot 2D
		yout = DblArr(nn+1,N_Elements(xint))
		labels = StrArr(nn+1)
		yout[0,*]=xint
		labels[0] = 'Interpolated abscissas'
		FOR i=1,nn DO BEGIN
		  labels[i] = mpp->value(iList[i-1],/Title)
	          yout[i,*] = Reform(yInt[i-1,*])
		ENDFOR
		XPlot,yout,No_Block=state.No_Block,ColTitles=labels,$
		 Wtitle='EXODUS: multicolumn 2D view'
		END
	   '2': BEGIN ; Xplot 3D
		npts = N_Elements(xint)
		yout = DblArr(3,npts*nn)
		FOR i=0,nn-1 DO BEGIN
	          yout[0,npts*i:(npts*i)+npts-1] = xint
	          yout[1,npts*i:(npts*i)+npts-1] = i
	          yout[2,npts*i:(npts*i)+npts-1] = Reform(yInt[i,*])
		ENDFOR
		p=0
		XPlot,yout,Parent=p,XCol=1,YCol=3,Wtitle='EXODUS: 3D view',$
		  /No_Block
		XPlot_Mesh,p,Flag=1,Kind=0,Col=2
		END
	   else: 
	   ENDCASE
	   END
	2: BEGIN  ; Average
	   exodus_interpolate,event,state,isel,xInt=xInt,yInt=yInt, $
	     leave_flag=leave_flag,ir_label=ir_label
	   IF leave_flag EQ 1 THEN GoTo,out
	   nn = N_Elements(yInt[*,0])
	   nxInt = N_Elements(xInt)
	   xOut = xInt


           IF group_nn LE 0 THEN group_nn = nn
           group_n = ceil(float(nn)/group_nn)
           FOR j=0,group_n-1 DO BEGIN
	     title = 'Average sets: '
             group_start = j*group_nn>0
             group_end = (group_start+ group_nn-1)<(nn-1)
             ;print,'<><> group, start, end: ',j,group_start,group_end

             ; average sets
	     yOut = FltArr(nxInt)
	     errOut = FltArr(nxInt)
	     FOR i=group_start,group_end DO BEGIN
		;print,'Index, values: ',i,yInt[i,*]
		yOut=yOut + Reform(yInt[i,*])
	     ENDFOR
	     yOut = yOut/Float(group_end-group_start+1)
	     FOR i=group_start,group_end DO BEGIN
		;print,'Index, values: ',i,yInt[i,*]
		errOut=errOut + (Reform(yInt[i,*])-yOut)^2
	     ENDFOR
             errOut = sqrt(errOut/Float(group_end-group_start+1))

	     data = Make_Set(xOut,yOut,errOut)

             FOR i=group_start,group_end DO BEGIN
               tmp = mpp->value(iList[i],/Title)
               tmp_pos = StrPos(tmp,'-')
               IF tmp_pos LE 0 THEN tmp_pos = 6
               title = title + StrMid(tmp,0,tmp_pos) +'; '
             ENDFOR

	     mpp->set,Value=data,Title=title, /OverAdd
	     state.ompp = mpp
             list = (state.ompp)->value(/Title)
             Widget_Control,state.wids.PRList,Set_Value=list,$
	          Set_List_Select=iList
             ENDFOR
	   END
	3: BEGIN  ; Sum
	   exodus_interpolate,event,state,isel,xInt=xInt,yInt=yInt, $
	     leave_flag=leave_flag,ir_label=ir_label
	   IF leave_flag EQ 1 THEN GoTo,out
	   nn = N_Elements(yInt[*,0])
	   nxInt = N_Elements(xInt)
	   xOut = xInt
	   yOut = FltArr(nxInt)
	   FOR i=0,nn-1 DO yOut=yOut + Reform(yInt[i,*])
	   yOut = yOut
	   data = Make_Set(xOut,yOut)

	   title = 'Sum sets: '
           FOR i=0,nn-1 DO BEGIN
             tmp = mpp->value(iList[i],/Title)
             tmp_pos = StrPos(tmp,']')
             IF tmp_pos LE 0 THEN tmp_pos = 6
             title = title + StrMid(tmp,0,tmp_pos+1) +'; '
           ENDFOR


	   mpp->set,Value=data,Title=title, /OverAdd
	   state.ompp = mpp
           list = (state.ompp)->value(/Title)
           Widget_Control,state.wids.PRList,Set_Value=list,$
	          Set_List_Select=iList
	   END
	4: BEGIN  ; Customized
	   exodus_interpolate,event,state,isel,xInt=xInt,yInt=yInt, $
	     leave_flag=leave_flag,ir_label=ir_label
	   IF leave_flag EQ 1 THEN GoTo,out
	   nn = N_Elements(yInt[*,0])
	   nxInt = N_Elements(xInt)
	   xOut = xInt
	   yOut = DblArr(nxInt)
	   buffer = [';','; Define your exit ordinates as yOut',$
	      '; nn is the number of data lines you selected', $
	      '; xInt (yInt) are the interpolated abscissas (ordinates)',$
	      '; As an example, the following code sums all selected lines',$
	      '; ',$
	      'FOR i=0,nn-1 DO yOut=yOut + Reform(yInt[i,*]) ']
	   XDisplayFile1,Text=buffer,Title='Customized operation',$
	     /Modal,Action=action, Dialog_Parent=event.top
	   IF action EQ 'DONT' THEN GoTo,out
	   FOR iii=0,N_Elements(buffer)-1 DO BEGIN
	     itmp = Execute(buffer[iii])
	     IF itmp NE 1 THEN BEGIN
	       jtmp = Dialog_Message(/Question,Dialog_Parent=event.top,$
	       ['Error executing IDL command:'+command[i],'Continue?'])
	       if jtmp EQ 'No' then GoTo,out
	     ENDIF
	   ENDFOR
	   txt = 'Result of customized operation'
	   XEdit,txt,InfoText=['Type a label for the new data '],$
		Title='Store result of operation',text='Label: ',$
		Action=action,Format='(a30)',Dialog_Parent=event.top
	   IF action EQ 'CANCEL' THEN GoTo,out
	   tmp = Make_Set(xOut,yOut)
	   mpp->set,Value=tmp,Title=txt,/OverAdd
	   state.ompp = mpp
           list = (state.ompp)->value(/Title)
           Widget_Control,state.wids.PRList,Set_Value=list, $
	     Set_List_Select=iList
	   END

	5: BEGIN  ; Fit LinCom
	   exodus_interpolate,event,state,isel,xInt=xInt,yInt=yInt, $
	     leave_flag=leave_flag,ir_label=ir_label
	   IF leave_flag EQ 1 THEN GoTo,out
	   nn = N_Elements(yInt[*,0])
	   nxInt = N_Elements(xInt)

	   END

	6: BEGIN  ; Customized for each set
	   IF N_Elements(stroper_cust) EQ 0 THEN BEGIN
	     stroper_cust = {output:['0','Add new sets to List','Substitute current sets'],$
		xFrom:0.0D0,xTo:0.0D0, $
		commandX:'x',command:'y',commandErr:''}
	   ENDIF

	   XScrMenu,stroper_cust,Titles=$
             ['Output to:', $
              'Use only points with x FROM:','TO [Default: set as FROM]',$
              'Command xNew=f(x,y)','Command yNew=f(x,y)',$
              'Command for error column errNew=f(err,x,y)' ],/NoType,/Interp,$
	     Dialog_Parent=event.top,Action=action,WTitle=$
		'Customized operations on individual sets',FieldLen=50
	   IF action EQ 'DONT' THEN GoTo,out
           Widget_Control,/HourGlass

	   ioutput = Fix(stroper_cust.output[0])
	   nn = N_Elements(iList)
	   FOR i=0,nn-1 DO BEGIN
		  title = mpp->value(iList[i],/Title)
	          data = mpp->value(iList[i])
                  IF (stroper_cust.xFrom NE stroper_cust.xTo) THEN BEGIN
                    cutset,data,data2,cutcol=0, $
                     xrange=[stroper_cust.xFrom,stroper_cust.xTo]
                    data=data2
                  ENDIF
		  x = Reform(data[0,*])
		  y = Reform(data[1,*])
		  IF N_Elements(data[*,0]) GT 2 THEN err = Reform(data[2,*])
		  
		  xnew=0
                  commandX = stroper_cust.commandX
                  IF StrUpCase (strMid(StrCompress(commandX,/Rem),0,5)) NE $
                        'XNEW='  THEN commandX = 'xnew='+commandX
		  itmp = Execute(commandX)
		  IF itmp NE 1 THEN BEGIN
			itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
			'Expression not understood: '+commandX)
			RETURN
		  ENDIF
                  x = xNew
		  ynew=0
                  commandY = stroper_cust.command
                  IF StrUpCase (strMid(StrCompress(commandY,/Rem),0,5)) NE $
                        'YNEW='  THEN commandY = 'ynew='+commandY
		  itmp = Execute(commandY)
		  IF itmp NE 1 THEN BEGIN
			itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
			'Expression not understood: '+commandY)
			RETURN
		  ENDIF
		  newdata = data
                  newData[0,*]=x
                  newData[1,*]=ynew
                  IF strCompress(stroper_cust.commandErr,/Rem) NE '' THEN BEGIN
                     commandErr=stroper_cust.commandErr
                     IF StrUpCase(strMid(StrCompress(commandErr,/Rem),0,7)) NE $
                        'ERRNEW='  THEN commandErr = 'errNew='+commandErr
		     itmp = Execute(commandErr)
		     IF itmp NE 1 THEN BEGIN
			itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
			'Expression not understood: '+commandErr)
			RETURN
		     ENDIF
                     IF N_Elements(newData[*,0]) GE 3 THEN BEGIN
                       newData[2,*]=errNew
                     ENDIF ELSE BEGIN
                       newData=Make_Set(x,y,errNew)
                     ENDELSE
                     
                  ENDIF 
		  ; newdata = Make_Set(x,ynew)
		  IF ioutput EQ 1 THEN BEGIN
		    mpp->set,iList[i],Value=newdata,Title=title+$
                    ' ; xnew='+commandX+' ynew='+commandY
		  ENDIF ELSE BEGIN
		    mpp->set,Value=newdata,Title=title+ $
                    ' ; xnew='+commandX+' ynew='+commandY,/OverAdd
		  ENDELSE
	   ENDFOR

	   state.ompp = mpp
           list = (state.ompp)->value(/Title)
           Widget_Control,state.wids.PRList,Set_Value=list, $
	     Set_List_Select=iList
	   END



	7: BEGIN  ; Xanes normalizetion
	   IF N_Elements(stroper_xanes_normalize) EQ 0 THEN BEGIN
	     stroper_xanes_normalize = { $
                output:['0','Add new sets to List','Substitute current sets'],$
		method:['1','Constant Jump','Polynomial fit'],$
                jump:0.0, degree:3,$
                e0:0.0, eMin:0.0, eMax:0.0 }
                
	   ENDIF
;
	   XScrMenu,stroper_xanes_normalize,Titles=$
             ['Output to:', 'PostEdge method:','Jump [0=Default]: ','Degree',$
              'Edge Eo [Default=0]: ', $
              'start interval [Default=0]: ','end interval [Default=0]: '], $
              Flags=['1','1','w(1) EQ 0','w(1) EQ 1','1','1','1'],$
              /NoType,/Interp,$
	     Dialog_Parent=event.top,Action=action,WTitle=$
		'Xanes Normalization',FieldLen=50
	   IF action EQ 'DONT' THEN GoTo,out
           Widget_Control,/HourGlass

	   ioutput = Fix(stroper_xanes_normalize.output[0])
	   nn = N_Elements(iList)
	   FOR i=0,nn-1 DO BEGIN
		  title = mpp->value(iList[i],/Title)
	          data = mpp->value(iList[i])
                  method=Fix(stroper_xanes_normalize.method[0])
                  degree=stroper_xanes_normalize.degree
                  jump=stroper_xanes_normalize.jump
                  e0=stroper_xanes_normalize.e0
                  eMin=stroper_xanes_normalize.eMin
                  eMax=stroper_xanes_normalize.eMax

                  newData = Xanes_Normalize(data,method=method,/verbose,$
                    degree=degree,e0=e0,jump=jump,eStart=emin,eEnd=emax,txt=txt)
                
		  IF ioutput EQ 1 THEN BEGIN
		    mpp->set,iList[i],Value=newdata,Title=title+$
                    ' ; '+txt
		  ENDIF ELSE BEGIN
		     mpp->set,Value=newdata,Title=title+ $
                    ' ; '+txt,/overAdd
		  ENDELSE
	   ENDFOR

	   state.ompp = mpp
           list = (state.ompp)->value(/Title)
           Widget_Control,state.wids.PRList,Set_Value=list, $
	     Set_List_Select=iList
	   END



	31: BEGIN  ; Append
	   nn = N_Elements(iList)
	   IF nn LT 2 THEN BEGIN
	      itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	      'Select at least two data sets ')
	      RETURN
           ENDIF
	   FOR i=0,nn-1 DO BEGIN
		  title = mpp->value(iList[i],/Title)
	          data = mpp->value(iList[i])
		  xi = Reform(data[0,*])
		  yi = Reform(data[1,*])
		  
		  IF i EQ 0 THEN BEGIN
		   x = xi
		   y = yi
                  ENDIF ELSE BEGIN
                   x=[x,xi]
                   y=[y,yi]
                  ENDELSE
	   ENDFOR

	   isort = sort(x)
           iUniq = Uniq(x,iSort)
	   xnew = x[iUniq]
	   IF N_Elements(xnew) NE N_Elements(x) THEN BEGIN ; duplicated values
	      itmp = Dialog_Message(/Question,/Cancel,Dialog_Parent=event.top, $
	      ['Some x points overlap (duplicated abscissas)',$
               'Sort and average ordinates? ','', $
               'Yes: sort and average', $
               'No: sort and use last value', $
               'Cancel: leave as they are'])
	       IF itmp NE 'Cancel' THEN BEGIN
		   tmp = LIndGen(N_Elements(x))
		   tmp[iUniq]=-1
		   xRedundant = x[Where(tmp NE -1)]
		   xRedUniq = xRedundant[ Uniq(xRedundant,Sort(xRedundant))]
		   print,'-- Redundant points: '
		   FOR i=0L,N_Elements(xRedUniq)-1 DO BEGIN
			tmp2 = where(x EQ xRedUniq[i])

		;     FOR j=0L,N_Elements(tmp2)-1 DO BEGIN
		;	  print,x[tmp2[j]],y[tmp2[j]]
		;     ENDFOR
		     yAv = Mean(y[tmp2])
		     IF itmp EQ 'Yes' THEN y[tmp2] = yAv 
		;     FOR j=0L,N_Elements(tmp2)-1 DO BEGIN
		;	    print,'     ',x[tmp2[j]],y[tmp2[j]]
		;     ENDFOR
                   ENDFOR
	           print,'--'
		   x = xNew
		   y = y[iUniq]
		      
		   END

	   ENDIF ELSE BEGIN ; no duplicated values, ask for sorting
             IF Total ( (isort-LIndGen(N_Elements(x)) )^2 ) NE 0 THEN BEGIN
	        itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
	        'Sort appended data? ')
	        IF itmp EQ 'Yes' THEN BEGIN
		  x = x[isort]
		  y = y[isort]
	        ENDIF
             ENDIF
           ENDELSE


	   newdata = Make_Set(x,y)
	   title = 'Appended sets: '
           FOR i=0,nn-1 DO BEGIN
             tmp = mpp->value(iList[i],/Title)
             tmp_pos = StrPos(tmp,']')
             IF tmp_pos LE 0 THEN tmp_pos = 6
             title = title + StrMid(tmp,0,tmp_pos+1) +'; '
           ENDFOR
           mpp->set,Value=newdata,Title=title,/OverAdd
;
	   state.ompp = mpp
           list = (state.ompp)->value(/Title)
           Widget_Control,state.wids.PRList,Set_Value=list, $
	     Set_List_Select=iList

	   END

	32: BEGIN  ; Merge
           ;
	   nn = N_Elements(iList)
	   IF (nn mod 2) NE 0 THEN BEGIN
	      itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	      'Select en EVEN number of lines (datasets) ')
	      RETURN
           ENDIF

           txt = ['Given a selection of 2N lines (datasets), this option',$
                  'permits to combine, one to one, the first N datasets',$
                  'with the last N datasets, resulting in N new datasets.',$
                  '  ','Continue?']
           itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
                  txt,Title='Merge two blocks: info')
           IF itmp EQ 'No' THEN RETURN


           ;stroper_merge = {x1:0D, x2:0D, deltaX:0D, $
           ;        combine:['0','by addition','by Y shift','None (just append)'], $
           ;        x11:0D, x22:0D}

           ;iInterpolation = 0
           ; need interpolation? And compute defaults...

	   FOR iii=0,(nn/2)-1 DO BEGIN
	          data1 = mpp->value(iList[iii])
	          data2 = mpp->value(iList[iii+(nn/2)])
		  x1 = Reform(data1[0,*])
		  x2 = Reform(data2[0,*])
                  ;
                  ; check if abscissas overlap
                  ;
                  x1_min = min(x1,max=x1_max)
                  x2_min = min(x2,max=x2_max)
                  ; overlapping region xx:
                  xx_min = max([x1_min,x2_min])
                  xx_max = min([x1_max,x2_max])

	          deltaX = x1[1]-x1[0]

                  ;IF max([x1_min,x2_min]) LE min([x1_max,x2_max]) THEN iInterpolation=1

           ENDFOR

           IF N_Elements(stroper_merge) EQ 0 THEN BEGIN
             stroper_merge = { $
                 combine:['0','None (just append)','Interpolate and sum', $
                 'Interpolate and match by Y shift',$
                 'Interpolate and match by Y scaling'],$
                 x1:min([x1,x2], max=tmp),x2:tmp, deltaX:x1[1]-x1[0], $
                 x11:xx_min, x22:xx_Max}
           ENDIF
	   action = 0
	   Titles=['Interpolation and matching method', $
                 'Interpolate from','Interpolate to','Interpolation step', $
                 'Match from','Match to']
           ;flags = ['w(3) NE 2','w(3) NE 2','w(3) NE 2','1','w(3) EQ 1','w(3) EQ 1']
           flags = ['1','w(0) GT 0','w(0) GT 0','w(0) GT 0','w(0) GE 2','w(0) GE 2']
	   XScrMenu,stroper_merge,/NoType,/Inter,Titles=titles,flags = flags,$
	        Dialog_Parent=event.top,WTitle='Interpolation parameters',$
	        Action=action
	   IF action EQ 'DONT' THEN RETURN
           Widget_Control,/HourGlass

           iInterpolation = Fix( (stroper_merge.combine)[0] )

           IF iInterpolation GE 1 THEN BEGIN
	      nPts = ((stroper_merge.x2-stroper_merge.x1)/stroper_merge.deltaX)+1
	      xInt = Abscissas(stroper_merge.x1,stroper_merge.x2,nPts)
           ENDIF
           
	   FOR iii=0,(nn/2)-1 DO BEGIN
		  title1 = mpp->value(iList[iii],/Title)
	          data1 = mpp->value(iList[iii])
		  title2 = mpp->value(iList[iii+(nn/2)],/Title)
	          data2 = mpp->value(iList[iii+(nn/2)])
		  x1 = Reform(data1[0,*])
		  y1 = Reform(data1[1,*])
		  x2 = Reform(data2[0,*])
		  y2 = Reform(data2[1,*])

                  IF iInterpolation GE 1 THEN BEGIN
                     yInt1 = Interpolate(y1,FIndex(x1,xInt), Missing=0)
                     yInt2 = Interpolate(y2,FIndex(x2,xInt), Missing=0)
                     ;
                     x=xInt
                     CASE iInterpolation OF
                        1: y=yInt1+yInt2
                        2: BEGIN
                           i1 = Where((x1 GE stroper_merge.x11) AND (x1 LE stroper_merge.x22))
                           i2 = Where((x2 GE stroper_merge.x11) AND (x2 LE stroper_merge.x22))
                           IF i1[0] NE -1 THEN y11 = mean(y1[i1]) ELSE y11=0
                           IF i2[0] NE -1 THEN y22 = mean(y2[i2]) ELSE y22=0
                           y=yInt1
                           IF min(x1) LE min(x2) THEN BEGIN
                             igood = where(xInt GE stroper_merge.x11)
                           ENDIF ELSE BEGIN
                             igood = where(xInt LE stroper_merge.x22)
                           ENDELSE
                           IF igood[0] NE -1 THEN y[igood] = yInt2[igood]+(y11-y22)
                           END
                        3: BEGIN
                           i1 = Where((x1 GE stroper_merge.x11) AND (x1 LE stroper_merge.x22))
                           i2 = Where((x2 GE stroper_merge.x11) AND (x2 LE stroper_merge.x22))
                           IF i1[0] NE -1 THEN y11 = mean(y1[i1]) ELSE y11=0
                           IF i2[0] NE -1 THEN y22 = mean(y2[i2]) ELSE y22=0
                           y=yInt1
                           IF min(x1) LE min(x2) THEN BEGIN
                             igood = where(xInt GE stroper_merge.x11)
                           ENDIF ELSE BEGIN
                             igood = where(xInt LE stroper_merge.x22)
                           ENDELSE
                           IF igood[0] NE -1 THEN y[igood] = yInt2[igood]/y22*y11
                           END

                        else: BEGIN  ; just in case 
                           x=[x1,x2]
                           y=[y1,y2]
                           ENDELSE
                     ENDCASE
                  ENDIF ELSE BEGIN
                     ;
                     ; merge
                     ;
                     x=[x1,x2]
                     y=[y1,y2]
                  ENDELSE
;
              	  newdata = Make_Set(x,y)
	          title = 'Merged sets: '+title1+' AND '+title2
                  mpp->set,Value=newdata,Title=title,/OverAdd
         
;;
	          state.ompp = mpp
                  list = (state.ompp)->value(/Title)
                  Widget_Control,state.wids.PRList,Set_Value=list, $
	            Set_List_Select=iList
	          ENDFOR

	   END


	33: BEGIN  ; Parametric
           ;
	   nn = N_Elements(iList)

           txt = ['Given a selection of N lines (datasets), this option',$
                  'permits to calculate a parameter from each set.',$
                  'The result will have as abscissas the index of ',$
                  'the data set (0,...,N-1), and as ordinates the ',$
                  'calculated parameter.',$
                  '  ','Continue?']
           itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
                  txt,Title='Parameter from blocks: info')
           IF itmp EQ 'No' THEN RETURN


;	   FOR iii=0,(nn)-1 DO BEGIN
;	          data1 = mpp->value(iList[iii])
;		  x1 = Reform(data1[0,*])
;           ENDFOR

           IF N_Elements(stroper_parametric) EQ 0 THEN BEGIN
             stroper_parametric = { command:'out=(moment(y))[1]',$
                      xMin:0.0, xMax:0.0 }
           ENDIF
	   action = 0
	   Titles=['Command to generate parameter', $
                 'Use only points with x FROM:',$
                 '   TO [Default: set as FROM]']
 
           flags = ['1','1','1']
help,stroper_merge,titles,flags
	   XScrMenu,stroper_parametric,/NoType,/Inter,Titles=titles,flags = flags,$
	        Dialog_Parent=event.top,WTitle='Interpolation parameters',$
	        Action=action
	   IF action EQ 'DONT' THEN RETURN
           Widget_Control,/HourGlass

           ;tmpOut=FltArr(nn)

	          FOR iii=0,(nn)-1 DO BEGIN
		  title1 = mpp->value(iList[iii],/Title)
	          data1 = mpp->value(iList[iii])
		  x = Reform(data1[0,*])
		  y = Reform(data1[1,*])
                  IF N_Elements(data1[*,0]) GE 3 THEN err = Reform(data1[2,*])

                  IF (stroper_parametric.xmin NE stroper_parametric.xmax) THEN BEGIN
                    igood = Where( (x GE stroper_parametric.xmin) AND $
                                   (x LE stroper_parametric.xmax) )
                    IF igood[0] EQ -1 THEN BEGIN
			itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $ 
                        ['No good points for dataset '+StrCompress(iii,/Rem),$
                         'in the interval '+$
                          vect2String([stroper_parametric.xmin,stroper_parametric.xmax]), $
                         '** using all defined points instead **'],/Cancel)
                        IF itmp EQ 'Cancel' THEN RETURN
                    ENDIF
                    x = x[igood]
                    y = y[igood]
                    IF N_Elements(err) GT 0 THEN err=err[igood]
                  ENDIF

                  out=0
                  itmp = Execute((stroper_parametric.command)[0])

		  IF itmp NE 1 THEN BEGIN
			itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
			'Expression not understood: '+stroper_parametric.command)
			RETURN
		  ENDIF
           
                  IF iii EQ 0 THEN BEGIN
                    nOut = N_Elements(out)
                    tmpOut = FltArr(nOut+1,nn)
                  ENDIF 
                  tmpOut[0,iii] = iii
                  tmpOut[1:nOut,iii] = out
                  ENDFOR

;
              	  
	          title = 'Parameter from sets. '+(stroper_parametric.command)[0]
                  mpp->set,Value=tmpOut,Title=title,/OverAdd
         
;;
	          state.ompp = mpp
                  list = (state.ompp)->value(/Title)
                  Widget_Control,state.wids.PRList,Set_Value=list, $
	            Set_List_Select=iList

	   END


	else: BEGIN 
	   END
	ENDCASE

	  

	END
    'XPLOT_RIGHT': BEGIN
	iList = Widget_Info(state.wids.PRList,/List_Select)
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    'Please select a data line')
	  GoTo,out
	ENDIF
	IF N_Elements(iList) GT 1 THEN BEGIN ; multiple plot
	  Widget_Control,/HourGlass
	  ompp = state.ompp
	  IF N_Elements(xwindowParms) EQ 0 THEN $
	    xwindowParms= $
	    {xMin:0.0,xMax:0.0,yMin:0.0,yMax:0.0,Legend:['1','No','Yes'],$
	     shiftValue:0.0}
	  XWindow,Buffer=$
	   [';','; It is NOT recommended  to change this line',$
	    '; Use the "Edit Parameters" button to customize',';',$
           "(data.mpp)->mplot,data.list,xRange=[parms.xMin,parms.xMax],xstyle=17,ystyle=17"+$
	   ",yRange=[parms.yMin,parms.yMax],Legend=Fix(parms.legend[0])"+$
	   ",shiftValue=parms.shiftValue,XLog="+$
              (state.pref.parameters.xlog)[0]+",YLog="+$
              (state.pref.parameters.ylog)[0] ],$
           Edit=1, Group=event.top,WTitle='EXODUS: Multiple plot', $
           Data={mpp:ompp,list:iList}, $
	   Parms=xwindowParms,$
	   Titles=['X Min (Set Min=Max for defaults)','X Max', $
		'Y Min','Y Max','Legend',$
		'Shift spectrum factor (0<f<1)'],/Interp,/NoType, $
		/ZoomFlag,ZBoxColor=8

	ENDIF ELSE BEGIN   ; single plot
	  IF Widget_Info(state.xplot,/Valid_Id) EQ 0 THEN BEGIN
	    Widget_Control,/HourGlass
	    p=0
	    Xplot,Group=event.top,Parent=p,No_Block=state.no_block
	    state.xplot=p
	  ENDIF
	  ;data = MPP_value(state.ptrMPP,iList)
	  data = state.ompp->value(iList)
	  Xplot_LoadFile,state.xPlot,data,/NoRefresh
	  Xplot_ChangeCol,state.xPlot,XCol=1,YCol=2
	ENDELSE
	END


    'MCA2SCAN': mca2scan,Group=event.top
    'MCA2MESH': mca2mesh,Group=event.top,/full_select,outFlag=1
    'MCA2MESH2': mca2mesh,Group=event.top,/full_select,outFlag=0
    'SPEC2EXCEL': BEGIN
	itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
	  ['This option creates Microsoft/Excel-XML file(s) from',$
	   'one (or several) SPEC file(s)',' ',$
	   'The output files have the same names as the input',$
	   'files but with the .xls extension'],/Cancel)
	IF itmp EQ 'Cancel' THEN Return
	files = Dialog_Pickfile(/Read,/Multiple,Dialog_Parent=event.top)
	nfiles = N_Elements(files)
	IF nfiles EQ 1 THEN op=1 ELSE op=0
	IF files[0] EQ '' THEN RETURN
	Widget_Control,/HourGlass
	FOR i=0L,nfiles-1 DO BEGIN
	   spec2excel,files[i],Dialog_Parent=event.top,open=op
	ENDFOR
	END
    'PRLIST': BEGIN
	IF event.clicks NE 2 THEN GoTo,out
	iList = event.Index
	
	IF Widget_Info(state.xplot,/Valid_Id) EQ 0 THEN BEGIN
	  Widget_Control,/HourGlass
	  p=0
	  Xplot,Group=event.top,Parent=p,No_Block=state.no_block
	  state.xplot=p
	ENDIF
	;data = MPP_value(state.ptrMPP,iList)
	data = state.ompp->value(iList)
	Xplot_LoadFile,state.xPlot,data,/NoRefresh
	Xplot_ChangeCol,state.xPlot,XCol=1,YCol=2
	END



    'ABOUT': XAID_Help,Group=event.top
    'HELP': BEGIN
	widget_control,event.id,get_value=tmp
	xhelp,tmp,GROUP=event.top
	END
    else: BEGIN
	;tmp=''
	;Widget_Control,event.id,Get_UValue=tmp,/Hourglass
	;command = tmp[0]+ $  ;',Group='+StrCompress(event.Id,/Rem)+'L'+$
	; ',No_Block='+StrCompress(state.no_block)
	;Message,/info,'Executing: '+command
	;itmp = Execute(command)
	END
EndCase


out:
if Widget_Info(stateid,/Valid_ID) then $
  Widget_Control, stateid, set_UValue=state ; , /No_Copy
end ; exodus_event
;
;====================================================================
;

PRO exodus, group=group, No_Block=no_Block,  $
  xafs=xafs, onlyRight=onlyRight, $
  Parent=parent,Quit_button=quit_button, $
  xlog=xlog, ylog=ylog, Wtitle=wtitle

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'EXODUS: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return
endif
;
; color setting
;
Device,Get_Decomposed=usingDecomposed
IF usingDecomposed EQ 1 THEN BEGIN
  itmp = Dialog_Message(/Question, $
     ['EXODUS does not work properli with decomposed colors',$
     'Turn decomposed colors off?'])
  IF itmp EQ 'Yes' THEN Device,Decomposed=0  
ENDIF
Tek_Color

IF N_Elements(no_block) EQ 0 THEN no_block=1
IF N_Elements(xafs) EQ 0 THEN xafs=0
IF N_Elements(onlyRight) EQ 0 THEN onlyRight=0
IF N_Elements(wtitle) EQ 0 THEN BEGIN
  wtitle='EXODUS '+Exodus_Version()
  IF xafs NE 0 THEN wtitle=wtitle+' (XAFS on)'
ENDIF


;
; default preferences
;
exodus_pref = xop_defaults('exodus') 
;      {rascii_skip:0, import_filter:'y=read_edf(file)', $
;       specNumbering:['1','No','Yes'],xcol:'col1',ycol:'col2'}
IF Keyword_Set(xlog) THEN BEGIN
  tmp=exodus_pref.parameters.xlog
  tmp[0]='1'
  exodus_pref.parameters.xlog=tmp
ENDIF
IF Keyword_Set(ylog) THEN BEGIN 
  tmp=exodus_pref.parameters.ylog
  tmp[0]='1'
  exodus_pref.parameters.ylog=tmp
ENDIF

;
; define widgets
;
wbase=widget_base(/col,title=wtitle,MBar=wMenuBar, $
	 /TLB_KILL_REQUEST_EVENTS, Event_Pro='Exodus_Quit')

wtmp = widget_base(wbase) ; to store state

wFileMenu = Widget_Button(wmenuBar,Value='File',/Menu)
  wtmp1 = Widget_Button(wFileMenu,Value='Preferences',/Menu, /Separator)
    wtmp = Widget_Button(wtmp1,Value='Edit Preferences...',UValue='PREFERENCES')
    wtmp = Widget_Button(wtmp1,Value='Save as default',UValue='PREFERENCES')
  wtmp = Widget_Button(wFileMenu,Value='Quit',UValue='QUIT',/Separator)
  IF Keyword_Set(xafs) THEN BEGIN
    wExafsToolsMenu = Widget_Button(wmenuBar,Value='EXAFS_Tools',/Menu)
      wtmp = Widget_Button(wExafsToolsMenu,Value='EXODUS (New window)',$
	UValue='EXODUS')
      wtmp = Widget_Button(wExafsToolsMenu,Value='Main window (XAID)',$
	UValue='XAID')
  ENDIF
wToolsMenu = Widget_Button(wmenuBar,Value='Tools',/Menu)
  wtmp = Widget_Button(wToolsMenu,Value='EXODUS (New window)',$
	UValue='EXODUS')
  wtmp = Widget_Button(wToolsMenu,Value='XPlot (Plotting tool)',$
	UValue='XPLOT')
  wtmp = Widget_Button(wToolsMenu,Value='Display file...',$
	UValue='DISPLAYFILE')
  wtmp = Widget_Button(wToolsMenu,Value='Convert MCA to SCAN',$
	UValue='MCA2SCAN')
  wtmpbase = Widget_Button(wToolsMenu,Value='Convert MCA to MESH',$
	/Menu)
    wtmp = Widget_Button(wtmpbase,Value='Output to file...',$
	UValue='MCA2MESH')
    wtmp = Widget_Button(wtmpbase,Value='Output to xplot window',$
	UValue='MCA2MESH2')
  wtmp = Widget_Button(wToolsMenu,Value='Convert SPEC to EXCEL',$
	UValue='SPEC2EXCEL')

wHelpMenu = Widget_Button(wmenuBar,Value='Help',/Help)
  IF Keyword_Set(xafs) THEN BEGIN
    wtmp = Widget_Button(wHelpMenu,Value='About XAID',UValue='ABOUT')
  ENDIF
  wtmp = Widget_Button(wHelpMenu,Value='exodus',UValue='HELP')

wPannelMain = Widget_Base(wBase,Row=1)
IF onlyRight NE 1 THEN BEGIN
  wPannelLeft = Widget_Base(wPannelMain,Col=1,/Frame)
  wPannelCenter = Widget_Base(wPannelMain,Col=1,/Frame)
ENDIF
wPannelRight = Widget_Base(wPannelMain,Col=1,/Frame)

yListSize = 25
xListSize=35

IF onlyRight NE 1 THEN BEGIN
;
; Left Pannel
; 
wPLBaseTop = Widget_Base(wPannelLeft,/Row)
  wtmp = Widget_DropList(wPLBaseTop, UValue='IMPORT', $
    Value=['Import SPEC: ','SPEC file','ReLoad SPEC file','MCA/SPEC file list'])
wPLList = Widget_List(wPannelLeft,value=SIndGen(yListSize),UValue='PLLIST',$
  ysize=yListSize,xSize=xListSize,/Multiple)
wPLBaseBottom = Widget_Base(wPannelLeft,/Col)
  wtmp1= Widget_Base(wPLBaseBottom,/Col)
    wSpecColBase= Widget_Base(wtmp1,/Row)
      wtmp = Widget_Label(wSpecColBase,Value='Columns to copy: ')
      wtmp = Widget_Button(wSpecColBase,Value='Help',UValue='HELPCOLUMNS')
    wSpecColBase= Widget_Base(wtmp1,/Row)
      wtmp = Widget_Label(wSpecColBase,Value='ColX = ')
      wXCol =Widget_Text(wSpecColBase,Value=exodus_pref.parameters.xcol,UValue='XCOL',XSize=25,/Edit)
    wSpecColBase= Widget_Base(wtmp1,/Row)
      wtmp = Widget_Label(wSpecColBase,Value='ColY = ')
      wYCol =Widget_Text(wSpecColBase,Value=exodus_pref.parameters.ycol,UValue='YCOL',XSize=25,/Edit)
  wtmp2= Widget_Base(wPLBaseBottom,/Row)
    wtmp= Widget_Button(wtmp2,Value='View Scan',UValue='XSPECVIEW')
    wtmp= Widget_Button(wtmp2,Value='Plot Scan',UValue='XPLOT_LEFT')

;
; Central Pannel
;
wPCCopy1= Widget_Button(wPannelCenter,Value='=>',UValue='COPY1',$
      ToolTip='Copy SELECTED sets from import SPEC panel to work panel')
wPCCopyAll= Widget_Button(wPannelCenter,Value='*=>',UValue='COPY1',$
      ToolTip='Copy ALL sets from import SPEC panel to work panel')
wPCCopyAll= Widget_Button(wPannelCenter,Value='?=>',UValue='COPY1', $
      ToolTip='Copy sets using a wildcard from import SPEC panel to work panel')

ENDIF ELSE BEGIN
wPLBaseTop = 0L
wPLList = 0L
wPLBaseBottom = 0L
wXCol = 0L
wYCol = 0L
wPCCopy1= 0L
wPCCopyAll= 0L
wspecColBase=0L
ENDELSE

;
; Right Pannel
; 
wPRBaseTop = Widget_Base(wPannelRight,/Row)
  IF Keyword_Set(quit_button) THEN $
	wtmp = Widget_Button(wPRBaseTop, Value='Quit',UValue='QUIT')
  wtmp = Widget_DropList(wPRBaseTop, UValue='IMPORTRIGHT', $
    Value=['Import from: ',$
           'Current xplot window','Xplot window ID...',$
           'ASCII file(s) [browser]','ASCII files [wildcard/list]', $
	   'Import filter [one file]','Import filter [wildcard]',$
           'SPEC File'])
  wtmp = Widget_DropList(wPRBaseTop, UValue='VIEWEDIT', $
    Value=['View/Edit: ','View Info','Edit Title'])

export_items = [ $
           'Export: ', $
           'ASCII file(s)', $
           'SPEC multiscan file',$
           'SPEC/MCA multiscan file',$
	   'XPLOT/SPEC mesh',$
	   'XPLOT2D/EDF mesh',$
           'XPLOT/SPEC multicolumn', $
           'EXCEL-XML multisheet file', $
           'EXCEL-XML multicolumn file', $
           'EXCEL-CSV multicolumn file', $
           'Application...', $
           'FuFiFa (fitting)']

IF Keyword_Set(xafs) THEN $
           export_items = [ export_items, $
	   ['XAID/mu2chi', $
           'XAID/ff', $
           'DELIA binary file']]

wtmp = Widget_DropList(wPRBaseTop, UValue='EXPORT', $
    Value=export_items)

;IF Keyword_Set(xafs) THEN BEGIN
;  wtmp = Widget_DropList(wPRBaseTop, UValue='EXPORT', $
;    Value=['Export: ','ASCII file(s)','SPEC multiscan file',$
;	   'XPLOT/SPEC mesh',$
;           'XPLOT/SPEC multicolumn', $
;           'EXCEL-XML multisheet file', $
;           'EXCEL-XML multicolumn file', $
;           'Application...', $
;           'FuFiFa (fitting)', $
;	   'XAID/mu2chi','XAID/ff','DELIA binary file'])
;ENDIF ELSE BEGIN
;  wtmp = Widget_DropList(wPRBaseTop, UValue='EXPORT', $
;    Value=['Export: ','ASCII file(s)','SPEC multiscan file', $
;	   'XPLOT/SPEC mesh',$
;           'XPLOT/SPEC multicolumn', $
;           'EXCEL-XML multisheet file', $
;           'EXCEL-XML multicolumn file', $
;	   'Application...',$
;	   'FuFiFa (fitting)'])
;ENDELSE

wPRList = Widget_List(wPannelRight,value=SIndGen(yListSize),UValue='PRLIST',$
  ysize=yListSize,xsize=xListSize,/Multiple)
wPRBaseBottom = Widget_Base(wPannelRight,/Col)
  wtmp1= Widget_Base(wPRBaseBottom,/Row)
    wtmp= Widget_Button(wtmp1,Value='*Delete',UValue='DELETEALL', $
      ToolTip='Delete ALL data sets')
    wtmp= Widget_Button(wtmp1,Value='Delete',UValue='DELETE', $
      ToolTip='Delete SELECTED data sets')
    wtmp= Widget_Button(wtmp1,Value='Plot',UValue='XPLOT_RIGHT', $
      ToolTip='Plot SELECTED data sets')
  wtmp2= Widget_Base(wPRBaseBottom,/Row)
    ;wtmp = Widget_DropList(wtmp2, UValue='OPERATIONS', $
    ;  Value=['Operations: ','Interpolation','Average','Sum','Customized',$
;	'Fit LinCom'])
    wtmpB = Widget_Button(wtmp2, /Menu, Value='Operations...')
      wtmp0 = Widget_Button(wtmpB, /Menu, Value='interpolate and: ')
        wtmp = Widget_Button(wtmp0, Value='Average',UValue='OPERATIONS')
        wtmp = Widget_Button(wtmp0, Value='Average every n sets',UValue='OPERATIONS')
        wtmp = Widget_Button(wtmp0, Value='Sum',UValue='OPERATIONS')
        wtmp = Widget_Button(wtmp0, Value='Customized',UValue='OPERATIONS')
        wtmp = Widget_Button(wtmp0, Value='Fit LinCom',UValue='OPERATIONS')
      wtmp0 = Widget_Button(wtmpB, /Menu, Value='do on each set: ')
        wtmp = Widget_Button(wtmp0, Value='Interpolate',UValue='OPERATIONS')
        wtmp = Widget_Button(wtmp0, Value='Customized operation',UValue='OPERATIONS')
IF Keyword_Set(xafs) THEN $
        wtmp = Widget_Button(wtmp0, Value='XANES Normalization',UValue='OPERATIONS')
      wtmp0 = Widget_Button(wtmpB, /Menu, Value='combine data: ')
        wtmp = Widget_Button(wtmp0, Value='Append',UValue='OPERATIONS')
      wtmp0 = Widget_Button(wtmpB, Value='Merge two blocks...',$
              UValue='OPERATIONS')
      wtmp0 = Widget_Button(wtmpB, Value='Parameters from blocks...',$
              UValue='OPERATIONS')


;
; set widget values
;
Widget_Control,wPRList,Set_value=['']
IF onlyRight NE 1 THEN Widget_Control,wPLList,Set_value=['']

;IF onlyRight NE 1 THEN BEGIN
;  Widget_Control,wPLList,Set_value=['']
;  IF xafs EQ 1 THEN BEGIN
;    xScale=[1.0D3,0.0D0]
;    Widget_Control,wXScale,Set_DropList_Select=1 
;  ENDIF ELSE BEGIN
;    xScale=[1.0D0,0.0D0]
;    Widget_Control,wXScale,Set_DropList_Select=0
;  ENDELSE
;ENDIF ELSE BEGIN
;    xScale=[1.0D0,0.0D0]
;ENDELSE
;
; parameters to store
;
dir=0 & cd,current=dir
ptrSpecFile = Ptr_New()
;ptrMPP = MPP_New()
wids = {Base: wBase, PLList:wPLList, PRList:wPRList, $
        xCol:wXCol, yCol:wYCol, specColBase:wSpecColBase}
state = {wids:wids, pref:exodus_pref, no_block:no_block, $
  ptrSpecFile:ptrSpecFile, $
  ompp:Obj_New('ompp'), Xplot:0L, FuFiFa:0L, $
  XAID_MU2CHI:0L, XAID_FF:0L, $
  specFile:'', importDir:dir, xafs:xafs, wildfile:'*.txt', $
  wildcard_transfer:'*'}

;
;
Parent=Widget_Info(wbase,/Child)
widget_control,parent,set_uvalue=state ; ,/no_copy

widget_control,wbase,/realize
xmanager,'exodus',wbase,GROUP=group,No_Block=no_block
end
