;+
;
; XPOWDER: X-ray diffraction patterns for powder samples
;
;
;
; This application calculates a  X-ray diffraction pattern for 
; a powder crystal. 
;
; The application is divided in three panels:
; "Asym Unit", "Unit Cell" and "Reflections"
;
; The main application is under the third panel "Reflection". The
; first two panels are optional with tools to help the user to 
; load and eventually compute the unit cell. 
;
; The calculation of the powder pattern is done in the routine
; xpowder_calchkl(), which basically 
;
; i) loads the information of the 
;    unit cell (either fom the "Unit Cell" panel or from a file *.ucell, 
;    being this a text ASCII file containing in the first line 
;    a b c alpha beta gamma (in Angstroms the lengths, and in deg the
;    angles) and then a number of lines, one per atom in the unit cell,
;    with Z, occupation number, x, y and z (coordinates in crystal units).
;
;    example: Silicon.ucell:
;    5.43070 5.43070 5.43070 90.0000 90.0000 90.0000
;    14 1.0 0.0 0.0 0.0
;    14 1.0 0.0 0.5 0.5
;    14 1.0 0.5 0.0 0.5
;    14 1.0 0.5 0.5 0.0
;    14 1.0 .25 .25 .25
;    14 1.0 .25 .75 .75
;    14 1.0 .75 .25 .75
;    14 1.0 .75 .75 .25
;    
;
; ii) Calculate all possible reflections (permutations of h k l indices)
;
; iii) Computes the Structure Factor for each reflection, and the
;      intensity (only unpolarized beams are used by now). 
;      Reflections with intensity inder a cutoff value (in %) are 
;      eliminated (considered forbidden reflections). 
;
; iv) Classifies and displays results. 
;
; 
; The Panel "Unit Cell" permits to load a unit cell, either from
; a *.ucell file or from the DABAX database. 
;
; The Panel "Asym Unit" contains tools to compute the unit cell. For
; that, the user must enter the atoms in the Asymmetric Unit, and the 
; symmetry operations to be applied. When the 
; "Compute Atoms in the Unit Cell"  button is pressed, the calculation
; is performed in the following way: every atoms in the asym. unit is
; transformed using all the symmetry operations. The redundant atoms 
; are eliminated (and counted for calculating their multiplicity). 
; Optionally, the operations of the 230 spatial symmetry groups can 
; be load from the SpatialGroups.dat DABAX file. Note that this file
; contains the 230 groups and their names as obtained from: 
; http://www.cryst.ehu.es/   (GENPOS application)
;
; 
; All the application information can be stored in a xpowder input file
; for being restored lately. In addition, the user can create text files
; with information that can be loaded in the table widgets, by tagging
; the inputs in the following way:
; 
; For Entering data in the "Asym Unit" panel, tag the inputs 
; in this way (example): 
;
;
;   <starttext>AUATOMS
;            57     0.9840        0.0        0.0        0.0         La
;             5        1.0     .19930         .5         .5          B
;   <endtext>AUATOMS
;   <starttext>AUSYM
;    x,y,z
;    -x,-y,z
;    -x,y,-z
;   <endtext>AUSYM
;
; This file can be reaf using File->Load xpowder input file...
;
; In the same way, information in the "Unit Cell" panel are tagged as:
;
; <starttext>UCATOMS
;       57.00     0.9840      0.000      0.000      0.000         La
;       5.000      1.000     0.1993     0.5000     0.5000          B
;       5.000      1.000     0.8007     0.5000     0.5000          B
;       5.000      1.000     0.5000     0.1993     0.5000          B
;       5.000      1.000     0.5000     0.8007     0.5000          B
;       5.000      1.000     0.5000     0.5000     0.1993          B
;       5.000      1.000     0.5000     0.5000     0.8007          B
; <endtext>UCATOMS
; <starttext>UCCELL
;     4.15271    4.15271    4.15271    90.0000    90.0000    90.0000
; <endtext>UCCELL
;
; 
;
;
; Author: Manuel Sanchez del Rio, March 2007
;
; Modification history:
;    2007-05-16 srio@esrf.fr added INFO tag,reference directory
;
;
;  COPYRIGHT:
; 	XPOWDER belongs to XOP package and it is distributed within XOP.
; 	PLEASE REFER TO THE XOP COPYRIGHT NOTICE
; 
;  REFERENCE:
; 	Published calculations made with XOP should refer:
; 
; 	M. Sanchez del Rio and R. J. Dejus 
;         "Status of XOP: an x-ray optics software toolkit"
;         SPIE Proceedings Vol. 5536 (2004) pp.171-174
; 
;         http://dx.doi.org/10.1117/12.560903
; 
;  LAST MODIFICATION: srio@esrf.eu 2008-02-01
; 
; 
;
;-


PRO xpowder_loadinputfile,wid,file=file,path=path,group=group

	strNew = tags2struct(file=file,path=path,group=group)
	IF Type(strNew) EQ Type(0) THEN RETURN
        Widget_Control,wid,Get_UValue=info
        Widget_Control,info.wids.info, Set_Value=' '
	str = info.str
	namesNew = Tag_Names(strNew)
	names = Tag_Names(str)
	FOR i=0,N_Tags(strNew)-1 DO BEGIN
	  tmp = strNew.(i)
	  CASE Type(tmp) OF
	  Type({a:0}): BEGIN
	     ii = Where(names EQ namesNew[i])
	     IF ii[0] NE -1 THEN BEGIN
		tmp1=str.(ii[0])
		Copy_Structure,strNew.(i),tmp1,/onlyFirstField
		str.(ii[0])=tmp1
	     ENDIF
	     END
	  Type(''): BEGIN
	     CASE namesNew[i] OF
               'INFO': BEGIN
                   Widget_Control,info.wids.info, Set_Value=tmp
		   END
               'AUATOMS': BEGIN
                   nTmp = N_Elements(tmp)
                   txt = strarr(6,nTmp)
                   FOR j=0,nTmp-1 DO BEGIN
                    tmp1=strsplit( tmp[j],' ',/Extract)
                    txt[*,j]=tmp1
                   ENDFOR
                   Widget_Control,info.wids.auAtoms,  $
                     Table_Ysize=nTmp,Set_Value=txt
		   END
               'AUSYM': BEGIN
                   Widget_Control,info.wids.auSym,  $
                     Table_Ysize=N_Elements(tmp),Set_Value=Transpose(tmp)
		   END
               'UCATOMS': BEGIN
                   nTmp = N_Elements(tmp)
                   txt = strarr(6,nTmp)
                   FOR j=0,nTmp-1 DO BEGIN
                    tmp1=strsplit( tmp[j],' ',/Extract)
                    txt[*,j]=tmp1
                   ENDFOR
                   Widget_Control,info.wids.ucAtoms,  $
                     Table_Ysize=nTmp,Set_Value=txt
		   END
               'UCCELL': BEGIN
                   txt=strsplit( tmp,' ',/Extract)
                   Widget_Control,info.wids.ucCell,  $
                     Table_Ysize=1,Set_Value=txt
		   END
               'HKL':  BEGIN
                   nTmp = N_Elements(tmp)
                   txt = strarr(10,nTmp)
                   FOR j=0,nTmp-1 DO BEGIN
                    tmp1=strsplit( tmp[j],' ',/Extract)
                    txt[*,j]=tmp1
                   ENDFOR
                   Widget_Control,info.wids.hkl,  $
                     Table_Ysize=nTmp,Set_Value=txt
		   END
               'AUAUX':  BEGIN
                   Widget_Control,info.wids.auAux,  $
                     Set_Value=tmp[0]
		   END
               ELSE: 
	     ENDCASE

             info.str=str
             Widget_Control,wid,Set_UValue=info
	     END

	  ELSE:
	  ENDCASE
	ENDFOR
	

END ; xpowder_loadInputFile

;
;===============================================================================
;

PRO xpowder_event,event


Catch, error_status
IF error_status NE 0 THEN BEGIN
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XPOWDER_EVENT: error caught: '+!err_string)
   Message,/Info,'error caught: '+!err_string
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

Widget_Control, event.id, get_UValue=eventUValue
if n_elements(eventuvalue) EQ 0 then eventuvalue = 'NONE'

Case eventUValue OF
    'QUIT': BEGIN
	Widget_Control,event.top,/Destroy
	END
    'LOAD_XPOWDER': BEGIN
	Widget_Control,event.id,Get_value=value
	IF value EQ 'References...' THEN BEGIN
;	  path=xop_getenv('XOP_HOME')+sdep(/ds)+'data'+sdep(/ds)+'xpowder'
	  path=xop_getenv('XOP_HOME')
          IF StrMid(path,0,1,/Reverse_OffSet) NE sdep(/ds) THEN $
             path=path+sdep(/ds)
	  path=path+'data'+sdep(/ds)+'xpowder'
	  xpowder_loadInputFile,event.top,path=path,group=event.top
        ENDIF ELSE BEGIN
	  xpowder_loadInputFile,event.top,group=event.top
        ENDELSE
	END
    'HELP': Xhelp,'xpowder',GROUP=event.top

    'SAVE_XPOWDER': BEGIN
        Widget_Control,event.top,Get_UValue=info
	str0 = info.str
	names = Tag_Names(str0)
	FOR i=0,N_Tags(str0)-1 DO BEGIN
           tmp = str0.(i)
	   IF Type(tmp) EQ Type({a:0}) THEN BEGIN
	      IF N_Elements(str1) EQ 0 THEN $
		str1 = Create_Struct(names[i],tmp) ELSE $
		str1 = Create_Struct(str1, names[i],tmp) 
	   ENDIF
	ENDFOR
	txtTags = ['INFO','AUATOMS','AUSYM','UCATOMS','UCCELL','HKL','AUAUX']
	FOR i=0,N_Elements(txtTags)-1 DO BEGIN
          txt = 0
          itmp = Execute('Widget_Control,info.wids.'+txtTags[i]+$
			',Get_Value=txt')
	  IF (Size(txt))[0] EQ 2 THEN BEGIN
	    n0=N_Elements(txt[0,*])
	    ncol=N_Elements(txt[*,0])
	    txt1 = StrArr(n0)
	    IF ncol EQ 1 THEN BEGIN
	      txt1=Reform(txt)
	    ENDIF ELSE BEGIN
	      FOR j=0,n0-1 DO BEGIN
                tmp2=String(txt[*,j],Format='('+StrCompress(ncol,/Rem)+'A11)')
		txt1[j]=tmp2
	      ENDFOR
            ENDELSE
	  ENDIF ELSE BEGIN
           IF txtTags[i] EQ 'INFO' THEN txt1=txt ELSE $
	   txt1=String(txt,Format='('+StrCompress(N_Elements(txt),/Rem)+'A25)')
	  ENDELSE
	  str1 = Create_Struct(str1, txtTags[i],txt1) 
        ENDFOR
        xop_input_save,str1,File='xpowder.xop',/tags, Group=event.top,  $
          /Write, Comment='; xop/xpowder(v1.0) input file on '+SysTime(),$
	  /NoFlagsTitles,Filter='*.xop'
	END

    'AULOADATOMS': BEGIN
	  filename = Dialog_Pickfile(Dialog_Parent=event.top,Title=$
	    'Select a 6 col file: Z,Occ,x,y,z,label')
	  IF filename EQ '' THEN RETURN
	  tmp = read_textfile(filename)
          NAtoms = N_Elements(tmp)
          atoms = strarr(6,NAtoms)
          FOR i=0,Natoms-1 DO BEGIN
            tmp1=strsplit( tmp[i],' ',/Extract)
            Atoms[*,i]=tmp1
          ENDFOR
          Widget_Control,event.top,Get_UValue=info
          Widget_Control,info.wids.auAtoms, Table_Ysize=Natoms, $
		Set_Value=Atoms
	END

    'AULOADSYM': BEGIN 
        Widget_Control,event.top,Get_UValue=info
	austr = info.str.auStr
	titles= info.str.auStr_Titles
	flags = info.str.auStr_Flags
        XSCRMENU,austr,/interpret,/notype,Titles=titles,flags=flags, $
	  Action=action,Dialog_Parent=event.top,WTitle='Load symmetry'
	IF action EQ 'DONT' THEN RETURN
	info.str.auStr = auStr
        Widget_Control,event.top,Set_UValue=info

	h = dabax_access('SpaceGroups.dat')
	IF Type(h) EQ Type(0) THEN RETURN
	n = h.scan.scan_n
	names = h.scan.name
	names2 = names
	itmp = StrPos(names2,':')
	FOR i=0,N_Elements(names)-1 DO BEGIN
	  IF itmp[i] GE 0 THEN names2[i]=StrMid(names2[i],0,itmp[i])
	ENDFOR

	iflag = StrCompress((austr.from)[0],/Rem)
	IF iflag EQ '2' THEN BEGIN ; File
	  filename = austr.file
	  oper = read_textfile(filename)
	  IF Type(oper) EQ Type(0) THEN RETURN
          IF filename NE '' THEN austr.file=filename
	ENDIF ELSE BEGIN ; Space group tables
	  CASE iflag OF
	    '0': BEGIN
		   index = Where(n EQ austr.number)
	           IF index[0] EQ -1 THEN BEGIN
		      itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		      'Space Group Number not found: '+StrCompress(austr.number))
		      RETURN
	           ENDIF
		 END
	    '1': BEGIN
		   index = Where(StrMatch(StrCompress(names2,/Rem),$
		      StrCompress(austr.name,/Rem)) EQ 1)
	           IF index[0] EQ -1 THEN BEGIN
		      itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		      ['Space Group Name not found: '+StrCompress(austr.name),$
                       ' ','Look for the available group names at: ','',$
		       'http://www.cryst.ehu.es/  (GENPOS)'] )
		      RETURN
	           ENDIF
		 END
	    else: 
	  ENDCASE

	  IF N_Elements(index) GT 1 THEN BEGIN
		itmp = Dialog_Message(/Question,/Cancel,$
		   Dialog_Parent=event.top, $
                 ['Two options are available for Space Group Number: '+$
		      StrCompress(austr.number),names[index], $
		  '','Do you want the first choice? ',$
		  '       Yes: '+names[index[0]],$
		  '       No: '+names[index[1]],$
		  '       Cancel: Abort'])
		IF itmp EQ 'Cancel' THEN RETURN
		IF itmp EQ 'Yes' THEN index = index[0]
		IF itmp EQ 'No' THEN index = index[1]
	  ENDIF 
          itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
               ['Space Group Number: '+StrCompress(austr.number),$
               'Space Group Name: '+spec_name(h,Fix(index[0])+1,/index)])
          IF itmp EQ 'Cancel' THEN RETURN

	  oper = spec_headers(h,index[0]+1,/Index,'USYMOPER',/Label_Remove)
	ENDELSE

	noper = N_Elements(oper)
	Widget_Control,event.top,Get_UValue=info
        Widget_Control,info.wids.auSym,Table_Ysize=noper, $
                Set_Value=Transpose(oper)
	END


;    'AUNATOMS': BEGIN
;	Widget_Control, event.top, get_UValue=info
;        Widget_Control,info.wids.auNAtoms,Get_Value=auNatoms
;        Widget_Control,info.wids.auAtoms,Get_Value=auAtoms
;        Widget_Control,info.wids.auAtoms,Table_Ysize=auNatoms
;	; extend the table values
;	IF auNatoms GT N_Elements(auAtoms[0,*]) THEN BEGIN
;	  txt = StrArr(6,auNatoms)
;	  txt[*,0:N_Elements(auAtoms[0,*])-1]=auAtoms
;          Widget_Control,info.wids.auAtoms,Set_Value=txt
;        ENDIF
;	END
;
;    'AUNSYM': BEGIN
;	Widget_Control, event.top, get_UValue=info
;        Widget_Control,info.wids.auNSym,Get_Value=auNSym
;        Widget_Control,info.wids.auSym,Get_Value=auSym
;        Widget_Control,info.wids.auSym,Table_Ysize=auNSym
;	; extend the table values
;	IF auNsym GT N_Elements(auSym[0,*]) THEN BEGIN
;	  txt = Transpose(Replicate('x,y,z',auNsym))
;	  txt[*,0:N_Elements(auSym[0,*])-1]=auSym
;          Widget_Control,info.wids.auSym,Set_Value=txt
;        ENDIF
;	END


    'AUCALC': BEGIN
	Widget_Control, event.top, get_UValue=info
        Widget_Control,info.wids.auAtoms,Get_Value=auAtoms
        Widget_Control,info.wids.auSym,Get_Value=auSym
        Widget_Control,info.wids.auAux,Get_Value=auAux

	atoms=auAtoms
	sym=auSym
	xpowder_calcunitcell,atoms,sym,out,aux=auAux[0]
	itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
	  'Results copied to Unit Cell TAB')
	Widget_Control, event.top, get_UValue=info
	nout = N_Elements(out[0,*])
        Widget_Control,info.wids.ucAtoms,Table_YSize=nout,Set_Value=out



	END
    'UCLOAD': BEGIN 
	Widget_Control, event.top, get_UValue=info
        xpowder_loaducstr = info.str.ucLoad
        titles=info.str.ucLoad_Titles
        flags=info.str.ucLoad_Flags

        XscrMenu,xpowder_loaducstr,Flags=flags,Titles=titles, $
	  Action=action,/NoType,/Interp, $
          Wtitle='Load unit cell',Dialog_Parent=event.top
	IF action EQ 'DONT' THEN RETURN
	info.str.ucLoad=xpowder_loaducstr

        CASE Fix( (xpowder_loaducstr.from)[0] ) OF
          0: BEGIN
               crystalIndex=Fix(xpowder_loaducstr.crystal[0])  ; index starts from zero
	       ;hStr = dabax_access('CrystalStructures.dat')
	       ;hCell = dabax_access('CrystalCell.dat')
               ;crystalStr = double(spec_data(hstr,crystalIndex+1,/index))
               ;crystalCell = double(spec_data(hcell,crystalIndex+1,/index))
		dabax_crystals,crystalIndex,str=crystalStr,cell=crystalCell
             END
          1: BEGIN
               file=StrCompress(xpowder_loaducstr.file,/Rem)
               IF file EQ '?' THEN BEGIN
                 file = Dialog_PickFile(Filter='*.ucell',Title='Select a file with unit cell')
                 IF file EQ '' THEN RETURN
                 IF CheckFile(file) NE 1 THEN BEGIN
                   itmp = Dialog_Message(/Error,'File not found: '+f)
                   Return
                 ENDIF
                 xpowder_loaducstr.file=file
               ENDIF
               crystalStr=rascii(file,Skip=1,Header=h)
               crystalCell=FltArr(6)
               openR,unit,file,/Get_Lun
               readF,unit,crystalCell
               Free_Lun,unit
               title='File: '+file
             END
        ENDCASE


	s = Size(crystalStr)
	atoms = StrArr(s[1]+1,s[2])
	atoms[0:s[1]-1,*] = String(crystalStr,Format='(G10.6)')
        Widget_Control,info.wids.ucAtoms,Table_Ysize=s[2], $
                Set_Value=atoms
        Widget_Control,info.wids.ucCell,Set_Value=String(crystalCell, $
	  Format='(G10.6)')

	Widget_Control,event.top,Set_UValue=info
	END

    'UCSAVE': BEGIN 

	backk:
        file = Dialog_Pickfile(FILTER='*.ucell',Title= $
          'Select a file for writing Unit Cell', $
	  File='xpowder.ucell',Dialog_Parent=event.top)
	IF file EQ '' THEN RETURN
	IF checkfile(file) EQ 1 THEN BEGIN
	  itmp = Dialog_Message(/Question,['File exists: '+file,$
		' ','Overwrite it? '],/Cancel,Dialog_Parent=event.top)
	  IF itmp EQ 'Cancel' THEN RETURN
	  IF itmp EQ 'No' THEN GoTo,backk
	ENDIF

	Widget_Control,event.top,Get_UValue=info
        Widget_Control,info.wids.ucAtoms, Get_Value=CrystalStr
        Widget_Control,info.wids.ucCell,Get_Value=CrystalCell
        ; 
        ; write ucell file
        ; 
	openw,unit,file,/Get_Lun
	printf,unit,CrystalCell
	FOR i=0L,N_Elements(crystalStr[0,*])-1 DO $
		printf,unit,crystalStr[0:4,i],Format='(I3,4G15.5)'
	free_lun,unit
	
	itmp = Dialog_Message(/Question,['File written to disk: '+file,$
		' ','Open it? '],/Default_No,Dialog_Parent=event.top)
	IF itmp EQ 'Yes' THEN XDisplayFile1,file


	END

    'HKLSET': BEGIN ; cp npointx and slp from first panel to third panel
	Widget_Control,event.top,Get_UValue=info
	str = info.str.hklStr
        titles=info.str.hklStr_Titles
	flags = info.str.hklStr_Flags

        helpdir = Xop_GetEnv('XOP_HOME')+SDep(/ds)+'doc'+SDep(/ds)+'txt'
        helpcmd="xdisplayfile1,'"+helpdir+sdep(/ds)+"xpowder_par.txt'

        XSCRMENU,str,/interpret,/notype,Titles=titles,flags=flags, $
	  Action=action,Dialog_Parent=event.top,NCOL=2, $
	  Wtitle='Calc Reflections',Help=helpcmd

	IF action EQ 'DONT' THEN RETURN

	info.str.hklStr  = str
	Widget_Control,event.top,Set_UValue=info

	; 
	; make calculations
	;
	Widget_Control,/HourGlass

	out2 = xpowder_calchkl(str,Group=event.top)

	; 
	; update widget_table
	;
	nout2 = N_Elements(out2[0,*])
	sout2 = StrCompress(out2)
	sout2[0:2,*] = String(sout2[0:2,*],Format='(I5)')
	sout2[7,*] = String(sout2[7,*],Format='(I3)')
        Widget_Control,info.wids.hkl,Table_Ysize=nout2,Table_Xsize=10, $
                Set_Value=sout2,Column_Labels=$
        ['h','k','l','2*Theta','d-spc','|F|','Int','M','IntAll','IntNorm']

	END

    'HKLSHOW': BEGIN ; cp npointx and slp from first panel to third panel
	Widget_Control,event.top,Get_UValue=info
        Widget_Control,info.wids.hkl, Get_Value=out2
	IF N_Elements(out2[*,0]) EQ 6 THEN BEGIN
	   itmp = Dialog_Message(/Error, ['No reflections are calculated.','','User "Set Parameteres" first'],Dialog_Parent=event.top)
	   RETURN
	ENDIF

        ;
        ; display out2 results
        ;
        ; 
        ;Display table with results 
        ; 
	Widget_Control,event.id,Get_Value=value
	CASE value OF
	'Plot Lines': BEGIN
            xplot,parent=p, Double(out2),xcol=4,ycol=9,/no_block,coltitles=$
            ['h','k','l','2*Theta','d-spc','|F|','Int','M','IntAll','IntNorm']
            UserSym,[0.,0.],[-1e9,0]
	    xplot_controls_action,p,PSymbol=8,PSYMSign='1',Clr_Lines=2
	    END
	'Export Table': BEGIN
            txt = Replicate('',N_Elements(out2[0,*])+1)
            txt[0]=String('h','k','l','2*Theta','d-spc','|F|','Int','M','IntAll','IntNorm',format='(3A4,4A15,A15,2A15)')
            isort = Reverse(Sort(out2[9,*]))
            FOR i=0,N_Elements(out2[0,*])-1 DO BEGIN
              ;ii = isort[i]
              ii = i
              ;print,out2[*,ii],format='(3I4,4G15.5,I15,2G15.5)'
              txt[i+1]=String(out2[*,ii],format='(3I4,4G15.5,I15,2G15.5)')
                ENDFOR
            XDisplayFile1,text=txt,title='hkl list'
	    END
	'Broad peaks...': BEGIN
	    lineWidthStr = info.str.broadStr
	    titles= info.str.broadStr_Titles
	    flags = info.str.broadStr_Flags
            XSCRMENU,lineWidthStr,/interpret,/notype,Titles=titles,$
		flags=flags, Action=action,Dialog_Parent=event.top, $
		WTitle='Broad peaks'
	    IF action EQ 'DONT' THEN RETURN
	    info.str.broadStr = lineWidthStr
	    Widget_Control,event.top,Set_UValue=info

	    out2 = Double(out2)
	    nlines = N_Elements(out2[0,*])
	    
	    twothetarange = [min(out2[3,*],max=mm),mm]
	    epsilon = abs(twothetarange[0]-twothetarange[1])*0.01
	    twothetarange = twothetarange + [-epsilon,epsilon]
	    twothetaN=abs((twothetarange[0]-twothetarange[1])/lineWidthStr.step)+1
	    twotheta = makearray1(twothetaN,twothetarange[0],twothetarange[1])
	    diff = twotheta*0
	    eta = lineWidthStr.voigtFac
	    IF StrCompress( (lineWidthStr.func)[0],/Rem) EQ '0' THEN eta=0D0
	    FOR i=0,nlines-1 DO BEGIN
	      i0=(out2[5,i])^2*out2[7,i]*bragg_lorentz(out2[3,i]/2.,Ret=4)
	      diff = diff+voigt1(twotheta,[i0,out2[3,i],lineWidthStr.width,eta],/Area)
	    ENDFOR
	    tmp = Make_Set(twotheta,diff,diff/max(diff)*100)
	    xplot,tmp,colTitles=['TwoTheta','Intens','Norm Intens'],/no_block

	    END
	ELSE: 
	ENDCASE
        
	END

    'NONE': 
    else: print,'EventUValue NOT FOUND: ',eventUValue
EndCase


;out:
;Widget_Control, stateid, set_UValue=state, /No_Copy
end
;
;====================================================================
;
PRO xpowder,input, Group=group, inputFile=inputFile

;
; color setting
;
Device,Get_Decomposed=usingDecomposed
IF usingDecomposed EQ 1 THEN BEGIN
  itmp = Dialog_Message(/Question, $
     ['XPOWDER does not work properli with decomposed colors',$
     'Turn decomposed colors off?'])
  IF itmp EQ 'Yes' THEN Device,Decomposed=0
ENDIF
Tek_Color


;
; input parameters
;

str = xpowder_defaults()



;
; widgets
;
wbase=widget_base(/col,title='xpowder',MBAR=wMenuBar)


;
; Menu bar
;
wtmp0 = widget_button(wMenuBar,Value='File')
  wtmp1 = widget_button(wtmp0,Value='Load xpowder input file',/MENU)
    wtmp = widget_button(wtmp1,Value='User file...',UValue='LOAD_XPOWDER')
    wtmp = widget_button(wtmp1,Value='References...',UValue='LOAD_XPOWDER')
  wtmp = widget_button(wtmp0,Value='Save xpowder input file...',UValue='SAVE_XPOWDER')
  wtmp = widget_button(wtmp0,Value='Quit',UValue='QUIT')
wtmp0 = widget_button(wMenuBar,Value='Help',/Help)
  wtmp = WIDGET_BUTTON(wtmp0, VALUE='xpowder', UVALUE='HELP')
 
;
; different bases
;
wTab=Widget_Tab(wbase,UVal='TAB') 
  wTXT=Widget_Base(wTab, Title='Info', Column=1) 
  wAU=Widget_Base(wTab, Title='Asym Unit', Column=1) 
  wUC=Widget_Base(wTab, Title='Unit Cell', Column=1) 
  wHKL=Widget_Base(wTab, Title='Reflections', Column=1) 
  ;wPLOT=Widget_Base(wTab, Title='Diffractogram', Column=1) 

;
; Info panel
;
w1 = Widget_Base(wTXT,/col,/Frame)
wtmp = Widget_Label(w1,Value='Enter/paste here a text with info on sample, references, etc.')
wInfo = Widget_Text(w1,/Edit,Value=[' '],XSize=75,YSize=40,/Scroll)


; AU Asymmetric Unit
;
aunatoms=2
w1 = Widget_Base(wAU,/col,/Frame)
wtmp = Widget_Label(w1,Value='** WARNING: THIS SECTION IS UNDER DEVELOPMENT - IT MAY NOT BE CORRECT **')
wtmp = Widget_Label(w1,Value='Asym Unit: DEFINITION OF ASYMMETRIC UNIT FOR COMPUTING UNIT CELL. OPTIONAL')
wtmp = Widget_Label(w1,Value=' ')

 wtmp = Widget_Label(w1,Value='Atoms in the Asymmetric Unit')
;wauNatoms = CW_Field(w1,Value=auNatoms,/Integer,Title='Number of atoms: ',$
;            UValue='AUNATOMS',XSize=4,/Return_Events)
 wtmp0 = Widget_Base(w1,/row)
   wtmp = Widget_Button(wtmp0,Value='Load from file...',UValue='AULOADATOMS')
   wauAux = CW_Field(wtmp0,Value=(str.auAux)[0],/String,$
            Title='Auxiliar variable: ',$
            UValue='NONE',XSize=25,/Return_Events)



;tmp = [ ['8','1.0','0.0','0.0','0.0','O1'], $
;        ['8','1.0','0.0','0.0','0.0','O2'] ]

tmp = str.auAtoms
auNAtoms = N_Elements(tmp)
auAtoms = strarr(6,auNAtoms)
FOR i=0,auNatoms-1 DO BEGIN
  tmp1=strsplit( tmp[i],' ',/Extract)
  auAtoms[*,i]=tmp1
ENDFOR


wauAtoms = Widget_Table (w1, xsize=6, scr_ysize=200, $
             value=auAtoms, /Editable, /All_Events, Scroll=0, $
             column_labels=['Z','Occ','x','y','z','Label'], $
             UValue='NONE',/Resizeable_Columns)

wtmp = Widget_Label(w1,Value='x')

auNsym = 2
wtmp = Widget_Label(w1,Value='Symmetry Operations')

wtmp0 = Widget_Base(w1,/Row)
  ;wauNsym = CW_Field(wtmp0,Value=auNsym,/Integer,$
  ;          Title='Number of operations: ',$
  ;          UValue='AUNSYM',XSize=4,/Return_Events)
  wtmp = Widget_Button(wtmp0,Value='Load...',UVALUE='AULOADSYM')


;tmp = [['x,y,z'],['-x,-y,z']] 
tmp = str.auSym

wauSym = Widget_Table (w1, xsize=1, scr_ysize=150, $
             value=tmp, /Editable, /All_Events, Scroll=0, $
             column_labels=['operation'],Column_Width=325, $
             UValue='NONE',/Resizeable_Columns)

wtmp = Widget_Label(w1,Value='=')
wtmp0 = Widget_Base(w1,/Row,/Align_center)
  wtmp = Widget_Button(wtmp0,Value='Compute Atoms Unit Cell',$
         UValue='AUCALC')

;
; UC Unit cell
;
aunatoms=2
w1 = Widget_Base(wUC,/Col,/Align_Center)
  wtmp = Widget_Label(w1,Value='Unit Cell: DEFINITION OF UNIT CELL.  NEEDED FOR REFLECTIONS')
  wtmp = Widget_Label(w1,Value=' ')
  wtmp = Widget_Button(w1,Value='Load unit cell...',UVALUE='UCLOAD',/Align_Center)
w1 = Widget_Base(wUC,/col,/Frame)
wtmp = Widget_Label(w1,Value='Atoms in the Unit Cell')
wtmp0 = Widget_Base(w1,/Row)
  ;wucNatoms = CW_Field(wtmp0,Value=auNatoms,/Integer,Title='Number of atoms: ',$
  ;          UValue='UCNATOMS',XSize=4,/Return_Events,/NoEdit)



;tmp = [ ['8','1.0','0.0','0.0','0.0','O1'], $
;        ['8','1.0','0.0','0.0','0.0','O2'] ]
;
tmp = str.ucAtoms
ucNAtoms = N_Elements(tmp)
ucAtoms = strarr(6,ucNAtoms)
FOR i=0,ucNatoms-1 DO BEGIN
  tmp1=strsplit( tmp[i],' ',/Extract)
  ucAtoms[*,i]=tmp1
ENDFOR

wucAtoms = Widget_Table (w1, xsize=6, scr_ysize=150, $
             value=ucAtoms, /Editable, /All_Events, Scroll=0, $
             column_labels=['Z','Occ','x','y','z','Label'], $
             UValue='NONE',/Resizeable_Columns)

wtmp = Widget_Label(w1,Value=';')

wtmp = Widget_Label(w1,Value='Unit Cell Dimensions')

;tmp = ['5.','5.','5.','90','90','90']
tmp = str.ucCell
ucCell=strsplit( tmp[0],' ',/Extract)

wucCell = Widget_Table (w1, xsize=6, ysize=1, $
   value=ucCell, /Editable, /All_Events, Scroll=0, $
   column_labels= $
   ['a[A]','b[A]','c[A]','alpha[deg]','beta[deg]','gamma[deg]'], $
   UValue='NONE',/Resizeable_Columns) 

wtmp = Widget_Button(w1,Value='Save unit cell...',UVALUE='UCSAVE',/Align_Center)

;
; HKL Reflections
;
w1 = Widget_Base(wHKL,/Col,/Align_Center)
  wtmp = Widget_Label(w1,Value='Reflections: CALCULATION OF X-RAY POWDER PATTERN. UNIT CELL MUST BE READY')
  wtmp = Widget_Label(w1,Value=' ')
  wtmp0 = Widget_Base(w1,/Row)
    wtmp = Widget_Button(wtmp0,Value='Set Parameters...',UVALUE='HKLSET', $
	/Align_Center)

  whkl = Widget_Table (w1, xsize=6, scr_ysize=200, $
         column_labels= $, 
         ['h','k','l','2*Theta','d-spc','|F|','Int','M','IntAll','IntNorm'],$
         UValue='NONE',/Resizeable_Columns)


  wtmp0 = Widget_Base(w1,/Row)
      wtmp = Widget_Button(wtmp0,Value='Plot Lines',UVALUE='HKLSHOW')
      wtmp = Widget_Button(wtmp0,Value='Export Table',UVALUE='HKLSHOW')
      wtmp = Widget_Button(wtmp0,Value='Broad peaks...',UVALUE='HKLSHOW')



;list=dabax_ls()

;str = {wavelength:1.0,limitFlg:['0','2Theta Max','dspacing min','HLK Max'], $
;       limitValue:30.0,debyeW:'1', $
;        f0file:['0',filef0], $
;        anomalousFlg:['0','No','Yes'], f1f2file:['0',filef1f2],$
;        ucFrom:['1','Unit Cell TAB','File'],ucellFile:'?' }



wids = { info:wInfo, $
         auAtoms:wauAtoms, auSym:wauSym, auAux:wauAux, $
         ucAtoms:wucAtoms, ucCell:wucCell, hkl:whkl }
state = {wids:wids, str:str}

;
;
;
widget_control,wbase,set_uvalue=state


widget_control,wbase,/realize

IF Keyword_Set(inputFile) THEN BEGIN
   xpowder_loadinputfile,wbase,file=inputFile
ENDIF


Widget_Control,wTab,Set_Tab_Current=2
xmanager,'xpowder',wbase,GROUP=group,/NO_BLOCK
end
