;+
; NAME:
;	nrbStruct
;
; PURPOSE:
;	This function packs a surface and its knots into a NURBS struature.
;
; CATEGORY:
;	NURBS library
;
; CALLING SEQUENCE:
;	n = nrbStruct(coefs, uknots, vknots)
;
; INPUTS:
;    coefs:	a (nx,4,ny) array with the surface (see nrbConvSurf)
;    uknots:	an array with the knots in u-direction
;    vknots:	an array with the knots in u-direction
;
;
; KEYWORD PARAMETERS:
;    knots = a structure containg {uknots:uknots, vknots:vknots}. In this case, 
;	the inputs uknots and vknots are ignored (can be omitted). 
;    w = an array with weights
;
; OUTPUTS:
;    a structure of the type (for a surface): 
;	  nrbsurf= {coefs: coefs, $
;		xgrid:xgrid, ygrid:ygrid, zgrid:zgrid, wgrid:wgrid, $
;		dim:[nx,ny],$
;	        uknots:uknots, vknots:vknots, p:p, q:q}
;
; PROCEDURE:
;
; EXAMPLE:
;	a=nrbconvsurf('testsurf.dat')
;	b=nrbgetknots(a,uu=uu,vv=vv)
;	n1 = nrbstruct(a,b.uknots,b.vknots)
;	help,/str,n1
;	nrbplot,n1,30,30
;
;	
; MODIFICATION HISTORY:
;	written by Davide Bianchi. ESRF. 2006
;
;-	

FUNCTION nrbstruct, coefs, uknots,vknots,knots=knots,w=w

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'NURBS: nrbstruct: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   RETURN,0
endif


;==DIMENSIONS==;
dim = size(coefs)

;==SURFACE STRUCTURE==;
IF dim[0] EQ 3 THEN BEGIN  ; SURFACE

  nx=uint(dim[1])
  ny=uint(dim[3])

  ;==KNOTS RENAME==;
  IF keyword_set(knots) THEN BEGIN
    IF size(knots,/type) EQ 8 THEN BEGIN
      uknots = knots.uknots
      vknots = knots.vknots
    ENDIF ELSE BEGIN
      message,'knots keyword must be a structure of the type {uknots:arr(),vknots:arr()}'
    ENDELSE
  ENDIF
  
  nku=n_elements(uknots) ;number of knots along the U direction
  nkv=n_elements(vknots) ;number of knots along the V direction
  
  
  Xgrid=transpose(reform(coefs[*,0,*]))    ;X grid coordinates
  Ygrid=transpose(reform(coefs[*,1,*]))    ;Y grid coordinates
  Zgrid=transpose(reform(coefs[*,2,*]))    ;Z grid coordinates
  IF dim[2] EQ 3 THEN BEGIN
    IF n_elements(w) EQ 0 THEN BEGIN
      temp=dblarr(nx,4,ny)
      temp[*,0:2,*]=coefs
      temp[*,3,*]=1D
      Wgrid=transpose(reform(temp[*,3,*]))
      coefs=temp
    ENDIF ELSE BEGIN
      temp=dblarr(nx,4,ny)
      temp[*,0:2,*]=coefs
      temp[*,3,*]=w
      Wgrid=transpose(reform(temp[*,3,*]))
      coefs=temp
    ENDELSE
  ENDIF ELSE BEGIN
    Wgrid=transpose(reform(coefs[*,3,*]))    ;W grid values
  ENDELSE 

  ;==SURFACE DEGREE==;
  p = uint(nku - nx - 1)  ;along U direction
  q = uint(nkv - ny - 1)  ;along V direction

  ;==NURBS STRUCTURE
  nrbsurf= {coefs: coefs, $
	xgrid:xgrid, ygrid:ygrid, zgrid:zgrid, wgrid:wgrid, $
	dim:[nx,ny],$
        uknots:uknots, vknots:vknots, p:p, q:q}
ENDIF

;=== CURVE STRUCTURE ===;

IF dim[0] EQ 2 THEN BEGIN

  nx=uint(dim[1])


  ;==KNOTS RENAME==;
  IF keyword_set(knots) THEN BEGIN

    IF size(knots,/type) EQ 8 THEN BEGIN
      uknots = knots.uknots
    ENDIF ELSE BEGIN
      message,'knots keyword must be a structure of the type {uknots:arr()}'
    ENDELSE
  ENDIF

  nku=n_elements(uknots) ;number of knots along the U direction

  ;==CURVES ARRAYS==;
  Xgrid=transpose(reform(coefs[*,0]))    ;X grid coordinates
  Ygrid=transpose(reform(coefs[*,1]))    ;Y grid coordinates
  Zgrid=transpose(reform(coefs[*,2]))    ;Z grid coordinates
  
  IF dim[2] EQ 3 THEN BEGIN
    temp=dblarr(nx,4)
    temp[*,0:2]=coefs
    temp[*,3]=1D
    Wgrid=transpose(reform(temp[*,3]))
    
    coefs=temp
  ENDIF ELSE BEGIN
    Wgrid=transpose(reform(coefs[*,3]))    ;W grid values

  ENDELSE

  ;==SURFACE DEGREE==;
  p = uint(nku - nx - 1)  ;along U direction

  ;==NURBS STRUCTURE
  nrbsurf= {coefs: coefs, $
	xgrid:xgrid, ygrid:ygrid, zgrid:zgrid, wgrid:wgrid, $
	dim:nx, uknots:uknots, p:p}
ENDIF

return, nrbsurf

END
