#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#
typedef double Rotation[3][3];

/*******************************************************************************
* rot_test_identity: Test if rotation is identity
*******************************************************************************/
int 
rot_test_identity(Rotation t)
{
  return (t[0][0] + t[1][1] + t[2][2] == 3);
}
/*******************************************************************************
* rot_copy: Copy a rotation transformation (arrays cannot be assigned in C).
*******************************************************************************/
void
rot_copy(Rotation dest, Rotation src)
{
  int i,j;
  for(i = 0; i < 3; i++)
    for(j = 0; j < 3; j++)
      dest[i][j] = src[i][j];
}


/*******************************************************************************
* rot_mul: Matrix multiplication of transformations (this corresponds to
* combining transformations). After rot_mul(T1, T2, T3), doing T3 is
* equal to doing first T2, then T1.
* Note that T3 must not alias (use the same array as) T1 or T2.
*******************************************************************************/
void
rot_mul(Rotation t1, Rotation t2, Rotation t3)
{
  if (rot_test_identity(t1)) {
    rot_copy(t3, t2);
  } else if (rot_test_identity(t2)) {
    rot_copy(t3, t1);
  } else {
    int i,j;
    for(i = 0; i < 3; i++)
      for(j = 0; j < 3; j++)
	t3[i][j] = t1[i][0]*t2[0][j] + t1[i][1]*t2[1][j] + t1[i][2]*t2[2][j];
  }
}

/*******************************************************************************
* rot_transpose: Matrix transposition, which is inversion for Rotation matrices
*******************************************************************************/
void
rot_transpose(Rotation src, Rotation dst)
{
  dst[0][0] = src[0][0];
  dst[0][1] = src[1][0];
  dst[0][2] = src[2][0];
  dst[1][0] = src[0][1];
  dst[1][1] = src[1][1];
  dst[1][2] = src[2][1];
  dst[2][0] = src[0][2];
  dst[2][1] = src[1][2];
  dst[2][2] = src[2][2];
}

double rot_determinant(Rotation R)
{
  double d;
  d=(R[0][0]*R[1][1]*R[2][2] + R[0][1]*R[1][2]*R[2][0] + R[0][2]*R[1][0]*R[2][1] -
      R[0][2]*R[1][1]*R[2][0] - R[0][1]*R[1][0]*R[2][2] - R[0][0]*R[1][2]*R[2][1]);
  return d;
}


int
sphere_intersect(double *l0, double *l1, double x, double y, double z,
                 double kx, double ky, double kz, double r)
{/*{{{*/
  double B, C, D, k;

  k = kx*kx + ky*ky + kz*kz;
  B = (x*kx + y*ky + z*kz);
  C = x*x + y*y + z*z - r*r;
  D = B*B - k*C;
  if(D < 0)
    return 0;
  D = sqrt(D);
  *l0 = (-B - D) / sqrt(k);
  *l1 = (-B + D) / sqrt(k);
  return 1;
} /* sphere_intersect *//*}}}*/
int
ellipsoid_intersect(double *l0, double *l1, double x, double y, double z,
    double kx, double ky, double kz, double a, double b, double c,
    Rotation Q)
{/*{{{*/
  Rotation A,Gamma,Q_t,Tmp;
  double u,v,w;
  
  /*set up the gamma matrix first to zeros*/ 
  Gamma[0][0]=Gamma[0][1]=Gamma[0][2]=0;
  Gamma[1][0]=Gamma[1][1]=Gamma[1][2]=0;
  Gamma[2][0]=Gamma[2][1]=Gamma[2][2]=0;
  /*now set diagonal to ellipsoid half axis if non-zero.
   * This way a zero value mean the sllipsoid extends infinitely along that axis,
   * which is useful for objects only curved in one direction*/ 
  if (a!=0){
    Gamma[0][0]=1/(a*a);
  }
  if (b!=0){
    Gamma[1][1]=1/(b*b);
  }
  if (c!=0){
    Gamma[2][2]=1/(c*c);
  }

  if (Q!=NULL){
    rot_transpose(Q,Q_t);
    rot_mul(Gamma,Q_t,Tmp);
    rot_mul(Q,Tmp,A);
  }else{
    rot_copy(A,Gamma);
  }
  /*to get the solutions as lengths in m use unit vector along k*/
  double ex,ey,ez,k;
  k=sqrt(kx*kx+ky*ky+kz*kz);
  ex=kx/k;
  ey=ky/k;
  ez=kz/k;

  u=ex*(A[0][0]*ex + A[1][0]*ey + A[2][0]*ez) + ey*( A[0][1]*ex + A[1][1]*ey + A[2][1]*ez) + ez*(A[0][2]*ex + A[1][2]*ey + A[2][2]*ez);
  v=x *(A[0][0]*ex + A[1][0]*ey + A[2][0]*ez) + ex*(A[0][0]*x + A[1][0]*y + A[2][0]*z) +
    y *(A[0][1]*ex + A[1][1]*ey + A[2][1]*ez) + ey*(A[0][1]*x + A[1][1]*y + A[2][1]*z) +
    z *(A[0][2]*ex + A[1][2]*ey + A[2][2]*ez) + ez*(A[0][2]*x + A[1][2]*y + A[2][2]*z);
  w=x*(A[0][0]*x + A[1][0]*y + A[2][0]*z) + y*(A[0][1]*x + A[1][1]*y + A[2][1]*z) + z*(A[0][2]*x + A[1][2]*y + A[2][2]*z);

  double D=v*v-4*u*w+4*u;
  if (D<0) return 0;

  D=sqrt(D);

  *l0=(-v-D) / (2*u);
  *l1=(-v+D) / (2*u);
  return 1;
  
}/*}}}*/

main(){
  double kx,ky,kz,l0,l1,x,y,z,R,a,b,c;

  kx=ky=0;kz=200;
  sphere_intersect(&l0,&l1,0,0,-1,kx,ky,kz,0.1);
  printf("%g %g\n",l0,l1);
  
  ellipsoid_intersect(&l0,&l1,0,0,-1,kx,ky,kz,0.1,0,0.1, NULL);
  printf("%g %g\n",l0,l1);

  kx=ky=0;kz=2;
  sphere_intersect(&l0,&l1,0,0,-1,kx,ky,kz,0.11);
  printf("%g %g\n",l0,l1);

  Rotation q;
  double xi=1.2*M_PI/180;
  q[0][0]=1;q[1][0]=0;q[2][0]=0;
  q[0][1]=0;q[1][1]=cos(xi);q[2][1]=sin(xi);
  q[0][2]=0;q[1][2]=-sin(xi);q[2][2]=cos(xi);

  ellipsoid_intersect(&l0,&l1,0,0,-1,kx,ky,kz,0,0.11,0.11, q);
  printf("%g %g\n",l0,l1);

  /*test cases*/
  
  q[0][0]=1;q[1][0]=0;q[2][0]=0;
  q[0][1]=0;q[1][1]=1;q[2][1]=0;
  q[0][2]=0;q[1][2]=0;q[2][2]=1;
  
  ellipsoid_intersect(&l0,&l1,0,0-0.4999,-1-2,kx,ky,kz,0,0.5,2, q);
  printf("1: %g %g\n",l0,l1);

  q[0][0]=1;q[1][0]=0;q[2][0]=0;
  q[0][1]=0;q[1][1]=0;q[2][1]=1;
  q[0][2]=0;q[1][2]=-1;q[2][2]=0;
  
  ellipsoid_intersect(&l0,&l1,0,0-1.9999,-1-2,kx,ky,kz,0,0.5,2, q);
  printf("2: %g %g\n",l0,l1);



}
