/* termios.cc: termios for WIN32.

   Copyright 1996, 1997, 1998 Cygnus Solutions.

   Written by Doug Evans and Steve Chamberlain of Cygnus Support
   dje@cygnus.com, sac@cygnus.com

This file is part of Cygwin.

This software is a copyrighted work licensed under the terms of the
Cygwin license.  Please consult the file "CYGWIN_LICENSE" for
details. */

#include <sys/termios.h>
#include <errno.h>
#include "winsup.h"

/* tcsendbreak: POSIX 7.2.2.1 */
extern "C"
int
tcsendbreak (int fd, int duration)
{
  int res = -1;

  if (dtable.not_open (fd))
    set_errno (EBADF);
  else if (!dtable[fd]->is_tty ())
    set_errno (ENOTTY);
  else
    res = dtable[fd]->tcsendbreak (duration);

  syscall_printf ("%d = tcsendbreak (%d, %d )", res, fd, duration);
  return res;
}

/* tcdrain: POSIX 7.2.2.1 */
extern "C"
int
tcdrain (int fd)
{
  int res = -1;

  termios_printf ("tcdrain");

  if (dtable.not_open (fd))
    set_errno (EBADF);
  else if (!dtable[fd]->is_tty ())
    set_errno (ENOTTY);
  else
    res = dtable[fd]->tcdrain ();

  syscall_printf ("%d = tcdrain (%d)", res, fd);
  return res;
}

/* tcflush: POSIX 7.2.2.1 */
extern "C"
int
tcflush (int fd, int queue)
{
  int res = -1;

  if (dtable.not_open (fd))
    set_errno (EBADF);
  else if (!dtable[fd]->is_tty ())
    set_errno (ENOTTY);
  else
    res = dtable[fd]->tcflush (queue);

  termios_printf ("%d = tcflush (%d, %d)", res, fd, queue);
  return res;
}

/* tcflow: POSIX 7.2.2.1 */
extern "C"
int
tcflow (int fd, int action)
{
  int res = -1;

  if (dtable.not_open (fd))
    set_errno (EBADF);
  else if (!dtable[fd]->is_tty ())
    set_errno (ENOTTY);
  else
    {
      sig_protect (here, 1);
      res = dtable[fd]->tcflow (action);
    }

  syscall_printf ("%d = tcflow (%d, %d)", res, fd, action);
  return res;
}

/* tcsetattr: POSIX96 7.2.1.1 */
extern "C"
int
tcsetattr (int fd, int a, const struct termios *t)
{
  int res = -1;

  if (dtable.not_open (fd))
    set_errno (EBADF);
  else if (!dtable[fd]->is_tty ())
    set_errno (ENOTTY);
  else
    {
      sig_protect (here, 1);
      res = dtable[fd]->tcsetattr (a, t);
    }

  termios_printf ("iflag %x, oflag %x, cflag %x, lflag %x",
	t->c_iflag, t->c_oflag, t->c_cflag, t->c_lflag);
  termios_printf ("%d = tcsetattr (%d, %d, %x)", res, fd, a, t);
  return res;
}

/* tcgetattr: POSIX 7.2.1.1 */
extern "C"
int
tcgetattr (int fd, struct termios *t)
{
  int res = -1;

  if (dtable.not_open (fd))
    set_errno (EBADF);
  else if (!dtable[fd]->is_tty ())
    set_errno (ENOTTY);
  else
    {
      sig_protect (here, 1);
      res = dtable[fd]->tcgetattr (t);
    }

  termios_printf ("%d = tcgetattr (%d, %x)", res, fd, t);
  return res;
}

/* tcgetpgrp: POSIX 7.2.3.1 */
extern "C"
int
tcgetpgrp (int fd)
{
  int res = -1;

  if (dtable.not_open (fd))
    set_errno (EBADF);
  else if (!dtable[fd]->is_tty ())
    set_errno (ENOTTY);
  else
    res = dtable[fd]->tcgetpgrp ();

  termios_printf ("%d = tcgetpgrp (%d)", res, fd);
  return res;
}

/* tcsetpgrp: POSIX 7.2.4.1 */
extern "C"
int
tcsetpgrp (int fd, pid_t pgid)
{
  int res = -1;

  if (dtable.not_open (fd))
    set_errno (EBADF);
  else if (!dtable[fd]->is_tty ())
    set_errno (ENOTTY);
  else
    res = dtable[fd]->tcsetpgrp (pgid);

  termios_printf ("%d = tcsetpgrp (%d, %x)", res, fd, pgid);
  return res;
}

/* NIST PCTS requires not macro-only implementation */
#undef cfgetospeed
#undef cfgetispeed
#undef cfsetospeed
#undef cfsetispeed

/* cfgetospeed: POSIX96 7.1.3.1 */
extern "C"
speed_t
cfgetospeed (struct termios *tp)
{
  return tp->c_ospeed;
}

/* cfgetispeed: POSIX96 7.1.3.1 */
extern "C"
speed_t
cfgetispeed (struct termios *tp)
{
  return tp->c_ispeed;
}

/* cfsetospeed: POSIX96 7.1.3.1 */
extern "C"
int
cfsetospeed (struct termios *tp, speed_t speed)
{
  tp->c_ospeed = speed;
  return 0;
}

/* cfsetispeed: POSIX96 7.1.3.1 */
extern "C"
int
cfsetispeed (struct termios *tp, speed_t speed)
{
  tp->c_ispeed = speed;
  return 0;
}

#if 0
static void
tdump (int)
{
  termios_printf ("fd %d rb %d wb %d len %d time %d",
		  fd,
		  myself->hmap[fd].r_binary,
		  myself->hmap[fd].w_binary,
		  myself->hmap[fd].vmin,
		  myself->hmap[fd].vtime);
}
#endif

#if 0
static void
ds (char *when, DCB *stty)
{
  termios_printf ("DCB state %s", when);
  termios_printf ("DCBlength %x", stty->DCBlength);
  termios_printf ("BaudRate; %d", stty->BaudRate);
  termios_printf ("fBinary; %d", stty->fBinary);
  termios_printf ("fParity; %d", stty->fParity);
  termios_printf ("fOutxCtsFlow	%d", stty->fOutxCtsFlow);
  termios_printf ("fOutxDsrFlow	%d", stty->fOutxDsrFlow);
  termios_printf ("fDtrControl %d", stty->fDtrControl);
  termios_printf ("fDsrSensitivity %d", stty->fDsrSensitivity);
  termios_printf ("fTXContinueOnXoff %d", stty->fTXContinueOnXoff);
  termios_printf ("1; %d", stty->fOutX);
  termios_printf ("1; %d", stty->fInX);
  termios_printf ("1; %d", stty->fErrorChar);
  termios_printf ("1; %d", stty->fNull);
  termios_printf ("fRtsControl %d", stty->fRtsControl);
  termios_printf ("fAbortOnError %d", stty->fAbortOnError);
  termios_printf ("%d", stty->fDummy2);
  termios_printf ("res1	%d", stty->res1);
  termios_printf ("XonLim; %d", stty->XonLim);
  termios_printf ("XoffLim; %d", stty->XoffLim);
  termios_printf ("ByteSize; %d", stty->ByteSize);
  termios_printf ("Parity; %d", stty->Parity);
  termios_printf ("StopBits; %d", stty->StopBits);
  termios_printf ("XonChar; %d", stty->XonChar);
  termios_printf ("XoffChar; %d", stty->XoffChar);
  termios_printf ("ErrorChar; %d", stty->ErrorChar);
  termios_printf ("EofChar; %d", stty->EofChar);
  termios_printf ("EvtChar; %d", stty->EvtChar);
  termios_printf ("res2	%d", stty->res2);
}
#endif
