/*
 * addshgrp.c: Add program manager group for SHADOW tools.
 *
 * Ideas and code lifted from GNU NT-Emacs distribution.
 *
 * ------------------------------------------------
 * Mumit Khan <khan@xraylith.wisc.edu>
 * Center for X-ray Lithography
 * University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 * Copyright (c) 1997 Mumit Khan
 */

#ifndef _WIN32
# error This program can only be compiled under Windows32 API.
#endif

#include <windows.h>
#include <ddeml.h>
#include <stdlib.h>
#include <stdio.h>

#ifdef __CYGWIN__
#include <limits.h>
#include <string.h>
#include <ctype.h>
extern int cygwin_conv_to_full_win32_path (const char *path, char *win32_path);
#endif

#ifndef SHADOW_VERSION
# error Must define SHADOW_VERSION via Makefile.
#endif

#ifndef SCRIPT_EXTENSION
# define SCRIPT_EXTENSION ""
#endif

HDDEDATA CALLBACK 
DdeCallback (UINT uType, UINT uFmt, HCONV hconv,
	     HSZ hsz1, HSZ hsz2, HDDEDATA hdata,
	     DWORD dwData1, DWORD dwData2)
{
    return ((HDDEDATA) NULL);
}

#define DdeCommand(str) 	\
	DdeClientTransaction (str, strlen (str)+1, HConversation, (HSZ)NULL, \
		              CF_TEXT, XTYP_EXECUTE, 30000, NULL)


/*
 * get_shroot_from_registry
 * 
 * returns:
 *    0 : all ok
 *    1 : major key not found
 *    2 : string value not found
 *    3 : other error
 */
int get_shroot_from_registry (char* value, size_t max_len) {

    DWORD size;
    HKEY key;

    const char *major_key = "Software\\CXrL\\SHADOW\\" SHADOW_VERSION;
    const char *var = "SHADOW_ROOT";

    if (RegOpenKeyEx (HKEY_LOCAL_MACHINE,
     		      major_key,
		      0,
		      KEY_READ, &key) != ERROR_SUCCESS)
    {
	key = INVALID_HANDLE_VALUE;
        return 1;
    }
    if (key == INVALID_HANDLE_VALUE)
        return 1;

    if (RegQueryValueEx(key, var, NULL, NULL, NULL, &size) == ERROR_SUCCESS) {
        if (size >= max_len)
	    return 3;
        RegQueryValueEx(key, var, NULL, NULL, value, &size);
    } else {
        return 2;
    }
    return 0;
}

int main (int argc, char *argv[]) {

    DWORD idDde = 0;
    HCONV HConversation;
    HSZ ProgMan;
    char additem[MAX_PATH*2 + 100];
    char shadow_root[MAX_PATH];
    char *p;

    /* If no args specified, use our location to set shadow_root.  */

    if (argc > 1) {
        strcpy (shadow_root, argv[1]);
    } else if ((p = getenv("SHADOW_ROOT"))) {
        strcpy (shadow_root, p);
    } else {
	if (get_shroot_from_registry (shadow_root, MAX_PATH)) {
	    MessageBox (NULL, 
	        "Error:Can't determine SHADOW root directory from Registry!",
	        "Install SHADOW icons", MB_OK | MB_ICONSTOP);
	    exit (1);
        }
    }

#ifdef __CYGWIN__
  /* We need to make sure that path to SHADOW_ROOT is native! */
    {
	char tmpbuf[ MAX_PATH + 1];
        if (! cygwin_conv_to_full_win32_path (shadow_root, tmpbuf))
	    strcpy (shadow_root, tmpbuf);
    }
#endif /* __CYGWIN__ */

    for (p = shadow_root; p && *p; p++)
      if (*p == '/')
        *p = '\\';


    /* Tell user what we are going to do.  */
    {
	char msg[ MAX_PATH ];
	sprintf (msg, "Add a program group SHADOW v" SHADOW_VERSION
		 " in the Taskbar using Installation Path: \"%s\"?\n\n"
		 "This is useful if you are installing SHADOW manually "
		 "instead of using the InstallShield installer. If you "
		 "have installed SHADOW using the installer, then you "
		 "probably do not want to do this unless you want to "
		 "repair your Taskbar/Start Menu icons/shortcuts.",
	         shadow_root);
	if (MessageBox (NULL, msg, "Install SHADOW icons", 
	    MB_OKCANCEL | MB_ICONQUESTION) == IDCANCEL)
	{
	    exit (1);
	}
    }

    DdeInitialize (&idDde, (PFNCALLBACK)DdeCallback, APPCMD_CLIENTONLY, 0);

    ProgMan = DdeCreateStringHandle (idDde, "PROGMAN", CP_WINANSI);

    HConversation = DdeConnect (idDde, ProgMan, ProgMan, NULL);
    if (HConversation != 0) {
	DdeCommand ("[CreateGroup (\"SHADOW v" SHADOW_VERSION "\")]");
	DdeCommand ("[ReplaceItem (\"Shadow GUI\")]");
	sprintf (additem, 
	         "[AddItem (\"%s\\bin\\shadow.exe\", \"Shadow GUI\")]",
	         shadow_root);
        DdeCommand (additem);
	DdeCommand ("[ReplaceItem (\"Shadow Plot\")]");
	sprintf (additem, 
	         "[AddItem (\"%s\\bin\\shplot.exe\", \"Shadow Plot\")]",
	         shadow_root);
        DdeCommand (additem);
	DdeCommand ("[ReplaceItem (\"What's New\")]");
	sprintf (additem, 
	         "[AddItem (\"%s\\NEWS.txt\", \"What's New\")]",
	         shadow_root);
        DdeCommand (additem);
	DdeCommand ("[ReplaceItem (\"GO Shell\")]");
	sprintf (additem, 
	         "[AddItem (\"%s\\bin\\cygwish80\" \"%s\\lib\\shadow\\launch.tcl\" -- GO%s, \"GO Shell\")]",
	         shadow_root, shadow_root, SCRIPT_EXTENSION);
        DdeCommand (additem);
	DdeCommand ("[ReplaceItem (\"Unix Shell\")]");
	sprintf (additem, 
	         "[AddItem (\"%s\\bin\\cygwish80\" \"%s\\lib\\shadow\\launch.tcl\" -- -shell, \"Unix Shell\")]",
	         shadow_root, shadow_root);
        DdeCommand (additem);
	DdeCommand ("[ReplaceItem (\"DOS Shell\")]");
	sprintf (additem, 
	         "[AddItem (\"%s\\bin\\cygwish80\" \"%s\\lib\\shadow\\launch.tcl\" -- -dosshell, \"DOS Shell\")]",
	         shadow_root, shadow_root);
        DdeCommand (additem);

        DdeDisconnect (HConversation);
    }

    DdeFreeStringHandle (idDde, ProgMan);

    DdeUninitialize (idDde);

    return (0);
}
