//
// tool_cmds.cc: Tcl commands related to Tools/ToolInstances.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994 Mumit Khan
//

#include <cassert>
#include <cstdlib>
#include <cstring>

#include "tool.h"
#include "value.h"
#include "variable.h"

#include "commands.h"
#include "xmenu.h"

#if NEED_DECLARATION_STRCASECMP
extern "C" int strcasecmp(const char*, const char*);
#endif
#if NEED_DECLARATION_STRNCASECMP
extern "C" int strncasecmp(const char*, const char*, std::size_t);
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

// ================ Tcl commands created here ======================== //
/*
 * ALL THE COMMANDS to "TOOL" in this interpreter:
 *
 * tool list
 * tool variables <toolname>
 * tool vget <toolname> <var>
 *
 */

// =================================================================== //

// =================================================================== //

static char g_buf[BUFSIZ];

// =================================================================== //

static inline char* string_to_charptr(const string& str) {
    return const_cast<char*>(str.c_str());
}

// =================================================================== //

static int list_tools (Tcl_Interp* interp) {
    const list<string>& tools = GLB_toolmgr->tools();
    list<string>::const_iterator it = tools.begin();
    for(; it != tools.end(); ++it) {
	const string& this_tool = *it;
	Tcl_AppendElement(interp, string_to_charptr(this_tool));
    }
    return TCL_OK;
}

static int list_tool_variables (Tcl_Interp* interp, char* toolname) {
    const Tool* tool = GLB_toolmgr->find(toolname);
    if (tool == nil) {
	Tcl_AppendResult (interp, 
	    "Tool \"", toolname, "\" does not exist.",
	    (char*) NULL
        );
	return TCL_ERROR;
    }
    const list<string>& variables = tool->variables();
    list<string>::const_iterator it = variables.begin();
    for(; it != variables.end(); ++it) {
	const string& this_variable = *it;
	Tcl_AppendElement(interp, string_to_charptr(this_variable));
    }
    return TCL_OK;
}

static int get_tool_variable (Tcl_Interp* interp, char* toolname, char* var) {
    const Tool* tool = GLB_toolmgr->find(toolname);
    if (tool == nil) {
	Tcl_AppendResult (interp, 
	    "Tool \"", toolname, "\" does not exist.",
	    (char*) NULL
        );
	return TCL_ERROR;
    }
    const Var* variable = tool->get_var(var);
    if (var == nil) {
	Tcl_AppendResult (interp, 
	    "No such variable \"", var, "\" in Tool \"", toolname, "\"",
	    (char*) NULL
        );
	return TCL_ERROR;
    }
    Tcl_AppendResult(interp, 
	const_cast<char*>(variable->getval()->getstrval()),
	(char*) NULL
    );
    return TCL_OK;
}

// =================================================================== //

/*
 *--------------------------------------------------------------
 *
 * XMenu_ToolCmd:
 *
 *      This procedure is invoked to process the "tool" Tcl
 *      command.  
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *--------------------------------------------------------------
 */

int XMenu_ToolCmd (ClientData, Tcl_Interp* interp, int argc, char** argv) {
    if (GLB_toolmgr == nil) {
	Tcl_AppendResult(interp, 
	    "Error: Tool Manager is not yet initialized!", (char*) NULL
	);
	return TCL_ERROR;
    }

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " option [args..]\"", (char *) NULL);
	return TCL_ERROR;
    }
    /*
     * tool list
     *
     */
    if (argv[1][0] == 'l' && strncasecmp(argv[1], "li", 2) == 0) {
        if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		    argv[0], " list\"", (char *) NULL);
	    return TCL_ERROR;
	}
        return list_tools(interp);
    }

    /*
     * tool variables <toolname>
     */
    if (argv[1][0] == 'v' && strncasecmp(argv[1], "va", 2) == 0) {
        if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " variables <toolname>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	} 
	return list_tool_variables(interp, argv[2]);
    }
    /*
     * tool vget <toolname> <var>
     */
    if (argv[1][0] == 'v' && strncasecmp(argv[1], "vg", 2) == 0) {
        if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"", 
		argv[0], " vget <toolname> <variable>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	} 
	return get_tool_variable(interp, argv[2], argv[3]);
    }
    Tcl_AppendResult(interp, "bad option \"", argv[1],
	"\":  should be list, variables, vget\"", (char*) NULL
    );
    return TCL_ERROR;
}
