//
// shdata_cmds.cc: Tcl commands for reading/writing SHADOW image files.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1996 Mumit Khan
//

// ======================================================================= //

#include <cassert>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

#include <string>
#include <algorithm>

#include "shdata_cmds.h"
#include "shdata_io.h"

#if NEED_DECLARATION_STRCASECMP
extern "C" int strcasecmp(const char*, const char*);
#endif
#if NEED_DECLARATION_STRNCASECMP
extern "C" int strncasecmp(const char*, const char*, std::size_t);
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

// ======================================================================= //

static int initialized = 0;
static Tcl_HashTable shdata_table;
static char buf[2048];

static const double TOCM = 1.239852e-4;
static const double TWOPI = 6.283185307179586467925287;

static const int AUTOMATIC_LIMIT = 0;
static const int CARTESIAN_LIMIT = 1;
static const int FILL_LIMIT = 2;

static const int BINSIZE = 25;

static const double FILL_FACTOR = 1.2;

// ======================================================================= //

#if (__GNUG__ && !USE_GNU_FREPO) || CXX_INSTANTIATE_TEMPLATES
template double const& min<>(double const&, double const&);
template double const& max<>(double const&, double const&);
#endif

// ======================================================================= //

static inline bool column_bad(int column) {
    bool badcol = 
	column < RAY_X && column > RAY_Z_PRIME &&
        column != RAY_LAMBDA && column != RAY_NA;
    return badcol;
}

static inline bool column_ok(int column) {
    return !column_bad(column);
}

//
// coordinate/limit routines. Modeled after ROUNDOUT in PLOTXY (FORTRAN)
//
static int compute_limits (double& xmin, double& xmax) {
    const double xdel = xmax - xmin;
    if (xdel <= 1.0e-15)		// CHECK/FIXME: (not) too close?
	return 0;

    int i0 = static_cast<int>(log10(xdel/2.0));
    if (i0 < 1) 
        i0 -= 1;
    double xrange = pow(10.0, i0);
    int imax = static_cast<int>(xmax/xrange) + 1;
    int imin = static_cast<int>(xmin/xrange) - 1;

    imax = static_cast<int>(imax/5.0);
    imin = static_cast<int>(imin/5.0);

    if (imax >= 0) imax += 1;
    if (imin <= 0) imin -= 1;

    imax = static_cast<int>(imax * 5.0);
    imin = static_cast<int>(imin * 5.0);

    xmax = imax * xrange;
    xmin = imin * xrange;
    return 0;
}

static int compute_cartesian_limits (
    const double xmin, const double xmax, 
    const double ymin, const double ymax, 
    double& xmin_out, double& xmax_out,
    double& ymin_out, double& ymax_out
) {
    double x_center = (xmin + xmax)/2.0;
    double y_center = (ymin + ymax)/2.0;
    double xdel = xmax - xmin;
    double ydel = ymax - ymin;
    double del = max(xdel, ydel);
    xmin_out = x_center - 0.5 * del;
    xmax_out = x_center + 0.5 * del;
    ymin_out = y_center - 0.5 * del;
    ymax_out = y_center + 0.5 * del;

    compute_limits(xmin_out, xmax_out);
    compute_limits(ymin_out, ymax_out);
    return 0;
}

static int compute_fill_cartesian_limits (
    const double xmin, const double xmax, 
    const double ymin, const double ymax, 
    double& xmin_out, double& xmax_out,
    double& ymin_out, double& ymax_out,
    const double& fill_factor
) {
    double x_center = (xmin + xmax)/2.0;
    double y_center = (ymin + ymax)/2.0;
    double xdel = xmax - xmin;
    double ydel = ymax - ymin;
    double del = fill_factor * max(xdel, ydel);

    xmin_out = x_center - 0.5 * del;
    xmax_out = x_center + 0.5 * del;
    ymin_out = y_center - 0.5 * del;
    ymax_out = y_center + 0.5 * del;

    return 0;
}

static int compute_auto_limits (
    const double xmin, const double xmax, 
    const double ymin, const double ymax, 
    double& xmin_out, double& xmax_out,
    double& ymin_out, double& ymax_out
) {
    xmin_out = xmin;
    xmax_out = xmax;
    ymin_out = ymin;
    ymax_out = ymax;
    compute_limits(xmin_out, xmax_out);
    compute_limits(ymin_out, ymax_out);
    return 0;
}

static int compute_data_limits(Tcl_Interp* interp, ShadowData* shdata,
    int column1, int column2, int limit, double fill_factor
) {
    if (column_bad(column1) || column_bad(column2)) {
	sprintf(buf, "Column %d and/or %d out of range", column1, column2);
        Tcl_AppendResult(interp, buf, 0);
        return TCL_ERROR;
    }
    --column1;
    --column2;
    const int which = shdata->which_rays;
    double xmin = shdata->column_info[column1][which][1];
    double xmax = shdata->column_info[column1][which][2];
    double ymin = shdata->column_info[column2][which][1];
    double ymax = shdata->column_info[column2][which][2];

    if (limit == AUTOMATIC_LIMIT) {
	compute_auto_limits (
	    xmin, xmax, ymin, ymax,		/* input            */
	    xmin, xmax, ymin, ymax		/*   output to same */
	);
    } else if (limit == CARTESIAN_LIMIT) {
	compute_cartesian_limits (
	    xmin, xmax, ymin, ymax,		/* input            */
	    xmin, xmax, ymin, ymax		/*   output to same */
	);
    } else if (limit == FILL_LIMIT) {
	compute_fill_cartesian_limits (
	    xmin, xmax, ymin, ymax,		/* input            */
	    xmin, xmax, ymin, ymax,		/*   output to same */
	    fill_factor
	);
    }
    sprintf (buf, "%12.5E %12.5E %12.5E %12.5E", xmin, xmax, ymin, ymax);
    Tcl_AppendResult(interp, buf, 0);
    return TCL_OK;
}

static int compute_data_limits(Tcl_Interp* interp,
    double& xmin, double& xmax, double& ymin, double& ymax, 
    int limit, double fill_factor
) {
    if (limit == AUTOMATIC_LIMIT) {
	compute_auto_limits (
	    xmin, xmax, ymin, ymax,		/* input            */
	    xmin, xmax, ymin, ymax		/*   output to same */
	);
    } else if (limit == CARTESIAN_LIMIT) {
	compute_cartesian_limits (
	    xmin, xmax, ymin, ymax,		/* input            */
	    xmin, xmax, ymin, ymax		/*   output to same */
	);
    } else if (limit == FILL_LIMIT) {
	compute_fill_cartesian_limits (
	    xmin, xmax, ymin, ymax,		/* input            */
	    xmin, xmax, ymin, ymax,		/*   output to same */
	    fill_factor
	);
    }
    sprintf (buf, "%12.5E %12.5E %12.5E %12.5E", xmin, xmax, ymin, ymax);
    Tcl_AppendResult(interp, buf, 0);
    return TCL_OK;
}

/*
 * compute_minmax: Column is 0 based here!
 * compuTE The min and max info for ALL three types of rays -- Good, Lost
 * and All and store these back into shdata->column_info mess.
 */

static int compute_column_minmax (Tcl_Interp* interp, ShadowData* shdata,
    int column, bool compute_variance = true
) {
    double* data = 0;
    if (extract_shadow_raw_column(interp, shdata, column+1, &data) !=
	TCL_OK
    ){
	Tcl_AppendResult(interp, "Error getting raw data", 0);
	return TCL_ERROR;
    }
    /* one element per type of ray -- good, lost and all */
    double minval[3];
    double maxval[3];
    double sum[3];
    double variance[3];
    int npcol[3];

    minval[0] = minval[1] = minval[2] =  1.0e+20;
    maxval[0] = maxval[1] = maxval[2] = -1.0e+20;
    sum[0] = sum[1] = sum[2] = 0.0;
    variance[0] = variance[1] = variance[2] = 0.0;
    npcol[0] = npcol[1] = npcol[2] = 0;

    for (int pt = 0; pt < shdata->npoints; ++pt) {
	const double& ptval = data[pt];
	const double ptval_square = ptval * ptval;

	/* do the ALL_RAYS case first */
	++npcol[ALL_RAYS];
	minval[ALL_RAYS] = min(minval[ALL_RAYS], ptval);
	maxval[ALL_RAYS]= max(maxval[ALL_RAYS], ptval);
	if (compute_variance) {
	    sum[ALL_RAYS] += ptval;
	    variance[ALL_RAYS] += ptval_square;
	}

	bool lost = (column_of_point(shdata, pt, RAY_GOOD-1) < 0.0);
	if (!lost) {
	    ++npcol[GOOD_RAYS];
	    minval[GOOD_RAYS] = min(minval[GOOD_RAYS], ptval);
	    maxval[GOOD_RAYS]= max(maxval[GOOD_RAYS], ptval);
	    if (compute_variance) {
		sum[GOOD_RAYS] += ptval;
		variance[GOOD_RAYS] += ptval_square;
	    }
	} else {
	    ++npcol[LOST_RAYS];
	    minval[LOST_RAYS] = min(minval[LOST_RAYS], ptval);
	    maxval[LOST_RAYS]= max(maxval[LOST_RAYS], ptval);
	    if (compute_variance) {
		sum[LOST_RAYS] += ptval;
		variance[LOST_RAYS] += ptval_square;
	    }
	}
    }

    for (int which = GOOD_RAYS; which <= ALL_RAYS; ++which) {
	double mean = 0.0;
	double stddev = 0.0;

	if (npcol[which] > 0) {
	    if (compute_variance) {
		mean = sum[which]/npcol[which];
		variance[which] = variance[which]/npcol[which] - mean * mean;
		stddev = (variance[which] >= 0.0) ? sqrt(variance[which]) : 0;
	    }
	} else {
	    minval[which] = maxval[which] = mean = stddev = 0.0;
	}

	shdata->column_info[column][which][0] = npcol[which];
	shdata->column_info[column][which][1] = minval[which];
	shdata->column_info[column][which][2] = maxval[which];
	if (compute_variance) {
	    shdata->column_info[column][which][3] = mean;
	    shdata->column_info[column][which][4] = stddev;
	}
    }

    return TCL_OK;
}

static int compute_NA_minmax (Tcl_Interp* interp, ShadowData* shdata) {
    /* one element per type of ray -- good, lost and all */
    double minval[3];
    double maxval[3];
    int npcol[3];

    minval[0] = minval[1] = minval[2] =  1.0e+20;
    maxval[0] = maxval[1] = maxval[2] = -1.0e+20;
    npcol[0] = npcol[1] = npcol[2] = 0;

    for (int pt = 0; pt < shdata->npoints; ++pt) {
	const double col4 = column_of_point(shdata, pt, 3);
	const double col5 = column_of_point(shdata, pt, 4);
	const double col6 = column_of_point(shdata, pt, 5);
	double thisval = sqrt(col4 * col4 + col6 * col6);
	thisval = fabs(thisval / col5);

	/* do the ALL_RAYS first */
	++npcol[ALL_RAYS];
	minval[ALL_RAYS] = min(minval[ALL_RAYS], thisval);
	maxval[ALL_RAYS]= max(maxval[ALL_RAYS], thisval);

	bool lost = (column_of_point(shdata, pt, RAY_GOOD-1) < 0.0);
	if (!lost) {				/* Good rays */
	    ++npcol[GOOD_RAYS];
	    minval[GOOD_RAYS] = min(minval[GOOD_RAYS], thisval);
	    maxval[GOOD_RAYS]= max(maxval[GOOD_RAYS], thisval);
	} else {
	    ++npcol[LOST_RAYS];
	    minval[LOST_RAYS] = min(minval[LOST_RAYS], thisval);
	    maxval[LOST_RAYS]= max(maxval[LOST_RAYS], thisval);
	}
    }

    for (int which = GOOD_RAYS; which <= ALL_RAYS; ++which) {
	if (npcol[which] == 0)
	    minval[which] = maxval[which] = 0.0;

	const int column = RAY_NA-1;
	shdata->column_info[column][which][0] = npcol[which];
	shdata->column_info[column][which][1] = minval[which];
	shdata->column_info[column][which][2] = maxval[which];
    }

    return TCL_OK;
}

static char* column_names[] = {
    "X ",
    "Y ",
    "Z ",
    "X'",
    "Y'",
    "Z'",
    "?",
    "?",
    "?",
    "?",
    "Photon Energy (eV)",
    "?",
    "?",
    "?",
    "?",
    "?",
    "?",
    "?",
    "?",
    "Numerical Aperture"
};

static int get_minmax_info (Tcl_Interp* interp, 
    const ShadowData* shdata, int which
) {
    strcpy(buf, "Col  NpCol  Par    Minimum       Maximum      Center"
        "      Std. Dev.\n"
    );
    strcat(buf, "===================================================="
        "================\n"
    );
    Tcl_AppendResult (interp, buf, 0);
    for (int column = 0; column <= 5; ++column) {
	sprintf(buf, "%-2d %6d    %s  %12.5E %12.5E %12.5E %12.5E\n",
	    column + 1,
	    static_cast<int>(shdata->column_info[column][which][0]),
	    column_names[column],
	    shdata->column_info[column][which][1],
	    shdata->column_info[column][which][2],
	    shdata->column_info[column][which][3],
	    shdata->column_info[column][which][4]
	);
	Tcl_AppendResult (interp, buf, 0);
    }

    sprintf(buf, "%-2d %6d %s\n"
        "                 %12.5E %12.5E\n",
	RAY_LAMBDA,
	static_cast<int>(shdata->column_info[RAY_LAMBDA-1][which][0]),
	column_names[RAY_LAMBDA-1],
	shdata->column_info[RAY_LAMBDA-1][which][1] * TOCM / TWOPI,
	shdata->column_info[RAY_LAMBDA-1][which][2] * TOCM / TWOPI
    );
    Tcl_AppendResult (interp, buf, 0);

    sprintf(buf, "%-2d %6d %s\n"
        "                 %12.5E %12.5E\n",
	RAY_NA,
	static_cast<int>(shdata->column_info[RAY_NA-1][which][0]),
	column_names[RAY_NA-1],
	shdata->column_info[RAY_NA-1][which][1],
	shdata->column_info[RAY_NA-1][which][2]
    );
    Tcl_AppendResult (interp, buf, 0);
    return TCL_OK;
}

static int compute_minmax(Tcl_Interp* interp, ShadowData* shdata) {
    assert(shdata != 0);
    for (int column = 0; column <= 5; ++column) {
	if (compute_column_minmax(interp, shdata, column) != TCL_OK)
	    return TCL_ERROR;
    }
    if (shdata->ncol > 12) {
	/* now column 13, which is needed for OPD stuff */
	if (
	    compute_column_minmax(interp, shdata, RAY_OPD-1, false
	    ) != TCL_OK
	) {
	    return TCL_ERROR;
	}
    }

    /* now the photon energy, but only need min/max for this one. */
    if (
	compute_column_minmax(interp, shdata, RAY_LAMBDA-1, false
	) != TCL_OK
    ) {
	return TCL_ERROR;
    }

    /* now the NA. Don't understand this yet, but give me time ... */
    if (compute_NA_minmax(interp, shdata) != TCL_OK)
	return TCL_ERROR;
    
    return 0;
}

/*
 *
 * list_shdata:
 *
 */
static int list_shdata(Tcl_Interp* interp) {
    Tcl_HashEntry* entry_ptr;
    Tcl_HashSearch search;

    for(
	entry_ptr = Tcl_FirstHashEntry(&shdata_table, &search); entry_ptr != 0; 
	entry_ptr = Tcl_NextHashEntry(&search)
    ) {
	char* value = (char*) Tcl_GetHashKey(&shdata_table, entry_ptr);
	Tcl_AppendElement(interp, value);
    }
    return TCL_OK;
}

/*
 * Tcl Commands:
 *
 * shdata <? | help>
 * shdata help
 * shdata compute minmax
 * shdata compute limits columns <col1> <col2> 
 *     <Automatic|Cartesian|Fill> [args]
 * shdata compute limits data <xmin> <ymin> <xmax> <ymax>
 * shdata column <col> <vector> 
 * shdata column <col> ?key <vector>? ?key <vector>? 
 * shdata histogram <bins> <col> <xvector> <yvector> <min> <max> ?scale?
 * shdata ray <ray> <vector> 
 * shdata info data ?which_rays?
 * shdata info rays
 * shdata info column <col> ?which_rays?
 * shdata info selected
 * shdata select <good|lost|all>
 * 
 */

int shdata_inst_cmd (ClientData client_data, 
    Tcl_Interp* interp, int argc, char** argv
) {
    ShadowData* shdata = (ShadowData*)client_data;
    assert(shdata != 0);
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " option [args..]\"", (char *) NULL);
	return TCL_ERROR;
    }
    size_t arg1len = strlen(argv[1]);

    /* shdata <? | help> */
    if (
        argv[1][0] == '?' || 
        (argv[1][0] == 'h' && strncmp(argv[1], "help", arg1len) == 0)
    ) {
	Tcl_AppendResult(interp, "No help available yet. Try back later.",
	    (char *) NULL
	);
	return TCL_OK;
    }

    /* shdata select <good|lost|all> */
    if (argv[1][0] == 's' && strncmp(argv[1], "select", arg1len) == 0) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " select <good|lost|all>\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	int which;
	switch (argv[2][0]) {
	    case 'g':
	    case 'G':
		which = GOOD_RAYS;
		break;

	    case 'l':
	    case 'L':
		which = LOST_RAYS;
		break;

	    case 'a':
	    case 'A':
		which = ALL_RAYS;
		break;
	    
	    default:
		Tcl_AppendResult (interp, "bad option \"", argv[2],
		    "\": Should be \"good, lost, all\".", 0
		);
		return TCL_ERROR;
	}
	shdata->which_rays = which;
	return TCL_OK;
    }

    /* 
     * shdata compute minmax
     * shdata compute limits columns <col1> <col2> 
     *     <Automatic|Cartesian|Fill> ?fill_factor?
     * shdata compute limits data <xmin> <ymin> <xmax> <ymax>
     *     <Automatic|Cartesian|Fill> ?fill_factor?
     */
    if (argv[1][0] == 'c' && strncmp(argv[1], "compute", arg1len) == 0) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " compute [args]\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	size_t arg2len = strlen(argv[2]);
	if (argv[2][0] == 'm' && strncmp(argv[2], "minmax", arg2len) == 0) {
	    if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # args:  should be \"",
		    argv[0], " ", argv[1], " minmax\"", NULL
		);
		return TCL_ERROR;
	    }
	    return compute_minmax(interp, shdata);
	} else if (
	    argv[2][0] == 'l' && 
	    strncmp(argv[2], "limits", arg2len) == 0
	){
	    if (argc < 7 || argc > 9) {
		Tcl_AppendResult(interp, "wrong # args:  should be \"",
		    argv[0], " ", argv[1], 
		    " limit <columns | data> [args]\"", NULL
		);
		return TCL_ERROR;
	    }
	    size_t arg3len = strlen(argv[3]);
	    if (
	        argv[3][0] == 'c' && 
		strncmp(argv[3], "columns", arg3len) == 0
	    ) {
		if (argc < 7 || argc > 8) {
		    Tcl_AppendResult (interp, "wrong # args: should be \"",
			argv[0], " ", argv[1], " ", argv[2], 
			" columns <col1> <col2> <Auto|Cartesian|Fill>"
			" ?fill_factor?\".", 0
		    );
		    return TCL_ERROR;
		}
		int col1 = atoi(argv[4]);
		int col2 = atoi(argv[5]);
		if (column_bad(col1) || column_bad(col2)) {
		    Tcl_AppendResult (interp, "column ", argv[4], " or ",
			argv[5], " out of range", 0);
		    return TCL_ERROR;
		}
		int limit;
		char lchar = argv[6][0];
		if (lchar == 'a' || lchar == 'A')
		    limit = AUTOMATIC_LIMIT;
		else if (lchar == 'c' || lchar == 'C')
		    limit = CARTESIAN_LIMIT;
		else if (lchar == 'f' || lchar == 'F')
		    limit = FILL_LIMIT;
		else {
		    Tcl_AppendResult(interp, "bad arg \"", argv[6], 
			"\":  should be \"Automatic, Cartesian, or Fill\"", 
			NULL
		    );
		    return TCL_ERROR;
		}
		double fill_factor = FILL_FACTOR;
		if (argc == 8 && 
		    Tcl_GetDouble(interp, argv[7], &fill_factor) != TCL_OK
		) {
		    Tcl_AppendResult(interp, 
			"Expecting a real number for fill-factor", 0
		    );
		    return TCL_ERROR;
		}
		return compute_data_limits(interp, 
		    shdata, col1, col2, limit, fill_factor
		);
	    }

	    if (argv[3][0] == 'd' && strncmp(argv[3], "data", arg3len) == 0) {
		if (argc <  9 || argc > 10) {
		    Tcl_AppendResult (interp, "wrong # args: should be \"",
			argv[0], " ", argv[1], " ", argv[2], 
			" data <xmin> <xmax> <ymin> <ymax> ",
			"<Auto|Cartesian|Fill> ?fill_factor?\".", 0
		    );
		    return TCL_ERROR;
		}
		/* next four numbers are x/y min/max values */
		double xmin, xmax, ymin, ymax;
		if (Tcl_GetDouble(interp, argv[4], &xmin) != TCL_OK ||
		    Tcl_GetDouble(interp, argv[5], &xmax) != TCL_OK ||
		    Tcl_GetDouble(interp, argv[6], &ymin) != TCL_OK ||
		    Tcl_GetDouble(interp, argv[7], &ymax) != TCL_OK
		) {
		    Tcl_AppendResult(interp, 
			"Expecting real number for x/y min/max values", 0
		    );
		    return TCL_ERROR;
		}
		int limit;
		char lchar = argv[8][0];
		if (lchar == 'a' || lchar == 'A')
		    limit = AUTOMATIC_LIMIT;
		else if (lchar == 'c' || lchar == 'C')
		    limit = CARTESIAN_LIMIT;
		else if (lchar == 'f' || lchar == 'F')
		    limit = FILL_LIMIT;
		else {
		    Tcl_AppendResult(interp, "bad arg \"", argv[8],
			"\":  should be \"Automatic, Cartesian, or Fill\"", 
			NULL
		    );
		    return TCL_ERROR;
		}
		double fill_factor = FILL_FACTOR;
		if (argc == 10 && 
		    Tcl_GetDouble(interp, argv[9], &fill_factor) != TCL_OK
		) {
		    Tcl_AppendResult(interp, 
			"Expecting a real number for fill-factor", 0
		    );
		    return TCL_ERROR;
		}
		return compute_data_limits(interp, 
		    xmin, xmax, ymin, ymax, limit, fill_factor
		);
	    }
	}
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
	    argv[0], " ", argv[1], " [args]\"", (char *) NULL
	);
	return TCL_ERROR;
    }
    
    /* 
     * shdata column <col> <vector> 
     * shdata column <col> ?key <vector>? ?key <vector>? 
     */
    if (argv[1][0] == 'c' && strncmp(argv[1], "column", arg1len) == 0) {
	if (argc < 4) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " ", argv[1], " <col> [args]\"",
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	int column = atoi(argv[2]);
	if (argc == 4) {
	    return extract_shadow_column(
	        interp, shdata, column, argv[3], shdata->which_rays
	    ); 
	} else {
	    int extra_args = argc - 3;
	    if (extra_args % 2 != 0) {
		Tcl_AppendResult(interp, "wrong # args:  should be \"",
		    argv[0], " ", argv[1], " ", argv[2],
		    " ?key <vector>? ?key <vector>? ...\"", 0
		);
		return TCL_ERROR;
	    }
	    for (int i = 3; i < argc; i += 2) {
		int which;
		switch (argv[i][0]) {
		    case 'g':
		    case 'G':
		        which = GOOD_RAYS;
			break;
		    
		    case 'l':
		    case 'L':
		        which = LOST_RAYS;
			break;

		    case 'a':
		    case 'A':
		        which = ALL_RAYS;
			break;
		    
		    default:
		        Tcl_AppendResult(interp, 
			    "Key must be All, Lost or Good", 0
			);
			return TCL_ERROR;
		}
		    
		int retcode = extract_shadow_column(
		    interp, shdata, column, argv[i+1], which
		); 
		if (retcode != TCL_OK)
		    return TCL_ERROR;
	    }
	    return TCL_OK;
	}
    }
    /* 
     * shdata histogram <bins> <col> <xvector> <yvector> <min> <max> ?scale?
     */
    if (argv[1][0] == 'h' && strncmp(argv[1], "histogram", arg1len) == 0) {
	if (argc < 8 || argc > 9) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], 
		" histogram <bins> <col> <xvector> <yvector> "
		"<min> <max> ?scale?\"", (char *) NULL
	    );
	    return TCL_ERROR;
	}
	int bins, column;
	if (Tcl_GetInt (interp, argv[2], &bins) != TCL_OK ||
	    Tcl_GetInt (interp, argv[3], &column) != TCL_OK
	) {
	    Tcl_AppendResult(interp, 
		"Expecting integers for both <bins> and <column> values", 0
	    );
	    return TCL_ERROR;
	}
	if (column_bad(column)) {
	    sprintf(buf, "Column %d out of range", column);
	    Tcl_AppendResult(interp, buf, 0);
	    return TCL_ERROR;
	}
	//
	// The min/max limits for computing the histograms is from the
	// limits of the scatter plot, not the actual data limits.
	//
	// double min = shdata->column_info[column-1][which][1];
	// double max = shdata->column_info[column-1][which][2];
	//
	//
	double min, max;
	if (Tcl_GetDouble (interp, argv[6], &min) != TCL_OK ||
	    Tcl_GetDouble (interp, argv[7], &max) != TCL_OK
	) {
	    Tcl_AppendResult(interp, 
		"Expecting real for both \"min\" and \"max\" arguments", 0
	    );
	    return TCL_ERROR;
	}
	double scale = 1.0;
	if (argc == 9 && Tcl_GetDouble (interp, argv[8], &scale) != TCL_OK) {
	    Tcl_AppendResult(interp, 
		"Expecting real for \"scale\" argument", 0
	    );
	    return TCL_ERROR;
	}
	int which = shdata->which_rays;
	char* xvecname = argv[4];
	char* yvecname = argv[5];
	return extract_shadow_histogram (interp, shdata, column, 
	    xvecname, yvecname, which, min, max, bins, scale
	);
    }

    /* shdata ray <ray> <vector> */
    if (argv[1][0] == 'r' && strncmp(argv[1], "ray", arg1len) == 0) {
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " ", argv[1], " <ray> <vector>\"",
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	int ray = atoi(argv[2]);
	if (extract_shadow_ray(interp, shdata, ray, argv[3]) != TCL_OK)
	    return TCL_ERROR;
	return TCL_OK;
    }

    /* 
     * shdata info data ?which_rays?
     * shdata info rays
     * shdata info column <col> ?which_rays?
     * shdata info selected
     */
    if (argv[1][0] == 'i' && strncmp(argv[1], "info", arg1len) == 0) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " ", argv[1], 
		" data, rays, column, selected [args]\"",
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	size_t arg2len = strlen(argv[2]);
	if (argv[2][0] == 'd' && strncmp(argv[2], "data", arg2len) == 0) {
	    if (argc != 3 && argc != 4) {
		Tcl_AppendResult (interp, "wrong # args: should be \"",
		    argv[0], " ", argv[1], " data ?which_rays?\".", 0
		);
		return TCL_ERROR;
	    }
	    int which = shdata->which_rays;
	    if (argc == 4) {
		switch (argv[3][0]) {
		    case 'g':
		    case 'G':
			which = GOOD_RAYS;
			break;
		    
		    case 'l':
		    case 'L':
			which = LOST_RAYS;
			break;

		    case 'a':
		    case 'A':
			which = ALL_RAYS;
			break;
		    
		    default:
			Tcl_AppendResult(interp, 
			    "Key must be All, Lost or Good", 0
			);
			return TCL_ERROR;
		}
	    }
	    sprintf (buf, "Found %d rays, %d good\n\n",
		shdata->npoints, shdata->good_rays
	    );
	    Tcl_AppendResult(interp, buf, 0);
	    return get_minmax_info(interp, shdata, which);
	}

	if (argv[2][0] == 'r' && strncmp(argv[2], "rays", arg2len) == 0) {
	    if (argc != 3) {
		Tcl_AppendResult (interp, "bad # args: should be \"",
		    argv[0], " ", argv[1], " rays\".", 0
		);
		return TCL_ERROR;
	    }
	    sprintf(buf, "%d %d", shdata->npoints, shdata->good_rays);
	    Tcl_AppendResult(interp, buf, 0);
	    return TCL_OK;
	}

	if (argv[2][0] == 'c' && strncmp(argv[2], "column", arg2len) == 0) {
	    if (argc != 4 && argc != 5) {
		Tcl_AppendResult (interp, "wrong # args: should be \"",
		    argv[0], " ", argv[1], " column <col> ?which_rays?\".",
		    0
		);
		return TCL_ERROR;
	    }
	    int column = atoi(argv[3]);
	    if (column == 0 || column > 20) {
	        Tcl_AppendResult(interp, 
		    "Column ", argv[3], " out of range", 0
		);
		return TCL_ERROR;
	    }

	    int which = shdata->which_rays;
	    if (argc == 5) {
		switch (argv[4][0]) {
		    case 'g':
		    case 'G':
			which = GOOD_RAYS;
			break;
		    
		    case 'l':
		    case 'L':
			which = LOST_RAYS;
			break;

		    case 'a':
		    case 'A':
			which = ALL_RAYS;
			break;
		    
		    default:
			Tcl_AppendResult(interp, 
			    "Key must be All, Lost or Good", 0
			);
			return TCL_ERROR;
		}
	    }

	    --column;
	    sprintf(buf, "%d", column+1);
	    Tcl_AppendElement(interp, buf);
	    sprintf(buf, "%d", 
	        static_cast<int>(shdata->column_info[column][which][0])
	    );
	    Tcl_AppendElement(interp, buf);
	    Tcl_AppendElement(interp, column_names[column]);

	    /* min, max, center, and std. dev. */
	    for (int i = 1; i < 5; ++i) {
		sprintf(buf, "%-12.5E", shdata->column_info[column][which][i]);
		Tcl_AppendElement(interp, buf);
	    }
	    return TCL_OK;
	}

	if (argv[2][0] == 's' && strncmp(argv[2], "selected", arg2len) == 0) {
	    if (argc != 3) {
		Tcl_AppendResult (interp, "bad # args: should be \"",
		    argv[0], " ", argv[1], " selected\".", 0
		);
		return TCL_ERROR;
	    }
	    char* which_rays;
	    switch (shdata->which_rays) {
		case GOOD_RAYS:
		    which_rays = "good";
		    break;
		
		case LOST_RAYS:
		    which_rays = "lost";
		    break;

		case ALL_RAYS:
		    which_rays = "all";
		    break;
		
		default:
		    break;
	    }
	    strcpy(buf, which_rays);
	    Tcl_AppendResult(interp, buf, 0);
	    return TCL_OK;
	}

	Tcl_AppendResult(interp, "wrong # args:  should be \"",
	    argv[0], " ", argv[1], " [args]\"", (char *) NULL
	);
	return TCL_ERROR;
    }
    
    Tcl_AppendResult(interp, "wrong # args:  should be \"",
	argv[0], 
	" help, select, compute, column, ray [args..]\"", 
	(char *) NULL
    );
    return TCL_ERROR;
}


/*
 * Tcl Commands:
 *
 * shdata <? | help>
 * shdata list
 * shdata list
 * shdata create <name> ?-load file?
 * shdata delete <name>
 *
 */

int shdata_cmd (ClientData /* client_data */, 
    Tcl_Interp* interp, int argc, char** argv
) {
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
	    argv[0], " list, create, delete, help [args]\"", (char *) NULL
	);
	return TCL_ERROR;
    }

    if (
	strcasecmp(argv[1], "?") == 0 || 
	strcasecmp(argv[1], "help") == 0
    ) {
	Tcl_AppendResult(interp, "No help available yet. Try back later.",
	    (char *) NULL
	);
	return TCL_OK;
    }
    else if (strcasecmp(argv[1], "list") == 0) {
	return list_shdata (interp);
    }
    else if (strcasecmp(argv[1], "create") == 0) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		    argv[0], " create <name> ?option?\"", (char *) NULL);
	    return TCL_ERROR;
	}
	//
	// create a new ShadowData object, but only if it hasn't been 
	// created already.
	//
	char* name = argv[2];

	//
	// first check if a Tcl command exists with a same name or not,
	// and then check for an existing data name.
	//
	Tcl_CmdInfo cmd_info;
	bool cmd_exists = Tcl_GetCommandInfo(interp, name, &cmd_info);
	bool data_exists = Tcl_FindHashEntry(&shdata_table, name) != 0;

	if (data_exists) {
	    Tcl_AppendResult(interp, "Data \"", name, "\" already exists. ",
		"Try another name for the data item.", (char *) NULL
	    );
	    return TCL_ERROR;
	}

	if (cmd_exists) {
	    Tcl_AppendResult(interp, 
		"Tcl already has a command called \"", name, "\".",
		"Try another name for the data item.",
		(char *) NULL
	    );
	    return TCL_ERROR;
	}

	ShadowData *shdata = create_shadow_data(interp, 0, 0, 0);
	Tcl_AppendResult (interp, name, 0);
	if (argc > 3) {
	    for(int ind = 3; ind < argc; ind++) {
		if (
		    strcasecmp(argv[ind], "-load") == 0 || 
		    strcasecmp(argv[ind], "-read") == 0
		) {
		    ind++;
		    if (ind >= argc) {
			Tcl_AppendResult(interp, "wrong # args: should be \"",
			    argv[0], " ", name, " -load <filename>\"", 
			    (char *) NULL
			);
			delete_shadow_data(shdata);
			return TCL_ERROR;
		    }
		    if (read_shadow_data(interp, argv[ind], shdata) != TCL_OK) {
			delete_shadow_data(shdata);
		        return TCL_ERROR;
		    }
		} else {
		    Tcl_AppendResult(interp, "Incorrect option: should be \"",
			argv[0], " <name> ?options?\"", (char *) NULL
		    );
		    delete_shadow_data(shdata);
		    return TCL_ERROR;
		}
	    }
	}
	//
	// add the new data to the hastable for listing, and create the 
	// Tcl command.
	//
	int new_entry = 0;
	Tcl_HashEntry* entry_ptr = Tcl_CreateHashEntry(
	    &shdata_table, name, &new_entry
	);
	assert(new_entry == 1);
	Tcl_SetHashValue(entry_ptr, shdata);

	Tcl_CreateCommand(interp, name, shdata_inst_cmd,
	    (ClientData)shdata, (Tcl_CmdDeleteProc*) NULL
	);
    }
    else if (strcasecmp(argv[1], "delete") == 0) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args:  should be \"",
		    argv[0], " delete <name> ?options?\"", (char *) NULL);
	    return TCL_ERROR;
	}
	//
	// deletes an existing shadow data object
	//
	char* instname = argv[2];

	Tcl_HashEntry* entry_ptr = Tcl_FindHashEntry(
	    &shdata_table, instname
	);
	if (entry_ptr == 0) {
	    Tcl_AppendResult(interp, "Shadow Data \"", instname, 
		"\" doesn't exist. Can't delete non-existent data object.", 
		(char *) NULL
	    );
	    return TCL_ERROR;
	}
	ShadowData *shdata = (ShadowData*) Tcl_GetHashValue(entry_ptr);
	assert(shdata != 0);

	Tcl_DeleteHashEntry(entry_ptr);
	delete_shadow_data(shdata);

	if (Tcl_DeleteCommand(interp, instname) != TCL_OK) {
	    Tcl_AppendResult(interp, "Error deleting SHADOW Data object \"", 
		instname, "\".", (char *) NULL
	    );
	    return TCL_ERROR;
	}
    }
    else {
	Tcl_AppendResult(interp, "Bad arguments:  should be \"",
	    argv[0], " list, create, delete, help [args\"", (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

// ======================================================================= //

static int shdata_delete_cmd(ClientData /* client_data */) {
    Tcl_DeleteHashTable(&shdata_table);
    initialized = 0;
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * Shdata_Init --
 *
 *	This procedure is invoked to initialize the Tcl command
 *	"shdata".
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Creates the new command and adds a new entry into a global Tcl
 *	hash table
 *
 * ------------------------------------------------------------------------
 */

int Shdata_Init(Tcl_Interp *interp) {

    /* Flag this so this routine can be run more than once */
    if (!initialized) {
	Tcl_InitHashTable(&shdata_table, TCL_STRING_KEYS);
	initialized = 1;
    }

    Tcl_CreateCommand(interp, "shdata", shdata_cmd, (ClientData)NULL, 
        (Tcl_CmdDeleteProc*)shdata_delete_cmd
    );
    return TCL_OK;
}

