//
// menu_cmds.cc: Tcl commands to manipulate MENU system.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994 Mumit Khan
//

#include <cassert>
#include <cstring>
#include <list>

#include "menu.h"
#include "commands.h"
#include "xmenu.h"

#if NEED_DECLARATION_STRCASECMP
extern "C" int strcasecmp(const char*, const char*);
#endif
#if NEED_DECLARATION_STRNCASECMP
extern "C" int strncasecmp(const char*, const char*, std::size_t);
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

// ================ Tcl commands created here ======================== //
/*
 *
 * ALL THE COMMANDS to "MENU" in this interpreter:
 *
 * xmenu load <file>
 * xmenu pages
 * xmenu dump 
 *
 */

// =================================================================== //

static inline char* string_to_charptr(const string& str) {
    return const_cast<char*>(str.c_str());
}

// =================================================================== //

static int load_menu(Tcl_Interp* interp, const char* file) {
    int retcode = TCL_OK;
    if (GLB_menu->load(file) != 0) {
	Tcl_AppendResult(
	    interp, "\"", file, "\": bad menu file format?", (char *) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}

static int dump_menu(Tcl_Interp* interp) {
    Tcl_AppendResult(
	interp, "dumping menu file ",  (char*) NULL
    );
    GLB_menu->dump();
    return TCL_OK;
}


static int list_menu_pages(Tcl_Interp* interp) {
    const list<string>& page_names = GLB_menu->get_pages();
    list<string>::const_iterator it = page_names.begin();
    for(; it != page_names.end(); ++it) {
	const string& page_name = *it;
	Tcl_AppendElement(interp, string_to_charptr(page_name));
    }
    return TCL_OK;
}

inline static int menu_suboption_error(Tcl_Interp* interp, 
    const char* cmd, const char* arg, const char* detail
) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", cmd,
	" ", arg, " ", detail, "\"", (char *) NULL
    );
    return TCL_ERROR;
}

inline static int menu_arg_error(Tcl_Interp* interp, 
    const char* cmd, const char* arg
) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", cmd,
	" ", arg, "\"", (char *) NULL
    );
    return TCL_ERROR;
}


/*
 *--------------------------------------------------------------
 *
 * XMenu_MenuCmd:
 *
 *      This procedure is invoked to process the "menu" Tcl
 *      command.  
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *--------------------------------------------------------------
 */

int XMenu_MenuCmd (ClientData, Tcl_Interp* interp, int argc, char** argv) {
    if (argc < 2 || argc > 3) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " option [args..]\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (strncasecmp(argv[1], "lo", 2) == 0) {
        if (argc != 3) {
	    return menu_suboption_error(interp, argv[0], argv[1], "MENU-FILE");
	}
	return load_menu(interp, argv[2]);
    } else if (strncasecmp(argv[1], "pa", 2) == 0) {
        if (argc != 2) {
	    return menu_arg_error(interp, argv[0], argv[1]);
	}
        return list_menu_pages(interp);
    } else if (strncasecmp(argv[1], "du", 2) == 0) {
        if (argc != 2) {
	    return menu_arg_error(interp, argv[0], argv[1]);
	}
        return dump_menu(interp);
    }
    Tcl_AppendResult(interp, "bad option \"", argv[1],
	"\":  should be load, pages, dump", (char*) NULL
    );
    return TCL_ERROR;
}

// =================================================================== //
