//
// tool.h: Interface to "tools"
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#ifndef shadow_tool_h
#define shadow_tool_h

// ====================================================================== //

#include <list>
#include <map>

#include "mystring.h"
#include "xrtypes.h"
#include "shadow-nml.h"

// ====================================================================== //
//
//  TOOLS
//
// ====================================================================== //

// 
// class Tool: Used to store definition of tool (1 for each tool)
//

class Tool {
public:
    typedef ShadowNamelist::VariableList VariableList;
public:
    Tool(const std::string& name);
    virtual ~Tool();

    virtual const std::string& name() const;

    virtual const ShadowNamelist& namelist() const;
    virtual const VariableList& variables() const;
    virtual bool exists_var(const std::string& varname) const;
    virtual const Var* get_var(const std::string& varname) const;
    virtual int add_var(Var* var); 

    virtual void dump() const;

private:
    std::string name_;
    ShadowNamelist nml_;
};

// ====================================================================== //
//
//  TOOL INSTANCE
//
// ====================================================================== //

//
// class ToolInstance: used to  store an instance of a tool in an experiment
//
class ToolInstance {
public:
    ToolInstance(const Tool* tool, int instance);
    ToolInstance(const ToolInstance&);
    virtual ~ToolInstance();

    virtual ToolInstance* clone() const;
    virtual int copy(const ToolInstance& from);

    const std::string& name() const;
    virtual const std::string& instance_name() const;
    const Tool* tool() const;
    int instance() const;

    virtual void set_instance_name(const std::string&);

    bool exists_var(const std::string& varname) const;
    const Var* get_var(const std::string& varname) const;
    const Value* get_varval(const std::string& varname) const;

    int add_var(Var* var);
    int set_varval(const std::string& varname, Value* val);

    virtual void dump() const;

private:
    const Tool* tool_;
    int instance_;
    std::string instname_;
    ShadowNamelist nml_;
};

// ====================================================================== //
//
//  TOOL Manager
//
// ====================================================================== //

class ToolMgr {
public:
    ToolMgr();
    virtual ~ToolMgr();

    const Tool* find(const std::string& name) const;
    Tool* find(const std::string& name);
    Tool* add(const std::string& name);
    int add_alias(const std::string& alias, const std::string& realname);

    //
    // for Tcl interpreter access.
    //
    const std::list<std::string>& tools() const;

    void dump() const;

private:
    int unalias(std::string& name) const;

private:
    typedef std::map<std::string, void*, std::less<std::string> > ToolMap;
    typedef std::map<Mystring, Mystring, std::less<Mystring> > ToolAliasMap;
    ToolMap tools_;
    ToolAliasMap tool_aliases_;
    std::list<std::string> toolnames_;
};

// ====================================================================== //

#endif/*shadow_tool_h*/
