/* choose_directory.c -- directory browser for win32. */

#ifndef _WIN32
# error This file is meant only for Windows32
#endif

#include <windows.h>

#ifdef _MSC_VER
#include <shlobj.h>
#endif

#include <tcl.h>
#include <tk.h>

#ifdef __CYGWIN__
#include <stdlib.h>
#include <string.h>
extern int cygwin_conv_to_full_win32_path (const char *path, char *win32_path);
#endif

/* things missing from Cygwin/Mingw/etc that uses Windows32api */
#ifndef BFFM_INITIALIZED

LPITEMIDLIST WINAPI SHBrowseForFolderA(LPBROWSEINFO lpbi);

/* browing for directories */
#define BIF_RETURNONLYFSDIRS	0x0001

/* message from browser */
#define BFFM_INITIALIZED        1
#define BFFM_SELCHANGED         2

/* messages to browser */
#define BFFM_SETSTATUSTEXTA     (WM_USER + 100)
#define BFFM_ENABLEOK           (WM_USER + 101)
#define BFFM_SETSELECTIONA      (WM_USER + 102)
#define BFFM_SETSELECTIONW      (WM_USER + 103)
#define BFFM_SETSTATUSTEXTW     (WM_USER + 104)

#ifdef UNICODE
#define SHBrowseForFolder   SHBrowseForFolderW
#define BFFM_SETSTATUSTEXT  BFFM_SETSTATUSTEXTW
#define BFFM_SETSELECTION   BFFM_SETSELECTIONW
#else
#define SHBrowseForFolder   SHBrowseForFolderA
#define BFFM_SETSTATUSTEXT  BFFM_SETSTATUSTEXTA
#define BFFM_SETSELECTION   BFFM_SETSELECTIONA
#endif

#endif /* ! BFFM_INITIALIZED */

int CALLBACK 
MyBrowseCallbackProc (HWND hwnd, UINT uMsg, LPARAM lParam, LPARAM lpData)
{
  if (uMsg == BFFM_INITIALIZED)
     SendMessage (hwnd, BFFM_SETSELECTION, (WPARAM) TRUE, (LPARAM) lpData);
  return 0;
}

static int
choose_directory_command (ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
  BROWSEINFO bi;
  char buf[MAX_PATH + 1];
  Tk_Window parent;
  int i;
  LPITEMIDLIST idlist;
  char *p;
  int atts;
  Tcl_DString tempBuffPtr;
  Tcl_DStringInit (&tempBuffPtr);

  bi.hwndOwner = NULL;
  bi.pidlRoot = NULL;
  bi.pszDisplayName = buf;
  bi.lpszTitle = NULL;
  bi.ulFlags = BIF_RETURNONLYFSDIRS;
  bi.lpfn = NULL;
  bi.lParam = 0;
  bi.iImage = 0;

  parent = Tk_MainWindow (interp);

  for (i = 1; i < argc; i += 2)
    {
      int v;
      int len;

      v = i + 1;
      len = strlen (argv[i]);

      if (strncmp (argv[i], "-parent", len) == 0)
	{
	  if (v == argc)
	    goto arg_missing;

	  parent = Tk_NameToWindow (interp, argv[v],
				    Tk_MainWindow (interp));
	  if (parent == NULL)
	    return TCL_ERROR;
	}
      else if (strncmp (argv[i], "-title", len) == 0)
	{
	  if (v == argc)
	    goto arg_missing;

	  bi.lpszTitle = argv[v];
	}
      else if (strncmp (argv[i], "-initialdir", len) == 0)
	{
	  char *initialdir;
	  if (v == argc)
	    goto arg_missing;
	  
	  initialdir = argv[v];
#ifdef __CYGWIN__
	  /* work around bug in Cygwin port of Tcl. file native doesn't
	     return the right thing.  */
          initialdir = (char *) alloca (PATH_MAX + 1);
	  if (cygwin_conv_to_full_win32_path (argv[v], initialdir))
	    initialdir = argv[v];
	  else 
	    {
	      /* FIXME/HACK: This is to get around the Cygwin API problem
		 where //c returns c:, not c:/. Argh.  */
	      if (isalpha (initialdir[0]) && initialdir[1] == ':' 
	          && initialdir[2] == '\0')
		strcat (initialdir, "\\");
	    }
#endif
	  /* bi.lParam will be passed to the callback function. 
	     (save the need for globals).  */
	  bi.lParam = (LPARAM) Tcl_TranslateFileName (interp, initialdir, 
	                                              &tempBuffPtr);
	  bi.lpfn   = MyBrowseCallbackProc;
	}
      else
	{
	  Tcl_AppendResult (interp, "unknown option \"", argv[i],
			    "\", must be -parent, -title or -initialdir", 
			    (char *) NULL);
	  return TCL_ERROR;
	}
    }

  if (Tk_WindowId (parent) == None)
    Tk_MakeWindowExist (parent);

  bi.hwndOwner = Tk_GetHWND (Tk_WindowId (parent));

  idlist = SHBrowseForFolder (&bi);

  if (idlist == NULL)
    {
      /* User pressed the cancel button.  */
      return TCL_OK;
    }

  if (! SHGetPathFromIDList (idlist, buf))
    {
      Tcl_SetResult (interp, "could not get path for directory", TCL_STATIC);
      return TCL_ERROR;
    }

  /* Ensure the directory exists.  */
  atts = GetFileAttributesA (buf);
  if (atts == -1 || ! (atts & FILE_ATTRIBUTE_DIRECTORY))
    {
      Tcl_AppendResult (interp, "path \"", buf, "\" is not a directory",
			(char *) NULL);
      /* FIXME: free IDLIST.  */
      return TCL_ERROR;
    }

  /* FIXME: We are supposed to free IDLIST using the shell task
     allocator, but cygwin32 doesn't define the required interfaces
     yet.  */

  /* Normalize the path for Tcl.  */
  for (p = buf; *p != '\0'; ++p)
    if (*p == '\\')
      *p = '/';

  Tcl_SetResult (interp, buf, TCL_VOLATILE);
  Tcl_DStringFree(&tempBuffPtr);

  return TCL_OK;

 arg_missing:
  Tcl_AppendResult(interp, "value for \"", argv[argc - 1], "\" missing",
		   NULL);
  return TCL_ERROR;
}

static char version_string[] = "0.9";

int __declspec(dllexport)
Choose_directory_Init (Tcl_Interp *interp)
{
  if (Tcl_PkgRequire(interp, "Tcl", TCL_VERSION, 0) == NULL 
      || Tcl_PkgRequire(interp, "Tk", TK_VERSION, 0) == NULL 
      || Tcl_PkgProvide (interp, "Choose_directory", version_string) != TCL_OK)
    return TCL_ERROR;

  if (Tcl_CreateCommand (interp, "choose_directory", choose_directory_command,
			 NULL, NULL) == NULL)
    return TCL_ERROR;

  return TCL_OK;
}

