//
// item_cmds.cc: Tcl commands to manipulate MENU items.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994 Mumit Khan
//

#ifdef __BORLANDC__
# include <assert>
# include <string>
#else
# include <cassert>
# include <cstring>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#if defined(NO_STRNCASECMP_PROTO)
extern "C" int strcasecmp(const char*, const char*);
extern "C" int strncasecmp(const char*, const char*, int);
#endif

#include "menu.h"
#include "beamline.h"
#include "utils.h"
#include "value.h"
#include "variable.h"

#include "commands.h"
#include "xmenu.h"

// ================ Tcl commands created here ======================== //
/*
 *
 * ALL THE COMMANDS to "ITEMS" in this interpreter:
 *
 * xitem list <page> <item>
 * xitem vget <page> <item>
 * xitem vset <page> <item> <new-value>
 * xitem vallowed <page> <item>
 * xitem dump <page> <item>
 *
 */

// =================================================================== //

static inline char* string_to_charptr(const string& str) {
    return const_cast<char*>(str.c_str());
}

// =================================================================== //

static int dump_item(Tcl_Interp* interp, const char* pname, 
    const char* item_name
) {
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    const MenuItem* item = page->find_item(item_name);
    if (item == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such item \"", item_name, "\" in page \"",
	    (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    Tcl_AppendResult(interp, 
	"dumping item \"", item_name, "\" from page \"", 
	(char*)pname, "\"...", (char*) NULL
    );
    item->dump();
    return TCL_OK;
}


static int list_item(Tcl_Interp* interp, const char* pname,
    const char* item_name
) {
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    const MenuItem* item = page->find_item(item_name);
    if (item == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such item \"", item_name, "\" in page \"",
	    (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    int retcode = TCL_OK;
    switch(item->type()) {
	case MenuItem::SKIP_ITEM:
	    Tcl_AppendElement(interp, "SKIP");
	    break;
	case MenuItem::TEXT_ITEM:
	    Tcl_AppendElement(interp, "TEXT");
	    {
		const TextMenuItem* titem = 
		    static_cast<const TextMenuItem*>(item);
		Tcl_AppendElement(interp, string_to_charptr(titem->text()));
		Tcl_AppendElement(interp, string_to_charptr(titem->submenu()));
	    }
	    break;
	case MenuItem::DATA_ITEM:
	    Tcl_AppendElement(interp, "DATA");
	    {
		const DataMenuItem* ditem = (DataMenuItem*)item;
		const Var* var = GLB_beamline->get_var(
		    ditem->record(), ditem->name()
		);
		if (var == nil) {
		    cerr << "variable \"" << ditem->name() << "\" is NIL"
		        << endl;
		}
		const char* strval = (var) ?
		    var->getval()->getstrval() : "";
		
		const char* flag = ditem->flag();
		bool flag_val = true;
		if (flag) {
		    flag_val = page->get_flag_value(flag);
		}
		bool force_update = 
		    ditem->flags() & DataMenuItem::FORCE_UPDATE;
		bool readonly = 
		    ditem->flags() & DataMenuItem::READ_ONLY
;
		Tcl_AppendElement(interp, string_to_charptr(ditem->record()));
		Tcl_AppendElement(interp, string_to_charptr(ditem->name()));
		Tcl_AppendElement(interp, 
		    (var) ? (char*)var->gettype() : "text"
		);
		Tcl_AppendElement(interp, string_to_charptr(ditem->prompt()));
		Tcl_AppendElement(interp, (flag_val) ? "" : "n/a");
		Tcl_AppendElement(interp, (force_update) ? "1" : "0");
		Tcl_AppendElement(interp, (readonly) ? "1" : "0");
		Tcl_AppendElement(interp, (char*)strval);
		Tcl_AppendElement(interp, string_to_charptr(ditem->submenu()));
	    }
	    break;
	default:
	    Tcl_AppendElement(interp, "UNKNOWN TYPE");
	    retcode = TCL_ERROR;
	    break;
    }
    return retcode;
}

static int get_value(Tcl_Interp* interp, const char* pname,
    const char* item_name
) {
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    const MenuItem* item = page->find_item(item_name);
    if (item == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such item \"", item_name, "\" in page \"",
	    (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    int retcode = TCL_OK;
    switch(item->type()) {
	case MenuItem::SKIP_ITEM:
	    Tcl_AppendResult(interp, "", (char*) NULL);
	    break;
	case MenuItem::TEXT_ITEM:
	    {
		const TextMenuItem* titem = (TextMenuItem*)item;
		Tcl_AppendResult(interp, 
		    string_to_charptr(titem->text()), (char*) NULL
		);
	    }
	    break;
	case MenuItem::DATA_ITEM:
	    {
		const DataMenuItem* ditem = (DataMenuItem*)item;
		const Var* var = GLB_beamline->get_var(
		    ditem->record(), ditem->name()
		);
		char const* strval = (var) ?
		    var->getval()->getstrval() : "";
		Tcl_AppendResult(interp, (char*)strval, (char*) NULL);
	    }
	    break;
	default:
	    Tcl_AppendResult(interp, "UNKNOWN ITEM", (char*) NULL);
	    retcode = TCL_ERROR;
	    break;
    }
    return retcode;
}

static int set_value(Tcl_Interp* interp, const char* pname,
    const char* item_name, const char* new_value
) {
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", const_cast<char*>(pname), "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    const MenuItem* item = page->find_item(item_name);
    if (item == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such item \"", item_name, "\" in page \"",
	    (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    if (item->type() != MenuItem::DATA_ITEM) {
	Tcl_AppendResult(interp,
	    "ERROR: Can only change the value of a data item.",
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    int retcode = TCL_OK;
    const DataMenuItem* ditem = static_cast<const DataMenuItem*>(item);
    const Var* var = GLB_beamline->get_var(
	ditem->record(), ditem->name()
    );
    assert (var != nil);
    if (GLB_beamline->set_var(ditem->record(), ditem->name(), new_value)) {
	Tcl_AppendResult(interp, 
	    "ERROR: bad value type for Namelist item \"", 
	    string_to_charptr(ditem->record()), "/", 
	    string_to_charptr(ditem->name()), "\"",
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    else {
	Tcl_AppendResult(interp, const_cast<char*>(new_value), (char*) NULL);
    }
    return retcode;
}

static int get_allowed_values(Tcl_Interp* interp, const char* pname,
    const char* item_name
) {
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    const MenuItem* item = page->find_item(item_name);
    if (item == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such item \"", item_name, "\" in page \"",
	    (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    if (item->type() != MenuItem::DATA_ITEM) {
	Tcl_AppendResult(interp,
	    "ERROR: Can only query the allowed values of a data item.",
	    (char*) NULL
	);
	return TCL_ERROR;
    }
    DataMenuItem* ditem = (DataMenuItem*)item;
    const Var* var = GLB_beamline->get_var(
	ditem->record(), ditem->name()
    );
    const Value* value = var->getval();
    if (value->isaenum()) {
	const EnumValue* enumval = (EnumValue*)var->getval();
	const Stq* alist = enumval->getenumlist();
	unsigned nlist = alist->size();
	for(unsigned i = 0; i < nlist; i++) {
	    Tcl_AppendElement(interp, (char*)alist->cycle());
	}
    }
    else if (value->isaboolean()) {
	Tcl_AppendElement(interp, "YES");
	Tcl_AppendElement(interp, "YES");
    }
    else if (value->isainteger()) {
	Tcl_AppendElement(interp, "(Any integer number)");
    }
    else if (value->isareal()) {
	Tcl_AppendElement(interp, "(Any floating point number)");
    }
    else if (value->isatext()) {
	Tcl_AppendElement(interp, "(Any text string)");
    }
    return TCL_OK;
}

static inline int item_arg_error(Tcl_Interp* interp, const char* cmd,
    const char* arg
) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", cmd,
	" ", arg, " PAGE-NAME ITEM-NAME\"", (char *) NULL
    );
    return TCL_ERROR;
}


/*
 *--------------------------------------------------------------
 *
 * XMenu_ItemCmd:
 *
 *      This procedure is invoked to process the "menu" Tcl
 *      command.  
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *--------------------------------------------------------------
 */

int XMenu_ItemCmd (ClientData, Tcl_Interp* interp, int argc, char** argv) {
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " option [args..]\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (argv[1][0] == 'l' && strncasecmp(argv[1], "li", 2) == 0) {
        if (argc != 4) {
	    return item_arg_error(interp, argv[0], argv[1]);
	}
        return list_item(interp, argv[2], argv[3]);
    }
    if (argv[1][0] == 'v') {
	if (strncasecmp(argv[1], "vg", 2) == 0) {
	    if (argc != 4) {
		return item_arg_error(interp, argv[0], argv[1]);
	    }
	    return get_value(interp, argv[2], argv[3]);
	} else if (strncasecmp(argv[1], "vs", 2) == 0) {
	    if (argc != 5) {
		Tcl_AppendResult(interp, "wrong # args:  should be \"",
		    argv[0], " ", argv[1], 
		    " vset PAGE-NAME ITEM-NAME NEW-VALUE\"", 
		    (char *) NULL
		);
		return TCL_ERROR;
	    }
	    return set_value(interp, argv[2], argv[3], argv[4]);
	} else if (strncasecmp(argv[1], "va", 2) == 0) {
	    if (argc != 4) {
		return item_arg_error(interp, argv[0], argv[1]);
	    }
	    return get_allowed_values(interp, argv[2], argv[3]);
	}
    }
    if (argv[1][0] == 'd' && strncasecmp(argv[1], "du", 2) == 0) {
        if (argc != 4) {
	    return item_arg_error(interp, argv[0], argv[1]);
	}
        return dump_item(interp, argv[2], argv[3]);
    }
    Tcl_AppendResult(interp, "bad option \"", argv[1],
	"\":  should be list, vget, vset, vallowed, dump", (char*) NULL
    );
    return TCL_ERROR;
}

// =================================================================== //
