//
// value.cc:
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

/***************************************************************************/

#ifdef __BORLANDC__
# include <stdio>
# include <string>
#else
# include <cstdio>
# include <cstring>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#ifdef __BORLANDC__
# define NO_STRNCASECMP_PROTO 1
#endif

#if defined(NO_STRNCASECMP_PROTO)
extern "C" int strcasecmp(const char*, const char*);
extern "C" int strncasecmp(const char*, const char*, int);
#endif

#include "pmu.h"
#include "utils.h"
#include "value.h"

/***************************************************************************/

Pmu* Value::pmu_ = new Pmu;

Value::Value() { strvalbuf_ = nil; }
Value::~Value() { delete[] strvalbuf_; strvalbuf_ = nil; }

bool Value::isequal(const Value*) const { return false; }
bool Value::isgt(const Value*) const { return false; }
bool Value::isainteger() const { return false; }
bool Value::isareal() const { return false; }
bool Value::isatext() const { return false; }
bool Value::isaboolean() const { return false; }
bool Value::isaenum() const { return false; }
bool Value::isarange() const { return false; }
bool Value::isafilename() const { return false; }

Value* Value::clone() const { return nil; }
Value* Value::clone(const char* /*str*/) const { return nil;  }
int Value::getinteger() const { return 0; }
double Value::getreal() const { return 0.0; }
const char* Value::gettext() const { return nil; }
bool Value::getboolean() const { return false; }
const char* Value::getenum() const { return nil; }
const Value* Value::getmin() const { return nil; }
const Value* Value::getmax() const { return nil; }
const Value* Value::getinc() const { return nil; }
const char* Value::getfilename() const { return nil; }
const char* Value::getstrval() const { return nil; }
int Value::setval(const char* /*str*/) { return 0; }

const Value* Value::add(const Value*) { return nil; }
const Value* Value::subtract(const Value*) { return nil; }
const Value* Value::mul(const Value*) { return nil; }
const Value* Value::div(const Value*) { return nil; }

void Value::dump() const { printf ("Generic value.\n"); }

IntegerValue::IntegerValue(int val) { val_ = val; }
IntegerValue::~IntegerValue() { }
Value* IntegerValue::clone() const { 
    return (new IntegerValue(val_)); 
}
bool IntegerValue::isequal(const Value* val) const { 
    bool equal = false;
    if (val->isainteger()) {
	equal = val_ == val->getinteger();
    }
    return equal;
}

bool IntegerValue::isgt(const Value* val) const { 
    bool gt = false;
    if (val->isainteger()) {
	gt = val_ > val->getinteger();
    }
    return gt;
}


bool IntegerValue::isainteger() const { return true; }
int IntegerValue::getinteger() const { return val_; }
const char* IntegerValue::getstrval() const {
    if (strvalbuf_ == nil) {
	((IntegerValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    sprintf (strvalbuf_, "%d", val_);  
    return strvalbuf_; 
}

const Value* IntegerValue::add(const Value* val) {
    bool sametype = val->isainteger();
    if (sametype) {
	val_ += val->getinteger();
    }
    return (sametype) ? this : nil;
}

const Value* IntegerValue::subtract(const Value* val) {
    bool sametype = val->isainteger();
    if (sametype) {
	val_ -= val->getinteger();
    }
    return (sametype) ? this : nil;
}

const Value* IntegerValue::mul(const Value* val) {
    bool sametype = val->isainteger();
    if (sametype) {
	val_ *= val->getinteger();
    }
    return (sametype) ? this : nil;
}

const Value* IntegerValue::div(const Value* val) {
    bool sametype = val->isainteger();
    if (sametype) {
	val_ /= val->getinteger();
    }
    return (sametype) ? this : nil;
}

void IntegerValue::dump() const { printf("Integer value '%d'.\n",val_); }


Value *IntegerValue::clone(const char* str) const {
   if (!pmu_->match(str,"<integer>")) return nil; 
   Value *newval = new IntegerValue (pmu_->getintfield(1));
   return newval;
}

int IntegerValue::setval(const char* str) {
   if (!pmu_->match(str,"<integer>")) return 0;
   val_ = pmu_->getintfield(1);
   return 1;
}

RealValue::RealValue(double val) { val_ = val; }
RealValue::~RealValue() { }
Value* RealValue::clone() const{ return (new RealValue(val_)); }

bool RealValue::isequal(const Value* val) const { 
    bool equal = false;
    if (val->isareal()) {
	equal = (val_ - val->getreal()) < 1.0e-28;
    }
    return equal;
}

bool RealValue::isgt(const Value* val) const { 
    bool gt = false;
    if (val->isareal()) {
	gt = val_ > val->getreal();
    }
    return gt;
}


bool RealValue::isareal() const{ return true; }
double RealValue::getreal() const { return val_; }
const char* RealValue::getstrval() const { 
    if (strvalbuf_ == nil) {
	((RealValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    sprintf (strvalbuf_, "%-.15g", val_);  
    return strvalbuf_; 
}

const Value* RealValue::add(const Value* val) {
    bool sametype = val->isareal();
    if (sametype) {
	val_ += val->getreal();
    }
    return (sametype) ? this : nil;
}

const Value* RealValue::subtract(const Value* val) {
    bool sametype = val->isareal();
    if (sametype) {
	val_ -= val->getreal();
    }
    return (sametype) ? this : nil;
}

const Value* RealValue::mul(const Value* val) {
    bool sametype = val->isareal();
    if (sametype) {
	val_ *= val->getreal();
    }
    return (sametype) ? this : nil;
}

const Value* RealValue::div(const Value* val) {
    bool sametype = val->isareal();
    if (sametype) {
	val_ /= val->getreal();
    }
    return (sametype) ? this : nil;
}

void RealValue::dump() const { printf ("Real value `%g'.\n", val_); }

Value *RealValue::clone(const char* str) const {
   if (!pmu_->match(str,"<real>")) return nil; 
   Value *newval = new RealValue (pmu_->getrealfield(1));
   return newval;
}

int RealValue::setval(const char* str) {
   if (!pmu_->match(str,"<real>")) return 0;
   val_ = pmu_->getrealfield(1);
   return 1;
}

Value* TextValue::clone() const { return (new TextValue(val_)); }
bool TextValue::isatext() const { return true; }
const char* TextValue::gettext() const { return val_; }
const char* TextValue::getstrval() const { 
    if (strvalbuf_ == nil) {
	((TextValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    strcpy(strvalbuf_, val_); 
    return strvalbuf_; 
}
void TextValue::dump() const { printf ("Text value '%s'.\n", val_); }


TextValue::TextValue(const char* text) { 
    val_ = strcpy_with_alloc(text);
}
TextValue::~TextValue() { delete[] val_; val_ = nil; }

bool TextValue::isequal(const Value* val) const { 
    bool equal = false;
    if (val->isatext()) {
	equal = strcmp(val_, val->gettext()) == 0;
    }
    return equal;
}

// CHECK: this is garbage!
bool TextValue::isgt(const Value* val) const { 
    bool gt = false;
    if (val->isatext()) {
	gt = strcmp(val_, val->gettext()) > 0;
    }
    return gt;
}


int TextValue::setval(const char* str) {
    delete[] val_;
    val_ = strcpy_with_alloc(str);
    return 1;
}

Value *TextValue::clone(const char* str) const {
   Value *newval = new TextValue (str);
   return newval;
}

BooleanValue::BooleanValue(bool val) { val_ = val; }
BooleanValue::~BooleanValue() { }
Value* BooleanValue::clone() const { return (new BooleanValue(val_)); }
bool BooleanValue::isaboolean() const { return true; }
bool BooleanValue::getboolean() const { return val_; }
const char* BooleanValue::getstrval() const { 
    if (strvalbuf_ == nil) {
	((BooleanValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    strcpy(strvalbuf_, (val_ == true) ? "YES" : "NO"); 
    return strvalbuf_; 
}

bool BooleanValue::isequal(const Value* val) const { 
    bool equal = false;
    if (val->isaboolean()) {
	equal = val_ == val->getboolean();
    }
    return equal;
}

void BooleanValue::dump() const { 
    printf ("Boolean value '%d'.\n", (int)val_);
}

Value *BooleanValue::clone(const char* str) const {
   int b = -1;
   if (pmu_->match(str,"<boolean>"))    b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<yesno>")) b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<yn>"))    b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<onoff>")) b = pmu_->getcodefield(1);
   if (b == -1) return nil;
   Value *newval = new BooleanValue (b);
   return newval;
}


int BooleanValue::setval(const char* str) {
   int b = -1;
   if (pmu_->match(str,"<boolean>"))    b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<yesno>")) b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<yn>"))    b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<onoff>")) b = pmu_->getcodefield(1);
   if (b == -1) return 0;
   val_ = b;
   return 1;
}

EnumValue::EnumValue(const Stq* enumlist) {
    vallist_ = new Stq;
    if (enumlist) {
	unsigned nenums = enumlist->size();
	for(unsigned i = 0; i < nenums; i++) {
	    const char* enum_val = (char*)enumlist->cycle();
	    vallist_->enq(strcpy_with_alloc(enum_val));
	}
    }
    val_ = (vallist_->size() > 0) ? (char*)vallist_->top() : nil;
}

EnumValue::~EnumValue() { 
    unsigned nenums = vallist_->size();
    for(unsigned i = 0; i < nenums; i++) {
	char* enum_val = (char*)vallist_->pop();
	delete[] enum_val;
    }
    delete vallist_;
}

Value* EnumValue::clone() const { 
    EnumValue* newval = new EnumValue(vallist_);
    newval->setenum(val_);
    return newval;
}

Value* EnumValue::clone(const char* str) const { 
    EnumValue* newval = nil;
    if (valid(str)) {
	newval = new EnumValue(vallist_);
	newval->setenum(str);
    }
    return newval;
}

int EnumValue::setval(const char* str) {
   bool ok = valid(str);
   if(ok)
       setenum(str);
   return ok;
}

bool EnumValue::isaenum() const { return true; }
const char* EnumValue::getenum() const { return val_; }
void EnumValue::setenum(const char* val) { 
    unsigned matchlen = strlen(val);
    unsigned nenums = vallist_->size();
    for(unsigned i = 0; i < nenums; i++) {
	char* enum_val = (char*)vallist_->cycle();
	if (strncasecmp(val, enum_val, matchlen) == 0)
	    val_ = enum_val;
    }
}

const Stq* EnumValue::getenumlist() const { return vallist_; }
void EnumValue::setenumlist(const Stq* enumlist) { 
    unsigned nenums = vallist_->size();
    for(unsigned i = 0; i < nenums; i++) {
	char* enum_val = (char*)vallist_->pop();
	delete[] enum_val;
    }
    delete vallist_;
    vallist_ = new Stq;
    if (enumlist) {
	unsigned nenums = enumlist->size();
	for(unsigned i = 0; i < nenums; i++) {
	    const char* enum_val = (char*)enumlist->cycle();
	    vallist_->enq(strcpy_with_alloc(enum_val));
	}
    }
}
bool EnumValue::valid(const char* val) const {
    unsigned matchlen = strlen(val);
    unsigned matches = 0;
    unsigned nenums = vallist_->size();
    for(unsigned i = 0; i < nenums; i++) {
	const char* enum_val = (char*)vallist_->cycle();
	if (strncasecmp(val, enum_val, matchlen) == 0)
	    ++matches;
    }
    return matches == 1;
}

const char* EnumValue::getstrval() const { 
    if (strvalbuf_ == nil) {
	((EnumValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    strcpy(strvalbuf_, val_); 
    return strvalbuf_; 
}

void EnumValue::dump() const { 
    printf ("Allowed values:");
    unsigned nvals = vallist_->size();
    for(unsigned i = 0; i < nvals; i++) {
	printf (" %s", (char*)vallist_->cycle());
    }
    printf ("\nENUM value `%s'.\n", (val_) ? val_ : "NONE DEFINED");
}

RangeValue::~RangeValue() { }
bool RangeValue::isainteger() const { return min_->isainteger(); }
bool RangeValue::isareal() const { return min_->isareal(); }
const Value* RangeValue::getmin() const { return min_; }
const Value* RangeValue::getmax() const { return max_; }
const Value* RangeValue::getinc() const { return inc_; }

bool FilenameValue::isafilename() const { return true; }
const char* FilenameValue::getfilename() const { return TextValue::gettext(); }
void FilenameValue::dump() const { 
    printf ("Filename value '%s'.\n", TextValue::getstrval()); 
}

FilenameValue::FilenameValue(const char* text) : TextValue(text) { }
FilenameValue::~FilenameValue() { }


