//
// tool.cc: SHADOW tool interface implementation
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#include <iostream>
#include <strstream>

#include "global.h"
#include "tool.h"
#include "value.h"
#include "variable.h"

// ====================================================================== //
//
//  TOOLS
//
// ====================================================================== //

Tool::Tool(const string& name) : name_(name) { }

Tool::~Tool() { }

const string& Tool::name() const { return name_; }

const ShadowNamelist& Tool::namelist() const { return nml_; }

const Tool::VariableList& Tool::variables() const { 
    return nml_.variables();
}

bool Tool::exists_var(const string& varname) const {
    return nml_.exists(varname);
}

const Var* Tool::get_var(const string& varname) const {
    return nml_.get(varname);
}

int Tool::add_var(Var* var) { 
    return nml_.add(var);
}

void Tool::dump() const {
    cerr << "==== Dumping tool " << name() << " =====" << endl;
    nml_.dump();
    cerr << "==== Done =====" << endl;
}


// ====================================================================== //
//
//  TOOL INSTANCE
//
// ====================================================================== //

ToolInstance::ToolInstance (const Tool* tool, int instance) 
    : tool_(tool), instance_(instance), nml_(&tool->namelist())
{
    ostrstream ostr;
    ostr << tool_->name();
    if (instance_ > 0) {
        ostr << "(" << instance_ << ")";
    }
    ostr << ends;
    instname_ = ostr.str();
#ifndef __BORLANDC__
    ostr.freeze(0);
#endif
}

ToolInstance::ToolInstance (const ToolInstance& ti) : 
    tool_(ti.tool()), 
    instance_(ti.instance()), 
    nml_(ti.nml_)		// note this copies the instance variables
{
    ostrstream ostr;
    ostr << tool_->name();
    if (instance_ > 0) {
        ostr << "(" << instance_ << ")";
    }
    ostr << ends;
    instname_ = ostr.str();
#ifndef __BORLANDC__
    ostr.freeze(0);
#endif
}

ToolInstance::~ToolInstance() { }

ToolInstance* ToolInstance::clone() const { 
    return new ToolInstance(*this);
}

int ToolInstance::copy(const ToolInstance& from) { 
    if (this == &from)
	return 0;

    if (tool() != from.tool()) {
	cerr << "ToolInstance::copy: Incompatible types!" << endl;
	return 1;
    }

    instance_ = from.instance();
    instname_ = from.instance_name();
    //
    // note that copying namelists only works right when they have the
    // same parent, which is the namelist of the parent Tool, so only
    // one exists in the system!
    //
    nml_.copy(from.nml_);
    return 0;
}

const string& ToolInstance::name() const { return tool_->name(); }
const string& ToolInstance::instance_name() const { return instname_; }
const Tool* ToolInstance::tool() const { return tool_; }
int ToolInstance::instance() const { return instance_; }

void ToolInstance::set_instance_name(const string& instname) { 
    instname_ = instname; 
}

bool ToolInstance::exists_var(const string& varname) const {
    return nml_.exists(varname);
}
const Var* ToolInstance::get_var(const string& varname) const {
    return nml_.get(varname);
}

const Value* ToolInstance::get_varval(const string& varname) const {
    const Var* var = nml_.get(varname);
    return (var) ? var->getval() : nil;
}

int ToolInstance::add_var(Var* var) {
    return nml_.add(var);
}

int ToolInstance::set_varval(const string& varname, Value* val) {
    return nml_.change(varname, val);
}

void ToolInstance::dump() const {
    cerr << "==== Dumping tool " << instance_name() << " =====" << endl;
    nml_.dump();
    cerr << "==== Done =====" << endl;
}

// ====================================================================== //
//
//  TOOL Mgr
//
// ====================================================================== //

ToolMgr::ToolMgr() { }

ToolMgr::~ToolMgr() { 
    ToolMap::iterator it = tools_.begin();
    for(; it != tools_.end(); ++it) {
        Tool* tool = static_cast<Tool*>((*it).second);
        delete tool;
    }
}

//
// finds the tool corresponding to the give name, which could be the
// real name or a short alias (for the sake of SHADOW menu file which
// only allows for 3 characters in tool names).
//
const Tool* ToolMgr::find(const string& given_name) const {
    string name(given_name);
    unalias(name);

    ToolMap::const_iterator it = tools_.find(name);
    return (it != tools_.end()) ? 
        reinterpret_cast<const Tool*>((*it).second) : nil;
}

Tool* ToolMgr::find(const string& given_name) {
    string name(given_name);
    unalias(name);

    ToolMap::iterator it = tools_.find(name);
    return (it != tools_.end()) ? static_cast<Tool*>((*it).second) : nil;
}

Tool* ToolMgr::add(const string& given_name) {
    string name(given_name);
    unalias(name);

    Tool* curtool = find(name);
    if (curtool) {
        delete curtool;
    } else {
        toolnames_.push_back(name);
    }
    Tool* new_tool = new Tool(name);
    tools_[name] = new_tool;
    return new_tool;
}

//
// if the supplied name is an alias, then return the real thing. If alias,
// return 0, else return 1.
//
int ToolMgr::unalias(string& name) const {
    ToolAliasMap::const_iterator it1 = tool_aliases_.find(name);
    int retcode = 1;
    if (it1 != tool_aliases_.end()) {
	retcode = 0;
        name = (*it1).second;
    }
    return retcode;
}

int ToolMgr::add_alias(const string& alias, const string& realname) {
    tool_aliases_[alias] = realname;
    return 0;
}

void ToolMgr::dump() const { 
    ToolMap::const_iterator it = tools_.begin();
    for(; it != tools_.end(); ++it) {
        const Tool* tool = reinterpret_cast<const Tool*>((*it).second);
        tool->dump();
    }
}

//
// for Tcl interpreter access.
//
const list<string>& ToolMgr::tools() const {
    return toolnames_;
}

// ====================================================================== //
