/*
 * shdata_io.h: Routines for reading/writing SHADOW image files.
 *
 * ------------------------------------------------
 * Mumit Khan <khan@xraylith.wisc.edu>
 * Center for X-ray Lithography
 * University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 * Copyright (c) 1996 Mumit Khan
 */

#ifndef shdata_io_h
#define shdata_io_h

/* ======================================================================= */

#include <stdlib.h>
#include <tcl.h>

/* ======================================================================= */

/*
 * ShadowData holds the info stored in SHADOW RAY (begin.dat, star.xx,
 * mirr.xx) files. 
 * 
 * Note that the data that holds the raw rays are not organized as in 
 * SHADOW proper -- RAY[npoints][ncol] in C and RAY[ncol][npoints] in
 * FORTRAN -- but rather as a long vector, where the first npoints
 * belong to first column, 2nd npoints belong to 2nd column and so on;
 * this allows us to extract an entire column w/out any computation.
 *
 * This class is intimately connected with Tcl for error reporting and
 * BLT for creating vectors.
 */
typedef struct {
    Tcl_Interp *interp;
    int ncol;
    int npoints;
    int flag;
    double *ray;
    double column_info[21][3][5];	/* npcol, min,max,center,stdev */
    int which_rays;			/* current selection */
    int npcol;				/* applicable number */
    int good_rays;
} ShadowData;

enum RAY_OPTS {GOOD_RAYS, LOST_RAYS, ALL_RAYS};
enum RAY_COLUMNS {
    RAY_X = 1,
    RAY_Y = 2,
    RAY_Z = 3,
    RAY_X_PRIME = 4,
    RAY_Y_PRIME = 5,
    RAY_Z_PRIME = 6,
    RAY_GOOD = 10,
    RAY_LAMBDA = 11,
    RAY_OPD = 13,
    RAY_NA = 20
};

//
// extract a RAY[COL][POINT] value. column and point are 0 based here.
//
static inline double column_of_point(const ShadowData* shdata, 
    const int point, const int column
) {
    size_t offset = shdata->npoints * column + point;
    return shdata->ray[offset];
}

extern ShadowData *create_shadow_data(Tcl_Interp*, int ncol, 
    int npoints, int flag);
extern int reset_shadow_data(ShadowData* shdata, int ncol,
    int npoints, int flag, int good_rays, double *ray);
extern int delete_shadow_data(ShadowData*);

extern int read_shadow_data(Tcl_Interp*, const char*, ShadowData*); 
extern int write_shadow_data(Tcl_Interp*, const char*, const ShadowData*); 
extern int extract_shadow_column (Tcl_Interp*, const ShadowData*, 
    int column, char* vecname, int which);
extern int extract_shadow_column (Tcl_Interp*, const ShadowData*, 
    int column, char* vecname_good, char* vecname_lost);
extern int extract_shadow_histogram (Tcl_Interp*, const ShadowData*, 
    int column, char* xvecname, char* yvecname, int which, 
    double min, double max, int bins, double scale);
extern int extract_shadow_ray (Tcl_Interp*, const ShadowData*, 
    int ray, char* vecname);
extern int extract_shadow_raw_column (Tcl_Interp*, const ShadowData*, 
    int column, double** data);
extern int extract_shadow_raw_ray (Tcl_Interp*, const ShadowData*, 
    int ray, double** data);

/* ======================================================================= */

#endif /* !shdata_io_h */
