//
// page_cmds.cc: Tcl commands to manipulate MENU pages.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994 Mumit Khan
//

#ifdef __GNUG__
# include <cassert>
# include <cstring>
#else
# include <assert>
# include <string>
#endif

#include "menu.h"

#include "commands.h"
#include "xmenu.h"

// ================ Tcl commands created here ======================== //
/*
 *
 * ALL THE COMMANDS to "XPAGE" in this interpreter:
 *
 * xpage title <page>
 * xpage type <page>
 * xpage items <page>
 * xpage eval <page>
 * xpage dump <page>
 *
 */

// =================================================================== //

static inline char* string_to_charptr(const string& str) {
    return const_cast<char*>(str.c_str());
}

// =================================================================== //

static int page_title(Tcl_Interp* interp, const char* pname) {
    int retcode = TCL_OK;
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", const_cast<char*>(pname), "\" exists.", 
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    else {
	const char* title = page->title();
	Tcl_AppendResult(interp, (char*)title, (char*) NULL);
    }
    return retcode;
}


static int page_type(Tcl_Interp* interp, const char* pname) {
    int retcode = TCL_OK;
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    else {
	const char* type = page->type();
	Tcl_AppendResult(interp, (char*)type, (char*)NULL);
    }
    return retcode;
}

static int list_page_items(Tcl_Interp* interp, const char* pname) {
    int retcode = TCL_OK;
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    else {
	const list<string>& item_names = page->get_items();
	list<string>::const_iterator it = item_names.begin();
	for(; it != item_names.end(); ++it) {
	    const string& item_name = *it;
	    Tcl_AppendElement(interp, string_to_charptr(item_name));
	}
    }
    return retcode;
}

static int eval_page_flags(Tcl_Interp* interp, const char* pname) {
    int retcode = TCL_OK;
    MenuPage* page = GLB_menu->find_page(pname);
    if (page == nil) {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    else {
	if (page->evaluate_flags(GLB_beamline) != 0)
	    retcode = TCL_ERROR;
    }
    return retcode;
}

static int dump_page(Tcl_Interp* interp, const char* pname) {
    int retcode = TCL_OK;
    const MenuPage* page = GLB_menu->find_page(pname);
    if (page) {
	Tcl_AppendResult(interp, 
	    "dumping menu page \"", (char*)pname, "\"...", 
	    (char*) NULL
	);
	page->dump();
    }
    else {
	Tcl_AppendResult(interp, 
	    "ERROR: no such page \"", (char*)pname, "\" exists.", 
	    (char*) NULL
	);
	retcode = TCL_ERROR;
    }
    return retcode;
}


inline static int page_arg_error(Tcl_Interp* interp, 
    const char* cmd, const char* arg
) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", cmd,
	" ", arg, " PAGE\"", (char *) NULL
    );
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * XMenu_PageCmd:
 *
 *      This procedure is invoked to process the "menu" Tcl
 *      command.  
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *--------------------------------------------------------------
 */

int XMenu_PageCmd (ClientData, Tcl_Interp* interp, int argc, char** argv) {
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " option [args..]\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (strcmp(argv[1], "title") == 0) {
        if (argc != 3) {
	    return page_arg_error(interp, argv[0], argv[1]);
	}
        return page_title(interp, argv[2]);
    }
    if (strcmp(argv[1], "type") == 0) {
        if (argc != 3) {
	    return page_arg_error(interp, argv[0], argv[1]);
	}
        return page_type(interp, argv[2]);
    }
    if (strcmp(argv[1], "items") == 0) {
        if (argc != 3) {
	    return page_arg_error(interp, argv[0], argv[1]);
	}
        return list_page_items(interp, argv[2]);
    }
    if (strcmp(argv[1], "eval") == 0) {
        if (argc != 3) {
	    return page_arg_error(interp, argv[0], argv[1]);
	}
        return eval_page_flags(interp, argv[2]);
    }
    if (strcmp(argv[1], "dump") == 0) {
        if (argc != 3) {
	    return page_arg_error(interp, argv[0], argv[1]);
	}
        return dump_page(interp, argv[2]);
    }
    Tcl_AppendResult(interp, "bad option \"", argv[1],
	"\":  should be title, type, items, eval, dump", (char*) NULL
    );
    return TCL_ERROR;
}

// =================================================================== //
